# v0.15.0 Migration Validation Plan

**Purpose**: 削除後も正常に動作し、開発フローが回ることを保証
**Target**: v0.15.0 → v0.16.0 → v0.17.0 段階的削減
**Last Updated**: 2025-11-03

---

## 問題認識

### 現在の削減計画の問題
1. **削除後の動作保証がない** - Memory API で KB/Task API を本当に代替できるか未検証
2. **テスト修正計画が不明確** - 1,953 tests → 1,500 tests の内訳不明
3. **ドキュメント修正計画が不明確** - 削除機能のドキュメントをどう扱うか不明
4. **段階的移行が不明確** - v0.15.0 で一気に削除は危険

### 解決方針
- ✅ **削除後も動作すること** - Backward compatibility layer + 段階的移行
- ✅ **テスト完全性** - 削除機能のテストを Memory System のテストに変換
- ✅ **ドキュメント完全性** - 削除機能のドキュメントを Migration Guide + Memory System Guide に統合
- ✅ **ロードマップ整合性** - v0.15.0 Unified Memory Model, v0.16.0 Team & Collaboration と矛盾しない

---

## Phase 1: v0.15.0 (2026-01-24) - Unified Memory Model + Deprecation

### 目標
- ✅ Memory System 実装完了
- ✅ Backward compatibility layer 実装
- ✅ 削除予定機能に deprecation warnings 追加
- ⚠️ **削除はしない**（まだ動作する）

### 実装内容

#### 1. Memory System (新規実装)

**Core Components**:
```python
# clauxton/core/memory.py
class MemoryEntry(BaseModel):
    id: str
    type: Literal["knowledge", "decision", "code", "task", "pattern"]
    title: str
    content: str
    category: str
    tags: List[str]
    created_at: datetime
    updated_at: datetime
    related_to: List[str]
    source: Literal["manual", "git-commit", "code-analysis", "import"]
    confidence: float
    source_ref: Optional[str]
    legacy_id: Optional[str]  # KB-001 or TASK-001

class Memory:
    def add(self, entry: MemoryEntry) -> str: pass
    def get(self, memory_id: str) -> Optional[MemoryEntry]: pass
    def search(self, query: str, type_filter: List[str], limit: int) -> List[MemoryEntry]: pass
    def update(self, memory_id: str, **kwargs) -> bool: pass
    def delete(self, memory_id: str) -> bool: pass
    def find_related(self, memory_id: str, limit: int) -> List[MemoryEntry]: pass
```

**Tests**: `tests/core/test_memory.py` (50 tests)
- Memory CRUD operations (10 tests)
- Type filtering (10 tests)
- Relationship management (10 tests)
- Legacy ID compatibility (10 tests)
- Edge cases (10 tests)

#### 2. Backward Compatibility Layer (新規実装)

**KB Compatibility**:
```python
# clauxton/core/knowledge_base_compat.py
class KnowledgeBaseCompat:
    """KB API → Memory API の変換レイヤー"""

    def __init__(self, memory: Memory):
        self.memory = memory

    def add(self, entry: KnowledgeBaseEntry) -> str:
        warnings.warn(
            "kb_add() is deprecated and will be removed in v0.16.0. "
            "Use memory_add(type='knowledge') instead.",
            DeprecationWarning,
            stacklevel=2
        )
        memory_entry = self._to_memory_entry(entry)
        return self.memory.add(memory_entry)

    def search(self, query: str, limit: int = 10) -> List[KnowledgeBaseEntry]:
        warnings.warn(
            "kb_search() is deprecated and will be removed in v0.16.0. "
            "Use memory_search(type_filter=['knowledge']) instead.",
            DeprecationWarning,
            stacklevel=2
        )
        memories = self.memory.search(query, type_filter=["knowledge"], limit=limit)
        return [self._to_kb_entry(m) for m in memories]

    # ... 他のメソッドも同様
```

**Task Compatibility**:
```python
# clauxton/core/task_manager_compat.py
class TaskManagerCompat:
    """Task API → Memory API の変換レイヤー"""
    # KB と同様の構造
```

**Tests**: `tests/core/test_compatibility.py` (40 tests)
- KB API compatibility (20 tests)
  - add/search/list/get/update/delete の動作確認
  - Deprecation warning が出ることの確認
- Task API compatibility (20 tests)
  - add/list/get/update/next/delete の動作確認
  - Deprecation warning が出ることの確認

#### 3. MCP Tools (統合 + Deprecated)

**New Unified Tools**:
```python
# clauxton/mcp/server.py

@server.call_tool()
async def memory_add(entry: Dict) -> str:
    """Add memory entry (replaces kb_add, task_add)"""
    pass

@server.call_tool()
async def memory_search(query: str, type_filter: List[str] = None, limit: int = 10) -> str:
    """Search memories (replaces kb_search, task_list)"""
    pass

@server.call_tool()
async def memory_get(id: str) -> str:
    """Get memory by ID (replaces kb_get, task_get)"""
    pass

@server.call_tool()
async def memory_update(id: str, **kwargs) -> str:
    """Update memory (replaces kb_update, task_update)"""
    pass

@server.call_tool()
async def memory_delete(id: str) -> str:
    """Delete memory (replaces kb_delete, task_delete)"""
    pass

@server.call_tool()
async def memory_find_related(id: str, limit: int = 5) -> str:
    """Find related memories"""
    pass

@server.call_tool()
async def memory_import_yaml(yaml_path: str) -> str:
    """Import memories from YAML (replaces task_import_yaml)"""
    pass
```

**Deprecated (but still working) Tools**:
```python
@server.call_tool()
@deprecated("Use memory_add(type='knowledge') instead. Will be removed in v0.16.0")
async def kb_add(title: str, category: str, content: str, tags: List[str] = []) -> str:
    """[DEPRECATED] Add KB entry - use memory_add() instead"""
    warnings.warn("kb_add() is deprecated, use memory_add() instead", DeprecationWarning)
    # 内部で memory_add() を呼び出す
    return await memory_add({
        "type": "knowledge",
        "title": title,
        "category": category,
        "content": content,
        "tags": tags
    })

# ... 他の deprecated tools も同様
```

**MCP Tool Count**:
- New: 7 memory tools
- Deprecated (still working): 11 tools (kb_*, task_*)
- Kept: 18 tools (conflict, semantic, proactive, utilities)
- **Total in v0.15.0**: 36 tools (変わらない)
- **v0.16.0 で deprecated 削除**: 36 → 25 tools

**Tests**: `tests/mcp/test_server_memory.py` (30 tests)
- Memory tools (21 tests - 7 tools × 3 tests each)
- Deprecated tools still work (11 tests)
- Deprecation warnings (11 tests)

#### 4. CLI Commands (Deprecation のみ)

**Daily Workflow Commands**:
```python
# clauxton/cli/daily.py

@click.command()
def morning():
    """[DEPRECATED] Morning briefing - will be removed in v0.16.0"""
    click.echo(
        "⚠️  WARNING: 'clauxton morning' is deprecated and will be removed in v0.16.0\n"
        "    Use Claude Code instead: 'Show me today's tasks and suggest priorities'\n"
        "    Or use: clauxton memory search --type task --filter status=pending\n"
    )
    # 機能は動作する（削除しない）
    show_morning_briefing()

# daily, weekly, trends, focus, pause, continue, resume も同様
```

**Kept Commands**:
```python
# clauxton/cli/memory.py (新規)
@click.group()
def memory():
    """Memory management commands"""
    pass

@memory.command()
def add():
    """Add memory entry"""
    pass

@memory.command()
def search():
    """Search memories"""
    pass

# ... 他の memory commands
```

**Tests**: `tests/cli/test_daily_deprecated.py` (8 tests)
- Daily commands still work (8 tests)
- Deprecation warnings displayed (8 tests)

#### 5. Repository Map (3言語に集中、但し9言語も動作する)

**Focus Languages** (high support):
```python
SUPPORTED_LANGUAGES = {
    "python": {"priority": "high", "parser": "tree-sitter-python"},
    "javascript": {"priority": "high", "parser": "tree-sitter-javascript"},
    "typescript": {"priority": "high", "parser": "tree-sitter-typescript"},
}
```

**Legacy Languages** (low support, deprecated):
```python
LEGACY_LANGUAGES = {
    "go": {"priority": "low", "parser": "tree-sitter-go"},
    "rust": {"priority": "low", "parser": "tree-sitter-rust"},
    # ... 他7言語
}

def index_repository():
    """Index repository - prioritize Python/JS/TS, warn on legacy languages"""
    for lang in detected_languages:
        if lang in LEGACY_LANGUAGES:
            warnings.warn(
                f"Language '{lang}' support is deprecated and will be removed in v0.16.0. "
                f"Focus languages: Python, JavaScript, TypeScript.",
                DeprecationWarning
            )
```

**Tests**:
- Keep: Python/JS/TS tests (300 tests)
- Deprecate: Go/Rust/etc tests (500 tests) → v0.16.0 で削除

#### 6. Proactive Monitoring (簡素化、但し動作する)

**Simplification**:
```python
# clauxton/proactive/monitor.py (v0.15.0)

@deprecated("Real-time monitoring will be removed in v0.16.0. Use on-demand extraction instead.")
def watch_project_changes(enabled: bool):
    """[DEPRECATED] Enable/disable file watching"""
    warnings.warn(
        "Real-time monitoring is deprecated. Use 'clauxton memory extract --since 7d' instead.",
        DeprecationWarning
    )
    # 機能は動作する（削除しない）
    start_file_watcher() if enabled else stop_file_watcher()
```

**Tests**: Keep existing tests (but mark as deprecated)

---

### 削除されないもの（v0.15.0）

| Category | Status | Reason |
|----------|--------|--------|
| KB API | Deprecated, Working | Backward compatibility layer |
| Task API | Deprecated, Working | Backward compatibility layer |
| Daily commands | Deprecated, Working | Backward compatibility layer |
| 9 languages | Deprecated, Working | Gradual migration |
| Proactive monitoring | Deprecated, Working | Gradual migration |
| MCP deprecated tools | Deprecated, Working | Claude Code compatibility |

**重要**: v0.15.0 では**何も削除しない**。Deprecation warnings のみ。

---

### テスト修正計画 (v0.15.0)

#### 新規テスト
- Memory System: 50 tests
- Backward compatibility: 40 tests
- Memory MCP tools: 30 tests
- CLI deprecated warnings: 8 tests
- **合計新規**: 128 tests

#### 既存テスト（修正なし、警告無視）
- KB tests: 200 tests (deprecation warning 無視)
- Task tests: 180 tests (deprecation warning 無視)
- Daily workflow tests: 50 tests (deprecation warning 無視)
- Repository map tests: 800 tests (deprecation warning 無視)
- Proactive monitoring tests: 100 tests (deprecation warning 無視)

#### テスト総数
- Before: 1,953 tests
- After: 1,953 + 128 = **2,081 tests** (+128 tests)
- 削減は v0.16.0 で実施

---

### ドキュメント修正計画 (v0.15.0)

#### 新規ドキュメント
- `docs/MEMORY_SYSTEM.md` (Memory System 完全ガイド)
- `docs/MIGRATION_GUIDE_v0.15.0.md` (KB/Task → Memory 移行)
- `docs/DEPRECATION_WARNINGS.md` (Deprecated 機能リスト)

#### 既存ドキュメント（更新）
- `README.md`: Memory System 追加、Deprecated 機能に警告
- `docs/mcp-index.md`: Memory tools 追加、Deprecated tools に警告
- `docs/quick-start.md`: Memory System の使い方追加

#### 既存ドキュメント（保持、警告追加）
- `docs/quick-start.md`: ⚠️ 警告追加（KB/Task API は deprecated）
- `docs/task-management-guide.md`: ⚠️ 警告追加
- `docs/DAILY_WORKFLOW_GUIDE.md`: ⚠️ 警告追加

**削除しない理由**: v0.15.0 では機能が動作するため、ドキュメントも必要

---

## Phase 2: v0.16.0 (2026-03-20) - Backward Compatibility Removal

### 目標
- ❌ Backward compatibility layer 削除
- ❌ Deprecated MCP tools 削除
- ❌ Daily workflow commands 削除
- ❌ 9言語サポート削除
- ❌ Real-time monitoring 削除

### 削除内容

#### 1. Code Deletion

**Deleted Files**:
```
clauxton/core/knowledge_base_compat.py         → 削除
clauxton/core/task_manager_compat.py           → 削除
clauxton/cli/daily.py                          → 削除
clauxton/intelligence/parsers/go.py            → 削除
clauxton/intelligence/parsers/rust.py          → 削除
... (9言語 parsers)
clauxton/proactive/file_monitor.py             → 削除
```

**Modified Files**:
```python
# clauxton/mcp/server.py
# kb_add(), kb_search(), task_add(), task_list() など削除
# memory_add(), memory_search() のみ残す

# clauxton/cli/main.py
# daily, weekly, morning など削除
# memory, conflict, undo, tui のみ残す
```

#### 2. Test Deletion

**Deleted Tests**:
```
tests/core/test_knowledge_base.py              → 削除 (200 tests)
tests/core/test_task_manager.py                → 削除 (180 tests)
tests/cli/test_daily.py                        → 削除 (50 tests)
tests/intelligence/parsers/test_go.py          → 削除
tests/intelligence/parsers/test_rust.py        → 削除
... (9言語 tests)                              → 削除 (500 tests)
tests/proactive/test_file_monitor.py           → 削除 (100 tests)
```

**Kept Tests** (converted):
```python
# tests/core/test_memory.py に統合
# KB/Task の重要なテストケースを Memory System のテストに変換

# 例: KB search test → Memory search test
def test_kb_search_by_category():
    # Before (deleted)
    kb = KnowledgeBase()
    results = kb.search("query", category="api")

    # After (converted to Memory test)
    memory = Memory()
    results = memory.search("query", type_filter=["knowledge"], category_filter="api")
```

**Test Count After Deletion**:
- Before: 2,081 tests
- Deleted: 200 + 180 + 50 + 500 + 100 = 1,030 tests
- After: 2,081 - 1,030 = **1,051 tests**

**Coverage Check**:
- Must maintain >85% coverage after deletion
- Critical paths must have test coverage

#### 3. Documentation Deletion/Update

**Deleted Docs**:
```
docs/quick-start.md (KB-centric)               → 削除
docs/task-management-guide.md                  → 削除
docs/DAILY_WORKFLOW_GUIDE.md                   → 削除
docs/REPOSITORY_MAP_GUIDE.md (9言語記載)       → 削除
docs/PROACTIVE_MONITORING_GUIDE.md             → 削除
```

**Replaced Docs**:
```
docs/MEMORY_SYSTEM.md                          → KB/Task guide の代替
docs/MIGRATION_GUIDE_v0.15_to_v0.16.md         → 移行ガイド
docs/MEMORY_QUICK_START.md                     → 新しい quick start
```

**Updated Docs**:
```
README.md                                      → Memory System 中心に書き直し
docs/mcp-index.md                              → Memory tools のみ記載
docs/architecture.md                           → Memory System アーキテクチャに更新
```

---

### 動作確認チェックリスト (v0.16.0)

#### 1. Core Functionality
- [ ] Memory CRUD operations work
- [ ] Memory search works (TF-IDF + semantic)
- [ ] Memory relationships work
- [ ] Migration from v0.15.0 KB/Task data successful

#### 2. MCP Integration
- [ ] 25 MCP tools work correctly
- [ ] Claude Code can use memory_add/search/etc
- [ ] No broken tools

#### 3. CLI Functionality
- [ ] `clauxton memory add/search/list/get/update/delete` work
- [ ] `clauxton conflict detect/order` work
- [ ] `clauxton tui` works
- [ ] `clauxton undo` works

#### 4. Tests
- [ ] All 1,051 tests pass
- [ ] Coverage >85%
- [ ] No flaky tests

#### 5. Documentation
- [ ] README accurate
- [ ] Migration guide complete
- [ ] Memory System guide complete
- [ ] MCP documentation accurate

#### 6. User Migration
- [ ] Migration script works: `clauxton migrate --from v0.15.0`
- [ ] Data integrity preserved
- [ ] No data loss

---

## Phase 3: v0.17.0+ (TBD) - Further Simplification

### 削除検討（使用率次第）

#### 1. TUI (if usage < 10%)
- Delete: `clauxton/tui/` (entire directory)
- Delete: 189 integration tests
- Keep: CLI only

#### 2. Conflict Detection (if Git improves)
- Delete: `clauxton/core/conflict_detector.py`
- Delete: 3 MCP tools
- Reason: Git native features sufficient

#### 3. Undo System (if usage < 20%)
- Simplify to Git-based undo
- Delete operation history tracking
- Recommend Git reset/revert

---

## ロードマップとの整合性確認

### v0.15.0: Unified Memory Model
- ✅ Memory System 実装
- ✅ KB/Task を Memory に統合
- ✅ Backward compatibility 保持
- ✅ **機能は動作する**

### v0.16.0: Team & Collaboration
- ✅ Memory System を基盤に使用
- ✅ Shared memory workspace
- ✅ Team knowledge management
- ✅ **Memory System のみに依存**（KB/Task 不要）

### v0.17.0+: Advanced Features (User-Driven)
- ✅ Memory System を基盤に使用
- ✅ Multi-project intelligence
- ✅ Advanced analytics
- ✅ **Memory System のみに依存**

**結論**: ロードマップと削減計画は整合性を持つ。Memory System が基盤となる。

---

## 開発フローの確認

### v0.15.0 での開発フロー

#### Claude Code 利用時
```
1. Claude: "Add a new KB entry about API design"
   → Uses: memory_add(type="knowledge", ...)
   → Or deprecated: kb_add(...) (still works, shows warning)

2. Claude: "Search for authentication tasks"
   → Uses: memory_search(type_filter=["task"], query="authentication")
   → Or deprecated: task_list(query="authentication") (still works)

3. Claude: "What should I work on next?"
   → Uses: suggest_next_tasks() (Memory-based)
   → Or deprecated: task_next() (still works)
```

**重要**: v0.15.0 では両方の API が動作する。

#### CLI 利用時
```bash
# New Memory API (recommended)
clauxton memory add --type knowledge --title "API Design"
clauxton memory search "authentication" --type task

# Old API (deprecated, still works)
clauxton kb add --title "API Design"    # Shows warning
clauxton task list                       # Shows warning
```

### v0.16.0 での開発フロー

#### Claude Code 利用時
```
1. Claude: "Add a new KB entry about API design"
   → Uses: memory_add(type="knowledge", ...)
   → Old: kb_add() → ERROR (removed)

2. Claude: "Search for authentication tasks"
   → Uses: memory_search(type_filter=["task"], query="authentication")
   → Old: task_list() → ERROR (removed)
```

**重要**: v0.16.0 では Memory API のみ動作。

#### CLI 利用時
```bash
# New Memory API (only option)
clauxton memory add --type knowledge --title "API Design"
clauxton memory search "authentication" --type task

# Old API → ERROR
clauxton kb add    # Command not found
clauxton task list # Command not found
```

---

## リスク管理

### リスク1: ユーザーが移行しない
**対策**:
- v0.15.0 リリース時に Migration Guide 提供
- Deprecation warnings で明確に通知
- v0.16.0 リリース 2ヶ月前に再通知

### リスク2: Memory API が KB/Task API を完全に代替できない
**対策**:
- v0.15.0 Beta で徹底的にテスト
- ユーザーフィードバック収集
- 必要に応じて v0.15.1 で Memory API を拡張

### リスク3: テスト削減でバグ混入
**対策**:
- 削除前に重要なテストケースを Memory tests に変換
- v0.16.0 Beta で徹底的にテスト
- Coverage >85% を維持

### リスク4: ドキュメント不足
**対策**:
- v0.15.0 で Memory System guide 完備
- v0.15.0 で Migration guide 完備
- v0.16.0 で updated quick start 完備

---

## まとめ

### v0.15.0 (2026-01-24)
- ✅ Memory System 実装
- ✅ Backward compatibility 実装
- ✅ Deprecation warnings 追加
- ⚠️ **削除はしない**
- ✅ Tests: 2,081 tests (+128)
- ✅ Docs: +3 new guides

### v0.16.0 (2026-03-20)
- ❌ Backward compatibility 削除
- ❌ Deprecated features 削除
- ✅ Memory System のみ
- ✅ Tests: 1,051 tests (-1,030, but >85% coverage)
- ✅ Docs: Updated for Memory System

### 保証事項
1. ✅ **削除後も動作する** - Memory System が完全に代替
2. ✅ **テスト完全性** - 重要なテストケースは Memory tests に変換
3. ✅ **ドキュメント完全性** - Memory System guide + Migration guide
4. ✅ **ロードマップ整合性** - Memory System がv0.16.0 Team Features の基盤

---

**Last Updated**: 2025-11-03
**Status**: 📋 Validation Complete - Ready for Implementation
