# Phase 3 Integration & QA Startup Guide

**Version**: v0.15.0 Unified Memory Model
**Phase**: Phase 3 Integration Testing & Quality Assurance
**Date**: 2025-11-03
**Status**: ✅ Ready to Execute

---

## 📋 Quick Start for New Claude Code Session

### Prerequisites Checklist

Phase 3 実装が完了しています。次は統合テストと品質保証です。

```bash
# 1. 現在のブランチを確認
git branch --show-current
# Expected: feature/v0.15.0-unified-memory

# 2. 最新のコミットを確認
git log --oneline | head -3
# Expected:
#   4c7e9b2 feat: implement Phase 3 Memory Intelligence (Q&A, Summarization, Graph Visualization)
#   23be304 docs: add Phase 3 startup guide with parallel agent execution plan
#   01aa67f feat: implement Phase 2 Smart Memory (Memory Extraction, Linking, CLI)

# 3. Phase 3 のテストが通過することを確認
./.venv/bin/pytest tests/semantic/test_memory_qa.py \
                   tests/semantic/test_memory_summarizer.py \
                   tests/visualization/test_memory_graph.py -v
# Expected: 62 passed

# 4. Working directory が clean であることを確認
git status
# Expected: "nothing to commit, working tree clean"

# 5. Python 環境を確認
./.venv/bin/python --version
# Expected: Python 3.11+
```

✅ **All checks passed?** → Proceed to Integration Testing

---

## 🎯 セッションの目的

**このセッションで達成すること**:

1. **Phase 2 ↔ Phase 3 統合テストの実装と実行**
2. **エンドツーエンドワークフローの検証**
3. **品質保証チェックの完了**
4. **ドキュメントの更新**

**期待される成果**:
- 統合テストが 50% 以上実装される
- すべてのテストが通過
- 品質スコアが Grade A (90/100) に到達
- Phase 3 が production-ready になる

---

## 🚀 推奨実行順序

### Step 1: 基本統合テストの実行 (30分)

既存のテストで Phase 2-3 の基本連携を確認します。

```bash
# 1. Phase 2 のテストが通ることを確認
./.venv/bin/pytest tests/semantic/test_memory_extractor.py \
                   tests/semantic/test_memory_linker.py \
                   tests/cli/test_memory_extract.py -v

# Expected: 74 passed, 1 deselected

# 2. Phase 3 のテストが通ることを確認
./.venv/bin/pytest tests/semantic/test_memory_qa.py \
                   tests/semantic/test_memory_summarizer.py \
                   tests/visualization/test_memory_graph.py -v

# Expected: 62 passed

# 3. すべてのセマンティックテストを一括実行
./.venv/bin/pytest tests/semantic/ tests/visualization/ -v

# Expected: 136 passed
```

**成功基準**:
- [ ] Phase 2 テスト: 74 passed
- [ ] Phase 3 テスト: 62 passed
- [ ] 全体: 136 passed, 0 failed

**もし失敗したら**: エラーメッセージを確認し、該当モジュールを修正

---

### Step 2: 簡易手動統合テスト (15分)

CLI を使って Phase 2-3 の連携を手動で確認します。

```bash
# テスト用の一時ディレクトリを作成
cd /tmp
mkdir -p clauxton-integration-test
cd clauxton-integration-test

# Clauxton を初期化
clauxton init

# Phase 2: Git コミットからメモリを抽出
# (注: 実際の git リポジトリがない場合はスキップ)
# clauxton memory extract --limit 5 --auto-add

# 代わりに手動でメモリを追加
clauxton memory add \
  --title "Switch to PostgreSQL" \
  --type decision \
  --category database \
  --content "Migrate from MySQL to PostgreSQL for better JSONB support and performance"

clauxton memory add \
  --title "Use JWT Authentication" \
  --type decision \
  --category authentication \
  --content "Implement JWT-based authentication with RS256 algorithm for API security"

clauxton memory add \
  --title "RESTful API Design" \
  --type pattern \
  --category api \
  --content "Follow RESTful principles with proper HTTP methods and status codes"

# Phase 3: 質問応答のテスト
echo "=== Testing Question Answering ==="
# Note: CLI コマンドが実装されていない場合、Python API でテスト

# Phase 3: サマリ生成のテスト
echo "=== Testing Summarization ==="
# Note: CLI コマンドが実装されていない場合、Python API でテスト

# Phase 3: グラフ生成のテスト
echo "=== Testing Graph Visualization ==="
clauxton memory graph --format mermaid --output test_graph.md
cat test_graph.md

clauxton memory graph --format dot --output test_graph.dot
cat test_graph.dot

clauxton memory graph --format json --output test_graph.json
cat test_graph.json

# クリーンアップ
cd /home/kishiyama-n/workspace/projects/clauxton
rm -rf /tmp/clauxton-integration-test
```

**成功基準**:
- [ ] メモリが正常に追加される
- [ ] グラフが正常に生成される (3形式)
- [ ] エラーが発生しない

---

### Step 3: 統合テストコードの実装 (2-3時間)

詳細は `docs/PHASE3_INTEGRATION_TEST_PLAN.md` を参照してください。

#### 3.1 Phase 2 ↔ Phase 3 統合テストの作成

**ファイル**: `tests/integration/test_phase2_phase3_integration.py`

実装するテスト:
1. `test_extract_and_ask()` - メモリ抽出 → 質問応答
2. `test_link_and_visualize()` - メモリリンク → グラフ生成
3. `test_extract_and_summarize()` - メモリ抽出 → サマリ生成

**テンプレート**:
```python
"""
Integration tests for Phase 2 ↔ Phase 3 interaction.

Tests:
- Memory extraction (Phase 2) → Question answering (Phase 3)
- Memory linking (Phase 2) → Graph visualization (Phase 3)
- Memory extraction (Phase 2) → Summarization (Phase 3)
"""

import pytest
from pathlib import Path
from datetime import datetime

from clauxton.core.memory import Memory, MemoryEntry
from clauxton.semantic.memory_extractor import MemoryExtractor
from clauxton.semantic.memory_linker import MemoryLinker
from clauxton.semantic.memory_qa import MemoryQA
from clauxton.semantic.memory_summarizer import MemorySummarizer
from clauxton.visualization.memory_graph import MemoryGraph


@pytest.fixture
def temp_project(tmp_path: Path) -> Path:
    """Create temporary project with git repo."""
    project = tmp_path / "test_project"
    project.mkdir()
    (project / ".clauxton").mkdir()

    # Initialize git repo (optional, for extraction tests)
    # subprocess.run(["git", "init"], cwd=project, check=True)

    return project


def test_extract_and_ask(temp_project: Path) -> None:
    """Test memory extraction → question answering flow."""
    # Phase 2: Add some memories manually (simulating extraction)
    memory = Memory(temp_project)
    now = datetime.now()

    mem1 = MemoryEntry(
        id="MEM-20251103-001",
        type="decision",
        title="Switch to PostgreSQL",
        content="Migrate from MySQL to PostgreSQL for better JSONB support",
        category="database",
        tags=["postgresql", "migration"],
        created_at=now,
        updated_at=now,
        source="git-commit",
    )
    memory.add(mem1)

    # Phase 3: Ask questions
    qa = MemoryQA(temp_project)
    answer, confidence, sources = qa.answer_question("Why did we switch to PostgreSQL?")

    # Assertions
    assert "PostgreSQL" in answer
    assert confidence > 0.5
    assert "MEM-20251103-001" in sources


def test_link_and_visualize(temp_project: Path) -> None:
    """Test memory linking → graph visualization flow."""
    # Phase 2: Create related memories
    memory = Memory(temp_project)
    now = datetime.now()

    mem1 = MemoryEntry(
        id="MEM-20251103-001",
        type="decision",
        title="Switch to PostgreSQL",
        content="Use PostgreSQL for database",
        category="database",
        tags=["postgresql", "database"],
        created_at=now,
        updated_at=now,
        source="manual",
    )
    mem2 = MemoryEntry(
        id="MEM-20251103-002",
        type="pattern",
        title="Database Connection Pool",
        content="Use connection pooling with PostgreSQL",
        category="database",
        tags=["postgresql", "performance"],
        created_at=now,
        updated_at=now,
        source="manual",
    )
    memory.add(mem1)
    memory.add(mem2)

    # Phase 2: Link memories
    linker = MemoryLinker(temp_project)
    linker.auto_link_all(threshold=0.5)

    # Phase 3: Generate graph
    graph = MemoryGraph(temp_project)
    graph_data = graph.generate_graph_data()

    # Assertions
    assert len(graph_data["nodes"]) == 2
    assert len(graph_data["edges"]) >= 1  # Should have at least one link


def test_extract_and_summarize(temp_project: Path) -> None:
    """Test memory extraction → summarization flow."""
    # Phase 2: Add multiple memories
    memory = Memory(temp_project)
    now = datetime.now()

    memories = [
        MemoryEntry(
            id=f"MEM-20251103-{i:03d}",
            type="decision" if i % 2 == 0 else "pattern",
            title=f"Decision {i}",
            content=f"Content {i}",
            category="api" if i % 2 == 0 else "database",
            tags=["tag1", "tag2"],
            created_at=now,
            updated_at=now,
            source="git-commit",
        )
        for i in range(1, 11)
    ]

    for mem in memories:
        memory.add(mem)

    # Phase 3: Generate summary
    summarizer = MemorySummarizer(temp_project)
    summary = summarizer.summarize_project()

    # Assertions
    assert "architecture_decisions" in summary
    assert "active_patterns" in summary
    assert summary["statistics"]["total"] == 10


# Add more integration tests here...
```

**実装手順**:
```bash
# 1. ディレクトリ作成
mkdir -p tests/integration

# 2. __init__.py 作成
touch tests/integration/__init__.py

# 3. テストファイル作成
# (上記テンプレートを使用)

# 4. テスト実行
pytest tests/integration/test_phase2_phase3_integration.py -v
```

#### 3.2 エンドツーエンドテストの作成

**ファイル**: `tests/integration/test_e2e_workflow.py`

実装するテスト:
1. `test_full_memory_lifecycle()` - 完全なメモリライフサイクル (7ステップ)

#### 3.3 MCP 統合テストの作成

**ファイル**: `tests/integration/test_mcp_integration.py`

実装するテスト:
1. すべての MCP ツールの動作確認
2. エラーハンドリングの確認

---

### Step 4: 品質保証チェック (1-2時間)

詳細は `docs/PHASE3_QUALITY_ASSURANCE_PLAN.md` を参照してください。

#### 4.1 コード品質チェック

```bash
# Type checking
mypy --strict clauxton/semantic/memory_qa.py
mypy --strict clauxton/semantic/memory_summarizer.py
mypy --strict clauxton/visualization/memory_graph.py

# Linting
ruff check clauxton/semantic/ clauxton/visualization/ tests/

# Complexity analysis
radon cc clauxton/semantic/ clauxton/visualization/ -a -nb
radon mi clauxton/semantic/ clauxton/visualization/ -s
```

**成功基準**:
- [ ] mypy: 0 errors
- [ ] ruff: All checks passed
- [ ] radon: CC <10, MI >70

#### 4.2 カバレッジチェック

```bash
# Phase 3 modules coverage
pytest --cov=clauxton/semantic/memory_qa \
       --cov=clauxton/semantic/memory_summarizer \
       --cov=clauxton/visualization/memory_graph \
       --cov-report=html \
       --cov-report=term-missing

# Open coverage report
xdg-open htmlcov/index.html  # Linux
open htmlcov/index.html      # macOS
```

**成功基準**:
- [ ] memory_qa.py: ≥90% (現在 85%)
- [ ] memory_summarizer.py: ≥90% (現在 98%)
- [ ] memory_graph.py: ≥85% (現在 100%)
- [ ] 全体: ≥90% (現在 88%)

**カバレッジ改善が必要な場合**:
```bash
# memory_qa.py の未カバー部分を特定
pytest --cov=clauxton/semantic/memory_qa \
       --cov-report=term-missing \
       tests/semantic/test_memory_qa.py -v

# 不足しているテストケースを追加
# - scikit-learn なし環境のテスト
# - Fallback エラーケースのテスト
```

#### 4.3 セキュリティスキャン

```bash
# Security linting
bandit -r clauxton/semantic/ clauxton/visualization/ -ll

# Dependency vulnerabilities
pip-audit

# License compliance
pip-licenses --format=markdown --output-file=licenses.md
```

**成功基準**:
- [ ] bandit: 0 high/medium severity issues
- [ ] pip-audit: 0 vulnerabilities
- [ ] すべての依存関係のライセンスが確認済み

---

### Step 5: ドキュメント更新 (1-2時間)

#### 5.1 README.md の更新

**追加する内容**:
```markdown
## Phase 3: Memory Intelligence (v0.15.0) ✨ NEW

### Question Answering
Ask natural language questions about your project:

\`\`\`bash
# Note: CLI command to be implemented
# Python API usage:
from clauxton.semantic.memory_qa import MemoryQA

qa = MemoryQA(Path("."))
answer, confidence, sources = qa.answer_question("Why did we switch to PostgreSQL?")
print(f"Answer: {answer} (confidence: {confidence:.2f})")
print(f"Sources: {sources}")
\`\`\`

### Project Summarization
Generate comprehensive project summaries:

\`\`\`bash
# Note: CLI command to be implemented
# Python API usage:
from clauxton.semantic.memory_summarizer import MemorySummarizer

summarizer = MemorySummarizer(Path("."))
summary = summarizer.summarize_project()
print(summary)
\`\`\`

### Memory Graph Visualization
Visualize memory relationships:

\`\`\`bash
# Generate Mermaid diagram
clauxton memory graph --format mermaid --output graph.md

# Generate Graphviz DOT
clauxton memory graph --format dot --output graph.dot
dot -Tpng graph.dot -o graph.png

# Generate JSON for web visualization
clauxton memory graph --format json --output graph.json
\`\`\`

### MCP Tools (for Claude Code)

Phase 3 adds 6 new MCP tools:
- \`answer_question(question, top_k)\` - Q&A with confidence scoring
- \`get_project_summary()\` - Comprehensive project summary
- \`suggest_next_tasks(limit)\` - AI task suggestions
- \`detect_knowledge_gaps()\` - Missing documentation detection
- (Graph visualization tools - future)
```

#### 5.2 User Guide の作成

**ファイル**: `docs/USER_GUIDE.md`

**内容**:
1. インストール手順
2. クイックスタート
3. すべての CLI コマンドの説明
4. 使用例とベストプラクティス
5. トラブルシューティング

#### 5.3 MCP Tools Documentation の完成

**ファイル**: `docs/mcp-phase3-tools.md`

**内容**:
- Phase 3 で追加された 6 つの MCP ツールの詳細説明
- 使用例
- パラメータとレスポンス形式

---

### Step 6: 最終確認とコミット (30分)

```bash
# 1. すべてのテストを実行
pytest tests/ -v

# 2. カバレッジレポート生成
pytest --cov=clauxton --cov-report=html --cov-report=term

# 3. 品質チェック
mypy --strict clauxton/
ruff check clauxton/ tests/

# 4. 変更をステージング
git add tests/integration/
git add docs/
git add README.md

# 5. コミット
git commit -m "test: add Phase 3 integration tests and update documentation

- Add Phase 2 ↔ Phase 3 integration tests
- Add end-to-end workflow tests
- Add MCP integration tests
- Update README.md with Phase 3 features
- Add User Guide and MCP documentation
- Improve test coverage to >90%

Test Results:
- Unit tests: 62/62 passing
- Integration tests: X/X passing
- Coverage: >90%

🤖 Generated with [Claude Code](https://claude.com/claude-code)

Co-Authored-By: Claude <noreply@anthropic.com>"
```

---

## ✅ 完了基準

このセッション終了時に達成すべきこと:

### 必須 (Must Have)
- [ ] Phase 2-3 統合テストが実装され、すべて通過
- [ ] エンドツーエンドワークフローテストが実装され、通過
- [ ] カバレッジが ≥88% (現状維持または改善)
- [ ] すべての品質チェックが通過
- [ ] README.md が更新されている

### 推奨 (Should Have)
- [ ] カバレッジが ≥90%
- [ ] MCP 統合テストが実装されている
- [ ] User Guide が作成されている
- [ ] セキュリティスキャンが完了している

### オプション (Nice to Have)
- [ ] パフォーマンステストが追加されている
- [ ] CI/CD パイプラインの設定が開始されている

---

## 📊 進捗トラッキング

セッション中に進捗を記録してください:

```markdown
## Progress Tracker

### Step 1: 基本統合テスト (30分)
- [ ] Phase 2 tests: 74 passed
- [ ] Phase 3 tests: 62 passed
- [ ] All semantic tests: 136 passed

### Step 2: 手動統合テスト (15分)
- [ ] Memory addition works
- [ ] Graph generation works (3 formats)
- [ ] No errors

### Step 3: 統合テストコード実装 (2-3時間)
- [ ] test_phase2_phase3_integration.py created
- [ ] test_e2e_workflow.py created
- [ ] test_mcp_integration.py created
- [ ] All integration tests passing

### Step 4: 品質保証チェック (1-2時間)
- [ ] mypy: Pass
- [ ] ruff: Pass
- [ ] radon: Pass
- [ ] Coverage: ≥88%
- [ ] Security scan: Pass

### Step 5: ドキュメント更新 (1-2時間)
- [ ] README.md updated
- [ ] User Guide created
- [ ] MCP docs completed

### Step 6: 最終確認 (30分)
- [ ] All tests passing
- [ ] Quality checks passing
- [ ] Changes committed
```

---

## 🚨 トラブルシューティング

### Issue: テストが失敗する

**Solution**: エラーメッセージを確認し、該当モジュールをデバッグ
```bash
pytest tests/integration/test_phase2_phase3_integration.py::test_extract_and_ask -v --tb=long
```

### Issue: カバレッジが目標に達しない

**Solution**: 未カバー部分を特定し、テストを追加
```bash
pytest --cov=clauxton/semantic/memory_qa --cov-report=term-missing
```

### Issue: パフォーマンスが遅い

**Solution**: プロファイリングを実行
```bash
python -m cProfile -o profile.stats -m pytest tests/integration/
python -m pstats profile.stats
```

---

## 📚 参考ドキュメント

- **統合テスト計画**: `docs/PHASE3_INTEGRATION_TEST_PLAN.md`
- **品質保証計画**: `docs/PHASE3_QUALITY_ASSURANCE_PLAN.md`
- **Phase 2-4 実行計画**: `docs/v0.15.0_PHASE2-4_EXECUTION_PLAN.md`
- **Code Style**: `CLAUDE.md`

---

## 🎉 成功！

すべての Step が完了したら:

1. 最終コミットを作成
2. 進捗を記録
3. Phase 4 (UX Polish) への移行を検討
4. または v0.15.0 リリース準備を開始

---

**Last Updated**: 2025-11-03
**Status**: ✅ Ready to Execute
**Next Action**: 新しい Claude Code セッションで Step 1 から開始
