"""Shared types for CLI environment discovery and configuration.

This module isolates shared dataclasses to prevent circular import dependencies
between env.py and extracted modules.
"""

from __future__ import annotations

from dataclasses import dataclass, field
from typing import Dict

from atlas.sdk.discovery import Candidate


# Constants used across env.py and related modules
DISCOVERY_FILENAME = "discover.json"
GENERATED_CONFIG_FILENAME = "generated_config.yaml"
VALIDATION_MARKER_FILENAME = ".validated"


@dataclass(slots=True)
class TargetSpec:
    """Specification for an environment or agent target discovered during onboarding.

    Attributes:
        candidate: Discovered candidate from AST analysis
        factory: Tuple of (module, function_name) for factory function
        kwargs: Default keyword arguments for factory/constructor
        config: Additional configuration metadata
        auto_wrapped: True if this was auto-generated by LLM synthesis
        metadata: Runtime metadata collected during discovery
    """
    candidate: Candidate | None = None
    factory: tuple[str, str] | None = None
    kwargs: Dict[str, object] = field(default_factory=dict)
    config: dict[str, object] | None = None
    auto_wrapped: bool = False
    metadata: dict[str, object] | None = None

    def dotted_path(self) -> str:
        """Return human-readable dotted path for this target."""
        if self.candidate is not None:
            return self.candidate.dotted_path()
        if self.factory is not None:
            return f"{self.factory[0]}:{self.factory[1]}"
        return "<unspecified>"


@dataclass(slots=True)
class SelectedTargets:
    """Pair of environment and agent targets selected for Atlas runtime."""
    environment: TargetSpec
    agent: TargetSpec
