WMStats.namespace("_StructBase");

WMStats._StructBase = function() {
    this._data = null;
};

WMStats._StructBase.prototype = {

    getData: function() {
        return this._data;
    },
    
    setData: function(couchData) {
        this._data = this.convertCouchData(couchData);
    },
    
    getDataByKey: function(key, combineFunc) {
        var data = {};
        var dataList = this._data;
        for (var i in dataList) {
            if (data[dataList[i][key]] === undefined) {
                data[dataList[i][key]] = dataList[i];
            } else if (combineFuc === undefined) {
                data[dataList[i][key]] = dataList[i];
            } else {
                data[dataList[i][key]] = combineFunc(data[dataList[i][key]], dataList[i]);
            }
            
        }
        return data;
    },
    
    convertCouchData: function(couchData) {
        // defaut conversion need to be over written
        return couchData;
    }
};
WMStats.namespace("GenericRequests");
WMStats.namespace("GenericRequestsSummary");
WMStats.namespace("RequestsByKey");

WMStats.GenericRequestsSummary = function (summaryStruct) {
    
    this._get = WMStats.Utils.get;
       
    this.summaryStruct = {length: 0};
    //default job status structure
    this.jobStatus = {
             success: 0,
             canceled: 0,
             transition: 0,
             queued: {first: 0, retry: 0},
             submitted: {first: 0, retry: 0},
             submitted: {pending: 0, running: 0},
             failure: {create: 0, submit: 0, exception: 0},
             cooloff: {create: 0, submit: 0, job: 0},
             paused: {create: 0, submit: 0, job: 0},
         };
    

    if (summaryStruct) {
        this.summaryStruct = WMStats.Utils.cloneObj(summaryStruct);
    };
};

WMStats.GenericRequestsSummary.prototype = {
    
    getJobStatus: function(statusStr) {
        return WMStats.Utils.get(this.jobStatus, statusStr, 0);
    },
    
    getAvgProgressSummary: function (doc) {
        
		var progressStat = {};
        var datasets = {};
        var numDataset = 0;
        if (doc.outputdatasets) {
        	numDataset = doc.outputdatasets.length;	
        }
        
        for (var task in doc.tasks) {
            for(var site in doc.tasks[task].sites) {
                for (var outputDS in doc.tasks[task].sites[site].dataset) {
                    if (datasets[outputDS] === undefined) {
                        if (!doc.outputdatasets) {
                        	//if outputdatasets is not defined calcuate from frwj
                        	numDataset += 1;
                        }
                        datasets[outputDS] = true;
                    }
                    WMStats.Utils.updateObj(progressStat, doc.tasks[task].sites[site].dataset[outputDS]);
                }
             }
        }
        for (var prop in progressStat) {
            progressStat[prop] = progressStat[prop] / numDataset;
        }
        progressStat.numDataset = numDataset;
        return progressStat;
    },
    
    getAvgEvents: function() {
        // handle legacy event calculation
        if (this.summaryStruct.progress === undefined || this.summaryStruct.progress.events === undefined) {
            return this.summaryStruct.processedEvents;
        } else {
            return this.summaryStruct.progress.events;
        }
    },
    
    getAvgLumis: function() {
        // handle legacy event calculation
        if (this.summaryStruct.progress.totalLumis === undefined) {
            return 0;
        } else {
            return this.summaryStruct.progress.totalLumis;
        }
    },
    
    getSummary: function(){
        return this.summaryStruct;
    },
    
    summaryStructUpdateFuction: null,
    
    update: function(summary) {
        WMStats.Utils.updateObj(this.summaryStruct, summary.summaryStruct, true, 
                                this.summaryStructUpdateFuction);
        WMStats.Utils.updateObj(this.jobStatus, summary.jobStatus);
    },
    
    updateFromRequestDoc: function(doc) {
         var summary = this.createSummaryFromRequestDoc(doc);
         this.update(summary);
    },
    
    getWMBSTotalJobs: function() {
        return (this.getJobStatus("success") +
                this.getJobStatus("canceled") +
                this.getJobStatus( "transition") +
                this.getTotalFailure() +
                this.getTotalCooloff() +
                this.getTotalPaused() +
                this.getTotalQueued() +
                this.getRunning() +
                this.getPending());
    },
    
    getTotalFailure: function() {
        return (this.getJobStatus("failure.create") + 
                this.getJobStatus("failure.submit") + 
                this.getJobStatus("failure.exception"));
    },
    
    getTotalSubmitted: function() {
        return (this.getJobStatus("submitted.first") + 
                this.getJobStatus("submitted.retry"));
    },

    getRunning: function() {
        return this.getJobStatus("submitted.running");
    },
    
    getPending: function() {
        return this.getJobStatus("submitted.pending");
    },
    getTotalCooloff:function() {
        return (this.getJobStatus("cooloff.create") + 
                this.getJobStatus("cooloff.submit") + 
                this.getJobStatus("cooloff.job"));
    },

    getTotalPaused: function() {
        return (this.getJobStatus("paused.create") + 
                this.getJobStatus("paused.submit") + 
                this.getJobStatus("paused.job"));
    },
    
    getTotalQueued: function() {
        return (this.getJobStatus("queued.first") + 
                this.getJobStatus("queued.retry"));
    },
    
    createSummaryFromRequestDoc: function(doc) {
        //THis is just interface which update summarySturct property
        var summary = WMStats.RequestsSummary();
        summary.summaryStruct.length = 1;
        summary.jobStatus = this._get(doc, 'status', {});
        return summary;
    }
};

WMStats.RequestStruct = function(requestName) {
    this._workflow = requestName;
    this._summary = WMStats.RequestsSummary();
    // number of requests in the data
	this._addJobs = WMStats.Utils.updateObj;
};

WMStats.RequestStruct.prototype = {
    
    getProgressStat: function () {
        var progressStat = {};
        for (var task in this.tasks) {
            for(var site in this.tasks[task].sites) {
                WMStats.Utils.updateObj(progressStat, this.tasks[task].sites[site].dataset);
            }
        }
        return progressStat;
    },

    getName: function() {
        return this._workflow;
    },
    
    getSummary: function() {
        return this._summary.createSummaryFromRequestDoc(this);
    },
    
    getTasks: function() {
    	return new WMStats.Tasks(this.tasks, this._workflow);
    },
    
    getLastState: function() {
        if (this.request_status){
            return this.request_status[this.request_status.length -1].status;
        };
        return null;
    },
    
    getLastStateAndTime: function() {
        if (this.request_status){
            return this.request_status[this.request_status.length -1];
        };
        return null;
    },
    
    getSkippedStatus: function () {
    	if (this.skipped) {
    		return this.skipped;
    	} else {
    		return false;
    	}
    },
    
    getSkippedDetail: function(fullTaskName) {
    	var skippedTasks = {};
    	if (this.skipped) {
    		for (var task in this.tasks) {
    			if (this.tasks[task].skipped) {
    				var taskName = task;
    				if (!fullTaskName){
    					var taskList = task.split('/');
    					taskName = taskList[taskList.length - 1];
    				}
    				skippedTasks[taskName] = this.tasks[task].skipped;
        		}
        	}
        }
        return skippedTasks;
    },
    
    updateFromCouchDoc: function (doc) {
        
        function _tasksUpdateFunction (baseObj, addObj, field) {
            if (field === "JobType") {
                baseObj[field] = addObj[field];
            } else if (field === "updated"){
                baseObj[field] = Math.max(baseObj[field], addObj[field]);
            } else {
                baseObj[field] += addObj[field];
            }
        }; 
        
        for (var field in doc) {
            //handles when request is split in more than one agents
            if (field == "AgentJobInfo") {
            	//skipping AgentJobInfo field. - added to handle newer ajax call to wmstats server
            	continue;
            } else if (this[field] && 
                (field == 'sites' || field == 'status')){
                this._addJobs(this[field], doc[field]);
            } else if (this[field] && field == 'tasks'){
            	//Also task['skipped'] value will be updated here
                this._addJobs(this[field], doc[field], true,  _tasksUpdateFunction);
            
            } else if (this[field] && field == 'output_progress') {
                var outProgress = this.output_progress;
                for (var index in outProgress){
                    for (var prop in doc[field][index]) {
                        outProgress[index][prop] += doc[field][index][prop];
                        //TODO: need combine dataset separately
                    }
                }
            } else if (this[field] && field == 'skipped') {
                this[field] = this[field] || doc[field]; 
            } else if (field == 'agent_url') {
                if (this[field] === undefined) this[field] = [];
                WMStats.Utils.addToSet(this[field], doc[field]);
            
            } else {
                this[field] = doc[field];
            }
        }
    }
};

WMStats.GenericRequests = function (data) {
    /*
     * Data structure for holding the request
     * it handles 3 types (not very robust and modular)
     * TODO: remove dependencies from different data type. (tier0, reqmgr)
     * if possible
     * reqmgr_request, agent_request, tier0_request.
     */
    // request data by workflow name
    this._dataByWorkflow = {};
    // request data by agent - only contains information from agent
    // i.e. job status.
    this._dataByWorkflowAgent = {};
    this._get = WMStats.Utils.get;
    this._filter = {};
    this._filteredRequests = null;
    if (data !== undefined) {
    	this.setFromRawData(data);
    }
};

WMStats.GenericRequests.prototype = {

    _mapProperty: function (workflowData, property) {
        if (property == 'request_status') {
            return workflowData[property][workflowData[property].length - 1].status;
        }
        if (property == 'inputdataset') {
            return WMStats.Utils.getInputDatasets(workflowData);
        }
        return workflowData[property];
    },
    
    _getRequestObj: function (request) {
        if (typeof(request) == "string") {
            return this.getData(request);
        } else {
            return request;
        }
    },
    
    _getStatusObj: function(request, level) {
        //level could be site, task, or request;
        var requestObj = this._getRequestObj(request);
        if (level == "task"){
            return requestObj.tasks.status;
        } else if (level == "site") {
            return requestObj.sites;
        } else {
            return requestObj.status;
        }
    },

    _requestDateSort: function(a, b) {
        for (var i in a.request_date) { 
            if (b.request_date[i] != a.request_date[i]) {
                return (Number(b.request_date[i]) - Number(a.request_date[i]));
            }
        }
        return 0;
    },

   _andFilter: function(base, filter) {
        var includeFlag = true;
        for (var property in filter) {
            if (!filter[property]) {
                continue;
            }else if (this._mapProperty(base, property) !== undefined &&
               this._contains(this._mapProperty(base, property), filter[property])) {
                continue;
            } else {
                includeFlag = false;
                break;
            }
        }
        return includeFlag;
    },
    
    _contains: function(a, b) {
        //TODO change to regular expression or handle numbers
        if ((typeof a) === "string") return (a.toLowerCase().indexOf(b.toLowerCase()) !== -1);
        else if ((typeof a) == "number") return (Number(b) == a);
        else if (a instanceof Array) {
            for (var i in a) {
                if (this._contains(a[i], b)) return true;
            }
            return false;
        } else {
            alert("value need to be either number or string");
        }
    },
        
    getProgressStat: function (request) {
        var requestObj = this._getRequestObj(request);
        return requestObj.getProgressStat();
    },
    
    getFilter: function() {
        return this._filter;
    },
    
    setFilter: function(filter) {
        this._filter = filter;
    },
    
    updateRequest: function(doc) {
        /*
         * 
         */
        var doc = WMStats.Globals.convertRequestDocToWMStatsFormat(doc);
        var workflow = doc.workflow;
        var agentURL = doc.agent_url;
        
        if (workflow && !this._dataByWorkflow[workflow]) {
            this._dataByWorkflow[workflow] = new WMStats.RequestStruct(workflow);;
        }
        
        if (agentURL && !this._dataByWorkflowAgent[workflow]) {
            this._dataByWorkflowAgent[workflow] = {};
        }
        //if it is new agent create one.
        if (agentURL && !this._dataByWorkflowAgent[workflow][agentURL]){
            this._dataByWorkflowAgent[workflow][agentURL] = new WMStats.RequestStruct(workflow);
        }
        
        // update both _databyWorkflow
        this.getData(workflow).updateFromCouchDoc(doc);
        if (agentURL) {
            this.getData(workflow, agentURL).updateFromCouchDoc(doc);
        }
    },
    
    updateBulkRequests: function(docList) {
        for (var row in docList) {
            //not sure why there is null case
            if (docList[row].doc) {
                this.updateRequest(docList[row].doc);
            }
        }
    },
    
    updateRequestFromWMStatsServer: function(doc) {
    	
    	var doc = WMStats.Globals.convertRequestDocToWMStatsFormat(doc);
        var workflow = doc.workflow;
        
        if (workflow && !this._dataByWorkflow[workflow]) {
            this._dataByWorkflow[workflow] = new WMStats.RequestStruct(workflow);
            this._dataByWorkflow[workflow].updateFromCouchDoc(doc);
        };
        
        if (doc.AgentJobInfo) {
        	
        	for (var agentURL in doc.AgentJobInfo) {
        		if (agentURL && !this._dataByWorkflowAgent[workflow]) {
            		this._dataByWorkflowAgent[workflow] = {};
        		};
        		this._dataByWorkflowAgent[workflow][agentURL] = new WMStats.RequestStruct(workflow);
        		this._dataByWorkflowAgent[workflow][agentURL].updateFromCouchDoc(doc.AgentJobInfo[agentURL]);
        		// legacy format which need to be updated
        		doc.AgentJobInfo[agentURL].agent_url = agentURL;
        		this._dataByWorkflow[workflow].updateFromCouchDoc(doc.AgentJobInfo[agentURL]);
        	};
        };

    },
    
    setFromRawData: function(data) {
    	/*   {"result": [
                                 * 	{"sryu_ReReco_reqmgr2_validation_150717_180724_9878": 
                                 *   {"InputDataset": "/QDTojWinc_NC_M-1200_TuneZ2star_8TeV-madgraph/Summer12pLHE-DMWM_Validation_DONOTDELETE_Alan_TEST-v1/GEN", 
                                 *    "Group": "DATAOPS", "CustodialSites": [], "OpenRunningTimeout": 1800, 
                                 *    "Comments": "MCFromGEN LumiBased splitting with 1l per job. Half an hour opened", 
                                 *    "Requestor": "sryu", "ProcessingString": "START53_V7C", "ScramArch": "slc6_amd64_gcc472", 
                                 *    "SizePerEvent": 1154, "ConfigCacheID": "1ad063a0d73c1d81143b4182cbf84793", "Memory": 2300, 
                                 *    "RunBlacklist": [], "PrepID": "B2G-Summer12-00736", "AutoApproveSubscriptionSites": [], 
                                 *    "BlockBlacklist": [], "BlockWhitelist": [], "CustodialSubType": "Move", 
                                 *    "RequestType": "ReReco", "TimePerEvent": 16.87
                                 *    "OutputDatasets": ["/QDTojWinc_NC_M-1200_TuneZ2star_8TeV-madgraph/Integ_Test-ReReco_SRYU_pnn-v1/GEN-SIM"], 
                                 *    "LumisPerJob": 1, "SoftwareVersions": ["CMSSW_5_3_19"], 
                                 *    "AcquisitionEra": "Integ_Test", "PrimaryDataset": "QDTojWinc_NC_M-1200_TuneZ2star_8TeV-madgraph", 
                                 *    "CouchDBName": "reqmgr_config_cache", "CMSSWVersion": "CMSSW_5_3_19", "NonCustodialSites": [], 
                                 *    "RequestSizeFiles": 0, "CouchWorkloadDBName": "reqmgr_workload_cache", "RequestPriority": 90000, 
                                 *    "SiteWhitelist": ["T1_US_FNAL", "T2_CH_CERN"], 
                                 *    "SubscriptionPriority": "Low", "ProcessingVersion": "1", "Team": "testbed-dev",
                                 *    "SplittingAlgo": "LumiBased", "TotalEstimatedJobs": 100, 
                                 *    "RequestTransition": [{"Status": "new", "DN": null, "UpdateTime": 1437149248}, 
                                 *                          {"Status": "assignment-approved", "DN": null, "UpdateTime": 1437149248}, 
                                 *                          {"Status": "assigned", "DN": null, "UpdateTime": 1437149249}, 
                                 *                          {"Status": "acquired", "DN": null, "UpdateTime": 1437150308}, 
                                 *                          {"Status": "running-open", "DN": null, "UpdateTime": 1437152703}, 
                                 *                          {"Status": "running-closed", "DN": null, "UpdateTime": 1437152709}, 
                                 *                          {"Status": "completed", "DN": null, "UpdateTime": 1437227104}], 
                                 *    "RequestName": "sryu_ReReco_reqmgr2_validation_150717_180724_9878", 
                                 *    "RequestString": "ReReco_reqmgr2_validation", 
                                 *    "InputDatasets": ["/QDTojWinc_NC_M-1200_TuneZ2star_8TeV-madgraph/Summer12pLHE-DMWM_Validation_DONOTDELETE_Alan_TEST-v1/GEN"], 
                                 *    "CouchURL": "https://reqmgr2-dev.cern.ch/couchdb", "TotalTime": 28800, 
                                 *    "RequestorDN": "....", 
                                 *    "RequestWorkflow": "https://reqmgr2-dev.cern.ch/couchdb/reqmgr_workload_cache/sryu_ReReco_reqmgr2_validation_150717_180724_9878/spec", "
                                 *    "Campaign": "Agent108_Validation", "GlobalTag": "START53_V7C::All", "RunWhitelist": [], 
                                 *    "FilterEfficiency": 1, "DbsUrl": "https://cmsweb-testbed.cern.ch/dbs/int/global/DBSReader", 
                                 *    "TotalInputLumis": 100, "RequestDate": [2015, 7, 17, 16, 7, 24], "NonCustodialSubType": "Replica", 
                                 *    "TotalInputFiles": 1, "SiteBlacklist": [], "TotalInputEvents": 2500, 
                                 *    "ConfigCacheUrl": "https://cmsweb.cern.ch/couchdb", 
                                 *    "_id": "sryu_ReReco_reqmgr2_validation_150717_180724_9878", 
                                 *    "RequestStatus": "completed", 
                                 *    "RequestNumEvents": 50000, 
                                 *    "AgentJobInfo": {"vocms008.cern.ch:9999": 
                                 *                       {"status": {"success": 107}, 
                                 *                        "agent_team": "testbed-dev", 
                                 *                        "workflow": "sryu_ReReco_reqmgr2_validation_150717_180724_9878", 
                                 *                        "timestamp": 1437498506, 
                                 *                        "_rev": "1-d792c9d73285ff1318d7f5c0e2b2f486", 
                                 *                        "sites": {"T2_CH_CERN": {"success": 107}}, 
                                 *                        "agent": "WMAgent", 
                                 *                        "tasks": {"/sryu_ReReco_reqmgr2_validation_150717_180724_9878/ReReco/ReRecoMergeRAWSIMoutput/ReRecoRAWSIMoutputMergeLogCollect": 
                                 *                                     {"status": {"success": 1}, 
                                 *                                      "sites": {"T2_CH_CERN": 
                                 *                                                  {"inputEvents": 0, 
                                 *                                                   "cmsRunCPUPerformance": {"totalJobCPU": 0, "totalJobTime": 0, "totalEventCPU": 0}, 
                                 *                                                   "wrappedTotalJobTime": 6, "success": 1, "dataset": {}}}}, 
                                 *                                   "/sryu_ReReco_reqmgr2_validation_150717_180724_9878/ReReco/LogCollect": 
                                 *                                      {"status": {"success": 1}, 
                                 *                                       "sites": .....}}}, ....}, 
                                 *                        "agent_url": "vocms008.cern.ch:9999", 
                                 *                        "_id": "98e5e3643f1c6575407b6de04bd6619a", 
                                 *                        "type": "agent_request"}},
                                 * "sryu_ReReco_reqmgr2_validation_150721_194418_5448": 
                                 *   {"InputDataset": .....
                                 *   }}]
                                 * }
                                 */
    	if (data.result.length == 1) {
    		for (var req in data.result[0]) {
    			this.updateRequestFromWMStatsServer(data.result[0][req]);
    		}
    	}
    },

    filterRequests: function(filter) {
        var requestData = this.getData();
        var filteredData = {};
        var requestWithAgentData = this.getDataWithAgent();
        var filteredDataWithAgent = {};
        if (filter === undefined) {filter = this._filter;}
        for (var workflowName in requestData) {
            if (this._andFilter(requestData[workflowName], filter)){
                filteredData[workflowName] =  requestData[workflowName];
                filteredDataWithAgent[workflowName] =  requestWithAgentData[workflowName];
            }
        }
        this._filteredRequests = WMStats.Requests();
        this._filteredRequests.setDataByWorkflow(filteredData, filteredDataWithAgent);
        return this._filteredRequests;
    },

    getKeyValue: function(request, keyString, defaultVal) {
        //keyString is opject property separte by '.'
        return this._get(this._dataByWorkflow[request], keyString, defaultVal);
    },
    
    getData: function(workflow, agentURL) {
        if (workflow && (agentURL === "all" || agentURL === "NA" )) {
            return this._dataByWorkflowAgent[workflow];
        } else if (workflow && agentURL) {
            return this._dataByWorkflowAgent[workflow][agentURL];
        } else if (workflow){
            return this._dataByWorkflow[workflow];
        } else{
            return this._dataByWorkflow;
        }
    },
    
    getDataWithAgent: function(workflow, agentURL) {
        if (workflow && (agentURL === "all" || agentURL === "NA" )) {
            return this._dataByWorkflowAgent[workflow];
        } else if (workflow && agentURL) {
            return this._dataByWorkflowAgent[workflow][agentURL];
        } else{
            return this._dataByWorkflowAgent;
        }
    },
    
    getFilteredRequests: function() {
        return this._filteredRequests;
    },
    
    setDataByWorkflow: function(data, agentData) {
        //keyString is opject property separte by '.'
        this._dataByWorkflow = data;
        this._dataByWorkflowAgent = agentData;
    },
    
    getList: function(sortFunc) {
        var list = [];
        for (var request in this.getData()) {
            list.push(this.getData(request));
        }
        if (sortFunc) {
            return list.sort(sortFunc);
        } else {
            return list.sort(this._requestDateSort);
        }
    },
    
    getRequestNames: function() {
        var list = [];
        for (var request in this.getData()) {
            list.push(request);
        }
        return list;
    },

    getSummary: function(workflow, agentURL) {
        
        var requests = this.getData(workflow, agentURL);
        if (workflow) {
            return requests.getSummary();
        } else {
            var summary =  WMStats.RequestsSummary();
            //TODO need to cache the information
            for (var requestName in requests) {
                summary.update(this.getData(requestName).getSummary());
            }
            return summary;
        }
    },
    
    getAlertRequests: function() {
        var alertRequests = [];
        for (var workflow in this.getData()) {
            var reqSummary = this.getSummary(workflow);
            var coolOff = reqSummary.getTotalCooloff();
            var paused = reqSummary.getTotalPaused();
            if (coolOff > 0 || paused > 0) {
                alertRequests.push(this.getData(workflow));
            }
        }
        return alertRequests;
    },
    
    getRequestStatusAndTime: function(workflowName) {
        var workflowData = this._dataByWorkflow[workflowName];
        return  workflowData["request_status"][workflowData["request_status"].length - 1];
    },
    
    getTasks: function(workflowName) {
    	return this._dataByWorkflow[workflowName].getTasks();
    }
    
};

WMStats.RequestsByKey = function (category, summaryFunc) {
    
    var _data = {};
    var _category = category;
    var _get = WMStats.Utils.get;
    
    function categorize(requestData) {
        
        function _getRequestData(workflow, agentURL){
            if (_category === "agent" && agentURL !== "all" && agentURL !== "NA" ) {
                return requestData.getData(workflow, agentURL);
            } else {
                return requestData.getData(workflow);
            }
        }
        
        function _getCategoryKey(workflow){
            if (_category === "agent") {
                var agentCategory = requestData.getData(workflow, "all");
                if (agentCategory === undefined) {
                    return "NA";
                } else {
                    return agentCategory;
                }
            } else {
                return requestData.getKeyValue(workflow, _category, "NA");
            }
        }
        function _updateData(key, workflow, summaryBase) {
            if (_data[key] === undefined) {
                //initial set up
                _data[key] = {};
                _data[key].requests = {};
                _data[key].summary =  summaryFunc();
                _data[key].key = key;
            }
            var requestInfo = _getRequestData(workflow, key);
            _data[key].requests[workflow] = requestInfo;
            _data[key].summary.updateFromRequestDoc(summaryBase);
        };
        
        var dataByWorkflow = requestData.getData();
        for (var workflow in dataByWorkflow) {
            var key = _getCategoryKey(workflow);
            if (typeof key == 'object') {
                if (key.length) {
                    // handles array case
                    for (var index in key) {
                        _updateData(key[index], workflow, requestData.getData(workflow));
                    }
                } else {
                    // handles agent, sites and tasks case
                    for (var prop in key) {
                        _updateData(prop, workflow, key[prop]);
                    }
                }
                
            } else {
                if (key == "NA" && _category == "sites" || _category == "tasks" || _category == "agent") {
                    // summary base shouldn't be higher level. since sites and tasks
                    // has sub hierarchy
                    _updateData(key, workflow, {});
                } else {
                    _updateData(key, workflow, requestData.getData(workflow));
                }
            }
            
        }
    };
    
    function getData(key){
        if (key === undefined) {
            return _data;
        } else {
            return _data[key];
        }
    };
    
    function getRequestData(key){
        var requestData = WMStats.Requests();
        if (_data[key] !== undefined) {
            requestData.setDataByWorkflow(_data[key].requests);
        }
        return requestData;
    };
    
    function getList(sortFunc) {
        var list = [];
        for (var key in _data) {
            list.push(_data[key]);
        }
        if (sortFunc) {
            return list.sort(sortFunc);
        } else {
            return list;
        }
    };
    
    return {
        categorize: categorize,
        getData: getData,
        getRequestData: getRequestData,
        category: _category,
        getList: getList
    };
};
WMStats.namespace("Tasks");
WMStats.namespace("TaskStruck");
WMStats.namespace("TasksSummary");

WMStats.TasksSummary = function(task) {
    var tier1Summary = {totalEvents: 0,
                        processedEvents: 0};
    if (task) {
        var taskSummaryStruct = task.subscription_status || {};
    } else {
        var taskSummaryStruct = {subscription_status: {finished: 0, open:0, total:0}};
    };
    
    
    var taskSummary = new WMStats.GenericRequestsSummary(taskSummaryStruct);
    
    taskSummary.getAvgProgressSummary = function () {
        
        var progressStat = {};
        var datasets = {};
        var numDataset = 0;
        
        for(var site in task.sites) {
            for (var outputDS in task.sites[site].dataset) {
                if (datasets[outputDS] === undefined) {
                     numDataset += 1;
                     datasets[outputDS] = true;
                }
                WMStats.Utils.updateObj(progressStat, task.sites[site].dataset[outputDS]);
            }
         }

        for (var prop in progressStat) {
            progressStat[prop] = progressStat[prop] / numDataset;
        }
        progressStat.numDataset = numDataset;
        return progressStat;
    };
    
    taskSummary.summaryStruct.progress = taskSummary.getAvgProgressSummary();
    
    taskSummary.jobStatus = task.status;
    
    
    
    return taskSummary;
};

WMStats.TaskStruct = function(taskName, task) {
    this._taskName = taskName;
    this._task = task;
    this._summary = WMStats.TasksSummary(task);
    // number of requests in the data
	this._addJobs = WMStats.Utils.updateObj;
};

WMStats.TaskStruct.prototype = {
   
   getProgressStat: function () {
        var progressStat = {};
        for(var site in this._task.sites) {
        	WMStats.Utils.updateObj(progressStat, this._task.sites[site].dataset);
        }
        return progressStat;
    },
    
    getName: function() {
        return this._taskName;
    },
    
    getSummary: function() {
        return this._summary;
    }
};

WMStats.Tasks = function (tasks, requestName) {
    /*
     * Data structure for holding the tasks.
     * tasks structure is 
     * {taskName: {'status': {'success: 0, running:10},
     *             'site': {'T2_US_FNAL': {'success: 0, running:10
     *             'jobtype': 'Processing',
     *             'subscription_status': {finished: 0, open:0, total:0}
     *       }}}
     */
    // request data by workflow name
    var taskStructDict = {};
    for (var taskName in tasks) {
    	taskStructDict[taskName] = new WMStats.TaskStruct(taskName, tasks[taskName]);
    }
 
    this._data = taskStructDict;
    this._workflow = requestName;
};

WMStats.Tasks.prototype = {
	
	getWorkflow: function() {
		return this._workflow;
	},
    getData: function () {return this._data;},
    			
    getSummary: function(taskName) {
    	// TODO need to add summary case for taskName undefined.
        return this._data[taskName].getSummary();
    }
};
WMStats.namespace("Agents");

WMStats.Agents = function (couchData) {
    
    var agentData = new WMStats._StructBase();
    agentData.agentNumber = {error: 0, stable:0};
    
    agentData.convertCouchData = function(data) {
                                     var dataRows = data.rows;
                                     var rows = [];
                                     for (var i in dataRows) {
                                         var tableRow = dataRows[i].value;
                                         rows.push(tableRow);
                                     };
                                     return rows;
                                };
    if (couchData) agentData.setData(couchData);
    
    agentData.getAlertList = function(){
        var currentTime = Math.round(new Date().getTime() / 1000);
        var dataList = this.getData();
        var agentPollingCycle = 600;
        agentData.agentNumber = {error: 0, warning:0, stable:0, drain: 0};
        
        function getStatus(agentInfo) {
            var lastUpdatedDuration = currentTime - agentInfo.timestamp;
            var dataUpdateDuration = -1; 
            if (agentInfo.data_last_update) {
                var dataUpdateDuration = currentTime - agentInfo.data_last_update; 
            }
            var report = {"status": "stable", "message": ""};
            
            // drain case            
            if (agentInfo.drain_mode) {
                report.status = "drain";
                report.message += "Draining Agent; ";
            }

            if (agentInfo.proxy_warning && agentInfo.status === "warning") {
                report.status = agentInfo.status;
                report.message += "Proxy expiration warning; ";
            } else if (agentInfo.proxy_warning && agentInfo.status === "error") {
                agentData.agentNumber.error += 1;
                report.status = agentInfo.status;
                report.message += "Proxy expiration error; ";
            }
            
            // warning case
            if (agentInfo.couch_process_warning) {
                agentData.agentNumber.error += 1;
                report.status = "error";
                report.message += "couchdb process maxed out: " + agentInfo.couch_process_warning;
            }
            
            if (agentInfo.disk_warning && (agentInfo.disk_warning.length > 0)) {
                agentData.agentNumber.warning += 1;
                report.status = "warning";
                report.message += "disk is almost full; ";
            }

            if (lastUpdatedDuration > agentPollingCycle * 2) {
                agentData.agentNumber.error += 1;
                report.status = "error"; 
                report.message += "Agent Data is not updated: AgentStatusWatcher is Down; ";
            }
            
            if (agentInfo.down_components.length > 0) {
                agentData.agentNumber.error += 1;
                report.status = "error";
                report.message += "Components or Thread down; ";
            }
            
            if (agentInfo.data_error && (agentInfo.data_error !== "ok")) {
                agentData.agentNumber.error += 1;
                report.status = "error";
                report.message += "Data collect error; ";
            }
            // If there is no drain/warning or errors set the message OK
            if (report.status === "stable") {
                agentData.agentNumber.stable += 1;
                report.message = "OK";
            };
            
            report["agent_update"] =  WMStats.Utils.formatDuration(lastUpdatedDuration);
            report["data_update"] = WMStats.Utils.formatDuration(dataUpdateDuration);
            return report;
        };
        
        for (var index in dataList) {
            dataList[index]['alert'] = getStatus(dataList[index]);
        };
        return dataList;
    };
    // initial calculation
    agentData.getAlertList();
    
    return agentData;
};
WMStats.namespace("Sites");

WMStats.Sites = function (couchData) {
    var _data;
    var baseColumns = ["timestamp", "site", "agent_url"];
    
    var siteData = new WMStats._StructBase();
    
    siteData.convertCouchData = function(data) {
                                    var dataRows = data.rows;
                                    var rows =[];
                                    for (var i in dataRows) {
                                        var tableRow = dataRows[i].value;
                                        for (var j = 0; j < baseColumns.length; j ++) {
                                            tableRow[baseColumns[j]] = dataRows[i].key[j];
                                        }
                                        rows.push(tableRow);
                                    }
                                    return rows;
                               };
    
    if (couchData) siteData.setData(couchData);
    
    return siteData;
};
WMStats.namespace("JobSummary");

WMStats.JobSummary = function (couchData) {
    
    var jobSummaryData = new WMStats._StructBase();
    
    jobSummaryData.convertCouchData = function(data) {
                                        var jobSummary = {};
                                        jobSummary.status = [];
                                        for (var i in data.rows){
                                            jobSummary.workflow = data.rows[i].key[0];
                                            var statusSummary = {};
                                            statusSummary.task = data.rows[i].key[1];
                                            statusSummary.status = data.rows[i].key[2];
                                            statusSummary.exitCode = data.rows[i].key[3];
                                            statusSummary.site = data.rows[i].key[4];
                                            if (typeof(statusSummary.site) === "object") {
                                                statusSummary.site = "{}";
                                            }
                                            statusSummary.acdcURL = data.rows[i].key[5];
                                            statusSummary.agentName = data.rows[i].key[6];
                                            statusSummary.errorMsg = data.rows[i].key[7];
                                            statusSummary.count = data.rows[i].value;
                                            jobSummary.status.push(statusSummary);
                                        }
                                        return jobSummary;
                                   };
    
    if (couchData) jobSummaryData.setData(couchData);
    
    return jobSummaryData;
};
WMStats.namespace("Campaigns");

WMStats.Campaigns = function (couchData) {
    
   var campaignData = new WMStats._StructBase();
   var _baseColumns = ["campaign"];
    
   campaignData.convertCouchData =  function(data) {
                                        dataRows = data.rows;
                                         //TODO sync with group level
                                        //This is base on the group_level ["campaign", "team", type]
                                        var rows =[];
                                        for (var i in dataRows) {
                                            var tableRow = dataRows[i].value;
                                            for (var j = 0; j < _baseColumns.length; j ++) {
                                                tableRow[_baseColumns[j]] = dataRows[i].key[j];
                                            }
                                            rows.push(tableRow);
                                        }
                                        return rows;
                                   };
    if (couchData) campaignData.setData(couchData);
    
    return campaignData;
};
WMStats.namespace("Alerts");

WMStats.Alerts = function (couchData) {
    
    var alertData = new WMStats._StructBase();
    
    alertData.convertCouchData =   function(data) {
                                        var dataRows = data.rows;
                                        var rows =[];
                                        for (var i in dataRows) {
                                            var tableRow = {};
                                            tableRow.workflow = dataRows[i].key;
                                            tableRow.count = dataRows[i].value;
                                            rows.push(tableRow);
                                        }
                                        return rows;
                                   };
    if (couchData) alertData.setData(couchData);
    
    return alertData;
};
WMStats.namespace("SiteSummary");

WMStats.SiteSummary = function() {
    var siteSummary = {totalEvents: 0,
                       processedEvents: 0,
                       numRequests: 0};
    var siteSummary = new WMStats.GenericRequestsSummary(siteSummary);
    
    siteSummary.createSummaryFromRequestDoc = function(doc) {
        var summary = WMStats.SiteSummary();
        summary.summaryStruct.totalEvents = Number(this._get(doc, "input_events", 0));
        summary.summaryStruct.processedEvents = this._get(doc, "output_progress.0.events", 0);
        summary.summaryStruct.numRequests = 1;
        summary.jobStatus = doc;
        //support legacy code which had cooloff jobs instead cooloff.create, cooloff.submit
        //cooloff.job
        if ((typeof summary.jobStatus.cooloff) === "number") {
            summary.jobStatus.cooloff = {create: 0, submit: 0, job: summary.jobStatus.cooloff};
        }
        return summary;
    };
    
    return siteSummary;
};
WMStats.namespace("JobDetails");

WMStats.JobDetails = function (couchData) {
    
    var jobDetailData = new WMStats._StructBase();
    
    jobDetailData.convertCouchData = function(data) {
                                     var dataRows = data.rows;
                                     var jobDetails = [];
                                     for (var i in dataRows){
                                         jobDetails.push(dataRows[i].doc);
                                     }
                                     return jobDetails;
                                 };
    
    if (couchData) jobDetailData.setData(couchData);
    
    return jobDetailData;
};
WMStats.namespace("WorkloadSummary");

WMStats.WorkloadSummary = function (couchData) {
    
    var workloadSummaryData = new WMStats._StructBase();
    
    workloadSummaryData.convertCouchData = function(data) {
                                     var dataRows = data.rows;
                                     var workloadSummary = [];
                                     for (var i in dataRows){
                                     	 var wmstatsDoc = WMStats.Globals.convertRequestDocToWMStatsFormat(dataRows[i].doc);
                                         workloadSummary.push(wmstatsDoc);
                                     }
                                     return workloadSummary;
                                 };
    
    if (couchData) workloadSummaryData.setData(couchData);
    
    return workloadSummaryData;
};
WMStats.namespace("History");
WMStats.namespace("TimeBucket");

WMStats.TimeBucket = function() {

    var _bucket = new Array();
    var _bucketSize = 24;
    var _interval = 3600; // 1 hour

    function addRow(currentTime, row) {
        var timestamp = row.key[0];
        var couchDoc = row.doc;
        var index = parseInt((currentTime - timestamp) / 3600);
        var updateFlag = false;
        if (_bucket[index] && _bucket[index][couchDoc.agent_url]) {
            if (_bucket[index][couchDoc.agent_url].timestamp < couchDoc.timestamp) {
                updteFlag = true;
            }
        } else if (_bucket[index]) {
            updateFlag = true;
        } else {
            _bucket[index] = {};
            _bucket[index][couchDoc.agent_url] = {};
            _bucket[index][couchDoc.agent_url].requests = {};
            updateFlag = true;
        }
        
        if (updateFlag) {
            _bucket[index][couchDoc.agent_url].timestamp = couchDoc.timestamp;
            _bucket[index][couchDoc.agent_url].requests[couchDoc.workflow] = couchDoc;
        }
    }
    
    function addHistory(couchData) {
        var dataRows = couchData.rows;
        var currentTime = Math.round((new Date()).getTime() / 1000);
        for (var i in dataRows) {
            addRow(currentTime, dataRows[i]);
        }
    }
    
    function getData() {
        return _bucket;
    }
    /*
    function getSiteFailureRate() {
        var requestData = []
        for (var i in _bucket) {
            requestData[i] = {}
            for (var request in _bucket[i]) {
                requestData[i][request]
            }
            requestData[i] = _bucket.requests
        }
    }
    */
    return {
        addHistory: addHistory,
        getData: getData
    };
};

WMStats.History = function (couchData) {
    var _data;
    
    var historyData = new WMStats._StructBase();
    
    historyData.convertCouchData = function(data) {
        var bucket = WMStats.TimeBucket();
        bucket.addHistory(data);
        return bucket.getData();
    };
    
    
    if (couchData) historyData.setData(couchData);
    
    return historyData;
};
WMStats.namespace("LogDBData");

WMStats.LogDBData = function (couchData) {
    
    var logDBData = new WMStats._StructBase();
    logDBData._sortedLogs = {};
    logDBData._errorLogIDList =[];
    logDBData._errorLogs = [];

    logDBData._timestampSort = function(a, b) {
    	// sort the record by descending order
        return (a.ts - b.ts);
    };
    
    logDBData._getValue = function(obj, key, defaultType) {
    	if (obj.key === undefined) {
    		if (defaultType !== undefined) {
    			obj[key] = defaultType;
    		} else {
    			obj[key] = {};	
    		}
    	} 
    	return obj[key];
    };
    
    logDBData._getThreadValue = function(obj, request, agent, thread) {
    	requestObj = this._getValue(this._sortedLogs, request);
    	agentObj = this._getValue(requestObj, agent);
    	threadObj = this._getValue(agentObj, thread, []);
    	return threadObj;
    };
    
    logDBData.convertCouchData = function (couchData) {
        var results = [];
        if (couchData.rows) {
        	for (var i in couchData.rows) {
        		row = couchData.rows[i]['value'];
        		row.request = couchData.rows[i]['key'];
        		row.id = couchData.rows[i].id;	
        		results.push(row);
        	}	
        }
        // and sortby ts stamp decending order
        results.sort(this._timestampSort); 
        return results;
    },
    
    logDBData._setLogWithLastestError = function () {
    	
    	for (var i in this._data) {
    		 thrResult = this._getThreadValue(this._sortedLogs, this._data[i].request,
    		 	                             this._data[i].agent, this._data[i].thr);
    		 if (this._data[i].type !== "message") {
    		 	thrResult.push(this._data[i]);	
    		 };
    	};
    	
    	var logs = this._sortedLogs;
    	for (var req in this._sortedLogs) {
    		for (var agent in logs[req]) {
    			for (var thread in logs[req][agent]) {
    			    for (var i in logs[req][agent][thread]) {
    					if (logs[req][agent][thread][i].type === "agent-error" ||
    					    logs[req][agent][thread][i].type === "agent-warning") {
    						this._errorLogs.push(logs[req][agent][thread][i]);
    						this._errorLogIDList.push(logs[req][agent][thread][i].id);
    					}
    				}
    			}
    		}
    	}
    	
    	return this._errorLogs;
    };
    
    logDBData.getErrorLogIDs = function () {
    	return this._errorLogIDList;
    };
    
    logDBData.setMessages = function (request, agent, thread, messages) {
    	this._sortedLogs[request][agent][thread][0].messages = messages;
    };
    
    logDBData.getLogWithLastestError = function() {
    	return this._errorLogs;
    };
    
    if (couchData) logDBData.setData(couchData);
    
    logDBData._setLogWithLastestError();
    
    return logDBData;
};
WMStats.namespace("LogMessage");

WMStats.LogMessage = function (couchData) {
    
    var logdbDetailData = new WMStats._StructBase();
  	
  	//set default value to []
    logdbDetailData._data =[];
    
    logdbDetailData.convertCouchData = function(data) {
                                        var dataRows = data.rows;
                                        var rows =[];
                                        for (var i in dataRows) {
                                            var tableRow = {};
                                            doc = dataRows[i].doc;
                                            tableRow.request = doc.request;
                                            tableRow.agent = doc.identifier;
                                            tableRow.thr = doc.thr;
                                            tableRow.messages = doc.messages;
                                            rows.push(tableRow);
                                        }
                                        return rows;
   	                               };
   	                               
   	logdbDetailData.setMessagesToLogDBData = function() {
   		var logDBData = WMStats.RequestLogModel.getData();
   		var d = this._data;
   		for (var i in d) {
   			logDBData.setMessages(d[i].request, d[i].agent, d[i].thr, d[i].messages);
   		}	
   	};
   	
    if (couchData) logdbDetailData.setData(couchData);
    
    logdbDetailData.setMessagesToLogDBData();
       
    return logdbDetailData;
};WMStats.namespace("Requests");
WMStats.namespace("RequestsSummary");

WMStats.RequestsSummary = function() {
    //TODO add specific tier0 summary structure
    var tier0Summary = {};
    var requestSummary = new WMStats.GenericRequestsSummary(tier0Summary);
    return requestSummary;
};

WMStats.Requests = function(data) {
    var tier0Requests = new WMStats.GenericRequests(data);
    
    tier0Requests.getRequestAlerts = function() {
        
        var alertRequests = {};
        alertRequests['configError'] = [];
        alertRequests['siteError'] = [];
        alertRequests['cPaused'] = [];
        alertRequests['sPaused'] = [];
        alertRequests['jPaused'] = [];
        var ignoreStatus = ["closed-out",
                            "announced",
                            "aborted",
                            "rejected"]; 
        for (var workflow in this.getData()) {
            var requestInfo = this.getData(workflow);
                
            var reqSummary = this.getSummary(workflow);
            var cooloff = reqSummary.getTotalCooloff();
            var paused = reqSummary.getTotalPaused();
            var failure = reqSummary.getTotalFailure();
            var success = reqSummary.getJobStatus("success");
            var sPaused = reqSummary.getJobStatus("paused.submit");
            var jPaused = reqSummary.getJobStatus("paused.job");
            var cPaused = reqSummary.getJobStatus("paused.create");
            var totalFailed = failure;
            if (cPaused > 0){
            	alertRequests['cPaused'].push(this.getData(workflow));
            } else if (sPaused > 0){
            	alertRequests['sPaused'].push(this.getData(workflow));
            } else if (jPaused > 0){
            	alertRequests['jPaused'].push(this.getData(workflow));
            }
            /* 
            else if ( totalFailed > 0) {
                if (success === 0) {
                    alertRequests['configError'].push(this.getData(workflow));
                } else if ((totalFailed / (totalFailed + success)) > 0.85) {
                    alertRequests['siteError'].push(this.getData(workflow));
                }
            }*/
        }
        return alertRequests;
    };
    
    tier0Requests.numOfRequestError = function() {
    	/***
    	 * TODO: need to implement this
    	 ***/
    	var alertData = this.getRequestAlerts();
    	var numError = {};
    	numError.alert = 0;
    	for (var error in alertData) {
        	numError.alert += alertData[error].length;
        };
        
        return numError;
    };
    return tier0Requests;
};
WMStats.namespace("RunSummary");
WMStats.namespace("RunStatus");

/*
 * run status is ordered as lower index determines combined status.
 * i.e. If one request is "Active" and the other is "Real Time Processing"
 * run status will be Active 
 * There is no way to determine given run has PromptReco "Real Timp Done" is moved
 * to right before Complete status. If run contains only Express and Repack workflows,
 * it will never move to Complete run status. However workflow status will be moved to 
 * completed by task archiver and eventually archived.
 */
WMStats.RunStatus = ["Active", "Real Time Processing", "Real Time Merge", 
                     "Real Time Harvesting", "PromptReco", "Reco Merge",
                     "Reco Harvest", "Processing Done", "Real Time Done","Complete"];
                     
WMStats.RunSummary = function() {
    var runSummary = {numRequests: 0};
    var runSummary = new WMStats.GenericRequestsSummary(runSummary);
    
    function getRunStatus(doc) {
        var wfStatus = doc["request_status"][doc["request_status"].length - 1].status;
        var wfPrefix = doc['workflow'].split("_")[0].toLowerCase();
        if (wfPrefix == "express" || wfPrefix == "repack") {
            if (wfStatus == "new") {
                return WMStats.RunStatus[0];
            } else if (wfStatus == "Closed") {
                return WMStats.RunStatus[1];
            } else if (wfStatus == "Merge") {
                return WMStats.RunStatus[2];
            } else if (wfStatus == "Harvesting") {
                return WMStats.RunStatus[3];
            } else if (wfStatus == "Processing Done") {
                return WMStats.RunStatus[8];
            }
        } else if (wfPrefix == "promptreco") {
            if (wfStatus == "new" || wfStatus == "Closed" || wfStatus == "AlcaSkim" ) {
                return WMStats.RunStatus[4];
            } else if (wfStatus == "Merge") {
                return WMStats.RunStatus[5];
            } else if (wfStatus == "Harvesting") {
                return WMStats.RunStatus[6];
            } else if (wfStatus == "Processing Done") {
                return WMStats.RunStatus[7];
            } 
        }
        return WMStats.RunStatus[9];
    }
    
    runSummary.summaryStructUpdateFuction = function(baseObj, additionObj, field) {
        if (field === "runStatus") {
            if (WMStats.RunStatus.indexOf(baseObj[field]) > WMStats.RunStatus.indexOf(additionObj[field])) {
                baseObj[field] = additionObj[field];
            }
        } else {
            baseObj[field] += additionObj[field];
        }
    };

    runSummary.createSummaryFromRequestDoc = function(doc) {
        var summary = WMStats.RunSummary();
        summary.summaryStruct.numRequests = 1;
        summary.summaryStruct.runStatus = getRunStatus(doc);
        summary.jobStatus = this._get(doc, 'status', {});
        
        return summary;
    };
    
    return runSummary;
};
/** abstract environment of the web page */
WMStats.namespace("_ViewModelBase");

WMStats._ViewModelBase = function (observableObj) {
    this._eventObj = {};
    this._eventObjName = "_eventObj";
    // special observable property holding represented data
    this.data = this.observable('data', null);
    if (observableObj !== undefined) {
        this._createObservableProperty(observableObj);
    };
    this._selector = null;
    this._data = null;
};

WMStats._ViewModelBase.prototype = {

    observable: function(property, defaultValue) {
        
        var _previousValue = defaultValue;
        var _property = property;
        
        function publish(value, noTrigger) {
            if (value !== undefined) {
                _previousValue = value;
                //TODO: check the equality
                /* trigger parent object changed */
                if (!noTrigger) {
                    $(this._eventObj).triggerHandler(this._eventObjName, this);
                    /* trigger property changed */
                    $(this._eventObj).triggerHandler(_property, value);
                
                }
                return this;
            } else {
                return _previousValue;
            }
        }
        return publish;
    },
    
    subscribe: function() {
        /*
         * takes 1 or 2 arguments 
         * if 1, it is callback function when this object changes
         * if 2. 1st is property name and 2nd is callback function
         * 
         */
        var callback;
        var eventName;
        if (arguments.length == 2) {
            callback = arguments[1];
            eventName = arguments[0];
        } else if (arguments.length == 1) {
            callback = arguments[0];
            eventName = this._eventObjName;
        }
        
        $(this._eventObj).on(eventName, callback);
    },
    
    id: function(selector) {
        // link the selector with the ViewModel
        // may need to connect multiple selector
        if (selector) {
            this._selector = selector;
            WMStats.ViewModel.Registry[selector] = this;
            return this;
        } else {
            return this._selector;
        } 
    },
    
    _createObservableProperty: function(observableObj) {
        for (var prop in observableObj) {
            this[prop] = this.observable(prop, observableObj[prop]);
        }
    }
};

WMStats.namespace("ViewModel");

WMStats.ViewModel = (function (){
    var properties = {page: null};
    var vm = new WMStats._ViewModelBase(properties);
    vm.propagateUpdate = function() {
        if (vm.page().propagateUpdate) {
            vm.page().propagateUpdate();
        }
     };
    return vm;
})();

// create WMStats.ViewModel properties
(function(vm) {
    
    vm.Registry  ={};

    vm.ActiveRequestPage = (function (){
        var properties = {view: null,
                          filter: {},
                          filteredStats: null,
                          refreshCount: 0};
        var requestPage = new WMStats._ViewModelBase(properties);
        
        requestPage.propagateUpdate = function() {
            var requestData = WMStats.ActiveRequestModel.getData();
            if (requestData === null) {
                return false;
            } else {
                var filter = vm.ActiveRequestPage.filter();
                vm.ActiveRequestPage.data(requestData.filterRequests(filter));
                if (requestPage.view().propagateUpdate) {
                    requestPage.view().propagateUpdate();
                }
                return true;
            }
        };
        
        return requestPage;
    })();
    
    vm.RequestAlertPage = (function (){
        var alertPage = new WMStats._ViewModelBase();
        
        alertPage.propagateUpdate = function() {
            var requestData = WMStats.ActiveRequestModel.getData();
            if (requestData === null) {
                return false;
            } else {
                vm.RequestAlertPage.data(requestData);
                return true;
            }
        };
        
        return alertPage;
    })();
    
    vm.AgentPage = (function (){
        var agentPage =new WMStats._ViewModelBase();
        
        agentPage.propagateUpdate = function() {
            var agentData = WMStats.AgentModel.getData();
            if (agentData === null) {
                return false;
            } else {
                vm.AgentPage.data(agentData);
                return true;
            }
        };
        return agentPage;
    })();
    
    vm.LogDBPage = (function (){
        var logDBPage =new WMStats._ViewModelBase();
        
        logDBPage.propagateUpdate = function() {
        
            var logDBData = WMStats.RequestLogModel.getData();
            if (logDBData === null) {
                return false;
            } else {
                vm.LogDBPage.data(logDBData);
                return true;
            }
        };
        return logDBPage;
    })();
    
    vm.SearchPage = (function (){
        /*
         * keys contain {searchCategory: blah, searchValue: blah}
         */
        var properties = {keys: null};
        searchPage = new WMStats._ViewModelBase(properties);
        
        searchPage.retrieveData = function(keys) {
            var selectedSearch = keys.searchCategory;
            var searchStr = keys.searchValue;
            var view;
            var options =  {'include_docs': true, 'reduce': false};
            if (selectedSearch === 'request') {
                view = "allDocs";
                options.key = searchStr;
            } else if (selectedSearch === 'outputdataset') {
                view = "byoutputdataset";
                options.key = searchStr;
            } else if (selectedSearch === 'inputdataset') {
                view = "byinputdataset";
                options.key = searchStr;
            } else if (selectedSearch === 'prep_id') {
                view = "byprepid";
                options.key = searchStr;
            } else if (selectedSearch === 'data_pileup') {
                view = "bydatapileup";
                options.key = searchStr;
            } else if (selectedSearch === 'mc_pileup') {
                view = "bymcpileup";
                options.key = searchStr;
            } else if (selectedSearch === 'request_date') {
                view = "bydate";
                var beginDate = $('input[name="dateRange1"]').val().split("/");
                var endDate = $('input[name="dateRange2"]').val().split("/");
                options.startkey = [Number(beginDate[0]), Number(beginDate[1]), Number(beginDate[2])];
                options.endkey = [Number(endDate[0]), Number(endDate[1]), Number(endDate[2]), {}];
            };
            
            WMStats.RequestSearchModel.retrieveData(view, options);
        };
        return searchPage;
    })();
    
    vm.CategoryView = (function (){
        
        var properties = {category: null,
                          detailView: null};
                          
        var categoryView = new WMStats._ViewModelBase(properties);
        
        categoryView.convertToCategoryData = function (requestData) {
            if (requestData === undefined) {
                requestData = vm.ActiveRequestPage.data();
            }
            var category = this.category().name();
            var summaryStruct = WMStats.CategorySummaryMap.get(this.category().name());
            var categoryData = WMStats.RequestsByKey(category, summaryStruct);
            categoryData.categorize(requestData);
            return categoryData;
        };
        
        categoryView.propagateUpdate = function() {
            if (vm.ActiveRequestPage.data() === null) {
                return false;
            };
            categoryView.data(categoryView.convertToCategoryData());
            return true;
        };
        
        return categoryView;
    })();
    
    vm.RequestView = (function (){
        /*
         * format is progress or numJobs - this is tied to name of the button
         * un tie.
         */
        
        var properties = {categoryKey: "all", 
                          format: null, 
                          detailView: null};
        
        var requestView = new WMStats._ViewModelBase(properties);
        
        requestView.propagateUpdate = function() {
            if (vm.ActiveRequestPage.data() === null) {
                return false;
            };
            if (requestView.categoryKey() == "all") {
                requestView.data(vm.ActiveRequestPage.data());
            } else {
                var categoryData = vm.CategoryView.convertToCategoryData();
                var data = categoryData.getRequestData(requestView.categoryKey());
                requestView.data(data);
            };
            return true;
        };
        //TODO: hack for assigning for category map
        requestView.categoryName = "requests";
        return requestView;
    })();
    
    vm.TaskView = (function (){
        /*
         * TODO: not done need to replace in TaskSummary Table.
         */
        
        var properties = {requestName: null};
        
        var taskView = new WMStats._ViewModelBase(properties);
        
        taskView.propagateUpdate = function() {
            if (taskView.requestName()) {
                taskView.data(WMStats.ActiveRequestModel.getData().getTasks(taskView.requestName()));
                return false;
            } else {
                return false;
            }
        };
        return taskView;
    })();
    
    function createJobSummaryView(){
        var properties = {requestName: null, detail: null};
        var jobView = new WMStats._ViewModelBase(properties);
        
        jobView.retrieveData = function(requestName) {
            WMStats.JobSummaryModel.setRequest(requestName);
            WMStats.JobSummaryModel.retrieveData();
        };
        
        jobView.propagateUpdate = function() {
            
            /* this part is needed if you want to refresh the job view when data is updated */
            if (jobView.requestName()) {
                jobView.retrieveData(jobView.requestName());
                return false;
            } else {
                return false;
            }
        };
        
        jobView.updateDataAndChild = function(data) {
            if (data) {jobView.data(data);}
            if (jobView.detail().propagateUpdate) {
                jobView.detail().propagateUpdate();
                return true;
            }
            return false;
        };
        
        return jobView;
    };
    
    vm.JobView = createJobSummaryView();
    vm.AlertJobView = createJobSummaryView();
    
    vm.CampaignCategory = (function (){
        return new WMStats._ViewModelBase({name: "campaign"});
    })();
    
    vm.SiteCategory = (function (){
        return new WMStats._ViewModelBase({name: "sites"});
    })();
    
    
    vm.CMSSWCategory = (function (){
        return new WMStats._ViewModelBase({name: "cmssw"});
    })();
    
    vm.AgentCategory = (function (){
        return new WMStats._ViewModelBase({name: "agent"});
    })();
    
    vm.RunCategory = (function (){
        return new WMStats._ViewModelBase({name: "run"});
    })();
    
    /* request view summary format */
    vm.RequestProgress = (function (){
        return new WMStats._ViewModelBase({name: "progress"});
    })();
    
    vm.RequestJobs = (function (){
        return new WMStats._ViewModelBase({name: "numJobs"});
    })();
    
    /* request view job detail */
    vm.CategoryDetail = (function (){
        var properties = {categoryKey: null};
        var categoryDetail = new WMStats._ViewModelBase(properties);
        
        categoryDetail.propagateUpdate = function() {
            if (categoryDetail.categoryKey()) {
                var allData = vm.CategoryView.data();
                categoryDetail.data(allData.getData(categoryDetail.categoryKey()));
            }
        };
        return categoryDetail;
    })();
    
   
    vm.RequestDetail = (function (){
        var properties = {requestName: null};
        var requestDetail = new WMStats._ViewModelBase(properties);
        requestDetail.open = false;
        return requestDetail;
    })();
    
    function createJobDetailView(){
        var properties = {keys: null, indexID: null};
        var jobDetail = new WMStats._ViewModelBase(properties);
        
        jobDetail.retrieveData = function(keys) {
            WMStats.JobDetailModel.setOptions(keys);
            WMStats.JobDetailModel.retrieveData();
        };
        
        jobDetail.propagateUpdate = function() {
            /* this part is needed if you want to refresh the job view when data is updated */
            if (jobDetail.keys() !== null) {
                jobDetail.retrieveData(jobDetail.keys());
                return true;
            } else {
                return false;
            }
        };
        
        return jobDetail;
    };
    
    vm.JobDetail = createJobDetailView();
    vm.AlertJobDetail = createJobDetailView();
    
    vm.Resubmission = (function (){
        /* resubmission keys contains 
         * requestName and taskName and acdc url
         * {requestName: blah, task: blah, acdcURL: blah}
         * */
       
        var properties = {keys: null};
        var resubmission = new WMStats._ViewModelBase(properties);
        
        resubmission.retrieveData = function(keys) {
            WMStats.ReqMgrRequestModel.retrieveDoc(keys.requestName);
        };
        
        resubmission.formatResubmissionData = function(reqMgrRequest) {
            var summary = {};
            // from resubmission keys
            summary.RequestType = "Resubmission";
            summary.OriginalRequestName = resubmission.keys().requestName;
            summary.InitialTaskPath = resubmission.keys().task;
            if (resubmission.keys().acdcURL) {
                //TODO: if there is not acdc_url don't create the button'
                var acdcServiceUrl = WMStats.Utils.splitCouchServiceURL(resubmission.keys().acdcURL);
                summary.ACDCServer = acdcServiceUrl.couchUrl;
                summary.ACDCDatabase = acdcServiceUrl.couchdb;
            }
            
            var requestInfo = reqMgrRequest.getData();
            // passed to wmstats view
            summary.RequestString = WMStats.Utils.acdcRequestSting(summary.OriginalRequestName, requestInfo.Requestor);
            summary.Campaign = requestInfo.Campaign;
            summary.RequestPriority = requestInfo.RequestPriority;

            //TODO get the site white list from acdc db. (site might be overflowed)
            // assign more value for acdc assignment
            summary.Team = requestInfo.Team;
            summary.TrustPUSitelists = false;
            summary.TrustSitelists = false;

            return summary;
        };
        
        resubmission.propagateUpdate = function() {
            if (resubmission.keys() !== null) {
                resubmission.retrieveData(resubmission.keys());
                return true;
            } else {
                return false;
            }
        };
        
        return resubmission;
    })();

    vm.initialize = (function (){ 
        //default setting
        vm.page(vm.ActiveRequestPage, true);
        vm.ActiveRequestPage.view(vm.CategoryView, true);
        vm.CategoryView.category(vm.CampaignCategory, true);
        vm.RequestView.format(vm.RequestProgress, true);
        vm.JobView.detail(vm.JobDetail, true);
    })();

})(WMStats.ViewModel);

// control inside the view model
(function(vm){
     // filter control
    vm.ActiveRequestPage.subscribe("filter", function() {
            vm.ActiveRequestPage.propagateUpdate();
        });
    
    vm.SearchPage.subscribe("keys", function(){
        vm.SearchPage.retrieveData(vm.SearchPage.keys());
    });
    
    vm.CategoryView.subscribe("category", function() {
        vm.CategoryView.propagateUpdate();
    });
    
    vm.RequestView.subscribe("categoryKey", function() {
        vm.ActiveRequestPage.view(vm.RequestView);
    });
    
    vm.JobView.subscribe("requestName", function() {
        vm.ActiveRequestPage.view(vm.JobView);
    });
    
    vm.AlertJobView.subscribe("requestName", function() {
        vm.AlertJobView.retrieveData(vm.AlertJobView.requestName());
    });
    
    vm.CategoryDetail.subscribe("categoryKey", function() {
        vm.CategoryDetail.propagateUpdate();
    });
    
    vm.JobDetail.subscribe("keys", function() {
        vm.JobDetail.retrieveData(vm.JobDetail.keys());
    });
    
    vm.AlertJobDetail.subscribe("keys", function() {
        vm.AlertJobDetail.retrieveData(vm.AlertJobDetail.keys());
    });
    
    vm.Resubmission.subscribe("keys", function() {
        vm.Resubmission.retrieveData(vm.Resubmission.keys());
    });

})(WMStats.ViewModel);
WMStats.namespace("CategorySummaryMap");
WMStats.namespace("CategoryTableMap");

WMStats.CategorySummaryMap = function(){
    var summaryMap = {};
    function add(category, summaryFunc) {
        summaryMap[category] = summaryFunc;
    }
    function get(category) {
        return summaryMap[category];
    }
    return {add: add, get: get};
}();

WMStats.CategoryTableMap = function(){
    var tableMap = {};
    var vm = WMStats.ViewModel;
    function add(category, view) {
        tableMap[category] = view;
    }
    function get(category, view) {
        if (category === vm.RequestView.categoryName) {
            return tableMap[category][vm.RequestView.format().name()];
        }else {
            return tableMap[category];
        }
        
    }
    return {add: add, get: get};
}();

(function(vm){
    //WMStats.CategoryTableMap.add(WMStats.Controls.requests, WMStats.ActiveRequestTableWithJob);
    
    // add controller
    vm.CategoryView.subscribe("data", function() {
        var view = WMStats.CategoryTableMap.get(vm.CategoryView.category().name());
        view(vm.CategoryView.data(), vm.CategoryView.category().id());
    });
    
    vm.RequestView.subscribe("data", function() {
        var view = WMStats.CategoryTableMap.get(vm.RequestView.categoryName);
        view(vm.RequestView.data(), vm.RequestView.format().id());
    });
})(WMStats.ViewModel);
WMStats.namespace("View");

WMStats.View.IndexHTML = function(){
    
    function applyTemplate(){
        var viewPane = $('#data_board div.viewPane');
        $('div.viewTemplate').children().clone().appendTo(viewPane);
    };
    
    function retrieveData() {
        WMStats.ActiveRequestModel.retrieveData();
        WMStats.AgentModel.retrieveData();
        //WMStats.HistoryModel.setOptions();
        //WMStats.HistoryModel.retrieveData();
    };

    $(document).ready(function() {
        $('#loading_page').addClass("front").show();
        //applyTemplate();
        WMStats.CommonControls.setLinkTabs("#link_tabs");
        WMStats.CommonControls.setUTCClock("#clock");
        WMStats.CommonControls.setWorkloadSummarySearch("#search_option_board");
        WMStats.Controls.setFilter("#filter_board");
        WMStats.Controls.setAllRequestButton("#status_board");
        WMStats.Controls.setTabs("#tab_board");
        WMStats.Controls.setCategoryButton("#category_bar");
        WMStats.Controls.setViewSwitchButton("#view_switch_bar");
        
        //view model bind
        var vm = WMStats.ViewModel;
        vm.ActiveRequestPage.id('#activeRequestPage');
        vm.RequestAlertPage.id('#requestAlertPage');
        vm.AgentPage.id('#agentInfoPage');
        vm.LogDBPage.id('#logDBPage');
        vm.SearchPage.id('#workloadSummaryPage');
        
        vm.CategoryView.id('#category_view');
        vm.RequestView.id('#request_view');
        vm.JobView.id('#job_view');
        
        // Category summary view
        //TODO: Tier1 Category - move this some other place which belongs Tier1
        vm.CampaignCategory.id('#category_view div.summary_data');
        vm.SiteCategory.id('#category_view div.summary_data');
        vm.CMSSWCategory.id('#category_view div.summary_data');
        vm.AgentCategory.id('#category_view div.summary_data');
        
        //TODO: Tier0 Category - move this some other place which belongs Tier0
        vm.RunCategory.id('#category_view div.summary_data');
        
        // Request summary view
        vm.RequestProgress.id('#request_view div.summary_data');
        vm.RequestJobs.id('#request_view div.summary_data');
        
        // Job summary view
        //To do  need to add
        
        vm.CategoryDetail.id('#category_view div.detail_data');
        vm.RequestDetail.id('#request_view div.detail_data');
        vm.JobDetail.id('#job_view div.detail_data');
        vm.Resubmission.id('#acdc_submission');
        
       
        // request alert view
        vm.AlertJobView.id("#alert_job_summary");
        vm.AlertJobDetail.id("#alert_job_detail");
        
        // view model controller
        var wsControl = WMStats.GenericController;
        
        //callback funtion
        function switchPage(event, data) {
            //data is page object
            wsControl.switchDiv(data.id(), ["#activeRequestPage", "#requestAlertPage", 
                                            "#agentInfoPage", "#logDBPage", "#workloadSummaryPage"]);
            vm.propagateUpdate();
        };
        
        function switchView(event, data) {
            //data is page object
            wsControl.switchDiv(data.id(), ["#category_view", "#request_view", "#job_view"]);
            vm.ActiveRequestPage.view().propagateUpdate();
        };
        
        // set the control bind
        vm.subscribe("page", switchPage);
        vm.ActiveRequestPage.subscribe("view", switchView);
        vm.RequestView.subscribe("categoryKey", function(event, categoryKey) {
             WMStats.CategoryTitle(categoryKey, '#category_title');
        });
        
        vm.JobView.subscribe("requestName", function(event, requestName) {
             WMStats.RequestTitle(requestName, '#request_title');
        });
        
        var E = WMStats.CustomEvents;
        $(WMStats.Globals.Event).on(E.AGENTS_LOADED, function(event, agentData) {
        	// refresh the agentData.agentNumber.error
        	agentData.getAlertList();
            if (agentData.agentNumber.error > 0) {
                $('#linkTabs a[href="#agentInfoPage"] strong').text("(" + agentData.agentNumber.error + ")");
            } else {
            	$('#linkTabs a[href="#agentInfoPage"] strong').text("");
            }
        });
        
        $(WMStats.Globals.Event).on(E.ERROR_LOG_LOADED, function(event, logdbDetailData) {
        	// refresh the agentData.agentNumber.error
        	var errorNum = logdbDetailData.getData().length;
            if ( errorNum > 0) {
                $('#linkTabs a[href="#logDBPage"] strong').text("(" + errorNum + ")");
            } else {
            	$('#linkTabs a[href="#logDBPage"] strong').text("");
            };
        });
        
        $(WMStats.Globals.Event).on(E.REQUESTS_LOADED, 
            function(event) {
                //only Tier1 case
                var numError = WMStats.ActiveRequestModel.getData().numOfRequestError();
                if ((numError.alert) > 0) {
                    $('#linkTabs a[href="#requestAlertPage"] strong').text("(" + numError.alert  + ")");
                } else {
                	$('#linkTabs a[href="#requestAlertPage"] strong').text("");
                };
        });
        
        // initialize page)
        vm.page(vm.ActiveRequestPage);
        vm.ActiveRequestPage.view(vm.CategoryView);
        //vm.CategoryView.subscribe("ecategory", switchCategory) 
        retrieveData();
        //$("div.draggable").draggable();
        // 3 min update
        setInterval(retrieveData, 180000);
     } );
};
WMStats.namespace("CommonControls");
WMStats.CommonControls = function($){
    
    var vm =  WMStats.ViewModel;
    var vmRegistry = WMStats.ViewModel.Registry;
    
    function setUTCClock(selector) {
       setInterval(function() {
            $(selector).text(WMStats.Utils.utcClock());
        }, 100);
    };

    function setLinkTabs(selector) {
       var linkTabs = 
        '<nav id="linkTabs" class="linkTabs">\
            <ul><li><a href="#activeRequestPage"> active request </a></li>\
                <li><a href="#requestAlertPage"> request alert <strong></strong></a></li>\
                <li><a href="#agentInfoPage"> agent info <strong></strong></a></li>\
                <li><a href="#logDBPage"> error logs <strong></strong></a></li>\
                <li><a href="#workloadSummaryPage"> search </a></li></ul>\
         </nav>';
        
        $(selector).append(linkTabs);
        
        // add controller for this view
        function changeTab(event, data) {
            $('#linkTabs li').removeClass("title-tab-selected").addClass("title-tab-hide");
            $('#linkTabs a[href="' + data.id() +'"]').parent().removeClass("title-tab-hide").addClass("title-tab-selected");
        }
        // viewModel -> view control
        vm.subscribe("page", changeTab);
        
        // view -> viewModel control
        $(document).on('click', "#linkTabs li a", function(event){
            vm.page(vmRegistry[this.hash]);
            event.preventDefault();
        });
    };

    function setWorkloadSummarySearch(selector) {
        var searchOption =
                '<fieldset id="SearchOptionsPane">\
                    <legend>Search WorkloadSummary</legend>\
                    <div id="searchPane">\
                        <div class="OptionBox">\
                            <select name="SearchOptions" class="searchSelector" title="Select a search option">\
                              <option value="request" data-search-type="stringMatch" selected="selected"> request name</option>\
                              <option value="outputdataset" data-search-type="stringMatch"> output dataset </option>\
                              <option value="inputdataset" data-search-type="stringMatch">input dataset</option>\
                              <option value="prep_id" data-search-type="stringMatch">prep id</option>\
                              <option value="data_pileup" data-search-type="stringMatch">data pileup</option>\
                              <option value="mc_pileup" data-search-type="stringMatch">mc pileup</option>\
                              <option value="request_date" data-search-type="dateRange">date range</option>\
                            </select>\
                        </div>\
                        <div class="SearchBox">\
                           <input type="text" size="100" name="workloadSummarySearch" value=""></input>\
                        </div>\
                     </div>\
                     <div>\
                        <button type="submit" id="WorkloadSummarySearchButton">submit</button>\
                     </div>\
                     </fieldset>';
                     
        $(selector).append(searchOption);
        
        // change the search options
        $(document).on('change', 'select[name="SearchOptions"]',function(){
            var filterType = $(':selected', this).attr('data-search-type');
            var searchBox = $('#searchPane .SearchBox');
            $(searchBox).empty();
            $('div.template.'+ filterType).children().clone().appendTo('#searchPane .SearchBox');
            $('#searchPane .SearchBox input[name="dateRange1"]').datepicker({
                altField: 'input[name="dateRange1"]', 
                altFormat: "yy/mm/dd", 
                changeYear: true, 
                yearRange: "2012:c"});
            $('#searchPane .SearchBox input[name="dateRange2"]').datepicker({
                altField: 'input[name="dateRange2"]', 
                altFormat: "yy/mm/dd", 
                changeYear: true, 
                yearRange: "2012:c"});
        });
        
        // control submit button
        $(document).on('click', '#WorkloadSummarySearchButton', function(event) {
            var keys = {};
            keys.searchCategory = $('#search_option_board select[name="SearchOptions"] :selected').val();
            keys.searchValue = $('input[name="workloadSummarySearch"]').val();
            vm.SearchPage.keys(keys);
            event.stopPropagation();
        });
    };
    
    return {
        setUTCClock: setUTCClock,
        setLinkTabs: setLinkTabs,
        setWorkloadSummarySearch: setWorkloadSummarySearch
    };
}(jQuery);WMStats.namespace("Table");
WMStats.Table = function(config, tableSetting) {

    var tableSetting = tableSetting || '<table cellpadding="0" cellspacing="0" border="0.5" class="display" width="100%"></table>';
    var tableConfig = {
        //"paginationType": "full_numbers",
        //"scrollX": "100%",
        //"scrollCollapse": true,
        "stateSave": true,
        "processing": true,
        //"iDisplayLength": 10,
        "dom": '<"top"pl>rt<"bottom"ip>',
        //"sDom": 'C<"clear">lfrtip',
        "autoWidth": true,
        "jQueryUI": true	
        };
    
    function updateConfig(config) {
        for (var prop in config) {
            tableConfig[prop] = config[prop];
        }
        
    }
    
    /* footer is needed to use columnFilter */ 
    function _footer() {
        var footer = '<tfoot><tr>';
      
        for (var i in tableConfig.aoColumns) {
            if (tableConfig.aoColumns[i].bVisible != false){
                footer += '<th>' + tableConfig.aoColumns[i]["title"] + '</th>';
            }
        }
        footer += '</tr></tfoot>';
        return footer;
    }
    
    function create(selector, filterConfig) {
        $(selector).empty();
        $(selector).html(tableSetting);
        
        tableConfig.stateSaveCallback = function(settings,data) {
            localStorage.setItem(selector, JSON.stringify(data));
        };
        tableConfig.stateLoadCallback = function(settings) {
        	return JSON.parse(localStorage.getItem(selector));
        };
		
        var oTable = $(selector + " table").DataTable(tableConfig);
        if ( oTable.length > 0 ) {
            oTable.columns.adjust().draw();
        }
        
        jQuery(WMStats.Globals.Event).triggerHandler(WMStats.CustomEvents.LOADING_DIV_END);
        
        //TODO: enable column filter
        //if (filterConfig) {
           //oTable.append(_footer());
           //https://datatables.net/reference/api/column().search()
        //}
        return oTable;
    }
    
    if (config) {updateConfig(config);}
    
    return {'config': tableConfig,
            'updateConfig': updateConfig,
            'create': create
           };
};
WMStats.namespace("JobSummaryTable");

WMStats.JobSummaryTable = function (data, containerDiv) {
    
    var tableConfig = {
        "dom": '<"top"plf>rt<"bottom"ip>',
        "pageLength": 25,
        "columns": [
            {"title": "L", 
             "defaultContent": 0,
             "render": function (data, type, row, meta) {
                            return WMStats.Utils.formatDetailButton("drill");
                        }},
            {  "render": function (data, type, row, meta) {
                if (type == 'display') {
                    var taskList = row.task.split('/');
                    return taskList[taskList.length - 1];
                }
                return row.task;
            }, "title": "task", "width": "150px"},
            { "data": "status", "title": "status"},
            { "data": "site", "title": "site"},
            { "data": "exitCode", "title": "exit code"},
            { "data": "count", "title": "jobs"},
            { "data": "errorMsg", "title": "error mesage", 
                           "defaultContent": ""},
            { "title": "acdc", defaultContent: "",
              "width": "15px",
              "render": function (data, type, row, meta) {
                            if (type == 'display') {
                                var taskList = row.task.split('/');
                                var endTask = taskList[taskList.length - 1];
                                if (row.status !== "success" && 
                                    !endTask.match(/LogCollect$/) && 
                                    !endTask.match(/Cleanup$/)) {
                                    return WMStats.Utils.formatDetailButton("acdc");
                                }
                            }
                            return "";
                        }
             }
         ],
         "order": [[1, 'asc']]
    };
    
    tableConfig.data = data.getData().status;
    
    var filterConfig = {};

    $(containerDiv).data('workflow', (data.getData()).workflow);
    
    // store the table data
    WMStats.JobSummaryTable.data = WMStats.Table(tableConfig).create(containerDiv, filterConfig);
    return WMStats.JobSummaryTable.data;
};

(function() {
    var vm = WMStats.ViewModel;
    
    vm.JobView.subscribe("data", function() {
        // need to create the lower level view
        var divSelector = vm.JobView.id() + " div.summary_data";
        WMStats.JobSummaryTable(vm.JobView.data(), divSelector);
    });
    
    /*
    vm.AlertJobView.subscribe("data", function() {
        WMStats.JobSummaryTable(vm.AlertJobView.data(), vm.AlertJobView.id());
    });
    */
})();
WMStats.namespace("WorkloadSummaryTable");

WMStats.WorkloadSummaryTable = function (data, containerDiv) {

    var formatReqDetailUrl = WMStats.Utils.formatReqDetailUrl;
    var formatWorkloadSummarylUrl = WMStats.Utils.formatWorkloadSummarylUrl;

    var tableConfig = {
        "pageLength": 25,
        "scrollX": true,
        "dom": 'lfrtip',
        "columns": [
            /*
            {"title": "D", 
             "defaultContent": 0,
             "width": "15px",
             "render": function (data, type, row, meta) {
                            return WMStats.Utils.formatDetailButton("detail");
                        }},
            {"title": "L", 
             "defaultContent": 0,
             "width": "15px",
             "render": function (data, type, row, meta) {
                            return WMStats.Utils.formatDetailButton("drill");
                        }},
            */
            { "data": "workflow", "title": "workflow",
             "render": function (data, type, row, meta) {
                            return formatReqDetailUrl(row._id);
                      },
              "width": "150px"
            },
            {"render": function (data, type, row, meta) {
              				if (type === "display") {
                            	return formatWorkloadSummarylUrl(row._id, 
                                	row.request_status[row.request_status.length -1].status);
                            }
                            return row.request_status[row.request_status.length -1].status;
                          },
            },
            { "data": "request_type", "title": "type", "defaultContent": ""},
            { "data": "priority", "title": "priority", "defaultContent": 0},
            { "data": "campaign", "title": "campaign", "defaultContent": ""}
        ]
    };
    
    var filterConfig = {};
    
    tableConfig.data = data.getData();
    
    return WMStats.Table(tableConfig).create(containerDiv, filterConfig);
};

(function() {
    
     var vm = WMStats.ViewModel;
     
     vm.SearchPage.subscribe("data", function(){
        WMStats.WorkloadSummaryTable(vm.SearchPage.data(), "#search_result_board");
     });
})();
    /*
 * Add EventHandler 
 */
(function($) {
    
    //custom events
    var E = WMStats.CustomEvents;
    var vm =  WMStats.ViewModel;

    // Super class for table event handling
    function TableEventHandler(containerID, populateRequestTable) {
        this.containerID = containerID;
        if (populateRequestTable) {
            this.populateRequestTable = populateRequestTable;
        }
    }
    // add Classmethod and property
    TableEventHandler.highlightRow = function(selector, currenElement) {
                                        $(selector).removeClass('mouseclickRow');
                                        $(currenElement).addClass('mouseclickRow');
                                   };

    TableEventHandler.prototype = { 
        constructor: TableEventHandler,
        
        tableRowBind: function(bind, parentSelector, func) {
            var currentObj = this;
            var selector =  parentSelector + " table tbody tr";
            $(document).on(bind, selector, function () {
                TableEventHandler.highlightRow(selector, this);
                currentObj[func](this);
            });
        },

        tableColumnBind: function(bind, parentSelector, name, func) {
            var currentObj = this;
            var selector =  parentSelector + ' table tbody tr td div[name="' + name + '"]';
            var rowSelector = parentSelector + ' table tbody tr';
            $(document).on(bind, selector, function () {
                var currentRow = $(this).closest("tr")[0];
                TableEventHandler.highlightRow(rowSelector, currentRow);
                currentObj[func](currentRow);
                event.preventDefault();
            });
        },
        
        populateRequestSummary: function(currentElement){
            // create the request table
            var nTds = $('td', currentElement);
            vm.RequestView.categoryKey($(nTds[2]).text());
        },
        
        populateJobSummary: function(currentElement){
            
            var nTds = $('td', currentElement);
            vm.JobView.requestName($(nTds[2]).text());
        },
        
        populateRequestDetail: function(currentElement){
            // create the request table
            var nTds = $('td', currentElement);
            vm.RequestDetail.requestName($(nTds[2]).text());
        },

        populateCategoryDetail: function(currentElement){
            // create the request table
            var nTds = $('td', currentElement);
            vm.CategoryDetail.categoryKey($(nTds[2]).text());
        },

        populateJobDetail: function (currentElement){
            // 2. create the job detail view
            var summary = {};
            // need to get the workflow name with out depending on the selector
            summary.workflow = $("#job_view div.summary_data").data("workflow");
            //summary.task = $(nTds[0]).text();
            var currentRow = $(currentElement).closest("tr")[0];
            var currentRowData = WMStats.JobSummaryTable.data.row(currentElement).data();
            summary.task = currentRowData.task;
            summary.status = currentRowData.status;
            summary.site = currentRowData.site;
            if (summary.site ==="{}"){
                summary.site = {};
            }
            summary.acdcURL = currentRowData.acdcURL;
            summary.exitCode = currentRowData.exitCode;
            
            vm.JobDetail.keys(summary);
            $(WMStats.Globals.Event).triggerHandler(E.AJAX_LOADING_START);
        },
        
        createACDCResubmission: function (currentElement){
            var workflow = $("#job_view div.summary_data").data("workflow");
            var summary = {};
            summary.requestName = workflow;
            //TODO: should be getting most information from request mgr database
            var currentRowData = WMStats.JobSummaryTable.data.row(currentElement).data();
            summary.task = currentRowData.task;
            summary.acdcURL = currentRowData.acdcURL;
            vm.Resubmission.keys(summary);
        }
    };

    var ActiveModelHandler = new TableEventHandler();
        ActiveModelHandler.tableColumnBind('click', "#category_view div.summary_data", "drill",
                                        "populateRequestSummary");
        ActiveModelHandler.tableColumnBind("click", "#request_view div.summary_data", "drill",
                                        "populateJobSummary");
        //ActiveModelHandler.tableRowBind("click","#job_view div.summary_data", 
        //                                "populateJobDetail");
        ActiveModelHandler.tableColumnBind("click","#job_view div.summary_data", "drill",
                                         "populateJobDetail");
        ActiveModelHandler.tableColumnBind("click","#job_view div.summary_data", "acdc",
                                         "createACDCResubmission");
        ActiveModelHandler.tableColumnBind('click', "#category_view div.summary_data", "detail",
                                        "populateCategoryDetail");
        ActiveModelHandler.tableColumnBind('click', "#request_view div.summary_data", "detail",
                                        "populateRequestDetail");

    // actual event binding codes
    // row mouse over/ mouse out events
    $(document).on('mouseover', 'tr', function(event) {
        $(this).addClass('mouseoverRow');
    });
    
    $(document).on('mouseout', 'tr', function(event) { 
        $(this).removeClass('mouseoverRow');
    });

})(jQuery);
WMStats.namespace('JobDetailList');
(function() { 
    
    var stateTransitionFormat = function(state) {
        return "<b>" + state['newstate'] + ":</b> " + 
                WMStats.Utils.utcClock(new Date(state['timestamp'] * 1000)) + 
                ",  " + state['location'];
    };
    
    var inputFileFormat = function(inputFile) {
        return inputFile['lfn'];
    };
    
    var lumiFormat = function(lumis) {
        
        function lumiRangeFormat() {
            if (startLumi == preLumi) {
               lumiFormat.push("[" + startLumi + "]");
            } else {
               lumiFormat.push("[" + startLumi + " - " + preLumi + "]");
            };
        };

        var preLumi = null;
        var startLumi = null;
        var lumiFormat = new Array();
        for (var i in lumis) {
            for (var j in lumis[i]) {
                for (var k in lumis[i][j]) {
                    var currentLumi = Number(lumis[i][j][k]);
                    if (startLumi === null) {
                        startLumi = currentLumi;
                    } else if ((preLumi + 1) !== currentLumi) {
                            lumiRangeFormat();
                            startLumi = currentLumi;
                    };
                    preLumi = currentLumi;
                };
            };
       };
       
       if (startLumi !== null) {
           lumiRangeFormat();
       };
       return lumiFormat;
    };
    
    var logArchiveFormat = function(archiveObj, key) {
        return key;
    };
    
    var format = function (data) {
        var jobDetails = data.getData();
        var requestData = WMStats.ActiveRequestModel.getData();
        var htmlstr = '<div><nav id="jobDetailNav" class="button-group"><ul>';
        var jobDivIDPrefix = "jobDetail-";
        for (var index in jobDetails) {
            var buttonNumber = Number(index) + 1;
            if (buttonNumber == 1) {
                htmlstr += '<li><a href="#' + jobDivIDPrefix + index + '" class="pageButton button-selected">' + buttonNumber +' </a></li>';
            } else {
                htmlstr += '<li><a href="#' + jobDivIDPrefix + index + '" class="pageButton button-unselected">' + buttonNumber +' </a></li>';
            }
        }
        htmlstr += '</nav></ul></div>';
        for (var index in jobDetails) {
            var jobDoc = jobDetails[index];
            if (index === "0") {
                htmlstr += "<div class='jobDetailBox' id='" + jobDivIDPrefix + index + "'>";
            } else {
                htmlstr += "<div class='jobDetailBox hideDiv' id='" + jobDivIDPrefix + index + "'>";
            }
            htmlstr += "<ul>";
            htmlstr += "<li><b>Job Name:</b> " + jobDoc._id + "</li>";
            htmlstr += "<li><b>WMBS job id:</b> " + jobDoc.wmbsid + "</li>";
            htmlstr += "<li><b>Workflow:</b> " + jobDoc.workflow + "</li>";
            htmlstr += "<li><b>Task:</b> " + jobDoc.task + "</li>";
            htmlstr += "<li><b>Status:</b> " + jobDoc.state + "</li>";
            htmlstr += "<li><b>Input dataset:</b> " + WMStats.Utils.getInputDatasets(requestData.getData(jobDoc.workflow)) + "</li>";
            if (typeof jobDoc.site == "object") {
                htmlstr += "<li><b>Site:</b> N/A </li>";
            } else {
                htmlstr += "<li><b>Site:</b> " + jobDoc.site + "</li>";
            }
            htmlstr += "<li><b>Agent:</b> " + jobDoc.agent_name + "</li>";
            htmlstr += "<li><b>ACDC URL:</b> " + jobDoc.acdc_url + "</li>";
            htmlstr += "<li>" + WMStats.Utils.expandFormat(jobDoc.state_history, "State Transition", stateTransitionFormat) + "</li>";
            htmlstr += "<li><b>Exit code:</b> " + jobDoc.exitcode + "</li>";
            htmlstr += "<li><b>Retry count:</b> " + jobDoc.retrycount + "</li>";
            htmlstr += "<li><b>Errors:</b> "; 
            for (var errorType in jobDoc.errors) {
                htmlstr += "<ul>";
                htmlstr += "<li><b>" + errorType + "</b></li>";
                for (var i in jobDoc.errors[errorType]){
                    htmlstr += "<ul>";
                    htmlstr += "<li><b>" + jobDoc.errors[errorType][i].type +" (Exit Code: " + jobDoc.errors[errorType][i].exitCode + ")</b></li>";
                    htmlstr += "<ul>";
                    htmlstr += "<li><pre>" + WMStats.Utils.escapeHtml(jobDoc.errors[errorType][i].details) +"</pre></li>";
                    htmlstr += "</ul>";
                    htmlstr += "</ul>";
                }
                 htmlstr += "</ul>";
            } 
            htmlstr += "</li>";
            
            htmlstr += "<li>" + WMStats.Utils.expandFormat(jobDoc.inputfiles, "Input files", inputFileFormat) + "</li>";
            htmlstr += "<li>" + WMStats.Utils.expandFormat(lumiFormat(jobDoc.lumis), "Lumis") + "</li>";
            
            htmlstr += "<li><b>Output:</b> "; 
            for (var i in jobDoc.output) {
                htmlstr += "<ul>";
				htmlstr += "<li><b>" + jobDoc.output[i].type + "</b></li>";
                htmlstr += "<ul>";
                htmlstr += "<li><b>lfn:</b> " + jobDoc.output[i].lfn +"</li>";
                
                htmlstr += "<li><b>location:</b> ";
                htmlstr += jobDoc.output[i].location;
                if (jobDoc.output[i].type === "logArchive") {
                    htmlstr += "<li class='pfn'><b>pfn:</b> ";
                    // call phedex to get lfn
                }
                /*
                for (var j in jobDoc.output[i].location) {
                    htmlstr += jobDoc.output[i].location[j] + " ";
                }*/
                htmlstr += "</li>";
                htmlstr += "<li><b>checksums: adler32:</b> " + jobDoc.output[i].checksums.adler32 + "</li>";
                htmlstr += "<li><b>size:</b> " + jobDoc.output[i].size + "</li>";
                htmlstr += "</ul>";
                htmlstr += "</ul>";
            }
            htmlstr += "</li>";
            if (jobDoc.worker_node_info) {
                htmlstr += "<li><b>Worker Node:</b>" + jobDoc.worker_node_info.HostName  + "</li>";
            };

            htmlstr += "<li>" + WMStats.Utils.expandFormat(jobDoc.logArchiveLFN, "log archive", logArchiveFormat) + "</li>";
            if (jobDoc.eos_log_url) {
                htmlstr += '<li><b>EOS log URL:</b><a href="' + jobDoc.eos_log_url.replace("eoscmshttp", "eoscmsweb") + '"> download log </a></li>';
            }
            htmlstr += "</ul>";
            htmlstr += "</div>";
        }
        return htmlstr;
    };
    
    
    WMStats.JobDetailList = function (data, containerDiv) {
         $(containerDiv).html(format(data));
    };
    
    // control job Detail
    
    var vm = WMStats.ViewModel;
    
    vm.JobDetail.subscribe("data", function() {
        WMStats.JobDetailList(vm.JobDetail.data(), vm.JobDetail.id());
    });
    /*
    vm.AlertJobDetail.subscribe("data", function() {
        WMStats.JobDetailList(vm.AlertJobDetail.data(), vm.AlertJobDetail.id());
    });
    */
    $(document).on('click', "#jobDetailNav li a", function(event){
        $('div.jobDetailBox').hide();
        $(this.hash).show();
        //TODO call phedex to get PFN
        
        $("#jobDetailNav li a").removeClass("button-selected").addClass("button-unselected");
        $(this).removeClass("button-unselected").addClass("button-selected");
        event.preventDefault();
    });
    
    $(WMStats.Globals.Event).on(WMStats.CustomEvents.PHEDEX_PFN_SUCCESS, 
        function(event, requestName) {
            $('#acdc_submission div.requestDetailBox').append(WMStats.Utils.formatReqDetailUrl(requestName));
    });
})();
WMStats.namespace("AgentDetailList");

(function () {
    
    var componentFormat = function(componentList) {
        var formatStr = "";
        for (var i in componentList) {
            formatStr += "<details> <summary>" + componentList[i].name +"</summary> <ul>";
            formatStr += "<li><b>worker thread:</b> " + componentList[i].worker_name +"</li>";
            formatStr += "<li><b>status:</b> " + componentList[i].state + "</li>";
            //formatStr += "<li><b>error:</b> " + WMStats.Utils.utcClock(new Date(componentList[i].last_error * 1000)) + "</li>";
            formatStr += "<li><b>last updated:</b> " + WMStats.Utils.utcClock(new Date(componentList[i].last_updated * 1000)) + "</li>";
            formatStr += "<li><b>pid:</b> " + componentList[i].pid + "</li>";
            formatStr += "<li><b>error message:</b> <pre>" + componentList[i].error_message + "</pre></li>";
            formatStr += "</ul></details>";
        }
        
        return formatStr;
    };
    
    var diskFullFormat = function(diskList) {
        var formatStr = "<details> <summary> disk list</summary> <ul>";
        for (var i in diskList) {
            formatStr += "<li><b>" + diskList[i].filesystem +"</b>:" + diskList[i].percent +"</li>";
        }
        formatStr += "</ul></details>";
        return formatStr;
    };
    
    var agentReportFormat = function (agentInfo, msgType) {
        var htmlstr = '';
        htmlstr += "<div class='" + msgType + " agent_detail_box'>";
        htmlstr += "<ul>";
        if (agentInfo) {
            agentVersion = '  (' + agentInfo.agent_version + ')'
            htmlstr += "<li><b>agent:</b> " + agentInfo.agent_url + agentVersion + "</li>";
            htmlstr += "<li><b>agent last updated:</b> " + WMStats.Utils.utcClock(new Date(agentInfo.timestamp * 1000)) +" : " +
                              agentInfo.alert.agent_update  + "</li>";
            htmlstr += "<li><b>data last updated:</b> " + agentInfo.alert.data_update  + "</li>";
            htmlstr += "<li><b>status:</b> " + agentInfo.alert.message + "</li>";
            if (agentInfo.agent_team) {
                htmlstr += "<li><b>team:</b> " + agentInfo.agent_team+ "</li>";
            }
        };
        var componentsDown = agentInfo.down_components;
        if (componentsDown && (componentsDown.length > 0)) {
            htmlstr += "<li><b>component errors for:</b> " + componentsDown;
            if (agentInfo.down_component_detail && (agentInfo.down_component_detail.length > 0)) {
                htmlstr += componentFormat(agentInfo.down_component_detail);
            }
            htmlstr +="</li>";
        };

        var diskInfo = agentInfo.disk_warning;
        if (diskInfo && (diskInfo.length > 0)) {
            htmlstr += "<li><b>disk warning:</b> ";
            htmlstr += diskFullFormat(diskInfo);
            htmlstr +="</li>";
           };

        if (agentInfo.proxy_warning) {
            htmlstr += "<li><b>proxy warning:</b> " + agentInfo.proxy_warning + "</li>";
           };

        if (agentInfo.drain_stats) {
            htmlstr += "<li><b>drain statistics:</b></li><ul>";
            htmlstr += "<li>workflows completed (" + agentInfo.drain_stats.workflows_completed + ")</li>";
            if (agentInfo.drain_stats.workflows_completed) {
                htmlstr += "<li>condor running (" + agentInfo.drain_stats.condor_status.running + ")</li>";
                htmlstr += "<li>condor idle (" + agentInfo.drain_stats.condor_status.idle + ")</li>";
                htmlstr += "<li>dbs open blocks (" + agentInfo.drain_stats.upload_status.dbs_open_blocks + ")</li>";
                htmlstr += "<li>dbs not uploaded (" + agentInfo.drain_stats.upload_status.dbs_notuploaded + ")</li>";
                htmlstr += "<li>phedex not uploaded (" + agentInfo.drain_stats.upload_status.phedex_notuploaded + ")</li>";
                for (var key in agentInfo.drain_stats.global_wq_status) {
                    htmlstr += "<li>global WQ in '" + key + "' status (" + agentInfo.drain_stats.global_wq_status[key] + ")</li>";
                }
                for (var key in agentInfo.drain_stats.local_wq_status) {
                    htmlstr += "<li>local WQ in '" + key + "' status (" + agentInfo.drain_stats.local_wq_status[key] + ")</li>";
                }
                for (var key in agentInfo.drain_stats.local_wqinbox_status) {
                    htmlstr += "<li>local WQInbox in '" + key + "' status (" + agentInfo.drain_stats.local_wqinbox_status[key] + ")</li>";
                }
                // only print job information that are != than 0
                htmlstr += "<li>there are no wmbs jobs in the agent, except for...</li>";
                for (var key in agentInfo.WMBS_INFO.wmbsCountByState) {
                    if (agentInfo.WMBS_INFO.wmbsCountByState[key]) {
                        htmlstr += "<li>  jobs in '" + key + "' state (" + agentInfo.WMBS_INFO.wmbsCountByState[key] + ")</li>";
                    }
                }
            }
            htmlstr += "</ul>"
        }


        var dataError = agentInfo.data_error;
        if (dataError && dataError !== 'ok') {
            htmlstr += "<li><b>data collect error:</b> ";
            htmlstr += dataError;
            htmlstr +="</li>";
           };
        
        var couchProcess = agentInfo.couch_process_warning;
        if (couchProcess && couchProcess > 0) {
            htmlstr += "<li><b>Couch Process maxed:</b> ";
            htmlstr += couchProcess;
            htmlstr +="</li>";
           };
        
        htmlstr += "</ul>";
        htmlstr += "</div>";
        return htmlstr;
    };

    var format = function (agentData) {
       var htmlstr = "";
       var agentsWithWarning = agentData.getAlertList();
       for (i in agentsWithWarning) {
           if (agentsWithWarning[i].alert && agentsWithWarning[i].alert.status) {
               htmlstr += agentReportFormat(agentsWithWarning[i], agentsWithWarning[i].alert.status);
           }
       };
       return htmlstr;
    };
    
    WMStats.AgentDetailList = function (data, containerDiv) {
         $(containerDiv).html(format(data));
    };
    
    // controller for this view to be triggered
    var vm = WMStats.ViewModel;
    vm.AgentPage.subscribe("data", function() {
        //TODO get id form the view
        var divID = '#agent_detail';
        WMStats.AgentDetailList(vm.AgentPage.data(), divID);
    });
        
})();
WMStats.namespace('WMStats.CategoryTitle');
(function() { 
    
    var format = function (data) {
        var categoryKey = data;
        
        var htmlstr = "";
        //htmlstr += "<legend>request</legend>";
        htmlstr += "<div class='requestSummaryBox'>";
        htmlstr += "<ul>";
        htmlstr += "<li><b> Category: " + categoryKey + "</b></li>";
        htmlstr += "</ul>";
        htmlstr += "</div>";
        return htmlstr;
    };
    
    WMStats.CategoryTitle = function (data, containerDiv) {
         $(containerDiv).html(format(data));
    };
})();
WMStats.namespace('WMStats.RequestTitle');
(function() { 
    var format = function (data) {
        var workflow;
        var dataType;
        if (typeof data === "string") {
            workflow = data;
            dataType = 0;
        } else {
            workflow = data.getData().workflow;
            dataType = 1;
        };
        var requestInfo = WMStats.ActiveRequestModel.getData().getData(workflow);

        var htmlstr = "";
        //htmlstr += "<legend>request</legend>";
        htmlstr += "<div class='requestSummaryBox'>";
        htmlstr += "<ul>";
        htmlstr += "<li><b>" + workflow + "</b></li>";
        if (dataType == 1) {
            htmlstr += "<li><b>agent: </b>" + requestInfo.agent_url + "</li>";
        };
        htmlstr += "</ul>";
        htmlstr += "</div>";
        return htmlstr;
    };
    
    WMStats.RequestTitle = function (data, containerDiv) {
        if (typeof data === "string" || (typeof data === "object" && data.getData().workflow !== undefined)) {
            $(containerDiv).html(format(data));
        };
    };
})();
WMStats.namespace('WMStats.RequestLogList');
(function() { 
	
	function createRequestLogList(res, containerDiv) {
	    var pat_war = /.*warning.*/;
	    var pat_err = /.*error.*/;
	    // construct list
	    $.each(res, function(i)
	    {
	        var req = $('<div/>')
	            .addClass('request')
	            .text(res[i].request);
	        $(containerDiv).append(req);
	        var obj = res[i];
	        var keys = Object.keys(obj);
	        $.each(keys, function(j) {
	            var thr = $('<div/>')
	                .addClass('thread')
	                .text(keys[j]);
	            $(containerDiv).append(thr);
	            var values = obj[keys[j]];
	            $.each(values, function(k) {
	                var val = values[k];
	                var type = val.type;
	                var msg_cls = 'agent-info';
	                var icon_cls = '';
	                if(val.type.match(pat_war)) {
	                    msg_cls = 'agent-warning';
	                    icon_cls = 'fa fa-bell-o medium agent-warning';
	                } else if(val.type.match(pat_err)) {
	                    msg_cls = 'agent-error';
	                    icon_cls = 'fa fa-exclamation-triangle medium agent-error';
	                } else {
	                    msg_cls = 'agent-info';
	                    icon_cls = 'fa fa-check medium agent-info';
	                }
	                var icon = $('<i/>').addClass(icon_cls).addClass('type-ts-msg');
	                var date = new Date(val.ts);
	                var span = $('<span/>').addClass('type-ts-msg '+msg_cls).text(val.type);
	                var rest = $('<span/>').text(' | '+date.toGMTString()+' | '+val.msg);
	                var ttm = $('<div/>')
	                    .addClass('type-ts-msg')
	                    .add(icon).add(rest).add('<br/>');
	                $(containerDiv).append(ttm);
	            });
	        });
	        $(containerDiv).append('<br/><hr/>');
	    });
	};
    
    var errorFormat = function(logData) {
        //var formatStr = "<div class='fa fa-exclamation-triangle medium agent-error'>";
        var formatStr = "";
        for (var i in logData) {
            formatStr += "<details> <summary> <b>" + logData[i].request + "</b></summary> <ul>";
            formatStr += "<li><b>agent</b>: " + logData[i].agent + "</li>";
            formatStr += "<li><b>thread</b>: " + logData[i].thr + "</li>";
            formatStr += "<li><b>type</b>: " + logData[i].type + "</li>";
            formatStr += "<li><b>update time</b>: " + WMStats.Utils.utcClock(new Date(logData[i].ts * 1000)) + "</li>";
            formatStr += "<li><b>error message</b>: <pre>" + logData[i].messages[0].msg + "</pre></li>";
            formatStr += "<li><b>id</b>: " + logData[i].id + "</li>";
            formatStr += "</ul></details>";
        }
        //formatStr += "</div>";
        return formatStr;
    };
    
    
    var format = function (data) {
       var htmlstr = "";
       htmlstr += errorFormat(data);
       return htmlstr;
    };
    
    WMStats.RequestLogList = function (data, containerDiv) {
         $(containerDiv).html(format(data));
    };
    
     // controller for this view to be triggered
    var vm = WMStats.ViewModel;
    vm.LogDBPage.subscribe("data", function() {
        //TODO get id form the view
        var divID = '#logdb_summary';
        logDBData = vm.LogDBPage.data();
        errors = logDBData.getLogWithLastestError();
        WMStats.RequestLogList(errors, divID);
    });
})();
WMStats.namespace("SiteHistoryGraph");

WMStats.SiteHistoryGraph = function (historyData, containerDiv) {
    
    var siteHistory = JSON.stringify(historyData);
    var htmlList = '<pre>' + siteHistory + '</pre>';
    $(containerDiv).html(htmlList);
};
WMStats.namespace("Controls");
WMStats.Controls = function($){
    var _filterSelector;
    var _categorySelector;
    var vm =  WMStats.ViewModel;
    var vmRegistry = WMStats.ViewModel.Registry;
    
    function setFilter(selector) {
        var inputFilter = '<div name="filter">\
                           workflow: <input name="workflow" value=""></input>\
                           status: <input name="request_status" value=""></input>\
                           run: <input name="run" value=""></input>\
                           </div>';
        $(selector).append(inputFilter);
        _filterSelector = selector + ' div[name="filter"] input';
        
        $(document).on('keyup', selector + " input", 
                function() { WMStats.Utils.delay(function() {
                    //change the view model filter value
                    WMStats.ViewModel.ActiveRequestPage.filter(WMStats.Utils.createInputFilter(_filterSelector));
                    
                }, 300)});
    }
    
    function setCategoryButton(selector) {

        vm.CategoryView.category(vm.RunCategory);
    };
    
    function setViewSwitchButton(selector) {
       var viewSwitchBottons = 
        '<nav id="view_switch_button" class="button-group">\
            <ul><li><a href="#progress" class="nav-button nav-button-selected"> progress </a></li>\
         </nav>';
        
        $(selector).append(viewSwitchBottons);

        $(document).on('click', "#view_switch_button li a", function(event){
            // format is either progress or numJobs need to decouple the name
            var buttonName = this.hash.substring(1);
            var requestFormat;
            if (buttonName === "progress") {
                requestFormat = vm.RequestProgress;
            } 
            vm.RequestView.format(requestFormat);
            // this might not be the efficient way. or directly update the table.
            vm.RequestView.propagateUpdate();
            $("#view_switch_button li a").removeClass("nav-button-selected").addClass("button-unselected");
            $(this).addClass("nav-button-selected");
            event.preventDefault();
        });
    };

   function setAllRequestButton(selector) {
        var requestButtons = 
        '<nav id="all_requests" class="button-group">\
            <ul><li><a href="#" class="nav-button"> all requests </a></li></ul>\
        </nav>';
        
        $(selector).append(requestButtons).addClass("button-group");
        
        $(document).on('click', "#all_requests li a", function(event){
            vm.RequestView.categoryKey("all");
            event.preventDefault();
           });
        
        vm.RequestView.subscribe("categoryKey", function(){
            var buttonSelector = "#all_requests li a";
            if (vm.RequestView.categoryKey() === "all") {
                $(buttonSelector).removeClass("button-unselected").addClass("nav-button-selected");
            } else {
                $(buttonSelector).removeClass("nav-button-selected").addClass("button-unselected");
            }
        }); 
    };
    
    function getFilter() {
        return WMStats.Utils.createInputFilter(_filterSelector);
    };
    
    
    function setTabs(selector) {
        var tabs = '<ul><li class="first"><a href="#category_view">Run</a></li>\
                    <li><a href="#request_view">&#187 Requests</a></li>\
                    <li><a href="#job_view">&#187 Jobs</a></li></ul>';
        
        $(selector).append(tabs).addClass("tabs");
        $(selector + " ul").addClass("tabs-nav");
                
        // add controller for this view
        function changeTab(event, data) {
            $(selector + ' li').removeClass("tabs-selected");
            $(selector + ' a[href="' + data.id() +'"]').parent().addClass("tabs-selected");
        }
        // viewModel -> view control
        vm.ActiveRequestPage.subscribe("view", changeTab);
        
        // view -> viewModel control
        $(document).on('click', selector + " li a", function(event){
            vm.ActiveRequestPage.view(vmRegistry[this.hash]);
            //vm.CategoryView.category(vm.RunCategory);
            event.preventDefault();
        });
    };
    
    return {
        setFilter: setFilter,
        setTabs: setTabs,
        setCategoryButton: setCategoryButton,
        setAllRequestButton: setAllRequestButton,
        setViewSwitchButton: setViewSwitchButton,
        getFilter: getFilter,
        requests: "requests",
        sites: "sites",
        run: "run"
    };
}(jQuery);
WMStats.namespace("ActiveRequestTable");

WMStats.ActiveRequestTable = function (requestData, containerDiv) {
    
    var formatReqDetailUrl = WMStats.Utils.formatReqDetailUrl;
    var formatWorkloadSummarylUrl = WMStats.Utils.formatWorkloadSummarylUrl;
    var tableConfig = {
        "pageLength": 25,
        "scrollX": "",
        "columns": [
            {"title": "D", 
             "defaultContent": 0,
             "render": function (data, type, row, meta) {
             		      if (type === "display") {
                            return WMStats.Utils.formatDetailButton("detail", row.skipped);
                          }
                          if (row.skipped) {
                          	return 1;
                          }
                        }},
            {"title": "L", 
             "defaultContent": 0,
             "render": function (data, type, row, meta) {
                            return WMStats.Utils.formatDetailButton("drill");
                        }},
            { "data": "workflow", "title": "workflow"},
            /*
            { "defaultContent": "new",
              "title": "status", 
              "render": function (data, type, row, meta) {
                           
                           var status = row.request_status[row.request_status.length -1].status;
                           var reqSummary = requestData.getSummary(row.workflow);
                           var totalJobs = reqSummary.getWMBSTotalJobs();
                           if (totalJobs > 0) {
                               if (reqSummary.getTotalSubmitted() > 0) {
                                   if (reqSummary.getJobStatus("submitted.running") > 0) {
                                       status = "running";
                                   } else {
                                       status = "submitted";
                                   }
                                   
                               } else {
                                   if (reqSummary.getTotalPaused() > 0) {
                                       status = "paused";
                                   }
                               }
                           }
                           return status
                          }
            },*/
            { "title": "status",
              "render": function (data, type, row, meta) {
              				if (type === "display") {
                            	return formatWorkloadSummarylUrl(row.workflow, 
                                	row.request_status[row.request_status.length -1].status);
                            }
                            return row.request_status[row.request_status.length -1].status;
                          }
            },
            { "title": "duration",
              "render": function (data, type, row, meta) {
              	            if (type === "display") {
	                            var currentTime = Math.round(new Date().getTime() / 1000);
	                            var startTime = row.request_status[row.request_status.length -1].update_time;
	                            return WMStats.Utils.formatDuration(currentTime - startTime);
                            }
                            return row.request_status[row.request_status.length -1].update_time;
                          },
            },
            { "defaultContent": 0,
              "title": "job progress", 
              "render": function (data, type, row, meta) {
                            var reqSummary = requestData.getSummary(row.workflow);
                            var totalJobs = reqSummary.getWMBSTotalJobs() || 1;
                            var result = (reqSummary.getJobStatus("success") + reqSummary.getTotalFailure()) /
                                     totalJobs * 100;
                            return  (result.toFixed(1) + "%");
                         },
              "type": "num-fmt"
            },
            { "defaultContent": 0,
              "title": "submitted", 
              "render": function (data, type, row, meta) {
                           var result = requestData.getKeyValue(row.workflow, "status.submitted.first", 0);
                           result += requestData.getKeyValue(row.workflow, "status.submitted.retry", 0);
                           return result;
                          }
            },
			{ "defaultContent": 0,
              "title": "pending", 
              "render": function (data, type, row, meta) {
                           var result = requestData.getKeyValue(row.workflow, "status.submitted.pending", 0);
                           return result;
                          }
            },
            { "defaultContent": 0,
              "title": "running", 
              "render": function (data, type, row, meta) {
                           var result = requestData.getKeyValue(row.workflow, "status.submitted.running", 0);
                           return result;
                          }
            },
            { "defaultContent": 0,
              "title": "cool off ", 
              "render": function (data, type, row, meta) {
                            var reqSummary = requestData.getSummary(row.workflow);
                            return reqSummary.getTotalCooloff();
                          }
            },
            { "defaultContent": 0,
              "title": "paused", 
              "render": function (data, type, row, meta) {
                            var reqSummary = requestData.getSummary(row.workflow);
                            return  reqSummary.getTotalPaused();
                          }
            },
            { "defaultContent": 0,
              "title": "failed", 
              "render": function (data, type, row, meta) {
                            var reqSummary = requestData.getSummary(row.workflow);
                            return  reqSummary.getTotalFailure();
                          }
            },
            { "defaultContent": 0,
              "title": "run", 
              "render": function (data, type, row, meta) {
                            return requestData.getKeyValue(row.workflow, "run", 0);
                          }
            }
            //TODO add more data (consult dataops)
        ]
    };
    
    function runNumerDesc(a, b) {
        return (Number(b.run) - Number(a.run));
    }
    
    tableConfig.data = requestData.getList(runNumerDesc);
    
    return WMStats.Table(tableConfig).create(containerDiv, null);
};
WMStats.namespace("TaskSummaryTable");

WMStats.TaskSummaryTable = function (data, containerDiv) {
    
    var taskData = data;
    var _activePageData = WMStats.ViewModel.ActiveRequestPage.data();
    var tableConfig = {
        "dom": '<"top"plf>rt<"bottom"ip>',
        "pageLength": 25,
        "columns": [
            { "render": function (data, type, row, meta) {
                if (type == 'display') {
                    var taskList = row[0].split('/');
                    return taskList[taskList.length - 1];
                }
                return row;
            }, "title": "task", "sWidth": "150px"},
            { "defaultContent": 0,
              "title": "created",
              "render": function (data, type, row, meta) {
                            var taskSummary = taskData.getSummary(row[0]);
                            var jobs = taskSummary.getWMBSTotalJobs();
                            return jobs;
                          }
            },
            { "defaultContent": 0,
              "title": "queued", 
              "render": function (data, type, row, meta) {
                            var taskSummary = taskData.getSummary(row[0]);
                            var jobs = taskSummary.getTotalQueued();
                            return jobs;
                          }
            },
            { "defaultContent": 0,
              "title": "pending ", 
              "render": function (data, type, row, meta) {
                                var taskSummary = taskData.getSummary(row[0]);
                                var jobs = taskSummary.getPending();
                                return jobs;
                              }
            },
            { "defaultContent": 0,
              "title": "running ", 
              "render": function (data, type, row, meta) {
                                var taskSummary = taskData.getSummary(row[0]);
                                var jobs = taskSummary.getRunning();
                                return jobs;
                              }
            },
            { "defaultContent": 0,
              "title": "success ", 
              "render": function (data, type, row, meta) {
                                var taskSummary = taskData.getSummary(row[0]);
                                var jobs = taskSummary.getJobStatus("success");
                                return jobs;
                              }
            },
            { "defaultContent": 0,
              "title": "failure ", 
              "render": function (data, type, row, meta) {
                                var taskSummary = taskData.getSummary(row[0]);
                                var jobs = taskSummary.getTotalFailure();
                                return jobs;
                              }
            },
            { "defaultContent": 0,
              "title": "cool off ", 
              "render": function (data, type, row, meta) {
                                var taskSummary = taskData.getSummary(row[0]);
                                var jobs = taskSummary.getTotalCooloff();
                                return jobs;
                              }
            },
            { "defaultContent": 0,
              "title": "paused", 
              "render": function (data, type, row, meta) {
                                var taskSummary = taskData.getSummary(row[0]);
                                var jobs = taskSummary.getTotalPaused();
                                return jobs;
                              }
            },
            { "defaultContent": 0,
              "title": "events processed", 
              "render": function (data, type, row, meta) {
                           var outputEvents = taskData.getSummary(row[0]).getAvgEvents() || 0;
                           return outputEvents;
                          }
            },
            { "defaultContent": 0,
              "title": "lumi processed", 
              "render": function (data, type, row, meta) {
                           var outputLumis = taskData.getSummary(row[0]).getAvgLumis();
                           return outputLumis;
                          }
            }
         ],
         "order": [[1, 'asc']]
    };
    
    var taskNamesArray = [];
    for (var taskNames in data.getData()){
    	taskNamesArray.push([taskNames]);
    }
    tableConfig.data = taskNamesArray;
    
    var filterConfig = {};
    
    return WMStats.Table(tableConfig).create(containerDiv, filterConfig);
};

(function() {
    var vm = WMStats.ViewModel;
    var E = WMStats.CustomEvents;
    
    vm.JobView.subscribe("requestName", function() {
        // empty job detail and resubmission window first
        vm.JobDetail.keys(null, true);
        $(vm.JobDetail.id()).empty();
        $(vm.Resubmission.id()).empty();
 	    // need to create the lower level view
        var divSelector = vm.JobView.id() + " div.task_summary";
        var tasks = WMStats.ActiveRequestModel.getData().getTasks(vm.JobView.requestName());
        WMStats.TaskSummaryTable(tasks, divSelector);
        $(WMStats.Globals.Event).triggerHandler(E.AJAX_LOADING_START);
    });
    
})();
WMStats.namespace("RunSummaryTable");

WMStats.RunSummaryTable = function (data, containerDiv) {
    
    var tableConfig = {
        "pageLength": 25,
        "scrollX": "",
        "columns": [
            {"title": "D", 
             "defaultContent": 0,
             "render": function (data, type, row, meta) {
                            return WMStats.Utils.formatDetailButton("detail");
                        }},
            {"title": "L", 
             "defaultContent": 0,
             "render": function (data, type, row, meta) {
                            return WMStats.Utils.formatDetailButton("drill");
                        }},
            { "data": "key", "title": "run"},               
            { "render": function (data, type, row, meta) { 
                              return row.summary.summaryStruct.numRequests;
                           }, "title": "requests", "defaultContent": 0, 
            },
            { "render": function (data, type, row, meta) { 
                              return row.summary.summaryStruct.runStatus;
                           }, "title": "run status", "defaultContent": "Active", 
            },
            { "render": function (data, type, row, meta) { 
                              return row.summary.getJobStatus("success");
                           }, "title": "success", "defaultContent": 0, 
            },
            { "render": function (data, type, row, meta) { 
                              return row.summary.getTotalFailure();
                           }, "title": "failure", "defaultContent": 0, 
            },
            { "defaultContent": 0,
              "title": "job progress", 
              "render": function (data, type, row, meta) { 
                            var totalJobs = row.summary.getWMBSTotalJobs() || 1;
                            var result = (row.summary.getJobStatus("success") + 
                                          row.summary.getTotalFailure()) /
                                          totalJobs * 100;
                            if (type === 'display') {
                                return result.toFixed(1) + "%";
                            }
                            return result.toFixed(1);
                          }
            },
            { "render": function (data, type, row, meta) { 
                              return row.summary.getJobStatus("submitted.pending");
                           }, "title": "pending", "defaultContent": 0, 
            },
            { "render": function (data, type, row, meta) { 
                              return row.summary.getJobStatus("submitted.running");
                           }, "title": "running", "defaultContent": 0, 
            },
            { "render": function (data, type, row, meta) { 
                              return row.summary.getTotalCooloff();
                           }, "title": "cool off", "defaultContent": 0, 
            },
            { "render": function (data, type, row, meta) { 
                              return row.summary.getTotalPaused();
                           }, "title": "paused", "defaultContent": 0, 
            }
        ]
    };
    
    function runNumerDesc(a, b) {
        return (Number(b.key) - Number(a.key));
    }
    
    tableConfig.data = data.getList(runNumerDesc);
    
    var filterConfig = {};
    
    return WMStats.Table(tableConfig).create(containerDiv,filterConfig);
};
//register mapping
(function(vm){
    WMStats.CategorySummaryMap.add(vm.SiteCategory.name(), WMStats.SiteSummary);
    WMStats.CategoryTableMap.add(vm.SiteCategory.name(), WMStats.SiteSummaryTable);
    
    WMStats.CategoryTableMap.add(vm.RequestView.categoryName, 
                                {'progress': WMStats.ActiveRequestTable});
    WMStats.CategorySummaryMap.add(vm.RunCategory.name(), WMStats.RunSummary);
    WMStats.CategoryTableMap.add(vm.RunCategory.name(), WMStats.RunSummaryTable);
})(WMStats.ViewModel);WMStats.namespace('RequestDetailList');
(function() { 
    
    var vm = WMStats.ViewModel;
    var transitionFormat = function(dataArray, maxLength, summaryStr) {
        var htmlstr = "";
        if (dataArray == undefined || dataArray.length == undefined ||
            dataArray.length <= maxLength) {
         
            htmlstr +=  dataArray;
         } else {
            htmlstr += "<details> <summary>" + summaryStr +"</summary><ul>";  
            for (var i in dataArray) {
                htmlstr += "<li> <b>" + dataArray[i].status + ":</b> " + WMStats.Utils.utcClock(new Date(dataArray[i].update_time * 1000)) + "</li>";
            }
            htmlstr += "</ul></details>";
        }
        return htmlstr;
    };
    
    var format = function (requestStruct) {
        var htmlstr = '<div class="closingButton">X</div>';
        var reqDoc = requestStruct.requests;
        var reqSummary = requestStruct.summary;
        
        htmlstr += "<div class='requestDetailBox'>";
        htmlstr += "<ul>";
        if (reqDoc) {
            
            htmlstr += "<li><b>category:</b> " + requestStruct.key + "</li>";
            htmlstr += "<li><b>state transition</b> " + transitionFormat(reqDoc[requestStruct.key].request_status, 0, "State List") + "</li>";
            htmlstr += "<li><b>queued (first):</b> " + reqSummary.getJobStatus("queued.first", 0) + "</li>";
            htmlstr += "<li><b>queued (retried):</b> " + reqSummary.getJobStatus("queued.retry", 0) + "</li>";
            htmlstr += "<li><b>created:</b> " + reqSummary.getWMBSTotalJobs() + "</li>";
            htmlstr += "<li><b>paused jobs:</b> " + reqSummary.getTotalPaused() + "</li>";
            htmlstr += "<li><b>cooloff jobs:</b> " + reqSummary.getTotalCooloff() + "</li>";
            htmlstr += "<li><b>submitted:</b> " + reqSummary.getTotalSubmitted() + "</li>";
            htmlstr += "<li><b>pending:</b> " + reqSummary.getJobStatus("submitted.pending", 0) + "</li>";
            htmlstr += "<li><b>running:</b> " + reqSummary.getJobStatus("submitted.running", 0) + "</li>";
            htmlstr += "<li><b>failure:</b> " + reqSummary.getTotalFailure()  + "</li>";
            htmlstr += "<li><b>success:</b> " + reqSummary.getJobStatus("success", 0) + "</li>";
            if (reqDoc.skipped) {
        		htmlstr += "<li>" + WMStats.Utils.expandFormat(reqDoc.getSkippedDetail(), "Skipped Summary", siteStatusFormat) + "</li>";
        	}
        }
        htmlstr += "</ul>";
        htmlstr += "</div>";
        return htmlstr;
    };
    
    WMStats.RequestDetailList = function (workflow, containerDiv) {
        var allRequests = vm.ActiveRequestPage.data();
        var reqDoc = allRequests.getData(workflow);
        var reqSummary = allRequests.getSummary(workflow);
        var requests = {};
        requests[workflow] = reqDoc;
        var data = {key: workflow, requests: requests, summary: reqSummary};
        $(containerDiv).html(format(data));
        $(containerDiv).show("slide", {}, 500);
        vm.RequestDetail.open = true;
    };
    
    var vm = WMStats.ViewModel;

    vm.RequestDetail.subscribe("requestName", function() {
        WMStats.RequestDetailList(vm.RequestDetail.requestName(), vm.RequestDetail.id());
    });
})();
WMStats.namespace("RequestAlertGUI");

WMStats.RequestAlertGUI = function (requestData, containerDiv) {
    
     function displayAlert(errorArray, title) {
        if (errorArray.length > 0) {
            htmlList += '<fieldset><legend>' + title + '</legend> <div><ul>';
        
            for (var i in errorArray) {
                var workflow = errorArray[i].getName();
                var summary = errorArray[i].getSummary();
                var running = summary.getRunning();
                var pending = summary.getPending();
                var cooloff = summary.getTotalCooloff();
                var failure =summary.getTotalFailure();
                var success = summary.getJobStatus("success");
                var sPaused = summary.getJobStatus("paused.submit");
                var jPaused = summary.getJobStatus("paused.job");
                var cPaused = summary.getJobStatus("paused.create");
                var lastState = errorArray[i].getLastStateAndTime();
                var lastStatus = "N/A";
                var lastUpdate = "N/A";
                if (lastState !== null) {
                	lastStatus = lastState.status;
                	lastUpdate = WMStats.Utils.utcClock(new Date(lastState.update_time * 1000));
                } 
                htmlList += ('<li> <a class="requestAlert">' + workflow + "</a>: status:" + 
                             lastStatus + " (" + lastUpdate + "), cooloff " + cooloff + 
                             " submit paused:" + sPaused + " job paused:" + jPaused + 
                             " create paused:" + cPaused + 
                             " failure:" + failure + " success:" + success +
                             " running:" + running + " pending:" + pending + '</li>');
            }
            htmlList += "</ul></div></fieldset>";
            
            errorFlag = true;
        };
    };
    
    
    function mapType(workflowName){
        // TODO: this is a hack to get jobType rely on the conventsion
        // need a better solution
        var workflowType = workflowName.split("_")[0].toLowerCase();
        if (workflowType == "express") {
            return 1;
        } else if (workflowType == "repack") {
            return 2;
        } else if (workflowType == "promptreco") {
            return 3;
        }
        return 4;
    }
    
    function alertSort(a, b) {
        var aRun = a.run;
        var bRun = b.run;
        if (aRun == bRun) {
            var aType = mapType(a.workflow);
            var bType = mapType(b.workflow);
            if ( aType == bType){
                var aJobs = a.getSummary().getTotalPaused() + a.getSummary().getTotalCooloff();
                var bJobs = b.getSummary().getTotalPaused() + b.getSummary().getTotalCooloff();
                return bJobs - aJobs;
            } else {
                return aType - bType;
            }
        } else {
            return aRun - bRun;
        }
    }
    
    var alertRequests = requestData.getRequestAlerts();
    var errorFlag = false;

    var htmlList = "";
    
    if (alertRequests.cPaused.length > 0) {
        var paused = alertRequests.cPaused.sort(alertSort);
        displayAlert(paused, "create paused");
    };
    if (alertRequests.sPaused.length > 0) {
        var paused = alertRequests.sPaused.sort(alertSort);
        displayAlert(paused, "submit paused");
    };
    if (alertRequests.jPaused.length > 0) {
        var paused = alertRequests.jPaused.sort(alertSort);
        displayAlert(paused, "job paused");
    };
    if (alertRequests.configError.length > 0) {
        var configError = alertRequests.configError.sort(alertSort);
        displayAlert(configError, "Config Error");
    };
    if (alertRequests.siteError.length > 0) {
        var siteError = alertRequests.siteError.sort(alertSort);
        displayAlert(siteError, "Site Error");
        
    };    

    
    $(containerDiv).addClass("request_error_box");
    if (errorFlag) {
        $(containerDiv).removeClass("stable warning").addClass("error").html(htmlList);
    } else {
        $(containerDiv).removeClass("warning error").addClass("stable").html("request alarm");
    }
};

// controller for request alert view
(function() {
    
     var vm = WMStats.ViewModel;
     vm.RequestAlertPage.subscribe("data", function(){
        var divSelector = vm.RequestAlertPage.id() + " div.summary_data";
        WMStats.RequestAlertGUI(vm.RequestAlertPage.data(), divSelector);
     });
    
     $(document).on('click', 'a.requestAlert', function() {
        var workflow = $(this).text();
        vm.JobView.requestName(workflow);
        //TODO: this cause calling one more time for retrieving data
        vm.ActiveRequestPage.view(vm.JobView);
        vm.page(vm.ActiveRequestPage);
        
        $(this).addClass('reviewed');
    });
})();

WMStats.namespace('CategoryDetailList');
(function() { 
    var format = function (requestStruct) {
        var htmlstr = '';
        var reqDoc = requestStruct.requests;
        var reqSummary = requestStruct.summary;
        
        htmlstr += "<div class='requestDetailBox'>";
        htmlstr += "<ul>";
        if (reqDoc) {
            
            htmlstr += "<li><b>category:</b> " + requestStruct.key + "</li>";
            htmlstr += "<li><b>queued (first):</b> " + reqSummary.getJobStatus("queued.first", 0) + "</li>";
            htmlstr += "<li><b>queued (retried):</b> " + reqSummary.getJobStatus("queued.retry", 0) + "</li>";
            htmlstr += "<li><b>created:</b> " + reqSummary.getWMBSTotalJobs() + "</li>";
            htmlstr += "<li><b>paused jobs:</b> " + reqSummary.getTotalPaused() + "</li>";
            htmlstr += "<li><b>cooloff jobs:</b> " + reqSummary.getTotalCooloff() + "</li>";
            htmlstr += "<li><b>submitted:</b> " + reqSummary.getTotalSubmitted() + "</li>";
            htmlstr += "<li><b>pending:</b> " + reqSummary.getJobStatus("submitted.pending", 0) + "</li>";
            htmlstr += "<li><b>running:</b> " + reqSummary.getJobStatus("submitted.running", 0) + "</li>";
            htmlstr += "<li><b>failure:</b> " + reqSummary.getTotalFailure()  + "</li>";
            htmlstr += "<li><b>success:</b> " + reqSummary.getJobStatus("success", 0) + "</li>";
        }
        htmlstr += "</ul>";
        htmlstr += "</div>";
        return htmlstr;
    };
    
    WMStats.CategoryDetailList = function (data, containerDiv) {
         $(containerDiv).html(format(data));
    };
    
    var vm = WMStats.ViewModel;
    
    vm.CategoryDetail.subscribe("data", function() {
        WMStats.CategoryDetailList(vm.CategoryDetail.data(), vm.CategoryDetail.id());
    });
})();
WMStats.namespace('RequestSummaryList');
(function() { 
    var format = function (summary) {
        var summaryStruct = summary.summaryStruct;
        var htmlstr = "";
        htmlstr += "<div class='requestSummaryBox'>";
        htmlstr += "<ul>";
        htmlstr += "<li><b>requests:</b> " + summary.summaryStruct.length + "</li>";
        htmlstr += "<li><b>created:</b> " + summary.getWMBSTotalJobs() + "</li>";
        htmlstr += "<li><b>cooloff:</b> " + summary.getTotalCooloff() + "</li>";
        htmlstr += "<li><b>success:</b> " + summary.getJobStatus('success') + "</li>";
        htmlstr += "<li><b>failure:</b> " + summary.getTotalFailure() + "</li>";
        htmlstr += "<li><b>queued:</b> " + summary.getTotalQueued() + "</li>";
        htmlstr += "<li><b>running:</b> " + summary.getJobStatus('submitted.running') + "</li>";
        htmlstr += "<li><b>pending:</b> " + summary.getJobStatus('submitted.pending') + "</li>";
        htmlstr += "</ul>";
        htmlstr += "</div>";
        return htmlstr;
    };
    
    WMStats.RequestSummaryList = function (data, containerDiv) {
         $(containerDiv).html(format(data));
    };
    
    var vm = WMStats.ViewModel;
    
    vm.ActiveRequestPage.subscribe("data", function() {
            var filteredData = vm.ActiveRequestPage.data();
            WMStats.RequestSummaryList(filteredData.getSummary(), "#filter_summary");
       });
})();
WMStats.namespace('RequestDataList');
(function() { 
    var format = function (summary) {
        var summaryStruct = summary.summaryStruct;
        var htmlstr = "";
        htmlstr += "<div class='requestSummaryBox'>";
        htmlstr += "<ul>";
        htmlstr += "<li> requests: " + summary.summaryStruct.length + "</li>";
        htmlstr += "<li> total created: " + summary.getWMBSTotalJobs() + "</li>";
        htmlstr += "<li> paused: " + summary.getTotalPaused() + "</li>";
        htmlstr += "<li> cooloff: " + summary.getTotalCooloff() + "</li>";
        htmlstr += "<li> success: " + summary.getJobStatus('success') + "</li>";
        htmlstr += "<li> failure: " + summary.getTotalFailure() + "</li>";
        htmlstr += "<li> queued: " + summary.getTotalQueued() + "</li>";
        htmlstr += "<li> running: " + summary.getJobStatus('submitted.running') + "</li>";
        htmlstr += "<li> pending: " + summary.getJobStatus('submitted.pending') + "</li>";
        htmlstr += "</ul>";
        htmlstr += "</div>";
        return htmlstr;
    };
    
    WMStats.RequestDataList = function (data, containerDiv) {
         $(containerDiv).html(format(data));
    };
})();
WMStats.namespace("_ModelBase");

WMStats._ModelBase = function(initView, options, dataStruct) {

    this._initialView = initView;
    this._options = options;
	this._dataStruct = dataStruct;
    this._trigger = null;
    this._data = null;
    // default _dbSource is wmstats database
    this._dbSource = WMStats.Couch;
};

WMStats._ModelBase.prototype = {

    setTrigger: function(triggerName) {
        this._trigger = triggerName;
    },
    
    getData: function() {
        return this._data;
    },
    
    dataReady: function(data) {
        this._data = this._dataStruct(data);
        if (this._trigger instanceof Array){
            for (var i in this._trigger) {
                jQuery(WMStats.Globals.Event).triggerHandler(this._trigger[i], this._data);
            }
        }else{
            jQuery(WMStats.Globals.Event).triggerHandler(this._trigger, this._data);
        }
        
    },

    retrieveData: function (view, options) {
        if (options === undefined){
            var options = this._options;
        }
        if (view === undefined) {
            var view = this._initialView;
        }
        
        if (view === "allDocs") {
            return this.retrieveAllDocs(options);
        } else if (view === "doc") {
            return this.retrieveDoc(options);
        }else {
            return this._dbSource.view(view, options, 
                               jQuery.proxy(this.callback, this));
        }
       
    },
    
    retrieveAllDocs: function (options) {
        if (options === undefined){
            var options = this._options;
        }
        return this._dbSource.allDocs(options, jQuery.proxy(this.callback, this));
    },
    
    retrieveDoc: function (docID) {
        return this._dbSource.openDoc(docID, jQuery.proxy(this.callback, this));
    },
    
    setDBSource: function(dbSource) {
        this._dbSource = dbSource;
    },
    callback: function (data) {
        // use current object context
        return this.dataReady(data);
    },
    
    clearData: function () {
        delete this._data;
    }
};
WMStats.namespace("_AjaxModelBase");

WMStats._AjaxModelBase = function(uri, dataStruct) {
	this._uri = uri;
	this._dataStruct = dataStruct;
    this._data = null;
    // default _dbSource is wmstats database
};

WMStats._AjaxModelBase.prototype = {
	
	setTrigger: function(triggerName) {
        this._trigger = triggerName;
    },
    
    getData: function() {
        return this._data;
    },
    
    dataReady: function(data) {
        this._data = this._dataStruct(data);
        if (this._trigger instanceof Array){
            for (var i in this._trigger) {
                jQuery(WMStats.Globals.Event).triggerHandler(this._trigger[i], this._data);
            }
        }else{
            jQuery(WMStats.Globals.Event).triggerHandler(this._trigger, this._data);
        }
        
    },
    
	retrieveData: function(args) {
            $.ajax(this._uri, 
                   {type: 'GET',
                    //accept: {json: "application/json"},
                    //contentType: "application/json",
                    headers: {"Accept": "application/json"},
                    processData: false,
                    data: args || "",
                    success: jQuery.proxy(this.callback, this),
                    error: function(jqXHR, textStatus, errorThrown){
                            console.log("call fails, response: " + jqXHR.responseText);
                        }
                    });
    },
  
    callback: function (data, textStatus, jqXHR) {
        // use current object context
        return this.dataReady(data);
    },
    
    clearData: function () {
        delete this._data;
    }
};
WMStats.namespace("_RequestModelBase");

WMStats._RequestModelBase = function(initView, options) {

    this._initialView = initView || 'bystatus';
    this._options = options || {'include_docs': true};
    this._data = null;
    this._trigger = "requestReady";
    this._dbSource = WMStats.Couch;
};
//Class method
WMStats._RequestModelBase.keysFromIDs = function(data) {
        var keys = [];
        for (var i in data.rows){
            if (data.rows[i].value && data.rows[i].value.id) {
                keys.push(data.rows[i].value.id);
            } else {
                keys.push(data.rows[i].id);
            }
        }
        return keys;
    };

WMStats._RequestModelBase.requestAgentUrlKeys = function(requestList, requestAgentData) {
        var keys = {};
        var requestAgentUrlList = [];
        for (var i in requestAgentData.rows){
            var request = requestAgentData.rows[i].key[0];
            if (!keys[request]) {
                keys[request] = [];
            }
            keys[request].push(requestAgentData.rows[i].key[1]);
        }
        
        for (var j=0; j < requestList.length; j++) {
            for (var k in keys[requestList[j]]) {
                requestAgentUrlList.push([requestList[j], keys[requestList[j]][k]]);
            }
        }
        return requestAgentUrlList;
   };

WMStats._RequestModelBase.prototype = {
    
    setInitView: function(initView) {
        this._initialView  = initView;
    },
    
    setTrigger: function(triggerName) {
        this._trigger = triggerName;
    },
    
    setDBSource: function(dbSource) {
        this._dbSource = dbSource;
    },
    
    // deprecated
    getData: function() {
        return this._data;
    },
    
    getRequests: function() {
        return this._data;
    },
    
    _getRequestDetailsAndTriggerEvent: function (agentIDs, reqmgrData, objPtr) {
        var options = {'keys': WMStats._RequestModelBase.keysFromIDs(agentIDs), 'reduce': false, 
                       'include_docs': true};
        WMStats.Couch.allDocs(options,
              function(agentData) {
                  //start loading sign disable the filter
                  jQuery(WMStats.Globals.Event).triggerHandler(WMStats.CustomEvents.LOADING_DIV_START);
                  // combine reqmgrData(reqmgr_request) and 
                  // agent_request(agentData) data 
                  var requestCache = WMStats.Requests();
                  requestCache.updateBulkRequests(reqmgrData.rows);
                  requestCache.updateBulkRequests(agentData.rows);
                  
                  // set the data cache
                  objPtr._data = requestCache;
                  // trigger custom events
                  jQuery(WMStats.Globals.Event).triggerHandler(objPtr._trigger, objPtr._data);
              });
    },

    _getLatestRequestAgentUrlAndCreateTable: function (overviewData, keys, objPtr) {
        var options = {"keys": keys, "reduce": false};
        WMStats.Couch.view('latestRequest', options,
              function(agentIDs) {
                  objPtr._getRequestDetailsAndTriggerEvent(agentIDs, overviewData, objPtr);
              });
    },
        
    _getLatestRequestIDsAndCreateTable: function (overviewData, objPtr) {
        /*
         * get list of request ids first from the couchDB then 
         * get the details of the requests.
         */
        var options = {"reduce": true, "group": true, "descending": true};
        var requestList =  WMStats._RequestModelBase.keysFromIDs(overviewData);
        WMStats.Couch.view('requestAgentUrl', options,
              function(requestAgentUrlData) {
                  var keys = WMStats._RequestModelBase.requestAgentUrlKeys(requestList, requestAgentUrlData);
                  objPtr._getLatestRequestAgentUrlAndCreateTable(overviewData, keys, objPtr);
               });
    },

    retrieveData: function (viewName, options) {
        
        if (!viewName) {viewName = this._initialView;}
        if (!options) {options = WMStats.Utils.cloneObj(this._options);}
        var objPtr = this;
        if (viewName == "allDocs") {
            this._dbSource.allDocs(options, function (overviewData) {
                objPtr._getLatestRequestIDsAndCreateTable(overviewData, objPtr);
            });
        } else {
            this._dbSource.view(viewName, options,  function (overviewData) {
                objPtr._getLatestRequestIDsAndCreateTable(overviewData, objPtr);
            });
        }
    },

    clearRequests: function () {
        delete this._data;
    }
};
WMStats.namespace("JobSummaryModel");

WMStats.JobSummaryModel = new WMStats._ModelBase('jobsByStatusWorkflow', {}, 
                                          WMStats.JobSummary);

WMStats.JobSummaryModel.setRequest = function(workflow) {
    this._options = {'reduce': true, 'group_level': 8, 'startkey':[workflow], 
                   'endkey':[workflow, {}]};
};
WMStats.JobSummaryModel.setTrigger([WMStats.CustomEvents.JOB_SUMMARY_READY,
                                    WMStats.CustomEvents.LOADING_DIV_END]);

/*
WMStats.JobSummaryModel.retrieveData = function(workflow) {
    WMStats.JobSummaryModel.setRequest(workflow);
    WMStats.JobSummaryModel.prototype.retrieveData();
}
*/WMStats.namespace("JobDetailModel");

WMStats.JobDetailModel = new WMStats._ModelBase('jobsByStatusWorkflow', {}, 
                                    WMStats.JobDetails);

WMStats.JobDetailModel.setOptions = function(summary) {
    var startkey = null;
    if ((typeof summary.site) == "object") {
        startkey = [summary.workflow, summary.task, summary.status, summary.exitCode];
    } else if (!summary.acdcURL) {
        startkey = [summary.workflow, summary.task, summary.status, summary.exitCode, summary.site];
    } else {
        startkey = [summary.workflow, summary.task, summary.status, summary.exitCode, summary.site, summary.acdcURL];
    }
    this._options= {'include_docs': true, 'reduce': false, 
              'startkey': startkey,
              'endkey': [summary.workflow, summary.task, summary.status, summary.exitCode, summary.site, {}],
              'limit': 10};
};

WMStats.JobDetailModel.setTrigger([WMStats.CustomEvents.JOB_DETAIL_READY,
                                  WMStats.CustomEvents.LOADING_DIV_END]);
WMStats.namespace("AgentModel");
WMStats.AgentModel = new WMStats._ModelBase('agentInfo', {}, WMStats.Agents);
WMStats.AgentModel.setTrigger(WMStats.CustomEvents.AGENTS_LOADED);
WMStats.namespace("WorkloadSummaryModel");

WMStats.WorkloadSummaryModel = new WMStats._ModelBase('', 
                                        {"reduce": false, 'include_docs': true}, 
                                        WMStats.WorkloadSummary);

WMStats.WorkloadSummaryModel.setDBSource(WMStats.WorkloadSummaryCouch);
WMStats.WorkloadSummaryModel.setTrigger(WMStats.CustomEvents.WORKLOAD_SUMMARY_READY);
WMStats.namespace("HistoryModel");

WMStats.HistoryModel = new WMStats._ModelBase('requestHistory', {}, WMStats.History);

WMStats.HistoryModel.setOptions = function() {
    var current = Math.round((new Date()).getTime() / 1000);
    this._options= {'reduce': false,
                    'startkey': [current, {}],
                    'endkey': [current - 600*60*24],
                    'include_docs': true,
                    'descending': true};
};

WMStats.HistoryModel.setTrigger(WMStats.CustomEvents.HISTORY_LOADED);
WMStats.namespace("ActiveRequestModel");
WMStats.ActiveRequestModel = function() {

    var uri = "/t0_reqmon/data/requestcache";
    var reqModel = new WMStats._AjaxModelBase(uri, WMStats.Requests);
    reqModel.setTrigger(WMStats.CustomEvents.REQUESTS_LOADED);
    
    return reqModel;
}();
WMStats.namespace("RequestModel");
//TODO: specify tableConfig and filterConfig for T0.
WMStats.RequestModel =  new WMStats._RequestModelBase();
WMStats.namespace("RequestSearchModel");

WMStats.RequestSearchModel = new WMStats._ModelBase('', 
                                        {"reduce": false, 'include_docs': true}, 
                                        WMStats.WorkloadSummary);

WMStats.RequestSearchModel.setDBSource(WMStats.T0Couch);
WMStats.RequestSearchModel.setTrigger(WMStats.CustomEvents.WORKLOAD_SUMMARY_READY);WMStats.namespace("RequestLogDetailModel");
// don't set the initial view - it is only used for doc retrieval
WMStats.RequestLogDetailModel = new WMStats._ModelBase('', {}, 
                                          WMStats.LogMessage);

WMStats.RequestLogDetailModel.setDBSource(WMStats.T0LogDBCouch);
WMStats.RequestLogDetailModel.setTrigger(WMStats.CustomEvents.ERROR_LOG_LOADED);

$(WMStats.Globals.Event).on(WMStats.CustomEvents.LOG_LOADED, 
    function(event) {
        var options = {"include_docs": true};
    	
    	var logData = WMStats.RequestLogModel.getData();
    	options.keys = logData.getErrorLogIDs();
        
        WMStats.RequestLogDetailModel.retrieveData("allDocs", options);
});WMStats.namespace("RequestLogModel");
// don't set the initial view - it is only used for doc retrieval
WMStats.RequestLogModel = new WMStats._ModelBase('', {}, 
                                          WMStats.LogDBData);

WMStats.RequestLogModel.setDBSource(WMStats.T0LogDBCouch);
WMStats.RequestLogModel.setTrigger(WMStats.CustomEvents.LOG_LOADED);

$(WMStats.Globals.Event).on(WMStats.CustomEvents.REQUESTS_LOADED, 
    function(event) {
        var options = {"stale": "update_after"};
        
        WMStats.RequestLogModel.retrieveData("logByRequest", options);
});WMStats.namespace("GenericController");

(function($) {
    var vm = WMStats.ViewModel;
 // collapsible bar
    function closeRequestDetail() {
        $("#request_view div.detail_data").hide('puff', {}, 500);
        $("#acdc_submission").hide('puff', {}, 500);
        vm.RequestDetail.open = false;
    };
    
    $(document).on('click', 'div.caption img', function(event){
        $(this).parent('div.caption').siblings('div.body').toggle('nomal');
    });

    $(document).on('click', 'div.closingButton', function(event){
        //$(this).parent('div').hide('puff', {}, 500);
        //TODO: generalize to all the button not just for request detail
        closeRequestDetail();
        event.preventDefault();
    });
   
    $(document).keyup(function(event) {
        if (vm.RequestDetail.open && event.keyCode == 27) {
            closeRequestDetail();
            event.preventDefault();
        }
    });
    
    
    WMStats.GenericController.switchDiv = function (showSelector, hideSelectors) {
        for (var i in hideSelectors) {
                $(hideSelectors[i]).hide();
        }
        $(showSelector).show();
    };

})(jQuery);
WMStats.namespace("ActiveRequestController");

(function($){
    
    var E = WMStats.CustomEvents;
   
    var vm = WMStats.ViewModel;
    
    $(WMStats.Globals.Event).on(E.REQUESTS_LOADED, 
        function(event) {
            vm.propagateUpdate();
        });

    $(WMStats.Globals.Event).on(E.AGENTS_LOADED, 
        function(event, agentData) {
            vm.propagateUpdate();
        });

	$(WMStats.Globals.Event).on(E.ERROR_LOG_LOADED, 
        function(event, logData) {
            vm.propagateUpdate();
        });
        
    $(WMStats.Globals.Event).on(E.JOB_SUMMARY_READY, 
        function(event, data) {
            vm.JobView.updateDataAndChild(data);
            //vm.AlertJobView.updateDataAndChild(data);
        });

    $(WMStats.Globals.Event).on(E.JOB_DETAIL_READY, 
		function(event, data) {
            vm.JobDetail.data(data);
        });

    $(WMStats.Globals.Event).on(E.RESUBMISSION_SUMMARY_READY, 
        function(event, data) {
            var rData = vm.Resubmission.formatResubmissionData(data);
            vm.Resubmission.data(rData);
    });

    $(WMStats.Globals.Event).on(E.LOADING_DIV_START, 
        function(event, data) {
            // TODO: need to update when partial_request happens)
            var count = vm.ActiveRequestPage.refreshCount();
            vm.ActiveRequestPage.refreshCount(count + 1);
            if (vm.page() === vm.ActiveRequestPage && vm.ActiveRequestPage.refreshCount() === 1) {
                     $('#loading_page').show();
                }
        });

    $(WMStats.Globals.Event).on(E.LOADING_DIV_END, 
        function(event, data) {
            $('#loading_page').hide();
        });
        
    $(WMStats.Globals.Event).on(E.AJAX_LOADING_START, 
        function(event, data) {
            $('#loading_page').show();
        });
    
})(jQuery);
WMStats.namespace("WorkloadSummaryController");

(function($){
    var vm = WMStats.ViewModel;
    var E = WMStats.CustomEvents;
    $(WMStats.Globals.Event).on(E.WORKLOAD_SUMMARY_READY, 
        function(event, data) {
            var data = WMStats.RequestSearchModel.getData();
            vm.SearchPage.data(data);
        });

})(jQuery);
