"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const elb = require("aws-cdk-lib/aws-elasticloadbalancingv2");
const defaults = require("@aws-solutions-constructs/core");
const ecs = require("aws-cdk-lib/aws-ecs");
const integ_tests_alpha_1 = require("@aws-cdk/integ-tests-alpha");
// Setup
const app = new aws_cdk_lib_1.App();
const stack = new aws_cdk_lib_1.Stack(app, defaults.generateIntegStackName(__filename), {
    env: { account: aws_cdk_lib_1.Aws.ACCOUNT_ID, region: 'us-east-1' },
});
stack.templateOptions.description = 'Integration Test for private HTTPS API with existing VPC, LoadBalancer and Service';
// This is a minimal web server in our account that passes health checks
const image = ecs.ContainerImage.fromRegistry('public.ecr.aws/m7z7i5e4/integration-test-image:latest');
const testExistingVpc = defaults.getTestVpc(stack);
const createFargateServiceResponse = defaults.CreateFargateService(stack, 'test', {
    constructVpc: testExistingVpc,
    clientContainerDefinitionProps: { image },
});
const existingAlb = new elb.ApplicationLoadBalancer(stack, 'test-alb', {
    vpc: testExistingVpc,
    internetFacing: true,
});
const testProps = {
    existingVpc: testExistingVpc,
    publicApi: true,
    existingFargateServiceObject: createFargateServiceResponse.service,
    existingContainerDefinitionObject: createFargateServiceResponse.containerDefinition,
    existingLoadBalancerObj: existingAlb,
    listenerProps: {
        protocol: 'HTTP'
    },
};
const albToFargate = new lib_1.AlbToFargate(stack, 'test-construct', testProps);
defaults.addCfnSuppressRules(existingAlb, [{ id: 'W52', reason: 'This ALB is created only for test purposes, is never accessed and is destroyed immediately' }]);
defaults.addCfnSuppressRules(albToFargate.listener, [
    { id: 'W56', reason: 'All integration tests must be HTTP because of certificate limitations.' },
]);
const newSecurityGroup = albToFargate.loadBalancer.connections.securityGroups[0].node.defaultChild;
defaults.addCfnSuppressRules(newSecurityGroup, [
    { id: 'W29', reason: 'CDK created rule that blocks all traffic.' },
    { id: 'W2', reason: 'Rule does not apply for ELB.' },
    { id: 'W9', reason: 'Rule does not apply for ELB.' }
]);
defaults.suppressCustomHandlerCfnNagWarnings(stack, 'Custom::VpcRestrictDefaultSGCustomResourceProvider');
// Synth
new integ_tests_alpha_1.IntegTest(stack, 'Integ', { testCases: [
        stack
    ] });
//# sourceMappingURL=data:application/json;base64,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