"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const elb = require("aws-cdk-lib/aws-elasticloadbalancingv2");
const ecs = require("aws-cdk-lib/aws-ecs");
const defaults = require("@aws-solutions-constructs/core");
const integ_tests_alpha_1 = require("@aws-cdk/integ-tests-alpha");
// Setup
const app = new aws_cdk_lib_1.App();
const stack = new aws_cdk_lib_1.Stack(app, defaults.generateIntegStackName(__filename), {
    env: { account: aws_cdk_lib_1.Aws.ACCOUNT_ID, region: 'us-east-1' },
});
stack.templateOptions.description = 'Integration Test for public HTTP API with new VPC, LoadBalancer and Service and 2 targets';
// This is a minimal web server in our account that passes health checks
const image = ecs.ContainerImage.fromRegistry('public.ecr.aws/m7z7i5e4/integration-test-image:latest');
const testProps = {
    publicApi: true,
    containerDefinitionProps: {
        image
    },
    listenerProps: {
        protocol: 'HTTP'
    },
    albLoggingBucketProps: {
        removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        autoDeleteObjects: true
    }
};
const firstConstruct = new lib_1.AlbToFargate(stack, 'test-construct', testProps);
const testPropsTwo = {
    existingVpc: firstConstruct.vpc,
    existingContainerDefinitionObject: firstConstruct.container,
    existingFargateServiceObject: firstConstruct.service,
    existingLoadBalancerObj: firstConstruct.loadBalancer,
    publicApi: true,
    ruleProps: {
        conditions: [elb.ListenerCondition.pathPatterns(["*admin*"])],
        priority: 10
    },
};
const albToFargate = new lib_1.AlbToFargate(stack, 'test-two-construct', testPropsTwo);
defaults.addCfnSuppressRules(albToFargate.listener, [
    { id: 'W56', reason: 'All integration tests must be HTTP because of certificate limitations.' },
]);
const newSecurityGroup = albToFargate.loadBalancer.connections.securityGroups[0].node.defaultChild;
defaults.addCfnSuppressRules(newSecurityGroup, [
    { id: 'W29', reason: 'CDK created rule that blocks all traffic.' },
    { id: 'W2', reason: 'Rule does not apply for ELB.' },
    { id: 'W9', reason: 'Rule does not apply for ELB.' }
]);
defaults.suppressCustomHandlerCfnNagWarnings(stack, 'Custom::S3AutoDeleteObjectsCustomResourceProvider');
defaults.suppressCustomHandlerCfnNagWarnings(stack, 'Custom::VpcRestrictDefaultSGCustomResourceProvider');
// Synth
new integ_tests_alpha_1.IntegTest(stack, 'Integ', { testCases: [
        stack
    ] });
//# sourceMappingURL=data:application/json;base64,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