from __future__ import annotations

import importlib
import os
import re
import sys
from pathlib import Path
from typing import Callable, Any, Optional

import click
from jinja2 import Environment, BaseLoader

from brickflow import _ilog, BrickflowProjectConstants, get_entrypoint_python
from brickflow.cli.commands import exec_command

PWD = Path(__file__).parent.absolute()
GITIGNORE_TEMPLATE = PWD / "gitignore_template.txt"
GIT_PATH = Path(".git")


class GitNotFoundError(Exception):
    pass


class GitIgnoreNotFoundError(Exception):
    pass


def _gitignore_exists() -> bool:
    return os.path.exists(".gitignore") and os.path.isfile(".gitignore")


def _create_gitignore_if_not_exists() -> None:
    if _gitignore_exists() is False:
        Path(".gitignore").touch(mode=0o755)


def _get_gitignore() -> str:
    return Path(".gitignore").read_text(encoding="utf-8")


def _get_gitignore_template() -> str:
    return GITIGNORE_TEMPLATE.read_text()


def _write_gitignore(data: str) -> None:
    Path(".gitignore").write_text(encoding="utf-8", data=data)


def _update_gitignore() -> None:
    search_regex = re.compile(
        r"(# GENERATED BY BRICKFLOW CLI --START--(.|\n)*# GENERATED BY BRICKFLOW CLI --END--)"
    )

    git_ignore_data = _get_gitignore()
    git_ignore_template = _get_gitignore_template()
    search = search_regex.findall(git_ignore_data)
    if len(search) > 0:
        search_match = search[0][0]
        gitignore_file_data = git_ignore_data.replace(search_match, git_ignore_template)
    else:
        gitignore_file_data = "\n\n".join([git_ignore_data, git_ignore_template])
    _write_gitignore(gitignore_file_data)


def _validate_package(path_str: str) -> str:
    folder_path: Path = Path(path_str)

    if not folder_path.exists():
        raise ImportError(f"Invalid pkg error: {folder_path.as_posix()}")

    sys.path.append(os.getcwd())
    folder_pkg_path: str = folder_path.as_posix().replace("/", ".")

    for module in folder_path.glob("**/*.py"):  # only find python files
        # ignore __init__.py
        if module.name == "__init__.py":
            continue
        module_name = module.as_posix().replace(".py", "").replace("/", ".")
        # import all the modules into the mod object and not actually import them using __import__
        mod = importlib.import_module(module_name)
        click.echo(f"Scanned module: {mod.__name__}")

    return folder_pkg_path


def render_template(**kwargs) -> str:  # type: ignore
    template = Path(__file__).parent.absolute() / "entrypoint.template"
    with template.open("r") as f:
        data = f.read()
        return Environment(loader=BaseLoader()).from_string(data).render(**kwargs)


def create_entry_point(working_dir: str, data: str) -> None:
    path = Path(working_dir) / "entrypoint.py"
    if path.exists():
        click.echo(f"Path: {str(path.absolute())} already exists...")
        # path = Path(working_dir) / "entrypoint.py.new"
    else:
        click.echo(f"Creating file in path: {str(path.absolute())}...")
        path.write_text(data)


def create_brickflow_project_root_marker() -> None:
    path = Path(
        f"{BrickflowProjectConstants.DEFAULT_MULTI_PROJECT_ROOT_FILE_NAME.value}."
        f"{BrickflowProjectConstants.DEFAULT_CONFIG_FILE_TYPE.value}"
    )
    if path.exists():
        click.echo(f"Path: {str(path.absolute())} already exists...")
        # path = Path(working_dir) / "entrypoint.py.new"
    else:
        click.echo(f"Creating file in path: {str(path.absolute())}...")
        path.write_text(
            "# DO NOT MODIFY THIS FILE - IT IS AUTO GENERATED BY BRICKFLOW AND RESERVED FOR FUTURE USAGE",
            encoding="utf-8",
        )


def bind_env_var(env_var: str) -> Callable:
    def callback(
        ctx: click.Context,  # noqa
        param: str,  # noqa
        value: Any,
    ) -> None:
        # pylint: disable=unused-argument
        if value is not None and len(value) > 0:
            _ilog.info("Setting env var: %s to %s...", env_var, value)
            if isinstance(value, list):
                os.environ[env_var] = ",".join(value)
            if isinstance(value, tuple):
                os.environ[env_var] = ",".join(value)
            elif isinstance(value, bool):
                os.environ[env_var] = str(value).lower()
            else:
                os.environ[env_var] = value

    return callback


def get_entrypoint(**kwargs: Any) -> str:
    wd: Optional[str] = kwargs.get("workflows_dir")
    if wd is None:
        raise ValueError(
            "workflows_dir not set, please set it using --workflows-dir or -wd"
        )
    return str(Path(wd) / "entrypoint.py")


def log_important_versions(bundle_cli: str) -> None:
    version = exec_command(bundle_cli, "--version", [], capture_output=True)
    _ilog.info("Using bundle version: %s", version)
    log_python_version()


def log_python_version() -> None:
    version = exec_command(
        get_entrypoint_python(), "--version", [], capture_output=True
    )
    _ilog.info("Using python version: %s", version)
