from typer import Typer, Option
from typing import Optional, List
from typing_extensions import Annotated
from .layout import dual_layout
from rich.live import Live
import time
from .console import console
from .tools import curry
from .slurm import (
    get_layout_pair,
    get_node_layout,
    get_hist_layout,
    get_job_layout,
    NotOnClusterError,
)

import mrich

# set up singletons
app = Typer()


# main CLI command
@app.callback(invoke_without_command=True)
def show(
    user: Annotated[
        str,
        Option(
            "-u",
            "--user",
            help="Query jobs for this user. Use 'all' to see all jobs",
        ),
    ] = None,
    long: Annotated[bool, Option("-v", "--long", help="More detailed output")] = False,
    idle: Annotated[bool, Option("-i", "--idle", help="Show available nodes")] = False,
    no_loop: Annotated[
        bool, Option("-nl", "--no-loop", help="Don't show a live-updating screen")
    ] = False,
    hist: Optional[List[str]] = Option(
        None,
        "--hist",
        "-h",
        help="Show historical jobs from the preceding specified time period, e.g. '4 weeks'",
    ),
    job: Annotated[
        int, Option("-j", "--job", help="Show details for a specific job")
    ] = None,
):

    loop = True
    screen = True
    disappear = not screen

    kwargs = {
        "user": user,
        "long": long,
        "idle": idle,
        "no_loop": no_loop,
        "hist": hist,
        "hist_unit": "weeks",
        "screen": screen,
        "disappear": disappear,
        "job": job,
    }

    # console.print(kwargs)

    if hist:
        hist, hist_unit = parse_history_string(hist)

        kwargs["hist"] = hist
        kwargs["hist_unit"] = hist_unit

    if no_loop:
        loop = False

    match (bool(idle), bool(hist), bool(job)):
        case (True, False, False):
            loop = False
            layout_func = get_node_layout
        case (False, True, False):
            loop = False
            layout_func = get_hist_layout
        case (False, False, True):
            layout_func = get_job_layout
        case (False, False, False):
            layout_func = curry(dual_layout, get_layout_pair)
        case _:
            raise Exception("Unsupported CLI options")

    try:

        # live updating layout
        if loop:

            layout = layout_func(**kwargs)

            with Live(
                layout,
                refresh_per_second=4,
                screen=screen,
                transient=disappear,
                vertical_overflow="visible",
            ) as live:

                try:
                    while True:
                        layout = layout_func(**kwargs)
                        live.update(layout)
                        time.sleep(1)
                except KeyboardInterrupt:
                    live.stop()

        # static layout
        else:

            kwargs["console_height"] = 500

            layout = layout_func(**kwargs)
            console.print(layout)

    except NotOnClusterError as e:
        console.print(f"[red bold]{e}[reset]")


def parse_history_string(hist: str, unit: str = "week") -> (int, str):
    """Parse one or two part string into integer value and string unit"""

    hist = " ".join(hist)
    hist = hist.split()

    assert len(hist) <= 2, "wrong number of 'hist' values"

    if len(hist) == 2:
        unit = hist[1]

    hist = int(hist[0])

    return hist, unit


# start Typer app
def main():
    app()


# start Typer app
if __name__ == "__main__":
    app()
