# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/coinbase.ipynb.

# %% auto 0
__all__ = ['retrieve_coinbase_price', 'coinbase_tokens', 'coinbase_usd_tokens', 'coinbase_price_history', 'save_file',
           'coinbase_to_file', 'coinbase_data_update', 'read_all_files', 'coinbase_price_last_day', 'binance_format']

# %% ../nbs/coinbase.ipynb 3
from datetime import datetime
import pandas as pd
import requests
import os
import datetime as dt
import time


# %% ../nbs/coinbase.ipynb 4
def retrieve_coinbase_price(pair="BTC-USD", time_interval=3600,
                          end_date=datetime.now().strftime('%Y-%m-%dT%H:%M:%SZ'),
                          start_date=(datetime.now()-pd.Timedelta(days=2)).strftime('%Y-%m-%dT%H:%M:%SZ')):
    """
    Retrieves historical price data from Coinbase for a given trading pair and time interval.

    Makes a GET request to the Coinbase candles endpoint to fetch OHLCV (Open, High, Low, Close, Volume) 
    data for the specified trading pair and time period.

    Args:
        pair (str, optional): Trading pair symbol (e.g. "BTC-USD"). Defaults to "BTC-USD".
        time_interval (int, optional): Candle interval in seconds. Defaults to 3600 (1 hour).
        end_date (str, optional): End datetime in ISO 8601 format. Defaults to current UTC time.
        start_date (str, optional): Start datetime in ISO 8601 format. Defaults to 2 days before end_date.

    Returns:
        pandas.DataFrame: DataFrame containing the OHLCV data with columns:
            - datetime: Timestamp for the candle (UTC)
            - low: Lowest traded price in the interval
            - high: Highest traded price in the interval  
            - open: Opening price of the interval
            - close: Closing price of the interval
            - volume: Trading volume in the interval
            - pair: Trading pair symbol
        Returns None if the API request fails or returns no data.

    Notes:
        - Limited to 300 requests per hour by the Coinbase API
        - Maximum of 200 candles can be retrieved per request
        - Attempting to exceed these limits will result in an error and return None
        - All datetime values are in UTC timezone
    """
    url = f"https://api.exchange.coinbase.com/products/{pair}/candles?granularity={time_interval}&start={start_date}&end={end_date}"
    response = requests.get(url)
    if response.status_code != 200:
        return None
    data = response.json()
    if not data:
        return None

    column_names = ['unix', 'low', 'high', 'open', 'close', 'volume']
    df = pd.DataFrame(data, columns=column_names)
    df['datetime'] = pd.to_datetime(df['unix'], unit='s')
    df['pair'] = pair
    df = df.sort_values(by='datetime')
    # drop unix column and put date first
    df = df.drop(columns=['unix'])
    df = df[['datetime'] + [col for col in df.columns if col != 'datetime']]
    return df


# %% ../nbs/coinbase.ipynb 5
def coinbase_tokens():
    """
    Retrieves all available trading pairs from the Coinbase Exchange API.

    Makes a GET request to the Coinbase products endpoint to fetch information about
    all trading pairs available on the exchange.

    Returns:
        pandas.DataFrame: DataFrame containing information about all trading pairs with columns:
            - id: Trading pair ID (e.g. 'BTC-USD')
            - base_currency: The cryptocurrency being traded (e.g. 'BTC') 
            - quote_currency: The currency used for pricing (e.g. 'USD')
            - quote_increment: Minimum price increment
            - base_increment: Minimum quantity increment
            - display_name: Human readable name of the trading pair
            - status: Trading status of the pair
            And other metadata columns provided by the Coinbase API
    """
    currencies_url = "https://api.exchange.coinbase.com/products"
    currencies_response = requests.get(currencies_url)
    currencies_data = currencies_response.json()
    return pd.DataFrame(currencies_data)


# %% ../nbs/coinbase.ipynb 6
def coinbase_usd_tokens():
    """
    Retrieves all trading pairs from Coinbase that have USD as the quote currency.

    Returns:
        pandas.DataFrame: DataFrame containing information about USD trading pairs with columns:
            - id: Trading pair ID (e.g. 'BTC-USD')
            - base_currency: The cryptocurrency being traded (e.g. 'BTC')
            - quote_currency: Always 'USD' for this filtered dataset
            - quote_increment: Minimum price increment
            - base_increment: Minimum quantity increment
            - display_name: Human readable name of the trading pair
            - status: Trading status of the pair
            And other metadata columns provided by the Coinbase API
    """
    tokens = coinbase_tokens()
    return tokens[tokens['quote_currency']=='USD']


# %% ../nbs/coinbase.ipynb 7
def coinbase_price_history(pair='BTC-USD', start_date='2024-01-01', end_date='2024-05-01', time_interval=3600, max_pull=250, verbose=False):
    """
    Downloads historical price data for a cryptocurrency pair from Coinbase, handling rate limits.

    Args:
        pair (str): Trading pair symbol (e.g. 'BTC-USD'). Defaults to 'BTC-USD'.
        start_date (str): Start date in 'YYYY-MM-DD' format. Defaults to '2024-01-01'.
        end_date (str): End date in 'YYYY-MM-DD' format. Defaults to '2024-05-01'.
        time_interval (int): Time interval between candles in seconds. Defaults to 3600 (1 hour).
        max_pull (int): Maximum number of candles per API request. Defaults to 250.
        verbose (bool): If True, prints progress messages. Defaults to False.

    Returns:
        pandas.DataFrame: DataFrame containing price history with columns:
            - datetime: Timestamp of the candle
            - low: Lowest price during the interval
            - high: Highest price during the interval  
            - open: Opening price of the interval
            - close: Closing price of the interval
            - volume: Trading volume during the interval
            - pair: Trading pair symbol

    Raises:
        ValueError: If end_date is earlier than start_date

    The function handles Coinbase's API limitations by automatically splitting requests
    into smaller chunks if the date range would exceed the maximum allowed candles
    per request.
    """
    # convert the dates to datetime
    start_date = pd.to_datetime(start_date)
    end_date = pd.to_datetime(end_date)
    if end_date < start_date:
        # raise an error
        raise ValueError("End date is smaller than start date")
    # calculate the number of hours between the start and end date
    n_hours = (end_date - start_date).days * 24
    if n_hours <= max_pull:
        df = retrieve_coinbase_price(pair,start_date=start_date.strftime('%Y-%m-%dT%H:%M:%SZ'),end_date=end_date.strftime('%Y-%m-%dT%H:%M:%SZ'),time_interval=time_interval)
        return df
    # create a timeline with the maximum number of dates per request
    time_line = pd.date_range(start=start_date, end=end_date, periods=n_hours//max_pull+1)
    price_list = []
    for i in range(len(time_line)-1):
        end = time_line[i+1]
        start = time_line[i]
        if verbose:
            print(f"Downloading data from {start} to {end}")
        df = retrieve_coinbase_price(pair,start_date=start.strftime('%Y-%m-%dT%H:%M:%SZ'),end_date=end.strftime('%Y-%m-%dT%H:%M:%SZ'),time_interval=time_interval)
        if df is not None:
            price_list.append(df)
    # concatenate the dataframes and sort by date and drop duplicates dates
    if price_list == []:
        print(f"{pair} is not listed.")
        return pd.DataFrame()
    df = pd.concat(price_list)
    df = df.sort_values(by='datetime')
    df = df.drop_duplicates(subset='datetime').reset_index(drop=True)
    return df

# %% ../nbs/coinbase.ipynb 8
def save_file(df, folder_path, file_name, type="csv"):
    """
    Save a pandas DataFrame to a file in either CSV or Parquet format.

    Args:
        df (pandas.DataFrame): The DataFrame to save
        folder_path (str): Directory path where the file will be saved
        file_name (str): Name of the file without extension
        type (str, optional): File format - either "csv" or "parquet". Defaults to "csv"

    The function saves the DataFrame to the specified path, handling the file extension automatically.
    For CSV files, the index is not saved. For Parquet files, default Parquet settings are used.
    """
    if type == "csv":
        df.to_csv(f"{folder_path}/{file_name}.csv",index=False)
    elif type == "parquet":
        df.to_parquet(f"{folder_path}/{file_name}.parquet")

# %% ../nbs/coinbase.ipynb 9
def coinbase_to_file(folder_path="../data/coinbase",token_list=coinbase_usd_tokens()['id'].tolist(),type="csv",
                     interval=3600,all_tokens=True,refresh_24h=False,pause=0):
    """
    Downloads and maintains historical price data for Coinbase tokens, saving to files.
    
    Args:
        folder_path (str): Path where token data files will be stored. Defaults to "../data/coinbase"
        token_list (list): List of token IDs to process. Defaults to all USD trading pairs from coinbase_usd_tokens()
        type (str): File format to save data - either "csv" or "parquet". Defaults to "csv"
        interval (int): Time interval in seconds between price points. Defaults to 3600 (1 hour)
        all_tokens (bool): If True, includes any additional tokens found in the folder path. Defaults to True
        refresh_24h (bool): If True, replaces at least 24 hours. Defaults to False
    
    The function:
    - Creates the folder_path if it doesn't exist
    - Date/Time is UTC
    - For each token, checks if data file exists:
        - If exists: Loads file and appends any new data since last recorded date
        - If not exists: Downloads full history starting from 2016
    - Saves data in specified format, handling duplicates and sorting by date
    - For hourly data (interval=3600), aligns to hour boundaries
    """
    # create the folder if it does not exist, and if it exists, read the file names
    if not os.path.exists(folder_path):
        os.makedirs(folder_path)
    else:
        if all_tokens:
            file_names = os.listdir(folder_path)
            # remove the file extension
            tokens_in_folder = [os.path.splitext(file_name)[0] for file_name in file_names]
            # join tokens in folder with token_list and remove the duplicates
            token_list = list(set(token_list + tokens_in_folder))
    # loop over the token list and open the file if it exists and append the new data
    for token in token_list:
        print(f"Processing {token}")
        file_name = f"{folder_path}/{token}.{type}"
        if os.path.exists(file_name):
            if type == "csv":
                df = pd.read_csv(file_name)
            elif type == "parquet":
                df = pd.read_parquet(file_name)
            else:
                raise ValueError(f"Type {type} not supported")
            # read last date in the file
            last_date = pd.to_datetime(df['datetime'].iloc[-1]).tz_localize(dt.UTC)
            today = datetime.now(tz=dt.UTC)
            #print(last_date.tz_localize(dt.UTC),today)
            print(last_date)
            print(today)
            first_date = today - dt.timedelta(hours=24)
            if first_date < last_date and refresh_24h:
                t_date = pd.to_datetime(df['datetime'],utc=True)
                df = df[t_date < first_date]
                last_date = first_date
            if last_date < today:
                try:
                    df_new = coinbase_price_history(pair=token,
                                                start_date=last_date.strftime('%Y-%m-%dT%H:%M:%SZ'),
                                                end_date=today.strftime('%Y-%m-%dT%H:%M:%SZ'),
                                                time_interval=interval)
                    df = pd.concat([df,df_new])
                    # drop duplicates and sort by date
                    df = df.drop_duplicates(subset='datetime').sort_values(by='datetime').reset_index(drop=True)
                    if not df.empty:
                        save_file(df,folder_path,token,type)
                except Exception as e:
                    print(f"Error adding data to {token}: {e}")
            else:
                print(f"File {token} is up to date")
        else:
            try:
                df = coinbase_price_history(pair=token,
                                        start_date='2016-01-01T00:00:00Z',
                                        end_date=datetime.now().strftime('%Y-%m-%dT%H:%M:%SZ'),
                                        time_interval=interval)
            except Exception as e:
                print(f"Error first download for {token}: {e}")
            if not df.empty:
                save_file(df,folder_path,token,type)
        if pause > 0:
            time.sleep(pause)

# %% ../nbs/coinbase.ipynb 10
def coinbase_data_update(folder_path="../data/coinbase",token='AAVE-USD',type="parquet",
                     interval=3600,refresh_24h=True):
    """
    Downloads new historical price data for Coinbase tokens... no saving
    
    Args:
        folder_path (str): Path where token data files will be stored. Defaults to "../data/coinbase"
        token (str): token IDs to process
        type (str): File format to save data - either "csv" or "parquet". Defaults to "parquet"
        interval (int): Time interval in seconds between price points. Defaults to 3600 (1 hour)
        refresh_24h (bool): If True, replaces at least 24 hours. Defaults to True
    
    Returns:
        df (pandas.DataFrame): DataFrame of historical price data
    
    The function:
    - Date/Time is UTC
    - For each token, checks if data file exists:
        - If exists: Loads file and appends any new data since last recorded date
        - If not exists: Downloads full history starting from 2016
    - For hourly data (interval=3600), aligns to hour boundaries
    """
    
    if not os.path.exists(folder_path):
        raise FileNotFoundError(f"Folder {folder_path} does not exist")
    
    print(f"Processing {token}")
    file_name = f"{folder_path}/{token}.{type}"
    if os.path.exists(file_name):
        if type == "csv":
            df = pd.read_csv(file_name)
        elif type == "parquet":
            df = pd.read_parquet(file_name)
        else:
            raise ValueError(f"Type {type} not supported")
        # read last date in the file
        last_date = pd.to_datetime(df['datetime'].iloc[-1]).tz_localize(dt.UTC)
        today = datetime.now(tz=dt.UTC)
        #print(last_date.tz_localize(dt.UTC),today)
        first_date = today - dt.timedelta(hours=24)
        if first_date < last_date and refresh_24h:
                t_date = pd.to_datetime(df['datetime'],utc=True)
                df = df[t_date < first_date]
                last_date = first_date
        if last_date < today:
            df_new = coinbase_price_history(pair=token,
                                            start_date=last_date.strftime('%Y-%m-%dT%H:%M:%SZ'),
                                            end_date=today.strftime('%Y-%m-%dT%H:%M:%SZ'),
                                            time_interval=interval)
            df = pd.concat([df,df_new])
                # drop duplicates and sort by date
            df = df.drop_duplicates(subset='datetime').sort_values(by='datetime').reset_index(drop=True)
        else:
            print(f"File {token} is up to date")
    else:
        df = coinbase_price_history(pair=token,
                                        start_date='2016-01-01T00:00:00Z',
                                        end_date=datetime.now().strftime('%Y-%m-%dT%H:%M:%SZ'),
                                        time_interval=interval)
    return df

# %% ../nbs/coinbase.ipynb 11
def read_all_files(folder_path="../data/coinbase",type="csv"):
    """Read and combine all files from a folder into a single DataFrame.
    
    Args:
        folder_path (str): Path to the folder containing the files. Defaults to "../data/coinbase".
        type (str): File type to read - either "csv" or "parquet". Defaults to "csv".
        
    Returns:
        pandas.DataFrame: Combined DataFrame containing data from all files in the folder.
        
    Raises:
        ValueError: If file type is not supported (must be "csv" or "parquet").
    """
    file_names = os.listdir(folder_path)
    df_list = []
    for file_name in file_names:
        if type == "csv":
            df = pd.read_csv(f"{folder_path}/{file_name}")
        elif type == "parquet":
            df = pd.read_parquet(f"{folder_path}/{file_name}")
        else:
            raise ValueError(f"Type {type} not supported")
        df_list.append(df)
    return pd.concat(df_list)



# %% ../nbs/coinbase.ipynb 12
def coinbase_price_last_day(pair='BTC-USD'):
    """
    Fetch the last day's price for a given Coinbase token.
    This fuction is a helper function around coinbase_price_history().
    
    Args:
        pair (str): Coinbase token pair, e.g., "BTC-USD". Defaults to "BTC-USD".
    
    Returns:
        pandas dataframe: Last 24 hours price data.
    """
    # fetch the data
    data = coinbase_price_history(pair=pair, 
                        start_date=(datetime.now(tz=dt.UTC) - dt.timedelta(days=1)).strftime('%Y-%m-%dT%H:%M:%SZ'), 
                        end_date=datetime.now(tz=dt.UTC).strftime('%Y-%m-%dT%H:%M:%SZ'))
    # return the last price
    return data

# %% ../nbs/coinbase.ipynb 13
def binance_format(df):
    """
    Convert coinbase dataframe to binance format.
    
    Args:
        df (pandas dataframe): Coinbase dataframe.
    
    Returns:
        pandas dataframe: Binance format dataframe.
    """
    df['date']  = pd.to_datetime(df['datetime'],utc=True)
    df['pair']  = df['pair'].apply(lambda x: str.lower(str.split(x,'-')[0]))
    df['Open'] = df['open']
    df['High'] = df['high']
    df['Low']  = df['low']
    df['Close'] = df['close']
    df['Volume'] = df['volume']
    return df[['date','Open','High','Low','Close','Volume','pair']]
