# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/hyper_account.ipynb.

# %% auto 0
__all__ = ['get_user_fills', 'get_user_funding_history', 'get_user_ledger_updates', 'get_account_summary']

# %% ../nbs/hyper_account.ipynb 4
from .hyperliquid import *
from hyperliquid.utils import constants
import pandas as pd
import numpy as np # Required for handling numerical operations

# %% ../nbs/hyper_account.ipynb 7
def get_user_fills(address, info=None, aggregated=False, include_hash_oid=False):
    """
    Retrieves user fill history from Hyperliquid and returns as a DataFrame.
    
    Args:
        address (str): User's wallet address
        info (Info, optional): Hyperliquid Info client. If None, creates a new one.
        aggregated (bool, optional): If True, returns aggregated fills. Defaults to False.
    
    Returns:
        pandas.DataFrame: DataFrame containing fill data indexed by datetime with columns:
            - coin: Trading pair/coin
            - px: Fill price
            - sz: Fill size
            - side: Trade side (A for ask/sell, B for bid/buy)
            - time: Original timestamp in milliseconds
            - startPosition: Position size before fill
            - dir: Direction of trade
            - closedPnl: Realized PnL from closing position
            - hash: Transaction hash
            - oid: Order ID
            - crossed: Whether order crossed the spread
            - fee: Trading fee paid
            - tid: Trade ID
            - feeToken: Token used for fee payment
        Returns None if no fills found or error occurs.
    
    Examples:
        # Get all user fills
        fills_df = get_user_fills(address, info=info)
        
        # Get aggregated fills
        fills_df = get_user_fills(address, info=info, aggregated=True)
        
        # Filter for specific coin
        eth_fills = fills_df[fills_df['coin'] == 'ETH']
        
        # Calculate total fees paid
        total_fees = fills_df['fee'].sum()
    
    Notes:
        - DataFrame is indexed by datetime (converted from milliseconds)
        - All numeric columns are converted to float for calculations
        - Sorted by datetime in ascending order
    """
    # Initialize info client if not provided
    if info is None:
        from hyperliquid.info import Info
        from hyperliquid.utils import constants
        address_temp, info, exchange = setup(base_url=constants.MAINNET_API_URL, skip_ws=True)
    
    try:
        # Get user fills
        if aggregated:
            user_fills = info.user_fills(address, aggregated=True)
        else:
            user_fills = info.user_fills(address)
        
        if not user_fills:
            print("No fills found for this address")
            return None
        
        # Convert to DataFrame
        fills_df = pd.DataFrame(user_fills)
        
        # Convert timestamp to datetime
        fills_df['datetime'] = pd.to_datetime(fills_df['time'], unit='ms')
        
        # Convert numeric columns to float
        numeric_cols = ['px', 'sz', 'startPosition', 'closedPnl', 'fee']
        for col in numeric_cols:
            if col in fills_df.columns:
                fills_df[col] = fills_df[col].astype(float)
        
        # Set datetime as index
        fills_df = fills_df.set_index('datetime')
        
        # Sort by datetime
        fills_df = fills_df.sort_index()
        
        # Add human-readable side column
        if 'side' in fills_df.columns:
            fills_df['side_readable'] = fills_df['side'].map({'A': 'Sell', 'B': 'Buy'})
        
        fills_df['fee_usdc'] = fills_df.apply(
            lambda row: row['fee'] * row['px'] if row['feeToken'] != 'USDC' else row['fee'],
            axis=1
        )
        # Calculate fee in basis points (bps)
        # Fee in bps = (fee / (price * size)) * 10000
        fills_df['fee_bps'] = (fills_df['fee_usdc'] / (fills_df['px'] * fills_df['sz'])) * 10000

        # Drop hash and oid columns if not requested
        if not include_hash_oid:
            cols_to_drop = [col for col in ['hash', 'oid','tid','time'] if col in fills_df.columns]
            if cols_to_drop:
                fills_df = fills_df.drop(columns=cols_to_drop)

        return fills_df
        
    except Exception as e:
        print(f"Error retrieving user fills: {e}")
        return None

# %% ../nbs/hyper_account.ipynb 11
def get_user_funding_history(address, start_date=None, end_date=None, lookback=30, info=None,include_hash_time=False):
    """
    Retrieves user funding payment history from Hyperliquid and returns as a DataFrame.
    
    Args:
        address (str): User's wallet address
        start_date (str, optional): Start date as string. Can be:
            - ISO format: "2024-01-15T10:30:00Z" or "2024-01-15T10:30:00"
            - Date only: "2024-01-15"
            - If None, calculated from lookback. Defaults to None.
        end_date (str, optional): End date as string (same formats as start_date).
            - If None, uses current UTC time. Defaults to None.
        lookback (int, optional): Number of days to look back from end_date if start_date is None.
            Defaults to 30.
        info (Info, optional): Hyperliquid Info client. If None, creates a new one.
    
    Returns:
        pandas.DataFrame: DataFrame containing funding payment data indexed by datetime with columns:
            - coin: Trading pair/coin
            - usdc: Funding payment amount in USDC (positive = received, negative = paid)
            - time: Original timestamp in milliseconds
        Returns None if no funding payments found or error occurs.
    
    Examples:
        # Get last 30 days of funding payments
        funding_df = get_user_funding_history(address, info=info)
        
        # Get funding for specific date range
        funding_df = get_user_funding_history(address, 
                                             start_date="2024-01-01",
                                             end_date="2024-01-31",
                                             info=info)
        
        # Calculate total funding received/paid
        total_funding = funding_df['usdc'].sum()
        
        # Get funding by coin
        funding_by_coin = funding_df.groupby('coin')['usdc'].sum()
    
    Notes:
        - DataFrame is indexed by datetime (converted from milliseconds)
        - Positive usdc values indicate funding received
        - Negative usdc values indicate funding paid
        - All numeric columns are converted to float for calculations
        - Sorted by datetime in ascending order
    """
    # Initialize info client if not provided
    if info is None:
        from hyperliquid.info import Info
        from hyperliquid.utils import constants
        address_temp, info, exchange = setup(base_url=constants.MAINNET_API_URL, skip_ws=True)
    
    # Handle end_date
    if end_date is None:
        end_dt = pd.Timestamp.now(tz='UTC')
    else:
        try:
            end_dt = pd.to_datetime(end_date)
        except Exception as e:
            print(f"Error parsing end_date '{end_date}': {e}")
            return None

    # Handle start_date
    if start_date is None:
        start_dt = end_dt - pd.Timedelta(days=lookback)
    else:
        try:
            start_dt = pd.to_datetime(start_date)
        except Exception as e:
            print(f"Error parsing start_date '{start_date}': {e}")
            return None
    
    # Convert to Unix milliseconds timestamps
    start_time_ms = int(start_dt.timestamp() * 1000)
    end_time_ms = int(end_dt.timestamp() * 1000)
    
    try:
        # Get user funding history
        funding_history = info.user_funding_history(address, start_time_ms, end_time_ms)
        
        if not funding_history:
            print("No funding payments found for this address in the specified period")
            return None
        
        # Flatten the nested dictionary structure
        flattened_data = []
        for record in funding_history:
            flat_record = {
                'time': record['time'],
                'hash': record['hash'],
                'coin': record['delta']['coin'],
                'usdc': float(record['delta']['usdc']),
                'szi': float(record['delta']['szi']),
                'fundingRate': float(record['delta']['fundingRate'])
            }
            flattened_data.append(flat_record)
        
        # Convert to DataFrame
        funding_df = pd.DataFrame(flattened_data)
        
        # Convert timestamp to datetime
        funding_df['datetime'] = pd.to_datetime(funding_df['time'], unit='ms')
        
        # Convert numeric columns to float
        numeric_cols = ['usdc', 'delta']
        for col in numeric_cols:
            if col in funding_df.columns:
                funding_df[col] = funding_df[col].astype(float)
        
        # Set datetime as index
        funding_df = funding_df.set_index('datetime')
        
        # Sort by datetime
        funding_df = funding_df.sort_index()
        
        # Drop hash and time columns if not requested
        if not include_hash_time:
            cols_to_drop = [col for col in ['hash', 'time'] if col in funding_df.columns]
            if cols_to_drop:
                funding_df = funding_df.drop(columns=cols_to_drop)
        
        return funding_df
        
    except Exception as e:
        print(f"Error retrieving user funding history: {e}")
        return None

# %% ../nbs/hyper_account.ipynb 18
def get_user_ledger_updates(address, start_date=None, end_date=None, lookback=30, info=None, include_hash=False):
    """
    Retrieves user non-funding ledger updates from Hyperliquid and returns as a DataFrame.
    
    This function captures account events like deposits, withdrawals, transfers, and other
    non-funding related balance changes.
    
    Args:
        address (str): User's wallet address
        start_date (str, optional): Start date as string. Can be:
            - ISO format: "2024-01-15T10:30:00Z" or "2024-01-15T10:30:00"
            - Date only: "2024-01-15"
            - If None, calculated from lookback. Defaults to None.
        end_date (str, optional): End date as string (same formats as start_date).
            - If None, uses current UTC time. Defaults to None.
        lookback (int, optional): Number of days to look back from end_date if start_date is None.
            Defaults to 30.
        info (Info, optional): Hyperliquid Info client. If None, creates a new one.
        include_hash (bool, optional): If True, includes 'hash' column with transaction hashes.
            Defaults to False.
    
    Returns:
        pandas.DataFrame: DataFrame containing ledger update data indexed by datetime with columns:
            - type: Type of ledger update (e.g., 'deposit', 'withdraw', 'accountClassTransfer')
            - usdc: Amount in USDC
            - toPerp: For transfers, whether transfer was to perpetual account (True) or from (False)
            - hash: Transaction hash (only if include_hash=True)
        Returns None if no ledger updates found or error occurs.
    
    Examples:
        # Get last 30 days of ledger updates
        ledger_df = get_user_ledger_updates(address, info=info)
        
        # Get ledger updates for specific date range
        ledger_df = get_user_ledger_updates(address, 
                                           start_date="2024-01-01",
                                           end_date="2024-01-31",
                                           info=info)
        
        # Calculate total deposits
        deposits = ledger_df[ledger_df['type'] == 'deposit']['usdc'].sum()
        
        # Calculate total withdrawals
        withdrawals = ledger_df[ledger_df['type'] == 'withdraw']['usdc'].sum()
        
        # Get all transfers to perpetual account
        to_perp = ledger_df[(ledger_df['type'] == 'accountClassTransfer') & 
                           (ledger_df['toPerp'] == True)]
    
    Notes:
        - DataFrame is indexed by datetime (converted from milliseconds)
        - All numeric columns are converted to float for calculations
        - Sorted by datetime in ascending order
        - Does not include funding payments (use get_user_funding_history for that)
        - Common transaction types:
            * 'deposit': Deposits to account
            * 'withdraw': Withdrawals from account
            * 'accountClassTransfer': Transfers between spot and perpetual accounts
    """
    # Initialize info client if not provided
    if info is None:
        from hyperliquid.info import Info
        from hyperliquid.utils import constants
        address_temp, info, exchange = setup(base_url=constants.MAINNET_API_URL, skip_ws=True)
    
    # Handle end_date
    if end_date is None:
        end_dt = pd.Timestamp.now(tz='UTC')
    else:
        try:
            end_dt = pd.to_datetime(end_date)
        except Exception as e:
            print(f"Error parsing end_date '{end_date}': {e}")
            return None

    # Handle start_date
    if start_date is None:
        start_dt = end_dt - pd.Timedelta(days=lookback)
    else:
        try:
            start_dt = pd.to_datetime(start_date)
        except Exception as e:
            print(f"Error parsing start_date '{start_date}': {e}")
            return None
    
    # Convert to Unix milliseconds timestamps
    start_time_ms = int(start_dt.timestamp() * 1000)
    end_time_ms = int(end_dt.timestamp() * 1000)
    
    try:
        # Get user non-funding ledger updates
        ledger_updates = info.user_non_funding_ledger_updates(address, start_time_ms, end_time_ms)
        
        if not ledger_updates:
            print("No ledger updates found for this address in the specified period")
            return None
        
        # Flatten the nested dictionary structure
        flattened_data = []
        for record in ledger_updates:
            flat_record = {
                'time': record['time'],
                'hash': record['hash'],
                'type': record['delta']['type'],
                'usdc': float(record['delta']['usdc'])
            }
            
            # Add toPerp field if it exists (for accountClassTransfer)
            if 'toPerp' in record['delta']:
                flat_record['toPerp'] = record['delta']['toPerp']
            else:
                flat_record['toPerp'] = None
            
            flattened_data.append(flat_record)
        
        # Convert to DataFrame
        ledger_df = pd.DataFrame(flattened_data)
        
        # Convert timestamp to datetime
        ledger_df['datetime'] = pd.to_datetime(ledger_df['time'], unit='ms')
        
        # Set datetime as index
        ledger_df = ledger_df.set_index('datetime')
        
        # Sort by datetime
        ledger_df = ledger_df.sort_index()
        
        # Drop time column (we have datetime as index)
        ledger_df = ledger_df.drop(columns=['time'])
        
        # Drop hash column if not requested
        if not include_hash:
            ledger_df = ledger_df.drop(columns=['hash'])
        
        return ledger_df
        
    except Exception as e:
        print(f"Error retrieving user ledger updates: {e}")
        return None

# %% ../nbs/hyper_account.ipynb 26
def get_account_summary(address, info=None, lookback_days=3650):
    """
    Retrieves a comprehensive account summary including deposits, withdrawals, and current value.
    
    Args:
        address (str): User's wallet address
        info (Info, optional): Hyperliquid Info client. If None, creates a new one.
        lookback_days (int, optional): Number of days to look back for ledger history. 
            Defaults to 3650 (~10 years).
    
    Returns:
        dict: Dictionary containing:
            - total_deposits (float): Total amount deposited in USDC
            - total_withdrawals (float): Total amount withdrawn in USDC
            - net_deposits (float): Net deposits (deposits - withdrawals) in USDC
            - current_value (float): Current total account value in USDC
            - spot_value (float): Value in spot account in USDC
            - perp_margin (float): Margin in perpetual account in USDC
            - perp_position_value (float): Current notional value of perpetual positions in USDC
            - total_pnl (float): Total profit/loss (current_value - net_deposits)
            - pnl_percentage (float): P&L as percentage of net deposits
            - unrealized_pnl (float): Unrealized P&L from open perpetual positions
        Returns None if error occurs.
    
    Examples:
        # Get account summary
        summary = get_account_summary(address, info=info)
        
        if summary:
            print(f"Net Deposits: ${summary['net_deposits']:,.2f}")
            print(f"Current Value: ${summary['current_value']:,.2f}")
            print(f"Total P&L: ${summary['total_pnl']:,.2f} ({summary['pnl_percentage']:.2f}%)")
    
    Notes:
        - All values are in USDC
        - P&L includes both realized and unrealized gains/losses
        - Current value = spot balance + perp margin + current position values
        - Uses current market prices from info.all_mids() to value positions
        - Requires access to user_state, spot_user_state, and ledger updates
    """
    # Initialize info client if not provided
    if info is None:
        from hyperliquid.info import Info
        from hyperliquid.utils import constants
        address_temp, info, exchange = setup(base_url=constants.MAINNET_API_URL, skip_ws=True)
    
    try:
        # Get ledger updates for deposits/withdrawals
        ledger_df = get_user_ledger_updates(address, info=info, lookback=lookback_days)
        
        if ledger_df is not None:
            deposits = ledger_df[ledger_df['type'] == 'deposit']['usdc'].sum()
            withdrawals = ledger_df[ledger_df['type'] == 'withdraw']['usdc'].sum()
            net_deposits = deposits - withdrawals
        else:
            deposits = 0
            withdrawals = 0
            net_deposits = 0
        
        # Get current market prices
        all_mids = info.all_mids()
        
        # PnL again ... using fills
        a = get_user_fills(address, info=info)
        a['last_px'] = a['coin'].apply(lambda x: float(all_mids.get(x,0)))
        a['sign'] = a['side_readable'].apply(lambda x: 1 if x == 'Buy' else -1)
        a['tc'] = a['feeToken'].apply(lambda x: 0 if x == 'USDC' else 1)
        a['pnl'] = a['sign'] * (a['last_px']-a['px']) * (a['sz']-a['tc']*a['fee'])
        a['pnl'] = np.where(a['coin']=='USDC', a['pnl']-a['fee'], a['pnl'])
        pnl_dollar = a.groupby('coin')['pnl'].apply(lambda x: x.sum().round(2)).to_dict()
        print(f"PnL: {pnl_dollar}")

        # Get user state for perpetual account
        user_state = info.user_state(address)
        margin_summary = user_state["marginSummary"]
        
        # Get margin value (this is the perp account)
        perp_margin = float(margin_summary["accountValue"])
        
        # Cash in the perp account
        cash_in_perp = float(user_state['withdrawable'])

        # Get spot balances
        spot_user_state = info.spot_user_state(address)
        spot_balances = spot_user_state["balances"]

        # Calculate spot value using current prices
        spot_value = 0
        positions = []
        for balance in spot_balances:
            coin = balance["coin"]
            if coin != "USDC":
                coin = spot_tickers_here(coin=coin, base='USDC',info=info)
            total = float(balance["total"])
            
            if coin == "USDC":
                spot_value += total
                current_price = 1.0
                usdc_value = total
            else:
                # Get current price for this coin
                current_price = float(all_mids.get(coin, 0))
                usdc_value = total * current_price
                spot_value += usdc_value
            unrealized_pnl = pnl_dollar.get(coin, 0)
            positions.append({
                    'coin': coin,
                    'signed_position': total,
                    'price': current_price,
                    'current_usdc_value': usdc_value,
                    'unrealized_pnl': unrealized_pnl,  # Spot positions don't have unrealized PnL tracked
                    'leverage':1,
                    'ROE': 0, #basically p&l
                    'liquidation_px': 0, # in USDC
                    'max_leverage': 0,
                    'cum_funding': 0,  # all time funding in USDC
                    'asset_type': 'spot'
                })
        
        # Get perpetual positions and calculate their current value
        perp_positions = user_state.get("assetPositions", [])
        unrealized_pnl = 0
        position_notional = 0
        
        for position in perp_positions: # Sum over all tokens.... # Only perpetual positions
            if position["position"]["szi"] != "0":  # Only non-zero positions
                coin = position["position"]["coin"]
                size = float(position["position"]["szi"])
                unrealized = float(position["position"]["unrealizedPnl"])
                leverage = float(position["position"]["leverage"]["value"])
                roe = float(position["position"]["returnOnEquity"])
                lpx = float(position["position"]["liquidationPx"])
                mxl = float(position["position"]["maxLeverage"])
                cfd = float(position["position"]["cumFunding"]["allTime"]) # since open and since change also available,but not extracted here
                # Get current market price
                current_price = float(all_mids.get(coin, 0))
                
                unrealized_pnl += unrealized
                # Current notional value = abs(size) * current_price
                usdc_value = abs(size) * current_price
                position_notional += usdc_value
                positions.append({
                    'coin': coin,
                    'signed_position': size,
                    'price': current_price,
                    'current_usdc_value': usdc_value,
                    'unrealized_pnl': unrealized, #pnl in USDC
                    'leverage':leverage,
                    'ROE': roe, #basically p&l
                    'liquidation_px': lpx, # in USDC
                    'max_leverage': mxl,
                    'cum_funding': cfd,  # all time funding in USDC
                    'asset_type': 'perp'
                })

        positions_df = pd.DataFrame(positions)
        # Total current value = spot value + perp margin + current position values
        current_value = spot_value + perp_margin #+ position_notional
        
        # Calculate P&L
        total_pnl = current_value - net_deposits
        pnl_percentage = (total_pnl / net_deposits * 100) if net_deposits > 0 else 0

        # info time
        balance_time = pd.to_datetime(user_state['time'], unit='ms')
        positions_df['datetime'] = balance_time

        summary = {
            'total_deposits': deposits,
            'total_withdrawals': withdrawals,
            'net_deposits': net_deposits,
            'current_value': current_value,
            'spot_value': spot_value,
            'perp_value': perp_margin,
            'perp_position_value': position_notional,
            'total_pnl': total_pnl,
            'pnl_percentage': pnl_percentage,
            'unrealized_pnl': unrealized_pnl,
            'cash_in_perp': cash_in_perp,
            'when': balance_time.strftime('%Y-%m-%d %H:%M:%S')  # info time
        }
        return summary, positions_df
        
    except Exception as e:
        print(f"Error retrieving account summary: {e}")
        return None
