# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/hyperliquid.ipynb.

# %% auto 0
__all__ = ['setup', 'retrieve_hyperliquid_perp_price', 'spot_tickers', 'retrieve_hyperliquid_spot_price', 'hyperliquid_tokens',
           'funding_calc', 'retrieve_hyperliquid_funding_history', 'retrieve_hyperliquid_data',
           'retrieve_hyperliquid_l2_snapshot', 'hyperliquid_mids', 'save_hyperliquid_file', 'HyperliquidDataManager',
           'HyperliquidPerpManager', 'HyperliquidSpotManager', 'HyperliquidFundingManager']

# %% ../nbs/hyperliquid.ipynb 3
#from nbdev.showdoc import *
import json
#from typing import List, Dict, Tuple, Optional, Union, Any, Callable
from hyperliquid.utils import constants
import os
import time

import eth_account
from eth_account.signers.local import LocalAccount
from hyperliquid.exchange import Exchange
from hyperliquid.info import Info
import pandas as pd
from datetime import datetime
import datetime as dt
import numpy as np

# %% ../nbs/hyperliquid.ipynb 5
def setup(base_url=None, skip_ws=False, perp_dexs=None,config='../config_hyperliquid.json'):
    # This function is copied from hyperliquid-python-sdk/examples/example_utils.py
    # for setting up the environment in our script.
    # config_path = os.path.join(os.path.dirname(__file__), "config.json")
    config_path = config
    with open(config_path) as f:
        config = json.load(f)
    account: LocalAccount = eth_account.Account.from_key(config["secret_key"])
    address = config["account_address"]
    if address == "":
        address = account.address
    print("Running with account address:", address)
    if address != account.address:
        print("Running with agent address:", account.address)
    info = Info(base_url, skip_ws, perp_dexs=perp_dexs)
    user_state = info.user_state(address)
    spot_user_state = info.spot_user_state(address)
    margin_summary = user_state["marginSummary"]
    if float(margin_summary["accountValue"]) == 0 and len(spot_user_state["balances"]) == 0:
        print("Not running the example because the provided account has no equity.")
        url = info.base_url.split(".", 1)[1]
        error_string = f"No accountValue:\nIf you think this is a mistake, make sure that {address} has a balance on {url}.\nIf address shown is your API wallet address, update the config to specify the address of your account, not the address of the API wallet."
        raise Exception(error_string)
    exchange = Exchange(account, base_url, account_address=address, perp_dexs=perp_dexs)
    return address, info, exchange

# %% ../nbs/hyperliquid.ipynb 9
def retrieve_hyperliquid_perp_price(coin="ETH", interval="1h", 
                                end_date=datetime.now(dt.timezone.utc).strftime('%Y-%m-%dT%H:%M:%SZ'),
                                start_date=(datetime.now(dt.timezone.utc)-pd.Timedelta(days=2)).strftime('%Y-%m-%dT%H:%M:%SZ'),
                                info=None):
    """
    Retrieves historical candle data from Hyperliquid for a given coin and time interval.

    Args:
        coin (str, optional): Coin symbol (e.g. "ETH"). Defaults to "ETH".
        interval (str, optional): Candle interval ("1m", "5m", "15m", "1h", "4h", "1d"). Defaults to "1h".
        end_date (str, optional): End datetime in ISO 8601 format. Defaults to current UTC time.
        start_date (str, optional): Start datetime in ISO 8601 format. Defaults to 2 days before end_date.
        info (Info, optional): Hyperliquid Info client. If None, creates a new one.

    Returns:
        pandas.DataFrame: DataFrame containing the OHLCV data with columns:
            - datetime: Timestamp for the candle (UTC)
            - open: Opening price of the interval
            - high: Highest traded price in the interval
            - low: Lowest traded price in the interval
            - close: Closing price of the interval
            - volume: Trading volume in the interval
            - coin: Coin symbol
        Returns None if the API request fails or returns no data.

    Notes:
        - All datetime values are in UTC timezone
        - Requires Hyperliquid Info client to be initialized
    """
    if info is None:
        from hyperliquid.info import Info
        from hyperliquid.utils import constants
        address, info, exchange = setup(base_url=constants.MAINNET_API_URL, skip_ws=True)
    try:
        # Convert datetime strings to Unix milliseconds timestamps
        start_dt = pd.to_datetime(start_date)
        end_dt = pd.to_datetime(end_date)
        
        start_time_ms = int(start_dt.timestamp() * 1000)
        end_time_ms = int(end_dt.timestamp() * 1000)
        
        # Get candles from Hyperliquid
        candles = info.candles_snapshot(name=coin, interval=interval, 
                                       startTime=start_time_ms, endTime=end_time_ms)
        
        if not candles:
            return None
        
        # Convert to DataFrame
        df = pd.DataFrame(candles)
        
        # Convert timestamp to datetime
        df['datetime'] = pd.to_datetime(df['t'], unit='ms')
        
        # Rename columns to match coinbase format
        df = df.rename(columns={
            'o': 'open',
            'h': 'high', 
            'l': 'low',
            'c': 'close',
            'v': 'volume'
        })
        
        # Add coin column
        df['coin'] = coin
        
        # Sort by datetime and reorder columns
        df = df.sort_values(by='datetime')
        df = df[['datetime', 'open', 'high', 'low', 'close', 'volume', 'coin']]
        df = df.astype({'open': 'float64', 'high': 'float64', 'low': 'float64', 'close': 'float64', 'volume': 'float64'})
        
        return df.reset_index(drop=True)
        
    except Exception as e:
        print(f"Error retrieving candles for {coin}: {e}")
        return None

# %% ../nbs/hyperliquid.ipynb 14
def spot_tickers(coin="ETH", base='USDC',info=None):
    """
    Retrieves current tickers for a given coin.

    Args:
        coin (str, optional): Coin symbol (e.g. "ETH"). Defaults to "ETH".
        base (str, optional): Coin symbol (e.g. "USDC"). Defaults to "USDC".
        info (Info, optional): Hyperliquid Info client. If None, creates a new one.

    Returns:
        spot ticker (non intuitive symbol)
        Returns None if the API request fails or returns no data.

    Notes:
        - Requires Hyperliquid Info client to be initialized
    """
    if info is None:
        address, info, exchange = setup(base_url=constants.MAINNET_API_URL, skip_ws=True)
    maps =info.spot_meta_and_asset_ctxs()
    # change of ticker for ETH and BTC.... they should have U in front of the name.... Hyperliquid's naming convention...
    if coin.upper() == "ETH":
        coin = "UETH"
    elif coin.upper() == "BTC":
        coin = "UBTC"
    elif coin.upper() == "DOGE":
        coin = "UDOGE"
    elif coin.upper() == "SOL":
        coin = "USOL"
    if base.upper() == "USDC":
        base = "USDC"
    # Find the index of the coin and base in the universe of tokens
    # If not found, return None
    id_coin, id_base = None, None
    for token_ctx in maps[0]['tokens']:
        if token_ctx['name'] == coin:
            id_coin = token_ctx['index']
        elif token_ctx['name'] == base:
            id_base = token_ctx['index']
    if id_coin is None or id_base is None:
        return None
    for i in maps[0]['universe']:
        if i['tokens'] == [id_coin,id_base]:
            return i['name']
    id_coin, id_base = None, None
    for token_ctx in maps[0]['tokens']:
        if token_ctx['name'] == coin:
            id_coin = token_ctx['index']
        elif token_ctx['name'] == base:
            id_base = token_ctx['index']
    if id_coin is None or id_base is None:
        return None
    for i in maps['universe']:
        if i['tokens'] == [id_coin,id_base]:
            return i['name']
    return None                

# %% ../nbs/hyperliquid.ipynb 18
def retrieve_hyperliquid_spot_price(coin="ETH", base='USDC',interval="1h", 
                                end_date=datetime.now(dt.timezone.utc).strftime('%Y-%m-%dT%H:%M:%SZ'),
                                start_date=(datetime.now(dt.timezone.utc)-pd.Timedelta(days=2)).strftime('%Y-%m-%dT%H:%M:%SZ'),
                                info=None,recheck=False):
    # get the ticker for the given coin
    if recheck:
        ticker = spot_tickers(coin=coin,base=base,info=info)
    else:
        spot_dict = {'BTC': '@142',
                        'ETH': '@151',
                        'SOL': '@156',
                        'HYPE': '@107',
                        'TRUMP': '@9',
                        'BERA': '@117',
                        'PUMP': '@20'}
        ticker = spot_dict.get(coin.upper(), None)
        if ticker is None:
            ticker = spot_tickers(coin=coin,base=base,info=info)

    if ticker is None:
        print(f"{coin} is not listed.")
        return pd.DataFrame()
    # get the price for the given ticker... same function used for perpetuals but
    # ticker price is different...
    try:
        price = retrieve_hyperliquid_perp_price(coin=ticker, interval=interval, 
                                end_date=end_date,
                                start_date=start_date,
                                info=info)
        price['coin'] = coin
        return price
    except Exception as e:
        print(f"Error retrieving price for {coin}: {e}")
        return None

# %% ../nbs/hyperliquid.ipynb 24
def hyperliquid_tokens(info=None,rm_delisted=True):
    if info is None:
        from hyperliquid.info import Info
        from hyperliquid.utils import constants
        address, info, exchange = setup(base_url=constants.MAINNET_API_URL, skip_ws=True)

    # Get universe details
    tokens = info.meta_and_asset_ctxs()[0].get('universe')
    
    # Create DataFrame with token details
    df = pd.DataFrame(tokens)
    df['isDelisted'] = ~df['isDelisted'].isna()
    df['onlyIsolated'] = ~df['onlyIsolated'].isna()
    df = df.loc[(~df['isDelisted']) & (~df['onlyIsolated'])]
    return df

# %% ../nbs/hyperliquid.ipynb 28
def funding_calc(rate,premium,max_rate=0.0005,min_rate=-0.0005):
    return premium+max(min(rate-premium, max_rate), min_rate)

# %% ../nbs/hyperliquid.ipynb 29
def retrieve_hyperliquid_funding_history(coin="ETH", 
                                        end_date=datetime.now(dt.timezone.utc).strftime('%Y-%m-%dT%H:%M:%SZ'),
                                        start_date=(datetime.now(dt.timezone.utc)-pd.Timedelta(days=2)).strftime('%Y-%m-%dT%H:%M:%SZ'),
                                        info=None,
                                        calc=False):
    """
    Retrieves funding rate history from Hyperliquid for a given coin and time period.

    Args:
        coin (str, optional): Coin symbol (e.g. "ETH"). Defaults to "ETH".
        end_date (str, optional): End datetime in ISO 8601 format. Defaults to current UTC time.
        start_date (str, optional): Start datetime in ISO 8601 format. Defaults to 7 days before end_date.
        info (Info, optional): Hyperliquid Info client. If None, creates a new one.

    Returns:
        pandas.DataFrame: DataFrame containing the funding history with columns:
            - datetime: Timestamp for the funding rate (UTC)
            - funding_rate: The funding rate value
            - premium: The premium component
            - coin: Coin symbol
        Returns None if the API request fails or returns no data.

    Notes:
        - All datetime values are in UTC timezone
        - Funding rates are typically updated every hour
        - Requires Hyperliquid Info client to be initialized
    """
    if info is None:
        from hyperliquid.info import Info
        from hyperliquid.utils import constants
        address, info, exchange = setup(base_url=constants.MAINNET_API_URL, skip_ws=True)
    
    try:
        # Convert datetime strings to Unix milliseconds timestamps
        start_dt = pd.to_datetime(start_date)
        end_dt = pd.to_datetime(end_date)
        
        start_time_ms = int(start_dt.timestamp() * 1000)
        end_time_ms = int(end_dt.timestamp() * 1000)
        
        # Get funding history from Hyperliquid
        funding_rates = info.funding_history(
            name=coin,
            startTime=start_time_ms,
            endTime=end_time_ms
        )
        
        if not funding_rates:
            return None
        
        # Convert to DataFrame
        df = pd.DataFrame(funding_rates)
        
        # Convert timestamp from milliseconds to datetime
        df['datetime'] = pd.to_datetime(df['time'], unit='ms')
        
        # Rename columns for clarity
        df = df.rename(columns={
            'fundingRate': 'funding_rate',
            'premium': 'premium'
        })
        
        # Convert multiple columns to float
        df[['funding_rate', 'premium']] = df[['funding_rate', 'premium']].astype(float)
        # Add coin column
        df['coin'] = coin
        
        # Sort by datetime and reorder columns
        df = df.sort_values(by='datetime')
        df = df[['datetime', 'funding_rate', 'premium', 'coin']]
        
        # Drop the original time column if it exists
        if 'time' in df.columns:
            df = df.drop(columns=['time'])
        
        df['fund_calc'] = df['funding_rate']
        if calc:
            vectorized_funding_calc = np.vectorize(funding_calc)
            df['fund_calc'] = vectorized_funding_calc(df['funding_rate'], df['premium'])
        
        return df.reset_index(drop=True)
        
    except Exception as e:
        print(f"Error retrieving funding history for {coin}: {e}")
        return None

# %% ../nbs/hyperliquid.ipynb 33
def retrieve_hyperliquid_data(ticker="ETH", 
                              data_type="perp",
                              start_date=None,
                              end_date=None,
                              lookback=2,
                              interval="1h",
                              base="USDC",
                              round_to_hour=False,
                              info=None):
    """
    Unified function to retrieve funding rates, perpetual prices, or spot prices from Hyperliquid.
    
    Args:
        ticker (str, optional): Coin symbol (e.g. "ETH", "BTC"). Defaults to "ETH".
        data_type (str, optional): Type of data to retrieve - "funding", "perp", or "spot". Defaults to "perp".
        start_date (str, optional): Start date as string. Can be:
            - ISO format: "2024-01-15T10:30:00Z" or "2024-01-15T10:30:00"
            - Date only: "2024-01-15"
            - If None, calculated from lookback. Defaults to None.
        end_date (str, optional): End date as string (same formats as start_date).
            - If None, uses current UTC time. Defaults to None.
        lookback (int, optional): Number of days to look back from end_date if start_date is None. 
            Defaults to 2.
        interval (str, optional): Candle interval for perp/spot data ("1m", "5m", "15m", "1h", "4h", "1d"). 
            Defaults to "1h". Not used for funding rates.
        base (str, optional): Base currency for spot prices (e.g. "USDC"). Defaults to "USDC".
            Not used for perp or funding rates.
        round_to_hour (bool, optional): If True, rounds start_date and end_date to nearest hour.
            Useful for funding rates which update hourly. Defaults to False.
        info (Info, optional): Hyperliquid Info client. If None, creates a new one.
    
    Returns:
        pandas.DataFrame: DataFrame containing the requested data with appropriate columns:
            - For "funding": datetime, funding_rate, premium, coin
            - For "perp": datetime, open, high, low, close, volume, coin
            - For "spot": datetime, open, high, low, close, volume, coin
        Returns None if the API request fails or returns no data.
    
    Examples:
        # Get 7 days of funding rates for ETH, rounded to hour
        df = retrieve_hyperliquid_data("ETH", "funding", lookback=7, round_to_hour=True, info=info)
        
        # Get perp prices between specific dates with 4h interval
        df = retrieve_hyperliquid_data("BTC", "perp", 
                                      start_date="2024-01-01", 
                                      end_date="2024-01-15",
                                      interval="4h", info=info)
        
        # Get spot prices for last 30 days with 1h interval
        df = retrieve_hyperliquid_data("ETH", "spot", lookback=30, 
                                      interval="1h", base="USDC", info=info)
    
    Notes:
        - All datetime values are in UTC timezone
        - Valid data_type values: "funding", "perp", "spot"
        - Funding rates are updated hourly, so round_to_hour=True is recommended
        - Requires Hyperliquid Info client to be initialized
    """
    # Validate data_type
    valid_types = ["funding", "perp", "spot"]
    if data_type not in valid_types:
        raise ValueError(f"data_type must be one of {valid_types}, got '{data_type}'")
    
    # Initialize info client if not provided
    if info is None:
        from hyperliquid.info import Info
        from hyperliquid.utils import constants
        address, info, exchange = setup(base_url=constants.MAINNET_API_URL, skip_ws=True)
    
    # Handle end_date
    if end_date is None:
        end_dt = pd.Timestamp.now(tz='UTC')
    else:
        # Parse end_date string
        try:
            end_dt = pd.to_datetime(end_date)
        except Exception as e:
            print(f"Error parsing end_date '{end_date}': {e}")
            return None

    # Handle start_date
    if start_date is None:
        # Calculate from lookback
        start_dt = end_dt - pd.Timedelta(days=lookback)
    else:
        # Parse start_date string
        try:
            start_dt = pd.to_datetime(start_date)
        except Exception as e:
            print(f"Error parsing start_date '{start_date}': {e}")
            return None
    
    # Convert to ISO format strings
    start_date_str = start_dt.strftime('%Y-%m-%dT%H:%M:%SZ')
    end_date_str = end_dt.strftime('%Y-%m-%dT%H:%M:%SZ')
    
    # Call appropriate function based on data_type
    try:
        if data_type == "funding":
            df = retrieve_hyperliquid_funding_history(
                coin=ticker,
                start_date=start_date_str,
                end_date=end_date_str,
                info=info
                )
            if round_to_hour:
                df['datetime'] = df['datetime'].apply(lambda x: x.round('h'))
            return df
        elif data_type == "perp":
            return retrieve_hyperliquid_perp_price(
                coin=ticker,
                interval=interval,
                start_date=start_date_str,
                end_date=end_date_str,
                info=info
            )
        elif data_type == "spot":
            return retrieve_hyperliquid_spot_price(
                coin=ticker,
                base=base,
                interval=interval,
                start_date=start_date_str,
                end_date=end_date_str,
                info=info
            )
    except Exception as e:
        print(f"Error retrieving {data_type} data for {ticker}: {e}")
        return None

# %% ../nbs/hyperliquid.ipynb 43
def retrieve_hyperliquid_l2_snapshot(coin="ETH", info=None):
    """
    Retrieves current L2 order book snapshot from Hyperliquid for a given coin.
    
    Args:
        coin (str, optional): Coin symbol (e.g. "ETH", "BTC"). Defaults to "ETH".
        info (Info, optional): Hyperliquid Info client. If None, creates a new one.
    
    Returns:
        pandas.DataFrame: DataFrame containing the order book snapshot with columns:
            - datetime: Timestamp of the snapshot (UTC)
            - side: Order side ("bid" or "ask")
            - price: Price level
            - size: Total size at this price level
            - num_orders: Number of orders at this price level
        Returns None if the API request fails or returns no data.
    
    Notes:
        - This is a snapshot at the moment the function is called
        - All datetime values are in UTC timezone
        - Bids are sorted from highest to lowest price
        - Asks are sorted from lowest to highest price
        - Requires Hyperliquid Info client to be initialized
    """
    if info is None:
        address, info, exchange = setup(base_url=constants.MAINNET_API_URL, skip_ws=True)
    
    try:
        # Get L2 snapshot from Hyperliquid
        l2_data = info.l2_snapshot(name=coin)
        
        if not l2_data or 'levels' not in l2_data:
            return None
        
        # Convert timestamp to datetime
        timestamp = pd.to_datetime(l2_data['time'], unit='ms')
        
        # Extract bid and ask levels
        bids = l2_data['levels'][0]  # First element is bids
        asks = l2_data['levels'][1]  # Second element is asks
        
        # Create list to store all rows
        rows = []
        
        # Process bids
        for bid in bids:
            rows.append({
                'datetime': timestamp,
                'side': 'bid',
                'price': float(bid['px']),
                'size': float(bid['sz']),
                'num_orders': int(bid['n'])
            })
        
        # Process asks
        for ask in asks:
            rows.append({
                'datetime': timestamp,
                'side': 'ask',
                'price': float(ask['px']),
                'size': float(ask['sz']),
                'num_orders': int(ask['n'])
            })
        
        # Create DataFrame
        df = pd.DataFrame(rows)
        
        # Reorder columns for consistency
        df = df[['datetime', 'side', 'price', 'size', 'num_orders']]
        
        return df
        
    except Exception as e:
        print(f"Error retrieving L2 snapshot for {coin}: {e}")
        return None

# %% ../nbs/hyperliquid.ipynb 47
def hyperliquid_mids(coin=None,info=None,typecast_to_float=True):
    """
    Retrieves current mid prices from Hyperliquid for specified coins or all available coins.
    
    Args:
        coins (list, optional): List of coin symbols (e.g. ["ETH", "BTC"]). 
            If None, retrieves mids for all available coins. Defaults to None.
        info (Info, optional): Hyperliquid Info client. If None, creates a new one.
    
    Returns:
        dictionary: Dictionary containing mid prices with keys as coin symbols and values as mid prices
        Returns None if the API request fails or returns no data.
    
    Examples:
        # Get mid prices for specific coins
        dic = hyperliquid_mids(info=info)
    
    Notes:
        - This is a snapshot at the moment the function is called
        - Mid price is calculated as (best_bid + best_ask) / 2
        - Requires Hyperliquid Info client to be initialized
    """
    if info is None:
        address, info, exchange = setup(base_url=constants.MAINNET_API_URL, skip_ws=True)
    
    try:
        # Get all mid prices from Hyperliquid
        mids_data = info.all_mids()
        
        if not mids_data:
            return None
        if coin:
            if coin not in mids_data.keys():
                return None
            return float(mids_data[coin])
            
        if typecast_to_float: # turn off to save time...
            for i in mids_data.keys():
                mids_data[i] = float(mids_data[i])

        return mids_data
        
    except Exception as e:
        print(f"Error retrieving mid prices: {e}")
        return None


# %% ../nbs/hyperliquid.ipynb 52
def save_hyperliquid_file(df, folder_path, file_name, type="parquet"):
    """
    Save a pandas DataFrame to a file in either CSV or Parquet format.

    Args:
        df (pandas.DataFrame): The DataFrame to save
        folder_path (str): Directory path where the file will be saved
        file_name (str): Name of the file without extension
        type (str, optional): File format - either "csv" or "parquet". Defaults to "parquet"

    The function saves the DataFrame to the specified path, handling the file extension automatically.
    For CSV files, the index is not saved. For Parquet files, default Parquet settings are used.
    Creates the folder if it doesn't exist.
    """
    # Create folder if it doesn't exist
    if not os.path.exists(folder_path):
        os.makedirs(folder_path)
    
    if type == "csv":
        df.to_csv(f"{folder_path}/{file_name}.csv", index=False)
    elif type == "parquet":
        df.to_parquet(f"{folder_path}/{file_name}.parquet")
    else:
        raise ValueError(f"Type {type} not supported. Use 'csv' or 'parquet'")

# %% ../nbs/hyperliquid.ipynb 53
class HyperliquidDataManager:
    """
    Base class for managing Hyperliquid data files.
    
    Handles reading, updating, and saving data for different data types (perp, spot, funding).
    Data is stored in organized folders by data type, with files named by token and interval.
    
    Args:
        ticker (str or list, optional): Token symbol(s) to manage. If None, uses all available tokens.
        data_dir (str, optional): Base directory for data storage. Defaults to "../data/hyperliquid"
        interval (str, optional): Time interval for data ("1m", "5m", "15m", "1h", "4h", "1d"). 
            Defaults to "1h". Not used for funding data.
        file_type (str, optional): File format - "parquet" or "csv". Defaults to "parquet"
        update (bool, optional): If True, checks for and downloads new data. Defaults to False
        save (bool, optional): If True, saves updated data back to file. Defaults to False
        refresh_hours (int, optional): Hours of data to refresh when updating. Defaults to 24
        info (Info, optional): Hyperliquid Info client. If None, creates a new one.
        verbose (bool, optional): If True, prints progress messages. Defaults to True
    
    Attributes:
        data_type (str): Type of data managed by this instance ("perp", "spot", "funding")
        data (dict): Dictionary mapping tickers to their DataFrames
    """
    
    def __init__(self, ticker=None, data_dir="../data/hyperliquid", interval="1h",
                 file_type="parquet", update=False, save=False, refresh_hours=24,
                 info=None, verbose=True,data_type=None):
        self.ticker = ticker
        self.data_dir = data_dir
        self.interval = interval
        self.file_type = file_type
        self.update = update
        self.save = save
        self.refresh_hours = refresh_hours
        self.info = info
        self.verbose = verbose
        self.data = {}
        self.data_type = data_type  # To be set by derived classes
        
        # Initialize info client if needed
        if self.info is None:
            if self.update:
                raise ValueError("To update data, Info client must be provided")
        
        # Get list of tickers to process
        self._initialize_tickers()
        
        # Create directory structure if needed
        self._ensure_directories()
    
    def _initialize_tickers(self):
        """Initialize the list of tickers to process."""
        if self.ticker is None:
            # Get all available tokens
            self.tickers = self._get_all_tickers()
        elif isinstance(self.ticker, str):
            self.tickers = [self.ticker]
        elif isinstance(self.ticker, list):
            self.tickers = self.ticker
        else:
            raise ValueError("ticker must be None, str, or list")
    
    def _get_all_tickers(self):
        """Get all available tickers for this data type. To be implemented by derived classes."""
        raise NotImplementedError("Derived classes must implement _get_all_tickers")
    
    def _ensure_directories(self):
        """Create directory structure if it doesn't exist."""
        if self.data_type is None:
            raise ValueError("data_type must be set by derived class")
        
        full_path = os.path.join(self.data_dir, self.data_type)
        if not os.path.exists(full_path):
            os.makedirs(full_path)
            if self.verbose:
                print(f"Created directory: {full_path}")
    
    def _get_file_path(self, ticker):
        """Get the file path for a given ticker."""
        file_name = f"{ticker}_{self.interval}.{self.file_type}" if self.data_type != "funding" else f"{ticker}.{self.file_type}"
        return os.path.join(self.data_dir, self.data_type, file_name)
    
    def _load_existing_data(self, ticker):
        """Load existing data from file if it exists."""
        file_path = self._get_file_path(ticker)
        
        if not os.path.exists(file_path):
            return None
        
        try:
            if self.file_type == "parquet":
                df = pd.read_parquet(file_path)
            elif self.file_type == "csv":
                df = pd.read_csv(file_path)
                df['datetime'] = pd.to_datetime(df['datetime'])
            else:
                raise ValueError(f"Unsupported file type: {self.file_type}")
            
            # Ensure datetime is timezone-aware
            if df['datetime'].dt.tz is None:
                df['datetime'] = pd.to_datetime(df['datetime'], utc=True)
            
            if self.verbose:
                print(f"Loaded {len(df)} rows for {ticker} from {file_path}")
            
            return df
        except Exception as e:
            print(f"Error loading data for {ticker}: {e}")
            return None
    
    def _get_new_data(self, ticker, start_date=None):
        """Retrieve new data from Hyperliquid. To be implemented by derived classes."""
        raise NotImplementedError("Derived classes must implement _get_new_data")
    
    def _update_data(self, ticker, existing_df):
        """Update existing data with new records."""
        import datetime as dt
        
        # Calculate start date for new data
        if existing_df is not None and not existing_df.empty:
            # Get last date and subtract refresh hours
            last_date = pd.to_datetime(existing_df['datetime'].max())
            cutoff_time = last_date - dt.timedelta(hours=self.refresh_hours)
            
            # Remove data within refresh window
            rows_before = len(existing_df)
            existing_df = existing_df[existing_df['datetime'] < cutoff_time]
            rows_removed = rows_before - len(existing_df)
            
            if self.verbose and rows_removed > 0:
                print(f"  Removed {rows_removed} rows from last {self.refresh_hours} hours for refresh")
            
            start_date = cutoff_time.strftime('%Y-%m-%dT%H:%M:%SZ')
        else:
            # No existing data, get default lookback
            start_date = None
        
        # Get new data
        new_df = self._get_new_data(ticker, start_date)
        
        if new_df is None or new_df.empty:
            if self.verbose:
                print(f"  No new data retrieved for {ticker}")
            return existing_df
        
        # Combine with existing data
        if existing_df is not None and not existing_df.empty:
            combined_df = pd.concat([existing_df, new_df], ignore_index=True)
            combined_df = combined_df.drop_duplicates(subset=['datetime'], keep='last')
            combined_df = combined_df.sort_values('datetime').reset_index(drop=True)
            
            if self.verbose:
                print(f"  Updated {ticker}: {len(existing_df)} -> {len(combined_df)} rows")
            
            return combined_df
        else:
            if self.verbose:
                print(f"  Downloaded {len(new_df)} rows for {ticker}")
            return new_df
    
    def _save_data(self, ticker, df):
        """Save data to file."""
        if df is None or df.empty:
            if self.verbose:
                print(f"  No data to save for {ticker}")
            return
        
        file_path = self._get_file_path(ticker)
        
        try:
            if self.file_type == "parquet":
                df.to_parquet(file_path, index=False)
            elif self.file_type == "csv":
                df.to_csv(file_path, index=False)
            else:
                raise ValueError(f"Unsupported file type: {self.file_type}")
            
            if self.verbose:
                print(f"  Saved {len(df)} rows for {ticker} to {file_path}")
        except Exception as e:
            print(f"Error saving data for {ticker}: {e}")
    
    def load_data(self):
        """
        Load data for all tickers.
        
        Returns:
            dict: Dictionary mapping tickers to their DataFrames
        """
        for ticker in self.tickers:
            if self.verbose:
                print(f"Processing {ticker}...")
            
            # Load existing data
            df = self._load_existing_data(ticker)
            
            # Update if requested
            if self.update:
                df = self._update_data(ticker, df)
                
                # Save if requested
                if self.save and df is not None:
                    self._save_data(ticker, df)
            
            # Store in data dictionary
            if df is not None:
                self.data[ticker] = df
        
        return self.data
    
    def get_data(self, ticker=None):
        """
        Get data for specific ticker(s).
        
        Args:
            ticker (str or list, optional): Ticker(s) to retrieve. If None, returns all data.
        
        Returns:
            pandas.DataFrame or dict: DataFrame if single ticker, dict if multiple
        """
        if ticker is None:
            return self.data
        elif isinstance(ticker, str):
            return self.data.get(ticker)
        elif isinstance(ticker, list):
            return {t: self.data.get(t) for t in ticker if t in self.data}
        else:
            raise ValueError("ticker must be None, str, or list")

# %% ../nbs/hyperliquid.ipynb 55
class HyperliquidPerpManager(HyperliquidDataManager):
    """
    Manager for Hyperliquid perpetual futures data.
    
    Handles reading, updating, and saving perpetual price data (OHLCV).
    Data is stored in the 'perp' subfolder with files named as {ticker}_{interval}.{file_type}
    
    Args:
        ticker (str or list, optional): Token symbol(s) to manage. If None, uses all available perp tokens.
        data_dir (str, optional): Base directory for data storage. Defaults to "../data/hyperliquid"
        interval (str, optional): Time interval for data ("1m", "5m", "15m", "1h", "4h", "1d"). 
            Defaults to "1h".
        file_type (str, optional): File format - "parquet" or "csv". Defaults to "parquet"
        update (bool, optional): If True, checks for and downloads new data. Defaults to False
        save (bool, optional): If True, saves updated data back to file. Defaults to False
        refresh_hours (int, optional): Hours of data to refresh when updating. Defaults to 24
        info (Info, optional): Hyperliquid Info client. If None, creates a new one.
        verbose (bool, optional): If True, prints progress messages. Defaults to True
    
    Examples:
        # Load existing perp data for ETH
        manager = HyperliquidPerpManager(ticker="ETH", interval="1h", info=info)
        eth_data = manager.data["ETH"]
        
        # Update and save data for multiple tokens
        manager = HyperliquidPerpManager(
            ticker=["ETH", "BTC", "SOL"],
            interval="4h",
            update=True,
            save=True,
            refresh_hours=48,
            info=info
        )
        
        # Load all available perp tokens
        manager = HyperliquidPerpManager(update=True, save=True, info=info)
    """
    
    def __init__(self, ticker=None, data_dir="../data/hyperliquid", interval="1h",
                 file_type="parquet", update=False, save=False, refresh_hours=24,
                 info=None, verbose=True):
        # Set data type before calling parent constructor
        self.data_type = "perp"
        
        # Call parent constructor
        super().__init__(ticker=ticker, data_dir=data_dir, interval=interval,
                        file_type=file_type, update=update, save=save,
                        refresh_hours=refresh_hours, info=info, verbose=verbose,data_type="perp")
        
        # Load and optionally update data for all tickers
        self._process_all_tickers()
    
    def _get_all_tokens_from_folder(self):
        """Get all tokens from the perp folder."""
        # Load all files in the perp folder
        file_list = os.listdir(self.data_dir+'/perp')
        #perp_files = [ff for ff in file_list if f"_{self.interval}" in ff)]
        
        # Extract token names from file names
        tickers = [f.split("_")[0] for f in file_list if self.interval in f]
        
        # Return unique tokens
        return list(set(tickers))

    def _get_all_tickers(self):
        """Get all available perpetual tokens from Hyperliquid."""
        try:
            if self.info is not None:
                tickers = hyperliquid_tokens(info=self.info)
                tickers = tickers['name'].tolist()
                if self.verbose:
                    print(f"Found {len(tickers)} perpetual tokens")
                return tickers
            else:
                # read the list of tokens from folder
                tickers = self._get_all_tokens_from_folder()
                if self.verbose:
                    print(f"Found {len(tickers)} perpetual tokens from folder")
                return tickers
        except Exception as e:
            print(f"Error getting perpetual tokens: {e}")
            return []
    
    def _get_new_data(self, ticker, start_date=None):
        """
        Retrieve new perpetual price data from Hyperliquid.
        
        Args:
            ticker (str): Token symbol
            start_date (str, optional): Start date for data retrieval. If None, uses refresh_hours.
        
        Returns:
            pandas.DataFrame: DataFrame with OHLCV data or None if error
        """
        try:
            # Calculate date range
            if start_date is None:
                end_date = None  # Will use current time
                lookback_days = self.refresh_hours / 24
            else:
                end_date = None
                lookback_days = None
            
            # Use retrieve_hyperliquid_data to get perp prices
            df = retrieve_hyperliquid_data(
                ticker=ticker,
                data_type="perp",
                start_date=start_date,
                end_date=end_date,
                lookback=lookback_days if lookback_days else 2,
                interval=self.interval,
                info=self.info
            )
            
            if df is not None and not df.empty:
                if self.verbose:
                    print(f"Retrieved {len(df)} new rows for {ticker}")
            
            return df
            
        except Exception as e:
            print(f"Error retrieving new data for {ticker}: {e}")
            return None
    
    def _update_data(self, ticker, existing_df):
        """
        Update existing perpetual data with new records.
        
        Args:
            ticker (str): Token symbol
            existing_df (pandas.DataFrame): Existing data
        
        Returns:
            pandas.DataFrame: Updated DataFrame with new data merged
        """
        import datetime as dt
        
        # Calculate start date for new data
        if existing_df is not None and not existing_df.empty:
            # Get the most recent datetime from existing data
            max_datetime = existing_df['datetime'].max()
            
            # Subtract refresh_hours to ensure overlap and catch any missing data
            start_datetime = max_datetime - pd.Timedelta(hours=self.refresh_hours)
            start_date = start_datetime.strftime('%Y-%m-%dT%H:%M:%SZ')
            
            if self.verbose:
                print(f"Updating {ticker} from {start_date}")
        else:
            # No existing data, get default lookback period
            start_date = None
            if self.verbose:
                print(f"No existing data for {ticker}, fetching initial data")
        
        # Get new data
        new_df = self._get_new_data(ticker, start_date)
        
        if new_df is None or new_df.empty:
            if self.verbose:
                print(f"No new data retrieved for {ticker}")
            return existing_df
        
        # Merge with existing data
        if existing_df is not None and not existing_df.empty:
            # Combine dataframes
            existing_df['datetime'] = existing_df['datetime'].dt.tz_localize(None)
            combined_df = pd.concat([existing_df, new_df], ignore_index=True)
            # Remove duplicates based on datetime, keeping the last occurrence
            combined_df = combined_df.drop_duplicates(subset=['datetime'], keep='last')
            # Sort by datetime
            combined_df = combined_df.sort_values('datetime').reset_index(drop=True)
            if self.verbose:
                new_rows = len(combined_df) - len(existing_df)
                print(f"Added {new_rows} new rows for {ticker}")
            
            return combined_df
        else:
            # No existing data, return new data
            return new_df.sort_values('datetime').reset_index(drop=True)
    
    def _process_all_tickers(self):
        """Load and optionally update data for all tickers."""
        n = len(self.tickers)
        if self.verbose:
            print(f"Processing {n} perpetual tokens...")
        max_per_batch = 1200/100 #120 per min and 100 weighrs per hyperliquid api
        waiting_time = 2
        if n>max_per_batch:
            waiting_time = 60/max_per_batch
        if self.update:
            print(f"Warning: Processing {n} perpetual tokens exceeds the maximum number of requests per minute. Adjusting waiting time to {waiting_time} seconds.")
        for ticker in self.tickers:
            try:
                # Load existing data
                existing_df = self._load_existing_data(ticker)
                
                # Update if requested
                if self.update:
                    df = self._update_data(ticker, existing_df)
                else:
                    df = existing_df
                
                # Store in data dictionary
                if df is not None and not df.empty:
                    self.data[ticker] = df
                    
                    # Save if requested
                    if self.save and df is not None:
                        file_path = self._get_file_path(ticker)
                        save_hyperliquid_file(df, os.path.dirname(file_path), 
                                            os.path.splitext(os.path.basename(file_path))[0],
                                            type=self.file_type)
                        if self.verbose:
                            print(f"Saved {len(df)} rows for {ticker}")
                
            except Exception as e:
                print(f"Error processing {ticker}: {e}")
                continue
            if self.update:
                time.sleep(waiting_time)  # wait to avoid rate limiting
    
    def get_data(self, ticker=None):
        """
        Get data for a specific ticker.
        
        Args:
            ticker (str): Token symbol
        
        Returns:
            pandas.DataFrame: Data for the ticker or None if not available
        """
        if ticker is None:
            #a = [self.data.get(i) for i in self.tickers]
            return pd.concat(self.data.values())
        return self.data.get(ticker)
    
    def refresh_ticker(self, ticker, save=None):
        """
        Refresh data for a specific ticker.
        
        Args:
            ticker (str): Token symbol
            save (bool, optional): Override instance save setting. If None, uses instance setting.
        
        Returns:
            pandas.DataFrame: Updated data for the ticker
        """
        if ticker not in self.tickers:
            print(f"Ticker {ticker} not in managed tickers")
            return None
        
        # Load existing data
        existing_df = self._load_existing_data(ticker)
        
        # Update data
        df = self._update_data(ticker, existing_df)
        
        # Store in data dictionary
        if df is not None and not df.empty:
            self.data[ticker] = df
            
            # Save if requested
            should_save = save if save is not None else self.save
            if should_save:
                file_path = self._get_file_path(ticker)
                save_hyperliquid_file(df, os.path.dirname(file_path),
                                    os.path.splitext(os.path.basename(file_path))[0],
                                    type=self.file_type)
                if self.verbose:
                    print(f"Saved {len(df)} rows for {ticker}")
        
        return df

# %% ../nbs/hyperliquid.ipynb 60
class HyperliquidSpotManager(HyperliquidDataManager):
    """
    Manager for Hyperliquid spot market data.
    
    Handles reading, updating, and saving spot price data (OHLCV).
    Data is stored in the 'spot' subfolder with files named as {ticker}_{base}_{interval}.{file_type}
    
    Args:
        ticker (str or list, optional): Token symbol(s) to manage. If None, uses all available spot tokens.
        base (str, optional): Base currency for spot pairs. Defaults to "USDC".
        data_dir (str, optional): Base directory for data storage. Defaults to "../data/hyperliquid"
        interval (str, optional): Time interval for data ("1m", "5m", "15m", "1h", "4h", "1d"). 
            Defaults to "1h".
        file_type (str, optional): File format - "parquet" or "csv". Defaults to "parquet"
        update (bool, optional): If True, checks for and downloads new data. Defaults to False
        save (bool, optional): If True, saves updated data back to file. Defaults to False
        refresh_hours (int, optional): Hours of data to refresh when updating. Defaults to 24
        info (Info, optional): Hyperliquid Info client. If None, creates a new one.
        verbose (bool, optional): If True, prints progress messages. Defaults to True
    
    Examples:
        # Load existing spot data for ETH/USDC
        manager = HyperliquidSpotManager(ticker="ETH", base="USDC", interval="1h", info=info)
        eth_data = manager.data["ETH"]
        
        # Update and save data for multiple tokens
        manager = HyperliquidSpotManager(
            ticker=["ETH", "BTC", "SOL"],
            base="USDC",
            interval="4h",
            update=True,
            save=True,
            refresh_hours=48,
            info=info
        )
        
        # Load all available spot tokens
        manager = HyperliquidSpotManager(base="USDC", update=True, save=True, info=info)
    """
    
    def __init__(self, ticker=None, base="USDC", data_dir="../data/hyperliquid", interval="1h",
                 file_type="parquet", update=False, save=False, refresh_hours=24,
                 info=None, verbose=True):
        # Set data type and base before calling parent constructor
        self.data_type = "spot"
        self.base = base
        
        # Call parent constructor
        super().__init__(ticker=ticker, data_dir=data_dir, interval=interval,
                        file_type=file_type, update=update, save=save,
                        refresh_hours=refresh_hours, info=info, verbose=verbose, data_type="spot")
        
        # Load and optionally update data for all tickers
        self._process_all_tickers()
    
    def _get_file_path(self, ticker):
        """
        Get the file path for a specific ticker, including base currency.
        
        Args:
            ticker (str): Token symbol
        
        Returns:
            str: Full file path
        """
        file_name = f"{ticker}_{self.base}_{self.interval}.{self.file_type}"
        return os.path.join(self.data_dir, self.data_type, file_name)
    
    def _get_all_tickers(self):
        """Get all available spot tokens from Hyperliquid for the specified base."""
        try:
            spot_dict = {'BTC': '@142',
                        'ETH': '@151',
                        'SOL': '@156',
                        'HYPE': '@107',
                        'TRUMP': '@9',
                        'BERA': '@117',
                        'PUMP': '@20'}
            
            tickers = list(spot_dict.keys())
            if self.verbose:
                print(f"Found {len(tickers)} spot tokens for {self.base}")
            return tickers
        except Exception as e:
            print(f"Error getting spot tokens: {e}")
            return []
    
    def _get_new_data(self, ticker, start_date=None):
        """
        Retrieve new spot price data from Hyperliquid.
        
        Args:
            ticker (str): Token symbol
            start_date (str, optional): Start date for data retrieval. If None, uses refresh_hours.
        
        Returns:
            pandas.DataFrame: DataFrame with OHLCV data or None if error
        """
        try:
            # Calculate date range
            if start_date is None:
                end_date = None  # Will use current time
                lookback_days = self.refresh_hours / 24
            else:
                end_date = None
                lookback_days = None
            
            # Use retrieve_hyperliquid_data to get spot prices
            df = retrieve_hyperliquid_data(
                ticker=ticker,
                data_type="spot",
                start_date=start_date,
                end_date=end_date,
                lookback=lookback_days if lookback_days else 2,
                interval=self.interval,
                base=self.base,
                info=self.info
            )
            
            if df is not None and not df.empty:
                if self.verbose:
                    print(f"Retrieved {len(df)} new rows for {ticker}/{self.base}")
            
            return df
            
        except Exception as e:
            print(f"Error retrieving new data for {ticker}/{self.base}: {e}")
            return None
    
    def _update_data(self, ticker, existing_df):
        """
        Update existing spot data with new records.
        
        Args:
            ticker (str): Token symbol
            existing_df (pandas.DataFrame): Existing data
        
        Returns:
            pandas.DataFrame: Updated DataFrame with new data merged
        """
        import datetime as dt
        
        # Calculate start date for new data
        if existing_df is not None and not existing_df.empty:
            # Get the most recent datetime from existing data
            max_datetime = existing_df['datetime'].max()
            
            # Subtract refresh_hours to ensure overlap and catch any missing data
            start_datetime = max_datetime - pd.Timedelta(hours=self.refresh_hours)
            start_date = start_datetime.strftime('%Y-%m-%dT%H:%M:%SZ')
            
            if self.verbose:
                print(f"Updating {ticker}/{self.base} from {start_date}")
        else:
            # No existing data, get default lookback period
            start_date = None
            if self.verbose:
                print(f"No existing data for {ticker}/{self.base}, fetching initial data")
        
        # Get new data
        new_df = self._get_new_data(ticker, start_date)
        
        if new_df is None or new_df.empty:
            if self.verbose:
                print(f"No new data retrieved for {ticker}/{self.base}")
            return existing_df
        
        # Merge with existing data
        if existing_df is not None and not existing_df.empty:
            # Combine dataframes
            existing_df['datetime'] = existing_df['datetime'].dt.tz_localize(None)
            combined_df = pd.concat([existing_df, new_df], ignore_index=True)
            # Remove duplicates based on datetime, keeping the last occurrence
            combined_df = combined_df.drop_duplicates(subset=['datetime'], keep='last')
            # Sort by datetime
            combined_df = combined_df.sort_values('datetime').reset_index(drop=True)
            if self.verbose:
                new_rows = len(combined_df) - len(existing_df)
                print(f"Added {new_rows} new rows for {ticker}/{self.base}")
            
            return combined_df
        else:
            # No existing data, return new data
            return new_df.sort_values('datetime').reset_index(drop=True)
    
    def _process_all_tickers(self):
        """Load and optionally update data for all tickers."""
        for ticker in self.tickers:
            try:
                # Load existing data
                existing_df = self._load_existing_data(ticker)
                
                # Update if requested
                if self.update:
                    df = self._update_data(ticker, existing_df)
                else:
                    df = existing_df
                
                # Store in data dictionary
                if df is not None and not df.empty:
                    self.data[ticker] = df
                    
                    # Save if requested
                    if self.save and df is not None:
                        file_path = self._get_file_path(ticker)
                        save_hyperliquid_file(df, os.path.dirname(file_path), 
                                            os.path.splitext(os.path.basename(file_path))[0],
                                            type=self.file_type)
                        if self.verbose:
                            print(f"Saved {len(df)} rows for {ticker}/{self.base}")
                
            except Exception as e:
                print(f"Error processing {ticker}/{self.base}: {e}")
                continue
    
    def get_data(self, ticker=None):
        """
        Get data for a specific ticker.
        
        Args:
            ticker (str): Token symbol
        
        Returns:
            pandas.DataFrame: Data for the ticker or None if not available
        """
        if ticker is None:
            return pd.concat(self.data.values())
        return self.data.get(ticker)
    
    def refresh_ticker(self, ticker, save=None):
        """
        Refresh data for a specific ticker.
        
        Args:
            ticker (str): Token symbol
            save (bool, optional): Override instance save setting. If None, uses instance setting.
        
        Returns:
            pandas.DataFrame: Updated data for the ticker
        """
        if ticker not in self.tickers:
            print(f"Ticker {ticker} not in managed tickers")
            return None
        
        # Load existing data
        existing_df = self._load_existing_data(ticker)
        
        # Update data
        df = self._update_data(ticker, existing_df)
        
        # Store in data dictionary
        if df is not None and not df.empty:
            self.data[ticker] = df
            
            # Save if requested
            should_save = save if save is not None else self.save
            if should_save:
                file_path = self._get_file_path(ticker)
                save_hyperliquid_file(df, os.path.dirname(file_path),
                                    os.path.splitext(os.path.basename(file_path))[0],
                                    type=self.file_type)
                if self.verbose:
                    print(f"Saved {len(df)} rows for {ticker}/{self.base}")
        
        return df

# %% ../nbs/hyperliquid.ipynb 68
class HyperliquidFundingManager(HyperliquidDataManager):
    """
    Manager for Hyperliquid funding rate data.
    
    Handles reading, updating, and saving funding rate data.
    Data is stored in the 'funding' subfolder with files named as {ticker}.{file_type}
    
    Args:
        ticker (str or list, optional): Token symbol(s) to manage. If None, uses all available tokens.
        data_dir (str, optional): Base directory for data storage. Defaults to "../data/hyperliquid"
        file_type (str, optional): File format - "parquet" or "csv". Defaults to "parquet"
        update (bool, optional): If True, checks for and downloads new data. Defaults to False
        save (bool, optional): If True, saves updated data back to file. Defaults to False
        refresh_hours (int, optional): Hours of data to refresh when updating. Defaults to 24
        round_to_hour (bool, optional): If True, rounds datetime to nearest hour. Defaults to True
        info (Info, optional): Hyperliquid Info client. If None, creates a new one.
        verbose (bool, optional): If True, prints progress messages. Defaults to True
    
    Examples:
        # Load existing funding data for ETH
        manager = HyperliquidFundingManager(ticker="ETH", info=info)
        eth_data = manager.data["ETH"]
        
        # Update and save data for multiple tokens
        manager = HyperliquidFundingManager(
            ticker=["ETH", "BTC", "SOL"],
            update=True,
            save=True,
            refresh_hours=48,
            info=info
        )
        
        # Load all available tokens
        manager = HyperliquidFundingManager(update=True, save=True, info=info)
    """
    
    def __init__(self, ticker=None, data_dir="../data/hyperliquid", 
                 file_type="parquet", update=False, save=False, refresh_hours=24,
                 round_to_hour=True, info=None, verbose=True):
        # Set data type and round_to_hour before calling parent constructor
        self.data_type = "funding"
        self.round_to_hour = round_to_hour
        
        # Call parent constructor (interval not used for funding data, but required by parent)
        super().__init__(ticker=ticker, data_dir=data_dir, interval="1h",
                        file_type=file_type, update=update, save=save,
                        refresh_hours=refresh_hours, info=info, verbose=verbose,
                        data_type="funding")
        
        # Load and optionally update data for all tickers
        self._process_all_tickers()
    
    def _get_all_tokens_from_folder(self):
        """Get all tokens from the perp folder."""
        # Load all files in the perp folder
        file_list = os.listdir(self.data_dir+'/funding')
        
        # Extract token names from file names
        tickers = [f.split(".")[0] for f in file_list]
        
        # Return unique tokens
        return list(set(tickers))

    def _get_all_tickers(self):
        """Get all available tokens from Hyperliquid."""
        try:
            if self.info is None:
                # read the list of tokens from folder
                tickers = self._get_all_tokens_from_folder()
                if self.verbose:
                    print(f"Found {len(tickers)} perpetual tokens from folder")
                return tickers
            tickers = hyperliquid_tokens(info=self.info)
            tickers = tickers['name'].tolist()
            if self.verbose:
                print(f"Found {len(tickers)} tokens")
            return tickers
        except Exception as e:
            print(f"Error getting tokens: {e}")
            return []
    
    def _get_file_path(self, ticker):
        """Get the file path for a given ticker (funding data doesn't use interval in filename)."""
        file_name = f"{ticker}.{self.file_type}"
        return os.path.join(self.data_dir, self.data_type, file_name)
    
    def _get_new_data(self, ticker, start_date=None):
        """
        Retrieve new funding rate data from Hyperliquid.
        
        Args:
            ticker (str): Token symbol
            start_date (str, optional): Start date for data retrieval. If None, uses refresh_hours.
        
        Returns:
            pandas.DataFrame: DataFrame with funding rate data or None if error
        """
        try:
            # Calculate date range
            if start_date is None:
                end_date = None  # Will use current time
                lookback_days = self.refresh_hours / 24
            else:
                end_date = None
                lookback_days = None
            
            # Use retrieve_hyperliquid_data to get funding rates
            df = retrieve_hyperliquid_data(
                ticker=ticker,
                data_type="funding",
                start_date=start_date,
                end_date=end_date,
                lookback=lookback_days if lookback_days else 2,
                round_to_hour=self.round_to_hour,
                info=self.info
            )
            
            if df is not None and not df.empty:
                if self.verbose:
                    print(f"Retrieved {len(df)} new rows for {ticker}")
            
            return df
            
        except Exception as e:
            print(f"Error retrieving new data for {ticker}: {e}")
            return None
    
    def _update_data(self, ticker, existing_df):
        """
        Update existing funding rate data with new records.
        
        Args:
            ticker (str): Token symbol
            existing_df (pandas.DataFrame): Existing data
        
        Returns:
            pandas.DataFrame: Updated DataFrame with new data merged
        """
        import datetime as dt
        
        # Calculate start date for new data
        if existing_df is not None and not existing_df.empty:
            # Get the most recent datetime from existing data
            max_datetime = existing_df['datetime'].max()
            
            # Subtract refresh_hours to ensure overlap and catch any missing data
            start_datetime = max_datetime - pd.Timedelta(hours=self.refresh_hours)
            start_date = start_datetime.strftime('%Y-%m-%dT%H:%M:%SZ')
            
            if self.verbose:
                print(f"Updating {ticker} from {start_date}")
        else:
            # No existing data, get default lookback period
            start_date = None
            if self.verbose:
                print(f"No existing data for {ticker}, fetching initial data")
        
        # Get new data
        new_df = self._get_new_data(ticker, start_date)
        
        if new_df is None or new_df.empty:
            if self.verbose:
                print(f"No new data retrieved for {ticker}")
            return existing_df
        
        # Merge with existing data
        if existing_df is not None and not existing_df.empty:
            # Combine dataframes
            existing_df['datetime'] = existing_df['datetime'].dt.tz_localize(None)
            combined_df = pd.concat([existing_df, new_df], ignore_index=True)
            # Remove duplicates based on datetime, keeping the last occurrence
            combined_df = combined_df.drop_duplicates(subset=['datetime'], keep='last')
            # Sort by datetime
            combined_df = combined_df.sort_values('datetime').reset_index(drop=True)
            if self.verbose:
                new_rows = len(combined_df) - len(existing_df)
                print(f"Added {new_rows} new rows for {ticker}")
            
            return combined_df
        else:
            # No existing data, return new data
            return new_df.sort_values('datetime').reset_index(drop=True)
    
    def _process_all_tickers(self):
        """Load and optionally update data for all tickers."""
        n = len(self.tickers)
        max_per_batch = 1200/100 #120 per min and 100 weighrs per hyperliquid api
        waiting_time = 2
        if n>max_per_batch:
            waiting_time = 60/max_per_batch
        if self.update:
            print(f"Warning: Processing {n} perpetual tokens exceeds the maximum number of requests per minute. Adjusting waiting time to {waiting_time} seconds.")
        for ticker in self.tickers:
            try:
                # Load existing data
                existing_df = self._load_existing_data(ticker)
                
                # Update if requested
                if self.update:
                    df = self._update_data(ticker, existing_df)
                else:
                    df = existing_df
                
                # Store in data dictionary
                if df is not None and not df.empty:
                    self.data[ticker] = df
                    
                    # Save if requested
                    if self.save and df is not None:
                        file_path = self._get_file_path(ticker)
                        save_hyperliquid_file(df, os.path.dirname(file_path), 
                                            os.path.splitext(os.path.basename(file_path))[0],
                                            type=self.file_type)
                        if self.verbose:
                            print(f"Saved {len(df)} rows for {ticker}")
                
            except Exception as e:
                print(f"Error processing {ticker}: {e}")
                continue
            if self.update:
                time.sleep(waiting_time)    
    def get_data(self, ticker=None):
        """
        Get data for a specific ticker.
        
        Args:
            ticker (str): Token symbol
        
        Returns:
            pandas.DataFrame: Data for the ticker or None if not available
        """
        if ticker is None:
            return pd.concat(self.data.values())
        return self.data.get(ticker)
    
    def refresh_ticker(self, ticker, save=None):
        """
        Refresh data for a specific ticker.
        
        Args:
            ticker (str): Token symbol
            save (bool, optional): Override instance save setting. If None, uses instance setting.
        
        Returns:
            pandas.DataFrame: Updated data for the ticker
        """
        if ticker not in self.tickers:
            print(f"Ticker {ticker} not in managed tickers")
            return None
        
        # Load existing data
        existing_df = self._load_existing_data(ticker)
        
        # Update data
        df = self._update_data(ticker, existing_df)
        
        # Store in data dictionary
        if df is not None and not df.empty:
            self.data[ticker] = df
            
            # Save if requested
            should_save = save if save is not None else self.save
            if should_save:
                file_path = self._get_file_path(ticker)
                save_hyperliquid_file(df, os.path.dirname(file_path),
                                    os.path.splitext(os.path.basename(file_path))[0],
                                    type=self.file_type)
                if self.verbose:
                    print(f"Saved {len(df)} rows for {ticker}")
        
        return df
