#!/usr/bin/env python3
"""
NOT NULL Column Addition Handler for DataFlow Migration System

Implements safe NOT NULL column addition to populated tables with comprehensive
default value strategies and data integrity preservation.

This handler addresses one of the most critical and risky migration scenarios
in production databases.
"""

import asyncio
import logging
from abc import ABC, abstractmethod
from dataclasses import dataclass
from datetime import datetime
from enum import Enum
from typing import Any, Dict, List, Optional, Tuple, Union

import asyncpg

logger = logging.getLogger(__name__)


class DefaultValueType(Enum):
    """Supported default value types for NOT NULL column addition."""

    STATIC = "static"
    COMPUTED = "computed"
    FUNCTION = "function"
    CONDITIONAL = "conditional"
    SEQUENCE = "sequence"
    FOREIGN_KEY = "foreign_key"


class AdditionResult(Enum):
    """Results of NOT NULL column addition operation."""

    SUCCESS = "success"
    CONSTRAINT_VIOLATION = "constraint_violation"
    PERFORMANCE_TIMEOUT = "performance_timeout"
    ROLLBACK_REQUIRED = "rollback_required"
    VALIDATION_FAILED = "validation_failed"


@dataclass
class ColumnDefinition:
    """Definition of a column to be added."""

    name: str
    data_type: str
    nullable: bool = False
    default_value: Optional[Any] = None
    default_type: DefaultValueType = DefaultValueType.STATIC
    default_expression: Optional[str] = None
    foreign_key_reference: Optional[str] = None
    check_constraints: List[str] = None
    unique: bool = False
    indexed: bool = False

    def __post_init__(self):
        if self.check_constraints is None:
            self.check_constraints = []


@dataclass
class ValidationResult:
    """Result of safety validation for NOT NULL column addition."""

    is_safe: bool
    issues: List[str]
    warnings: List[str]
    estimated_time: Optional[float] = None  # seconds
    recommended_batch_size: Optional[int] = None

    def __post_init__(self):
        if self.issues is None:
            self.issues = []
        if self.warnings is None:
            self.warnings = []


@dataclass
class NotNullAdditionPlan:
    """Plan for NOT NULL column addition execution."""

    table_name: str
    column: ColumnDefinition
    execution_strategy: str
    batch_size: int = 10000
    timeout_seconds: int = 300
    rollback_on_failure: bool = True
    validate_constraints: bool = True
    performance_monitoring: bool = True

    # Execution details
    estimated_duration: Optional[float] = None
    affected_rows: Optional[int] = None
    constraint_dependencies: List[str] = None
    rollback_plan: Optional[Dict[str, Any]] = None

    def __post_init__(self):
        if self.constraint_dependencies is None:
            self.constraint_dependencies = []


@dataclass
class AdditionExecutionResult:
    """Result of executing NOT NULL column addition."""

    result: AdditionResult
    execution_time: float
    affected_rows: int
    rollback_executed: bool = False
    error_message: Optional[str] = None
    constraint_violations: List[str] = None
    performance_metrics: Optional[Dict[str, Any]] = None

    def __post_init__(self):
        if self.constraint_violations is None:
            self.constraint_violations = []


class DefaultValueStrategy(ABC):
    """Abstract base class for default value strategies."""

    @abstractmethod
    def generate_default_expression(self, column: ColumnDefinition) -> str:
        """Generate SQL expression for default value."""
        pass

    @abstractmethod
    def validate_against_constraints(
        self, column: ColumnDefinition, existing_constraints: List[Dict[str, Any]]
    ) -> ValidationResult:
        """Validate default value against existing constraints."""
        pass

    @abstractmethod
    def estimate_performance_impact(
        self, table_name: str, row_count: int, column: ColumnDefinition
    ) -> Dict[str, Any]:
        """Estimate performance impact of applying this default."""
        pass


class StaticDefaultStrategy(DefaultValueStrategy):
    """Strategy for static default values."""

    def generate_default_expression(self, column: ColumnDefinition) -> str:
        """Generate SQL for static default value."""
        if column.default_value is None:
            raise ValueError("Static default strategy requires a default_value")

        # Handle different data types appropriately
        if column.data_type.upper() in ["VARCHAR", "TEXT", "CHAR"]:
            return f"'{column.default_value}'"
        elif column.data_type.upper() in ["INTEGER", "INT", "BIGINT", "SMALLINT"]:
            return str(column.default_value)
        elif column.data_type.upper() in ["BOOLEAN", "BOOL"]:
            return "TRUE" if column.default_value else "FALSE"
        elif column.data_type.upper() in ["FLOAT", "DOUBLE", "DECIMAL", "NUMERIC"]:
            return str(column.default_value)
        elif column.data_type.upper().startswith("TIMESTAMP"):
            if isinstance(column.default_value, datetime):
                return f"'{column.default_value.isoformat()}'"
            return f"'{column.default_value}'"
        else:
            # Generic approach - quote if string, otherwise literal
            if isinstance(column.default_value, str):
                return f"'{column.default_value}'"
            return str(column.default_value)

    def validate_against_constraints(
        self, column: ColumnDefinition, existing_constraints: List[Dict[str, Any]]
    ) -> ValidationResult:
        """Validate static default against constraints."""
        issues = []
        warnings = []

        # Validate against check constraints
        for constraint in existing_constraints:
            # Handle case where constraint might be a string (error recovery)
            if isinstance(constraint, str):
                warnings.append(f"Constraint parsing issue: {constraint}")
                continue
            elif not isinstance(constraint, dict):
                warnings.append(f"Unexpected constraint format: {type(constraint)}")
                continue

            if constraint.get("constraint_type") == "CHECK":
                # This would require SQL parsing/evaluation - simplified for now
                warnings.append(
                    f"Manual validation needed for CHECK constraint: {constraint.get('constraint_definition', 'unknown')}"
                )

        # Validate data type compatibility
        try:
            self._validate_type_compatibility(column)
        except ValueError as e:
            issues.append(str(e))

        return ValidationResult(
            is_safe=len(issues) == 0, issues=issues, warnings=warnings
        )

    def estimate_performance_impact(
        self, table_name: str, row_count: int, column: ColumnDefinition
    ) -> Dict[str, Any]:
        """Estimate performance for static default."""
        # Static defaults are very fast - single DDL operation
        estimated_time = min(0.1 + (row_count * 0.000001), 5.0)  # Max 5 seconds

        return {
            "estimated_seconds": estimated_time,
            "strategy": "single_ddl",
            "batch_required": row_count > 1000000,
            "recommended_batch_size": 50000 if row_count > 1000000 else None,
        }

    def _validate_type_compatibility(self, column: ColumnDefinition) -> None:
        """Validate that default value is compatible with column type."""
        if column.default_value is None:
            return

        data_type = column.data_type.upper()
        value = column.default_value

        if data_type in ["INTEGER", "INT", "BIGINT", "SMALLINT"]:
            if not isinstance(value, (int, float)) or (
                isinstance(value, float) and not value.is_integer()
            ):
                raise ValueError(
                    f"Default value {value} is not compatible with integer type {data_type}"
                )

        elif data_type in ["BOOLEAN", "BOOL"]:
            if not isinstance(value, (bool, int)) or (
                isinstance(value, int) and value not in [0, 1]
            ):
                raise ValueError(
                    f"Default value {value} is not compatible with boolean type"
                )


class ComputedDefaultStrategy(DefaultValueStrategy):
    """Strategy for computed default values based on existing data."""

    def generate_default_expression(self, column: ColumnDefinition) -> str:
        """Generate SQL for computed default value."""
        if not column.default_expression:
            raise ValueError("Computed default strategy requires default_expression")
        return column.default_expression

    def validate_against_constraints(
        self, column: ColumnDefinition, existing_constraints: List[Dict[str, Any]]
    ) -> ValidationResult:
        """Validate computed default against constraints."""
        issues = []
        warnings = []

        if not column.default_expression:
            issues.append("Computed strategy requires a default_expression")
            return ValidationResult(is_safe=False, issues=issues, warnings=warnings)

        # Basic SQL expression validation (simplified)
        if not self._is_valid_sql_expression(column.default_expression):
            issues.append(f"Invalid SQL expression: {column.default_expression}")

        warnings.append(
            "Computed defaults require careful testing with production data"
        )

        return ValidationResult(
            is_safe=len(issues) == 0, issues=issues, warnings=warnings
        )

    def estimate_performance_impact(
        self, table_name: str, row_count: int, column: ColumnDefinition
    ) -> Dict[str, Any]:
        """Estimate performance for computed default."""
        # Computed defaults can be expensive - need row-by-row evaluation
        # Adjusted for modern PostgreSQL performance characteristics
        # Must be slower than static defaults due to batched processing overhead
        base_time = 0.15  # Base overhead for batched processing (higher than static)
        per_row_time = (
            0.000005  # 5μs per row (realistic for simple computed expressions)
        )
        estimated_time = base_time + (row_count * per_row_time)

        return {
            "estimated_seconds": estimated_time,
            "strategy": "batched_update",
            "batch_required": True,
            "recommended_batch_size": 10000,
            "requires_table_scan": True,
        }

    def _is_valid_sql_expression(self, expression: str) -> bool:
        """Basic validation of SQL expression."""
        # Simplified validation - in production this would be more sophisticated
        dangerous_keywords = ["DROP", "DELETE", "TRUNCATE", "ALTER", "CREATE"]
        expression_upper = expression.upper()

        for keyword in dangerous_keywords:
            if keyword in expression_upper:
                return False

        # Must contain CASE or function calls or column references
        return any(keyword in expression_upper for keyword in ["CASE", "(", "WHEN"])


class FunctionDefaultStrategy(DefaultValueStrategy):
    """Strategy for function-based default values."""

    def generate_default_expression(self, column: ColumnDefinition) -> str:
        """Generate SQL for function default value."""
        if not column.default_expression:
            raise ValueError(
                "Function default strategy requires default_expression with function name"
            )

        # Handle common function patterns
        func_expr = column.default_expression.upper()

        if func_expr in ["CURRENT_TIMESTAMP", "NOW()"]:
            return "CURRENT_TIMESTAMP"
        elif func_expr in ["CURRENT_DATE"]:
            return "CURRENT_DATE"
        elif func_expr.startswith("GENERATE_UUID"):
            return "gen_random_uuid()"
        else:
            return column.default_expression

    def validate_against_constraints(
        self, column: ColumnDefinition, existing_constraints: List[Dict[str, Any]]
    ) -> ValidationResult:
        """Validate function default against constraints."""
        issues = []
        warnings = []

        if not column.default_expression:
            issues.append("Function strategy requires a default_expression")
            return ValidationResult(is_safe=False, issues=issues, warnings=warnings)

        # Validate function exists and is appropriate
        func_name = column.default_expression.upper()

        if func_name in ["CURRENT_TIMESTAMP", "NOW()", "CURRENT_DATE"]:
            if not column.data_type.upper().startswith(("TIMESTAMP", "DATE")):
                issues.append(
                    f"Function {func_name} incompatible with type {column.data_type}"
                )

        elif "UUID" in func_name:
            if column.data_type.upper() not in ["UUID", "VARCHAR", "TEXT"]:
                issues.append(
                    f"UUID function incompatible with type {column.data_type}"
                )

        return ValidationResult(
            is_safe=len(issues) == 0, issues=issues, warnings=warnings
        )

    def estimate_performance_impact(
        self, table_name: str, row_count: int, column: ColumnDefinition
    ) -> Dict[str, Any]:
        """Estimate performance for function default."""
        # Function defaults are usually fast - single DDL
        # Adjusted to be faster than computed but competitive with static
        estimated_time = min(0.12 + (row_count * 0.0000005), 10.0)

        return {
            "estimated_seconds": estimated_time,
            "strategy": "single_ddl",
            "batch_required": False,
            "function_overhead": True,
        }


class NotNullColumnHandler:
    """
    Handles safe NOT NULL column addition to populated tables.

    Provides comprehensive default value strategies, constraint validation,
    and safe execution with rollback capabilities.
    """

    def __init__(self, connection_manager: Optional[Any] = None):
        """Initialize the NOT NULL column handler."""
        self.connection_manager = connection_manager
        self.logger = logging.getLogger(f"{__name__}.{self.__class__.__name__}")

        # Initialize strategies
        self.strategies = {
            DefaultValueType.STATIC: StaticDefaultStrategy(),
            DefaultValueType.COMPUTED: ComputedDefaultStrategy(),
            DefaultValueType.FUNCTION: FunctionDefaultStrategy(),
            # TODO: Add more strategies in TODO-136B
        }

    async def plan_not_null_addition(
        self,
        table_name: str,
        column: ColumnDefinition,
        connection: Optional[asyncpg.Connection] = None,
    ) -> NotNullAdditionPlan:
        """
        Plan NOT NULL column addition with comprehensive analysis.

        Args:
            table_name: Target table name
            column: Column definition with default value strategy
            connection: Database connection (optional)

        Returns:
            NotNullAdditionPlan with execution strategy and safety analysis
        """
        self.logger.info(f"Planning NOT NULL addition: {table_name}.{column.name}")

        if connection is None:
            connection = await self._get_connection()

        try:
            # Analyze table structure and constraints
            table_info = await self._analyze_table_structure(table_name, connection)
            row_count = table_info["row_count"]
            existing_constraints = table_info["constraints"]

            # Validate column definition
            validation = await self._validate_column_definition(
                column, existing_constraints
            )
            if not validation.is_safe:
                raise ValueError(
                    f"Column definition validation failed: {validation.issues}"
                )

            # Select optimal strategy
            strategy_name = self._select_execution_strategy(column, row_count)

            # Estimate performance
            strategy = self.strategies[column.default_type]
            perf_estimate = strategy.estimate_performance_impact(
                table_name, row_count, column
            )

            # Create execution plan
            plan = NotNullAdditionPlan(
                table_name=table_name,
                column=column,
                execution_strategy=strategy_name,
                batch_size=perf_estimate.get("recommended_batch_size", 10000),
                timeout_seconds=min(
                    max(int(perf_estimate["estimated_seconds"] * 2), 60), 1800
                ),
                estimated_duration=perf_estimate["estimated_seconds"],
                affected_rows=row_count,
                constraint_dependencies=self._extract_constraint_dependencies(
                    existing_constraints
                ),
            )

            # Generate rollback plan
            plan.rollback_plan = await self._generate_rollback_plan(plan, connection)

            self.logger.info(
                f"Plan created for {table_name}.{column.name}: "
                f"{strategy_name}, {row_count} rows, ~{perf_estimate['estimated_seconds']:.1f}s"
            )

            return plan

        except Exception as e:
            self.logger.error(f"Planning failed for {table_name}.{column.name}: {e}")
            raise

    async def validate_addition_safety(
        self, plan: NotNullAdditionPlan, connection: Optional[asyncpg.Connection] = None
    ) -> ValidationResult:
        """
        Validate safety of NOT NULL column addition plan.

        Args:
            plan: Execution plan to validate
            connection: Database connection (optional)

        Returns:
            ValidationResult with safety assessment
        """
        self.logger.info(
            f"Validating addition safety: {plan.table_name}.{plan.column.name}"
        )

        if connection is None:
            connection = await self._get_connection()

        issues = []
        warnings = []

        try:
            # Validate table exists and is accessible
            table_exists = await self._validate_table_access(
                plan.table_name, connection
            )
            if not table_exists:
                issues.append(
                    f"Table {plan.table_name} does not exist or is not accessible"
                )
                return ValidationResult(is_safe=False, issues=issues, warnings=warnings)

            # Validate column doesn't already exist
            column_exists = await self._check_column_exists(
                plan.table_name, plan.column.name, connection
            )
            if column_exists:
                issues.append(
                    f"Column {plan.column.name} already exists in table {plan.table_name}"
                )

            # Get table constraints and validate against default value
            constraints = await self._get_table_constraints(plan.table_name, connection)
            strategy = self.strategies[plan.column.default_type]
            strategy_validation = strategy.validate_against_constraints(
                plan.column, constraints
            )

            issues.extend(strategy_validation.issues)
            warnings.extend(strategy_validation.warnings)

            # Validate foreign key references if applicable
            if plan.column.foreign_key_reference:
                fk_validation = await self._validate_foreign_key_reference(
                    plan.column.foreign_key_reference, connection
                )
                if not fk_validation:
                    issues.append(
                        f"Invalid foreign key reference: {plan.column.foreign_key_reference}"
                    )

            # Performance validation
            if plan.estimated_duration and plan.estimated_duration > 300:  # 5 minutes
                warnings.append(
                    f"Long execution time estimated: {plan.estimated_duration:.1f} seconds"
                )

            # Check for concurrent operations
            concurrent_ops = await self._check_concurrent_operations(
                plan.table_name, connection
            )
            if concurrent_ops:
                warnings.append("Concurrent operations detected on target table")

            result = ValidationResult(
                is_safe=len(issues) == 0,
                issues=issues,
                warnings=warnings,
                estimated_time=plan.estimated_duration,
                recommended_batch_size=plan.batch_size,
            )

            self.logger.info(
                f"Validation complete for {plan.table_name}.{plan.column.name}: "
                f"Safe={result.is_safe}, Issues={len(issues)}, Warnings={len(warnings)}"
            )

            return result

        except Exception as e:
            self.logger.error(f"Validation failed: {e}")
            issues.append(f"Validation error: {str(e)}")
            return ValidationResult(is_safe=False, issues=issues, warnings=warnings)

    async def execute_not_null_addition(
        self, plan: NotNullAdditionPlan, connection: Optional[asyncpg.Connection] = None
    ) -> AdditionExecutionResult:
        """
        Execute NOT NULL column addition according to plan.

        Args:
            plan: Validated execution plan
            connection: Database connection (optional)

        Returns:
            AdditionExecutionResult with execution details
        """
        start_time = datetime.now()
        self.logger.info(
            f"Executing NOT NULL addition: {plan.table_name}.{plan.column.name}"
        )

        if connection is None:
            connection = await self._get_connection()

        try:
            # Start transaction for rollback capability
            async with connection.transaction():
                # Execute based on strategy
                if plan.execution_strategy == "single_ddl":
                    result = await self._execute_single_ddl_addition(plan, connection)
                elif plan.execution_strategy == "batched_update":
                    result = await self._execute_batched_addition(plan, connection)
                else:
                    raise ValueError(
                        f"Unknown execution strategy: {plan.execution_strategy}"
                    )

                # Validate constraints after addition
                if plan.validate_constraints:
                    constraint_validation = (
                        await self._validate_constraints_post_addition(plan, connection)
                    )
                    if not constraint_validation:
                        result.result = AdditionResult.CONSTRAINT_VIOLATION
                        result.constraint_violations.append(
                            "Post-addition constraint validation failed"
                        )
                        raise Exception("Constraint validation failed after addition")

                execution_time = (datetime.now() - start_time).total_seconds()
                result.execution_time = execution_time

                self.logger.info(
                    f"Addition completed successfully: {plan.table_name}.{plan.column.name} "
                    f"in {execution_time:.2f}s, {result.affected_rows} rows affected"
                )

                return result

        except Exception as e:
            execution_time = (datetime.now() - start_time).total_seconds()
            self.logger.error(f"Addition failed: {e}")

            return AdditionExecutionResult(
                result=AdditionResult.ROLLBACK_REQUIRED,
                execution_time=execution_time,
                affected_rows=0,
                rollback_executed=True,  # Transaction will rollback automatically
                error_message=str(e),
                constraint_violations=[str(e)],
            )

    async def rollback_not_null_addition(
        self, plan: NotNullAdditionPlan, connection: Optional[asyncpg.Connection] = None
    ) -> AdditionExecutionResult:
        """
        Rollback NOT NULL column addition.

        Args:
            plan: Original execution plan
            connection: Database connection (optional)

        Returns:
            AdditionExecutionResult with rollback details
        """
        start_time = datetime.now()
        self.logger.warning(
            f"Rolling back NOT NULL addition: {plan.table_name}.{plan.column.name}"
        )

        if connection is None:
            connection = await self._get_connection()

        try:
            # Drop the column if it exists
            column_exists = await self._check_column_exists(
                plan.table_name, plan.column.name, connection
            )
            if column_exists:
                await connection.execute(
                    f"ALTER TABLE {plan.table_name} DROP COLUMN IF EXISTS {plan.column.name}"
                )
                affected_rows = await connection.fetchval(
                    f"SELECT COUNT(*) FROM {plan.table_name}"
                )
            else:
                affected_rows = 0

            execution_time = (datetime.now() - start_time).total_seconds()

            self.logger.info(
                f"Rollback completed: {plan.table_name}.{plan.column.name} "
                f"in {execution_time:.2f}s"
            )

            return AdditionExecutionResult(
                result=AdditionResult.SUCCESS,
                execution_time=execution_time,
                affected_rows=affected_rows,
                rollback_executed=True,
            )

        except Exception as e:
            execution_time = (datetime.now() - start_time).total_seconds()
            self.logger.error(f"Rollback failed: {e}")

            return AdditionExecutionResult(
                result=AdditionResult.VALIDATION_FAILED,
                execution_time=execution_time,
                affected_rows=0,
                rollback_executed=False,
                error_message=f"Rollback failed: {str(e)}",
            )

    # Private helper methods

    async def _get_connection(self) -> asyncpg.Connection:
        """Get database connection."""
        if self.connection_manager:
            return await self.connection_manager.get_connection()
        else:
            # This would need to be configured properly in production
            raise NotImplementedError("Connection manager not configured")

    async def _analyze_table_structure(
        self, table_name: str, connection: asyncpg.Connection
    ) -> Dict[str, Any]:
        """Analyze table structure and return comprehensive information."""
        # Get row count
        row_count = await connection.fetchval(f"SELECT COUNT(*) FROM {table_name}")

        # Get constraints
        constraints_query = """
        SELECT conname, contype, pg_get_constraintdef(oid) as definition
        FROM pg_constraint
        WHERE conrelid = $1::regclass
        """
        constraints_raw = await connection.fetch(constraints_query, table_name)

        constraints = []
        for row in constraints_raw:
            constraints.append(
                {
                    "name": row["conname"],
                    "constraint_type": {
                        "c": "CHECK",
                        "f": "FOREIGN KEY",
                        "p": "PRIMARY KEY",
                        "u": "UNIQUE",
                    }.get(row["contype"], "UNKNOWN"),
                    "constraint_definition": row["definition"],
                }
            )

        return {"row_count": row_count, "constraints": constraints}

    async def _validate_column_definition(
        self, column: ColumnDefinition, existing_constraints: List[Dict[str, Any]]
    ) -> ValidationResult:
        """Validate column definition against table structure."""
        issues = []
        warnings = []

        # Basic validation
        if not column.name:
            issues.append("Column name is required")
        if not column.data_type:
            issues.append("Column data type is required")

        # Validate default value strategy
        if column.default_type not in self.strategies:
            issues.append(f"Unsupported default value type: {column.default_type}")
        else:
            strategy = self.strategies[column.default_type]
            strategy_validation = strategy.validate_against_constraints(
                column, existing_constraints
            )
            issues.extend(strategy_validation.issues)
            warnings.extend(strategy_validation.warnings)

        return ValidationResult(
            is_safe=len(issues) == 0, issues=issues, warnings=warnings
        )

    def _select_execution_strategy(
        self, column: ColumnDefinition, row_count: int
    ) -> str:
        """Select optimal execution strategy based on column and table characteristics."""
        if column.default_type == DefaultValueType.STATIC and row_count < 1000000:
            return "single_ddl"
        elif column.default_type in [
            DefaultValueType.COMPUTED,
            DefaultValueType.CONDITIONAL,
        ]:
            return "batched_update"
        elif row_count > 10000000:  # Very large table
            return "batched_update"
        else:
            return "single_ddl"

    def _extract_constraint_dependencies(
        self, constraints: List[Dict[str, Any]]
    ) -> List[str]:
        """Extract constraint dependencies that might affect the addition."""
        dependencies = []

        for constraint in constraints:
            if constraint["constraint_type"] in ["FOREIGN KEY", "CHECK"]:
                dependencies.append(constraint["name"])

        return dependencies

    async def _generate_rollback_plan(
        self, plan: NotNullAdditionPlan, connection: asyncpg.Connection
    ) -> Dict[str, Any]:
        """Generate rollback plan for the addition."""
        return {
            "strategy": "drop_column",
            "sql": f"ALTER TABLE {plan.table_name} DROP COLUMN IF EXISTS {plan.column.name}",
            "estimated_time": 0.1,  # Very fast operation
            "requires_transaction": True,
        }

    async def _validate_table_access(
        self, table_name: str, connection: asyncpg.Connection
    ) -> bool:
        """Validate that table exists and is accessible."""
        try:
            result = await connection.fetchval(
                "SELECT EXISTS(SELECT 1 FROM information_schema.tables WHERE table_name = $1)",
                table_name,
            )
            return result
        except Exception:
            return False

    async def _check_column_exists(
        self, table_name: str, column_name: str, connection: asyncpg.Connection
    ) -> bool:
        """Check if column already exists in table."""
        try:
            result = await connection.fetchval(
                """SELECT EXISTS(
                    SELECT 1 FROM information_schema.columns
                    WHERE table_name = $1 AND column_name = $2
                )""",
                table_name,
                column_name,
            )
            return result
        except Exception:
            return False

    async def _get_table_constraints(
        self, table_name: str, connection: asyncpg.Connection
    ) -> List[Dict[str, Any]]:
        """Get all constraints for the table."""
        return await self._analyze_table_structure(table_name, connection)

    async def _validate_foreign_key_reference(
        self, reference: str, connection: asyncpg.Connection
    ) -> bool:
        """Validate foreign key reference exists."""
        try:
            # Parse reference (format: "table(column)")
            if "(" in reference:
                ref_table, ref_column = reference.split("(")
                ref_column = ref_column.rstrip(")")

                result = await connection.fetchval(
                    """SELECT EXISTS(
                        SELECT 1 FROM information_schema.columns
                        WHERE table_name = $1 AND column_name = $2
                    )""",
                    ref_table,
                    ref_column,
                )
                return result
            return False
        except Exception:
            return False

    async def _check_concurrent_operations(
        self, table_name: str, connection: asyncpg.Connection
    ) -> bool:
        """Check for concurrent operations on the table."""
        try:
            # Check for locks on the table
            result = await connection.fetchval(
                """SELECT COUNT(*) FROM pg_locks l
                   JOIN pg_class c ON l.relation = c.oid
                   WHERE c.relname = $1 AND l.mode != 'AccessShareLock'""",
                table_name,
            )
            return result > 0
        except Exception:
            return False

    async def _execute_single_ddl_addition(
        self, plan: NotNullAdditionPlan, connection: asyncpg.Connection
    ) -> AdditionExecutionResult:
        """Execute NOT NULL addition as single DDL operation."""
        strategy = self.strategies[plan.column.default_type]
        default_expr = strategy.generate_default_expression(plan.column)

        # Build ALTER TABLE statement
        sql = f"""
        ALTER TABLE {plan.table_name}
        ADD COLUMN {plan.column.name} {plan.column.data_type}
        NOT NULL DEFAULT {default_expr}
        """

        # Execute the DDL
        await connection.execute(sql)

        # Get affected row count
        affected_rows = await connection.fetchval(
            f"SELECT COUNT(*) FROM {plan.table_name}"
        )

        return AdditionExecutionResult(
            result=AdditionResult.SUCCESS,
            execution_time=0.0,  # Will be set by caller
            affected_rows=affected_rows,
        )

    async def _execute_batched_addition(
        self, plan: NotNullAdditionPlan, connection: asyncpg.Connection
    ) -> AdditionExecutionResult:
        """Execute NOT NULL addition in batches for large tables."""
        strategy = self.strategies[plan.column.default_type]
        default_expr = strategy.generate_default_expression(plan.column)

        # Check if this is a computed expression that references columns
        # PostgreSQL doesn't allow column references in DEFAULT expressions
        is_computed_with_column_refs = (
            plan.column.default_type == DefaultValueType.COMPUTED
            and self._has_column_references(default_expr)
        )

        if is_computed_with_column_refs:
            # Step 1: Add nullable column WITHOUT default (PostgreSQL limitation)
            await connection.execute(
                f"""
                ALTER TABLE {plan.table_name}
                ADD COLUMN {plan.column.name} {plan.column.data_type}
            """
            )
        else:
            # Step 1: Add nullable column with default (works for static/function defaults)
            await connection.execute(
                f"""
                ALTER TABLE {plan.table_name}
                ADD COLUMN {plan.column.name} {plan.column.data_type}
                DEFAULT {default_expr}
            """
            )

        # Step 2: Update all NULL values in batches using the computed expression
        batch_size = plan.batch_size
        total_updated = 0

        while True:
            updated = await connection.fetchval(
                f"""
                WITH batch AS (
                    SELECT ctid FROM {plan.table_name}
                    WHERE {plan.column.name} IS NULL
                    LIMIT {batch_size}
                )
                UPDATE {plan.table_name}
                SET {plan.column.name} = {default_expr}
                FROM batch
                WHERE {plan.table_name}.ctid = batch.ctid
                RETURNING 1
            """
            )

            if not updated:
                break

            total_updated += len(updated) if isinstance(updated, list) else 1

            # Add small delay for very large batches
            if batch_size > 10000:
                await asyncio.sleep(0.01)

        # Step 3: Add NOT NULL constraint
        await connection.execute(
            f"""
            ALTER TABLE {plan.table_name}
            ALTER COLUMN {plan.column.name} SET NOT NULL
        """
        )

        # Step 4: Add default constraint if it's not a column-referencing computed expression
        if not is_computed_with_column_refs:
            await connection.execute(
                f"""
                ALTER TABLE {plan.table_name}
                ALTER COLUMN {plan.column.name} SET DEFAULT {default_expr}
            """
            )

        return AdditionExecutionResult(
            result=AdditionResult.SUCCESS,
            execution_time=0.0,  # Will be set by caller
            affected_rows=total_updated,
        )

    def _has_column_references(self, expression: str) -> bool:
        """Check if expression contains column references that PostgreSQL can't handle in DEFAULT."""
        import re

        # Look for potential column references in the expression
        # This is a simplified heuristic - in production, this could be more sophisticated
        # Common patterns that suggest column references:
        # - CASE WHEN column_name ...
        # - column_name > value
        # - column_name IS NOT NULL
        # - etc.
        # Remove string literals and function calls to avoid false positives
        cleaned = re.sub(r"'[^']*'", "", expression)  # Remove string literals
        cleaned = re.sub(r"\b\w+\s*\(", "", cleaned)  # Remove function calls

        # Look for identifier patterns that could be column names
        # This pattern looks for words that are not SQL keywords and are used in comparison/logic contexts
        potential_columns = re.findall(r"\b([a-zA-Z_][a-zA-Z0-9_]*)\b", cleaned)

        # Filter out known SQL keywords
        sql_keywords = {
            "CASE",
            "WHEN",
            "THEN",
            "ELSE",
            "END",
            "AND",
            "OR",
            "NOT",
            "IN",
            "IS",
            "NULL",
            "TRUE",
            "FALSE",
            "LIKE",
            "BETWEEN",
            "EXISTS",
            "SELECT",
            "FROM",
            "WHERE",
        }

        # If we find identifiers that aren't keywords, assume they're column references
        for identifier in potential_columns:
            if identifier.upper() not in sql_keywords:
                return True

        return False

    async def _validate_constraints_post_addition(
        self, plan: NotNullAdditionPlan, connection: asyncpg.Connection
    ) -> bool:
        """Validate all constraints after column addition."""
        try:
            # Check for constraint violations
            violations = await connection.fetchval(
                f"""
                SELECT COUNT(*) FROM {plan.table_name}
                WHERE {plan.column.name} IS NULL
            """
            )

            return violations == 0
        except Exception as e:
            self.logger.error(f"Post-addition constraint validation failed: {e}")
            return False
