"""Property-based tests using Hypothesis for Blinter.

These tests use property-based testing to verify invariants and edge cases
across a wide range of inputs automatically generated by Hypothesis.
"""

# pylint: disable=too-many-lines

from pathlib import Path
import tempfile
from typing import Dict, List, Optional, Set

from hypothesis import HealthCheck, assume, given, settings, strategies as st
import pytest

from blinter import (
    RULES,
    BlinterConfig,
    LintIssue,
    Rule,
    RuleSeverity,
    _check_advanced_escaping_rules,
    _check_advanced_for_rules,
    _check_advanced_global_patterns,
    _check_advanced_performance,
    _check_advanced_process_mgmt,
    _check_advanced_security,
    _check_advanced_style_patterns,
    _check_advanced_style_rules,
    _check_advanced_vars,
    _check_basic_line_ending_issues,
    _check_bat_cmd_differences,
    _check_call_ambiguity,
    _check_call_labels,
    _check_cmd_case_consistency,
    _check_cmd_error_handling,
    _check_code_documentation,
    _check_code_duplication,
    _check_command_typos,
    _check_command_warnings,
    _check_compatibility_warnings,
    _check_continuation_spaces,
    _check_deprecated_commands,
    _check_double_percent_escaping,
    _check_doublecolon_risks,
    _check_echo_unicode_risk,
    _check_enhanced_commands,
    _check_enhanced_performance,
    _check_enhanced_security_rules,
    _check_errorlevel_comparison,
    _check_errorlevel_syntax,
    _check_extended_non_ascii,
    _check_external_error_handling,
    _check_for_f_options,
    _check_for_loop_syntax,
    _check_for_loop_var_syntax,
    _check_function_docs,
    _check_general_unicode_risk,
    _check_global_priv_security,
    _check_global_style_rules,
    _check_goto_call_risks,
    _check_goto_colon_consistency,
    _check_goto_labels,
    _check_if_comparison_quotes,
    _check_if_exist_mixing,
    _check_if_statement_formatting,
    _check_improper_caret_escape,
    _check_inconsistent_indentation,
    _check_inefficient_modifiers,
    _check_info_disclosure_sec,
    _check_input_validation_sec,
    _check_line_ending_rules,
    _check_line_length,
    _check_magic_numbers,
    _check_malware_security,
    _check_missing_documentation,
    _check_missing_header_doc,
    _check_missing_pause,
    _check_multibyte_risks,
    _check_multilevel_escaping,
    _check_nested_for_loops,
    _check_new_global_rules,
    _check_non_ascii_chars,
    _check_parameter_modifiers,
    _check_path_security,
    _check_path_syntax,
    _check_percent_tilde_syntax,
    _check_performance_issues,
    _check_privilege_security,
    _check_quote_escaping,
    _check_quotes,
    _check_redundant_operations,
    _check_restart_limits,
    _check_search_unicode_risk,
    _check_security_issues,
    _check_self_modification,
    _check_set_a_expression,
    _check_set_a_quoting,
    _check_setlocal_redundancy,
    _check_string_operation_syntax,
    _check_style_issues,
    _check_subroutine_call,
    _check_syntax_errors,
    _check_timeout_ping_numbers,
    _check_unc_path,
    _check_undefined_variables,
    _check_unicode_filenames,
    _check_unicode_handling_issue,
    _check_unnecessary_output_p014,
    _check_unquoted_variables,
    _check_unreachable_code,
    _check_var_naming,
    _check_variable_expansion,
    _check_variable_naming,
    _check_warning_issues,
    _collect_indented_lines,
    _collect_labels,
    _collect_set_variables,
    _detect_line_endings,
    _find_single_line_mixed_indent,
    _has_multibyte_chars,
    _is_command_in_safe_context,
    _is_comment_line,
    _is_truly_executable_command,
    _line_makes_code_reachable,
    _parse_suppression_comments,
    _set_min_severity,
    _update_paren_depth,
    find_batch_files,
    group_issues,
    lint_batch_file,
    read_file_with_encoding,
)

# ============================================================================
# Custom Strategies for Batch File Testing
# ============================================================================


@st.composite
def rule_severity_strategy(draw: st.DrawFn) -> RuleSeverity:
    """Generate valid RuleSeverity enum values."""
    return draw(st.sampled_from(list(RuleSeverity)))


@st.composite
def rule_code_strategy(draw: st.DrawFn) -> str:
    """Generate valid rule codes (E001-E999, W001-W999, etc.)."""
    prefix = draw(st.sampled_from(["E", "W", "S", "P", "SEC"]))
    if prefix == "SEC":
        number = draw(st.integers(min_value=1, max_value=99))
        return f"{prefix}{number:03d}"
    number = draw(st.integers(min_value=1, max_value=999))
    return f"{prefix}{number:03d}"


@st.composite
def rule_strategy(draw: st.DrawFn) -> Rule:
    """Generate valid Rule instances."""
    code = draw(rule_code_strategy())
    name = draw(st.text(min_size=1, max_size=100).filter(lambda x: x.strip()))
    severity = draw(rule_severity_strategy())
    explanation = draw(st.text(min_size=1, max_size=500).filter(lambda x: x.strip()))
    recommendation = draw(st.text(min_size=1, max_size=500).filter(lambda x: x.strip()))

    return Rule(
        code=code,
        name=name,
        severity=severity,
        explanation=explanation,
        recommendation=recommendation,
    )


@st.composite
def lint_issue_strategy(draw: st.DrawFn) -> LintIssue:
    """Generate valid LintIssue instances."""
    line_number = draw(st.integers(min_value=1, max_value=10000))
    rule = draw(rule_strategy())
    context = draw(st.text(max_size=500))

    return LintIssue(line_number=line_number, rule=rule, context=context)


@st.composite
def blinter_config_strategy(draw: st.DrawFn) -> BlinterConfig:
    """Generate valid BlinterConfig instances."""
    recursive = draw(st.booleans())
    show_summary = draw(st.booleans())
    max_line_length = draw(st.integers(min_value=80, max_value=500))

    # Generate sets of rule codes
    enabled_rules: Optional[Set[str]] = None
    disabled_rules: Optional[Set[str]] = None

    if draw(st.booleans()):
        enabled_rules = draw(st.sets(rule_code_strategy(), max_size=10))

    if draw(st.booleans()):
        disabled_rules = draw(st.sets(rule_code_strategy(), max_size=10))

    min_severity = draw(st.none() | rule_severity_strategy())

    return BlinterConfig(
        recursive=recursive,
        show_summary=show_summary,
        max_line_length=max_line_length,
        enabled_rules=enabled_rules,
        disabled_rules=disabled_rules,
        min_severity=min_severity,
    )


@st.composite
def batch_line_strategy(draw: st.DrawFn) -> str:
    """Generate plausible batch file lines."""
    line_types = [
        "comment",
        "echo",
        "set",
        "if",
        "for",
        "call",
        "goto",
        "label",
        "rem",
        "empty",
        "command",
    ]
    line_type = draw(st.sampled_from(line_types))

    # Use a dictionary to map line types to their generators
    def gen_set() -> str:
        var = draw(st.text(min_size=1, max_size=20).filter(lambda x: x.isalnum()))
        val = draw(st.text(max_size=50))
        return f"SET {var}={val}"

    generators = {
        "comment": lambda: f"REM {draw(st.text(max_size=100))}",
        "echo": lambda: f"ECHO {draw(st.text(max_size=100))}",
        "set": gen_set,
        "if": lambda: "IF EXIST test.txt ECHO File exists",
        "for": lambda: "FOR %%i IN (*.txt) DO ECHO %%i",
        "call": lambda: "CALL :subroutine",
        "goto": lambda: "GOTO end",
        "label": lambda: f":{draw(st.text(min_size=1, max_size=20).filter(lambda x: x.isalnum()))}",
        "rem": lambda: f":: {draw(st.text(max_size=100))}",
        "empty": lambda: "",
        "command": lambda: draw(st.text(max_size=100)),
    }

    return generators[line_type]()


# ============================================================================
# Data Class Property Tests
# ============================================================================


class TestRuleProperties:
    """Property-based tests for Rule dataclass."""

    @given(rule=rule_strategy())
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_rule_creation_valid(self, rule: Rule) -> None:
        """Valid rules should be created successfully."""
        assert isinstance(rule, Rule)
        assert rule.code
        assert rule.name
        assert isinstance(rule.severity, RuleSeverity)
        assert rule.explanation
        assert rule.recommendation

    @given(
        code=st.one_of(st.none(), st.just("")),
        rule=rule_strategy(),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_rule_invalid_code(
        self,
        code: Optional[str],
        rule: Rule,
    ) -> None:
        """Rules with invalid codes should raise ValueError."""
        with pytest.raises(ValueError, match="Rule code must be a non-empty string"):
            Rule(
                code=code if code is not None else "",
                name=rule.name,
                severity=rule.severity,
                explanation=rule.explanation,
                recommendation=rule.recommendation,
            )

    @given(
        code=rule_code_strategy(),
        severity=rule_severity_strategy(),
        explanation=st.text(min_size=1, max_size=500),
        recommendation=st.text(min_size=1, max_size=500),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_rule_invalid_name(
        self,
        code: str,
        severity: RuleSeverity,
        explanation: str,
        recommendation: str,
    ) -> None:
        """Rules with invalid names should raise ValueError."""
        with pytest.raises(ValueError, match="Rule name must be a non-empty string"):
            Rule(
                code=code,
                name="",
                severity=severity,
                explanation=explanation,
                recommendation=recommendation,
            )


class TestLintIssueProperties:
    """Property-based tests for LintIssue dataclass."""

    @given(issue=lint_issue_strategy())
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_lint_issue_creation_valid(self, issue: LintIssue) -> None:
        """Valid lint issues should be created successfully."""
        assert isinstance(issue, LintIssue)
        assert issue.line_number >= 1
        assert isinstance(issue.rule, Rule)

    @given(
        line_number=st.integers(max_value=0),
        rule=rule_strategy(),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_lint_issue_invalid_line_number(self, line_number: int, rule: Rule) -> None:
        """Lint issues with invalid line numbers should raise ValueError."""
        with pytest.raises(ValueError, match="Line number must be positive"):
            LintIssue(line_number=line_number, rule=rule, context="")


class TestBlinterConfigProperties:
    """Property-based tests for BlinterConfig dataclass."""

    @given(config=blinter_config_strategy())
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_config_creation_valid(self, config: BlinterConfig) -> None:
        """Valid configs should be created successfully."""
        assert isinstance(config, BlinterConfig)
        assert isinstance(config.recursive, bool)
        assert isinstance(config.show_summary, bool)
        assert config.max_line_length >= 80

    @given(config=blinter_config_strategy(), rule_code=rule_code_strategy())
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_config_is_rule_enabled_consistency(
        self, config: BlinterConfig, rule_code: str
    ) -> None:
        """Rule enablement should be consistent with configuration."""
        result = config.is_rule_enabled(rule_code)
        assert isinstance(result, bool)

        # If rule is explicitly disabled, it should return False (disabled takes precedence)
        if config.disabled_rules and rule_code in config.disabled_rules:
            assert result is False
        # If enabled_rules is not empty and rule is in it (and not disabled), should be True
        elif config.enabled_rules and rule_code in config.enabled_rules:
            assert result is True

    @given(config=blinter_config_strategy(), severity=rule_severity_strategy())
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_config_should_include_severity_consistency(
        self, config: BlinterConfig, severity: RuleSeverity
    ) -> None:
        """Severity filtering should be consistent."""
        result = config.should_include_severity(severity)
        assert isinstance(result, bool)

        # If no min_severity is set, all severities should be included
        if config.min_severity is None:
            assert result is True


# ============================================================================
# String Parsing Property Tests
# ============================================================================


class TestStringParsingProperties:
    """Property-based tests for string parsing functions."""

    @given(line=st.text())
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_is_comment_line_deterministic(self, line: str) -> None:
        """_is_comment_line should always return the same result for the same input."""
        result1 = _is_comment_line(line)
        result2 = _is_comment_line(line)
        assert result1 == result2
        assert isinstance(result1, bool)

    @given(line=st.text())
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_is_command_in_safe_context_deterministic(self, line: str) -> None:
        """_is_command_in_safe_context should be deterministic."""
        result1 = _is_command_in_safe_context(line)
        result2 = _is_command_in_safe_context(line)
        assert result1 == result2
        assert isinstance(result1, bool)

    @given(line=st.text())
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_is_truly_executable_command_deterministic(self, line: str) -> None:
        """_is_truly_executable_command should be deterministic."""
        result1 = _is_truly_executable_command(line)
        result2 = _is_truly_executable_command(line)
        assert result1 == result2
        assert isinstance(result1, bool)

    @given(line=st.text())
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_line_makes_code_reachable_deterministic(self, line: str) -> None:
        """_line_makes_code_reachable should be deterministic."""
        result1 = _line_makes_code_reachable(line)
        result2 = _line_makes_code_reachable(line)
        assert result1 == result2
        assert isinstance(result1, bool)

    @given(
        line=st.text(),
        current_depth=st.integers(min_value=0, max_value=10),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_update_paren_depth_bounded(self, line: str, current_depth: int) -> None:
        """_update_paren_depth should return reasonable values."""
        result = _update_paren_depth(line, current_depth)
        assert isinstance(result, int)
        # Depth shouldn't change drastically (by more than 1)
        assert abs(result - current_depth) <= 1


# ============================================================================
# Collection and Analysis Property Tests
# ============================================================================


class TestCollectionProperties:
    """Property-based tests for collection functions."""

    @given(lines=st.lists(batch_line_strategy(), max_size=50))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_collect_labels_returns_valid_dict(self, lines: List[str]) -> None:
        """_collect_labels should return a dict mapping labels to line numbers."""
        labels, issues = _collect_labels(lines)
        assert isinstance(labels, dict)
        assert isinstance(issues, list)

        # All keys should be strings
        for key in labels:
            assert isinstance(key, str)

        # All values should be positive integers
        for value in labels.values():
            assert isinstance(value, int)
            assert value >= 1

        # All issues should be LintIssue instances
        for issue in issues:
            assert isinstance(issue, LintIssue)

    @given(lines=st.lists(batch_line_strategy(), max_size=50))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_collect_set_variables_returns_valid_set(self, lines: List[str]) -> None:
        """_collect_set_variables should return a set of variable names."""
        variables = _collect_set_variables(lines)
        assert isinstance(variables, set)

        # All elements should be strings
        for var in variables:
            assert isinstance(var, str)

    @given(lines=st.lists(batch_line_strategy(), max_size=50))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_parse_suppression_comments_returns_valid_dict(self, lines: List[str]) -> None:
        """_parse_suppression_comments should return a dict of line numbers to rule codes."""
        suppressions = _parse_suppression_comments(lines)
        assert isinstance(suppressions, dict)

        # All keys should be integers
        for key in suppressions:
            assert isinstance(key, int)

        # All values should be sets of strings
        for value in suppressions.values():
            assert isinstance(value, set)
            for rule_code in value:
                assert isinstance(rule_code, str)

    @given(lines=st.lists(st.text(), max_size=50))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_collect_indented_lines_returns_valid_list(self, lines: List[str]) -> None:
        """_collect_indented_lines should return a list of tuples."""
        result = _collect_indented_lines(lines)
        assert isinstance(result, list)

        # All elements should be tuples of (int, str)
        for item in result:
            assert isinstance(item, tuple)
            assert len(item) == 2
            assert isinstance(item[0], int)
            assert isinstance(item[1], str)
            assert item[0] >= 1

    @given(lines=st.lists(st.text(), max_size=50))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_has_multibyte_chars_returns_valid_tuple(self, lines: List[str]) -> None:
        """_has_multibyte_chars should return (bool, list of int)."""
        has_multibyte, line_numbers = _has_multibyte_chars(lines)
        assert isinstance(has_multibyte, bool)
        assert isinstance(line_numbers, list)

        # All elements should be integers
        for line_num in line_numbers:
            assert isinstance(line_num, int)
            assert line_num >= 1


# ============================================================================
# Checking Function Property Tests
# ============================================================================


class TestCheckingFunctionProperties:
    """Property-based tests for checking functions."""

    @given(
        stripped=st.text(max_size=200),
        line_num=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_goto_labels_returns_list(self, stripped: str, line_num: int) -> None:
        """_check_goto_labels should return a list of LintIssue."""
        labels: Dict[str, int] = {}
        result = _check_goto_labels(stripped, line_num, labels)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)
            assert issue.line_number == line_num

    @given(
        stripped=st.text(max_size=200),
        line_num=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_if_statement_formatting_returns_list(self, stripped: str, line_num: int) -> None:
        """_check_if_statement_formatting should return a list of LintIssue."""
        result = _check_if_statement_formatting(stripped, line_num)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)
            assert issue.line_number == line_num

    @given(
        stripped=st.text(max_size=200),
        line_num=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_path_syntax_returns_list(self, stripped: str, line_num: int) -> None:
        """_check_path_syntax should return a list of LintIssue."""
        result = _check_path_syntax(stripped, line_num)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)
            assert issue.line_number == line_num

    @given(
        stripped=st.text(max_size=200),
        line_num=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_unquoted_variables_returns_list(self, stripped: str, line_num: int) -> None:
        """_check_unquoted_variables should return a list of LintIssue."""
        result = _check_unquoted_variables(stripped, line_num)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)
            assert issue.line_number == line_num

    @given(lines=st.lists(batch_line_strategy(), min_size=1, max_size=50))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_cmd_case_consistency_returns_list(self, lines: List[str]) -> None:
        """_check_cmd_case_consistency should return a list of LintIssue."""
        result = _check_cmd_case_consistency(lines)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(indented_lines=st.lists(st.tuples(st.integers(min_value=1), st.text()), max_size=50))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_find_single_line_mixed_indent_returns_list(
        self, indented_lines: List[tuple[int, str]]
    ) -> None:
        """_find_single_line_mixed_indent should return a list of LintIssue."""
        result = _find_single_line_mixed_indent(indented_lines)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)


# ============================================================================
# File Operation Property Tests
# ============================================================================


class TestFileOperationProperties:
    """Property-based tests for file operations."""

    @given(recursive=st.booleans())
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_find_batch_files_returns_list(self, recursive: bool) -> None:
        """find_batch_files should return a list of Path objects."""
        # Create a temporary directory
        with tempfile.TemporaryDirectory() as tmpdir:
            result = find_batch_files(tmpdir, recursive=recursive)
            assert isinstance(result, list)
            for path in result:
                assert isinstance(path, Path)

    @given(issues=st.lists(lint_issue_strategy(), max_size=50))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_group_issues_returns_dict(self, issues: List[LintIssue]) -> None:
        """group_issues should return a defaultdict mapping severity to issues."""
        result = group_issues(issues)
        assert isinstance(result, dict)

        # All keys should be RuleSeverity
        for key in result.keys():
            assert isinstance(key, RuleSeverity)

        # All values should be lists of LintIssue
        for value in result.values():
            assert isinstance(value, list)
            for issue in value:
                assert isinstance(issue, LintIssue)

    def test_detect_line_endings_with_temp_files(self) -> None:
        """_detect_line_endings should handle files with various line endings."""
        line_endings = ["\r\n", "\n", "\r"]

        for ending in line_endings:
            with tempfile.NamedTemporaryFile(mode="wb", suffix=".bat", delete=False) as tmp:
                content = f"@ECHO OFF{ending}ECHO test{ending}".encode("utf-8")
                tmp.write(content)
                tmp_path = tmp.name

            try:
                result = _detect_line_endings(tmp_path)
                assert isinstance(result, tuple)
                assert len(result) == 5
                ending_type, has_crlf, crlf_count, lf_count, cr_count = result
                assert isinstance(ending_type, str)
                assert isinstance(has_crlf, bool)
                assert isinstance(crlf_count, int)
                assert isinstance(lf_count, int)
                assert isinstance(cr_count, int)
            finally:
                Path(tmp_path).unlink()


# ============================================================================
# Configuration Property Tests
# ============================================================================


class TestConfigurationProperties:
    """Property-based tests for configuration functions."""

    @given(severity_str=st.sampled_from(["ERROR", "WARNING", "STYLE", "SECURITY", "PERFORMANCE"]))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_set_min_severity_valid(self, severity_str: str) -> None:
        """_set_min_severity should handle valid severity strings."""
        config = BlinterConfig()
        _set_min_severity(config, severity_str)
        assert config.min_severity is not None
        assert isinstance(config.min_severity, RuleSeverity)

    @given(
        severity_str=st.text().filter(
            lambda x: x.upper() not in ["ERROR", "WARNING", "STYLE", "SECURITY", "PERFORMANCE"]
        )
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_set_min_severity_invalid(self, severity_str: str) -> None:
        """_set_min_severity should not crash on invalid severity strings."""
        assume(severity_str.upper() not in ["ERROR", "WARNING", "STYLE", "SECURITY", "PERFORMANCE"])
        config = BlinterConfig()
        # Should not raise an exception, just log a warning
        _set_min_severity(config, severity_str)
        # min_severity should remain None for invalid input
        assert config.min_severity is None


# ============================================================================
# Integration Property Tests
# ============================================================================


class TestIntegrationProperties:
    """Integration property-based tests."""

    @given(
        lines=st.lists(batch_line_strategy(), min_size=1, max_size=20),
        extension=st.sampled_from([".bat", ".cmd"]),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None, max_examples=20)
    def test_lint_batch_file_with_generated_content(self, lines: List[str], extension: str) -> None:
        """lint_batch_file should handle generated batch file content."""
        content = "\r\n".join(lines)

        with tempfile.NamedTemporaryFile(
            mode="w", suffix=extension, delete=False, encoding="utf-8", newline=""
        ) as tmp:
            tmp.write(content)
            tmp_path = tmp.name

        try:
            result = lint_batch_file(tmp_path)
            assert isinstance(result, list)
            for issue in result:
                assert isinstance(issue, LintIssue)
                assert issue.line_number >= 1
        finally:
            Path(tmp_path).unlink()

    @given(rule_code=st.sampled_from(list(RULES.keys())))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_all_rules_are_valid(self, rule_code: str) -> None:
        """All rules in RULES dict should be valid Rule instances."""
        rule = RULES[rule_code]
        assert isinstance(rule, Rule)
        assert rule.code == rule_code
        assert isinstance(rule.severity, RuleSeverity)
        assert rule.name
        assert rule.explanation
        assert rule.recommendation


# ============================================================================
# Edge Case Property Tests
# ============================================================================


class TestEdgeCaseProperties:
    """Property-based tests for edge cases."""

    @given(line=st.text(alphabet=st.characters(blacklist_categories=("Cs",)), max_size=1000))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_parsing_functions_handle_unicode(self, line: str) -> None:
        """Parsing functions should handle unicode without crashing."""
        # These should not raise exceptions
        _is_comment_line(line)
        _is_command_in_safe_context(line)
        _is_truly_executable_command(line)
        _line_makes_code_reachable(line)

    @given(lines=st.lists(st.text(max_size=10), max_size=0))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_collection_functions_handle_empty_lists(self, lines: List[str]) -> None:
        """Collection functions should handle empty lists gracefully."""
        assert len(lines) == 0

        labels, issues = _collect_labels(lines)
        assert isinstance(labels, dict)
        assert isinstance(issues, list)

        variables = _collect_set_variables(lines)
        assert isinstance(variables, set)

        suppressions = _parse_suppression_comments(lines)
        assert isinstance(suppressions, dict)

    @given(
        line=st.text(
            alphabet=st.characters(
                whitelist_categories=("Lu", "Ll", "Nd", "Zs"),
                whitelist_characters="@%:()[]{}=<>|&^!",
            ),
            max_size=500,
        )
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_functions_dont_crash_on_valid_chars(self, line: str) -> None:
        """Check functions should handle lines with typical batch characters."""
        line_num = 1
        labels: Dict[str, int] = {}

        # These should not raise exceptions
        _check_goto_labels(line, line_num, labels)
        _check_if_statement_formatting(line, line_num)
        _check_path_syntax(line, line_num)
        _check_unquoted_variables(line, line_num)

    @given(
        lines=st.lists(
            st.text(max_size=200).filter(lambda x: "\0" not in x),
            min_size=1,
            max_size=20,
        )
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None, max_examples=10)
    def test_lint_batch_file_handles_various_content(self, lines: List[str]) -> None:
        """lint_batch_file should handle various content without crashing."""
        content = "\r\n".join(lines)

        with tempfile.NamedTemporaryFile(
            mode="w", suffix=".bat", delete=False, encoding="utf-8", newline=""
        ) as tmp:
            tmp.write(content)
            tmp_path = tmp.name

        try:
            # Should not raise an exception
            result = lint_batch_file(tmp_path)
            assert isinstance(result, list)
        finally:
            Path(tmp_path).unlink()


# ============================================================================
# Invariant Property Tests
# ============================================================================


class TestInvariantProperties:
    """Tests for invariants that should always hold."""

    @given(config=blinter_config_strategy(), rule_code=rule_code_strategy())
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_disabled_rules_always_disabled(self, config: BlinterConfig, rule_code: str) -> None:
        """If a rule is in disabled_rules, it should always return False."""
        if config.disabled_rules is None:
            config.disabled_rules = set()
        config.disabled_rules.add(rule_code)
        assert config.is_rule_enabled(rule_code) is False

    @given(config=blinter_config_strategy(), rule_code=rule_code_strategy())
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_enabled_rules_override_disabled(self, config: BlinterConfig, rule_code: str) -> None:
        """Disabled rules should take precedence over enabled rules."""
        if config.disabled_rules is None:
            config.disabled_rules = set()
        if config.enabled_rules is None:
            config.enabled_rules = set()

        config.disabled_rules.add(rule_code)
        config.enabled_rules.add(rule_code)

        # Disabled should take precedence
        assert config.is_rule_enabled(rule_code) is False

    @given(issues=st.lists(lint_issue_strategy(), min_size=1, max_size=50))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_group_issues_preserves_all_issues(self, issues: List[LintIssue]) -> None:
        """group_issues should preserve all issues without loss."""
        grouped = group_issues(issues)

        # Count total issues in grouped
        total_grouped = sum(len(issue_list) for issue_list in grouped.values())
        assert total_grouped == len(issues)

    @given(
        issues=st.lists(lint_issue_strategy(), min_size=1, max_size=50),
        severity=rule_severity_strategy(),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_group_issues_correct_severity(
        self, issues: List[LintIssue], severity: RuleSeverity
    ) -> None:
        """Issues grouped by severity should have the correct severity."""
        # Create issues with specific severity
        specific_issues = []
        for issue in issues:
            new_issue = LintIssue(
                line_number=issue.line_number,
                rule=Rule(
                    code=issue.rule.code,
                    name=issue.rule.name,
                    severity=severity,
                    explanation=issue.rule.explanation,
                    recommendation=issue.rule.recommendation,
                ),
                context=issue.context,
            )
            specific_issues.append(new_issue)

        grouped = group_issues(specific_issues)

        # All issues should be in the correct severity group
        assert len(grouped[severity]) == len(specific_issues)
        for issue in grouped[severity]:
            assert issue.rule.severity == severity


# ============================================================================
# Additional Checking Function Property Tests
# ============================================================================


class TestAdditionalCheckingFunctionProperties:
    """Property-based tests for additional checking functions."""

    @given(
        stripped=st.text(max_size=200),
        line_num=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_call_labels_returns_list(self, stripped: str, line_num: int) -> None:
        """_check_call_labels should return a list of LintIssue."""
        result = _check_call_labels(stripped, line_num)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)
            assert issue.line_number == line_num

    @given(
        stripped=st.text(max_size=200),
        line_num=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_errorlevel_syntax_returns_list(self, stripped: str, line_num: int) -> None:
        """_check_errorlevel_syntax should return a list of LintIssue."""
        result = _check_errorlevel_syntax(stripped, line_num)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)
            assert issue.line_number == line_num

    @given(
        stripped=st.text(max_size=200),
        line_num=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_if_exist_mixing_returns_list(self, stripped: str, line_num: int) -> None:
        """_check_if_exist_mixing should return a list of LintIssue."""
        result = _check_if_exist_mixing(stripped, line_num)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)
            assert issue.line_number == line_num

    @given(
        line=st.text(max_size=200),
        line_num=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_quotes_returns_list(self, line: str, line_num: int) -> None:
        """_check_quotes should return a list of LintIssue."""
        result = _check_quotes(line, line_num)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)
            assert issue.line_number == line_num

    @given(
        stripped=st.text(max_size=200),
        line_num=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_for_loop_syntax_returns_list(self, stripped: str, line_num: int) -> None:
        """_check_for_loop_syntax should return a list of LintIssue."""
        result = _check_for_loop_syntax(stripped, line_num)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)
            assert issue.line_number == line_num

    @given(
        stripped=st.text(max_size=200),
        line_num=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_variable_expansion_returns_list(self, stripped: str, line_num: int) -> None:
        """_check_variable_expansion should return a list of LintIssue."""
        result = _check_variable_expansion(stripped, line_num)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)
            assert issue.line_number == line_num

    @given(
        stripped=st.text(max_size=200),
        line_num=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_subroutine_call_returns_list(self, stripped: str, line_num: int) -> None:
        """_check_subroutine_call should return a list of LintIssue."""
        labels: Dict[str, int] = {}
        result = _check_subroutine_call(stripped, line_num, labels)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)
            assert issue.line_number == line_num

    @given(
        stripped=st.text(max_size=200),
        line_num=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_command_typos_returns_list(self, stripped: str, line_num: int) -> None:
        """_check_command_typos should return a list of LintIssue."""
        result = _check_command_typos(stripped, line_num)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)
            assert issue.line_number == line_num

    @given(
        stripped=st.text(max_size=200),
        line_num=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_parameter_modifiers_returns_list(self, stripped: str, line_num: int) -> None:
        """_check_parameter_modifiers should return a list of LintIssue."""
        result = _check_parameter_modifiers(stripped, line_num)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)
            assert issue.line_number == line_num

    @given(
        stripped=st.text(max_size=200),
        line_num=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_unc_path_returns_list(self, stripped: str, line_num: int) -> None:
        """_check_unc_path should return a list of LintIssue."""
        result = _check_unc_path(stripped, line_num)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)
            assert issue.line_number == line_num

    @given(
        stripped=st.text(max_size=200),
        line_num=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_quote_escaping_returns_list(self, stripped: str, line_num: int) -> None:
        """_check_quote_escaping should return a list of LintIssue."""
        result = _check_quote_escaping(stripped, line_num)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)
            assert issue.line_number == line_num

    @given(
        stripped=st.text(max_size=200),
        line_num=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_set_a_expression_returns_list(self, stripped: str, line_num: int) -> None:
        """_check_set_a_expression should return a list of LintIssue."""
        result = _check_set_a_expression(stripped, line_num)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)
            assert issue.line_number == line_num

    @given(
        line=st.text(max_size=200),
        line_num=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_syntax_errors_returns_list(self, line: str, line_num: int) -> None:
        """_check_syntax_errors should return a list of LintIssue."""
        labels: Dict[str, int] = {}
        result = _check_syntax_errors(line, line_num, labels)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(
        stripped=st.text(max_size=200),
        line_num=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_non_ascii_chars_returns_list(self, stripped: str, line_num: int) -> None:
        """_check_non_ascii_chars should return a list of LintIssue."""
        result = _check_non_ascii_chars(stripped, line_num)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)
            assert issue.line_number == line_num

    @given(
        stripped=st.text(max_size=200),
        line_num=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_errorlevel_comparison_returns_list(self, stripped: str, line_num: int) -> None:
        """_check_errorlevel_comparison should return a list of LintIssue."""
        result = _check_errorlevel_comparison(stripped, line_num)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)
            assert issue.line_number == line_num

    @given(
        stripped=st.text(max_size=200),
        line_num=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_inefficient_modifiers_returns_list(self, stripped: str, line_num: int) -> None:
        """_check_inefficient_modifiers should return a list of LintIssue."""
        result = _check_inefficient_modifiers(stripped, line_num)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)
            assert issue.line_number == line_num

    @given(
        stripped=st.text(max_size=200),
        line_num=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_extended_non_ascii_returns_list(self, stripped: str, line_num: int) -> None:
        """_check_extended_non_ascii should return a list of LintIssue."""
        result = _check_extended_non_ascii(stripped, line_num)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)
            assert issue.line_number == line_num

    @given(
        stripped=st.text(max_size=200),
        line_num=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_unicode_filenames_returns_list(self, stripped: str, line_num: int) -> None:
        """_check_unicode_filenames should return a list of LintIssue."""
        result = _check_unicode_filenames(stripped, line_num)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)
            assert issue.line_number == line_num

    @given(
        stripped=st.text(max_size=200),
        line_num=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_call_ambiguity_returns_list(self, stripped: str, line_num: int) -> None:
        """_check_call_ambiguity should return a list of LintIssue."""
        result = _check_call_ambiguity(stripped, line_num)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)
            assert issue.line_number == line_num

    @given(
        stripped=st.text(max_size=200),
        line_num=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_timeout_ping_numbers_returns_list(self, stripped: str, line_num: int) -> None:
        """_check_timeout_ping_numbers should return a list of LintIssue."""
        result = _check_timeout_ping_numbers(stripped, line_num)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)
            assert issue.line_number == line_num

    @given(
        stripped=st.text(max_size=200),
        line_num=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_info_disclosure_sec_returns_list(self, stripped: str, line_num: int) -> None:
        """_check_info_disclosure_sec should return a list of LintIssue."""
        result = _check_info_disclosure_sec(stripped, line_num)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)
            assert issue.line_number == line_num

    @given(
        stripped=st.text(max_size=200),
        line_num=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_malware_security_returns_list(self, stripped: str, line_num: int) -> None:
        """_check_malware_security should return a list of LintIssue."""
        result = _check_malware_security(stripped, line_num)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)
            assert issue.line_number == line_num

    @given(
        line=st.text(max_size=200),
        line_num=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_security_issues_returns_list(self, line: str, line_num: int) -> None:
        """_check_security_issues should return a list of LintIssue."""
        result = _check_security_issues(line, line_num)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(
        stripped=st.text(max_size=200),
        line_number=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_improper_caret_escape_returns_list(
        self, stripped: str, line_number: int
    ) -> None:
        """_check_improper_caret_escape should return a list of LintIssue."""
        result = _check_improper_caret_escape(stripped, line_number)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)
            assert issue.line_number == line_number

    @given(
        stripped=st.text(max_size=200),
        line_number=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_multilevel_escaping_returns_list(self, stripped: str, line_number: int) -> None:
        """_check_multilevel_escaping should return a list of LintIssue."""
        result = _check_multilevel_escaping(stripped, line_number)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)
            assert issue.line_number == line_number

    @given(
        line=st.text(max_size=200),
        line_number=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_continuation_spaces_returns_list(self, line: str, line_number: int) -> None:
        """_check_continuation_spaces should return a list of LintIssue."""
        stripped = line.strip()
        result = _check_continuation_spaces(line, stripped, line_number)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)
            assert issue.line_number == line_number

    @given(
        stripped=st.text(max_size=200),
        line_number=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_double_percent_escaping_returns_list(
        self, stripped: str, line_number: int
    ) -> None:
        """_check_double_percent_escaping should return a list of LintIssue."""
        result = _check_double_percent_escaping(stripped, line_number)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)
            assert issue.line_number == line_number

    @given(
        line=st.text(max_size=200),
        line_number=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_advanced_escaping_rules_returns_list(self, line: str, line_number: int) -> None:
        """_check_advanced_escaping_rules should return a list of LintIssue."""
        result = _check_advanced_escaping_rules(line, line_number)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(
        line=st.text(max_size=200),
        line_number=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_advanced_for_rules_returns_list(self, line: str, line_number: int) -> None:
        """_check_advanced_for_rules should return a list of LintIssue."""
        result = _check_advanced_for_rules(line, line_number)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(
        line=st.text(max_size=200),
        line_number=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_advanced_process_mgmt_returns_list(self, line: str, line_number: int) -> None:
        """_check_advanced_process_mgmt should return a list of LintIssue."""
        result = _check_advanced_process_mgmt(line, line_number)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(
        stripped=st.text(max_size=200),
        line_number=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_percent_tilde_syntax_returns_list(self, stripped: str, line_number: int) -> None:
        """_check_percent_tilde_syntax should return a list of LintIssue."""
        result = _check_percent_tilde_syntax(stripped, line_number)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)
            assert issue.line_number == line_number

    @given(
        stripped=st.text(max_size=200),
        line_number=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_for_loop_var_syntax_returns_list(self, stripped: str, line_number: int) -> None:
        """_check_for_loop_var_syntax should return a list of LintIssue."""
        result = _check_for_loop_var_syntax(stripped, line_number)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)
            assert issue.line_number == line_number

    @given(
        stripped=st.text(max_size=200),
        line_number=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_string_operation_syntax_returns_list(
        self, stripped: str, line_number: int
    ) -> None:
        """_check_string_operation_syntax should return a list of LintIssue."""
        result = _check_string_operation_syntax(stripped, line_number)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)
            assert issue.line_number == line_number

    @given(
        stripped=st.text(max_size=200),
        line_number=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_set_a_quoting_returns_list(self, stripped: str, line_number: int) -> None:
        """_check_set_a_quoting should return a list of LintIssue."""
        result = _check_set_a_quoting(stripped, line_number)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)
            assert issue.line_number == line_number

    @given(
        stripped=st.text(max_size=200),
        line_number=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_deprecated_commands_returns_list(self, stripped: str, line_number: int) -> None:
        """_check_deprecated_commands should return a list of LintIssue."""
        result = _check_deprecated_commands(stripped, line_number)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)
            assert issue.line_number == line_number

    @given(
        line=st.text(max_size=200),
        line_number=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_magic_numbers_returns_list(self, line: str, line_number: int) -> None:
        """_check_magic_numbers should return a list of LintIssue."""
        result = _check_magic_numbers(line, line_number)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)
            assert issue.line_number == line_number

    @given(
        line=st.text(max_size=200),
        line_number=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_line_length_returns_list(self, line: str, line_number: int) -> None:
        """_check_line_length should return a list of LintIssue."""
        result = _check_line_length(line, line_number)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)
            assert issue.line_number == line_number


# ============================================================================
# Boolean Return Function Property Tests
# ============================================================================


class TestBooleanReturnFunctionProperties:
    """Property-based tests for boolean return functions."""

    @given(stripped=st.text(max_size=200))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_echo_unicode_risk_returns_bool(self, stripped: str) -> None:
        """_check_echo_unicode_risk should return a boolean."""
        result = _check_echo_unicode_risk(stripped)
        assert isinstance(result, bool)

    @given(stripped=st.text(max_size=200))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_search_unicode_risk_returns_bool(self, stripped: str) -> None:
        """_check_search_unicode_risk should return a boolean."""
        result = _check_search_unicode_risk(stripped)
        assert isinstance(result, bool)

    @given(stripped=st.text(max_size=200))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_general_unicode_risk_returns_bool(self, stripped: str) -> None:
        """_check_general_unicode_risk should return a boolean."""
        result = _check_general_unicode_risk(stripped)
        assert isinstance(result, bool)


# ============================================================================
# Optional Return Function Property Tests
# ============================================================================


class TestOptionalReturnFunctionProperties:
    """Property-based tests for functions that return Optional[LintIssue]."""

    @given(
        stripped=st.text(max_size=200),
        line_num=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_unicode_handling_issue_returns_optional(
        self, stripped: str, line_num: int
    ) -> None:
        """_check_unicode_handling_issue should return Optional[LintIssue]."""
        result = _check_unicode_handling_issue(stripped, line_num)
        assert result is None or isinstance(result, LintIssue)
        if result is not None:
            assert result.line_number == line_num

    @given(
        stripped=st.text(max_size=200),
        line_number=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_for_f_options_returns_optional(self, stripped: str, line_number: int) -> None:
        """_check_for_f_options should return Optional[LintIssue]."""
        result = _check_for_f_options(stripped, line_number)
        assert result is None or isinstance(result, LintIssue)
        if result is not None:
            assert result.line_number == line_number

    @given(
        stripped=st.text(max_size=200),
        line_number=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_if_comparison_quotes_returns_optional(
        self, stripped: str, line_number: int
    ) -> None:
        """_check_if_comparison_quotes should return Optional[LintIssue]."""
        result = _check_if_comparison_quotes(stripped, line_number)
        assert result is None or isinstance(result, LintIssue)
        if result is not None:
            assert result.line_number == line_number

    @given(
        lines=st.lists(st.text(max_size=100), min_size=2, max_size=50),
        i=st.integers(min_value=0),
        stripped=st.text(max_size=200),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_unnecessary_output_p014_returns_optional(
        self, lines: List[str], i: int, stripped: str
    ) -> None:
        """_check_unnecessary_output_p014 should return Optional[LintIssue]."""
        assume(i < len(lines) - 1)
        result = _check_unnecessary_output_p014(lines, i, stripped)
        assert result is None or isinstance(result, LintIssue)


# ============================================================================
# Global Checking Function Property Tests (List[str] input)
# ============================================================================


class TestGlobalCheckingFunctionProperties:
    """Property-based tests for checking functions that take List[str]."""

    @given(lines=st.lists(batch_line_strategy(), max_size=50))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_undefined_variables_returns_list(self, lines: List[str]) -> None:
        """_check_undefined_variables should return a list of LintIssue."""
        set_vars: Set[str] = set()
        result = _check_undefined_variables(lines, set_vars)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(lines=st.lists(batch_line_strategy(), max_size=50))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_code_documentation_returns_list(self, lines: List[str]) -> None:
        """_check_code_documentation should return a list of LintIssue."""
        result = _check_code_documentation(lines)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(lines=st.lists(batch_line_strategy(), max_size=50))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_var_naming_returns_list(self, lines: List[str]) -> None:
        """_check_var_naming should return a list of LintIssue."""
        result = _check_var_naming(lines)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(lines=st.lists(batch_line_strategy(), max_size=50))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_setlocal_redundancy_returns_list(self, lines: List[str]) -> None:
        """_check_setlocal_redundancy should return a list of LintIssue."""
        result = _check_setlocal_redundancy(lines)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(lines=st.lists(batch_line_strategy(), max_size=50))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_unreachable_code_returns_list(self, lines: List[str]) -> None:
        """_check_unreachable_code should return a list of LintIssue."""
        result = _check_unreachable_code(lines)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(lines=st.lists(batch_line_strategy(), max_size=50))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_redundant_operations_returns_list(self, lines: List[str]) -> None:
        """_check_redundant_operations should return a list of LintIssue."""
        result = _check_redundant_operations(lines)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(lines=st.lists(batch_line_strategy(), max_size=50))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_code_duplication_returns_list(self, lines: List[str]) -> None:
        """_check_code_duplication should return a list of LintIssue."""
        result = _check_code_duplication(lines)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(lines=st.lists(batch_line_strategy(), max_size=50))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_missing_pause_returns_list(self, lines: List[str]) -> None:
        """_check_missing_pause should return a list of LintIssue."""
        result = _check_missing_pause(lines)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(lines=st.lists(batch_line_strategy(), max_size=50))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_missing_header_doc_returns_list(self, lines: List[str]) -> None:
        """_check_missing_header_doc should return a list of LintIssue."""
        result = _check_missing_header_doc(lines)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(lines=st.lists(batch_line_strategy(), max_size=50))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_global_priv_security_returns_list(self, lines: List[str]) -> None:
        """_check_global_priv_security should return a list of LintIssue."""
        result = _check_global_priv_security(lines)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(lines=st.lists(batch_line_strategy(), max_size=50))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_nested_for_loops_returns_list(self, lines: List[str]) -> None:
        """_check_nested_for_loops should return a list of LintIssue."""
        result = _check_nested_for_loops(lines)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(lines=st.lists(batch_line_strategy(), max_size=50))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_external_error_handling_returns_list(self, lines: List[str]) -> None:
        """_check_external_error_handling should return a list of LintIssue."""
        result = _check_external_error_handling(lines)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(lines=st.lists(batch_line_strategy(), max_size=50))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_restart_limits_returns_list(self, lines: List[str]) -> None:
        """_check_restart_limits should return a list of LintIssue."""
        result = _check_restart_limits(lines)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(lines=st.lists(batch_line_strategy(), max_size=50))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_advanced_vars_returns_list(self, lines: List[str]) -> None:
        """_check_advanced_vars should return a list of LintIssue."""
        result = _check_advanced_vars(lines)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(lines=st.lists(batch_line_strategy(), max_size=50))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_enhanced_commands_returns_list(self, lines: List[str]) -> None:
        """_check_enhanced_commands should return a list of LintIssue."""
        result = _check_enhanced_commands(lines)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(lines=st.lists(batch_line_strategy(), max_size=50))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_advanced_style_rules_returns_list(self, lines: List[str]) -> None:
        """_check_advanced_style_rules should return a list of LintIssue."""
        result = _check_advanced_style_rules(lines)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(lines=st.lists(batch_line_strategy(), max_size=50))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_enhanced_security_rules_returns_list(self, lines: List[str]) -> None:
        """_check_enhanced_security_rules should return a list of LintIssue."""
        result = _check_enhanced_security_rules(lines)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(lines=st.lists(batch_line_strategy(), max_size=50))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_enhanced_performance_returns_list(self, lines: List[str]) -> None:
        """_check_enhanced_performance should return a list of LintIssue."""
        result = _check_enhanced_performance(lines)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(
        lines=st.lists(batch_line_strategy(), max_size=50),
        ending_type=st.sampled_from(["CRLF", "LF", "CR", "MIXED"]),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_multibyte_risks_returns_list(self, lines: List[str], ending_type: str) -> None:
        """_check_multibyte_risks should return a list of LintIssue."""
        result = _check_multibyte_risks(lines, ending_type)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(
        lines=st.lists(batch_line_strategy(), max_size=50),
        ending_type=st.sampled_from(["CRLF", "LF", "CR", "MIXED"]),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_goto_call_risks_returns_list(self, lines: List[str], ending_type: str) -> None:
        """_check_goto_call_risks should return a list of LintIssue."""
        result = _check_goto_call_risks(lines, ending_type)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(
        lines=st.lists(batch_line_strategy(), max_size=50),
        ending_type=st.sampled_from(["CRLF", "LF", "CR", "MIXED"]),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_doublecolon_risks_returns_list(self, lines: List[str], ending_type: str) -> None:
        """_check_doublecolon_risks should return a list of LintIssue."""
        result = _check_doublecolon_risks(lines, ending_type)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)


# ============================================================================
# File Path Dependent Function Property Tests
# ============================================================================


class TestFilePathDependentFunctionProperties:
    """Property-based tests for functions that require file paths."""

    @given(lines=st.lists(batch_line_strategy(), max_size=50))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None, max_examples=20)
    def test_check_line_ending_rules_with_temp_file(self, lines: List[str]) -> None:
        """_check_line_ending_rules should return a list of LintIssue."""
        with tempfile.NamedTemporaryFile(
            mode="w", suffix=".bat", delete=False, encoding="utf-8", newline=""
        ) as tmp:
            content = "\r\n".join(lines)
            tmp.write(content)
            tmp_path = tmp.name

        try:
            result = _check_line_ending_rules(lines, tmp_path)
            assert isinstance(result, list)
            for issue in result:
                assert isinstance(issue, LintIssue)
        finally:
            Path(tmp_path).unlink()

    @given(lines=st.lists(batch_line_strategy(), max_size=50))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None, max_examples=20)
    def test_check_global_style_rules_with_temp_file(self, lines: List[str]) -> None:
        """_check_global_style_rules should return a list of LintIssue."""
        with tempfile.NamedTemporaryFile(
            mode="w", suffix=".bat", delete=False, encoding="utf-8", newline=""
        ) as tmp:
            content = "\r\n".join(lines)
            tmp.write(content)
            tmp_path = tmp.name

        try:
            result = _check_global_style_rules(lines, tmp_path)
            assert isinstance(result, list)
            for issue in result:
                assert isinstance(issue, LintIssue)
        finally:
            Path(tmp_path).unlink()

    @given(lines=st.lists(batch_line_strategy(), max_size=50))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None, max_examples=20)
    def test_check_new_global_rules_with_temp_file(self, lines: List[str]) -> None:
        """_check_new_global_rules should return a list of LintIssue."""
        with tempfile.NamedTemporaryFile(
            mode="w", suffix=".bat", delete=False, encoding="utf-8", newline=""
        ) as tmp:
            content = "\r\n".join(lines)
            tmp.write(content)
            tmp_path = tmp.name

        try:
            result = _check_new_global_rules(lines, tmp_path)
            assert isinstance(result, list)
            for issue in result:
                assert isinstance(issue, LintIssue)
        finally:
            Path(tmp_path).unlink()

    @given(lines=st.lists(batch_line_strategy(), max_size=50))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None, max_examples=20)
    def test_check_bat_cmd_differences_with_temp_file(self, lines: List[str]) -> None:
        """_check_bat_cmd_differences should return a list of LintIssue."""
        with tempfile.NamedTemporaryFile(
            mode="w", suffix=".bat", delete=False, encoding="utf-8", newline=""
        ) as tmp:
            content = "\r\n".join(lines)
            tmp.write(content)
            tmp_path = tmp.name

        try:
            result = _check_bat_cmd_differences(lines, tmp_path)
            assert isinstance(result, list)
            for issue in result:
                assert isinstance(issue, LintIssue)
        finally:
            Path(tmp_path).unlink()

    @given(lines=st.lists(batch_line_strategy(), max_size=50))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None, max_examples=20)
    def test_check_advanced_global_patterns_with_temp_file(self, lines: List[str]) -> None:
        """_check_advanced_global_patterns should return a list of LintIssue."""
        with tempfile.NamedTemporaryFile(
            mode="w", suffix=".bat", delete=False, encoding="utf-8", newline=""
        ) as tmp:
            content = "\r\n".join(lines)
            tmp.write(content)
            tmp_path = tmp.name

        try:
            result = _check_advanced_global_patterns(lines, tmp_path)
            assert isinstance(result, list)
            for issue in result:
                assert isinstance(issue, LintIssue)
        finally:
            Path(tmp_path).unlink()

    @given(lines=st.lists(batch_line_strategy(), max_size=50))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None, max_examples=20)
    def test_check_self_modification_with_temp_file(self, lines: List[str]) -> None:
        """_check_self_modification should return a list of LintIssue."""
        with tempfile.NamedTemporaryFile(
            mode="w", suffix=".bat", delete=False, encoding="utf-8", newline=""
        ) as tmp:
            content = "\r\n".join(lines)
            tmp.write(content)
            tmp_path = tmp.name

        try:
            result = _check_self_modification(lines, tmp_path)
            assert isinstance(result, list)
            for issue in result:
                assert isinstance(issue, LintIssue)
        finally:
            Path(tmp_path).unlink()


# ============================================================================
# File Encoding Property Tests
# ============================================================================


class TestFileEncodingProperties:  # pylint: disable=too-few-public-methods
    """Property-based tests for file encoding functions."""

    @given(lines=st.lists(batch_line_strategy(), min_size=1, max_size=50))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None, max_examples=20)
    def test_read_file_with_encoding(self, lines: List[str]) -> None:
        """read_file_with_encoding should handle various encodings."""
        content = "\r\n".join(lines)

        with tempfile.NamedTemporaryFile(
            mode="w", suffix=".bat", delete=False, encoding="utf-8", newline=""
        ) as tmp:
            tmp.write(content)
            tmp_path = tmp.name

        try:
            result_lines, encoding = read_file_with_encoding(tmp_path)
            assert isinstance(result_lines, list)
            assert isinstance(encoding, str)
            for line in result_lines:
                assert isinstance(line, str)
        finally:
            Path(tmp_path).unlink()


# ============================================================================
# Complex Signature Function Property Tests
# ============================================================================


class TestComplexSignatureFunctionProperties:
    """Property-based tests for functions with complex signatures."""

    @given(
        line=st.text(max_size=200),
        line_num=st.integers(min_value=1, max_value=10000),
        stripped=st.text(max_size=200),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_compatibility_warnings_returns_list(
        self, line: str, line_num: int, stripped: str
    ) -> None:
        """_check_compatibility_warnings should return a list of LintIssue."""
        result = _check_compatibility_warnings(line, line_num, stripped)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(
        line=st.text(max_size=200),
        line_num=st.integers(min_value=1, max_value=10000),
        stripped=st.text(max_size=200),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_command_warnings_returns_list(
        self, line: str, line_num: int, stripped: str
    ) -> None:
        """_check_command_warnings should return a list of LintIssue."""
        result = _check_command_warnings(line, line_num, stripped)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(
        line=st.text(max_size=200),
        line_num=st.integers(min_value=1, max_value=10000),
        delayed_expansion_enabled=st.booleans(),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_warning_issues_returns_list(
        self, line: str, line_num: int, delayed_expansion_enabled: bool
    ) -> None:
        """_check_warning_issues should return a list of LintIssue."""
        set_vars: Set[str] = set()
        result = _check_warning_issues(line, line_num, set_vars, delayed_expansion_enabled)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(
        line=st.text(max_size=200),
        line_num=st.integers(min_value=1, max_value=10000),
        stripped=st.text(max_size=200),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_input_validation_sec_returns_list(
        self, line: str, line_num: int, stripped: str
    ) -> None:
        """_check_input_validation_sec should return a list of LintIssue."""
        result = _check_input_validation_sec(line, line_num, stripped)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(
        stripped=st.text(max_size=200),
        line_num=st.integers(min_value=1, max_value=10000),
        lines=st.lists(batch_line_strategy(), max_size=50),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_privilege_security_returns_list(
        self, stripped: str, line_num: int, lines: List[str]
    ) -> None:
        """_check_privilege_security should return a list of LintIssue."""
        result = _check_privilege_security(stripped, line_num, lines)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(
        line=st.text(max_size=200),
        stripped=st.text(max_size=200),
        line_num=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_path_security_returns_list(
        self, line: str, stripped: str, line_num: int
    ) -> None:
        """_check_path_security should return a list of LintIssue."""
        result = _check_path_security(line, stripped, line_num)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(
        lines=st.lists(batch_line_strategy(), min_size=1, max_size=50),
        line_number=st.integers(min_value=0),
        line=st.text(max_size=200),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_advanced_performance_returns_list(
        self, lines: List[str], line_number: int, line: str
    ) -> None:
        """_check_advanced_performance should return a list of LintIssue."""
        assume(line_number < len(lines))
        result = _check_advanced_performance(lines, line_number, line)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(
        line=st.text(max_size=200),
        line_number=st.integers(min_value=1, max_value=10000),
        lines=st.lists(batch_line_strategy(), max_size=50),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_advanced_security_returns_list(
        self, line: str, line_number: int, lines: List[str]
    ) -> None:
        """_check_advanced_security should return a list of LintIssue."""
        labels: Dict[str, int] = {}
        result = _check_advanced_security(line, line_number, lines, labels)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(
        line=st.text(max_size=200),
        line_number=st.integers(min_value=1, max_value=10000),
        lines=st.lists(batch_line_strategy(), max_size=50),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_advanced_style_patterns_returns_list(
        self, line: str, line_number: int, lines: List[str]
    ) -> None:
        """_check_advanced_style_patterns should return a list of LintIssue."""
        result = _check_advanced_style_patterns(line, line_number, lines)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(
        line=st.text(max_size=200),
        line_number=st.integers(min_value=1, max_value=10000),
        lines=st.lists(batch_line_strategy(), max_size=50),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_function_docs_returns_list(
        self, line: str, line_number: int, lines: List[str]
    ) -> None:
        """_check_function_docs should return a list of LintIssue."""
        result = _check_function_docs(line, line_number, lines)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(
        stripped=st.text(max_size=200),
        line_number=st.integers(min_value=1, max_value=10000),
        lines=st.lists(batch_line_strategy(), max_size=50),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_cmd_error_handling_returns_optional(
        self, stripped: str, line_number: int, lines: List[str]
    ) -> None:
        """_check_cmd_error_handling should return Optional[LintIssue]."""
        result = _check_cmd_error_handling(stripped, line_number, lines)
        assert result is None or isinstance(result, LintIssue)
        if result is not None:
            assert result.line_number == line_number

    @given(
        line=st.text(max_size=200),
        line_number=st.integers(min_value=1, max_value=10000),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_variable_naming_returns_list(self, line: str, line_number: int) -> None:
        """_check_variable_naming should return a list of LintIssue."""
        variables_seen: Dict[str, str] = {}
        result = _check_variable_naming(line, line_number, variables_seen)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(lines=st.lists(batch_line_strategy(), max_size=50))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_goto_colon_consistency_returns_list(self, lines: List[str]) -> None:
        """_check_goto_colon_consistency should return a list of LintIssue."""
        result = _check_goto_colon_consistency(lines)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(
        ending_info=st.tuples(
            st.sampled_from(["CRLF", "LF", "CR", "MIXED"]),
            st.booleans(),
            st.integers(min_value=0, max_value=100),
            st.integers(min_value=0, max_value=100),
            st.integers(min_value=0, max_value=100),
        )
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_basic_line_ending_issues_returns_list(
        self, ending_info: tuple[str, bool, int, int, int]
    ) -> None:
        """_check_basic_line_ending_issues should return a list of LintIssue."""
        result = _check_basic_line_ending_issues(ending_info)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(lines=st.lists(batch_line_strategy(), max_size=50))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_inconsistent_indentation_returns_list(self, lines: List[str]) -> None:
        """_check_inconsistent_indentation should return a list of LintIssue."""
        result = _check_inconsistent_indentation(lines)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(lines=st.lists(batch_line_strategy(), max_size=50))
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_missing_documentation_returns_list(self, lines: List[str]) -> None:
        """_check_missing_documentation should return a list of LintIssue."""
        result = _check_missing_documentation(lines)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(
        line=st.text(max_size=200),
        line_num=st.integers(min_value=1, max_value=10000),
        max_line_length=st.integers(min_value=80, max_value=500),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    def test_check_style_issues_returns_list(
        self, line: str, line_num: int, max_line_length: int
    ) -> None:
        """_check_style_issues should return a list of LintIssue."""
        result = _check_style_issues(line, line_num, max_line_length)
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)

    @given(
        lines=st.lists(batch_line_strategy(), max_size=50),
        line_num=st.integers(min_value=1, max_value=10000),
        line=st.text(max_size=200),
        has_setlocal=st.booleans(),
        has_set_commands=st.booleans(),
        has_delayed_expansion=st.booleans(),
        uses_delayed_vars=st.booleans(),
    )
    @settings(suppress_health_check=[HealthCheck.too_slow], deadline=None)
    # pylint: disable=too-many-arguments,too-many-positional-arguments
    def test_check_performance_issues_returns_list(
        self,
        lines: List[str],
        line_num: int,
        line: str,
        has_setlocal: bool,
        has_set_commands: bool,
        has_delayed_expansion: bool,
        uses_delayed_vars: bool,
    ) -> None:
        """_check_performance_issues should return a list of LintIssue."""
        result = _check_performance_issues(
            lines,
            line_num,
            line,
            has_setlocal,
            has_set_commands,
            has_delayed_expansion,
            uses_delayed_vars,
            False,  # has_disable_delayed_expansion
            False,  # has_literal_exclamations
            False,  # has_disable_expansion_lines
        )
        assert isinstance(result, list)
        for issue in result:
            assert isinstance(issue, LintIssue)
