# -*- coding: utf-8 -*-
#
# Copyright (C) 2015-2023 Bitergia
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
#
# Authors:
#   Alvaro del Castillo San Felix <acs@bitergia.com>
#

import csv
import logging
import re

from .enrich import Enrich, metadata
from ..elastic_mapping import Mapping as BaseMapping
from grimoirelab_toolkit.datetime import str_to_datetime

logger = logging.getLogger(__name__)


class Mapping(BaseMapping):

    @staticmethod
    def get_elastic_mappings(es_major):
        """Get Elasticsearch mapping.

        :param es_major: major version of Elasticsearch, as string
        :returns:        dictionary with a key, 'items', with the mapping
        """

        mapping = """
        {
            "properties": {
                "fullDisplayName_analyzed": {
                    "type": "text",
                    "index": true
                }
           }
        } """

        return {"items": mapping}


class JenkinsEnrich(Enrich):

    mapping = Mapping

    MAIN_NODE_NAME = "main"

    def __init__(self, db_sortinghat=None, json_projects_map=None,
                 db_user='', db_password='', db_host='', db_path=None,
                 db_port=None, db_ssl=False, db_verify_ssl=True, db_tenant=None,
                 node_regex=None):
        super().__init__(db_sortinghat=db_sortinghat, json_projects_map=json_projects_map,
                         db_user=db_user, db_password=db_password, db_host=db_host,
                         db_port=db_port, db_path=db_path, db_ssl=db_ssl, db_verify_ssl=db_verify_ssl,
                         db_tenant=db_tenant)
        self.nodes_rename_file = None
        self.nodes_rename = {}
        self.node_regex = node_regex

    def set_jenkins_rename_file(self, nodes_rename_file):
        """ File with nodes renaming mapping:

        Node,Comment
        arm-build1,remove
        arm-build2,keep
        ericsson-build3,merge into ericsson-build1
        ....

        Once set in the next enrichment the rename will be done
        """
        self.nodes_rename_file = nodes_rename_file
        self.__load_node_renames()
        logger.info("[jenkins] Node rename file active: {}".format(nodes_rename_file))

    def __load_node_renames(self):
        # In OPNFV nodes could be renamed
        if not self.nodes_rename_file:
            logger.debug("[jenkis] Node rename file not defined.")
            return
        try:
            with open(self.nodes_rename_file, 'r') as csvfile:
                nodes = csv.reader(csvfile, delimiter=',')
                for node in nodes:
                    name = node[0]
                    action = node[1]
                    rename = action.split("merge into ")
                    if len(rename) > 1:
                        self.nodes_rename[name] = rename[1]
                logger.debug("[jenkins] Total node renames: {}".format(len(self.nodes_rename.keys())))
        except FileNotFoundError:
            logger.info("[jenkins] Node rename file not found {}".format(self.nodes_rename_file))

    def get_field_author(self):
        # In Jenkins there are no identities support
        return None

    def get_identities(self, item):
        """ Return the identities from an item """
        identities = []

        return identities

    def has_identities(self):
        """ Return whether the enriched items contains identities """

        return False

    def get_fields_from_job_name(self, job_name):
        """Analyze a Jenkins job name, producing a dictionary

        The produced dictionary will include information about the category
        and subcategory of the job name, and any extra information which
        could be useful.

        For each deployment of a Jenkins dashboard, an implementation of
        this function should be produced, according to the needs of the users.

        :param job: job name to Analyze
        :returns:   dictionary with categorization information

        """

        extra_fields = {
            'category': None,
            'installer': None,
            'scenario': None,
            'testproject': None,
            'pod': None,
            'loop': None,
            'branch': None
        }

        try:
            components = job_name.split('-')

            if len(components) < 2:
                return extra_fields

            kind = components[1]
            if kind == 'os':
                extra_fields['category'] = 'parent/main'
                extra_fields['installer'] = components[0]
                extra_fields['scenario'] = '-'.join(components[2:-3])
            elif kind == 'deploy':
                extra_fields['category'] = 'deploy'
                extra_fields['installer'] = components[0]
            else:
                extra_fields['category'] = 'test'
                extra_fields['testproject'] = components[0]
                extra_fields['installer'] = components[1]

            extra_fields['pod'] = components[-3]
            extra_fields['loop'] = components[-2]
            extra_fields['branch'] = components[-1]
        except IndexError as ex:
            # Just DEBUG level because it is just for OPNFV
            logger.debug('[jenkins] Problems parsing job name {}'.format(job_name))
            logger.debug(ex)

        return extra_fields

    def extract_builton(self, built_on, regex):
        """Extracts node name using a regular expression. Node name is expected to
        be group 1.
        """
        pattern = re.compile(regex, re.M | re.I)
        match = pattern.search(built_on)
        if match and len(match.groups()) >= 1:
            node_name = match.group(1)
        else:
            msg = "[jenkins] Node name not extracted, using builtOn as it is: {}:{}".format(regex, built_on)
            logger.warning(msg)
            node_name = built_on

        return node_name

    @metadata
    def get_rich_item(self, item):
        eitem = {}

        self.copy_raw_fields(self.RAW_FIELDS_COPY, item, eitem)
        # The real data
        build = item['data']

        # data fields to copy
        copy_fields = ["fullDisplayName", "url", "result", "duration", "builtOn"]
        for f in copy_fields:
            if f in build:
                eitem[f] = build[f]
            else:
                eitem[f] = None
        # main node names
        if not eitem["builtOn"]:
            eitem["builtOn"] = self.MAIN_NODE_NAME
        # Nodes renaming
        if eitem["builtOn"] in self.nodes_rename:
            eitem["builtOn"] = self.nodes_rename[eitem["builtOn"]]

        # Rename nodes if regex is provided
        if self.node_regex:
            eitem["builtOn"] = self.extract_builton(eitem["builtOn"],
                                                    self.node_regex)

        # Fields which names are translated
        map_fields = {"fullDisplayName": "fullDisplayName_analyzed",
                      "number": "build"
                      }
        for fn in map_fields:
            eitem[map_fields[fn]] = build[fn]

        # Job url: remove the last /build_id from job_url/build_id/
        eitem['job_url'] = eitem['url'].rsplit("/", 2)[0]
        eitem['job_name'] = eitem['url'].rsplit('/', 3)[1]
        eitem['job_build'] = eitem['job_name'] + '/' + str(eitem['build'])

        # Enrich dates
        eitem["build_date"] = str_to_datetime(item["metadata__updated_on"]).isoformat()

        # Add duration in days
        if "duration" in eitem:
            seconds_day = float(60 * 60 * 24)
            duration_days = eitem["duration"] / (1000 * seconds_day)
            eitem["duration_days"] = float('%.2f' % duration_days)

        # Add extra fields extracted from job name
        eitem.update(self.get_fields_from_job_name(eitem['job_name']))

        eitem.update(self.get_grimoire_fields(item["metadata__updated_on"], "job"))

        if self.prjs_map:
            eitem.update(self.get_item_project(eitem))

        self.add_repository_labels(eitem)
        self.add_metadata_filter_raw(eitem)
        return eitem
