from __future__ import annotations

__all__ = []

import json
from collections.abc import Sequence
from pathlib import Path
from typing import Any, Literal

from maite_datasets._base import (
    BaseDatasetNumpyMixin,
    BaseODDataset,
    DataLocation,
    NumpyArray,
    NumpyObjectDetectionTarget,
    NumpyObjectDetectionTransform,
)
from maite_datasets._fileio import _ensure_exists


class SeaDrone(
    BaseODDataset[
        NumpyArray,
        NumpyObjectDetectionTarget,
        list[tuple[list[int], list[list[float]]]],
        tuple[list[int], list[list[float]]],
    ],
    BaseDatasetNumpyMixin,
):
    """
    A UAV dataset focused on open water object detection.

    The dataset comes from the paper
    `SeaDronesSee: A Maritime Benchmark for Detecting Humans in Open Water <https://openaccess.thecvf.com/content/WACV2022/html/Varga_SeaDronesSee_A_Maritime_Benchmark_for_Detecting_Humans_in_Open_Water_WACV_2022_paper.html>`_
    by L.A. Varga et. al. (2022).

    This contains the compressed (.jpg) version of the `object detection v2 dataset <https://seadronessee.cs.uni-tuebingen.de/dataset>`_.
    Dataset is 9.96 GB.
    Images were collected at varying altitudes (5-260 m), at a variety of viewing angles (0-90&deg;) and drone speeds.
    All images contain the above metadata as well as some additional information (based on the drone),
    while only the training and validation images contain ground truth labels.
    Three different drones were used to collect the images, 2 quadcopters and 1 fixed-wing.

    There are 14,227 images: 8930 images in the training set, 1547 images in the validation set,
    and 3750 images in the test set.
    There are 5 classes in the dataset: 1-swimmer, 2-boat, 3-jetski, 4-life_saving_appliances, 5-buoy.
    The annotation files provide the metadata and the ground-truth labels in the COCO format.
    Bounding boxes are (x, y, w, h).
    The images come in a variety of sizes, with 3632 x 5456 (149 images) or 5456 x 3632 (47 images) being the largest
    size, 2160 x 3840 (1030 images) being the most common size, and 932 x 1230 (5 images) being the smallest size.

    Parameters
    ----------
    root : str or pathlib.Path
        Root directory of dataset where the ``milco`` folder exists.
    image_set: "train", "val", "test", or "base", default "train"
        If "base", then the full dataset is selected (train, val and test).
    transforms : Transform, Sequence[Transform] or None, default None
        Transform(s) to apply to the data.
    download : bool, default False
        If True, downloads the dataset from the internet and puts it in root directory.
        Class checks to see if data is already downloaded to ensure it does not create a duplicate download.
    verbose : bool, default False
        If True, outputs print statements.

    Attributes
    ----------
    path : pathlib.Path
        Location of the folder containing the data.
    image_set : "train", "val", "test", or "base"
        The selected image set from the dataset.
    index2label : dict[int, str]
        Dictionary which translates from class integers to the associated class strings.
    label2index : dict[str, int]
        Dictionary which translates from class strings to the associated class integers.
    metadata : DatasetMetadata
        Typed dictionary containing dataset metadata, such as `id` which returns the dataset class name.
    transforms : Sequence[Transform]
        The transforms to be applied to the data.
    size : int
        The size of the dataset.

    Note
    ----
    Data License: `Creative Commons CC0 1.0 Universal <https://creativecommons.org/publicdomain/zero/1.0/deed.en>`_
    """

    _resources = [
        DataLocation(
            url="https://cloud.cs.uni-tuebingen.de/public.php/dav/files/ZZxX65FGnQ8zjBP/Compressed%20Version/images/train/?accept=zip&files=%5B%222.jpg%22%2C%225.jpg%22%2C%227.jpg%22%2C%228.jpg%22%2C%229.jpg%22%2C%2211.jpg%22%2C%2212.jpg%22%2C%2213.jpg%22%2C%2215.jpg%22%2C%2219.jpg%22%2C%2220.jpg%22%2C%2221.jpg%22%2C%2222.jpg%22%2C%2224.jpg%22%2C%2225.jpg%22%2C%2226.jpg%22%2C%2229.jpg%22%2C%2230.jpg%22%2C%2234.jpg%22%2C%2236.jpg%22%2C%2237.jpg%22%2C%2238.jpg%22%2C%2240.jpg%22%2C%2241.jpg%22%2C%2242.jpg%22%2C%2243.jpg%22%2C%2244.jpg%22%2C%2247.jpg%22%2C%2250.jpg%22%2C%2252.jpg%22%2C%2253.jpg%22%2C%2255.jpg%22%2C%2259.jpg%22%2C%2260.jpg%22%2C%2261.jpg%22%2C%2266.jpg%22%2C%2268.jpg%22%2C%2269.jpg%22%2C%2272.jpg%22%2C%2273.jpg%22%2C%2276.jpg%22%2C%2278.jpg%22%2C%2281.jpg%22%2C%2283.jpg%22%2C%2284.jpg%22%2C%2286.jpg%22%2C%2287.jpg%22%2C%2291.jpg%22%2C%2294.jpg%22%2C%2296.jpg%22%2C%22100.jpg%22%2C%22101.jpg%22%2C%22102.jpg%22%2C%22104.jpg%22%2C%22106.jpg%22%2C%22107.jpg%22%2C%22110.jpg%22%2C%22111.jpg%22%2C%22112.jpg%22%2C%22115.jpg%22%2C%22117.jpg%22%2C%22121.jpg%22%2C%22123.jpg%22%2C%22125.jpg%22%2C%22131.jpg%22%2C%22140.jpg%22%2C%22144.jpg%22%2C%22149.jpg%22%2C%22152.jpg%22%2C%22156.jpg%22%2C%22157.jpg%22%2C%22158.jpg%22%2C%22159.jpg%22%2C%22160.jpg%22%2C%22165.jpg%22%2C%22167.jpg%22%2C%22170.jpg%22%2C%22171.jpg%22%2C%22176.jpg%22%2C%22178.jpg%22%2C%22180.jpg%22%2C%22181.jpg%22%2C%22190.jpg%22%2C%22191.jpg%22%2C%22192.jpg%22%2C%22193.jpg%22%2C%22195.jpg%22%2C%22197.jpg%22%2C%22200.jpg%22%2C%22201.jpg%22%2C%22205.jpg%22%2C%22208.jpg%22%2C%22211.jpg%22%2C%22213.jpg%22%2C%22214.jpg%22%2C%22217.jpg%22%2C%22224.jpg%22%2C%22228.jpg%22%2C%22229.jpg%22%2C%22230.jpg%22%2C%22231.jpg%22%2C%22233.jpg%22%2C%22237.jpg%22%2C%22238.jpg%22%2C%22247.jpg%22%2C%22248.jpg%22%2C%22249.jpg%22%2C%22251.jpg%22%2C%22255.jpg%22%2C%22259.jpg%22%2C%22260.jpg%22%2C%22261.jpg%22%2C%22266.jpg%22%2C%22271.jpg%22%2C%22273.jpg%22%2C%22274.jpg%22%2C%22275.jpg%22%2C%22278.jpg%22%2C%22279.jpg%22%2C%22280.jpg%22%2C%22281.jpg%22%2C%22283.jpg%22%2C%22285.jpg%22%2C%22287.jpg%22%2C%22289.jpg%22%2C%22291.jpg%22%2C%22293.jpg%22%2C%22295.jpg%22%2C%22296.jpg%22%2C%22301.jpg%22%2C%22307.jpg%22%2C%22308.jpg%22%2C%22309.jpg%22%2C%22311.jpg%22%2C%22313.jpg%22%2C%22316.jpg%22%2C%22318.jpg%22%2C%22319.jpg%22%2C%22322.jpg%22%2C%22327.jpg%22%2C%22328.jpg%22%2C%22330.jpg%22%2C%22332.jpg%22%2C%22334.jpg%22%2C%22336.jpg%22%2C%22341.jpg%22%2C%22343.jpg%22%2C%22344.jpg%22%2C%22353.jpg%22%2C%22354.jpg%22%2C%22358.jpg%22%2C%22359.jpg%22%2C%22360.jpg%22%2C%22362.jpg%22%2C%22363.jpg%22%2C%22364.jpg%22%2C%22365.jpg%22%2C%22366.jpg%22%2C%22367.jpg%22%2C%22368.jpg%22%2C%22369.jpg%22%2C%22375.jpg%22%2C%22376.jpg%22%2C%22378.jpg%22%2C%22380.jpg%22%2C%22384.jpg%22%2C%22385.jpg%22%2C%22390.jpg%22%2C%22392.jpg%22%2C%22394.jpg%22%2C%22395.jpg%22%2C%22396.jpg%22%2C%22403.jpg%22%2C%22406.jpg%22%2C%22409.jpg%22%2C%22415.jpg%22%2C%22418.jpg%22%2C%22419.jpg%22%2C%22421.jpg%22%2C%22423.jpg%22%2C%22431.jpg%22%2C%22432.jpg%22%2C%22433.jpg%22%2C%22434.jpg%22%2C%22436.jpg%22%2C%22437.jpg%22%2C%22438.jpg%22%2C%22440.jpg%22%2C%22441.jpg%22%2C%22443.jpg%22%2C%22450.jpg%22%2C%22452.jpg%22%2C%22453.jpg%22%2C%22457.jpg%22%2C%22459.jpg%22%2C%22461.jpg%22%2C%22463.jpg%22%2C%22467.jpg%22%2C%22468.jpg%22%2C%22469.jpg%22%2C%22470.jpg%22%2C%22473.jpg%22%2C%22477.jpg%22%2C%22481.jpg%22%2C%22484.jpg%22%2C%22487.jpg%22%2C%22488.jpg%22%2C%22491.jpg%22%2C%22492.jpg%22%2C%22493.jpg%22%2C%22494.jpg%22%2C%22499.jpg%22%2C%22504.jpg%22%2C%22507.jpg%22%2C%22508.jpg%22%2C%22509.jpg%22%2C%22511.jpg%22%2C%22513.jpg%22%2C%22514.jpg%22%2C%22515.jpg%22%2C%22516.jpg%22%2C%22517.jpg%22%2C%22519.jpg%22%2C%22522.jpg%22%2C%22524.jpg%22%2C%22527.jpg%22%2C%22531.jpg%22%2C%22532.jpg%22%2C%22533.jpg%22%2C%22534.jpg%22%2C%22536.jpg%22%2C%22538.jpg%22%2C%22539.jpg%22%2C%22540.jpg%22%2C%22541.jpg%22%2C%22545.jpg%22%2C%22546.jpg%22%2C%22547.jpg%22%2C%22548.jpg%22%2C%22553.jpg%22%2C%22554.jpg%22%2C%22556.jpg%22%2C%22558.jpg%22%2C%22561.jpg%22%2C%22562.jpg%22%2C%22564.jpg%22%2C%22567.jpg%22%2C%22568.jpg%22%2C%22570.jpg%22%2C%22571.jpg%22%2C%22572.jpg%22%2C%22574.jpg%22%2C%22576.jpg%22%2C%22578.jpg%22%2C%22581.jpg%22%2C%22582.jpg%22%2C%22583.jpg%22%2C%22587.jpg%22%2C%22588.jpg%22%2C%22590.jpg%22%2C%22591.jpg%22%2C%22592.jpg%22%2C%22594.jpg%22%2C%22596.jpg%22%2C%22598.jpg%22%2C%22599.jpg%22%2C%22600.jpg%22%2C%22601.jpg%22%2C%22602.jpg%22%2C%22606.jpg%22%2C%22608.jpg%22%2C%22609.jpg%22%2C%22610.jpg%22%2C%22611.jpg%22%2C%22612.jpg%22%2C%22613.jpg%22%2C%22614.jpg%22%2C%22615.jpg%22%2C%22617.jpg%22%2C%22620.jpg%22%2C%22621.jpg%22%2C%22623.jpg%22%2C%22624.jpg%22%2C%22627.jpg%22%2C%22631.jpg%22%2C%22632.jpg%22%2C%22636.jpg%22%2C%22639.jpg%22%2C%22640.jpg%22%2C%22641.jpg%22%2C%22642.jpg%22%2C%22643.jpg%22%2C%22644.jpg%22%2C%22645.jpg%22%2C%22647.jpg%22%2C%22648.jpg%22%2C%22649.jpg%22%2C%22650.jpg%22%2C%22651.jpg%22%2C%22653.jpg%22%2C%22656.jpg%22%2C%22660.jpg%22%2C%22661.jpg%22%2C%22662.jpg%22%2C%22663.jpg%22%2C%22664.jpg%22%2C%22665.jpg%22%2C%22667.jpg%22%2C%22668.jpg%22%2C%22672.jpg%22%2C%22673.jpg%22%2C%22674.jpg%22%2C%22675.jpg%22%2C%22678.jpg%22%2C%22679.jpg%22%2C%22680.jpg%22%2C%22681.jpg%22%2C%22682.jpg%22%2C%22683.jpg%22%2C%22684.jpg%22%2C%22686.jpg%22%2C%22688.jpg%22%2C%22689.jpg%22%2C%22691.jpg%22%2C%22692.jpg%22%2C%22694.jpg%22%2C%22695.jpg%22%2C%22696.jpg%22%2C%22698.jpg%22%2C%22699.jpg%22%2C%22701.jpg%22%2C%22702.jpg%22%2C%22703.jpg%22%2C%22705.jpg%22%2C%22706.jpg%22%2C%22709.jpg%22%2C%22710.jpg%22%2C%22711.jpg%22%2C%22712.jpg%22%2C%22714.jpg%22%2C%22715.jpg%22%2C%22716.jpg%22%2C%22717.jpg%22%2C%22720.jpg%22%2C%22721.jpg%22%2C%22724.jpg%22%2C%22725.jpg%22%2C%22726.jpg%22%2C%22727.jpg%22%2C%22728.jpg%22%2C%22730.jpg%22%2C%22731.jpg%22%2C%22732.jpg%22%2C%22733.jpg%22%2C%22736.jpg%22%2C%22738.jpg%22%2C%22739.jpg%22%2C%22740.jpg%22%2C%22742.jpg%22%2C%22743.jpg%22%2C%22744.jpg%22%2C%22745.jpg%22%2C%22746.jpg%22%2C%22747.jpg%22%2C%22750.jpg%22%2C%22752.jpg%22%2C%22755.jpg%22%2C%22756.jpg%22%2C%22759.jpg%22%2C%22762.jpg%22%2C%22763.jpg%22%2C%22764.jpg%22%2C%22766.jpg%22%2C%22768.jpg%22%2C%22775.jpg%22%2C%22776.jpg%22%2C%22778.jpg%22%2C%22782.jpg%22%2C%22783.jpg%22%2C%22785.jpg%22%2C%22786.jpg%22%2C%22787.jpg%22%2C%22788.jpg%22%2C%22797.jpg%22%2C%22799.jpg%22%2C%22800.jpg%22%2C%22801.jpg%22%2C%22802.jpg%22%2C%22804.jpg%22%2C%22806.jpg%22%2C%22809.jpg%22%2C%22810.jpg%22%2C%22811.jpg%22%2C%22812.jpg%22%2C%22814.jpg%22%2C%22816.jpg%22%2C%22817.jpg%22%2C%22819.jpg%22%2C%22820.jpg%22%2C%22821.jpg%22%2C%22822.jpg%22%2C%22823.jpg%22%2C%22825.jpg%22%2C%22826.jpg%22%2C%22829.jpg%22%2C%22831.jpg%22%2C%22833.jpg%22%2C%22834.jpg%22%2C%22835.jpg%22%2C%22837.jpg%22%2C%22839.jpg%22%2C%22841.jpg%22%2C%22846.jpg%22%2C%22848.jpg%22%2C%22849.jpg%22%2C%22850.jpg%22%2C%22855.jpg%22%2C%22864.jpg%22%2C%22867.jpg%22%2C%22872.jpg%22%2C%22873.jpg%22%2C%22875.jpg%22%2C%22876.jpg%22%2C%22880.jpg%22%2C%22881.jpg%22%2C%22893.jpg%22%2C%22897.jpg%22%2C%22900.jpg%22%2C%22906.jpg%22%2C%22907.jpg%22%2C%22910.jpg%22%2C%22913.jpg%22%2C%22916.jpg%22%2C%22919.jpg%22%2C%22920.jpg%22%2C%22922.jpg%22%2C%22924.jpg%22%2C%22925.jpg%22%2C%22926.jpg%22%2C%22927.jpg%22%2C%22930.jpg%22%2C%22931.jpg%22%2C%22935.jpg%22%2C%22936.jpg%22%2C%22937.jpg%22%2C%22938.jpg%22%2C%22939.jpg%22%2C%22941.jpg%22%2C%22943.jpg%22%2C%22950.jpg%22%5D",
            filename="train1.zip",
            md5=False,
            checksum="5e2216cd4da5925a47f59c7d2e0bc597d4e7bb50f6e58bad390bbd83ba5dede0",
        ),
        DataLocation(
            url="https://cloud.cs.uni-tuebingen.de/public.php/dav/files/ZZxX65FGnQ8zjBP/Compressed%20Version/images/train/?accept=zip&files=%5B%22954.jpg%22%2C%22958.jpg%22%2C%22960.jpg%22%2C%22964.jpg%22%2C%22968.jpg%22%2C%22974.jpg%22%2C%22975.jpg%22%2C%22976.jpg%22%2C%22978.jpg%22%2C%22981.jpg%22%2C%22988.jpg%22%2C%22997.jpg%22%2C%22999.jpg%22%2C%221000.jpg%22%2C%221002.jpg%22%2C%221004.jpg%22%2C%221009.jpg%22%2C%221011.jpg%22%2C%221013.jpg%22%2C%221014.jpg%22%2C%221019.jpg%22%2C%221020.jpg%22%2C%221022.jpg%22%2C%221025.jpg%22%2C%221031.jpg%22%2C%221034.jpg%22%2C%221035.jpg%22%2C%221038.jpg%22%2C%221042.jpg%22%2C%221044.jpg%22%2C%221048.jpg%22%2C%221051.jpg%22%2C%221053.jpg%22%2C%221055.jpg%22%2C%221056.jpg%22%2C%221057.jpg%22%2C%221060.jpg%22%2C%221063.jpg%22%2C%221066.jpg%22%2C%221068.jpg%22%2C%221069.jpg%22%2C%221077.jpg%22%2C%221078.jpg%22%2C%221079.jpg%22%2C%221080.jpg%22%2C%221081.jpg%22%2C%221082.jpg%22%2C%221084.jpg%22%2C%221085.jpg%22%2C%221090.jpg%22%2C%221093.jpg%22%2C%221095.jpg%22%2C%221098.jpg%22%2C%221099.jpg%22%2C%221101.jpg%22%2C%221108.jpg%22%2C%221112.jpg%22%2C%221115.jpg%22%2C%221116.jpg%22%2C%221117.jpg%22%2C%221119.jpg%22%2C%221121.jpg%22%2C%221127.jpg%22%2C%221130.jpg%22%2C%221134.jpg%22%2C%221140.jpg%22%2C%221148.jpg%22%2C%221152.jpg%22%2C%221153.jpg%22%2C%221156.jpg%22%2C%221159.jpg%22%2C%221162.jpg%22%2C%221172.jpg%22%2C%221179.jpg%22%2C%221184.jpg%22%2C%221186.jpg%22%2C%221187.jpg%22%2C%221189.jpg%22%2C%221190.jpg%22%2C%221193.jpg%22%2C%221194.jpg%22%2C%221197.jpg%22%2C%221198.jpg%22%2C%221199.jpg%22%2C%221200.jpg%22%2C%221201.jpg%22%2C%221207.jpg%22%2C%221208.jpg%22%2C%221212.jpg%22%2C%221213.jpg%22%2C%221218.jpg%22%2C%221219.jpg%22%2C%221225.jpg%22%2C%221226.jpg%22%2C%221237.jpg%22%2C%221238.jpg%22%2C%221245.jpg%22%2C%221246.jpg%22%2C%221248.jpg%22%2C%221249.jpg%22%2C%221252.jpg%22%2C%221257.jpg%22%2C%221264.jpg%22%2C%221265.jpg%22%2C%221268.jpg%22%2C%221274.jpg%22%2C%221278.jpg%22%2C%221282.jpg%22%2C%221284.jpg%22%2C%221285.jpg%22%2C%221287.jpg%22%2C%221288.jpg%22%2C%221289.jpg%22%2C%221290.jpg%22%2C%221303.jpg%22%2C%221304.jpg%22%2C%221305.jpg%22%2C%221307.jpg%22%2C%221310.jpg%22%2C%221312.jpg%22%2C%221315.jpg%22%2C%221322.jpg%22%2C%221324.jpg%22%2C%221328.jpg%22%2C%221333.jpg%22%2C%221336.jpg%22%2C%221339.jpg%22%2C%221340.jpg%22%2C%221344.jpg%22%2C%221345.jpg%22%2C%221346.jpg%22%2C%221347.jpg%22%2C%221352.jpg%22%2C%221354.jpg%22%2C%221363.jpg%22%2C%221368.jpg%22%2C%221369.jpg%22%2C%221370.jpg%22%2C%221371.jpg%22%2C%221372.jpg%22%2C%221373.jpg%22%2C%221374.jpg%22%2C%221377.jpg%22%2C%221378.jpg%22%2C%221379.jpg%22%2C%221380.jpg%22%2C%221381.jpg%22%2C%221384.jpg%22%2C%221386.jpg%22%2C%221387.jpg%22%2C%221391.jpg%22%2C%221392.jpg%22%2C%221395.jpg%22%2C%221396.jpg%22%2C%221397.jpg%22%2C%221398.jpg%22%2C%221400.jpg%22%2C%221401.jpg%22%2C%221402.jpg%22%2C%221406.jpg%22%2C%221411.jpg%22%2C%221415.jpg%22%2C%221417.jpg%22%2C%221420.jpg%22%2C%221427.jpg%22%2C%221428.jpg%22%2C%221429.jpg%22%2C%221430.jpg%22%2C%221432.jpg%22%2C%221433.jpg%22%2C%221435.jpg%22%2C%221440.jpg%22%2C%221444.jpg%22%2C%221447.jpg%22%2C%221449.jpg%22%2C%221450.jpg%22%2C%221459.jpg%22%2C%221464.jpg%22%2C%221465.jpg%22%2C%221476.jpg%22%2C%221478.jpg%22%2C%221482.jpg%22%2C%221484.jpg%22%2C%221485.jpg%22%2C%221486.jpg%22%2C%221488.jpg%22%2C%221490.jpg%22%2C%221495.jpg%22%2C%221496.jpg%22%2C%221497.jpg%22%2C%221498.jpg%22%2C%221501.jpg%22%2C%221504.jpg%22%2C%221508.jpg%22%2C%221509.jpg%22%2C%221511.jpg%22%2C%221514.jpg%22%2C%221516.jpg%22%2C%221518.jpg%22%2C%221521.jpg%22%2C%221522.jpg%22%2C%221523.jpg%22%2C%221529.jpg%22%2C%221531.jpg%22%2C%221532.jpg%22%2C%221533.jpg%22%2C%221534.jpg%22%2C%221535.jpg%22%2C%221536.jpg%22%2C%221537.jpg%22%2C%221538.jpg%22%2C%221540.jpg%22%2C%221541.jpg%22%2C%221547.jpg%22%2C%221554.jpg%22%2C%221559.jpg%22%2C%221563.jpg%22%2C%221564.jpg%22%2C%221565.jpg%22%2C%221567.jpg%22%2C%221569.jpg%22%2C%221570.jpg%22%2C%221575.jpg%22%2C%221577.jpg%22%2C%221584.jpg%22%2C%221590.jpg%22%2C%221600.jpg%22%2C%221601.jpg%22%2C%221606.jpg%22%2C%221608.jpg%22%2C%221609.jpg%22%2C%221611.jpg%22%2C%221613.jpg%22%2C%221614.jpg%22%2C%221616.jpg%22%2C%221618.jpg%22%2C%221619.jpg%22%2C%221620.jpg%22%2C%221622.jpg%22%2C%221625.jpg%22%2C%221626.jpg%22%2C%221627.jpg%22%2C%221630.jpg%22%2C%221634.jpg%22%2C%221635.jpg%22%2C%221636.jpg%22%2C%221638.jpg%22%2C%221640.jpg%22%2C%221642.jpg%22%2C%221646.jpg%22%2C%221648.jpg%22%2C%221649.jpg%22%2C%221650.jpg%22%2C%221651.jpg%22%2C%221653.jpg%22%2C%221654.jpg%22%2C%221657.jpg%22%2C%221662.jpg%22%2C%221666.jpg%22%2C%221668.jpg%22%2C%221672.jpg%22%2C%221673.jpg%22%2C%221674.jpg%22%2C%221679.jpg%22%2C%221681.jpg%22%2C%221683.jpg%22%2C%221684.jpg%22%2C%221685.jpg%22%2C%221697.jpg%22%2C%221700.jpg%22%2C%221707.jpg%22%2C%221708.jpg%22%2C%221711.jpg%22%2C%221712.jpg%22%2C%221713.jpg%22%2C%221715.jpg%22%2C%221719.jpg%22%2C%221720.jpg%22%2C%221723.jpg%22%2C%221725.jpg%22%2C%221729.jpg%22%2C%221730.jpg%22%2C%221736.jpg%22%2C%221739.jpg%22%2C%221751.jpg%22%2C%221753.jpg%22%2C%221754.jpg%22%2C%221755.jpg%22%2C%221760.jpg%22%2C%221761.jpg%22%2C%221763.jpg%22%2C%221770.jpg%22%2C%221776.jpg%22%2C%221777.jpg%22%2C%221779.jpg%22%2C%221780.jpg%22%2C%221782.jpg%22%2C%221784.jpg%22%2C%221788.jpg%22%2C%221790.jpg%22%2C%221794.jpg%22%2C%221795.jpg%22%2C%221797.jpg%22%2C%221799.jpg%22%2C%221801.jpg%22%2C%221802.jpg%22%2C%221804.jpg%22%2C%221808.jpg%22%2C%221811.jpg%22%2C%221813.jpg%22%2C%221814.jpg%22%2C%221815.jpg%22%2C%221818.jpg%22%2C%221822.jpg%22%2C%221825.jpg%22%2C%221829.jpg%22%2C%221831.jpg%22%2C%221833.jpg%22%2C%221834.jpg%22%2C%221836.jpg%22%2C%221837.jpg%22%2C%221838.jpg%22%2C%221841.jpg%22%2C%221843.jpg%22%2C%221846.jpg%22%2C%221849.jpg%22%2C%221851.jpg%22%2C%221855.jpg%22%2C%221856.jpg%22%2C%221857.jpg%22%2C%221860.jpg%22%2C%221873.jpg%22%2C%221874.jpg%22%2C%221877.jpg%22%2C%221878.jpg%22%2C%221886.jpg%22%2C%221887.jpg%22%2C%221888.jpg%22%2C%221889.jpg%22%2C%221890.jpg%22%2C%221899.jpg%22%2C%221901.jpg%22%2C%221905.jpg%22%2C%221907.jpg%22%2C%221909.jpg%22%2C%221912.jpg%22%2C%221914.jpg%22%2C%221916.jpg%22%2C%221917.jpg%22%2C%221919.jpg%22%2C%221920.jpg%22%2C%221923.jpg%22%2C%221927.jpg%22%2C%221930.jpg%22%2C%221932.jpg%22%2C%221934.jpg%22%2C%221937.jpg%22%2C%221938.jpg%22%2C%221939.jpg%22%2C%221942.jpg%22%2C%221949.jpg%22%2C%221956.jpg%22%2C%221964.jpg%22%2C%221967.jpg%22%2C%221971.jpg%22%2C%221972.jpg%22%2C%221973.jpg%22%2C%221979.jpg%22%2C%221981.jpg%22%2C%221989.jpg%22%2C%221992.jpg%22%2C%221993.jpg%22%2C%221994.jpg%22%2C%221996.jpg%22%2C%221997.jpg%22%2C%222000.jpg%22%2C%222004.jpg%22%2C%222006.jpg%22%2C%222007.jpg%22%2C%222009.jpg%22%2C%222012.jpg%22%2C%222013.jpg%22%2C%222025.jpg%22%2C%222028.jpg%22%2C%222029.jpg%22%2C%222031.jpg%22%2C%222034.jpg%22%2C%222038.jpg%22%2C%222045.jpg%22%2C%222047.jpg%22%2C%222053.jpg%22%2C%222055.jpg%22%2C%222057.jpg%22%2C%222061.jpg%22%2C%222063.jpg%22%2C%222069.jpg%22%2C%222074.jpg%22%2C%222075.jpg%22%2C%222080.jpg%22%2C%222081.jpg%22%2C%222083.jpg%22%2C%222085.jpg%22%2C%222087.jpg%22%2C%222089.jpg%22%2C%222092.jpg%22%2C%222095.jpg%22%2C%222096.jpg%22%2C%222104.jpg%22%2C%222120.jpg%22%2C%222123.jpg%22%2C%222125.jpg%22%2C%222126.jpg%22%2C%222127.jpg%22%2C%222128.jpg%22%2C%222129.jpg%22%2C%222130.jpg%22%2C%222131.jpg%22%2C%222132.jpg%22%2C%222137.jpg%22%2C%222138.jpg%22%2C%222141.jpg%22%2C%222142.jpg%22%2C%222143.jpg%22%2C%222145.jpg%22%2C%222146.jpg%22%2C%222147.jpg%22%2C%222148.jpg%22%2C%222151.jpg%22%2C%222155.jpg%22%2C%222157.jpg%22%2C%222159.jpg%22%2C%222161.jpg%22%2C%222165.jpg%22%2C%222169.jpg%22%2C%222172.jpg%22%2C%222173.jpg%22%2C%222174.jpg%22%2C%222176.jpg%22%2C%222178.jpg%22%2C%222179.jpg%22%2C%222180.jpg%22%2C%222183.jpg%22%2C%222184.jpg%22%2C%222189.jpg%22%2C%222190.jpg%22%2C%222194.jpg%22%2C%222195.jpg%22%2C%222196.jpg%22%2C%222197.jpg%22%2C%222199.jpg%22%5D",
            filename="train2.zip",
            md5=False,
            checksum="799642f2e2a322b20dc63033b5a0b74371315d58cd2e66183d372858c258bacc",
        ),
        DataLocation(
            url="https://cloud.cs.uni-tuebingen.de/public.php/dav/files/ZZxX65FGnQ8zjBP/Compressed%20Version/images/train/?accept=zip&files=%5B%222200.jpg%22%2C%222202.jpg%22%2C%222203.jpg%22%2C%222204.jpg%22%2C%222212.jpg%22%2C%222213.jpg%22%2C%222214.jpg%22%2C%222216.jpg%22%2C%222217.jpg%22%2C%222221.jpg%22%2C%222222.jpg%22%2C%222223.jpg%22%2C%222224.jpg%22%2C%222226.jpg%22%2C%222229.jpg%22%2C%222230.jpg%22%2C%222233.jpg%22%2C%222235.jpg%22%2C%222240.jpg%22%2C%222243.jpg%22%2C%222246.jpg%22%2C%222247.jpg%22%2C%222251.jpg%22%2C%222252.jpg%22%2C%222257.jpg%22%2C%222258.jpg%22%2C%222261.jpg%22%2C%222262.jpg%22%2C%222263.jpg%22%2C%222264.jpg%22%2C%222269.jpg%22%2C%222270.jpg%22%2C%222273.jpg%22%2C%222279.jpg%22%2C%222280.jpg%22%2C%222290.jpg%22%2C%222293.jpg%22%2C%222294.jpg%22%2C%222295.jpg%22%2C%222298.jpg%22%2C%222299.jpg%22%2C%222302.jpg%22%2C%222308.jpg%22%2C%222311.jpg%22%2C%222312.jpg%22%2C%222317.jpg%22%2C%222319.jpg%22%2C%222320.jpg%22%2C%222321.jpg%22%2C%222323.jpg%22%2C%222324.jpg%22%2C%222327.jpg%22%2C%222329.jpg%22%2C%222330.jpg%22%2C%222335.jpg%22%2C%222338.jpg%22%2C%222339.jpg%22%2C%222344.jpg%22%2C%222350.jpg%22%2C%222351.jpg%22%2C%222352.jpg%22%2C%222356.jpg%22%2C%222357.jpg%22%2C%222361.jpg%22%2C%222362.jpg%22%2C%222363.jpg%22%2C%222367.jpg%22%2C%222376.jpg%22%2C%222383.jpg%22%2C%222384.jpg%22%2C%222385.jpg%22%2C%222391.jpg%22%2C%222393.jpg%22%2C%222394.jpg%22%2C%222397.jpg%22%2C%222399.jpg%22%2C%222403.jpg%22%2C%222408.jpg%22%2C%222409.jpg%22%2C%222410.jpg%22%2C%222412.jpg%22%2C%222413.jpg%22%2C%222414.jpg%22%2C%222418.jpg%22%2C%222419.jpg%22%2C%222420.jpg%22%2C%222423.jpg%22%2C%222424.jpg%22%2C%222425.jpg%22%2C%222427.jpg%22%2C%222432.jpg%22%2C%222434.jpg%22%2C%222435.jpg%22%2C%222437.jpg%22%2C%222445.jpg%22%2C%222448.jpg%22%2C%222450.jpg%22%2C%222451.jpg%22%2C%222458.jpg%22%2C%222462.jpg%22%2C%222464.jpg%22%2C%222465.jpg%22%2C%222466.jpg%22%2C%222467.jpg%22%2C%222468.jpg%22%2C%222471.jpg%22%2C%222479.jpg%22%2C%222482.jpg%22%2C%222484.jpg%22%2C%222489.jpg%22%2C%222490.jpg%22%2C%222493.jpg%22%2C%222494.jpg%22%2C%222495.jpg%22%2C%222499.jpg%22%2C%222500.jpg%22%2C%222501.jpg%22%2C%222505.jpg%22%2C%222506.jpg%22%2C%222509.jpg%22%2C%222510.jpg%22%2C%222511.jpg%22%2C%222512.jpg%22%2C%222514.jpg%22%2C%222516.jpg%22%2C%222518.jpg%22%2C%222521.jpg%22%2C%222522.jpg%22%2C%222523.jpg%22%2C%222524.jpg%22%2C%222525.jpg%22%2C%222526.jpg%22%2C%222529.jpg%22%2C%222530.jpg%22%2C%222532.jpg%22%2C%222533.jpg%22%2C%222534.jpg%22%2C%222536.jpg%22%2C%222538.jpg%22%2C%222539.jpg%22%2C%222540.jpg%22%2C%222543.jpg%22%2C%222545.jpg%22%2C%222546.jpg%22%2C%222549.jpg%22%2C%222550.jpg%22%2C%222553.jpg%22%2C%222554.jpg%22%2C%222555.jpg%22%2C%222556.jpg%22%2C%222557.jpg%22%2C%222558.jpg%22%2C%222559.jpg%22%2C%222560.jpg%22%2C%222561.jpg%22%2C%222562.jpg%22%2C%222564.jpg%22%2C%222565.jpg%22%2C%222567.jpg%22%2C%222568.jpg%22%2C%222569.jpg%22%2C%222570.jpg%22%2C%222571.jpg%22%2C%222572.jpg%22%2C%222573.jpg%22%2C%222576.jpg%22%2C%222577.jpg%22%2C%222578.jpg%22%2C%222580.jpg%22%2C%222581.jpg%22%2C%222582.jpg%22%2C%222584.jpg%22%2C%222586.jpg%22%2C%222587.jpg%22%2C%222591.jpg%22%2C%222595.jpg%22%2C%222598.jpg%22%2C%222599.jpg%22%2C%222600.jpg%22%2C%222601.jpg%22%2C%222603.jpg%22%2C%222605.jpg%22%2C%222606.jpg%22%2C%222608.jpg%22%2C%222611.jpg%22%2C%222613.jpg%22%2C%222614.jpg%22%2C%222615.jpg%22%2C%222616.jpg%22%2C%222617.jpg%22%2C%222618.jpg%22%2C%222622.jpg%22%2C%222623.jpg%22%2C%222624.jpg%22%2C%222626.jpg%22%2C%222627.jpg%22%2C%222628.jpg%22%2C%222629.jpg%22%2C%222630.jpg%22%2C%222633.jpg%22%2C%222634.jpg%22%2C%222637.jpg%22%2C%222638.jpg%22%2C%222639.jpg%22%2C%222643.jpg%22%2C%222645.jpg%22%2C%222647.jpg%22%2C%222649.jpg%22%2C%222650.jpg%22%2C%222651.jpg%22%2C%222652.jpg%22%2C%222654.jpg%22%2C%222656.jpg%22%2C%222657.jpg%22%2C%222658.jpg%22%2C%222661.jpg%22%2C%222664.jpg%22%2C%222666.jpg%22%2C%222667.jpg%22%2C%222669.jpg%22%2C%222670.jpg%22%2C%222671.jpg%22%2C%222672.jpg%22%2C%222673.jpg%22%2C%222675.jpg%22%2C%222677.jpg%22%2C%222678.jpg%22%2C%222679.jpg%22%2C%222681.jpg%22%2C%222682.jpg%22%2C%222683.jpg%22%2C%222685.jpg%22%2C%222687.jpg%22%2C%222689.jpg%22%2C%222691.jpg%22%2C%222692.jpg%22%2C%222694.jpg%22%2C%222695.jpg%22%2C%222697.jpg%22%2C%222699.jpg%22%2C%222700.jpg%22%2C%222703.jpg%22%2C%222704.jpg%22%2C%222706.jpg%22%2C%222709.jpg%22%2C%222711.jpg%22%2C%222713.jpg%22%2C%222715.jpg%22%2C%222719.jpg%22%2C%222720.jpg%22%2C%222724.jpg%22%2C%222725.jpg%22%2C%222726.jpg%22%2C%222728.jpg%22%2C%222731.jpg%22%2C%222732.jpg%22%2C%222733.jpg%22%2C%222734.jpg%22%2C%222736.jpg%22%2C%222739.jpg%22%2C%222741.jpg%22%2C%222742.jpg%22%2C%222744.jpg%22%2C%222745.jpg%22%2C%222746.jpg%22%2C%222747.jpg%22%2C%222748.jpg%22%2C%222750.jpg%22%2C%222752.jpg%22%2C%222755.jpg%22%2C%222756.jpg%22%2C%222757.jpg%22%2C%222758.jpg%22%2C%222761.jpg%22%2C%222762.jpg%22%2C%222765.jpg%22%2C%222766.jpg%22%2C%222769.jpg%22%2C%222771.jpg%22%2C%222773.jpg%22%2C%222774.jpg%22%2C%222776.jpg%22%2C%222777.jpg%22%2C%222778.jpg%22%2C%222779.jpg%22%2C%222780.jpg%22%2C%222785.jpg%22%2C%222786.jpg%22%2C%222787.jpg%22%2C%222788.jpg%22%2C%222789.jpg%22%2C%222791.jpg%22%2C%222792.jpg%22%2C%222793.jpg%22%2C%222794.jpg%22%2C%222795.jpg%22%2C%222796.jpg%22%2C%222797.jpg%22%2C%222798.jpg%22%2C%222801.jpg%22%2C%222802.jpg%22%2C%222804.jpg%22%2C%222807.jpg%22%2C%222809.jpg%22%2C%222810.jpg%22%2C%222811.jpg%22%2C%222813.jpg%22%2C%222814.jpg%22%2C%222815.jpg%22%2C%222816.jpg%22%2C%222822.jpg%22%2C%222826.jpg%22%2C%222827.jpg%22%2C%222828.jpg%22%2C%222829.jpg%22%2C%222830.jpg%22%2C%222831.jpg%22%2C%222834.jpg%22%2C%222835.jpg%22%2C%222836.jpg%22%2C%222838.jpg%22%2C%222840.jpg%22%2C%222841.jpg%22%2C%222842.jpg%22%2C%222844.jpg%22%2C%222845.jpg%22%2C%222846.jpg%22%2C%222847.jpg%22%2C%222849.jpg%22%2C%222851.jpg%22%2C%222853.jpg%22%2C%222854.jpg%22%2C%222855.jpg%22%2C%222857.jpg%22%2C%222858.jpg%22%2C%222859.jpg%22%2C%222860.jpg%22%2C%222862.jpg%22%2C%222864.jpg%22%2C%222865.jpg%22%2C%222866.jpg%22%2C%222869.jpg%22%2C%222870.jpg%22%2C%222871.jpg%22%2C%222872.jpg%22%2C%222874.jpg%22%2C%222875.jpg%22%2C%222876.jpg%22%2C%222879.jpg%22%2C%222880.jpg%22%2C%222883.jpg%22%2C%222884.jpg%22%2C%222889.jpg%22%2C%222892.jpg%22%2C%222893.jpg%22%2C%222895.jpg%22%2C%222899.jpg%22%2C%222900.jpg%22%2C%222901.jpg%22%2C%222903.jpg%22%2C%222906.jpg%22%2C%222907.jpg%22%2C%222908.jpg%22%2C%222909.jpg%22%2C%222910.jpg%22%2C%222913.jpg%22%2C%222915.jpg%22%2C%222916.jpg%22%2C%222922.jpg%22%2C%222923.jpg%22%2C%222924.jpg%22%2C%222927.jpg%22%2C%222928.jpg%22%2C%222929.jpg%22%2C%222930.jpg%22%2C%222933.jpg%22%2C%222934.jpg%22%2C%222935.jpg%22%2C%222940.jpg%22%2C%222941.jpg%22%2C%222942.jpg%22%2C%222944.jpg%22%2C%222945.jpg%22%2C%222948.jpg%22%2C%222950.jpg%22%2C%222952.jpg%22%2C%222953.jpg%22%2C%222957.jpg%22%2C%222958.jpg%22%2C%222961.jpg%22%2C%222962.jpg%22%2C%222965.jpg%22%2C%222967.jpg%22%2C%222969.jpg%22%2C%222970.jpg%22%2C%222971.jpg%22%2C%222972.jpg%22%2C%222975.jpg%22%2C%222976.jpg%22%2C%222977.jpg%22%2C%222978.jpg%22%2C%222980.jpg%22%2C%222981.jpg%22%2C%222985.jpg%22%2C%222986.jpg%22%2C%222987.jpg%22%2C%222989.jpg%22%2C%222991.jpg%22%2C%222997.jpg%22%2C%222998.jpg%22%2C%223000.jpg%22%2C%223002.jpg%22%2C%223003.jpg%22%2C%223006.jpg%22%2C%223007.jpg%22%2C%223008.jpg%22%2C%223009.jpg%22%2C%223013.jpg%22%2C%223015.jpg%22%2C%223017.jpg%22%2C%223020.jpg%22%2C%223024.jpg%22%2C%223026.jpg%22%2C%223027.jpg%22%2C%223028.jpg%22%2C%223029.jpg%22%2C%223030.jpg%22%2C%223032.jpg%22%2C%223035.jpg%22%2C%223038.jpg%22%2C%223039.jpg%22%2C%223040.jpg%22%2C%223041.jpg%22%2C%223042.jpg%22%2C%223043.jpg%22%2C%223044.jpg%22%2C%223046.jpg%22%2C%223047.jpg%22%2C%223049.jpg%22%2C%223050.jpg%22%2C%223051.jpg%22%2C%223052.jpg%22%2C%223053.jpg%22%2C%223056.jpg%22%2C%223058.jpg%22%2C%223059.jpg%22%2C%223062.jpg%22%2C%223063.jpg%22%2C%223065.jpg%22%2C%223068.jpg%22%5D",
            filename="train3.zip",
            md5=False,
            checksum="f77a27e23974895bb9ab5bddd4dd3a4d011c76c62ae950cdd1f922a340ee142d",
        ),
        DataLocation(
            url="https://cloud.cs.uni-tuebingen.de/public.php/dav/files/ZZxX65FGnQ8zjBP/Compressed%20Version/images/train/?accept=zip&files=%5B%223069.jpg%22%2C%223072.jpg%22%2C%223073.jpg%22%2C%223074.jpg%22%2C%223076.jpg%22%2C%223080.jpg%22%2C%223081.jpg%22%2C%223082.jpg%22%2C%223084.jpg%22%2C%223085.jpg%22%2C%223086.jpg%22%2C%223087.jpg%22%2C%223091.jpg%22%2C%223092.jpg%22%2C%223094.jpg%22%2C%223095.jpg%22%2C%223096.jpg%22%2C%223098.jpg%22%2C%223100.jpg%22%2C%223101.jpg%22%2C%223103.jpg%22%2C%223104.jpg%22%2C%223105.jpg%22%2C%223107.jpg%22%2C%223108.jpg%22%2C%223109.jpg%22%2C%223110.jpg%22%2C%223112.jpg%22%2C%223113.jpg%22%2C%223114.jpg%22%2C%223116.jpg%22%2C%223117.jpg%22%2C%223120.jpg%22%2C%223121.jpg%22%2C%223122.jpg%22%2C%223125.jpg%22%2C%223127.jpg%22%2C%223128.jpg%22%2C%223129.jpg%22%2C%223130.jpg%22%2C%223133.jpg%22%2C%223134.jpg%22%2C%223139.jpg%22%2C%223140.jpg%22%2C%223142.jpg%22%2C%223143.jpg%22%2C%223144.jpg%22%2C%223146.jpg%22%2C%223147.jpg%22%2C%223148.jpg%22%2C%223149.jpg%22%2C%223151.jpg%22%2C%223152.jpg%22%2C%223156.jpg%22%2C%223160.jpg%22%2C%223162.jpg%22%2C%223164.jpg%22%2C%223165.jpg%22%2C%223166.jpg%22%2C%223167.jpg%22%2C%223170.jpg%22%2C%223171.jpg%22%2C%223173.jpg%22%2C%223175.jpg%22%2C%223176.jpg%22%2C%223177.jpg%22%2C%223178.jpg%22%2C%223179.jpg%22%2C%223182.jpg%22%2C%223183.jpg%22%2C%223184.jpg%22%2C%223185.jpg%22%2C%223186.jpg%22%2C%223187.jpg%22%2C%223188.jpg%22%2C%223190.jpg%22%2C%223191.jpg%22%2C%223192.jpg%22%2C%223193.jpg%22%2C%223194.jpg%22%2C%223196.jpg%22%2C%223197.jpg%22%2C%223199.jpg%22%2C%223200.jpg%22%2C%223202.jpg%22%2C%223204.jpg%22%2C%223205.jpg%22%2C%223206.jpg%22%2C%223208.jpg%22%2C%223212.jpg%22%2C%223213.jpg%22%2C%223214.jpg%22%2C%223217.jpg%22%2C%223218.jpg%22%2C%223219.jpg%22%2C%223221.jpg%22%2C%223223.jpg%22%2C%223224.jpg%22%2C%223225.jpg%22%2C%223226.jpg%22%2C%223227.jpg%22%2C%223229.jpg%22%2C%223230.jpg%22%2C%223235.jpg%22%2C%223236.jpg%22%2C%223240.jpg%22%2C%223241.jpg%22%2C%223244.jpg%22%2C%223245.jpg%22%2C%223246.jpg%22%2C%223248.jpg%22%2C%223249.jpg%22%2C%223250.jpg%22%2C%223251.jpg%22%2C%223252.jpg%22%2C%223256.jpg%22%2C%223258.jpg%22%2C%223260.jpg%22%2C%223261.jpg%22%2C%223262.jpg%22%2C%223263.jpg%22%2C%223265.jpg%22%2C%223267.jpg%22%2C%223268.jpg%22%2C%223270.jpg%22%2C%223272.jpg%22%2C%223274.jpg%22%2C%223275.jpg%22%2C%223276.jpg%22%2C%223278.jpg%22%2C%223280.jpg%22%2C%223281.jpg%22%2C%223282.jpg%22%2C%223283.jpg%22%2C%223285.jpg%22%2C%223286.jpg%22%2C%223288.jpg%22%2C%223289.jpg%22%2C%223291.jpg%22%2C%223292.jpg%22%2C%223294.jpg%22%2C%223295.jpg%22%2C%223296.jpg%22%2C%223297.jpg%22%2C%223298.jpg%22%2C%223301.jpg%22%2C%223302.jpg%22%2C%223303.jpg%22%2C%223308.jpg%22%2C%223309.jpg%22%2C%223311.jpg%22%2C%223313.jpg%22%2C%223314.jpg%22%2C%223315.jpg%22%2C%223317.jpg%22%2C%223319.jpg%22%2C%223320.jpg%22%2C%223321.jpg%22%2C%223323.jpg%22%2C%223324.jpg%22%2C%223325.jpg%22%2C%223326.jpg%22%2C%223327.jpg%22%2C%223330.jpg%22%2C%223331.jpg%22%2C%223332.jpg%22%2C%223333.jpg%22%2C%223334.jpg%22%2C%223335.jpg%22%2C%223338.jpg%22%2C%223339.jpg%22%2C%223340.jpg%22%2C%223341.jpg%22%2C%223342.jpg%22%2C%223343.jpg%22%2C%223344.jpg%22%2C%223345.jpg%22%2C%223346.jpg%22%2C%223347.jpg%22%2C%223349.jpg%22%2C%223352.jpg%22%2C%223353.jpg%22%2C%223354.jpg%22%2C%223356.jpg%22%2C%223357.jpg%22%2C%223358.jpg%22%2C%223361.jpg%22%2C%223362.jpg%22%2C%223363.jpg%22%2C%223364.jpg%22%2C%223366.jpg%22%2C%223367.jpg%22%2C%223368.jpg%22%2C%223370.jpg%22%2C%223371.jpg%22%2C%223374.jpg%22%2C%223376.jpg%22%2C%223378.jpg%22%2C%223380.jpg%22%2C%223381.jpg%22%2C%223382.jpg%22%2C%223383.jpg%22%2C%223385.jpg%22%2C%223386.jpg%22%2C%223388.jpg%22%2C%223389.jpg%22%2C%223390.jpg%22%2C%223391.jpg%22%2C%223392.jpg%22%2C%223393.jpg%22%2C%223394.jpg%22%2C%223395.jpg%22%2C%223396.jpg%22%2C%223397.jpg%22%2C%223398.jpg%22%2C%223399.jpg%22%2C%223400.jpg%22%2C%223401.jpg%22%2C%223402.jpg%22%2C%223403.jpg%22%2C%223404.jpg%22%2C%223405.jpg%22%2C%223406.jpg%22%2C%223407.jpg%22%2C%223408.jpg%22%2C%223409.jpg%22%2C%223410.jpg%22%2C%223411.jpg%22%2C%223412.jpg%22%2C%223413.jpg%22%2C%223414.jpg%22%2C%223415.jpg%22%2C%223416.jpg%22%2C%223417.jpg%22%2C%223418.jpg%22%2C%223419.jpg%22%2C%223420.jpg%22%2C%223421.jpg%22%2C%223422.jpg%22%2C%223423.jpg%22%2C%223424.jpg%22%2C%223425.jpg%22%2C%223426.jpg%22%2C%223427.jpg%22%2C%223428.jpg%22%2C%223429.jpg%22%2C%223430.jpg%22%2C%223431.jpg%22%2C%223432.jpg%22%2C%223433.jpg%22%2C%223434.jpg%22%2C%223435.jpg%22%2C%223436.jpg%22%2C%223437.jpg%22%2C%223438.jpg%22%2C%223439.jpg%22%2C%223440.jpg%22%2C%223441.jpg%22%2C%223442.jpg%22%2C%223443.jpg%22%2C%223444.jpg%22%2C%223445.jpg%22%2C%223446.jpg%22%2C%223447.jpg%22%2C%223448.jpg%22%2C%223449.jpg%22%2C%223450.jpg%22%2C%223451.jpg%22%2C%223452.jpg%22%2C%223453.jpg%22%2C%223454.jpg%22%2C%223455.jpg%22%2C%223456.jpg%22%2C%223457.jpg%22%2C%223458.jpg%22%2C%223459.jpg%22%2C%223460.jpg%22%2C%223461.jpg%22%2C%223462.jpg%22%2C%223463.jpg%22%2C%223619.jpg%22%2C%223620.jpg%22%2C%223621.jpg%22%2C%223622.jpg%22%2C%223623.jpg%22%2C%223624.jpg%22%2C%223625.jpg%22%2C%223626.jpg%22%2C%223627.jpg%22%2C%223628.jpg%22%2C%223629.jpg%22%2C%223630.jpg%22%2C%223631.jpg%22%2C%223632.jpg%22%2C%223633.jpg%22%2C%223634.jpg%22%2C%223635.jpg%22%2C%223636.jpg%22%2C%223637.jpg%22%2C%223638.jpg%22%2C%223639.jpg%22%2C%223640.jpg%22%2C%223641.jpg%22%2C%223642.jpg%22%2C%223643.jpg%22%2C%223644.jpg%22%2C%223645.jpg%22%2C%223646.jpg%22%2C%223647.jpg%22%2C%223648.jpg%22%2C%223649.jpg%22%2C%223650.jpg%22%2C%223651.jpg%22%2C%223652.jpg%22%2C%223676.jpg%22%2C%223677.jpg%22%2C%223678.jpg%22%2C%223679.jpg%22%2C%223680.jpg%22%2C%223681.jpg%22%2C%223682.jpg%22%2C%223683.jpg%22%2C%223684.jpg%22%2C%223685.jpg%22%2C%223686.jpg%22%2C%223687.jpg%22%2C%223688.jpg%22%2C%223689.jpg%22%2C%223690.jpg%22%2C%223691.jpg%22%2C%223692.jpg%22%2C%223693.jpg%22%2C%223694.jpg%22%2C%223695.jpg%22%2C%223696.jpg%22%2C%223697.jpg%22%2C%223698.jpg%22%2C%223699.jpg%22%2C%223700.jpg%22%2C%223701.jpg%22%2C%223702.jpg%22%2C%223703.jpg%22%2C%223704.jpg%22%2C%223705.jpg%22%2C%223706.jpg%22%2C%223707.jpg%22%2C%223708.jpg%22%2C%223709.jpg%22%2C%223710.jpg%22%2C%223711.jpg%22%2C%223712.jpg%22%2C%223713.jpg%22%2C%223714.jpg%22%2C%223715.jpg%22%2C%223758.jpg%22%2C%223759.jpg%22%2C%223760.jpg%22%2C%223761.jpg%22%2C%223762.jpg%22%2C%223763.jpg%22%2C%223764.jpg%22%2C%223765.jpg%22%2C%223766.jpg%22%2C%223767.jpg%22%2C%223768.jpg%22%2C%223769.jpg%22%2C%223770.jpg%22%2C%223771.jpg%22%2C%223772.jpg%22%2C%223773.jpg%22%2C%223774.jpg%22%2C%223775.jpg%22%2C%223776.jpg%22%2C%223777.jpg%22%2C%223778.jpg%22%2C%223779.jpg%22%2C%223780.jpg%22%2C%223781.jpg%22%2C%223782.jpg%22%2C%223783.jpg%22%2C%223784.jpg%22%2C%223785.jpg%22%2C%223786.jpg%22%2C%223787.jpg%22%2C%223788.jpg%22%2C%223789.jpg%22%2C%223790.jpg%22%2C%223791.jpg%22%2C%223792.jpg%22%2C%223793.jpg%22%2C%223794.jpg%22%2C%223795.jpg%22%2C%223796.jpg%22%2C%223797.jpg%22%2C%223798.jpg%22%2C%223799.jpg%22%2C%223800.jpg%22%2C%223801.jpg%22%2C%223802.jpg%22%2C%223803.jpg%22%2C%223804.jpg%22%2C%223805.jpg%22%2C%223806.jpg%22%2C%223807.jpg%22%2C%223808.jpg%22%2C%223809.jpg%22%2C%223810.jpg%22%2C%223811.jpg%22%2C%223812.jpg%22%2C%223813.jpg%22%2C%223814.jpg%22%2C%223815.jpg%22%2C%223816.jpg%22%2C%223817.jpg%22%2C%223818.jpg%22%2C%223819.jpg%22%2C%223820.jpg%22%2C%223821.jpg%22%2C%223822.jpg%22%2C%223823.jpg%22%2C%223824.jpg%22%2C%223825.jpg%22%2C%223826.jpg%22%2C%223827.jpg%22%2C%223828.jpg%22%2C%223829.jpg%22%2C%223830.jpg%22%2C%223831.jpg%22%2C%223832.jpg%22%2C%223833.jpg%22%2C%223834.jpg%22%2C%223835.jpg%22%2C%223836.jpg%22%2C%223837.jpg%22%2C%223838.jpg%22%2C%223839.jpg%22%2C%223840.jpg%22%2C%223841.jpg%22%2C%223842.jpg%22%2C%223843.jpg%22%2C%223844.jpg%22%2C%223845.jpg%22%2C%223846.jpg%22%2C%223847.jpg%22%2C%223848.jpg%22%2C%223849.jpg%22%2C%223850.jpg%22%2C%223851.jpg%22%2C%223852.jpg%22%2C%223853.jpg%22%5D",
            filename="train4.zip",
            md5=False,
            checksum="fcc2d9269869d0263aebb05ceef4e518e028a7d98e763201a99fc8ae7b503323",
        ),
        DataLocation(
            url="https://cloud.cs.uni-tuebingen.de/public.php/dav/files/ZZxX65FGnQ8zjBP/Compressed%20Version/images/train/?accept=zip&files=%5B%223854.jpg%22%2C%223855.jpg%22%2C%223856.jpg%22%2C%223857.jpg%22%2C%223858.jpg%22%2C%223859.jpg%22%2C%223860.jpg%22%2C%223861.jpg%22%2C%223862.jpg%22%2C%223863.jpg%22%2C%223864.jpg%22%2C%223865.jpg%22%2C%223866.jpg%22%2C%223867.jpg%22%2C%223868.jpg%22%2C%223869.jpg%22%2C%223870.jpg%22%2C%223871.jpg%22%2C%223872.jpg%22%2C%223873.jpg%22%2C%223874.jpg%22%2C%223875.jpg%22%2C%223876.jpg%22%2C%223877.jpg%22%2C%223878.jpg%22%2C%223879.jpg%22%2C%223880.jpg%22%2C%223881.jpg%22%2C%223882.jpg%22%2C%223883.jpg%22%2C%223884.jpg%22%2C%223885.jpg%22%2C%223886.jpg%22%2C%223887.jpg%22%2C%223888.jpg%22%2C%223889.jpg%22%2C%223890.jpg%22%2C%223891.jpg%22%2C%223892.jpg%22%2C%223893.jpg%22%2C%223894.jpg%22%2C%223895.jpg%22%2C%223896.jpg%22%2C%223897.jpg%22%2C%223898.jpg%22%2C%223899.jpg%22%2C%223900.jpg%22%2C%223901.jpg%22%2C%223902.jpg%22%2C%223903.jpg%22%2C%223904.jpg%22%2C%223905.jpg%22%2C%223906.jpg%22%2C%223907.jpg%22%2C%223908.jpg%22%2C%223909.jpg%22%2C%223910.jpg%22%2C%223911.jpg%22%2C%223912.jpg%22%2C%223913.jpg%22%2C%223914.jpg%22%2C%223915.jpg%22%2C%223916.jpg%22%2C%223917.jpg%22%2C%223918.jpg%22%2C%223919.jpg%22%2C%223920.jpg%22%2C%223921.jpg%22%2C%223922.jpg%22%2C%223923.jpg%22%2C%223924.jpg%22%2C%223925.jpg%22%2C%223926.jpg%22%2C%223927.jpg%22%2C%223928.jpg%22%2C%223929.jpg%22%2C%223930.jpg%22%2C%223931.jpg%22%2C%223932.jpg%22%2C%223933.jpg%22%2C%223934.jpg%22%2C%223935.jpg%22%2C%223936.jpg%22%2C%223937.jpg%22%2C%223938.jpg%22%2C%223939.jpg%22%2C%223940.jpg%22%2C%223941.jpg%22%2C%223942.jpg%22%2C%223943.jpg%22%2C%223944.jpg%22%2C%223945.jpg%22%2C%223946.jpg%22%2C%223947.jpg%22%2C%223948.jpg%22%2C%223949.jpg%22%2C%223950.jpg%22%2C%223951.jpg%22%2C%223952.jpg%22%2C%223953.jpg%22%2C%223954.jpg%22%2C%223955.jpg%22%2C%223956.jpg%22%2C%223957.jpg%22%2C%223958.jpg%22%2C%223959.jpg%22%2C%223960.jpg%22%2C%223961.jpg%22%2C%223962.jpg%22%2C%223963.jpg%22%2C%223964.jpg%22%2C%223965.jpg%22%2C%223966.jpg%22%2C%223967.jpg%22%2C%223968.jpg%22%2C%223969.jpg%22%2C%223970.jpg%22%2C%223971.jpg%22%2C%223972.jpg%22%2C%223973.jpg%22%2C%223974.jpg%22%2C%223975.jpg%22%2C%223976.jpg%22%2C%223977.jpg%22%2C%223978.jpg%22%2C%223979.jpg%22%2C%223980.jpg%22%2C%223981.jpg%22%2C%223982.jpg%22%2C%223983.jpg%22%2C%223984.jpg%22%2C%223985.jpg%22%2C%223986.jpg%22%2C%223987.jpg%22%2C%223988.jpg%22%2C%223989.jpg%22%2C%223990.jpg%22%2C%223991.jpg%22%2C%223992.jpg%22%2C%223993.jpg%22%2C%223994.jpg%22%2C%223995.jpg%22%2C%223996.jpg%22%2C%223997.jpg%22%2C%223998.jpg%22%2C%223999.jpg%22%2C%224000.jpg%22%2C%224001.jpg%22%2C%224002.jpg%22%2C%224003.jpg%22%2C%224004.jpg%22%2C%224005.jpg%22%2C%224006.jpg%22%2C%224007.jpg%22%2C%224008.jpg%22%2C%224009.jpg%22%2C%224010.jpg%22%2C%224011.jpg%22%2C%224012.jpg%22%2C%224197.jpg%22%2C%224198.jpg%22%2C%224199.jpg%22%2C%224200.jpg%22%2C%224201.jpg%22%2C%224202.jpg%22%2C%224203.jpg%22%2C%224204.jpg%22%2C%224205.jpg%22%2C%224206.jpg%22%2C%224207.jpg%22%2C%224208.jpg%22%2C%224209.jpg%22%2C%224210.jpg%22%2C%224211.jpg%22%2C%224212.jpg%22%2C%224213.jpg%22%2C%224214.jpg%22%2C%224215.jpg%22%2C%224216.jpg%22%2C%224217.jpg%22%2C%224218.jpg%22%2C%224219.jpg%22%2C%224220.jpg%22%2C%224221.jpg%22%2C%224222.jpg%22%2C%224223.jpg%22%2C%224224.jpg%22%2C%224225.jpg%22%2C%224226.jpg%22%2C%224227.jpg%22%2C%224228.jpg%22%2C%224229.jpg%22%2C%224230.jpg%22%2C%224231.jpg%22%2C%224232.jpg%22%2C%224233.jpg%22%2C%224234.jpg%22%2C%224235.jpg%22%2C%224236.jpg%22%2C%224264.jpg%22%2C%224265.jpg%22%2C%224266.jpg%22%2C%224267.jpg%22%2C%224268.jpg%22%2C%224269.jpg%22%2C%224270.jpg%22%2C%224271.jpg%22%2C%224272.jpg%22%2C%224273.jpg%22%2C%224274.jpg%22%2C%224275.jpg%22%2C%224276.jpg%22%2C%224277.jpg%22%2C%224278.jpg%22%2C%224279.jpg%22%2C%224280.jpg%22%2C%224281.jpg%22%2C%224282.jpg%22%2C%224283.jpg%22%2C%224284.jpg%22%2C%224285.jpg%22%2C%224286.jpg%22%2C%224287.jpg%22%2C%224288.jpg%22%2C%224289.jpg%22%2C%224290.jpg%22%2C%224291.jpg%22%2C%224292.jpg%22%2C%224293.jpg%22%2C%224294.jpg%22%2C%224295.jpg%22%2C%224296.jpg%22%2C%224297.jpg%22%2C%224298.jpg%22%2C%224299.jpg%22%2C%224300.jpg%22%2C%224301.jpg%22%2C%224302.jpg%22%2C%224303.jpg%22%2C%224304.jpg%22%2C%224305.jpg%22%2C%224306.jpg%22%2C%224307.jpg%22%2C%224308.jpg%22%2C%224309.jpg%22%2C%224310.jpg%22%2C%224311.jpg%22%2C%224312.jpg%22%2C%224313.jpg%22%2C%224314.jpg%22%2C%224315.jpg%22%2C%224316.jpg%22%2C%224317.jpg%22%2C%224318.jpg%22%2C%224319.jpg%22%2C%224320.jpg%22%2C%224321.jpg%22%2C%224322.jpg%22%2C%224323.jpg%22%2C%224324.jpg%22%2C%224325.jpg%22%2C%224326.jpg%22%2C%224327.jpg%22%2C%224328.jpg%22%2C%224329.jpg%22%2C%224330.jpg%22%2C%224396.jpg%22%2C%224397.jpg%22%2C%224398.jpg%22%2C%224399.jpg%22%2C%224400.jpg%22%2C%224401.jpg%22%2C%224402.jpg%22%2C%224403.jpg%22%2C%224404.jpg%22%2C%224405.jpg%22%2C%224406.jpg%22%2C%224407.jpg%22%2C%224408.jpg%22%2C%224409.jpg%22%2C%224410.jpg%22%2C%224411.jpg%22%2C%224412.jpg%22%2C%224413.jpg%22%2C%224414.jpg%22%2C%224415.jpg%22%2C%224416.jpg%22%2C%224417.jpg%22%2C%224418.jpg%22%2C%224419.jpg%22%2C%224420.jpg%22%2C%224421.jpg%22%2C%224422.jpg%22%2C%224423.jpg%22%2C%224424.jpg%22%2C%224425.jpg%22%2C%224426.jpg%22%2C%224427.jpg%22%2C%224428.jpg%22%2C%224429.jpg%22%2C%224430.jpg%22%2C%224431.jpg%22%2C%224432.jpg%22%2C%224433.jpg%22%2C%224434.jpg%22%2C%224435.jpg%22%2C%224463.jpg%22%2C%224464.jpg%22%2C%224465.jpg%22%2C%224466.jpg%22%2C%224467.jpg%22%2C%224468.jpg%22%2C%224469.jpg%22%2C%224470.jpg%22%2C%224471.jpg%22%2C%224472.jpg%22%2C%224473.jpg%22%2C%224474.jpg%22%2C%224475.jpg%22%2C%224476.jpg%22%2C%224477.jpg%22%2C%224478.jpg%22%2C%224479.jpg%22%2C%224480.jpg%22%2C%224481.jpg%22%2C%224509.jpg%22%2C%224510.jpg%22%2C%224511.jpg%22%2C%224512.jpg%22%2C%224513.jpg%22%2C%224514.jpg%22%2C%224515.jpg%22%2C%224516.jpg%22%2C%224517.jpg%22%2C%224518.jpg%22%2C%224519.jpg%22%2C%224520.jpg%22%2C%224521.jpg%22%2C%224522.jpg%22%2C%224523.jpg%22%2C%224524.jpg%22%2C%224525.jpg%22%2C%224526.jpg%22%2C%224527.jpg%22%2C%224528.jpg%22%2C%224529.jpg%22%2C%224530.jpg%22%2C%224531.jpg%22%2C%224532.jpg%22%2C%224533.jpg%22%2C%224534.jpg%22%2C%224535.jpg%22%2C%224536.jpg%22%2C%224537.jpg%22%2C%224538.jpg%22%2C%224539.jpg%22%2C%224540.jpg%22%2C%224541.jpg%22%2C%224542.jpg%22%2C%224543.jpg%22%2C%224544.jpg%22%2C%224545.jpg%22%2C%224546.jpg%22%2C%224547.jpg%22%2C%224548.jpg%22%2C%224549.jpg%22%2C%224550.jpg%22%2C%224551.jpg%22%2C%224552.jpg%22%2C%224553.jpg%22%2C%224554.jpg%22%2C%224555.jpg%22%2C%224556.jpg%22%2C%224557.jpg%22%2C%224558.jpg%22%2C%224559.jpg%22%2C%224560.jpg%22%2C%224561.jpg%22%2C%224562.jpg%22%2C%224563.jpg%22%2C%224564.jpg%22%2C%224565.jpg%22%2C%224566.jpg%22%2C%224567.jpg%22%2C%224568.jpg%22%2C%224569.jpg%22%2C%224570.jpg%22%2C%224571.jpg%22%2C%224572.jpg%22%2C%224573.jpg%22%2C%224574.jpg%22%2C%224575.jpg%22%2C%224576.jpg%22%2C%224577.jpg%22%2C%224578.jpg%22%2C%224579.jpg%22%2C%224580.jpg%22%2C%224581.jpg%22%2C%224582.jpg%22%2C%224583.jpg%22%2C%224584.jpg%22%2C%224585.jpg%22%2C%224586.jpg%22%2C%224587.jpg%22%2C%224588.jpg%22%2C%224589.jpg%22%2C%224590.jpg%22%2C%224591.jpg%22%2C%224592.jpg%22%2C%224593.jpg%22%2C%224594.jpg%22%2C%224595.jpg%22%2C%224596.jpg%22%2C%224597.jpg%22%2C%224598.jpg%22%2C%224599.jpg%22%2C%224600.jpg%22%2C%224601.jpg%22%2C%224602.jpg%22%2C%224603.jpg%22%2C%224604.jpg%22%2C%224605.jpg%22%2C%224606.jpg%22%2C%224607.jpg%22%2C%224608.jpg%22%2C%224609.jpg%22%2C%224610.jpg%22%2C%224611.jpg%22%2C%224612.jpg%22%2C%224613.jpg%22%2C%224614.jpg%22%2C%224615.jpg%22%2C%224616.jpg%22%2C%224617.jpg%22%2C%224618.jpg%22%2C%224619.jpg%22%2C%224620.jpg%22%2C%224621.jpg%22%2C%224622.jpg%22%2C%224623.jpg%22%2C%224624.jpg%22%2C%224625.jpg%22%2C%224626.jpg%22%2C%224627.jpg%22%2C%224628.jpg%22%2C%224629.jpg%22%2C%224630.jpg%22%2C%224631.jpg%22%2C%224632.jpg%22%2C%224633.jpg%22%5D",
            filename="train5.zip",
            md5=False,
            checksum="928301132f2a503eea3f8528e6395329ca3c4face43774d53519b6c322c23228",
        ),
        DataLocation(
            url="https://cloud.cs.uni-tuebingen.de/public.php/dav/files/ZZxX65FGnQ8zjBP/Compressed%20Version/images/train/?accept=zip&files=%5B%224634.jpg%22%2C%224635.jpg%22%2C%224636.jpg%22%2C%224637.jpg%22%2C%224638.jpg%22%2C%224639.jpg%22%2C%224640.jpg%22%2C%224641.jpg%22%2C%224642.jpg%22%2C%224643.jpg%22%2C%224644.jpg%22%2C%224645.jpg%22%2C%224646.jpg%22%2C%224647.jpg%22%2C%224648.jpg%22%2C%224649.jpg%22%2C%224650.jpg%22%2C%224651.jpg%22%2C%224652.jpg%22%2C%224653.jpg%22%2C%224654.jpg%22%2C%224655.jpg%22%2C%224656.jpg%22%2C%224657.jpg%22%2C%224658.jpg%22%2C%224659.jpg%22%2C%224660.jpg%22%2C%224661.jpg%22%2C%224662.jpg%22%2C%224663.jpg%22%2C%224664.jpg%22%2C%224665.jpg%22%2C%224666.jpg%22%2C%224667.jpg%22%2C%224668.jpg%22%2C%224669.jpg%22%2C%224670.jpg%22%2C%224671.jpg%22%2C%224672.jpg%22%2C%224673.jpg%22%2C%224674.jpg%22%2C%224675.jpg%22%2C%224676.jpg%22%2C%224677.jpg%22%2C%224678.jpg%22%2C%224679.jpg%22%2C%224680.jpg%22%2C%224681.jpg%22%2C%224682.jpg%22%2C%224683.jpg%22%2C%224684.jpg%22%2C%224685.jpg%22%2C%224686.jpg%22%2C%224687.jpg%22%2C%224688.jpg%22%2C%224689.jpg%22%2C%224690.jpg%22%2C%224691.jpg%22%2C%224692.jpg%22%2C%224693.jpg%22%2C%224694.jpg%22%2C%224695.jpg%22%2C%224696.jpg%22%2C%224697.jpg%22%2C%224698.jpg%22%2C%224699.jpg%22%2C%224700.jpg%22%2C%224701.jpg%22%2C%224702.jpg%22%2C%224703.jpg%22%2C%224704.jpg%22%2C%224705.jpg%22%2C%224706.jpg%22%2C%224707.jpg%22%2C%224708.jpg%22%2C%224709.jpg%22%2C%224710.jpg%22%2C%224711.jpg%22%2C%224712.jpg%22%2C%224713.jpg%22%2C%224714.jpg%22%2C%224715.jpg%22%2C%224716.jpg%22%2C%224717.jpg%22%2C%224718.jpg%22%2C%224719.jpg%22%2C%224720.jpg%22%2C%224721.jpg%22%2C%224722.jpg%22%2C%224723.jpg%22%2C%224724.jpg%22%2C%224725.jpg%22%2C%224726.jpg%22%2C%224727.jpg%22%2C%224728.jpg%22%2C%224729.jpg%22%2C%224730.jpg%22%2C%224731.jpg%22%2C%224732.jpg%22%2C%224733.jpg%22%2C%224734.jpg%22%2C%224735.jpg%22%2C%224736.jpg%22%2C%224737.jpg%22%2C%224738.jpg%22%2C%224739.jpg%22%2C%224740.jpg%22%2C%224741.jpg%22%2C%224742.jpg%22%2C%224743.jpg%22%2C%224744.jpg%22%2C%224745.jpg%22%2C%224746.jpg%22%2C%224747.jpg%22%2C%224748.jpg%22%2C%224749.jpg%22%2C%224750.jpg%22%2C%224751.jpg%22%2C%224752.jpg%22%2C%224753.jpg%22%2C%224754.jpg%22%2C%224755.jpg%22%2C%224756.jpg%22%2C%224757.jpg%22%2C%224758.jpg%22%2C%224759.jpg%22%2C%224760.jpg%22%2C%224761.jpg%22%2C%224762.jpg%22%2C%224763.jpg%22%2C%224764.jpg%22%2C%224765.jpg%22%2C%224766.jpg%22%2C%224767.jpg%22%2C%224768.jpg%22%2C%224769.jpg%22%2C%224770.jpg%22%2C%224771.jpg%22%2C%224772.jpg%22%2C%224773.jpg%22%2C%224774.jpg%22%2C%224775.jpg%22%2C%224776.jpg%22%2C%224777.jpg%22%2C%224778.jpg%22%2C%224779.jpg%22%2C%224780.jpg%22%2C%224781.jpg%22%2C%224782.jpg%22%2C%224783.jpg%22%2C%224784.jpg%22%2C%224785.jpg%22%2C%224786.jpg%22%2C%224787.jpg%22%2C%224789.jpg%22%2C%224790.jpg%22%2C%224793.jpg%22%2C%224794.jpg%22%2C%224795.jpg%22%2C%224796.jpg%22%2C%224797.jpg%22%2C%224798.jpg%22%2C%224799.jpg%22%2C%224800.jpg%22%2C%224801.jpg%22%2C%224802.jpg%22%2C%224803.jpg%22%2C%224804.jpg%22%2C%224805.jpg%22%2C%224806.jpg%22%2C%224807.jpg%22%2C%224808.jpg%22%2C%224809.jpg%22%2C%224810.jpg%22%2C%224811.jpg%22%2C%224812.jpg%22%2C%224813.jpg%22%2C%224814.jpg%22%2C%224815.jpg%22%2C%224816.jpg%22%2C%224817.jpg%22%2C%224818.jpg%22%2C%224819.jpg%22%2C%224820.jpg%22%2C%224821.jpg%22%2C%224822.jpg%22%2C%224823.jpg%22%2C%224824.jpg%22%2C%224825.jpg%22%2C%224826.jpg%22%2C%224827.jpg%22%2C%224828.jpg%22%2C%224829.jpg%22%2C%224830.jpg%22%2C%224831.jpg%22%2C%224832.jpg%22%2C%224833.jpg%22%2C%224834.jpg%22%2C%224835.jpg%22%2C%224836.jpg%22%2C%224837.jpg%22%2C%224838.jpg%22%2C%224839.jpg%22%2C%224840.jpg%22%2C%224841.jpg%22%2C%224842.jpg%22%2C%224843.jpg%22%2C%224844.jpg%22%2C%224845.jpg%22%2C%224846.jpg%22%2C%224847.jpg%22%2C%224848.jpg%22%2C%224849.jpg%22%2C%224850.jpg%22%2C%224851.jpg%22%2C%224852.jpg%22%2C%224853.jpg%22%2C%224854.jpg%22%2C%224855.jpg%22%2C%224856.jpg%22%2C%224857.jpg%22%2C%224858.jpg%22%2C%224859.jpg%22%2C%224860.jpg%22%2C%224861.jpg%22%2C%224862.jpg%22%2C%224863.jpg%22%2C%224864.jpg%22%2C%225106.jpg%22%2C%225107.jpg%22%2C%225108.jpg%22%2C%225109.jpg%22%2C%225110.jpg%22%2C%225111.jpg%22%2C%225112.jpg%22%2C%225113.jpg%22%2C%225114.jpg%22%2C%225115.jpg%22%2C%225116.jpg%22%2C%225117.jpg%22%2C%225118.jpg%22%2C%225119.jpg%22%2C%225120.jpg%22%2C%225121.jpg%22%2C%225122.jpg%22%2C%225123.jpg%22%2C%225124.jpg%22%2C%225125.jpg%22%2C%225126.jpg%22%2C%225127.jpg%22%2C%225128.jpg%22%2C%225129.jpg%22%2C%225130.jpg%22%2C%225131.jpg%22%2C%225132.jpg%22%2C%225133.jpg%22%2C%225134.jpg%22%2C%225135.jpg%22%2C%225136.jpg%22%2C%225137.jpg%22%2C%225138.jpg%22%2C%225139.jpg%22%2C%225140.jpg%22%2C%225141.jpg%22%2C%225142.jpg%22%2C%225143.jpg%22%2C%225144.jpg%22%2C%225145.jpg%22%2C%225173.jpg%22%2C%225174.jpg%22%2C%225175.jpg%22%2C%225176.jpg%22%2C%225177.jpg%22%2C%225178.jpg%22%2C%225179.jpg%22%2C%225180.jpg%22%2C%225181.jpg%22%2C%225182.jpg%22%2C%225183.jpg%22%2C%225184.jpg%22%2C%225185.jpg%22%2C%225186.jpg%22%2C%225187.jpg%22%2C%225188.jpg%22%2C%225189.jpg%22%2C%225190.jpg%22%2C%225191.jpg%22%2C%225192.jpg%22%2C%225193.jpg%22%2C%225194.jpg%22%2C%225195.jpg%22%2C%225196.jpg%22%2C%225197.jpg%22%2C%225198.jpg%22%2C%225199.jpg%22%2C%225200.jpg%22%2C%225201.jpg%22%2C%225202.jpg%22%2C%225203.jpg%22%2C%225204.jpg%22%2C%225205.jpg%22%2C%225206.jpg%22%2C%225207.jpg%22%2C%225208.jpg%22%2C%225209.jpg%22%2C%225210.jpg%22%2C%225211.jpg%22%2C%225212.jpg%22%2C%225213.jpg%22%2C%225214.jpg%22%2C%225215.jpg%22%2C%225216.jpg%22%2C%225217.jpg%22%2C%225218.jpg%22%2C%225219.jpg%22%2C%225220.jpg%22%2C%225221.jpg%22%2C%225222.jpg%22%2C%225223.jpg%22%2C%225224.jpg%22%2C%225225.jpg%22%2C%225226.jpg%22%2C%225227.jpg%22%2C%225228.jpg%22%2C%225229.jpg%22%2C%225230.jpg%22%2C%225231.jpg%22%2C%225232.jpg%22%2C%225233.jpg%22%2C%225234.jpg%22%2C%225235.jpg%22%2C%225236.jpg%22%2C%225237.jpg%22%2C%225238.jpg%22%2C%225239.jpg%22%2C%225240.jpg%22%2C%225241.jpg%22%2C%225242.jpg%22%2C%225243.jpg%22%2C%225244.jpg%22%2C%225245.jpg%22%2C%225246.jpg%22%2C%225247.jpg%22%2C%225248.jpg%22%2C%225249.jpg%22%2C%225250.jpg%22%2C%225251.jpg%22%2C%225252.jpg%22%2C%225253.jpg%22%2C%225254.jpg%22%2C%225255.jpg%22%2C%225268.jpg%22%2C%225269.jpg%22%2C%225270.jpg%22%2C%225271.jpg%22%2C%225272.jpg%22%2C%225273.jpg%22%2C%225274.jpg%22%2C%225275.jpg%22%2C%225276.jpg%22%2C%225277.jpg%22%2C%225278.jpg%22%2C%225279.jpg%22%2C%225280.jpg%22%2C%225281.jpg%22%2C%225282.jpg%22%2C%225283.jpg%22%2C%225284.jpg%22%2C%225285.jpg%22%2C%225286.jpg%22%2C%225287.jpg%22%2C%225288.jpg%22%2C%225289.jpg%22%2C%225290.jpg%22%2C%225291.jpg%22%2C%225292.jpg%22%2C%225293.jpg%22%2C%225294.jpg%22%2C%225295.jpg%22%2C%225296.jpg%22%2C%225297.jpg%22%2C%225298.jpg%22%2C%225299.jpg%22%2C%225300.jpg%22%2C%225301.jpg%22%2C%225302.jpg%22%2C%225303.jpg%22%2C%225304.jpg%22%2C%225305.jpg%22%2C%225306.jpg%22%2C%225307.jpg%22%2C%225335.jpg%22%2C%225336.jpg%22%2C%225337.jpg%22%2C%225338.jpg%22%2C%225339.jpg%22%2C%225340.jpg%22%2C%225341.jpg%22%2C%225342.jpg%22%2C%225343.jpg%22%2C%225344.jpg%22%2C%225345.jpg%22%2C%225346.jpg%22%2C%225347.jpg%22%2C%225348.jpg%22%2C%225349.jpg%22%2C%225350.jpg%22%2C%225351.jpg%22%2C%225352.jpg%22%2C%225353.jpg%22%2C%225354.jpg%22%2C%225355.jpg%22%2C%225356.jpg%22%2C%225357.jpg%22%2C%225358.jpg%22%2C%225359.jpg%22%2C%225360.jpg%22%2C%225361.jpg%22%2C%225362.jpg%22%2C%225363.jpg%22%2C%225364.jpg%22%2C%225365.jpg%22%2C%225366.jpg%22%2C%225367.jpg%22%2C%225368.jpg%22%2C%225369.jpg%22%2C%225370.jpg%22%2C%225371.jpg%22%2C%225372.jpg%22%2C%225373.jpg%22%2C%225374.jpg%22%2C%225375.jpg%22%2C%225376.jpg%22%2C%225377.jpg%22%2C%225378.jpg%22%2C%225379.jpg%22%2C%225380.jpg%22%2C%225381.jpg%22%2C%225382.jpg%22%2C%225383.jpg%22%2C%225384.jpg%22%2C%225385.jpg%22%2C%225386.jpg%22%2C%225387.jpg%22%2C%225388.jpg%22%2C%225389.jpg%22%2C%225390.jpg%22%2C%225391.jpg%22%2C%225392.jpg%22%2C%225393.jpg%22%5D",
            filename="train6.zip",
            md5=False,
            checksum="aca75dc54ee3c4cced9c31e703764360aedf3ede2fc9e01a4e6c5023021e0b7d",
        ),
        DataLocation(
            url="https://cloud.cs.uni-tuebingen.de/public.php/dav/files/ZZxX65FGnQ8zjBP/Compressed%20Version/images/train/?accept=zip&files=%5B%225394.jpg%22%2C%225395.jpg%22%2C%225396.jpg%22%2C%225397.jpg%22%2C%225398.jpg%22%2C%225399.jpg%22%2C%225400.jpg%22%2C%225401.jpg%22%2C%225402.jpg%22%2C%225403.jpg%22%2C%225404.jpg%22%2C%225405.jpg%22%2C%225406.jpg%22%2C%225407.jpg%22%2C%225408.jpg%22%2C%225409.jpg%22%2C%225410.jpg%22%2C%225411.jpg%22%2C%225412.jpg%22%2C%225413.jpg%22%2C%225414.jpg%22%2C%225415.jpg%22%2C%225416.jpg%22%2C%225417.jpg%22%2C%225418.jpg%22%2C%225419.jpg%22%2C%225420.jpg%22%2C%225421.jpg%22%2C%225422.jpg%22%2C%225423.jpg%22%2C%225424.jpg%22%2C%225425.jpg%22%2C%225426.jpg%22%2C%225427.jpg%22%2C%225428.jpg%22%2C%225429.jpg%22%2C%225430.jpg%22%2C%225431.jpg%22%2C%225432.jpg%22%2C%225433.jpg%22%2C%225434.jpg%22%2C%225435.jpg%22%2C%225436.jpg%22%2C%225437.jpg%22%2C%225438.jpg%22%2C%225439.jpg%22%2C%225440.jpg%22%2C%225441.jpg%22%2C%225442.jpg%22%2C%225443.jpg%22%2C%225444.jpg%22%2C%225445.jpg%22%2C%225446.jpg%22%2C%225447.jpg%22%2C%225448.jpg%22%2C%225449.jpg%22%2C%225450.jpg%22%2C%225451.jpg%22%2C%225452.jpg%22%2C%225453.jpg%22%2C%225454.jpg%22%2C%225455.jpg%22%2C%225456.jpg%22%2C%225457.jpg%22%2C%225458.jpg%22%2C%225459.jpg%22%2C%225460.jpg%22%2C%225461.jpg%22%2C%225462.jpg%22%2C%225463.jpg%22%2C%225464.jpg%22%2C%225465.jpg%22%2C%225466.jpg%22%2C%225467.jpg%22%2C%225468.jpg%22%2C%225469.jpg%22%2C%225470.jpg%22%2C%225471.jpg%22%2C%225472.jpg%22%2C%225473.jpg%22%2C%225474.jpg%22%2C%225475.jpg%22%2C%225476.jpg%22%2C%225477.jpg%22%2C%225478.jpg%22%2C%225479.jpg%22%2C%225480.jpg%22%2C%225481.jpg%22%2C%225482.jpg%22%2C%225483.jpg%22%2C%225484.jpg%22%2C%225485.jpg%22%2C%225486.jpg%22%2C%225487.jpg%22%2C%225488.jpg%22%2C%225489.jpg%22%2C%225490.jpg%22%2C%225491.jpg%22%2C%225492.jpg%22%2C%225493.jpg%22%2C%225494.jpg%22%2C%225495.jpg%22%2C%225496.jpg%22%2C%225497.jpg%22%2C%225498.jpg%22%2C%225499.jpg%22%2C%225500.jpg%22%2C%225501.jpg%22%2C%225502.jpg%22%2C%225503.jpg%22%2C%225504.jpg%22%2C%225505.jpg%22%2C%225506.jpg%22%2C%225507.jpg%22%2C%225508.jpg%22%2C%225509.jpg%22%2C%225510.jpg%22%2C%225511.jpg%22%2C%225512.jpg%22%2C%225513.jpg%22%2C%225514.jpg%22%2C%225515.jpg%22%2C%225516.jpg%22%2C%225517.jpg%22%2C%225518.jpg%22%2C%225519.jpg%22%2C%225520.jpg%22%2C%225521.jpg%22%2C%225522.jpg%22%2C%225523.jpg%22%2C%225524.jpg%22%2C%225525.jpg%22%2C%225526.jpg%22%2C%225527.jpg%22%2C%225528.jpg%22%2C%225529.jpg%22%2C%225530.jpg%22%2C%225531.jpg%22%2C%225532.jpg%22%2C%225533.jpg%22%2C%225534.jpg%22%2C%225535.jpg%22%2C%225536.jpg%22%2C%225537.jpg%22%2C%225538.jpg%22%2C%225539.jpg%22%2C%225540.jpg%22%2C%225541.jpg%22%2C%225542.jpg%22%2C%225682.jpg%22%2C%225683.jpg%22%2C%225684.jpg%22%2C%225685.jpg%22%2C%225686.jpg%22%2C%225687.jpg%22%2C%225688.jpg%22%2C%225689.jpg%22%2C%225690.jpg%22%2C%225691.jpg%22%2C%225692.jpg%22%2C%225693.jpg%22%2C%225694.jpg%22%2C%225695.jpg%22%2C%225696.jpg%22%2C%225697.jpg%22%2C%225698.jpg%22%2C%225699.jpg%22%2C%225700.jpg%22%2C%225701.jpg%22%2C%225702.jpg%22%2C%225703.jpg%22%2C%225704.jpg%22%2C%225705.jpg%22%2C%225706.jpg%22%2C%225707.jpg%22%2C%225708.jpg%22%2C%225709.jpg%22%2C%225710.jpg%22%2C%225711.jpg%22%2C%225712.jpg%22%2C%225713.jpg%22%2C%225714.jpg%22%2C%225715.jpg%22%2C%225716.jpg%22%2C%225717.jpg%22%2C%225718.jpg%22%2C%225719.jpg%22%2C%225720.jpg%22%2C%225721.jpg%22%2C%225722.jpg%22%2C%225723.jpg%22%2C%225724.jpg%22%2C%225725.jpg%22%2C%225726.jpg%22%2C%225727.jpg%22%2C%225728.jpg%22%2C%225729.jpg%22%2C%225730.jpg%22%2C%225731.jpg%22%2C%225732.jpg%22%2C%225733.jpg%22%2C%225734.jpg%22%2C%225735.jpg%22%2C%225736.jpg%22%2C%225737.jpg%22%2C%225738.jpg%22%2C%225739.jpg%22%2C%225740.jpg%22%2C%225741.jpg%22%2C%225742.jpg%22%2C%225743.jpg%22%2C%225744.jpg%22%2C%225745.jpg%22%2C%225746.jpg%22%2C%225747.jpg%22%2C%225748.jpg%22%2C%225749.jpg%22%2C%225750.jpg%22%2C%225751.jpg%22%2C%225752.jpg%22%2C%225753.jpg%22%2C%225754.jpg%22%2C%225755.jpg%22%2C%225756.jpg%22%2C%225757.jpg%22%2C%225758.jpg%22%2C%225759.jpg%22%2C%225760.jpg%22%2C%225761.jpg%22%2C%225762.jpg%22%2C%225763.jpg%22%2C%225764.jpg%22%2C%225765.jpg%22%2C%225766.jpg%22%2C%225767.jpg%22%2C%225768.jpg%22%2C%225769.jpg%22%2C%225770.jpg%22%2C%225771.jpg%22%2C%225772.jpg%22%2C%225773.jpg%22%2C%225774.jpg%22%2C%225775.jpg%22%2C%225776.jpg%22%2C%225777.jpg%22%2C%225778.jpg%22%2C%225779.jpg%22%2C%225780.jpg%22%2C%225781.jpg%22%2C%225782.jpg%22%2C%225783.jpg%22%2C%225784.jpg%22%2C%225785.jpg%22%2C%225786.jpg%22%2C%225787.jpg%22%2C%225788.jpg%22%2C%225789.jpg%22%2C%225790.jpg%22%2C%225791.jpg%22%2C%225792.jpg%22%2C%225793.jpg%22%2C%225794.jpg%22%2C%225795.jpg%22%2C%225796.jpg%22%2C%225797.jpg%22%2C%225798.jpg%22%2C%225799.jpg%22%2C%225800.jpg%22%2C%225801.jpg%22%2C%225802.jpg%22%2C%225803.jpg%22%2C%225804.jpg%22%2C%225805.jpg%22%2C%225806.jpg%22%2C%225807.jpg%22%2C%225808.jpg%22%2C%225809.jpg%22%2C%225810.jpg%22%2C%225811.jpg%22%2C%225812.jpg%22%2C%225813.jpg%22%2C%225814.jpg%22%2C%225815.jpg%22%2C%225816.jpg%22%2C%225817.jpg%22%2C%225818.jpg%22%2C%225819.jpg%22%2C%225820.jpg%22%2C%225821.jpg%22%2C%225822.jpg%22%2C%225823.jpg%22%2C%225824.jpg%22%2C%225825.jpg%22%2C%225826.jpg%22%2C%225827.jpg%22%2C%225828.jpg%22%2C%225829.jpg%22%2C%225830.jpg%22%2C%225831.jpg%22%2C%225832.jpg%22%2C%225833.jpg%22%2C%225834.jpg%22%2C%225835.jpg%22%2C%225836.jpg%22%2C%225837.jpg%22%2C%225838.jpg%22%2C%225839.jpg%22%2C%225840.jpg%22%2C%225841.jpg%22%2C%225842.jpg%22%2C%225843.jpg%22%2C%225844.jpg%22%2C%225845.jpg%22%2C%225846.jpg%22%2C%225847.jpg%22%2C%225848.jpg%22%2C%225849.jpg%22%2C%225850.jpg%22%2C%225851.jpg%22%2C%225852.jpg%22%2C%225853.jpg%22%2C%225854.jpg%22%2C%225855.jpg%22%2C%225856.jpg%22%2C%225857.jpg%22%2C%225858.jpg%22%2C%225859.jpg%22%2C%225860.jpg%22%2C%225861.jpg%22%2C%225862.jpg%22%2C%225863.jpg%22%2C%225864.jpg%22%2C%225865.jpg%22%2C%225866.jpg%22%2C%225867.jpg%22%2C%225868.jpg%22%2C%225869.jpg%22%2C%225870.jpg%22%2C%225871.jpg%22%2C%225872.jpg%22%2C%225873.jpg%22%2C%225874.jpg%22%2C%225875.jpg%22%2C%225876.jpg%22%2C%225877.jpg%22%2C%225878.jpg%22%2C%225879.jpg%22%2C%225880.jpg%22%2C%225881.jpg%22%2C%225882.jpg%22%2C%225883.jpg%22%2C%225884.jpg%22%2C%225885.jpg%22%2C%225886.jpg%22%2C%225887.jpg%22%2C%225888.jpg%22%2C%225889.jpg%22%2C%225890.jpg%22%2C%225891.jpg%22%2C%225892.jpg%22%2C%225893.jpg%22%2C%225894.jpg%22%2C%225895.jpg%22%2C%225896.jpg%22%2C%225897.jpg%22%2C%225898.jpg%22%2C%225899.jpg%22%2C%225900.jpg%22%2C%225901.jpg%22%2C%225902.jpg%22%2C%225903.jpg%22%2C%225904.jpg%22%2C%225905.jpg%22%2C%225906.jpg%22%2C%225907.jpg%22%2C%225908.jpg%22%2C%225909.jpg%22%2C%225910.jpg%22%2C%225911.jpg%22%2C%225912.jpg%22%2C%225913.jpg%22%2C%225914.jpg%22%2C%225915.jpg%22%2C%225916.jpg%22%2C%225917.jpg%22%2C%225918.jpg%22%2C%225919.jpg%22%2C%225920.jpg%22%2C%225921.jpg%22%2C%225922.jpg%22%2C%225923.jpg%22%2C%225924.jpg%22%2C%225925.jpg%22%2C%225926.jpg%22%2C%225927.jpg%22%2C%225928.jpg%22%2C%225929.jpg%22%2C%225930.jpg%22%2C%225931.jpg%22%2C%225932.jpg%22%2C%225933.jpg%22%2C%225934.jpg%22%2C%225935.jpg%22%2C%225936.jpg%22%2C%225937.jpg%22%2C%225938.jpg%22%2C%225939.jpg%22%2C%225940.jpg%22%2C%225941.jpg%22%2C%225942.jpg%22%2C%225943.jpg%22%2C%225944.jpg%22%2C%225945.jpg%22%2C%225946.jpg%22%2C%225947.jpg%22%2C%225948.jpg%22%2C%225949.jpg%22%2C%225950.jpg%22%2C%225951.jpg%22%2C%225952.jpg%22%2C%225953.jpg%22%2C%225954.jpg%22%2C%225955.jpg%22%2C%225956.jpg%22%2C%225957.jpg%22%2C%225958.jpg%22%2C%225959.jpg%22%2C%225960.jpg%22%2C%225961.jpg%22%2C%225962.jpg%22%2C%225963.jpg%22%2C%225964.jpg%22%2C%225965.jpg%22%2C%225966.jpg%22%2C%225967.jpg%22%2C%225968.jpg%22%2C%225969.jpg%22%2C%225970.jpg%22%2C%225971.jpg%22%2C%225972.jpg%22%2C%225973.jpg%22%2C%225974.jpg%22%2C%225975.jpg%22%2C%225976.jpg%22%2C%225977.jpg%22%2C%225978.jpg%22%2C%225979.jpg%22%2C%225980.jpg%22%2C%225981.jpg%22%2C%225982.jpg%22%5D",
            filename="train7.zip",
            md5=False,
            checksum="59c3418110f8d8fd1db5e437b97690e47b2622f5ad718ac993717275bf48f732",
        ),
        DataLocation(
            url="https://cloud.cs.uni-tuebingen.de/public.php/dav/files/ZZxX65FGnQ8zjBP/Compressed%20Version/images/train/?accept=zip&files=%5B%225983.jpg%22%2C%225984.jpg%22%2C%225985.jpg%22%2C%225986.jpg%22%2C%225987.jpg%22%2C%225988.jpg%22%2C%225989.jpg%22%2C%225990.jpg%22%2C%225991.jpg%22%2C%225992.jpg%22%2C%225993.jpg%22%2C%225994.jpg%22%2C%225995.jpg%22%2C%225996.jpg%22%2C%225997.jpg%22%2C%225998.jpg%22%2C%225999.jpg%22%2C%226000.jpg%22%2C%226001.jpg%22%2C%226002.jpg%22%2C%226003.jpg%22%2C%226004.jpg%22%2C%226005.jpg%22%2C%226006.jpg%22%2C%226007.jpg%22%2C%226008.jpg%22%2C%226009.jpg%22%2C%226010.jpg%22%2C%226011.jpg%22%2C%226012.jpg%22%2C%226013.jpg%22%2C%226014.jpg%22%2C%226015.jpg%22%2C%226016.jpg%22%2C%226017.jpg%22%2C%226018.jpg%22%2C%226019.jpg%22%2C%226020.jpg%22%2C%226021.jpg%22%2C%226022.jpg%22%2C%226023.jpg%22%2C%226024.jpg%22%2C%226025.jpg%22%2C%226026.jpg%22%2C%226027.jpg%22%2C%226028.jpg%22%2C%226029.jpg%22%2C%226030.jpg%22%2C%226031.jpg%22%2C%226032.jpg%22%2C%226033.jpg%22%2C%226034.jpg%22%2C%226035.jpg%22%2C%226036.jpg%22%2C%226037.jpg%22%2C%226038.jpg%22%2C%226280.jpg%22%2C%226281.jpg%22%2C%226282.jpg%22%2C%226283.jpg%22%2C%226284.jpg%22%2C%226285.jpg%22%2C%226286.jpg%22%2C%226287.jpg%22%2C%226288.jpg%22%2C%226289.jpg%22%2C%226290.jpg%22%2C%226291.jpg%22%2C%226292.jpg%22%2C%226293.jpg%22%2C%226294.jpg%22%2C%226295.jpg%22%2C%226296.jpg%22%2C%226297.jpg%22%2C%226298.jpg%22%2C%226299.jpg%22%2C%226300.jpg%22%2C%226301.jpg%22%2C%226302.jpg%22%2C%226303.jpg%22%2C%226304.jpg%22%2C%226305.jpg%22%2C%226306.jpg%22%2C%226307.jpg%22%2C%226308.jpg%22%2C%226309.jpg%22%2C%226310.jpg%22%2C%226311.jpg%22%2C%226312.jpg%22%2C%226313.jpg%22%2C%226314.jpg%22%2C%226315.jpg%22%2C%226316.jpg%22%2C%226317.jpg%22%2C%226318.jpg%22%2C%226319.jpg%22%2C%226320.jpg%22%2C%226321.jpg%22%2C%226322.jpg%22%2C%226323.jpg%22%2C%226324.jpg%22%2C%226325.jpg%22%2C%226326.jpg%22%2C%226327.jpg%22%2C%226328.jpg%22%2C%226329.jpg%22%2C%226330.jpg%22%2C%226331.jpg%22%2C%226332.jpg%22%2C%226333.jpg%22%2C%226334.jpg%22%2C%226335.jpg%22%2C%226336.jpg%22%2C%226337.jpg%22%2C%226338.jpg%22%2C%226339.jpg%22%2C%226340.jpg%22%2C%226341.jpg%22%2C%226342.jpg%22%2C%226343.jpg%22%2C%226344.jpg%22%2C%226345.jpg%22%2C%226346.jpg%22%2C%226347.jpg%22%2C%226348.jpg%22%2C%226349.jpg%22%2C%226350.jpg%22%2C%226351.jpg%22%2C%226352.jpg%22%2C%226353.jpg%22%2C%226354.jpg%22%2C%226355.jpg%22%2C%226356.jpg%22%2C%226357.jpg%22%2C%226358.jpg%22%2C%226359.jpg%22%2C%226360.jpg%22%2C%226361.jpg%22%2C%226362.jpg%22%2C%226363.jpg%22%2C%226364.jpg%22%2C%226365.jpg%22%2C%226366.jpg%22%2C%226367.jpg%22%2C%226368.jpg%22%2C%226369.jpg%22%2C%226370.jpg%22%2C%226371.jpg%22%2C%226372.jpg%22%2C%226373.jpg%22%2C%226374.jpg%22%2C%226375.jpg%22%2C%226376.jpg%22%2C%226377.jpg%22%2C%226378.jpg%22%2C%226379.jpg%22%2C%226380.jpg%22%2C%226381.jpg%22%2C%226382.jpg%22%2C%226383.jpg%22%2C%226384.jpg%22%2C%226385.jpg%22%2C%226386.jpg%22%2C%226387.jpg%22%2C%226388.jpg%22%2C%226389.jpg%22%2C%226390.jpg%22%2C%226391.jpg%22%2C%226392.jpg%22%2C%226393.jpg%22%2C%226394.jpg%22%2C%226395.jpg%22%2C%226396.jpg%22%2C%226397.jpg%22%2C%226398.jpg%22%2C%226399.jpg%22%2C%226400.jpg%22%2C%226401.jpg%22%2C%226402.jpg%22%2C%226403.jpg%22%2C%226404.jpg%22%2C%226405.jpg%22%2C%226406.jpg%22%2C%226407.jpg%22%2C%226408.jpg%22%2C%226409.jpg%22%2C%226410.jpg%22%2C%226411.jpg%22%2C%226412.jpg%22%2C%226413.jpg%22%2C%226414.jpg%22%2C%226415.jpg%22%2C%226416.jpg%22%2C%226417.jpg%22%2C%226418.jpg%22%2C%226419.jpg%22%2C%226420.jpg%22%2C%226421.jpg%22%2C%226422.jpg%22%2C%226423.jpg%22%2C%226424.jpg%22%2C%226425.jpg%22%2C%226426.jpg%22%2C%226427.jpg%22%2C%226428.jpg%22%2C%226429.jpg%22%2C%226430.jpg%22%2C%226431.jpg%22%2C%226432.jpg%22%2C%226433.jpg%22%2C%226434.jpg%22%2C%226435.jpg%22%2C%226436.jpg%22%2C%226437.jpg%22%2C%226438.jpg%22%2C%226439.jpg%22%2C%226440.jpg%22%2C%226441.jpg%22%2C%226442.jpg%22%2C%226443.jpg%22%2C%226444.jpg%22%2C%226445.jpg%22%2C%226446.jpg%22%2C%226447.jpg%22%2C%226448.jpg%22%2C%226449.jpg%22%2C%226450.jpg%22%2C%226451.jpg%22%2C%226452.jpg%22%2C%226453.jpg%22%2C%226454.jpg%22%2C%226455.jpg%22%2C%226456.jpg%22%2C%226457.jpg%22%2C%226458.jpg%22%2C%226459.jpg%22%2C%226460.jpg%22%2C%226461.jpg%22%2C%226462.jpg%22%2C%226463.jpg%22%2C%226464.jpg%22%2C%226465.jpg%22%2C%226466.jpg%22%2C%226467.jpg%22%2C%226468.jpg%22%2C%226469.jpg%22%2C%226470.jpg%22%2C%226471.jpg%22%2C%226472.jpg%22%2C%226473.jpg%22%2C%226474.jpg%22%2C%226475.jpg%22%2C%226476.jpg%22%2C%226477.jpg%22%2C%226478.jpg%22%2C%226479.jpg%22%2C%226480.jpg%22%2C%226481.jpg%22%2C%226482.jpg%22%2C%226483.jpg%22%2C%226484.jpg%22%2C%226485.jpg%22%2C%226486.jpg%22%2C%226487.jpg%22%2C%226488.jpg%22%2C%226489.jpg%22%2C%226490.jpg%22%2C%226491.jpg%22%2C%226492.jpg%22%2C%226493.jpg%22%2C%226494.jpg%22%2C%226495.jpg%22%2C%226496.jpg%22%2C%226497.jpg%22%2C%226498.jpg%22%2C%226499.jpg%22%2C%226500.jpg%22%2C%226501.jpg%22%2C%226502.jpg%22%2C%226503.jpg%22%2C%226504.jpg%22%2C%226505.jpg%22%2C%226506.jpg%22%2C%226507.jpg%22%2C%226508.jpg%22%2C%226509.jpg%22%2C%226510.jpg%22%2C%226511.jpg%22%2C%226512.jpg%22%2C%226513.jpg%22%2C%226514.jpg%22%2C%226515.jpg%22%2C%226516.jpg%22%2C%226517.jpg%22%2C%226518.jpg%22%2C%226519.jpg%22%2C%226520.jpg%22%2C%226521.jpg%22%2C%226522.jpg%22%2C%226523.jpg%22%2C%226524.jpg%22%2C%226525.jpg%22%2C%226526.jpg%22%2C%226527.jpg%22%2C%226528.jpg%22%2C%226529.jpg%22%2C%226530.jpg%22%2C%226531.jpg%22%2C%226532.jpg%22%2C%226533.jpg%22%2C%226534.jpg%22%2C%226535.jpg%22%2C%226536.jpg%22%2C%226537.jpg%22%2C%226538.jpg%22%2C%226539.jpg%22%2C%226540.jpg%22%2C%226541.jpg%22%2C%226542.jpg%22%2C%226543.jpg%22%2C%226544.jpg%22%2C%226545.jpg%22%2C%226546.jpg%22%2C%226547.jpg%22%2C%226548.jpg%22%2C%226549.jpg%22%2C%226550.jpg%22%2C%226551.jpg%22%2C%226552.jpg%22%2C%226553.jpg%22%2C%226554.jpg%22%2C%226555.jpg%22%2C%226556.jpg%22%2C%226557.jpg%22%2C%226558.jpg%22%2C%226559.jpg%22%2C%226560.jpg%22%2C%226561.jpg%22%2C%226562.jpg%22%2C%226563.jpg%22%2C%226564.jpg%22%2C%226565.jpg%22%2C%226566.jpg%22%2C%226567.jpg%22%2C%226568.jpg%22%2C%226569.jpg%22%2C%226570.jpg%22%2C%226571.jpg%22%2C%226572.jpg%22%2C%226573.jpg%22%2C%226574.jpg%22%2C%226575.jpg%22%2C%226576.jpg%22%2C%226577.jpg%22%2C%226578.jpg%22%2C%226579.jpg%22%2C%226580.jpg%22%2C%226581.jpg%22%2C%226582.jpg%22%2C%226583.jpg%22%2C%226584.jpg%22%2C%226585.jpg%22%2C%226586.jpg%22%2C%226587.jpg%22%2C%226588.jpg%22%2C%226589.jpg%22%2C%226590.jpg%22%2C%226591.jpg%22%2C%226592.jpg%22%2C%226593.jpg%22%2C%226594.jpg%22%2C%226595.jpg%22%2C%226596.jpg%22%2C%226597.jpg%22%2C%226598.jpg%22%2C%226599.jpg%22%2C%226600.jpg%22%2C%226601.jpg%22%2C%226602.jpg%22%2C%226603.jpg%22%2C%226604.jpg%22%2C%226605.jpg%22%2C%226606.jpg%22%2C%226607.jpg%22%2C%226608.jpg%22%2C%226609.jpg%22%2C%226610.jpg%22%2C%226611.jpg%22%2C%226612.jpg%22%2C%226613.jpg%22%2C%226614.jpg%22%2C%226615.jpg%22%2C%226616.jpg%22%2C%226617.jpg%22%2C%226618.jpg%22%2C%226619.jpg%22%2C%226620.jpg%22%2C%226621.jpg%22%2C%226622.jpg%22%2C%226623.jpg%22%2C%226624.jpg%22%2C%226625.jpg%22%2C%226626.jpg%22%2C%226627.jpg%22%2C%226628.jpg%22%2C%226629.jpg%22%2C%226630.jpg%22%2C%226631.jpg%22%2C%226632.jpg%22%2C%226633.jpg%22%2C%226634.jpg%22%2C%226635.jpg%22%2C%226636.jpg%22%2C%226637.jpg%22%2C%226638.jpg%22%2C%226639.jpg%22%2C%226640.jpg%22%2C%226882.jpg%22%2C%226883.jpg%22%2C%226884.jpg%22%2C%226885.jpg%22%2C%226886.jpg%22%2C%226887.jpg%22%2C%226888.jpg%22%2C%226889.jpg%22%2C%226890.jpg%22%2C%226891.jpg%22%2C%226892.jpg%22%2C%226893.jpg%22%2C%226894.jpg%22%2C%226895.jpg%22%2C%226896.jpg%22%2C%226897.jpg%22%2C%226898.jpg%22%2C%226899.jpg%22%2C%226900.jpg%22%2C%226901.jpg%22%2C%226902.jpg%22%2C%226903.jpg%22%2C%226904.jpg%22%2C%226905.jpg%22%2C%226906.jpg%22%2C%226907.jpg%22%2C%226908.jpg%22%2C%226909.jpg%22%2C%226910.jpg%22%2C%226911.jpg%22%2C%226912.jpg%22%2C%226913.jpg%22%2C%226914.jpg%22%5D",
            filename="train8.zip",
            md5=False,
            checksum="c1f31da134065f15a9810d5997e002fb516ab9cb732529794f28c27609e26a22",
        ),
        DataLocation(
            url="https://cloud.cs.uni-tuebingen.de/public.php/dav/files/ZZxX65FGnQ8zjBP/Compressed%20Version/images/train/?accept=zip&files=%5B%226915.jpg%22%2C%226916.jpg%22%2C%226917.jpg%22%2C%226918.jpg%22%2C%226919.jpg%22%2C%226920.jpg%22%2C%226921.jpg%22%2C%226922.jpg%22%2C%226923.jpg%22%2C%226924.jpg%22%2C%226925.jpg%22%2C%226926.jpg%22%2C%226927.jpg%22%2C%226928.jpg%22%2C%226929.jpg%22%2C%226930.jpg%22%2C%226931.jpg%22%2C%226932.jpg%22%2C%226933.jpg%22%2C%226934.jpg%22%2C%226935.jpg%22%2C%226936.jpg%22%2C%226937.jpg%22%2C%226938.jpg%22%2C%226939.jpg%22%2C%226940.jpg%22%2C%226941.jpg%22%2C%226942.jpg%22%2C%226943.jpg%22%2C%226944.jpg%22%2C%226945.jpg%22%2C%226946.jpg%22%2C%226947.jpg%22%2C%226948.jpg%22%2C%226949.jpg%22%2C%226950.jpg%22%2C%226951.jpg%22%2C%226952.jpg%22%2C%226953.jpg%22%2C%226954.jpg%22%2C%226955.jpg%22%2C%226956.jpg%22%2C%226957.jpg%22%2C%226958.jpg%22%2C%226959.jpg%22%2C%226960.jpg%22%2C%226961.jpg%22%2C%226962.jpg%22%2C%226963.jpg%22%2C%226964.jpg%22%2C%226965.jpg%22%2C%226966.jpg%22%2C%226967.jpg%22%2C%226968.jpg%22%2C%226969.jpg%22%2C%226970.jpg%22%2C%226971.jpg%22%2C%226972.jpg%22%2C%226973.jpg%22%2C%226974.jpg%22%2C%226975.jpg%22%2C%226976.jpg%22%2C%226977.jpg%22%2C%226978.jpg%22%2C%226979.jpg%22%2C%226980.jpg%22%2C%226981.jpg%22%2C%226982.jpg%22%2C%226983.jpg%22%2C%226984.jpg%22%2C%226985.jpg%22%2C%226986.jpg%22%2C%226987.jpg%22%2C%226988.jpg%22%2C%226989.jpg%22%2C%226990.jpg%22%2C%226991.jpg%22%2C%226992.jpg%22%2C%226993.jpg%22%2C%226994.jpg%22%2C%226995.jpg%22%2C%226996.jpg%22%2C%226997.jpg%22%2C%226998.jpg%22%2C%226999.jpg%22%2C%227000.jpg%22%2C%227001.jpg%22%2C%227002.jpg%22%2C%227003.jpg%22%2C%227004.jpg%22%2C%227005.jpg%22%2C%227006.jpg%22%2C%227007.jpg%22%2C%227008.jpg%22%2C%227009.jpg%22%2C%227010.jpg%22%2C%227011.jpg%22%2C%227012.jpg%22%2C%227013.jpg%22%2C%227014.jpg%22%2C%227015.jpg%22%2C%227016.jpg%22%2C%227017.jpg%22%2C%227018.jpg%22%2C%227019.jpg%22%2C%227020.jpg%22%2C%227021.jpg%22%2C%227022.jpg%22%2C%227023.jpg%22%2C%227024.jpg%22%2C%227025.jpg%22%2C%227026.jpg%22%2C%227027.jpg%22%2C%227028.jpg%22%2C%227029.jpg%22%2C%227030.jpg%22%2C%227031.jpg%22%2C%227032.jpg%22%2C%227033.jpg%22%2C%227034.jpg%22%2C%227035.jpg%22%2C%227036.jpg%22%2C%227037.jpg%22%2C%227038.jpg%22%2C%227039.jpg%22%2C%227040.jpg%22%2C%227041.jpg%22%2C%227042.jpg%22%2C%227043.jpg%22%2C%227044.jpg%22%2C%227045.jpg%22%2C%227046.jpg%22%2C%227047.jpg%22%2C%227048.jpg%22%2C%227049.jpg%22%2C%227050.jpg%22%2C%227051.jpg%22%2C%227052.jpg%22%2C%227053.jpg%22%2C%227054.jpg%22%2C%227055.jpg%22%2C%227056.jpg%22%2C%227057.jpg%22%2C%227058.jpg%22%2C%227059.jpg%22%2C%227060.jpg%22%2C%227061.jpg%22%2C%227062.jpg%22%2C%227063.jpg%22%2C%227064.jpg%22%2C%227065.jpg%22%2C%227066.jpg%22%2C%227067.jpg%22%2C%227068.jpg%22%2C%227069.jpg%22%2C%227070.jpg%22%2C%227071.jpg%22%2C%227072.jpg%22%2C%227073.jpg%22%2C%227074.jpg%22%2C%227075.jpg%22%2C%227076.jpg%22%2C%227077.jpg%22%2C%227078.jpg%22%2C%227079.jpg%22%2C%227080.jpg%22%2C%227081.jpg%22%2C%227082.jpg%22%2C%227083.jpg%22%2C%227084.jpg%22%2C%227085.jpg%22%2C%227086.jpg%22%2C%227087.jpg%22%2C%227088.jpg%22%2C%227089.jpg%22%2C%227090.jpg%22%2C%227091.jpg%22%2C%227092.jpg%22%2C%227093.jpg%22%2C%227094.jpg%22%2C%227095.jpg%22%2C%227096.jpg%22%2C%227097.jpg%22%2C%227098.jpg%22%2C%227099.jpg%22%2C%227100.jpg%22%2C%227101.jpg%22%2C%227102.jpg%22%2C%227103.jpg%22%2C%227104.jpg%22%2C%227105.jpg%22%2C%227106.jpg%22%2C%227107.jpg%22%2C%227108.jpg%22%2C%227109.jpg%22%2C%227110.jpg%22%2C%227111.jpg%22%2C%227112.jpg%22%2C%227113.jpg%22%2C%227114.jpg%22%2C%227115.jpg%22%2C%227116.jpg%22%2C%227117.jpg%22%2C%227118.jpg%22%2C%227119.jpg%22%2C%227120.jpg%22%2C%227121.jpg%22%2C%227122.jpg%22%2C%227123.jpg%22%2C%227124.jpg%22%2C%227125.jpg%22%2C%227126.jpg%22%2C%227127.jpg%22%2C%227128.jpg%22%2C%227129.jpg%22%2C%227130.jpg%22%2C%227131.jpg%22%2C%227132.jpg%22%2C%227133.jpg%22%2C%227134.jpg%22%2C%227135.jpg%22%2C%227136.jpg%22%2C%227137.jpg%22%2C%227138.jpg%22%2C%227139.jpg%22%2C%227140.jpg%22%2C%227141.jpg%22%2C%227142.jpg%22%2C%227143.jpg%22%2C%227144.jpg%22%2C%227145.jpg%22%2C%227146.jpg%22%2C%227147.jpg%22%2C%227148.jpg%22%2C%227149.jpg%22%2C%227150.jpg%22%2C%227151.jpg%22%2C%227152.jpg%22%2C%227153.jpg%22%2C%227154.jpg%22%2C%227155.jpg%22%2C%227156.jpg%22%2C%227157.jpg%22%2C%227158.jpg%22%2C%227159.jpg%22%2C%227160.jpg%22%2C%227161.jpg%22%2C%227162.jpg%22%2C%227163.jpg%22%2C%227164.jpg%22%2C%227165.jpg%22%2C%227166.jpg%22%2C%227167.jpg%22%2C%227168.jpg%22%2C%227169.jpg%22%2C%227170.jpg%22%2C%227171.jpg%22%2C%227386.jpg%22%2C%227387.jpg%22%2C%227388.jpg%22%2C%227389.jpg%22%2C%227390.jpg%22%2C%227391.jpg%22%2C%227392.jpg%22%2C%227393.jpg%22%2C%227394.jpg%22%2C%227395.jpg%22%2C%227396.jpg%22%2C%227397.jpg%22%2C%227398.jpg%22%2C%227399.jpg%22%2C%227400.jpg%22%2C%227401.jpg%22%2C%227402.jpg%22%2C%227403.jpg%22%2C%227404.jpg%22%2C%227405.jpg%22%2C%227406.jpg%22%2C%227407.jpg%22%2C%227408.jpg%22%2C%227409.jpg%22%2C%227410.jpg%22%2C%227411.jpg%22%2C%227412.jpg%22%2C%227413.jpg%22%2C%227414.jpg%22%2C%227415.jpg%22%2C%227416.jpg%22%2C%227417.jpg%22%2C%227418.jpg%22%2C%227419.jpg%22%2C%227420.jpg%22%2C%227421.jpg%22%2C%227422.jpg%22%2C%227423.jpg%22%2C%227424.jpg%22%2C%227425.jpg%22%2C%227426.jpg%22%2C%227427.jpg%22%2C%227428.jpg%22%2C%227429.jpg%22%2C%227430.jpg%22%2C%227431.jpg%22%2C%227432.jpg%22%2C%227433.jpg%22%2C%227434.jpg%22%2C%227435.jpg%22%2C%227436.jpg%22%2C%227437.jpg%22%2C%227438.jpg%22%2C%227439.jpg%22%2C%227440.jpg%22%2C%227441.jpg%22%2C%227442.jpg%22%2C%227443.jpg%22%2C%227444.jpg%22%2C%227445.jpg%22%2C%227446.jpg%22%2C%227447.jpg%22%2C%227448.jpg%22%2C%227449.jpg%22%2C%227450.jpg%22%2C%227451.jpg%22%2C%227452.jpg%22%2C%227453.jpg%22%2C%227454.jpg%22%2C%227455.jpg%22%2C%227456.jpg%22%2C%227457.jpg%22%2C%227458.jpg%22%2C%227459.jpg%22%2C%227460.jpg%22%2C%227461.jpg%22%2C%227462.jpg%22%2C%227463.jpg%22%2C%227464.jpg%22%2C%227465.jpg%22%2C%227466.jpg%22%2C%227467.jpg%22%2C%227468.jpg%22%2C%227469.jpg%22%2C%227470.jpg%22%2C%227471.jpg%22%2C%227472.jpg%22%2C%227473.jpg%22%2C%227474.jpg%22%2C%227475.jpg%22%2C%227476.jpg%22%2C%227477.jpg%22%2C%227478.jpg%22%2C%227479.jpg%22%2C%227480.jpg%22%2C%227481.jpg%22%2C%227482.jpg%22%2C%227483.jpg%22%2C%227484.jpg%22%2C%227485.jpg%22%2C%227486.jpg%22%2C%227487.jpg%22%2C%227488.jpg%22%2C%227489.jpg%22%2C%227490.jpg%22%2C%227491.jpg%22%2C%227492.jpg%22%2C%227493.jpg%22%2C%227494.jpg%22%2C%227495.jpg%22%2C%227496.jpg%22%2C%227497.jpg%22%2C%227498.jpg%22%2C%227499.jpg%22%2C%227500.jpg%22%2C%227501.jpg%22%2C%227502.jpg%22%2C%227503.jpg%22%2C%227504.jpg%22%2C%227505.jpg%22%2C%227506.jpg%22%2C%227507.jpg%22%2C%227508.jpg%22%2C%227509.jpg%22%2C%227510.jpg%22%2C%227511.jpg%22%2C%227512.jpg%22%2C%227513.jpg%22%2C%227514.jpg%22%2C%227515.jpg%22%2C%227516.jpg%22%2C%227517.jpg%22%2C%227518.jpg%22%2C%227519.jpg%22%2C%227520.jpg%22%2C%227521.jpg%22%2C%227522.jpg%22%2C%227523.jpg%22%2C%227524.jpg%22%2C%227525.jpg%22%2C%227526.jpg%22%2C%227527.jpg%22%2C%227528.jpg%22%2C%227529.jpg%22%2C%227530.jpg%22%2C%227531.jpg%22%2C%227532.jpg%22%2C%227533.jpg%22%2C%227534.jpg%22%2C%227535.jpg%22%2C%227536.jpg%22%2C%227537.jpg%22%2C%227538.jpg%22%2C%227539.jpg%22%2C%227540.jpg%22%2C%227541.jpg%22%2C%227542.jpg%22%2C%227543.jpg%22%2C%227544.jpg%22%2C%227545.jpg%22%2C%227546.jpg%22%2C%227547.jpg%22%2C%227548.jpg%22%2C%227549.jpg%22%2C%227550.jpg%22%2C%227551.jpg%22%2C%227552.jpg%22%2C%227553.jpg%22%2C%227554.jpg%22%2C%227555.jpg%22%2C%227556.jpg%22%2C%227557.jpg%22%2C%227558.jpg%22%2C%227559.jpg%22%2C%227560.jpg%22%2C%227561.jpg%22%2C%227562.jpg%22%2C%227563.jpg%22%2C%227564.jpg%22%2C%227565.jpg%22%2C%227566.jpg%22%2C%227567.jpg%22%2C%227568.jpg%22%2C%227569.jpg%22%2C%227570.jpg%22%2C%227571.jpg%22%2C%227572.jpg%22%2C%227573.jpg%22%2C%227574.jpg%22%2C%227575.jpg%22%2C%227576.jpg%22%2C%227577.jpg%22%2C%227578.jpg%22%5D",
            filename="train9.zip",
            md5=False,
            checksum="a33891186a6e1a1c57d048ba3b942dabe10c30b5d972b82df93280edc75feb2b",
        ),
        DataLocation(
            url="https://cloud.cs.uni-tuebingen.de/public.php/dav/files/ZZxX65FGnQ8zjBP/Compressed%20Version/images/train/?accept=zip&files=%5B%227579.jpg%22%2C%227580.jpg%22%2C%227581.jpg%22%2C%227582.jpg%22%2C%227583.jpg%22%2C%227584.jpg%22%2C%227585.jpg%22%2C%227586.jpg%22%2C%227587.jpg%22%2C%227588.jpg%22%2C%227589.jpg%22%2C%227590.jpg%22%2C%227591.jpg%22%2C%227592.jpg%22%2C%227593.jpg%22%2C%227594.jpg%22%2C%227595.jpg%22%2C%227596.jpg%22%2C%227597.jpg%22%2C%227598.jpg%22%2C%227599.jpg%22%2C%227600.jpg%22%2C%227601.jpg%22%2C%227602.jpg%22%2C%227603.jpg%22%2C%227604.jpg%22%2C%227605.jpg%22%2C%227606.jpg%22%2C%227607.jpg%22%2C%227608.jpg%22%2C%227609.jpg%22%2C%227610.jpg%22%2C%227611.jpg%22%2C%227612.jpg%22%2C%227613.jpg%22%2C%227614.jpg%22%2C%227615.jpg%22%2C%227616.jpg%22%2C%227617.jpg%22%2C%227618.jpg%22%2C%227619.jpg%22%2C%227620.jpg%22%2C%227621.jpg%22%2C%227622.jpg%22%2C%227623.jpg%22%2C%227624.jpg%22%2C%227625.jpg%22%2C%227626.jpg%22%2C%227627.jpg%22%2C%227628.jpg%22%2C%227629.jpg%22%2C%227630.jpg%22%2C%227631.jpg%22%2C%227632.jpg%22%2C%227633.jpg%22%2C%227634.jpg%22%2C%227635.jpg%22%2C%227636.jpg%22%2C%227637.jpg%22%2C%227638.jpg%22%2C%227639.jpg%22%2C%227640.jpg%22%2C%227641.jpg%22%2C%227642.jpg%22%2C%227643.jpg%22%2C%227644.jpg%22%2C%227645.jpg%22%2C%227646.jpg%22%2C%227647.jpg%22%2C%227648.jpg%22%2C%227649.jpg%22%2C%227650.jpg%22%2C%227651.jpg%22%2C%227652.jpg%22%2C%227653.jpg%22%2C%227654.jpg%22%2C%227655.jpg%22%2C%227656.jpg%22%2C%227657.jpg%22%2C%227658.jpg%22%2C%227659.jpg%22%2C%227660.jpg%22%2C%227661.jpg%22%2C%227662.jpg%22%2C%227663.jpg%22%2C%227664.jpg%22%2C%227665.jpg%22%2C%227666.jpg%22%2C%227667.jpg%22%2C%227668.jpg%22%2C%227669.jpg%22%2C%227670.jpg%22%2C%227671.jpg%22%2C%227672.jpg%22%2C%227673.jpg%22%2C%227674.jpg%22%2C%227675.jpg%22%2C%227676.jpg%22%2C%227677.jpg%22%2C%227678.jpg%22%2C%227679.jpg%22%2C%227680.jpg%22%2C%227681.jpg%22%2C%227682.jpg%22%2C%227683.jpg%22%2C%227684.jpg%22%2C%227685.jpg%22%2C%227686.jpg%22%2C%227687.jpg%22%2C%227688.jpg%22%2C%227689.jpg%22%2C%227690.jpg%22%2C%227691.jpg%22%2C%227692.jpg%22%2C%227693.jpg%22%2C%227694.jpg%22%2C%227695.jpg%22%2C%227696.jpg%22%2C%227697.jpg%22%2C%227698.jpg%22%2C%227699.jpg%22%2C%227700.jpg%22%2C%227701.jpg%22%2C%227702.jpg%22%2C%227703.jpg%22%2C%227704.jpg%22%2C%227705.jpg%22%2C%227706.jpg%22%2C%227707.jpg%22%2C%227708.jpg%22%2C%227709.jpg%22%2C%227710.jpg%22%2C%227711.jpg%22%2C%227712.jpg%22%2C%227713.jpg%22%2C%227714.jpg%22%2C%227715.jpg%22%2C%227716.jpg%22%2C%227717.jpg%22%2C%227718.jpg%22%2C%227719.jpg%22%2C%227720.jpg%22%2C%227721.jpg%22%2C%227722.jpg%22%2C%227723.jpg%22%2C%227724.jpg%22%2C%227725.jpg%22%2C%227726.jpg%22%2C%227727.jpg%22%2C%227728.jpg%22%2C%227729.jpg%22%2C%227730.jpg%22%2C%227731.jpg%22%2C%227732.jpg%22%2C%227733.jpg%22%2C%227734.jpg%22%2C%227735.jpg%22%2C%227736.jpg%22%2C%227737.jpg%22%2C%227738.jpg%22%2C%227739.jpg%22%2C%227740.jpg%22%2C%227741.jpg%22%2C%227742.jpg%22%2C%227743.jpg%22%2C%227744.jpg%22%2C%227745.jpg%22%2C%227746.jpg%22%2C%227982.jpg%22%2C%227983.jpg%22%2C%227984.jpg%22%2C%227985.jpg%22%2C%227986.jpg%22%2C%227987.jpg%22%2C%227988.jpg%22%2C%227989.jpg%22%2C%227990.jpg%22%2C%227991.jpg%22%2C%227992.jpg%22%2C%227993.jpg%22%2C%227994.jpg%22%2C%227995.jpg%22%2C%227996.jpg%22%2C%227997.jpg%22%2C%227998.jpg%22%2C%227999.jpg%22%2C%228000.jpg%22%2C%228001.jpg%22%2C%228002.jpg%22%2C%228003.jpg%22%2C%228004.jpg%22%2C%228005.jpg%22%2C%228006.jpg%22%2C%228007.jpg%22%2C%228008.jpg%22%2C%228009.jpg%22%2C%228010.jpg%22%2C%228011.jpg%22%2C%228012.jpg%22%2C%228013.jpg%22%2C%228014.jpg%22%2C%228015.jpg%22%2C%228016.jpg%22%2C%228017.jpg%22%2C%228018.jpg%22%2C%228019.jpg%22%2C%228020.jpg%22%2C%228021.jpg%22%2C%228022.jpg%22%2C%228023.jpg%22%2C%228024.jpg%22%2C%228025.jpg%22%2C%228026.jpg%22%2C%228027.jpg%22%2C%228028.jpg%22%2C%228029.jpg%22%2C%228030.jpg%22%2C%228031.jpg%22%2C%228032.jpg%22%2C%228033.jpg%22%2C%228034.jpg%22%2C%228035.jpg%22%2C%228036.jpg%22%2C%228037.jpg%22%2C%228038.jpg%22%2C%228039.jpg%22%2C%228040.jpg%22%2C%228041.jpg%22%2C%228042.jpg%22%2C%228043.jpg%22%2C%228044.jpg%22%2C%228045.jpg%22%2C%228046.jpg%22%2C%228047.jpg%22%2C%228048.jpg%22%2C%228049.jpg%22%2C%228050.jpg%22%2C%228051.jpg%22%2C%228052.jpg%22%2C%228053.jpg%22%2C%228054.jpg%22%2C%228055.jpg%22%2C%228056.jpg%22%2C%228057.jpg%22%2C%228058.jpg%22%2C%228059.jpg%22%2C%228060.jpg%22%2C%228061.jpg%22%2C%228062.jpg%22%2C%228063.jpg%22%2C%228064.jpg%22%2C%228065.jpg%22%2C%228066.jpg%22%2C%228067.jpg%22%2C%228068.jpg%22%2C%228069.jpg%22%2C%228070.jpg%22%2C%228071.jpg%22%2C%228072.jpg%22%2C%228073.jpg%22%2C%228074.jpg%22%2C%228075.jpg%22%2C%228076.jpg%22%2C%228077.jpg%22%2C%228078.jpg%22%2C%228079.jpg%22%2C%228080.jpg%22%2C%228081.jpg%22%2C%228082.jpg%22%2C%228083.jpg%22%2C%228084.jpg%22%2C%228085.jpg%22%2C%228086.jpg%22%2C%228087.jpg%22%2C%228088.jpg%22%2C%228089.jpg%22%2C%228090.jpg%22%2C%228091.jpg%22%2C%228092.jpg%22%2C%228093.jpg%22%2C%228094.jpg%22%2C%228095.jpg%22%2C%228096.jpg%22%2C%228097.jpg%22%2C%228098.jpg%22%2C%228099.jpg%22%2C%228100.jpg%22%2C%228101.jpg%22%2C%228102.jpg%22%2C%228103.jpg%22%2C%228104.jpg%22%2C%228105.jpg%22%2C%228106.jpg%22%2C%228107.jpg%22%2C%228108.jpg%22%2C%228109.jpg%22%2C%228110.jpg%22%2C%228111.jpg%22%2C%228112.jpg%22%2C%228113.jpg%22%2C%228114.jpg%22%2C%228257.jpg%22%2C%228258.jpg%22%2C%228259.jpg%22%2C%228260.jpg%22%2C%228261.jpg%22%2C%228262.jpg%22%2C%228263.jpg%22%2C%228264.jpg%22%2C%228265.jpg%22%2C%228266.jpg%22%2C%228267.jpg%22%2C%228268.jpg%22%2C%228269.jpg%22%2C%228270.jpg%22%2C%228271.jpg%22%2C%228272.jpg%22%2C%228273.jpg%22%2C%228274.jpg%22%2C%228275.jpg%22%2C%228276.jpg%22%2C%228277.jpg%22%2C%228278.jpg%22%2C%228279.jpg%22%2C%228280.jpg%22%2C%228281.jpg%22%2C%228282.jpg%22%2C%228283.jpg%22%2C%228284.jpg%22%2C%228285.jpg%22%2C%228286.jpg%22%2C%228287.jpg%22%2C%228288.jpg%22%2C%228289.jpg%22%2C%228290.jpg%22%2C%228291.jpg%22%2C%228292.jpg%22%2C%228293.jpg%22%2C%228294.jpg%22%2C%228295.jpg%22%2C%228296.jpg%22%2C%228297.jpg%22%2C%228298.jpg%22%2C%228299.jpg%22%2C%228300.jpg%22%2C%228301.jpg%22%2C%228302.jpg%22%2C%228303.jpg%22%2C%228304.jpg%22%2C%228305.jpg%22%2C%228306.jpg%22%2C%228307.jpg%22%2C%228308.jpg%22%2C%228309.jpg%22%2C%228310.jpg%22%2C%228311.jpg%22%2C%228312.jpg%22%2C%228313.jpg%22%2C%228314.jpg%22%2C%228315.jpg%22%2C%228316.jpg%22%2C%228317.jpg%22%2C%228318.jpg%22%2C%228319.jpg%22%2C%228320.jpg%22%2C%228321.jpg%22%2C%228322.jpg%22%2C%228323.jpg%22%2C%228324.jpg%22%2C%228325.jpg%22%2C%228326.jpg%22%2C%228327.jpg%22%2C%228328.jpg%22%2C%228329.jpg%22%2C%228330.jpg%22%2C%228331.jpg%22%2C%228332.jpg%22%2C%228333.jpg%22%2C%228334.jpg%22%2C%228335.jpg%22%2C%228336.jpg%22%2C%228337.jpg%22%2C%228338.jpg%22%2C%228339.jpg%22%2C%228340.jpg%22%2C%228341.jpg%22%2C%228342.jpg%22%2C%228343.jpg%22%2C%228344.jpg%22%2C%228345.jpg%22%2C%228346.jpg%22%2C%228347.jpg%22%2C%228348.jpg%22%2C%228349.jpg%22%2C%228350.jpg%22%2C%228351.jpg%22%2C%228352.jpg%22%2C%228353.jpg%22%2C%228354.jpg%22%2C%228355.jpg%22%2C%228356.jpg%22%2C%228357.jpg%22%2C%228358.jpg%22%2C%228359.jpg%22%2C%228360.jpg%22%2C%228361.jpg%22%2C%228362.jpg%22%2C%228363.jpg%22%2C%228364.jpg%22%2C%228365.jpg%22%2C%228366.jpg%22%2C%228367.jpg%22%2C%228368.jpg%22%2C%228369.jpg%22%2C%228370.jpg%22%2C%228371.jpg%22%2C%228372.jpg%22%2C%228373.jpg%22%2C%228374.jpg%22%2C%228375.jpg%22%2C%228376.jpg%22%2C%228377.jpg%22%2C%228378.jpg%22%2C%228379.jpg%22%2C%228380.jpg%22%2C%228381.jpg%22%2C%228382.jpg%22%2C%228383.jpg%22%2C%228384.jpg%22%2C%228385.jpg%22%2C%228386.jpg%22%2C%228387.jpg%22%2C%228388.jpg%22%2C%228389.jpg%22%2C%228390.jpg%22%2C%228391.jpg%22%2C%228392.jpg%22%2C%228393.jpg%22%2C%228394.jpg%22%2C%228395.jpg%22%2C%228396.jpg%22%2C%228397.jpg%22%2C%228398.jpg%22%2C%228399.jpg%22%2C%228400.jpg%22%2C%228401.jpg%22%2C%228402.jpg%22%2C%228403.jpg%22%2C%228404.jpg%22%2C%228405.jpg%22%5D",
            filename="train10.zip",
            md5=False,
            checksum="c9ceb57376faf64dab817e6a18edffaeab928fef69336e960081ffe8a5dc848f",
        ),
        DataLocation(
            url="https://cloud.cs.uni-tuebingen.de/public.php/dav/files/ZZxX65FGnQ8zjBP/Compressed%20Version/images/train/?accept=zip&files=%5B%228406.jpg%22%2C%228407.jpg%22%2C%228408.jpg%22%2C%228409.jpg%22%2C%228410.jpg%22%2C%228411.jpg%22%2C%228412.jpg%22%2C%228413.jpg%22%2C%228414.jpg%22%2C%228415.jpg%22%2C%228416.jpg%22%2C%228417.jpg%22%2C%228418.jpg%22%2C%228419.jpg%22%2C%228420.jpg%22%2C%228421.jpg%22%2C%228422.jpg%22%2C%228423.jpg%22%2C%228424.jpg%22%2C%228425.jpg%22%2C%228426.jpg%22%2C%228427.jpg%22%2C%228428.jpg%22%2C%228429.jpg%22%2C%228430.jpg%22%2C%228431.jpg%22%2C%228432.jpg%22%2C%228433.jpg%22%2C%228434.jpg%22%2C%228435.jpg%22%2C%228436.jpg%22%2C%228437.jpg%22%2C%228438.jpg%22%2C%228439.jpg%22%2C%228440.jpg%22%2C%228441.jpg%22%2C%228442.jpg%22%2C%228443.jpg%22%2C%228444.jpg%22%2C%228445.jpg%22%2C%228446.jpg%22%2C%228447.jpg%22%2C%228448.jpg%22%2C%228449.jpg%22%2C%228450.jpg%22%2C%228451.jpg%22%2C%228452.jpg%22%2C%228453.jpg%22%2C%228454.jpg%22%2C%228455.jpg%22%2C%228456.jpg%22%2C%228457.jpg%22%2C%228458.jpg%22%2C%228459.jpg%22%2C%228460.jpg%22%2C%228461.jpg%22%2C%228462.jpg%22%2C%228463.jpg%22%2C%228464.jpg%22%2C%228465.jpg%22%2C%228466.jpg%22%2C%228467.jpg%22%2C%228468.jpg%22%2C%228469.jpg%22%2C%228470.jpg%22%2C%228471.jpg%22%2C%228472.jpg%22%2C%228473.jpg%22%2C%228474.jpg%22%2C%228475.jpg%22%2C%228476.jpg%22%2C%228477.jpg%22%2C%228478.jpg%22%2C%228479.jpg%22%2C%228480.jpg%22%2C%228481.jpg%22%2C%228482.jpg%22%2C%228483.jpg%22%2C%228484.jpg%22%2C%228485.jpg%22%2C%228486.jpg%22%2C%228487.jpg%22%2C%228488.jpg%22%2C%228489.jpg%22%2C%228490.jpg%22%2C%228491.jpg%22%2C%228492.jpg%22%2C%228493.jpg%22%2C%228494.jpg%22%2C%228495.jpg%22%2C%228496.jpg%22%2C%228497.jpg%22%2C%228498.jpg%22%2C%228499.jpg%22%2C%228500.jpg%22%2C%228501.jpg%22%2C%228502.jpg%22%2C%228503.jpg%22%2C%228504.jpg%22%2C%228505.jpg%22%2C%228506.jpg%22%2C%228507.jpg%22%2C%228508.jpg%22%2C%228509.jpg%22%2C%228510.jpg%22%2C%228511.jpg%22%2C%228512.jpg%22%2C%228513.jpg%22%2C%228514.jpg%22%2C%228515.jpg%22%2C%228516.jpg%22%2C%228517.jpg%22%2C%228518.jpg%22%2C%228519.jpg%22%2C%228520.jpg%22%2C%228521.jpg%22%2C%228522.jpg%22%2C%228523.jpg%22%2C%228524.jpg%22%2C%228525.jpg%22%2C%228526.jpg%22%2C%228527.jpg%22%2C%228528.jpg%22%2C%228529.jpg%22%2C%228530.jpg%22%2C%228531.jpg%22%2C%228532.jpg%22%2C%228533.jpg%22%2C%228534.jpg%22%2C%228535.jpg%22%2C%228536.jpg%22%2C%228537.jpg%22%2C%228538.jpg%22%2C%228539.jpg%22%2C%228540.jpg%22%2C%228541.jpg%22%2C%228542.jpg%22%2C%228543.jpg%22%2C%228544.jpg%22%2C%228545.jpg%22%2C%228546.jpg%22%2C%228547.jpg%22%2C%228548.jpg%22%2C%228549.jpg%22%2C%228550.jpg%22%2C%228551.jpg%22%2C%228552.jpg%22%2C%228553.jpg%22%2C%228554.jpg%22%2C%228555.jpg%22%2C%228556.jpg%22%2C%228557.jpg%22%2C%228558.jpg%22%2C%228559.jpg%22%2C%228560.jpg%22%2C%228561.jpg%22%2C%228562.jpg%22%2C%228563.jpg%22%2C%228564.jpg%22%2C%228565.jpg%22%2C%228566.jpg%22%2C%228567.jpg%22%2C%228568.jpg%22%2C%228569.jpg%22%2C%228570.jpg%22%2C%228571.jpg%22%2C%228572.jpg%22%2C%228573.jpg%22%2C%228574.jpg%22%2C%228575.jpg%22%2C%228576.jpg%22%2C%228577.jpg%22%2C%228578.jpg%22%2C%228579.jpg%22%2C%228580.jpg%22%2C%228581.jpg%22%2C%228582.jpg%22%2C%228583.jpg%22%2C%228584.jpg%22%2C%228585.jpg%22%2C%228586.jpg%22%2C%228587.jpg%22%2C%228588.jpg%22%2C%228589.jpg%22%2C%228590.jpg%22%2C%228591.jpg%22%2C%228592.jpg%22%2C%228593.jpg%22%2C%228594.jpg%22%2C%228595.jpg%22%2C%228596.jpg%22%2C%228597.jpg%22%2C%228598.jpg%22%2C%228599.jpg%22%2C%228600.jpg%22%2C%228601.jpg%22%2C%228602.jpg%22%2C%228603.jpg%22%2C%228604.jpg%22%2C%228605.jpg%22%2C%228606.jpg%22%2C%228607.jpg%22%2C%228608.jpg%22%2C%228609.jpg%22%2C%228610.jpg%22%2C%228611.jpg%22%2C%228612.jpg%22%2C%228613.jpg%22%2C%228614.jpg%22%2C%228615.jpg%22%2C%228616.jpg%22%2C%228617.jpg%22%2C%228859.jpg%22%2C%228860.jpg%22%2C%228861.jpg%22%2C%228862.jpg%22%2C%228863.jpg%22%2C%228864.jpg%22%2C%228865.jpg%22%2C%228866.jpg%22%2C%228867.jpg%22%2C%228868.jpg%22%2C%228869.jpg%22%2C%228870.jpg%22%2C%228871.jpg%22%2C%228872.jpg%22%2C%228873.jpg%22%2C%228874.jpg%22%2C%228875.jpg%22%2C%228876.jpg%22%2C%228877.jpg%22%2C%228878.jpg%22%2C%228879.jpg%22%2C%228880.jpg%22%2C%228881.jpg%22%2C%228882.jpg%22%2C%228883.jpg%22%2C%228884.jpg%22%2C%228885.jpg%22%2C%228886.jpg%22%2C%228887.jpg%22%2C%228888.jpg%22%2C%228889.jpg%22%2C%228890.jpg%22%2C%228891.jpg%22%2C%228892.jpg%22%2C%228893.jpg%22%2C%228894.jpg%22%2C%228895.jpg%22%2C%228896.jpg%22%2C%228897.jpg%22%2C%228898.jpg%22%2C%228899.jpg%22%2C%228900.jpg%22%2C%228901.jpg%22%2C%228902.jpg%22%2C%228903.jpg%22%2C%228904.jpg%22%2C%228905.jpg%22%2C%228906.jpg%22%2C%228907.jpg%22%2C%228908.jpg%22%2C%228909.jpg%22%2C%228910.jpg%22%2C%228911.jpg%22%2C%228912.jpg%22%2C%228913.jpg%22%2C%228914.jpg%22%2C%228915.jpg%22%2C%228916.jpg%22%2C%228917.jpg%22%2C%228918.jpg%22%2C%228919.jpg%22%2C%228920.jpg%22%2C%228921.jpg%22%2C%228922.jpg%22%2C%228923.jpg%22%2C%228924.jpg%22%2C%228925.jpg%22%2C%228926.jpg%22%2C%228927.jpg%22%2C%228928.jpg%22%2C%228929.jpg%22%2C%228930.jpg%22%2C%228931.jpg%22%2C%228932.jpg%22%2C%228933.jpg%22%2C%228934.jpg%22%2C%228935.jpg%22%2C%228936.jpg%22%2C%228937.jpg%22%2C%228938.jpg%22%2C%228939.jpg%22%2C%228940.jpg%22%2C%228941.jpg%22%2C%228942.jpg%22%2C%228943.jpg%22%2C%228944.jpg%22%2C%228945.jpg%22%2C%228946.jpg%22%2C%228947.jpg%22%2C%228948.jpg%22%2C%228949.jpg%22%2C%228950.jpg%22%2C%228951.jpg%22%2C%228952.jpg%22%2C%228953.jpg%22%2C%228954.jpg%22%2C%228955.jpg%22%2C%228956.jpg%22%2C%228957.jpg%22%2C%228958.jpg%22%2C%228959.jpg%22%2C%228960.jpg%22%2C%228961.jpg%22%2C%228962.jpg%22%2C%228963.jpg%22%2C%228964.jpg%22%2C%228965.jpg%22%2C%228966.jpg%22%2C%228967.jpg%22%2C%228968.jpg%22%2C%228969.jpg%22%2C%228970.jpg%22%2C%228971.jpg%22%2C%228972.jpg%22%2C%228973.jpg%22%2C%228974.jpg%22%2C%228975.jpg%22%2C%228976.jpg%22%2C%228977.jpg%22%2C%228978.jpg%22%2C%228979.jpg%22%2C%228980.jpg%22%2C%228981.jpg%22%2C%228982.jpg%22%2C%228983.jpg%22%2C%228984.jpg%22%2C%228985.jpg%22%2C%228986.jpg%22%2C%228987.jpg%22%2C%228988.jpg%22%2C%228989.jpg%22%2C%228990.jpg%22%2C%228991.jpg%22%2C%228992.jpg%22%2C%228993.jpg%22%2C%228994.jpg%22%2C%228995.jpg%22%2C%228996.jpg%22%2C%228997.jpg%22%2C%228998.jpg%22%2C%228999.jpg%22%2C%229000.jpg%22%2C%229001.jpg%22%2C%229002.jpg%22%2C%229003.jpg%22%2C%229004.jpg%22%2C%229005.jpg%22%2C%229006.jpg%22%2C%229007.jpg%22%2C%229008.jpg%22%2C%229009.jpg%22%2C%229010.jpg%22%2C%229011.jpg%22%2C%229012.jpg%22%2C%229013.jpg%22%2C%229014.jpg%22%2C%229015.jpg%22%2C%229016.jpg%22%2C%229017.jpg%22%2C%229018.jpg%22%2C%229019.jpg%22%2C%229020.jpg%22%2C%229021.jpg%22%2C%229022.jpg%22%2C%229023.jpg%22%2C%229024.jpg%22%2C%229025.jpg%22%2C%229026.jpg%22%2C%229027.jpg%22%2C%229028.jpg%22%2C%229029.jpg%22%2C%229030.jpg%22%2C%229031.jpg%22%2C%229032.jpg%22%2C%229033.jpg%22%2C%229034.jpg%22%2C%229035.jpg%22%2C%229036.jpg%22%2C%229037.jpg%22%2C%229038.jpg%22%2C%229039.jpg%22%2C%229040.jpg%22%2C%229041.jpg%22%2C%229042.jpg%22%2C%229043.jpg%22%2C%229044.jpg%22%2C%229045.jpg%22%2C%229046.jpg%22%2C%229047.jpg%22%2C%229048.jpg%22%2C%229049.jpg%22%2C%229050.jpg%22%2C%229051.jpg%22%2C%229052.jpg%22%2C%229053.jpg%22%2C%229054.jpg%22%2C%229055.jpg%22%2C%229056.jpg%22%2C%229057.jpg%22%2C%229058.jpg%22%2C%229059.jpg%22%2C%229060.jpg%22%2C%229061.jpg%22%2C%229062.jpg%22%2C%229063.jpg%22%2C%229064.jpg%22%2C%229065.jpg%22%2C%229066.jpg%22%2C%229067.jpg%22%2C%229068.jpg%22%2C%229069.jpg%22%2C%229070.jpg%22%2C%229071.jpg%22%2C%229072.jpg%22%2C%229073.jpg%22%2C%229074.jpg%22%2C%229075.jpg%22%2C%229076.jpg%22%2C%229077.jpg%22%2C%229078.jpg%22%2C%229079.jpg%22%2C%229080.jpg%22%2C%229081.jpg%22%2C%229082.jpg%22%2C%229083.jpg%22%2C%229084.jpg%22%2C%229085.jpg%22%2C%229086.jpg%22%2C%229087.jpg%22%2C%229088.jpg%22%2C%229089.jpg%22%2C%229090.jpg%22%2C%229091.jpg%22%2C%229092.jpg%22%2C%229093.jpg%22%2C%229094.jpg%22%2C%229095.jpg%22%2C%229096.jpg%22%5D",
            filename="train11.zip",
            md5=False,
            checksum="7f64b7c3455c1bf466b528ccbae3098b311046d44518d9fd429fa3ff8682d9ae",
        ),
        DataLocation(
            url="https://cloud.cs.uni-tuebingen.de/public.php/dav/files/ZZxX65FGnQ8zjBP/Compressed%20Version/images/train/?accept=zip&files=%5B%229097.jpg%22%2C%229098.jpg%22%2C%229099.jpg%22%2C%229100.jpg%22%2C%229101.jpg%22%2C%229102.jpg%22%2C%229103.jpg%22%2C%229104.jpg%22%2C%229105.jpg%22%2C%229106.jpg%22%2C%229107.jpg%22%2C%229108.jpg%22%2C%229109.jpg%22%2C%229110.jpg%22%2C%229111.jpg%22%2C%229112.jpg%22%2C%229113.jpg%22%2C%229114.jpg%22%2C%229115.jpg%22%2C%229116.jpg%22%2C%229117.jpg%22%2C%229118.jpg%22%2C%229119.jpg%22%2C%229120.jpg%22%2C%229121.jpg%22%2C%229122.jpg%22%2C%229123.jpg%22%2C%229124.jpg%22%2C%229125.jpg%22%2C%229126.jpg%22%2C%229127.jpg%22%2C%229128.jpg%22%2C%229129.jpg%22%2C%229130.jpg%22%2C%229131.jpg%22%2C%229132.jpg%22%2C%229133.jpg%22%2C%229134.jpg%22%2C%229135.jpg%22%2C%229136.jpg%22%2C%229137.jpg%22%2C%229138.jpg%22%2C%229139.jpg%22%2C%229140.jpg%22%2C%229141.jpg%22%2C%229142.jpg%22%2C%229143.jpg%22%2C%229144.jpg%22%2C%229145.jpg%22%2C%229146.jpg%22%2C%229147.jpg%22%2C%229148.jpg%22%2C%229149.jpg%22%2C%229150.jpg%22%2C%229151.jpg%22%2C%229152.jpg%22%2C%229153.jpg%22%2C%229154.jpg%22%2C%229155.jpg%22%2C%229156.jpg%22%2C%229157.jpg%22%2C%229158.jpg%22%2C%229159.jpg%22%2C%229160.jpg%22%2C%229161.jpg%22%2C%229162.jpg%22%2C%229163.jpg%22%2C%229164.jpg%22%2C%229165.jpg%22%2C%229166.jpg%22%2C%229167.jpg%22%2C%229168.jpg%22%2C%229169.jpg%22%2C%229170.jpg%22%2C%229171.jpg%22%2C%229172.jpg%22%2C%229173.jpg%22%2C%229174.jpg%22%2C%229175.jpg%22%2C%229176.jpg%22%2C%229177.jpg%22%2C%229178.jpg%22%2C%229179.jpg%22%2C%229180.jpg%22%2C%229181.jpg%22%2C%229182.jpg%22%2C%229183.jpg%22%2C%229184.jpg%22%2C%229185.jpg%22%2C%229186.jpg%22%2C%229187.jpg%22%2C%229188.jpg%22%2C%229189.jpg%22%2C%229190.jpg%22%2C%229191.jpg%22%2C%229192.jpg%22%2C%229193.jpg%22%2C%229194.jpg%22%2C%229195.jpg%22%2C%229196.jpg%22%2C%229197.jpg%22%2C%229198.jpg%22%2C%229199.jpg%22%2C%229200.jpg%22%2C%229201.jpg%22%2C%229202.jpg%22%2C%229203.jpg%22%2C%229204.jpg%22%2C%229205.jpg%22%2C%229206.jpg%22%2C%229207.jpg%22%2C%229208.jpg%22%2C%229209.jpg%22%2C%229210.jpg%22%2C%229211.jpg%22%2C%229212.jpg%22%2C%229213.jpg%22%2C%229214.jpg%22%2C%229215.jpg%22%2C%229501.jpg%22%2C%229502.jpg%22%2C%229503.jpg%22%2C%229504.jpg%22%2C%229505.jpg%22%2C%229506.jpg%22%2C%229533.jpg%22%2C%229534.jpg%22%2C%229535.jpg%22%2C%229536.jpg%22%2C%229537.jpg%22%2C%229538.jpg%22%2C%229539.jpg%22%2C%229540.jpg%22%2C%229541.jpg%22%2C%229542.jpg%22%2C%229543.jpg%22%2C%229544.jpg%22%2C%229545.jpg%22%2C%229546.jpg%22%2C%229547.jpg%22%2C%229548.jpg%22%2C%229549.jpg%22%2C%229550.jpg%22%2C%229551.jpg%22%2C%229552.jpg%22%2C%229553.jpg%22%2C%229554.jpg%22%2C%229555.jpg%22%2C%229556.jpg%22%2C%229557.jpg%22%2C%229558.jpg%22%2C%229559.jpg%22%2C%229560.jpg%22%2C%229561.jpg%22%2C%229562.jpg%22%2C%229563.jpg%22%2C%229564.jpg%22%2C%229565.jpg%22%2C%229566.jpg%22%2C%229567.jpg%22%2C%229568.jpg%22%2C%229569.jpg%22%2C%229570.jpg%22%2C%229571.jpg%22%2C%229572.jpg%22%2C%229573.jpg%22%2C%229574.jpg%22%2C%229575.jpg%22%2C%229576.jpg%22%2C%229577.jpg%22%2C%229578.jpg%22%2C%229579.jpg%22%2C%229580.jpg%22%2C%229581.jpg%22%2C%229582.jpg%22%2C%229583.jpg%22%2C%229584.jpg%22%2C%229585.jpg%22%2C%229586.jpg%22%2C%229587.jpg%22%2C%229588.jpg%22%2C%229589.jpg%22%2C%229590.jpg%22%2C%229591.jpg%22%2C%229592.jpg%22%2C%229593.jpg%22%2C%229594.jpg%22%2C%229595.jpg%22%2C%229596.jpg%22%2C%229597.jpg%22%2C%229598.jpg%22%2C%229599.jpg%22%2C%229600.jpg%22%2C%229601.jpg%22%2C%229602.jpg%22%2C%229603.jpg%22%2C%229604.jpg%22%2C%229605.jpg%22%2C%229606.jpg%22%2C%229607.jpg%22%2C%229608.jpg%22%2C%229609.jpg%22%2C%229610.jpg%22%2C%229611.jpg%22%2C%229612.jpg%22%2C%229613.jpg%22%2C%229614.jpg%22%2C%229615.jpg%22%2C%229616.jpg%22%2C%229617.jpg%22%2C%229618.jpg%22%2C%229619.jpg%22%2C%229620.jpg%22%2C%229621.jpg%22%2C%229622.jpg%22%2C%229623.jpg%22%2C%229624.jpg%22%2C%229625.jpg%22%2C%229626.jpg%22%2C%229627.jpg%22%2C%229628.jpg%22%2C%229629.jpg%22%2C%229630.jpg%22%2C%229631.jpg%22%2C%229632.jpg%22%2C%229633.jpg%22%2C%229634.jpg%22%2C%229635.jpg%22%2C%229636.jpg%22%2C%229637.jpg%22%2C%229638.jpg%22%2C%229639.jpg%22%2C%229640.jpg%22%2C%229641.jpg%22%2C%229642.jpg%22%2C%229643.jpg%22%2C%229644.jpg%22%2C%229645.jpg%22%2C%229646.jpg%22%2C%229647.jpg%22%2C%229648.jpg%22%2C%229649.jpg%22%2C%229650.jpg%22%2C%229651.jpg%22%2C%229652.jpg%22%2C%229653.jpg%22%2C%229654.jpg%22%2C%229655.jpg%22%2C%229656.jpg%22%2C%229657.jpg%22%2C%229658.jpg%22%2C%229659.jpg%22%2C%229660.jpg%22%2C%229661.jpg%22%2C%229662.jpg%22%2C%229663.jpg%22%2C%229664.jpg%22%2C%229665.jpg%22%2C%229666.jpg%22%2C%229667.jpg%22%2C%229668.jpg%22%2C%229669.jpg%22%2C%229670.jpg%22%2C%229671.jpg%22%2C%229672.jpg%22%2C%229673.jpg%22%2C%229674.jpg%22%2C%229675.jpg%22%2C%229676.jpg%22%2C%229677.jpg%22%2C%229678.jpg%22%2C%229679.jpg%22%2C%229680.jpg%22%2C%229681.jpg%22%2C%229682.jpg%22%2C%229683.jpg%22%2C%229684.jpg%22%2C%229685.jpg%22%2C%229686.jpg%22%2C%229687.jpg%22%2C%229688.jpg%22%2C%229689.jpg%22%2C%229690.jpg%22%2C%229691.jpg%22%2C%229692.jpg%22%2C%229693.jpg%22%2C%229694.jpg%22%2C%229695.jpg%22%2C%229696.jpg%22%2C%229697.jpg%22%2C%229698.jpg%22%2C%229699.jpg%22%2C%229700.jpg%22%2C%229701.jpg%22%2C%229702.jpg%22%2C%229703.jpg%22%2C%229704.jpg%22%2C%229705.jpg%22%2C%229706.jpg%22%2C%229707.jpg%22%2C%229708.jpg%22%2C%229709.jpg%22%2C%229710.jpg%22%2C%229711.jpg%22%2C%229712.jpg%22%2C%229713.jpg%22%2C%229714.jpg%22%2C%229715.jpg%22%2C%229716.jpg%22%2C%229717.jpg%22%2C%229718.jpg%22%2C%229719.jpg%22%2C%229720.jpg%22%2C%229721.jpg%22%2C%229722.jpg%22%2C%229723.jpg%22%2C%229724.jpg%22%2C%229725.jpg%22%2C%229726.jpg%22%2C%229727.jpg%22%2C%229728.jpg%22%2C%229729.jpg%22%2C%229730.jpg%22%2C%229731.jpg%22%2C%229732.jpg%22%2C%229733.jpg%22%2C%229734.jpg%22%2C%229735.jpg%22%2C%229736.jpg%22%2C%229737.jpg%22%2C%229738.jpg%22%2C%229739.jpg%22%2C%229740.jpg%22%2C%229741.jpg%22%2C%229742.jpg%22%2C%229743.jpg%22%2C%229744.jpg%22%2C%229745.jpg%22%2C%229746.jpg%22%2C%229747.jpg%22%2C%229748.jpg%22%2C%229749.jpg%22%2C%229750.jpg%22%2C%229751.jpg%22%2C%229752.jpg%22%2C%229753.jpg%22%2C%229754.jpg%22%2C%229755.jpg%22%2C%229756.jpg%22%2C%229757.jpg%22%2C%229758.jpg%22%2C%229759.jpg%22%2C%229760.jpg%22%2C%229761.jpg%22%2C%229762.jpg%22%2C%229763.jpg%22%2C%229764.jpg%22%2C%229765.jpg%22%2C%229766.jpg%22%2C%229767.jpg%22%2C%229768.jpg%22%2C%229769.jpg%22%2C%229770.jpg%22%2C%229771.jpg%22%2C%229772.jpg%22%2C%229773.jpg%22%2C%229774.jpg%22%2C%229775.jpg%22%2C%229776.jpg%22%2C%229777.jpg%22%2C%229778.jpg%22%2C%229779.jpg%22%2C%229780.jpg%22%2C%229781.jpg%22%2C%229782.jpg%22%2C%229783.jpg%22%2C%229784.jpg%22%2C%229785.jpg%22%2C%229786.jpg%22%2C%229787.jpg%22%2C%229788.jpg%22%2C%229789.jpg%22%2C%229790.jpg%22%2C%229791.jpg%22%2C%229792.jpg%22%2C%229793.jpg%22%2C%229794.jpg%22%2C%229795.jpg%22%2C%229796.jpg%22%2C%229797.jpg%22%2C%229798.jpg%22%2C%229799.jpg%22%2C%229800.jpg%22%2C%229801.jpg%22%2C%229802.jpg%22%2C%229803.jpg%22%2C%229804.jpg%22%2C%229805.jpg%22%2C%229806.jpg%22%2C%229807.jpg%22%2C%229808.jpg%22%2C%229809.jpg%22%2C%229810.jpg%22%2C%229811.jpg%22%2C%2210119.jpg%22%2C%2210120.jpg%22%2C%2210121.jpg%22%2C%2210122.jpg%22%2C%2210123.jpg%22%2C%2210124.jpg%22%2C%2210125.jpg%22%2C%2210126.jpg%22%2C%2210127.jpg%22%2C%2210128.jpg%22%2C%2210129.jpg%22%2C%2210130.jpg%22%2C%2210131.jpg%22%2C%2210132.jpg%22%2C%2210133.jpg%22%2C%2210134.jpg%22%2C%2210135.jpg%22%2C%2210136.jpg%22%2C%2210137.jpg%22%2C%2210138.jpg%22%2C%2210139.jpg%22%2C%2210140.jpg%22%2C%2210141.jpg%22%2C%2210142.jpg%22%2C%2210143.jpg%22%2C%2210144.jpg%22%2C%2210145.jpg%22%2C%2210146.jpg%22%2C%2210147.jpg%22%2C%2210148.jpg%22%2C%2210149.jpg%22%2C%2210150.jpg%22%2C%2210151.jpg%22%2C%2210152.jpg%22%2C%2210153.jpg%22%2C%2210154.jpg%22%2C%2210155.jpg%22%2C%2210156.jpg%22%2C%2210157.jpg%22%2C%2210158.jpg%22%2C%2210159.jpg%22%2C%2210160.jpg%22%2C%2210161.jpg%22%2C%2210162.jpg%22%2C%2210163.jpg%22%2C%2210164.jpg%22%5D",
            filename="train12.zip",
            md5=False,
            checksum="32ab89ffc68018aa621f2eae2995fe0c4334a83891a0e5edec02d43ab4c0303a",
        ),
        DataLocation(
            url="https://cloud.cs.uni-tuebingen.de/public.php/dav/files/ZZxX65FGnQ8zjBP/Compressed%20Version/images/train/?accept=zip&files=%5B%2210165.jpg%22%2C%2210166.jpg%22%2C%2210167.jpg%22%2C%2210168.jpg%22%2C%2210169.jpg%22%2C%2210170.jpg%22%2C%2210171.jpg%22%2C%2210172.jpg%22%2C%2210173.jpg%22%2C%2210174.jpg%22%2C%2210175.jpg%22%2C%2210176.jpg%22%2C%2210177.jpg%22%2C%2210178.jpg%22%2C%2210179.jpg%22%2C%2210180.jpg%22%2C%2210181.jpg%22%2C%2210182.jpg%22%2C%2210183.jpg%22%2C%2210184.jpg%22%2C%2210185.jpg%22%2C%2210186.jpg%22%2C%2210187.jpg%22%2C%2210188.jpg%22%2C%2210189.jpg%22%2C%2210190.jpg%22%2C%2210191.jpg%22%2C%2210192.jpg%22%2C%2210193.jpg%22%2C%2210194.jpg%22%2C%2210195.jpg%22%2C%2210196.jpg%22%2C%2210197.jpg%22%2C%2210198.jpg%22%2C%2210199.jpg%22%2C%2210200.jpg%22%2C%2210201.jpg%22%2C%2210202.jpg%22%2C%2210203.jpg%22%2C%2210204.jpg%22%2C%2210205.jpg%22%2C%2210206.jpg%22%2C%2210207.jpg%22%2C%2210208.jpg%22%2C%2210209.jpg%22%2C%2210210.jpg%22%2C%2210211.jpg%22%2C%2210212.jpg%22%2C%2210213.jpg%22%2C%2210214.jpg%22%2C%2210215.jpg%22%2C%2210216.jpg%22%2C%2210217.jpg%22%2C%2210218.jpg%22%2C%2210219.jpg%22%2C%2210220.jpg%22%2C%2210221.jpg%22%2C%2210222.jpg%22%2C%2210223.jpg%22%2C%2210224.jpg%22%2C%2210225.jpg%22%2C%2210226.jpg%22%2C%2210227.jpg%22%2C%2210228.jpg%22%2C%2210229.jpg%22%2C%2210230.jpg%22%2C%2210231.jpg%22%2C%2210232.jpg%22%2C%2210233.jpg%22%2C%2210234.jpg%22%2C%2210235.jpg%22%2C%2210236.jpg%22%2C%2210237.jpg%22%2C%2210238.jpg%22%2C%2210239.jpg%22%2C%2210240.jpg%22%2C%2210241.jpg%22%2C%2210242.jpg%22%2C%2210243.jpg%22%2C%2210244.jpg%22%2C%2210245.jpg%22%2C%2210246.jpg%22%2C%2210247.jpg%22%2C%2210248.jpg%22%2C%2210249.jpg%22%2C%2210250.jpg%22%2C%2210251.jpg%22%2C%2210252.jpg%22%2C%2210253.jpg%22%2C%2210254.jpg%22%2C%2210255.jpg%22%2C%2210256.jpg%22%2C%2210257.jpg%22%2C%2210258.jpg%22%2C%2210259.jpg%22%2C%2210260.jpg%22%2C%2210261.jpg%22%2C%2210262.jpg%22%2C%2210263.jpg%22%2C%2210264.jpg%22%2C%2210265.jpg%22%2C%2210266.jpg%22%2C%2210267.jpg%22%2C%2210268.jpg%22%2C%2210269.jpg%22%2C%2210270.jpg%22%2C%2210271.jpg%22%2C%2210272.jpg%22%2C%2210273.jpg%22%2C%2210274.jpg%22%2C%2210275.jpg%22%2C%2210276.jpg%22%2C%2210277.jpg%22%2C%2210278.jpg%22%2C%2210279.jpg%22%2C%2210280.jpg%22%2C%2210281.jpg%22%2C%2210282.jpg%22%2C%2210283.jpg%22%2C%2210284.jpg%22%2C%2210285.jpg%22%2C%2210286.jpg%22%2C%2210287.jpg%22%2C%2210288.jpg%22%2C%2210289.jpg%22%2C%2210290.jpg%22%2C%2210291.jpg%22%2C%2210292.jpg%22%2C%2210293.jpg%22%2C%2210294.jpg%22%2C%2210295.jpg%22%2C%2210296.jpg%22%2C%2210297.jpg%22%2C%2210298.jpg%22%2C%2210299.jpg%22%2C%2210300.jpg%22%2C%2210301.jpg%22%2C%2210302.jpg%22%2C%2210303.jpg%22%2C%2210304.jpg%22%2C%2210305.jpg%22%2C%2210306.jpg%22%2C%2210307.jpg%22%2C%2210308.jpg%22%2C%2210309.jpg%22%2C%2210310.jpg%22%2C%2210311.jpg%22%2C%2210312.jpg%22%2C%2210313.jpg%22%2C%2210314.jpg%22%2C%2210315.jpg%22%2C%2210316.jpg%22%2C%2210317.jpg%22%2C%2210318.jpg%22%2C%2210319.jpg%22%2C%2210320.jpg%22%2C%2210321.jpg%22%2C%2210322.jpg%22%2C%2210323.jpg%22%2C%2210324.jpg%22%2C%2210325.jpg%22%2C%2210326.jpg%22%2C%2210327.jpg%22%2C%2210328.jpg%22%2C%2210329.jpg%22%2C%2210330.jpg%22%2C%2210331.jpg%22%2C%2210332.jpg%22%2C%2210333.jpg%22%2C%2210334.jpg%22%2C%2210335.jpg%22%2C%2210336.jpg%22%2C%2210337.jpg%22%2C%2210338.jpg%22%2C%2210339.jpg%22%2C%2210340.jpg%22%2C%2210341.jpg%22%2C%2210342.jpg%22%2C%2210343.jpg%22%2C%2210344.jpg%22%2C%2210345.jpg%22%2C%2210346.jpg%22%2C%2210347.jpg%22%2C%2210348.jpg%22%2C%2210349.jpg%22%2C%2210350.jpg%22%2C%2210351.jpg%22%2C%2210352.jpg%22%2C%2210353.jpg%22%2C%2210354.jpg%22%2C%2210355.jpg%22%2C%2210356.jpg%22%2C%2210357.jpg%22%2C%2210358.jpg%22%2C%2210359.jpg%22%2C%2210360.jpg%22%2C%2210361.jpg%22%2C%2210362.jpg%22%2C%2210363.jpg%22%2C%2210364.jpg%22%2C%2210365.jpg%22%2C%2210366.jpg%22%2C%2210367.jpg%22%2C%2210368.jpg%22%2C%2210369.jpg%22%2C%2210370.jpg%22%2C%2210371.jpg%22%2C%2210372.jpg%22%2C%2210373.jpg%22%2C%2210374.jpg%22%2C%2210375.jpg%22%2C%2210376.jpg%22%2C%2210377.jpg%22%2C%2210378.jpg%22%2C%2210379.jpg%22%2C%2210380.jpg%22%2C%2210381.jpg%22%2C%2210382.jpg%22%2C%2210383.jpg%22%2C%2210384.jpg%22%2C%2210385.jpg%22%2C%2210386.jpg%22%2C%2210387.jpg%22%2C%2210388.jpg%22%2C%2210389.jpg%22%2C%2210390.jpg%22%2C%2210391.jpg%22%2C%2210392.jpg%22%2C%2210393.jpg%22%2C%2210394.jpg%22%2C%2210395.jpg%22%2C%2210396.jpg%22%2C%2210397.jpg%22%2C%2210398.jpg%22%2C%2210399.jpg%22%2C%2210400.jpg%22%2C%2210401.jpg%22%2C%2210402.jpg%22%2C%2210403.jpg%22%2C%2210404.jpg%22%2C%2210405.jpg%22%2C%2210406.jpg%22%2C%2210407.jpg%22%2C%2210408.jpg%22%2C%2210409.jpg%22%2C%2210410.jpg%22%2C%2210411.jpg%22%2C%2210412.jpg%22%2C%2210413.jpg%22%2C%2210414.jpg%22%2C%2210415.jpg%22%2C%2210617.jpg%22%2C%2210618.jpg%22%2C%2210619.jpg%22%2C%2210620.jpg%22%2C%2210621.jpg%22%2C%2210622.jpg%22%2C%2210623.jpg%22%2C%2210624.jpg%22%2C%2210625.jpg%22%2C%2210626.jpg%22%2C%2210627.jpg%22%2C%2210628.jpg%22%2C%2210629.jpg%22%2C%2210630.jpg%22%2C%2210631.jpg%22%2C%2210632.jpg%22%2C%2210633.jpg%22%2C%2210634.jpg%22%2C%2210635.jpg%22%2C%2210636.jpg%22%2C%2210637.jpg%22%2C%2210638.jpg%22%2C%2210639.jpg%22%2C%2210640.jpg%22%2C%2210641.jpg%22%2C%2210642.jpg%22%2C%2210643.jpg%22%2C%2210644.jpg%22%2C%2210645.jpg%22%2C%2210646.jpg%22%2C%2210647.jpg%22%2C%2210648.jpg%22%2C%2210649.jpg%22%2C%2210650.jpg%22%2C%2210651.jpg%22%2C%2210652.jpg%22%2C%2210653.jpg%22%2C%2210654.jpg%22%2C%2210655.jpg%22%2C%2210656.jpg%22%2C%2210657.jpg%22%2C%2210658.jpg%22%2C%2210659.jpg%22%2C%2210660.jpg%22%2C%2210661.jpg%22%2C%2210662.jpg%22%2C%2210663.jpg%22%2C%2210664.jpg%22%2C%2210665.jpg%22%2C%2210666.jpg%22%2C%2210667.jpg%22%2C%2210668.jpg%22%2C%2210669.jpg%22%2C%2210670.jpg%22%2C%2210671.jpg%22%2C%2210672.jpg%22%2C%2210673.jpg%22%2C%2210674.jpg%22%2C%2210675.jpg%22%2C%2210676.jpg%22%2C%2210677.jpg%22%2C%2210678.jpg%22%2C%2210679.jpg%22%2C%2210680.jpg%22%2C%2210681.jpg%22%2C%2210682.jpg%22%2C%2210683.jpg%22%2C%2210684.jpg%22%2C%2210685.jpg%22%2C%2210686.jpg%22%2C%2210687.jpg%22%2C%2210688.jpg%22%2C%2210689.jpg%22%2C%2210690.jpg%22%2C%2210691.jpg%22%2C%2210692.jpg%22%2C%2210693.jpg%22%2C%2210694.jpg%22%2C%2210695.jpg%22%2C%2210696.jpg%22%2C%2210697.jpg%22%2C%2210698.jpg%22%2C%2210699.jpg%22%2C%2210700.jpg%22%2C%2210701.jpg%22%2C%2210702.jpg%22%2C%2210703.jpg%22%2C%2210704.jpg%22%2C%2210705.jpg%22%2C%2210706.jpg%22%2C%2210707.jpg%22%2C%2210708.jpg%22%2C%2210709.jpg%22%2C%2210710.jpg%22%2C%2210711.jpg%22%2C%2210712.jpg%22%2C%2210713.jpg%22%2C%2210714.jpg%22%2C%2210715.jpg%22%2C%2210716.jpg%22%2C%2210717.jpg%22%2C%2210718.jpg%22%2C%2210719.jpg%22%2C%2210720.jpg%22%2C%2210721.jpg%22%2C%2210722.jpg%22%2C%2210723.jpg%22%2C%2210724.jpg%22%2C%2210725.jpg%22%2C%2210726.jpg%22%2C%2210727.jpg%22%2C%2210728.jpg%22%2C%2210729.jpg%22%2C%2210730.jpg%22%2C%2210731.jpg%22%2C%2210732.jpg%22%2C%2210733.jpg%22%2C%2210734.jpg%22%2C%2210735.jpg%22%2C%2210736.jpg%22%2C%2210737.jpg%22%2C%2210738.jpg%22%2C%2210739.jpg%22%2C%2211141.jpg%22%2C%2211142.jpg%22%2C%2211143.jpg%22%2C%2211144.jpg%22%2C%2211145.jpg%22%2C%2211146.jpg%22%2C%2211147.jpg%22%2C%2211148.jpg%22%2C%2211149.jpg%22%2C%2211150.jpg%22%2C%2211151.jpg%22%2C%2211152.jpg%22%2C%2211153.jpg%22%2C%2211154.jpg%22%2C%2211155.jpg%22%2C%2211156.jpg%22%2C%2211157.jpg%22%2C%2211158.jpg%22%2C%2211159.jpg%22%2C%2211160.jpg%22%2C%2211161.jpg%22%2C%2211162.jpg%22%2C%2211163.jpg%22%2C%2211164.jpg%22%2C%2211165.jpg%22%2C%2211166.jpg%22%2C%2211167.jpg%22%2C%2211168.jpg%22%2C%2211169.jpg%22%2C%2211170.jpg%22%2C%2211171.jpg%22%2C%2211172.jpg%22%2C%2211173.jpg%22%2C%2211174.jpg%22%2C%2211175.jpg%22%2C%2211176.jpg%22%2C%2211177.jpg%22%2C%2211178.jpg%22%2C%2211179.jpg%22%2C%2211180.jpg%22%2C%2211181.jpg%22%2C%2211182.jpg%22%2C%2211183.jpg%22%2C%2211184.jpg%22%2C%2211197.jpg%22%2C%2211205.jpg%22%2C%2211206.jpg%22%2C%2211207.jpg%22%2C%2211208.jpg%22%2C%2211209.jpg%22%2C%2211210.jpg%22%2C%2211211.jpg%22%2C%2211212.jpg%22%2C%2211213.jpg%22%2C%2211214.jpg%22%2C%2211215.jpg%22%2C%2211216.jpg%22%2C%2211217.jpg%22%2C%2211218.jpg%22%2C%2211226.jpg%22%2C%2211227.jpg%22%2C%2211228.jpg%22%2C%2211229.jpg%22%2C%2211230.jpg%22%2C%2211231.jpg%22%2C%2211232.jpg%22%2C%2211315.jpg%22%2C%2211317.jpg%22%5D",
            filename="train13.zip",
            md5=False,
            checksum="b5b43dfc46efdc59ddb42e537b1f1c16020d64bb5df690cd2405c3465f7fec4b",
        ),
        DataLocation(
            url="https://cloud.cs.uni-tuebingen.de/public.php/dav/files/ZZxX65FGnQ8zjBP/Compressed%20Version/images/train/?accept=zip&files=%5B%2211318.jpg%22%2C%2211319.jpg%22%2C%2211320.jpg%22%2C%2211321.jpg%22%2C%2211322.jpg%22%2C%2211323.jpg%22%2C%2211324.jpg%22%2C%2211325.jpg%22%2C%2211326.jpg%22%2C%2211327.jpg%22%2C%2211328.jpg%22%2C%2211329.jpg%22%2C%2211330.jpg%22%2C%2211331.jpg%22%2C%2211332.jpg%22%2C%2211333.jpg%22%2C%2211334.jpg%22%2C%2211335.jpg%22%2C%2211336.jpg%22%2C%2211337.jpg%22%2C%2211338.jpg%22%2C%2211339.jpg%22%2C%2211340.jpg%22%2C%2211341.jpg%22%2C%2211342.jpg%22%2C%2211343.jpg%22%2C%2211344.jpg%22%2C%2211345.jpg%22%2C%2211346.jpg%22%2C%2211347.jpg%22%2C%2211348.jpg%22%2C%2211349.jpg%22%2C%2211350.jpg%22%2C%2211351.jpg%22%2C%2211352.jpg%22%2C%2211353.jpg%22%2C%2211354.jpg%22%2C%2211355.jpg%22%2C%2211356.jpg%22%2C%2211357.jpg%22%2C%2211358.jpg%22%2C%2211359.jpg%22%2C%2211360.jpg%22%2C%2211361.jpg%22%2C%2211362.jpg%22%2C%2211363.jpg%22%2C%2211364.jpg%22%2C%2211365.jpg%22%2C%2211366.jpg%22%2C%2211367.jpg%22%2C%2211368.jpg%22%2C%2211369.jpg%22%2C%2211370.jpg%22%2C%2211371.jpg%22%2C%2211372.jpg%22%2C%2211373.jpg%22%2C%2211374.jpg%22%2C%2211375.jpg%22%2C%2211376.jpg%22%2C%2211377.jpg%22%2C%2211378.jpg%22%2C%2211379.jpg%22%2C%2211380.jpg%22%2C%2211381.jpg%22%2C%2211382.jpg%22%2C%2211383.jpg%22%2C%2211384.jpg%22%2C%2211385.jpg%22%2C%2211386.jpg%22%2C%2211387.jpg%22%2C%2211388.jpg%22%2C%2211389.jpg%22%2C%2211390.jpg%22%2C%2211391.jpg%22%2C%2211392.jpg%22%2C%2211393.jpg%22%2C%2211394.jpg%22%2C%2211395.jpg%22%2C%2211396.jpg%22%2C%2211397.jpg%22%2C%2211398.jpg%22%2C%2211399.jpg%22%2C%2211400.jpg%22%2C%2211401.jpg%22%2C%2211402.jpg%22%2C%2211403.jpg%22%2C%2211404.jpg%22%2C%2211405.jpg%22%2C%2211406.jpg%22%2C%2211407.jpg%22%2C%2211408.jpg%22%2C%2211409.jpg%22%2C%2211410.jpg%22%2C%2211411.jpg%22%2C%2211412.jpg%22%2C%2211413.jpg%22%2C%2211414.jpg%22%2C%2211415.jpg%22%2C%2211416.jpg%22%2C%2211417.jpg%22%2C%2211418.jpg%22%2C%2211419.jpg%22%2C%2211420.jpg%22%2C%2211421.jpg%22%2C%2211422.jpg%22%2C%2211423.jpg%22%2C%2211424.jpg%22%2C%2211425.jpg%22%2C%2211426.jpg%22%2C%2211427.jpg%22%2C%2211428.jpg%22%2C%2211429.jpg%22%2C%2211430.jpg%22%2C%2211431.jpg%22%2C%2211432.jpg%22%2C%2211433.jpg%22%2C%2211434.jpg%22%2C%2211435.jpg%22%2C%2211436.jpg%22%2C%2211437.jpg%22%2C%2211438.jpg%22%2C%2211439.jpg%22%2C%2211440.jpg%22%2C%2211441.jpg%22%2C%2211442.jpg%22%2C%2211443.jpg%22%2C%2211444.jpg%22%2C%2211445.jpg%22%2C%2211446.jpg%22%2C%2211447.jpg%22%2C%2211448.jpg%22%2C%2211449.jpg%22%2C%2211450.jpg%22%2C%2211451.jpg%22%2C%2211452.jpg%22%2C%2211453.jpg%22%2C%2211454.jpg%22%2C%2211455.jpg%22%2C%2211456.jpg%22%2C%2211457.jpg%22%2C%2211458.jpg%22%2C%2211459.jpg%22%2C%2211460.jpg%22%2C%2211461.jpg%22%2C%2211462.jpg%22%2C%2211463.jpg%22%2C%2211464.jpg%22%2C%2211465.jpg%22%2C%2211466.jpg%22%2C%2211467.jpg%22%2C%2211468.jpg%22%2C%2211469.jpg%22%2C%2211470.jpg%22%2C%2211471.jpg%22%2C%2211472.jpg%22%2C%2211473.jpg%22%2C%2211474.jpg%22%2C%2211475.jpg%22%2C%2211476.jpg%22%2C%2211477.jpg%22%2C%2211478.jpg%22%2C%2211479.jpg%22%2C%2211480.jpg%22%2C%2211481.jpg%22%2C%2211482.jpg%22%2C%2211483.jpg%22%2C%2211484.jpg%22%2C%2211485.jpg%22%2C%2211486.jpg%22%2C%2211487.jpg%22%2C%2211488.jpg%22%2C%2211489.jpg%22%2C%2211490.jpg%22%2C%2211491.jpg%22%2C%2211492.jpg%22%2C%2211493.jpg%22%2C%2211494.jpg%22%2C%2211495.jpg%22%2C%2211496.jpg%22%2C%2211497.jpg%22%2C%2211498.jpg%22%2C%2211499.jpg%22%2C%2211500.jpg%22%2C%2211501.jpg%22%2C%2211502.jpg%22%2C%2211503.jpg%22%2C%2211504.jpg%22%2C%2211505.jpg%22%2C%2211506.jpg%22%2C%2211507.jpg%22%2C%2211508.jpg%22%2C%2211509.jpg%22%2C%2211510.jpg%22%2C%2211511.jpg%22%2C%2211512.jpg%22%2C%2211513.jpg%22%2C%2211514.jpg%22%2C%2211515.jpg%22%2C%2211516.jpg%22%2C%2211517.jpg%22%2C%2211518.jpg%22%2C%2211519.jpg%22%2C%2211520.jpg%22%2C%2211521.jpg%22%2C%2211522.jpg%22%2C%2211523.jpg%22%2C%2211524.jpg%22%2C%2211525.jpg%22%2C%2211526.jpg%22%2C%2211527.jpg%22%2C%2211528.jpg%22%2C%2211529.jpg%22%2C%2211530.jpg%22%2C%2211531.jpg%22%2C%2211532.jpg%22%2C%2211533.jpg%22%2C%2211534.jpg%22%2C%2211535.jpg%22%2C%2211536.jpg%22%2C%2211537.jpg%22%2C%2211538.jpg%22%2C%2211539.jpg%22%2C%2211540.jpg%22%2C%2211541.jpg%22%2C%2211542.jpg%22%2C%2211543.jpg%22%2C%2211544.jpg%22%2C%2211545.jpg%22%2C%2211546.jpg%22%2C%2211547.jpg%22%2C%2211548.jpg%22%2C%2211549.jpg%22%2C%2211550.jpg%22%2C%2211551.jpg%22%2C%2211552.jpg%22%2C%2211553.jpg%22%2C%2211554.jpg%22%2C%2211555.jpg%22%2C%2211556.jpg%22%2C%2211557.jpg%22%2C%2211558.jpg%22%2C%2211559.jpg%22%2C%2211560.jpg%22%2C%2211561.jpg%22%2C%2211562.jpg%22%2C%2211563.jpg%22%2C%2211564.jpg%22%2C%2211565.jpg%22%2C%2211566.jpg%22%2C%2211567.jpg%22%2C%2211568.jpg%22%2C%2211569.jpg%22%2C%2211570.jpg%22%2C%2211571.jpg%22%2C%2211572.jpg%22%2C%2211573.jpg%22%2C%2211574.jpg%22%2C%2211575.jpg%22%2C%2211576.jpg%22%2C%2211577.jpg%22%2C%2211578.jpg%22%2C%2211579.jpg%22%2C%2211756.jpg%22%2C%2211757.jpg%22%2C%2211758.jpg%22%2C%2211759.jpg%22%2C%2211760.jpg%22%2C%2211761.jpg%22%2C%2211762.jpg%22%2C%2211763.jpg%22%2C%2211764.jpg%22%2C%2211765.jpg%22%2C%2211766.jpg%22%2C%2211767.jpg%22%2C%2211768.jpg%22%2C%2211769.jpg%22%2C%2211770.jpg%22%2C%2211771.jpg%22%2C%2211772.jpg%22%2C%2211773.jpg%22%2C%2211774.jpg%22%2C%2211775.jpg%22%2C%2211776.jpg%22%2C%2211777.jpg%22%2C%2211778.jpg%22%2C%2211779.jpg%22%2C%2211780.jpg%22%2C%2211781.jpg%22%2C%2211782.jpg%22%2C%2211783.jpg%22%2C%2211784.jpg%22%2C%2211785.jpg%22%2C%2211786.jpg%22%2C%2211787.jpg%22%2C%2211788.jpg%22%2C%2211789.jpg%22%2C%2211790.jpg%22%2C%2211791.jpg%22%2C%2211792.jpg%22%2C%2211793.jpg%22%2C%2211794.jpg%22%2C%2211795.jpg%22%2C%2211796.jpg%22%2C%2211797.jpg%22%2C%2211798.jpg%22%2C%2211799.jpg%22%2C%2211800.jpg%22%2C%2211801.jpg%22%2C%2211802.jpg%22%2C%2211803.jpg%22%2C%2211804.jpg%22%2C%2211805.jpg%22%2C%2211806.jpg%22%2C%2211807.jpg%22%2C%2211808.jpg%22%2C%2211809.jpg%22%2C%2211810.jpg%22%2C%2211811.jpg%22%2C%2211812.jpg%22%2C%2211813.jpg%22%2C%2211814.jpg%22%2C%2211815.jpg%22%2C%2211816.jpg%22%2C%2211817.jpg%22%2C%2211818.jpg%22%2C%2211819.jpg%22%2C%2211820.jpg%22%2C%2211821.jpg%22%2C%2211822.jpg%22%2C%2211823.jpg%22%2C%2211824.jpg%22%2C%2211825.jpg%22%2C%2211826.jpg%22%2C%2211827.jpg%22%2C%2211828.jpg%22%2C%2211829.jpg%22%2C%2211830.jpg%22%2C%2211831.jpg%22%2C%2211832.jpg%22%2C%2211833.jpg%22%2C%2211834.jpg%22%2C%2211835.jpg%22%2C%2211836.jpg%22%2C%2211837.jpg%22%2C%2211838.jpg%22%2C%2211839.jpg%22%2C%2211840.jpg%22%2C%2211841.jpg%22%2C%2211842.jpg%22%2C%2211843.jpg%22%2C%2211844.jpg%22%2C%2211845.jpg%22%2C%2211846.jpg%22%2C%2211847.jpg%22%2C%2211848.jpg%22%2C%2211849.jpg%22%2C%2211850.jpg%22%2C%2211851.jpg%22%2C%2211852.jpg%22%2C%2211853.jpg%22%2C%2211854.jpg%22%2C%2211855.jpg%22%2C%2211856.jpg%22%2C%2211857.jpg%22%2C%2211858.jpg%22%2C%2211859.jpg%22%2C%2211860.jpg%22%2C%2211861.jpg%22%2C%2211862.jpg%22%2C%2211863.jpg%22%2C%2211864.jpg%22%2C%2211865.jpg%22%2C%2211866.jpg%22%2C%2211867.jpg%22%2C%2211868.jpg%22%2C%2211869.jpg%22%2C%2211870.jpg%22%2C%2211871.jpg%22%2C%2211875.jpg%22%2C%2211876.jpg%22%2C%2211877.jpg%22%2C%2211878.jpg%22%2C%2211879.jpg%22%2C%2211880.jpg%22%2C%2211881.jpg%22%2C%2211882.jpg%22%2C%2211883.jpg%22%2C%2211884.jpg%22%2C%2211885.jpg%22%2C%2211886.jpg%22%2C%2211887.jpg%22%2C%2211888.jpg%22%2C%2211889.jpg%22%2C%2211890.jpg%22%2C%2211891.jpg%22%2C%2211892.jpg%22%2C%2211893.jpg%22%2C%2211894.jpg%22%2C%2211895.jpg%22%2C%2211896.jpg%22%2C%2211897.jpg%22%2C%2211898.jpg%22%2C%2211899.jpg%22%2C%2211900.jpg%22%2C%2211901.jpg%22%2C%2211902.jpg%22%2C%2211903.jpg%22%2C%2211904.jpg%22%2C%2211905.jpg%22%2C%2211906.jpg%22%2C%2211907.jpg%22%2C%2211908.jpg%22%2C%2211909.jpg%22%2C%2211910.jpg%22%2C%2211911.jpg%22%2C%2211912.jpg%22%2C%2211913.jpg%22%2C%2211914.jpg%22%2C%2211915.jpg%22%2C%2211916.jpg%22%2C%2211917.jpg%22%2C%2211918.jpg%22%2C%2211919.jpg%22%2C%2211920.jpg%22%2C%2211921.jpg%22%2C%2211922.jpg%22%2C%2211923.jpg%22%2C%2211924.jpg%22%2C%2211925.jpg%22%2C%2211926.jpg%22%2C%2211927.jpg%22%2C%2211928.jpg%22%2C%2211929.jpg%22%2C%2211930.jpg%22%2C%2211931.jpg%22%2C%2211932.jpg%22%2C%2211933.jpg%22%2C%2211934.jpg%22%2C%2211935.jpg%22%2C%2211936.jpg%22%2C%2211937.jpg%22%2C%2211938.jpg%22%5D",
            filename="train14.zip",
            md5=False,
            checksum="35544cf46737d7f2c56bb1a80db0f32ea6666bbf70bb86fe4ac1a8359f4e0814",
        ),
        DataLocation(
            url="https://cloud.cs.uni-tuebingen.de/public.php/dav/files/ZZxX65FGnQ8zjBP/Compressed%20Version/images/train/?accept=zip&files=%5B%2211939.jpg%22%2C%2211940.jpg%22%2C%2211941.jpg%22%2C%2211942.jpg%22%2C%2211943.jpg%22%2C%2211944.jpg%22%2C%2211945.jpg%22%2C%2211946.jpg%22%2C%2211947.jpg%22%2C%2211948.jpg%22%2C%2211949.jpg%22%2C%2211950.jpg%22%2C%2211951.jpg%22%2C%2211952.jpg%22%2C%2211953.jpg%22%2C%2211954.jpg%22%2C%2211955.jpg%22%2C%2211956.jpg%22%2C%2211957.jpg%22%2C%2211958.jpg%22%2C%2211959.jpg%22%2C%2211960.jpg%22%2C%2211961.jpg%22%2C%2211962.jpg%22%2C%2211963.jpg%22%2C%2211964.jpg%22%2C%2211965.jpg%22%2C%2211966.jpg%22%2C%2211967.jpg%22%2C%2211968.jpg%22%2C%2211969.jpg%22%2C%2211970.jpg%22%2C%2211971.jpg%22%2C%2211972.jpg%22%2C%2211973.jpg%22%2C%2211974.jpg%22%2C%2211975.jpg%22%2C%2211976.jpg%22%2C%2211977.jpg%22%2C%2211978.jpg%22%2C%2211979.jpg%22%2C%2211980.jpg%22%2C%2211981.jpg%22%2C%2211982.jpg%22%2C%2211983.jpg%22%2C%2211984.jpg%22%2C%2211985.jpg%22%2C%2211986.jpg%22%2C%2211987.jpg%22%2C%2211988.jpg%22%2C%2211989.jpg%22%2C%2211990.jpg%22%2C%2211991.jpg%22%2C%2211992.jpg%22%2C%2211993.jpg%22%2C%2211994.jpg%22%2C%2211995.jpg%22%2C%2211996.jpg%22%2C%2211997.jpg%22%2C%2211998.jpg%22%2C%2211999.jpg%22%2C%2212000.jpg%22%2C%2212001.jpg%22%2C%2212002.jpg%22%2C%2212003.jpg%22%2C%2212004.jpg%22%2C%2212005.jpg%22%2C%2212194.jpg%22%2C%2212195.jpg%22%2C%2212196.jpg%22%2C%2212197.jpg%22%2C%2212198.jpg%22%2C%2212199.jpg%22%2C%2212200.jpg%22%2C%2212201.jpg%22%2C%2212202.jpg%22%2C%2212203.jpg%22%2C%2212204.jpg%22%2C%2212205.jpg%22%2C%2212206.jpg%22%2C%2212207.jpg%22%2C%2212208.jpg%22%2C%2212209.jpg%22%2C%2212210.jpg%22%2C%2212211.jpg%22%2C%2212212.jpg%22%2C%2212213.jpg%22%2C%2212214.jpg%22%2C%2212215.jpg%22%2C%2212216.jpg%22%2C%2212217.jpg%22%2C%2212218.jpg%22%2C%2212219.jpg%22%2C%2212220.jpg%22%2C%2212221.jpg%22%2C%2212222.jpg%22%2C%2212223.jpg%22%2C%2212224.jpg%22%2C%2212225.jpg%22%2C%2212226.jpg%22%2C%2212227.jpg%22%2C%2212228.jpg%22%2C%2212229.jpg%22%2C%2212230.jpg%22%2C%2212231.jpg%22%2C%2212232.jpg%22%2C%2212233.jpg%22%2C%2212234.jpg%22%2C%2212235.jpg%22%2C%2212268.jpg%22%2C%2212269.jpg%22%2C%2212270.jpg%22%2C%2212271.jpg%22%2C%2212272.jpg%22%2C%2212273.jpg%22%2C%2212274.jpg%22%2C%2212275.jpg%22%2C%2212276.jpg%22%2C%2212277.jpg%22%2C%2212278.jpg%22%2C%2212279.jpg%22%2C%2212280.jpg%22%2C%2212281.jpg%22%2C%2212282.jpg%22%2C%2212283.jpg%22%2C%2212284.jpg%22%2C%2212286.jpg%22%2C%2212287.jpg%22%2C%2212288.jpg%22%2C%2212289.jpg%22%2C%2212290.jpg%22%2C%2212291.jpg%22%2C%2212292.jpg%22%2C%2212293.jpg%22%2C%2212294.jpg%22%2C%2212295.jpg%22%2C%2212296.jpg%22%2C%2212297.jpg%22%2C%2212298.jpg%22%2C%2212299.jpg%22%2C%2212300.jpg%22%2C%2212301.jpg%22%2C%2212302.jpg%22%2C%2212303.jpg%22%2C%2212304.jpg%22%2C%2212305.jpg%22%2C%2212306.jpg%22%2C%2212307.jpg%22%2C%2212308.jpg%22%2C%2212309.jpg%22%2C%2212310.jpg%22%2C%2212311.jpg%22%2C%2212312.jpg%22%2C%2212313.jpg%22%2C%2212314.jpg%22%2C%2212315.jpg%22%2C%2212316.jpg%22%2C%2212317.jpg%22%2C%2212318.jpg%22%2C%2212319.jpg%22%2C%2212320.jpg%22%2C%2212321.jpg%22%2C%2212322.jpg%22%2C%2212323.jpg%22%2C%2212324.jpg%22%2C%2212325.jpg%22%2C%2212326.jpg%22%2C%2212327.jpg%22%2C%2212328.jpg%22%2C%2212329.jpg%22%2C%2212330.jpg%22%2C%2212331.jpg%22%2C%2212332.jpg%22%2C%2212333.jpg%22%2C%2212334.jpg%22%2C%2212335.jpg%22%2C%2212336.jpg%22%2C%2212337.jpg%22%2C%2212338.jpg%22%2C%2212339.jpg%22%2C%2212340.jpg%22%2C%2212341.jpg%22%2C%2212342.jpg%22%2C%2212343.jpg%22%2C%2212344.jpg%22%2C%2212345.jpg%22%2C%2212346.jpg%22%2C%2212347.jpg%22%2C%2212348.jpg%22%2C%2212349.jpg%22%2C%2212350.jpg%22%2C%2212351.jpg%22%2C%2212352.jpg%22%2C%2212353.jpg%22%2C%2212354.jpg%22%2C%2212355.jpg%22%2C%2212356.jpg%22%2C%2212357.jpg%22%2C%2212358.jpg%22%2C%2212359.jpg%22%2C%2212360.jpg%22%2C%2212361.jpg%22%2C%2212362.jpg%22%2C%2212363.jpg%22%2C%2212364.jpg%22%2C%2212365.jpg%22%2C%2212366.jpg%22%2C%2212367.jpg%22%2C%2212368.jpg%22%2C%2212369.jpg%22%2C%2212370.jpg%22%2C%2212371.jpg%22%2C%2212372.jpg%22%2C%2212373.jpg%22%2C%2212374.jpg%22%2C%2212375.jpg%22%2C%2212376.jpg%22%2C%2212377.jpg%22%2C%2212378.jpg%22%2C%2212379.jpg%22%2C%2212380.jpg%22%2C%2212381.jpg%22%2C%2212382.jpg%22%2C%2212383.jpg%22%2C%2212384.jpg%22%2C%2212385.jpg%22%2C%2212386.jpg%22%2C%2212387.jpg%22%2C%2212388.jpg%22%2C%2212389.jpg%22%2C%2212390.jpg%22%2C%2212391.jpg%22%2C%2212392.jpg%22%2C%2212393.jpg%22%2C%2212394.jpg%22%2C%2212395.jpg%22%2C%2212396.jpg%22%2C%2212397.jpg%22%2C%2212398.jpg%22%2C%2212399.jpg%22%2C%2212400.jpg%22%2C%2212401.jpg%22%2C%2212402.jpg%22%2C%2212403.jpg%22%2C%2212404.jpg%22%2C%2212405.jpg%22%2C%2212406.jpg%22%2C%2212407.jpg%22%2C%2212408.jpg%22%2C%2212409.jpg%22%2C%2212410.jpg%22%2C%2212411.jpg%22%2C%2212412.jpg%22%2C%2212413.jpg%22%2C%2212414.jpg%22%2C%2212415.jpg%22%2C%2212417.jpg%22%2C%2212418.jpg%22%2C%2212419.jpg%22%2C%2212420.jpg%22%2C%2212421.jpg%22%2C%2212422.jpg%22%2C%2212423.jpg%22%2C%2212424.jpg%22%2C%2212425.jpg%22%2C%2212426.jpg%22%2C%2212427.jpg%22%2C%2212428.jpg%22%2C%2212429.jpg%22%2C%2212430.jpg%22%2C%2212431.jpg%22%2C%2212432.jpg%22%2C%2212433.jpg%22%2C%2212434.jpg%22%2C%2212435.jpg%22%2C%2212436.jpg%22%2C%2212437.jpg%22%2C%2212438.jpg%22%2C%2212439.jpg%22%2C%2212440.jpg%22%2C%2212441.jpg%22%2C%2212442.jpg%22%2C%2212443.jpg%22%2C%2212444.jpg%22%2C%2212445.jpg%22%2C%2212446.jpg%22%2C%2212447.jpg%22%2C%2212448.jpg%22%2C%2212449.jpg%22%2C%2212450.jpg%22%2C%2212451.jpg%22%2C%2212452.jpg%22%2C%2212453.jpg%22%2C%2212454.jpg%22%2C%2212455.jpg%22%2C%2212456.jpg%22%2C%2212457.jpg%22%2C%2212458.jpg%22%2C%2212459.jpg%22%2C%2212460.jpg%22%2C%2212461.jpg%22%2C%2212462.jpg%22%2C%2212463.jpg%22%2C%2212464.jpg%22%2C%2212465.jpg%22%2C%2212466.jpg%22%2C%2212467.jpg%22%2C%2212468.jpg%22%2C%2212469.jpg%22%2C%2212470.jpg%22%2C%2212471.jpg%22%2C%2212472.jpg%22%2C%2212473.jpg%22%2C%2212474.jpg%22%2C%2212475.jpg%22%2C%2212476.jpg%22%2C%2212477.jpg%22%2C%2212478.jpg%22%2C%2212479.jpg%22%2C%2212480.jpg%22%2C%2212481.jpg%22%2C%2212482.jpg%22%2C%2212483.jpg%22%2C%2212484.jpg%22%2C%2212485.jpg%22%2C%2212486.jpg%22%2C%2212487.jpg%22%2C%2212488.jpg%22%2C%2212489.jpg%22%2C%2212490.jpg%22%2C%2212491.jpg%22%2C%2212492.jpg%22%2C%2212493.jpg%22%2C%2212494.jpg%22%2C%2212495.jpg%22%2C%2212496.jpg%22%2C%2212497.jpg%22%2C%2212498.jpg%22%2C%2212499.jpg%22%2C%2212500.jpg%22%2C%2212501.jpg%22%2C%2212502.jpg%22%2C%2212503.jpg%22%2C%2212504.jpg%22%2C%2212505.jpg%22%2C%2212506.jpg%22%2C%2212507.jpg%22%2C%2212508.jpg%22%2C%2212509.jpg%22%2C%2212510.jpg%22%2C%2212511.jpg%22%2C%2212512.jpg%22%2C%2212513.jpg%22%2C%2212514.jpg%22%2C%2212515.jpg%22%2C%2212516.jpg%22%2C%2212517.jpg%22%2C%2212518.jpg%22%2C%2212519.jpg%22%2C%2212520.jpg%22%2C%2212521.jpg%22%2C%2212522.jpg%22%2C%2212523.jpg%22%2C%2212524.jpg%22%2C%2212525.jpg%22%2C%2212526.jpg%22%2C%2212527.jpg%22%2C%2212528.jpg%22%2C%2212529.jpg%22%2C%2212530.jpg%22%2C%2212531.jpg%22%2C%2212532.jpg%22%2C%2212533.jpg%22%2C%2212534.jpg%22%2C%2212535.jpg%22%2C%2212536.jpg%22%2C%2212537.jpg%22%2C%2212538.jpg%22%2C%2212539.jpg%22%2C%2212540.jpg%22%2C%2212541.jpg%22%2C%2212542.jpg%22%2C%2212543.jpg%22%2C%2212544.jpg%22%2C%2212545.jpg%22%2C%2212546.jpg%22%2C%2212547.jpg%22%2C%2212548.jpg%22%2C%2212549.jpg%22%2C%2212550.jpg%22%2C%2212551.jpg%22%2C%2212552.jpg%22%2C%2212820.jpg%22%2C%2212821.jpg%22%2C%2212822.jpg%22%2C%2212823.jpg%22%2C%2212824.jpg%22%2C%2212825.jpg%22%2C%2212826.jpg%22%2C%2212827.jpg%22%2C%2212828.jpg%22%2C%2212829.jpg%22%2C%2212830.jpg%22%2C%2212831.jpg%22%2C%2212832.jpg%22%2C%2212833.jpg%22%2C%2212834.jpg%22%2C%2212835.jpg%22%2C%2212836.jpg%22%2C%2212837.jpg%22%2C%2212838.jpg%22%2C%2212839.jpg%22%2C%2212840.jpg%22%2C%2212841.jpg%22%2C%2212842.jpg%22%2C%2212843.jpg%22%2C%2212844.jpg%22%2C%2212845.jpg%22%2C%2212846.jpg%22%2C%2212847.jpg%22%2C%2212848.jpg%22%2C%2212849.jpg%22%2C%2212850.jpg%22%2C%2212851.jpg%22%2C%2212852.jpg%22%2C%2212853.jpg%22%2C%2212854.jpg%22%2C%2212855.jpg%22%2C%2212856.jpg%22%2C%2212857.jpg%22%2C%2212858.jpg%22%2C%2212859.jpg%22%2C%2212860.jpg%22%2C%2212861.jpg%22%2C%2212862.jpg%22%2C%2212863.jpg%22%2C%2212864.jpg%22%2C%2212865.jpg%22%2C%2212866.jpg%22%2C%2212867.jpg%22%2C%2212868.jpg%22%2C%2212869.jpg%22%5D",
            filename="train15.zip",
            md5=False,
            checksum="09a07fab209ba1461a9c524d52b7774ad5360e5fc62edeafeb9cf14014fa257b",
        ),
        DataLocation(
            url="https://cloud.cs.uni-tuebingen.de/public.php/dav/files/ZZxX65FGnQ8zjBP/Compressed%20Version/images/train/?accept=zip&files=%5B%2212870.jpg%22%2C%2212871.jpg%22%2C%2212872.jpg%22%2C%2212873.jpg%22%2C%2212874.jpg%22%2C%2212875.jpg%22%2C%2212876.jpg%22%2C%2212877.jpg%22%2C%2212878.jpg%22%2C%2212879.jpg%22%2C%2212880.jpg%22%2C%2212881.jpg%22%2C%2212882.jpg%22%2C%2212883.jpg%22%2C%2212884.jpg%22%2C%2212885.jpg%22%2C%2212886.jpg%22%2C%2212887.jpg%22%2C%2212888.jpg%22%2C%2212889.jpg%22%2C%2212890.jpg%22%2C%2212891.jpg%22%2C%2212892.jpg%22%2C%2212893.jpg%22%2C%2212894.jpg%22%2C%2212895.jpg%22%2C%2212896.jpg%22%2C%2212897.jpg%22%2C%2212898.jpg%22%2C%2212899.jpg%22%2C%2212900.jpg%22%2C%2212901.jpg%22%2C%2212902.jpg%22%2C%2212903.jpg%22%2C%2212904.jpg%22%2C%2212905.jpg%22%2C%2212906.jpg%22%2C%2212907.jpg%22%2C%2212908.jpg%22%2C%2212909.jpg%22%2C%2212910.jpg%22%2C%2212911.jpg%22%2C%2212912.jpg%22%2C%2212913.jpg%22%2C%2212914.jpg%22%2C%2212915.jpg%22%2C%2212916.jpg%22%2C%2212917.jpg%22%2C%2212918.jpg%22%2C%2212919.jpg%22%2C%2212920.jpg%22%2C%2212921.jpg%22%2C%2212922.jpg%22%2C%2212923.jpg%22%2C%2212924.jpg%22%2C%2212925.jpg%22%2C%2212926.jpg%22%2C%2212927.jpg%22%2C%2212928.jpg%22%2C%2212929.jpg%22%2C%2212930.jpg%22%2C%2212931.jpg%22%2C%2212932.jpg%22%2C%2212933.jpg%22%2C%2212934.jpg%22%2C%2212935.jpg%22%2C%2212936.jpg%22%2C%2212937.jpg%22%2C%2212938.jpg%22%2C%2212939.jpg%22%2C%2212940.jpg%22%2C%2212941.jpg%22%2C%2212942.jpg%22%2C%2212943.jpg%22%2C%2212944.jpg%22%2C%2212945.jpg%22%2C%2212946.jpg%22%2C%2212947.jpg%22%2C%2212948.jpg%22%2C%2212949.jpg%22%2C%2212950.jpg%22%2C%2212951.jpg%22%2C%2212952.jpg%22%2C%2212953.jpg%22%2C%2212954.jpg%22%2C%2212955.jpg%22%2C%2212956.jpg%22%2C%2212957.jpg%22%2C%2212958.jpg%22%2C%2212959.jpg%22%2C%2212960.jpg%22%2C%2212961.jpg%22%2C%2212962.jpg%22%2C%2212963.jpg%22%2C%2212964.jpg%22%2C%2212965.jpg%22%2C%2212966.jpg%22%2C%2212967.jpg%22%2C%2212968.jpg%22%2C%2212969.jpg%22%2C%2212970.jpg%22%2C%2212971.jpg%22%2C%2212972.jpg%22%2C%2212973.jpg%22%2C%2212974.jpg%22%2C%2212975.jpg%22%2C%2212976.jpg%22%2C%2212977.jpg%22%2C%2212978.jpg%22%2C%2212979.jpg%22%2C%2212980.jpg%22%2C%2212981.jpg%22%2C%2212982.jpg%22%2C%2212983.jpg%22%2C%2212984.jpg%22%2C%2212985.jpg%22%2C%2212986.jpg%22%2C%2212987.jpg%22%2C%2212988.jpg%22%2C%2212989.jpg%22%2C%2212990.jpg%22%2C%2212991.jpg%22%2C%2212992.jpg%22%2C%2212993.jpg%22%2C%2212994.jpg%22%2C%2212995.jpg%22%2C%2212996.jpg%22%2C%2212997.jpg%22%2C%2212998.jpg%22%2C%2212999.jpg%22%2C%2213000.jpg%22%2C%2213001.jpg%22%2C%2213002.jpg%22%2C%2213003.jpg%22%2C%2213004.jpg%22%2C%2213005.jpg%22%2C%2213006.jpg%22%2C%2213007.jpg%22%2C%2213008.jpg%22%2C%2213009.jpg%22%2C%2213010.jpg%22%2C%2213011.jpg%22%2C%2213012.jpg%22%2C%2213013.jpg%22%2C%2213014.jpg%22%2C%2213015.jpg%22%2C%2213016.jpg%22%2C%2213017.jpg%22%2C%2213018.jpg%22%2C%2213019.jpg%22%2C%2213020.jpg%22%2C%2213021.jpg%22%2C%2213022.jpg%22%2C%2213023.jpg%22%2C%2213024.jpg%22%2C%2213025.jpg%22%2C%2213026.jpg%22%2C%2213027.jpg%22%2C%2213028.jpg%22%2C%2213029.jpg%22%2C%2213030.jpg%22%2C%2213031.jpg%22%2C%2213032.jpg%22%2C%2213033.jpg%22%2C%2213034.jpg%22%2C%2213035.jpg%22%2C%2213036.jpg%22%2C%2213037.jpg%22%2C%2213038.jpg%22%2C%2213039.jpg%22%2C%2213040.jpg%22%2C%2213041.jpg%22%2C%2213042.jpg%22%2C%2213043.jpg%22%2C%2213044.jpg%22%2C%2213045.jpg%22%2C%2213046.jpg%22%2C%2213047.jpg%22%2C%2213048.jpg%22%2C%2213049.jpg%22%2C%2213050.jpg%22%2C%2213051.jpg%22%2C%2213052.jpg%22%2C%2213053.jpg%22%2C%2213054.jpg%22%2C%2213055.jpg%22%2C%2213056.jpg%22%2C%2213057.jpg%22%2C%2213058.jpg%22%2C%2213059.jpg%22%2C%2213060.jpg%22%2C%2213061.jpg%22%2C%2213062.jpg%22%2C%2213063.jpg%22%2C%2213064.jpg%22%2C%2213065.jpg%22%2C%2213066.jpg%22%2C%2213067.jpg%22%2C%2213068.jpg%22%2C%2213069.jpg%22%2C%2213070.jpg%22%2C%2213071.jpg%22%2C%2213072.jpg%22%2C%2213073.jpg%22%2C%2213074.jpg%22%2C%2213075.jpg%22%2C%2213076.jpg%22%2C%2213077.jpg%22%2C%2213078.jpg%22%2C%2213079.jpg%22%2C%2213080.jpg%22%2C%2213081.jpg%22%2C%2213082.jpg%22%2C%2213083.jpg%22%2C%2213084.jpg%22%2C%2213085.jpg%22%2C%2213086.jpg%22%2C%2213087.jpg%22%2C%2213088.jpg%22%2C%2213089.jpg%22%2C%2213090.jpg%22%2C%2213091.jpg%22%2C%2213092.jpg%22%2C%2213093.jpg%22%2C%2213094.jpg%22%2C%2213095.jpg%22%2C%2213096.jpg%22%2C%2213097.jpg%22%2C%2213098.jpg%22%2C%2213099.jpg%22%2C%2213100.jpg%22%2C%2213101.jpg%22%2C%2213102.jpg%22%2C%2213103.jpg%22%2C%2213104.jpg%22%2C%2213105.jpg%22%2C%2213106.jpg%22%2C%2213107.jpg%22%2C%2213108.jpg%22%2C%2213109.jpg%22%2C%2213110.jpg%22%2C%2213111.jpg%22%2C%2213112.jpg%22%2C%2213113.jpg%22%2C%2213114.jpg%22%2C%2213115.jpg%22%2C%2213116.jpg%22%2C%2213117.jpg%22%2C%2213118.jpg%22%2C%2213119.jpg%22%2C%2213120.jpg%22%2C%2213121.jpg%22%2C%2213122.jpg%22%2C%2213123.jpg%22%2C%2213124.jpg%22%2C%2213125.jpg%22%2C%2213126.jpg%22%2C%2213127.jpg%22%2C%2213128.jpg%22%2C%2213129.jpg%22%2C%2213130.jpg%22%2C%2213131.jpg%22%2C%2213132.jpg%22%2C%2213133.jpg%22%2C%2213134.jpg%22%2C%2213135.jpg%22%2C%2213136.jpg%22%2C%2213137.jpg%22%2C%2213138.jpg%22%2C%2213139.jpg%22%2C%2213140.jpg%22%2C%2213141.jpg%22%2C%2213142.jpg%22%2C%2213143.jpg%22%2C%2213144.jpg%22%2C%2213145.jpg%22%2C%2213146.jpg%22%2C%2213147.jpg%22%2C%2213148.jpg%22%2C%2213149.jpg%22%2C%2213150.jpg%22%2C%2213151.jpg%22%2C%2213152.jpg%22%2C%2213153.jpg%22%2C%2213154.jpg%22%2C%2213155.jpg%22%2C%2213156.jpg%22%2C%2213157.jpg%22%2C%2213158.jpg%22%2C%2213159.jpg%22%2C%2213160.jpg%22%2C%2213161.jpg%22%2C%2213162.jpg%22%2C%2213163.jpg%22%2C%2213164.jpg%22%2C%2213165.jpg%22%2C%2213166.jpg%22%2C%2213167.jpg%22%2C%2213168.jpg%22%2C%2213169.jpg%22%2C%2213170.jpg%22%2C%2213171.jpg%22%2C%2213172.jpg%22%2C%2213173.jpg%22%2C%2213174.jpg%22%2C%2213175.jpg%22%2C%2213176.jpg%22%2C%2213177.jpg%22%2C%2213178.jpg%22%2C%2213179.jpg%22%2C%2213180.jpg%22%2C%2213181.jpg%22%2C%2213182.jpg%22%2C%2213183.jpg%22%2C%2213184.jpg%22%2C%2213185.jpg%22%2C%2213186.jpg%22%2C%2213187.jpg%22%2C%2213188.jpg%22%2C%2213189.jpg%22%2C%2213190.jpg%22%2C%2213191.jpg%22%2C%2213192.jpg%22%2C%2213193.jpg%22%2C%2213194.jpg%22%2C%2213195.jpg%22%2C%2213196.jpg%22%2C%2213197.jpg%22%2C%2213198.jpg%22%2C%2213199.jpg%22%2C%2213200.jpg%22%2C%2213201.jpg%22%2C%2213202.jpg%22%2C%2213203.jpg%22%2C%2213204.jpg%22%2C%2213205.jpg%22%2C%2213206.jpg%22%2C%2213207.jpg%22%2C%2213208.jpg%22%2C%2213209.jpg%22%2C%2213210.jpg%22%2C%2213211.jpg%22%2C%2213212.jpg%22%2C%2213213.jpg%22%2C%2213214.jpg%22%2C%2213215.jpg%22%2C%2213216.jpg%22%2C%2213217.jpg%22%2C%2213218.jpg%22%2C%2213219.jpg%22%2C%2213220.jpg%22%2C%2213221.jpg%22%2C%2213222.jpg%22%2C%2213223.jpg%22%2C%2213224.jpg%22%2C%2213225.jpg%22%2C%2213226.jpg%22%2C%2213227.jpg%22%2C%2213228.jpg%22%2C%2213229.jpg%22%2C%2213230.jpg%22%2C%2213231.jpg%22%2C%2213232.jpg%22%2C%2213233.jpg%22%2C%2213234.jpg%22%2C%2213235.jpg%22%2C%2213236.jpg%22%2C%2213237.jpg%22%2C%2213238.jpg%22%2C%2213239.jpg%22%2C%2213240.jpg%22%2C%2213241.jpg%22%2C%2213242.jpg%22%2C%2213243.jpg%22%2C%2213244.jpg%22%2C%2213245.jpg%22%2C%2213246.jpg%22%2C%2213247.jpg%22%2C%2213248.jpg%22%2C%2213249.jpg%22%2C%2213250.jpg%22%2C%2213251.jpg%22%2C%2213252.jpg%22%2C%2213253.jpg%22%2C%2213254.jpg%22%2C%2213255.jpg%22%2C%2213256.jpg%22%2C%2213257.jpg%22%2C%2213258.jpg%22%2C%2213259.jpg%22%2C%2213260.jpg%22%2C%2213261.jpg%22%2C%2213262.jpg%22%2C%2213263.jpg%22%2C%2213264.jpg%22%2C%2213265.jpg%22%2C%2213266.jpg%22%2C%2213267.jpg%22%2C%2213268.jpg%22%2C%2213269.jpg%22%2C%2213270.jpg%22%2C%2213271.jpg%22%2C%2213272.jpg%22%2C%2213273.jpg%22%2C%2213274.jpg%22%2C%2213275.jpg%22%2C%2213276.jpg%22%2C%2213277.jpg%22%2C%2213278.jpg%22%2C%2213279.jpg%22%2C%2213280.jpg%22%2C%2213281.jpg%22%2C%2213282.jpg%22%2C%2213283.jpg%22%2C%2213284.jpg%22%2C%2213285.jpg%22%2C%2213286.jpg%22%2C%2213287.jpg%22%2C%2213288.jpg%22%2C%2213289.jpg%22%2C%2213290.jpg%22%2C%2213291.jpg%22%2C%2213292.jpg%22%2C%2213293.jpg%22%2C%2213294.jpg%22%2C%2213295.jpg%22%2C%2213296.jpg%22%2C%2213297.jpg%22%2C%2213298.jpg%22%2C%2213299.jpg%22%2C%2213300.jpg%22%2C%2213301.jpg%22%2C%2213302.jpg%22%2C%2213303.jpg%22%2C%2213304.jpg%22%2C%2213305.jpg%22%2C%2213306.jpg%22%2C%2213307.jpg%22%2C%2213308.jpg%22%2C%2213309.jpg%22%2C%2213310.jpg%22%2C%2213311.jpg%22%5D",
            filename="train16.zip",
            md5=False,
            checksum="6b14ebef37ef25003f2c9ead4180478dc50745bd48ebb25765daee15257efdfe",
        ),
        DataLocation(
            url="https://cloud.cs.uni-tuebingen.de/public.php/dav/files/ZZxX65FGnQ8zjBP/Compressed%20Version/images/train/?accept=zip&files=%5B%2213312.jpg%22%2C%2213313.jpg%22%2C%2213314.jpg%22%2C%2213315.jpg%22%2C%2213316.jpg%22%2C%2213317.jpg%22%2C%2213318.jpg%22%2C%2213319.jpg%22%2C%2213320.jpg%22%2C%2213321.jpg%22%2C%2213322.jpg%22%2C%2213323.jpg%22%2C%2213324.jpg%22%2C%2213325.jpg%22%2C%2213326.jpg%22%2C%2213327.jpg%22%2C%2213328.jpg%22%2C%2213329.jpg%22%2C%2213330.jpg%22%2C%2213331.jpg%22%2C%2213332.jpg%22%2C%2213333.jpg%22%2C%2213334.jpg%22%2C%2213335.jpg%22%2C%2213336.jpg%22%2C%2213337.jpg%22%2C%2213338.jpg%22%2C%2213339.jpg%22%2C%2213340.jpg%22%2C%2213341.jpg%22%2C%2213342.jpg%22%2C%2213343.jpg%22%2C%2213344.jpg%22%2C%2213345.jpg%22%2C%2213346.jpg%22%2C%2213347.jpg%22%2C%2213348.jpg%22%2C%2213349.jpg%22%2C%2213350.jpg%22%2C%2213351.jpg%22%2C%2213352.jpg%22%2C%2213353.jpg%22%2C%2213354.jpg%22%2C%2213355.jpg%22%2C%2213356.jpg%22%2C%2213357.jpg%22%2C%2213358.jpg%22%2C%2213359.jpg%22%2C%2213360.jpg%22%2C%2213361.jpg%22%2C%2213362.jpg%22%2C%2213363.jpg%22%2C%2213364.jpg%22%2C%2213365.jpg%22%2C%2213366.jpg%22%2C%2213367.jpg%22%2C%2213368.jpg%22%2C%2213369.jpg%22%2C%2213370.jpg%22%2C%2213371.jpg%22%2C%2213372.jpg%22%2C%2213373.jpg%22%2C%2213374.jpg%22%2C%2213375.jpg%22%2C%2213376.jpg%22%2C%2213377.jpg%22%2C%2213378.jpg%22%2C%2213379.jpg%22%2C%2213380.jpg%22%2C%2213381.jpg%22%2C%2213382.jpg%22%2C%2213383.jpg%22%2C%2213384.jpg%22%2C%2213385.jpg%22%2C%2213386.jpg%22%2C%2213387.jpg%22%2C%2213388.jpg%22%2C%2213389.jpg%22%2C%2213390.jpg%22%2C%2213391.jpg%22%2C%2213392.jpg%22%2C%2213393.jpg%22%2C%2213394.jpg%22%2C%2213395.jpg%22%2C%2213396.jpg%22%2C%2213397.jpg%22%2C%2213398.jpg%22%2C%2213399.jpg%22%2C%2213400.jpg%22%2C%2213401.jpg%22%2C%2213402.jpg%22%2C%2213403.jpg%22%2C%2213404.jpg%22%2C%2213405.jpg%22%2C%2213406.jpg%22%2C%2213407.jpg%22%2C%2213408.jpg%22%2C%2213409.jpg%22%2C%2213410.jpg%22%2C%2213411.jpg%22%2C%2213412.jpg%22%2C%2213413.jpg%22%2C%2213414.jpg%22%2C%2213415.jpg%22%2C%2213416.jpg%22%2C%2213417.jpg%22%2C%2213418.jpg%22%2C%2213419.jpg%22%2C%2213420.jpg%22%2C%2213421.jpg%22%2C%2213422.jpg%22%2C%2213423.jpg%22%2C%2213424.jpg%22%2C%2213425.jpg%22%2C%2213426.jpg%22%2C%2213427.jpg%22%2C%2213428.jpg%22%2C%2213429.jpg%22%2C%2213430.jpg%22%2C%2213431.jpg%22%2C%2213432.jpg%22%2C%2213433.jpg%22%2C%2213434.jpg%22%2C%2213435.jpg%22%2C%2213436.jpg%22%2C%2213437.jpg%22%2C%2213438.jpg%22%2C%2213439.jpg%22%2C%2213440.jpg%22%2C%2213441.jpg%22%2C%2213442.jpg%22%2C%2213443.jpg%22%2C%2213444.jpg%22%2C%2213445.jpg%22%2C%2213446.jpg%22%2C%2213447.jpg%22%2C%2213448.jpg%22%2C%2213449.jpg%22%2C%2213450.jpg%22%2C%2213451.jpg%22%2C%2213452.jpg%22%2C%2213453.jpg%22%2C%2213454.jpg%22%2C%2213455.jpg%22%2C%2213456.jpg%22%2C%2213457.jpg%22%2C%2213458.jpg%22%2C%2213459.jpg%22%2C%2213460.jpg%22%2C%2213461.jpg%22%2C%2213462.jpg%22%2C%2213463.jpg%22%2C%2213464.jpg%22%2C%2213465.jpg%22%2C%2213466.jpg%22%2C%2213467.jpg%22%2C%2213468.jpg%22%2C%2213469.jpg%22%2C%2213470.jpg%22%2C%2213471.jpg%22%2C%2213472.jpg%22%2C%2213473.jpg%22%2C%2213474.jpg%22%2C%2213475.jpg%22%2C%2213476.jpg%22%2C%2213477.jpg%22%2C%2213478.jpg%22%2C%2213479.jpg%22%2C%2213480.jpg%22%2C%2213481.jpg%22%2C%2213482.jpg%22%2C%2213483.jpg%22%2C%2213484.jpg%22%2C%2213485.jpg%22%2C%2213486.jpg%22%2C%2213487.jpg%22%2C%2213488.jpg%22%2C%2213489.jpg%22%2C%2213490.jpg%22%2C%2213491.jpg%22%2C%2213492.jpg%22%2C%2213493.jpg%22%2C%2213494.jpg%22%2C%2213495.jpg%22%2C%2213496.jpg%22%2C%2213497.jpg%22%2C%2213498.jpg%22%2C%2213499.jpg%22%2C%2213500.jpg%22%2C%2213501.jpg%22%2C%2213502.jpg%22%2C%2213503.jpg%22%2C%2213504.jpg%22%2C%2213505.jpg%22%2C%2213506.jpg%22%2C%2213507.jpg%22%2C%2213508.jpg%22%2C%2213509.jpg%22%2C%2213510.jpg%22%2C%2213511.jpg%22%2C%2213512.jpg%22%2C%2213513.jpg%22%2C%2213514.jpg%22%2C%2213515.jpg%22%2C%2213516.jpg%22%2C%2213517.jpg%22%2C%2213518.jpg%22%2C%2213519.jpg%22%2C%2213520.jpg%22%2C%2213521.jpg%22%2C%2213522.jpg%22%2C%2213523.jpg%22%2C%2213524.jpg%22%2C%2213525.jpg%22%2C%2213526.jpg%22%2C%2213527.jpg%22%2C%2213528.jpg%22%2C%2213529.jpg%22%2C%2213530.jpg%22%2C%2213531.jpg%22%2C%2213532.jpg%22%2C%2213533.jpg%22%2C%2213534.jpg%22%2C%2213535.jpg%22%2C%2213536.jpg%22%2C%2213537.jpg%22%2C%2213538.jpg%22%2C%2213539.jpg%22%2C%2213540.jpg%22%2C%2213541.jpg%22%2C%2213542.jpg%22%2C%2213543.jpg%22%2C%2213544.jpg%22%2C%2213545.jpg%22%2C%2213546.jpg%22%2C%2213547.jpg%22%2C%2213548.jpg%22%2C%2213549.jpg%22%2C%2213550.jpg%22%2C%2213551.jpg%22%2C%2213552.jpg%22%2C%2213553.jpg%22%2C%2213554.jpg%22%2C%2213555.jpg%22%2C%2213556.jpg%22%2C%2213557.jpg%22%2C%2213558.jpg%22%2C%2213559.jpg%22%2C%2213560.jpg%22%2C%2213561.jpg%22%2C%2213562.jpg%22%2C%2213563.jpg%22%2C%2213564.jpg%22%2C%2213565.jpg%22%2C%2213566.jpg%22%2C%2213567.jpg%22%2C%2213568.jpg%22%2C%2213569.jpg%22%2C%2213570.jpg%22%2C%2213571.jpg%22%2C%2213572.jpg%22%2C%2213573.jpg%22%2C%2213574.jpg%22%2C%2213575.jpg%22%2C%2213576.jpg%22%2C%2213577.jpg%22%2C%2213578.jpg%22%2C%2213579.jpg%22%2C%2213580.jpg%22%2C%2213581.jpg%22%2C%2213582.jpg%22%2C%2213583.jpg%22%2C%2213584.jpg%22%2C%2213585.jpg%22%2C%2213586.jpg%22%2C%2213587.jpg%22%2C%2213588.jpg%22%2C%2213589.jpg%22%2C%2213590.jpg%22%2C%2213591.jpg%22%2C%2213592.jpg%22%2C%2213593.jpg%22%2C%2213594.jpg%22%2C%2213595.jpg%22%2C%2213596.jpg%22%2C%2213597.jpg%22%2C%2213598.jpg%22%2C%2213599.jpg%22%2C%2213600.jpg%22%2C%2213601.jpg%22%2C%2213602.jpg%22%2C%2213603.jpg%22%2C%2213604.jpg%22%2C%2213605.jpg%22%2C%2213606.jpg%22%2C%2213607.jpg%22%2C%2213608.jpg%22%2C%2213609.jpg%22%2C%2213610.jpg%22%2C%2213611.jpg%22%2C%2213612.jpg%22%2C%2213613.jpg%22%2C%2213614.jpg%22%2C%2213615.jpg%22%2C%2213616.jpg%22%2C%2213617.jpg%22%2C%2213618.jpg%22%2C%2213619.jpg%22%2C%2213620.jpg%22%2C%2213621.jpg%22%2C%2213622.jpg%22%2C%2213623.jpg%22%2C%2213624.jpg%22%2C%2213625.jpg%22%2C%2213626.jpg%22%2C%2213627.jpg%22%2C%2213628.jpg%22%2C%2213629.jpg%22%2C%2213630.jpg%22%2C%2213631.jpg%22%2C%2213632.jpg%22%2C%2213633.jpg%22%2C%2213634.jpg%22%2C%2213635.jpg%22%2C%2213636.jpg%22%2C%2213637.jpg%22%2C%2213638.jpg%22%2C%2213639.jpg%22%2C%2213640.jpg%22%2C%2213641.jpg%22%2C%2213642.jpg%22%2C%2213643.jpg%22%2C%2213644.jpg%22%2C%2213645.jpg%22%2C%2213646.jpg%22%2C%2213647.jpg%22%2C%2213648.jpg%22%2C%2213649.jpg%22%2C%2213650.jpg%22%2C%2213651.jpg%22%2C%2213652.jpg%22%2C%2213653.jpg%22%2C%2213654.jpg%22%2C%2213655.jpg%22%2C%2213656.jpg%22%2C%2213657.jpg%22%2C%2213658.jpg%22%2C%2213659.jpg%22%2C%2213660.jpg%22%2C%2213661.jpg%22%2C%2213662.jpg%22%2C%2213663.jpg%22%2C%2213664.jpg%22%2C%2213665.jpg%22%2C%2213666.jpg%22%2C%2213667.jpg%22%2C%2213668.jpg%22%2C%2213669.jpg%22%2C%2213670.jpg%22%2C%2213671.jpg%22%2C%2213672.jpg%22%2C%2213673.jpg%22%2C%2213674.jpg%22%2C%2213675.jpg%22%2C%2213676.jpg%22%2C%2213677.jpg%22%2C%2213678.jpg%22%2C%2213679.jpg%22%2C%2213680.jpg%22%2C%2213681.jpg%22%2C%2213682.jpg%22%2C%2213683.jpg%22%2C%2213684.jpg%22%2C%2213685.jpg%22%2C%2213686.jpg%22%2C%2213687.jpg%22%2C%2213688.jpg%22%2C%2213689.jpg%22%2C%2213690.jpg%22%2C%2213691.jpg%22%2C%2213692.jpg%22%2C%2213693.jpg%22%2C%2213694.jpg%22%2C%2213695.jpg%22%2C%2213696.jpg%22%2C%2213697.jpg%22%2C%2213698.jpg%22%2C%2213699.jpg%22%2C%2213700.jpg%22%2C%2213701.jpg%22%2C%2213702.jpg%22%2C%2213703.jpg%22%2C%2213704.jpg%22%2C%2213705.jpg%22%2C%2213706.jpg%22%2C%2213707.jpg%22%2C%2213708.jpg%22%2C%2213709.jpg%22%2C%2213710.jpg%22%2C%2213711.jpg%22%2C%2213712.jpg%22%2C%2213713.jpg%22%2C%2213714.jpg%22%2C%2213715.jpg%22%2C%2213716.jpg%22%2C%2213717.jpg%22%2C%2213718.jpg%22%2C%2213719.jpg%22%2C%2213720.jpg%22%2C%2213721.jpg%22%2C%2213722.jpg%22%2C%2213723.jpg%22%2C%2213724.jpg%22%2C%2213725.jpg%22%2C%2213726.jpg%22%2C%2213727.jpg%22%2C%2213728.jpg%22%2C%2213729.jpg%22%2C%2213730.jpg%22%2C%2213731.jpg%22%2C%2213732.jpg%22%2C%2213733.jpg%22%2C%2213734.jpg%22%2C%2213735.jpg%22%2C%2213736.jpg%22%2C%2213737.jpg%22%2C%2213738.jpg%22%2C%2213739.jpg%22%2C%2213740.jpg%22%2C%2213741.jpg%22%2C%2213742.jpg%22%2C%2213743.jpg%22%2C%2213744.jpg%22%2C%2213745.jpg%22%2C%2213746.jpg%22%2C%2213747.jpg%22%2C%2213748.jpg%22%2C%2213749.jpg%22%2C%2213750.jpg%22%2C%2213751.jpg%22%2C%2213752.jpg%22%2C%2213753.jpg%22%5D",
            filename="train17.zip",
            md5=False,
            checksum="d38f3f249ad79e66e28f0578a9cfdffdc67292e94b5cdc1762836a11ff6ce6b6",
        ),
        DataLocation(
            url="https://cloud.cs.uni-tuebingen.de/public.php/dav/files/ZZxX65FGnQ8zjBP/Compressed%20Version/images/train/?accept=zip&files=%5B%2213754.jpg%22%2C%2213755.jpg%22%2C%2213756.jpg%22%2C%2213757.jpg%22%2C%2213758.jpg%22%2C%2213759.jpg%22%2C%2213760.jpg%22%2C%2213761.jpg%22%2C%2213762.jpg%22%2C%2213763.jpg%22%2C%2213764.jpg%22%2C%2213765.jpg%22%2C%2213766.jpg%22%2C%2213767.jpg%22%2C%2213768.jpg%22%2C%2213769.jpg%22%2C%2213770.jpg%22%2C%2213771.jpg%22%2C%2213772.jpg%22%2C%2213773.jpg%22%2C%2213774.jpg%22%2C%2213775.jpg%22%2C%2213776.jpg%22%2C%2213777.jpg%22%2C%2213778.jpg%22%2C%2213779.jpg%22%2C%2213780.jpg%22%2C%2213781.jpg%22%2C%2213782.jpg%22%2C%2213783.jpg%22%2C%2213784.jpg%22%2C%2213785.jpg%22%2C%2213786.jpg%22%2C%2213787.jpg%22%2C%2213788.jpg%22%2C%2213789.jpg%22%2C%2213790.jpg%22%2C%2213791.jpg%22%2C%2213792.jpg%22%2C%2213793.jpg%22%2C%2213794.jpg%22%2C%2213795.jpg%22%2C%2213796.jpg%22%2C%2213797.jpg%22%2C%2213798.jpg%22%2C%2213799.jpg%22%2C%2213800.jpg%22%2C%2213801.jpg%22%2C%2213802.jpg%22%2C%2213803.jpg%22%2C%2213804.jpg%22%2C%2213805.jpg%22%2C%2213806.jpg%22%2C%2213807.jpg%22%2C%2213808.jpg%22%2C%2213809.jpg%22%2C%2213810.jpg%22%2C%2213811.jpg%22%2C%2213812.jpg%22%2C%2213813.jpg%22%2C%2213814.jpg%22%2C%2213815.jpg%22%2C%2213816.jpg%22%2C%2213817.jpg%22%2C%2213818.jpg%22%2C%2213819.jpg%22%2C%2213820.jpg%22%2C%2213821.jpg%22%2C%2213822.jpg%22%2C%2213823.jpg%22%2C%2213824.jpg%22%2C%2213825.jpg%22%2C%2213826.jpg%22%2C%2213827.jpg%22%2C%2213828.jpg%22%2C%2213829.jpg%22%2C%2213830.jpg%22%2C%2213831.jpg%22%2C%2213832.jpg%22%2C%2213833.jpg%22%2C%2213834.jpg%22%2C%2213835.jpg%22%2C%2213836.jpg%22%2C%2213837.jpg%22%2C%2213838.jpg%22%2C%2213839.jpg%22%2C%2213840.jpg%22%2C%2213841.jpg%22%2C%2213842.jpg%22%2C%2213843.jpg%22%2C%2213844.jpg%22%2C%2213845.jpg%22%2C%2213846.jpg%22%2C%2213847.jpg%22%2C%2213848.jpg%22%2C%2213849.jpg%22%2C%2213850.jpg%22%2C%2213851.jpg%22%2C%2213852.jpg%22%2C%2213853.jpg%22%2C%2213854.jpg%22%2C%2213855.jpg%22%2C%2213856.jpg%22%2C%2213857.jpg%22%2C%2213858.jpg%22%2C%2213859.jpg%22%2C%2213860.jpg%22%2C%2213861.jpg%22%2C%2213862.jpg%22%2C%2213863.jpg%22%2C%2213864.jpg%22%2C%2213865.jpg%22%2C%2213866.jpg%22%2C%2213867.jpg%22%2C%2213868.jpg%22%2C%2213869.jpg%22%2C%2213870.jpg%22%2C%2213871.jpg%22%2C%2213872.jpg%22%2C%2213873.jpg%22%2C%2213874.jpg%22%2C%2213875.jpg%22%2C%2213876.jpg%22%2C%2213877.jpg%22%2C%2213878.jpg%22%2C%2213879.jpg%22%2C%2213880.jpg%22%2C%2213881.jpg%22%2C%2213882.jpg%22%2C%2213883.jpg%22%2C%2213884.jpg%22%2C%2213885.jpg%22%2C%2213886.jpg%22%2C%2213887.jpg%22%2C%2213888.jpg%22%2C%2213889.jpg%22%2C%2213890.jpg%22%2C%2213891.jpg%22%2C%2213892.jpg%22%2C%2213893.jpg%22%2C%2213894.jpg%22%2C%2213895.jpg%22%2C%2213896.jpg%22%2C%2213897.jpg%22%2C%2213898.jpg%22%2C%2213899.jpg%22%2C%2213900.jpg%22%2C%2213901.jpg%22%2C%2213902.jpg%22%2C%2213903.jpg%22%2C%2213904.jpg%22%2C%2213905.jpg%22%2C%2213906.jpg%22%2C%2214849.jpg%22%2C%2214850.jpg%22%2C%2214851.jpg%22%2C%2214852.jpg%22%2C%2214853.jpg%22%2C%2214854.jpg%22%2C%2214855.jpg%22%2C%2214856.jpg%22%2C%2214857.jpg%22%2C%2214858.jpg%22%2C%2214859.jpg%22%2C%2214860.jpg%22%2C%2214861.jpg%22%2C%2214862.jpg%22%2C%2214863.jpg%22%2C%2214864.jpg%22%2C%2214865.jpg%22%2C%2214866.jpg%22%2C%2214867.jpg%22%2C%2214868.jpg%22%2C%2214869.jpg%22%2C%2214870.jpg%22%2C%2214871.jpg%22%2C%2214872.jpg%22%2C%2214873.jpg%22%2C%2214874.jpg%22%2C%2214875.jpg%22%2C%2214876.jpg%22%2C%2214877.jpg%22%2C%2214878.jpg%22%2C%2214879.jpg%22%2C%2214880.jpg%22%2C%2214881.jpg%22%2C%2214882.jpg%22%2C%2214883.jpg%22%2C%2214884.jpg%22%2C%2214885.jpg%22%2C%2214886.jpg%22%2C%2214887.jpg%22%2C%2214888.jpg%22%2C%2214889.jpg%22%2C%2214890.jpg%22%2C%2214891.jpg%22%2C%2214892.jpg%22%2C%2214893.jpg%22%2C%2214894.jpg%22%2C%2214895.jpg%22%2C%2214896.jpg%22%2C%2214897.jpg%22%2C%2214898.jpg%22%2C%2214899.jpg%22%2C%2214900.jpg%22%2C%2214901.jpg%22%2C%2214902.jpg%22%2C%2214903.jpg%22%2C%2214904.jpg%22%2C%2214905.jpg%22%2C%2214906.jpg%22%2C%2214907.jpg%22%2C%2214908.jpg%22%2C%2214909.jpg%22%2C%2214910.jpg%22%2C%2214911.jpg%22%2C%2214912.jpg%22%2C%2214913.jpg%22%2C%2214914.jpg%22%2C%2214915.jpg%22%2C%2214916.jpg%22%2C%2214917.jpg%22%2C%2214918.jpg%22%2C%2214919.jpg%22%2C%2214920.jpg%22%2C%2214921.jpg%22%2C%2214922.jpg%22%2C%2214923.jpg%22%2C%2214924.jpg%22%2C%2214925.jpg%22%2C%2214926.jpg%22%2C%2214927.jpg%22%2C%2214928.jpg%22%2C%2214929.jpg%22%2C%2214930.jpg%22%2C%2214931.jpg%22%2C%2214932.jpg%22%2C%2214933.jpg%22%2C%2214934.jpg%22%2C%2214935.jpg%22%2C%2214936.jpg%22%2C%2214937.jpg%22%2C%2214938.jpg%22%2C%2214939.jpg%22%2C%2214940.jpg%22%2C%2214941.jpg%22%2C%2214942.jpg%22%2C%2214943.jpg%22%2C%2214944.jpg%22%2C%2214945.jpg%22%2C%2214946.jpg%22%2C%2214947.jpg%22%2C%2214948.jpg%22%2C%2214949.jpg%22%2C%2214950.jpg%22%2C%2214951.jpg%22%2C%2214952.jpg%22%2C%2214953.jpg%22%2C%2214954.jpg%22%2C%2214955.jpg%22%2C%2214956.jpg%22%2C%2214957.jpg%22%2C%2214958.jpg%22%2C%2214959.jpg%22%2C%2214960.jpg%22%2C%2214961.jpg%22%2C%2214962.jpg%22%2C%2214963.jpg%22%2C%2214964.jpg%22%2C%2214965.jpg%22%2C%2214966.jpg%22%2C%2214967.jpg%22%2C%2214968.jpg%22%2C%2214969.jpg%22%2C%2214970.jpg%22%2C%2214971.jpg%22%2C%2214972.jpg%22%2C%2214973.jpg%22%2C%2214974.jpg%22%2C%2214975.jpg%22%2C%2214976.jpg%22%2C%2214977.jpg%22%2C%2214978.jpg%22%2C%2214979.jpg%22%2C%2214980.jpg%22%2C%2214981.jpg%22%2C%2214982.jpg%22%2C%2214983.jpg%22%2C%2214984.jpg%22%2C%2214985.jpg%22%2C%2214986.jpg%22%2C%2214987.jpg%22%2C%2214988.jpg%22%2C%2214989.jpg%22%2C%2214990.jpg%22%2C%2214991.jpg%22%2C%2214992.jpg%22%2C%2214993.jpg%22%2C%2214994.jpg%22%2C%2214995.jpg%22%2C%2214996.jpg%22%2C%2214997.jpg%22%2C%2214998.jpg%22%2C%2214999.jpg%22%2C%2215000.jpg%22%2C%2215001.jpg%22%2C%2215002.jpg%22%2C%2215003.jpg%22%2C%2215004.jpg%22%2C%2215005.jpg%22%2C%2215006.jpg%22%2C%2215007.jpg%22%2C%2215008.jpg%22%2C%2215009.jpg%22%2C%2215010.jpg%22%2C%2215011.jpg%22%2C%2215012.jpg%22%2C%2215013.jpg%22%2C%2215014.jpg%22%2C%2215015.jpg%22%2C%2215016.jpg%22%2C%2215017.jpg%22%2C%2215018.jpg%22%2C%2215019.jpg%22%2C%2215020.jpg%22%2C%2215021.jpg%22%2C%2215022.jpg%22%2C%2215023.jpg%22%2C%2215024.jpg%22%2C%2215025.jpg%22%2C%2215026.jpg%22%2C%2215027.jpg%22%2C%2215028.jpg%22%2C%2215029.jpg%22%2C%2215030.jpg%22%2C%2215031.jpg%22%2C%2215032.jpg%22%2C%2215033.jpg%22%2C%2215034.jpg%22%2C%2215035.jpg%22%2C%2215036.jpg%22%2C%2215037.jpg%22%2C%2215038.jpg%22%2C%2215039.jpg%22%2C%2215040.jpg%22%2C%2215041.jpg%22%2C%2215042.jpg%22%2C%2215043.jpg%22%2C%2215044.jpg%22%2C%2215045.jpg%22%2C%2215046.jpg%22%2C%2215047.jpg%22%2C%2215048.jpg%22%2C%2215049.jpg%22%2C%2215050.jpg%22%2C%2215051.jpg%22%2C%2215052.jpg%22%2C%2215053.jpg%22%2C%2215054.jpg%22%2C%2215055.jpg%22%2C%2215056.jpg%22%2C%2215057.jpg%22%2C%2215058.jpg%22%2C%2215059.jpg%22%2C%2215060.jpg%22%2C%2215061.jpg%22%2C%2215062.jpg%22%2C%2215063.jpg%22%2C%2215064.jpg%22%2C%2215065.jpg%22%2C%2215066.jpg%22%2C%2215067.jpg%22%2C%2215068.jpg%22%2C%2215069.jpg%22%2C%2215070.jpg%22%2C%2215071.jpg%22%2C%2215072.jpg%22%2C%2215073.jpg%22%2C%2215074.jpg%22%2C%2215075.jpg%22%2C%2215076.jpg%22%2C%2215077.jpg%22%2C%2215078.jpg%22%2C%2215079.jpg%22%2C%2215080.jpg%22%2C%2215081.jpg%22%2C%2215082.jpg%22%2C%2215083.jpg%22%2C%2215084.jpg%22%2C%2215085.jpg%22%2C%2215086.jpg%22%2C%2215087.jpg%22%2C%2215088.jpg%22%2C%2215089.jpg%22%2C%2215090.jpg%22%2C%2215091.jpg%22%2C%2215092.jpg%22%2C%2215093.jpg%22%2C%2215094.jpg%22%2C%2215095.jpg%22%2C%2215096.jpg%22%2C%2215097.jpg%22%2C%2215098.jpg%22%2C%2215099.jpg%22%2C%2215100.jpg%22%2C%2215101.jpg%22%2C%2215102.jpg%22%2C%2215103.jpg%22%2C%2215104.jpg%22%2C%2215105.jpg%22%2C%2215106.jpg%22%2C%2215107.jpg%22%2C%2215108.jpg%22%2C%2215109.jpg%22%2C%2215110.jpg%22%2C%2215111.jpg%22%2C%2215112.jpg%22%2C%2215113.jpg%22%2C%2215114.jpg%22%2C%2215115.jpg%22%2C%2215116.jpg%22%2C%2215117.jpg%22%2C%2215118.jpg%22%2C%2215119.jpg%22%2C%2215120.jpg%22%2C%2215121.jpg%22%2C%2215122.jpg%22%2C%2215123.jpg%22%2C%2215124.jpg%22%2C%2215125.jpg%22%2C%2215126.jpg%22%2C%2215127.jpg%22%2C%2215128.jpg%22%2C%2215129.jpg%22%2C%2215130.jpg%22%2C%2215131.jpg%22%2C%2215132.jpg%22%2C%2215133.jpg%22%2C%2215134.jpg%22%2C%2215135.jpg%22%2C%2215136.jpg%22%2C%2215137.jpg%22%5D",
            filename="train18.zip",
            md5=False,
            checksum="8dd873fb19b31f5054ece479d6d2454155e24c4d66398f858a9ce42f6cffb9bc",
        ),
        DataLocation(
            url="https://cloud.cs.uni-tuebingen.de/public.php/dav/files/ZZxX65FGnQ8zjBP/Compressed%20Version/images/train/?accept=zip&files=%5B%2215138.jpg%22%2C%2215139.jpg%22%2C%2215140.jpg%22%2C%2215141.jpg%22%2C%2215142.jpg%22%2C%2215143.jpg%22%2C%2215144.jpg%22%2C%2215145.jpg%22%2C%2215146.jpg%22%2C%2215147.jpg%22%2C%2215148.jpg%22%2C%2215149.jpg%22%2C%2215150.jpg%22%2C%2215151.jpg%22%2C%2215152.jpg%22%2C%2215153.jpg%22%2C%2215154.jpg%22%2C%2215155.jpg%22%2C%2215156.jpg%22%2C%2215157.jpg%22%2C%2215158.jpg%22%2C%2215159.jpg%22%2C%2215160.jpg%22%2C%2215161.jpg%22%2C%2215162.jpg%22%2C%2215163.jpg%22%2C%2215164.jpg%22%2C%2215165.jpg%22%2C%2215166.jpg%22%2C%2215167.jpg%22%2C%2215168.jpg%22%2C%2215169.jpg%22%2C%2215170.jpg%22%2C%2215171.jpg%22%2C%2215172.jpg%22%2C%2215173.jpg%22%2C%2215174.jpg%22%2C%2215175.jpg%22%2C%2215176.jpg%22%2C%2215177.jpg%22%2C%2215178.jpg%22%2C%2215179.jpg%22%2C%2215180.jpg%22%2C%2215181.jpg%22%2C%2215182.jpg%22%2C%2215183.jpg%22%2C%2215184.jpg%22%2C%2215185.jpg%22%2C%2215186.jpg%22%2C%2215187.jpg%22%2C%2215188.jpg%22%2C%2215189.jpg%22%2C%2215190.jpg%22%2C%2215191.jpg%22%2C%2215192.jpg%22%2C%2215193.jpg%22%2C%2215194.jpg%22%2C%2215195.jpg%22%2C%2215196.jpg%22%2C%2215197.jpg%22%2C%2215198.jpg%22%2C%2215199.jpg%22%2C%2215200.jpg%22%2C%2215201.jpg%22%2C%2215202.jpg%22%2C%2215203.jpg%22%2C%2215204.jpg%22%2C%2215205.jpg%22%2C%2215206.jpg%22%2C%2215207.jpg%22%2C%2215208.jpg%22%2C%2215209.jpg%22%2C%2215210.jpg%22%2C%2215211.jpg%22%2C%2215212.jpg%22%2C%2215213.jpg%22%2C%2215214.jpg%22%2C%2215215.jpg%22%2C%2215216.jpg%22%2C%2215217.jpg%22%2C%2215218.jpg%22%2C%2215219.jpg%22%2C%2215220.jpg%22%2C%2215221.jpg%22%2C%2215222.jpg%22%2C%2215223.jpg%22%2C%2215224.jpg%22%2C%2215225.jpg%22%2C%2215226.jpg%22%2C%2215227.jpg%22%2C%2215228.jpg%22%2C%2215229.jpg%22%2C%2215230.jpg%22%2C%2215231.jpg%22%2C%2215232.jpg%22%2C%2215233.jpg%22%2C%2215234.jpg%22%2C%2215235.jpg%22%2C%2215236.jpg%22%2C%2215237.jpg%22%2C%2215238.jpg%22%2C%2215239.jpg%22%2C%2215240.jpg%22%2C%2215241.jpg%22%2C%2215242.jpg%22%2C%2215243.jpg%22%2C%2215244.jpg%22%2C%2215245.jpg%22%2C%2215246.jpg%22%2C%2215247.jpg%22%2C%2215248.jpg%22%2C%2215249.jpg%22%2C%2215250.jpg%22%2C%2215251.jpg%22%2C%2215252.jpg%22%2C%2215253.jpg%22%2C%2215254.jpg%22%2C%2215255.jpg%22%2C%2215256.jpg%22%2C%2215257.jpg%22%2C%2215258.jpg%22%2C%2215259.jpg%22%2C%2215260.jpg%22%2C%2215261.jpg%22%2C%2215262.jpg%22%2C%2215263.jpg%22%2C%2215264.jpg%22%2C%2215265.jpg%22%2C%2215266.jpg%22%2C%2215267.jpg%22%2C%2215268.jpg%22%2C%2215269.jpg%22%2C%2215270.jpg%22%2C%2215271.jpg%22%2C%2215272.jpg%22%2C%2215273.jpg%22%2C%2215274.jpg%22%2C%2215275.jpg%22%2C%2215276.jpg%22%2C%2215277.jpg%22%2C%2215278.jpg%22%2C%2215279.jpg%22%2C%2215280.jpg%22%2C%2215281.jpg%22%2C%2215282.jpg%22%2C%2215283.jpg%22%2C%2215284.jpg%22%2C%2215285.jpg%22%2C%2215286.jpg%22%2C%2215287.jpg%22%2C%2215288.jpg%22%2C%2215289.jpg%22%2C%2215290.jpg%22%2C%2215291.jpg%22%2C%2215292.jpg%22%2C%2215293.jpg%22%2C%2215294.jpg%22%2C%2215295.jpg%22%2C%2215296.jpg%22%2C%2215297.jpg%22%2C%2215298.jpg%22%2C%2215299.jpg%22%2C%2215300.jpg%22%2C%2215301.jpg%22%2C%2215302.jpg%22%2C%2215303.jpg%22%2C%2215304.jpg%22%2C%2215305.jpg%22%2C%2215306.jpg%22%2C%2215307.jpg%22%2C%2215308.jpg%22%2C%2215309.jpg%22%2C%2215310.jpg%22%2C%2215311.jpg%22%2C%2215312.jpg%22%2C%2215313.jpg%22%2C%2215314.jpg%22%2C%2215315.jpg%22%2C%2215316.jpg%22%2C%2215317.jpg%22%2C%2215318.jpg%22%2C%2215319.jpg%22%2C%2215320.jpg%22%2C%2215321.jpg%22%2C%2215322.jpg%22%2C%2215323.jpg%22%2C%2215324.jpg%22%2C%2215325.jpg%22%2C%2215326.jpg%22%2C%2215327.jpg%22%2C%2215328.jpg%22%2C%2215329.jpg%22%2C%2215330.jpg%22%2C%2215331.jpg%22%2C%2215332.jpg%22%2C%2215333.jpg%22%2C%2215334.jpg%22%2C%2215335.jpg%22%2C%2215336.jpg%22%2C%2215337.jpg%22%2C%2215808.jpg%22%2C%2215809.jpg%22%2C%2215810.jpg%22%2C%2215811.jpg%22%2C%2215812.jpg%22%2C%2215813.jpg%22%2C%2215814.jpg%22%2C%2215815.jpg%22%2C%2215816.jpg%22%2C%2215817.jpg%22%2C%2215818.jpg%22%2C%2215819.jpg%22%2C%2215820.jpg%22%2C%2215821.jpg%22%2C%2215822.jpg%22%2C%2215890.jpg%22%2C%2215891.jpg%22%2C%2215892.jpg%22%2C%2215893.jpg%22%2C%2215894.jpg%22%2C%2215895.jpg%22%2C%2215896.jpg%22%2C%2215897.jpg%22%2C%2215898.jpg%22%2C%2215899.jpg%22%2C%2215900.jpg%22%2C%2215901.jpg%22%2C%2215902.jpg%22%2C%2215903.jpg%22%2C%2215904.jpg%22%2C%2215905.jpg%22%2C%2215906.jpg%22%2C%2215907.jpg%22%2C%2215908.jpg%22%2C%2215909.jpg%22%2C%2215910.jpg%22%2C%2215911.jpg%22%2C%2215912.jpg%22%2C%2215913.jpg%22%2C%2215914.jpg%22%2C%2215915.jpg%22%2C%2215916.jpg%22%2C%2215917.jpg%22%2C%2215918.jpg%22%2C%2215919.jpg%22%2C%2215920.jpg%22%2C%2215921.jpg%22%2C%2215922.jpg%22%2C%2215923.jpg%22%2C%2215924.jpg%22%2C%2215925.jpg%22%2C%2215926.jpg%22%2C%2215927.jpg%22%2C%2215928.jpg%22%2C%2215929.jpg%22%2C%2215930.jpg%22%2C%2215931.jpg%22%2C%2215932.jpg%22%2C%2215933.jpg%22%2C%2215934.jpg%22%2C%2215935.jpg%22%2C%2215936.jpg%22%2C%2215937.jpg%22%2C%2215938.jpg%22%2C%2215939.jpg%22%2C%2215940.jpg%22%2C%2215941.jpg%22%2C%2215942.jpg%22%2C%2215943.jpg%22%2C%2215944.jpg%22%2C%2215945.jpg%22%2C%2215946.jpg%22%2C%2215947.jpg%22%2C%2215948.jpg%22%2C%2215949.jpg%22%2C%2215950.jpg%22%2C%2215951.jpg%22%2C%2215952.jpg%22%2C%2215953.jpg%22%2C%2215954.jpg%22%2C%2215955.jpg%22%2C%2215956.jpg%22%2C%2215957.jpg%22%2C%2215958.jpg%22%2C%2215959.jpg%22%2C%2215960.jpg%22%2C%2215961.jpg%22%2C%2215962.jpg%22%2C%2215963.jpg%22%2C%2215964.jpg%22%2C%2215965.jpg%22%2C%2215966.jpg%22%2C%2215967.jpg%22%2C%2215968.jpg%22%2C%2215969.jpg%22%2C%2216024.jpg%22%2C%2216025.jpg%22%2C%2216026.jpg%22%2C%2216027.jpg%22%2C%2216028.jpg%22%2C%2216029.jpg%22%2C%2216030.jpg%22%2C%2216031.jpg%22%2C%2216032.jpg%22%2C%2216033.jpg%22%2C%2216034.jpg%22%2C%2216035.jpg%22%2C%2216036.jpg%22%2C%2216037.jpg%22%2C%2216038.jpg%22%2C%2216039.jpg%22%2C%2216040.jpg%22%2C%2216041.jpg%22%2C%2216042.jpg%22%2C%2216043.jpg%22%2C%2216044.jpg%22%2C%2216045.jpg%22%2C%2216046.jpg%22%2C%2216047.jpg%22%2C%2216048.jpg%22%2C%2216049.jpg%22%2C%2216050.jpg%22%2C%2216051.jpg%22%2C%2216052.jpg%22%2C%2216053.jpg%22%2C%2216054.jpg%22%2C%2216055.jpg%22%2C%2216056.jpg%22%2C%2216057.jpg%22%2C%2216058.jpg%22%2C%2216059.jpg%22%2C%2216060.jpg%22%2C%2216061.jpg%22%2C%2216062.jpg%22%2C%2216063.jpg%22%2C%2216064.jpg%22%2C%2216065.jpg%22%2C%2216066.jpg%22%2C%2216067.jpg%22%2C%2216068.jpg%22%2C%2216069.jpg%22%2C%2216070.jpg%22%2C%2216071.jpg%22%2C%2216072.jpg%22%2C%2216073.jpg%22%2C%2216074.jpg%22%2C%2216111.jpg%22%2C%2216113.jpg%22%2C%2216114.jpg%22%2C%2216115.jpg%22%2C%2216116.jpg%22%2C%2216117.jpg%22%2C%2216118.jpg%22%2C%2216119.jpg%22%2C%2216120.jpg%22%2C%2216121.jpg%22%2C%2216122.jpg%22%2C%2216123.jpg%22%2C%2216124.jpg%22%2C%2216125.jpg%22%2C%2216126.jpg%22%2C%2216127.jpg%22%2C%2216128.jpg%22%2C%2216129.jpg%22%2C%2216130.jpg%22%2C%2216131.jpg%22%2C%2216132.jpg%22%2C%2216133.jpg%22%2C%2216134.jpg%22%2C%2216135.jpg%22%2C%2216136.jpg%22%2C%2216137.jpg%22%2C%2216138.jpg%22%2C%2216139.jpg%22%2C%2216140.jpg%22%2C%2216141.jpg%22%2C%2216142.jpg%22%2C%2216143.jpg%22%2C%2216144.jpg%22%2C%2216145.jpg%22%2C%2216146.jpg%22%2C%2216147.jpg%22%2C%2216148.jpg%22%2C%2216149.jpg%22%2C%2216150.jpg%22%2C%2216151.jpg%22%2C%2216152.jpg%22%2C%2216153.jpg%22%2C%2216154.jpg%22%2C%2216155.jpg%22%2C%2216156.jpg%22%2C%2216157.jpg%22%2C%2216158.jpg%22%2C%2216159.jpg%22%2C%2216160.jpg%22%2C%2216161.jpg%22%2C%2216162.jpg%22%2C%2216163.jpg%22%2C%2216164.jpg%22%2C%2216165.jpg%22%2C%2216166.jpg%22%2C%2216167.jpg%22%2C%2216168.jpg%22%2C%2216169.jpg%22%2C%2216170.jpg%22%2C%2216171.jpg%22%2C%2216172.jpg%22%2C%2216173.jpg%22%2C%2216174.jpg%22%2C%2216175.jpg%22%2C%2216176.jpg%22%2C%2216177.jpg%22%2C%2216178.jpg%22%2C%2216179.jpg%22%2C%2216180.jpg%22%2C%2216181.jpg%22%2C%2216182.jpg%22%2C%2216183.jpg%22%2C%2216184.jpg%22%2C%2216185.jpg%22%2C%2216186.jpg%22%2C%2216187.jpg%22%2C%2216188.jpg%22%2C%2216189.jpg%22%2C%2216190.jpg%22%2C%2216191.jpg%22%2C%2216192.jpg%22%2C%2216270.jpg%22%2C%2216271.jpg%22%2C%2216272.jpg%22%2C%2216273.jpg%22%2C%2216274.jpg%22%2C%2216275.jpg%22%2C%2216276.jpg%22%2C%2216277.jpg%22%2C%2216278.jpg%22%2C%2216279.jpg%22%2C%2216280.jpg%22%2C%2216281.jpg%22%2C%2216282.jpg%22%2C%2216283.jpg%22%2C%2216284.jpg%22%5D",
            filename="train19.zip",
            md5=False,
            checksum="0aa1c6bcdb64e3cbb3d3b637ddecd69b3d8be038f49f34f43d9c33512158d60b",
        ),
        DataLocation(
            url="https://cloud.cs.uni-tuebingen.de/public.php/dav/files/ZZxX65FGnQ8zjBP/Compressed%20Version/images/train/?accept=zip&files=%5B%2216285.jpg%22%2C%2216286.jpg%22%2C%2216287.jpg%22%2C%2216288.jpg%22%2C%2216289.jpg%22%2C%2216290.jpg%22%2C%2216291.jpg%22%2C%2216292.jpg%22%2C%2216293.jpg%22%2C%2216294.jpg%22%2C%2216295.jpg%22%2C%2216296.jpg%22%2C%2216297.jpg%22%2C%2216298.jpg%22%2C%2216299.jpg%22%2C%2216300.jpg%22%2C%2216301.jpg%22%2C%2216302.jpg%22%2C%2216303.jpg%22%2C%2216304.jpg%22%2C%2216305.jpg%22%2C%2216306.jpg%22%2C%2216307.jpg%22%2C%2216308.jpg%22%2C%2216309.jpg%22%2C%2216310.jpg%22%2C%2216311.jpg%22%2C%2216312.jpg%22%2C%2216313.jpg%22%2C%2216314.jpg%22%2C%2216315.jpg%22%2C%2216316.jpg%22%2C%2216317.jpg%22%2C%2216318.jpg%22%2C%2216319.jpg%22%2C%2216320.jpg%22%2C%2216321.jpg%22%2C%2216322.jpg%22%2C%2216323.jpg%22%2C%2216324.jpg%22%2C%2216325.jpg%22%2C%2216326.jpg%22%2C%2216327.jpg%22%2C%2216328.jpg%22%2C%2216329.jpg%22%2C%2216330.jpg%22%2C%2216331.jpg%22%2C%2216332.jpg%22%2C%2216333.jpg%22%2C%2216334.jpg%22%2C%2216335.jpg%22%2C%2216336.jpg%22%2C%2216337.jpg%22%2C%2216338.jpg%22%2C%2216339.jpg%22%2C%2216340.jpg%22%2C%2216341.jpg%22%2C%2216342.jpg%22%2C%2216343.jpg%22%2C%2216344.jpg%22%2C%2216345.jpg%22%2C%2216346.jpg%22%2C%2216347.jpg%22%2C%2216348.jpg%22%2C%2216349.jpg%22%2C%2216350.jpg%22%2C%2216351.jpg%22%2C%2216352.jpg%22%2C%2216353.jpg%22%2C%2216354.jpg%22%2C%2216355.jpg%22%2C%2216356.jpg%22%2C%2216357.jpg%22%2C%2216358.jpg%22%2C%2216359.jpg%22%2C%2216360.jpg%22%2C%2216361.jpg%22%2C%2216362.jpg%22%2C%2216363.jpg%22%2C%2216364.jpg%22%2C%2216365.jpg%22%2C%2216366.jpg%22%2C%2216367.jpg%22%2C%2216368.jpg%22%2C%2216369.jpg%22%2C%2216370.jpg%22%2C%2216371.jpg%22%2C%2216372.jpg%22%2C%2216373.jpg%22%2C%2216374.jpg%22%2C%2216375.jpg%22%2C%2216376.jpg%22%2C%2216377.jpg%22%2C%2216378.jpg%22%2C%2216379.jpg%22%2C%2216380.jpg%22%2C%2216381.jpg%22%2C%2216382.jpg%22%2C%2216394.jpg%22%2C%2216395.jpg%22%2C%2216396.jpg%22%2C%2216397.jpg%22%2C%2216398.jpg%22%2C%2216399.jpg%22%2C%2216400.jpg%22%2C%2216401.jpg%22%2C%2216402.jpg%22%2C%2216403.jpg%22%2C%2216404.jpg%22%2C%2216405.jpg%22%2C%2216406.jpg%22%2C%2216407.jpg%22%2C%2216408.jpg%22%2C%2216409.jpg%22%2C%2216410.jpg%22%2C%2216411.jpg%22%2C%2216412.jpg%22%2C%2216413.jpg%22%2C%2216414.jpg%22%2C%2216415.jpg%22%2C%2216416.jpg%22%2C%2216417.jpg%22%2C%2216418.jpg%22%2C%2216419.jpg%22%2C%2216420.jpg%22%2C%2216421.jpg%22%2C%2216422.jpg%22%2C%2216423.jpg%22%2C%2216424.jpg%22%2C%2216425.jpg%22%2C%2216426.jpg%22%2C%2216427.jpg%22%2C%2216428.jpg%22%2C%2216429.jpg%22%2C%2216430.jpg%22%2C%2216431.jpg%22%2C%2216432.jpg%22%2C%2216433.jpg%22%2C%2216434.jpg%22%2C%2216435.jpg%22%2C%2216436.jpg%22%2C%2216437.jpg%22%2C%2216438.jpg%22%2C%2216439.jpg%22%2C%2216440.jpg%22%2C%2216441.jpg%22%2C%2216442.jpg%22%2C%2216443.jpg%22%2C%2216444.jpg%22%2C%2216445.jpg%22%2C%2216446.jpg%22%2C%2216447.jpg%22%2C%2216448.jpg%22%2C%2216449.jpg%22%2C%2216450.jpg%22%2C%2216451.jpg%22%2C%2216452.jpg%22%2C%2216453.jpg%22%2C%2216454.jpg%22%2C%2216455.jpg%22%2C%2216456.jpg%22%2C%2216457.jpg%22%2C%2216458.jpg%22%2C%2216459.jpg%22%2C%2216460.jpg%22%2C%2216461.jpg%22%2C%2216462.jpg%22%2C%2216463.jpg%22%2C%2216464.jpg%22%2C%2216465.jpg%22%2C%2216466.jpg%22%2C%2216467.jpg%22%2C%2216468.jpg%22%2C%2216469.jpg%22%2C%2216470.jpg%22%2C%2216471.jpg%22%2C%2216472.jpg%22%2C%2216473.jpg%22%2C%2216474.jpg%22%2C%2216475.jpg%22%2C%2216476.jpg%22%2C%2216477.jpg%22%2C%2216478.jpg%22%2C%2216479.jpg%22%2C%2216480.jpg%22%2C%2216481.jpg%22%2C%2216482.jpg%22%2C%2216483.jpg%22%2C%2216484.jpg%22%2C%2216485.jpg%22%2C%2216486.jpg%22%2C%2216487.jpg%22%2C%2216488.jpg%22%2C%2216489.jpg%22%2C%2216490.jpg%22%2C%2216491.jpg%22%2C%2216492.jpg%22%2C%2216493.jpg%22%2C%2216494.jpg%22%2C%2216495.jpg%22%2C%2216496.jpg%22%2C%2216497.jpg%22%2C%2216498.jpg%22%2C%2216499.jpg%22%2C%2216500.jpg%22%2C%2216501.jpg%22%2C%2216502.jpg%22%2C%2216503.jpg%22%2C%2216504.jpg%22%2C%2216505.jpg%22%2C%2216506.jpg%22%2C%2216507.jpg%22%2C%2216508.jpg%22%2C%2216509.jpg%22%2C%2216510.jpg%22%2C%2216511.jpg%22%2C%2216512.jpg%22%2C%2216513.jpg%22%2C%2216514.jpg%22%2C%2216515.jpg%22%2C%2216516.jpg%22%2C%2216517.jpg%22%2C%2216518.jpg%22%2C%2216519.jpg%22%2C%2216520.jpg%22%2C%2216521.jpg%22%2C%2216522.jpg%22%2C%2216523.jpg%22%2C%2216524.jpg%22%2C%2216525.jpg%22%2C%2216526.jpg%22%2C%2216527.jpg%22%2C%2216528.jpg%22%2C%2216529.jpg%22%2C%2216530.jpg%22%2C%2216531.jpg%22%2C%2216532.jpg%22%2C%2216533.jpg%22%2C%2216534.jpg%22%2C%2216535.jpg%22%2C%2216536.jpg%22%2C%2216537.jpg%22%2C%2216538.jpg%22%2C%2216539.jpg%22%2C%2216540.jpg%22%2C%2216541.jpg%22%2C%2216542.jpg%22%2C%2216543.jpg%22%2C%2216544.jpg%22%2C%2216545.jpg%22%2C%2216546.jpg%22%2C%2216547.jpg%22%2C%2216548.jpg%22%2C%2216549.jpg%22%2C%2216550.jpg%22%2C%2216551.jpg%22%2C%2216552.jpg%22%2C%2216553.jpg%22%2C%2216554.jpg%22%2C%2216555.jpg%22%2C%2216556.jpg%22%2C%2216557.jpg%22%2C%2216558.jpg%22%2C%2216559.jpg%22%2C%2216560.jpg%22%2C%2216561.jpg%22%2C%2216562.jpg%22%2C%2216563.jpg%22%2C%2216564.jpg%22%2C%2216565.jpg%22%2C%2216566.jpg%22%2C%2216567.jpg%22%2C%2216568.jpg%22%2C%2216569.jpg%22%2C%2216570.jpg%22%2C%2216571.jpg%22%2C%2216572.jpg%22%2C%2216573.jpg%22%2C%2216574.jpg%22%2C%2216575.jpg%22%2C%2216576.jpg%22%2C%2216577.jpg%22%2C%2216578.jpg%22%2C%2216579.jpg%22%2C%2216580.jpg%22%2C%2216581.jpg%22%2C%2216582.jpg%22%2C%2216583.jpg%22%2C%2216584.jpg%22%2C%2216585.jpg%22%2C%2216586.jpg%22%2C%2216587.jpg%22%2C%2216588.jpg%22%2C%2216589.jpg%22%2C%2216590.jpg%22%2C%2216591.jpg%22%2C%2216592.jpg%22%2C%2216593.jpg%22%2C%2216594.jpg%22%2C%2216595.jpg%22%2C%2216596.jpg%22%2C%2216597.jpg%22%2C%2216598.jpg%22%2C%2216599.jpg%22%2C%2216600.jpg%22%2C%2216601.jpg%22%2C%2216602.jpg%22%2C%2216603.jpg%22%2C%2216604.jpg%22%2C%2216605.jpg%22%2C%2216606.jpg%22%2C%2216607.jpg%22%2C%2216608.jpg%22%2C%2216609.jpg%22%2C%2216610.jpg%22%2C%2216611.jpg%22%2C%2216612.jpg%22%2C%2216613.jpg%22%2C%2216614.jpg%22%2C%2216615.jpg%22%2C%2216616.jpg%22%2C%2216617.jpg%22%2C%2216618.jpg%22%2C%2216619.jpg%22%2C%2216620.jpg%22%2C%2216621.jpg%22%2C%2216622.jpg%22%2C%2216623.jpg%22%2C%2216624.jpg%22%2C%2216625.jpg%22%2C%2216626.jpg%22%2C%2216627.jpg%22%2C%2216628.jpg%22%2C%2216629.jpg%22%2C%2216630.jpg%22%2C%2216631.jpg%22%2C%2216632.jpg%22%2C%2216633.jpg%22%2C%2216634.jpg%22%2C%2216635.jpg%22%2C%2216636.jpg%22%2C%2216637.jpg%22%2C%2216638.jpg%22%2C%2216639.jpg%22%2C%2216640.jpg%22%2C%2216641.jpg%22%2C%2216642.jpg%22%2C%2216643.jpg%22%2C%2216644.jpg%22%2C%2216645.jpg%22%2C%2216646.jpg%22%2C%2216647.jpg%22%2C%2216648.jpg%22%2C%2216649.jpg%22%2C%2216650.jpg%22%2C%2216651.jpg%22%2C%2216652.jpg%22%2C%2216653.jpg%22%2C%2216654.jpg%22%2C%2216655.jpg%22%2C%2216656.jpg%22%2C%2216657.jpg%22%2C%2216658.jpg%22%2C%2216659.jpg%22%2C%2216660.jpg%22%2C%2216661.jpg%22%2C%2216662.jpg%22%2C%2216663.jpg%22%2C%2216664.jpg%22%2C%2216665.jpg%22%2C%2216666.jpg%22%2C%2216667.jpg%22%2C%2216668.jpg%22%2C%2216669.jpg%22%2C%2216670.jpg%22%2C%2216671.jpg%22%2C%2216672.jpg%22%2C%2216673.jpg%22%2C%2216674.jpg%22%2C%2216675.jpg%22%2C%2216676.jpg%22%2C%2216677.jpg%22%2C%2216678.jpg%22%2C%2216679.jpg%22%2C%2216680.jpg%22%2C%2216681.jpg%22%2C%2216682.jpg%22%2C%2216683.jpg%22%2C%2216684.jpg%22%2C%2216685.jpg%22%2C%2216686.jpg%22%2C%2216687.jpg%22%2C%2216688.jpg%22%2C%2216689.jpg%22%2C%2216690.jpg%22%2C%2216691.jpg%22%2C%2216692.jpg%22%2C%2216693.jpg%22%2C%2216694.jpg%22%2C%2216695.jpg%22%2C%2216696.jpg%22%2C%2216697.jpg%22%2C%2216698.jpg%22%2C%2216699.jpg%22%2C%2216700.jpg%22%2C%2216701.jpg%22%2C%2216702.jpg%22%2C%2216703.jpg%22%2C%2216704.jpg%22%2C%2216705.jpg%22%2C%2216706.jpg%22%2C%2216707.jpg%22%2C%2216708.jpg%22%2C%2216709.jpg%22%2C%2216710.jpg%22%2C%2216711.jpg%22%2C%2216712.jpg%22%2C%2216713.jpg%22%2C%2216714.jpg%22%2C%2216715.jpg%22%2C%2216716.jpg%22%2C%2216717.jpg%22%2C%2216718.jpg%22%2C%2216719.jpg%22%2C%2216720.jpg%22%2C%2216721.jpg%22%2C%2216722.jpg%22%2C%2216723.jpg%22%2C%2216724.jpg%22%2C%2216725.jpg%22%2C%2216726.jpg%22%2C%2216727.jpg%22%2C%2216728.jpg%22%2C%2216729.jpg%22%2C%2216730.jpg%22%2C%2216731.jpg%22%5D",
            filename="train20.zip",
            md5=False,
            checksum="e6219803dc1e597307a1c5ba26641d1157eb87f280442bd841db06698ea65582",
        ),
        DataLocation(
            url="https://cloud.cs.uni-tuebingen.de/public.php/dav/files/ZZxX65FGnQ8zjBP/Compressed%20Version/images/val/?accept=zip&files=%5B%2248.jpg%22%2C%2271.jpg%22%2C%2285.jpg%22%2C%22113.jpg%22%2C%22116.jpg%22%2C%22119.jpg%22%2C%22124.jpg%22%2C%22143.jpg%22%2C%22151.jpg%22%2C%22162.jpg%22%2C%22172.jpg%22%2C%22179.jpg%22%2C%22187.jpg%22%2C%22189.jpg%22%2C%22242.jpg%22%2C%22267.jpg%22%2C%22276.jpg%22%2C%22300.jpg%22%2C%22310.jpg%22%2C%22351.jpg%22%2C%22352.jpg%22%2C%22370.jpg%22%2C%22379.jpg%22%2C%22382.jpg%22%2C%22393.jpg%22%2C%22460.jpg%22%2C%22465.jpg%22%2C%22479.jpg%22%2C%22490.jpg%22%2C%22501.jpg%22%2C%22510.jpg%22%2C%22518.jpg%22%2C%22526.jpg%22%2C%22529.jpg%22%2C%22537.jpg%22%2C%22550.jpg%22%2C%22552.jpg%22%2C%22566.jpg%22%2C%22595.jpg%22%2C%22604.jpg%22%2C%22619.jpg%22%2C%22622.jpg%22%2C%22628.jpg%22%2C%22637.jpg%22%2C%22646.jpg%22%2C%22652.jpg%22%2C%22655.jpg%22%2C%22669.jpg%22%2C%22676.jpg%22%2C%22690.jpg%22%2C%22693.jpg%22%2C%22704.jpg%22%2C%22735.jpg%22%2C%22765.jpg%22%2C%22767.jpg%22%2C%22774.jpg%22%2C%22780.jpg%22%2C%22792.jpg%22%2C%22803.jpg%22%2C%22813.jpg%22%2C%22824.jpg%22%2C%22828.jpg%22%2C%22830.jpg%22%2C%22863.jpg%22%2C%22868.jpg%22%2C%22884.jpg%22%2C%22887.jpg%22%2C%22944.jpg%22%2C%22984.jpg%22%2C%221003.jpg%22%2C%221018.jpg%22%2C%221032.jpg%22%2C%221036.jpg%22%2C%221037.jpg%22%2C%221073.jpg%22%2C%221089.jpg%22%2C%221096.jpg%22%2C%221109.jpg%22%2C%221123.jpg%22%2C%221125.jpg%22%2C%221129.jpg%22%2C%221136.jpg%22%2C%221144.jpg%22%2C%221157.jpg%22%2C%221171.jpg%22%2C%221181.jpg%22%2C%221224.jpg%22%2C%221230.jpg%22%2C%221240.jpg%22%2C%221254.jpg%22%2C%221283.jpg%22%2C%221309.jpg%22%2C%221350.jpg%22%2C%221389.jpg%22%2C%221436.jpg%22%2C%221471.jpg%22%2C%221474.jpg%22%2C%221479.jpg%22%2C%221500.jpg%22%2C%221506.jpg%22%2C%221520.jpg%22%2C%221542.jpg%22%2C%221549.jpg%22%2C%221550.jpg%22%2C%221560.jpg%22%2C%221572.jpg%22%2C%221617.jpg%22%2C%221621.jpg%22%2C%221661.jpg%22%2C%221663.jpg%22%2C%221716.jpg%22%2C%221724.jpg%22%2C%221765.jpg%22%2C%221771.jpg%22%2C%221774.jpg%22%2C%221786.jpg%22%2C%221793.jpg%22%2C%221821.jpg%22%2C%221824.jpg%22%2C%221835.jpg%22%2C%221842.jpg%22%2C%221845.jpg%22%2C%221852.jpg%22%2C%221869.jpg%22%2C%221871.jpg%22%2C%221880.jpg%22%2C%221896.jpg%22%2C%221900.jpg%22%2C%221908.jpg%22%2C%221910.jpg%22%2C%221911.jpg%22%2C%221915.jpg%22%2C%221931.jpg%22%2C%221944.jpg%22%2C%221959.jpg%22%2C%221975.jpg%22%2C%221990.jpg%22%2C%222005.jpg%22%2C%222018.jpg%22%2C%222019.jpg%22%2C%222035.jpg%22%2C%222049.jpg%22%2C%222058.jpg%22%2C%222060.jpg%22%2C%222070.jpg%22%2C%222077.jpg%22%2C%222078.jpg%22%2C%222094.jpg%22%2C%222109.jpg%22%2C%222110.jpg%22%2C%222114.jpg%22%2C%222160.jpg%22%2C%222162.jpg%22%2C%222185.jpg%22%2C%222198.jpg%22%2C%222211.jpg%22%2C%222228.jpg%22%2C%222231.jpg%22%2C%222234.jpg%22%2C%222241.jpg%22%2C%222276.jpg%22%2C%222286.jpg%22%2C%222360.jpg%22%2C%222364.jpg%22%2C%222366.jpg%22%2C%222371.jpg%22%2C%222378.jpg%22%2C%222382.jpg%22%2C%222390.jpg%22%2C%222398.jpg%22%2C%222428.jpg%22%2C%222453.jpg%22%2C%222459.jpg%22%2C%222486.jpg%22%2C%222488.jpg%22%2C%222497.jpg%22%2C%222503.jpg%22%2C%222507.jpg%22%2C%222508.jpg%22%2C%222513.jpg%22%2C%222535.jpg%22%2C%222551.jpg%22%2C%222563.jpg%22%2C%222583.jpg%22%2C%222593.jpg%22%2C%222610.jpg%22%2C%222621.jpg%22%2C%222636.jpg%22%2C%222642.jpg%22%2C%222648.jpg%22%2C%222655.jpg%22%2C%222684.jpg%22%2C%222690.jpg%22%2C%222698.jpg%22%2C%222716.jpg%22%2C%222718.jpg%22%2C%222749.jpg%22%2C%222754.jpg%22%2C%222759.jpg%22%2C%222781.jpg%22%2C%222782.jpg%22%2C%222885.jpg%22%2C%222887.jpg%22%2C%222890.jpg%22%2C%222894.jpg%22%2C%222898.jpg%22%2C%222905.jpg%22%2C%222911.jpg%22%2C%222931.jpg%22%2C%222963.jpg%22%2C%222983.jpg%22%2C%222990.jpg%22%2C%223001.jpg%22%2C%223004.jpg%22%2C%223010.jpg%22%2C%223011.jpg%22%2C%223012.jpg%22%2C%223016.jpg%22%2C%223022.jpg%22%2C%223023.jpg%22%2C%223025.jpg%22%2C%223033.jpg%22%2C%223060.jpg%22%2C%223066.jpg%22%2C%223067.jpg%22%2C%223070.jpg%22%2C%223071.jpg%22%2C%223075.jpg%22%2C%223078.jpg%22%2C%223102.jpg%22%2C%223111.jpg%22%2C%223126.jpg%22%2C%223138.jpg%22%2C%223157.jpg%22%2C%223181.jpg%22%2C%223189.jpg%22%2C%223203.jpg%22%2C%223211.jpg%22%2C%223220.jpg%22%2C%223231.jpg%22%2C%223233.jpg%22%2C%223234.jpg%22%2C%223239.jpg%22%2C%223242.jpg%22%2C%223247.jpg%22%2C%223255.jpg%22%2C%223257.jpg%22%2C%223264.jpg%22%2C%223287.jpg%22%2C%223290.jpg%22%2C%223304.jpg%22%2C%223312.jpg%22%2C%223318.jpg%22%2C%223328.jpg%22%2C%223329.jpg%22%2C%223336.jpg%22%2C%223372.jpg%22%2C%223379.jpg%22%2C%223387.jpg%22%2C%223464.jpg%22%2C%223465.jpg%22%2C%223466.jpg%22%2C%223467.jpg%22%2C%223468.jpg%22%2C%223469.jpg%22%2C%223470.jpg%22%2C%223471.jpg%22%2C%223472.jpg%22%2C%223473.jpg%22%2C%223474.jpg%22%2C%223475.jpg%22%2C%223476.jpg%22%2C%223477.jpg%22%2C%223478.jpg%22%2C%223479.jpg%22%2C%223480.jpg%22%2C%223481.jpg%22%2C%223482.jpg%22%2C%223483.jpg%22%2C%223484.jpg%22%2C%223485.jpg%22%2C%223486.jpg%22%2C%223487.jpg%22%2C%223488.jpg%22%2C%223489.jpg%22%2C%223490.jpg%22%2C%223491.jpg%22%2C%223492.jpg%22%2C%223493.jpg%22%2C%223494.jpg%22%2C%223495.jpg%22%2C%223496.jpg%22%2C%223497.jpg%22%2C%223498.jpg%22%2C%223499.jpg%22%2C%223500.jpg%22%2C%223501.jpg%22%2C%223502.jpg%22%2C%223503.jpg%22%2C%223504.jpg%22%2C%223505.jpg%22%2C%223506.jpg%22%2C%223507.jpg%22%2C%223508.jpg%22%2C%223509.jpg%22%2C%223510.jpg%22%2C%223511.jpg%22%2C%223512.jpg%22%2C%223513.jpg%22%2C%223514.jpg%22%2C%223515.jpg%22%2C%223516.jpg%22%2C%223517.jpg%22%2C%223518.jpg%22%2C%223519.jpg%22%2C%223520.jpg%22%2C%223521.jpg%22%2C%223522.jpg%22%2C%223523.jpg%22%2C%223653.jpg%22%2C%223654.jpg%22%2C%223655.jpg%22%2C%223656.jpg%22%2C%223657.jpg%22%2C%223658.jpg%22%2C%223716.jpg%22%2C%223717.jpg%22%2C%223718.jpg%22%2C%223719.jpg%22%2C%223720.jpg%22%2C%223721.jpg%22%2C%223722.jpg%22%2C%224013.jpg%22%2C%224014.jpg%22%2C%224015.jpg%22%2C%224016.jpg%22%2C%224017.jpg%22%2C%224018.jpg%22%2C%224019.jpg%22%2C%224020.jpg%22%2C%224021.jpg%22%2C%224022.jpg%22%2C%224023.jpg%22%2C%224024.jpg%22%2C%224025.jpg%22%2C%224026.jpg%22%2C%224027.jpg%22%2C%224028.jpg%22%2C%224029.jpg%22%2C%224030.jpg%22%2C%224031.jpg%22%2C%224032.jpg%22%2C%224033.jpg%22%2C%224034.jpg%22%2C%224035.jpg%22%2C%224036.jpg%22%2C%224037.jpg%22%2C%224038.jpg%22%2C%224039.jpg%22%2C%224040.jpg%22%2C%224041.jpg%22%2C%224042.jpg%22%2C%224043.jpg%22%2C%224044.jpg%22%2C%224045.jpg%22%2C%224046.jpg%22%2C%224047.jpg%22%2C%224048.jpg%22%2C%224049.jpg%22%2C%224050.jpg%22%2C%224051.jpg%22%2C%224052.jpg%22%2C%224053.jpg%22%2C%224054.jpg%22%2C%224237.jpg%22%2C%224238.jpg%22%2C%224239.jpg%22%2C%224240.jpg%22%2C%224241.jpg%22%2C%224242.jpg%22%2C%224243.jpg%22%2C%224436.jpg%22%2C%224437.jpg%22%2C%224438.jpg%22%2C%224439.jpg%22%2C%224440.jpg%22%2C%224441.jpg%22%2C%224442.jpg%22%2C%224482.jpg%22%2C%224483.jpg%22%2C%224484.jpg%22%2C%224485.jpg%22%2C%224486.jpg%22%2C%224487.jpg%22%2C%224488.jpg%22%2C%224865.jpg%22%2C%224866.jpg%22%2C%224867.jpg%22%2C%224868.jpg%22%2C%224869.jpg%22%2C%224870.jpg%22%2C%224871.jpg%22%2C%224872.jpg%22%2C%224873.jpg%22%2C%224874.jpg%22%2C%224875.jpg%22%2C%224876.jpg%22%2C%224877.jpg%22%2C%224878.jpg%22%2C%224879.jpg%22%2C%224880.jpg%22%2C%224881.jpg%22%2C%224882.jpg%22%2C%224883.jpg%22%2C%224884.jpg%22%2C%224885.jpg%22%2C%224886.jpg%22%2C%224887.jpg%22%2C%224888.jpg%22%2C%224889.jpg%22%2C%224890.jpg%22%2C%224891.jpg%22%2C%224892.jpg%22%2C%224893.jpg%22%2C%224894.jpg%22%2C%224895.jpg%22%2C%224896.jpg%22%2C%224897.jpg%22%2C%224898.jpg%22%2C%224899.jpg%22%2C%224900.jpg%22%2C%224901.jpg%22%2C%224902.jpg%22%2C%224903.jpg%22%2C%224904.jpg%22%2C%224905.jpg%22%2C%224906.jpg%22%2C%224907.jpg%22%2C%224908.jpg%22%2C%224909.jpg%22%2C%224910.jpg%22%2C%224911.jpg%22%2C%224912.jpg%22%2C%224913.jpg%22%2C%224914.jpg%22%2C%224915.jpg%22%2C%224916.jpg%22%2C%224917.jpg%22%2C%224918.jpg%22%2C%224919.jpg%22%5D",
            filename="val1.zip",
            md5=False,
            checksum="2468b71467d81b7730d740c8dc5c12cdf16154eafbc27a02ebd5667ab178b799",
        ),
        DataLocation(
            url="https://cloud.cs.uni-tuebingen.de/public.php/dav/files/ZZxX65FGnQ8zjBP/Compressed%20Version/images/val/?accept=zip&files=%5B%224920.jpg%22%2C%224921.jpg%22%2C%224922.jpg%22%2C%224923.jpg%22%2C%224924.jpg%22%2C%225146.jpg%22%2C%225147.jpg%22%2C%225148.jpg%22%2C%225149.jpg%22%2C%225150.jpg%22%2C%225151.jpg%22%2C%225152.jpg%22%2C%225256.jpg%22%2C%225257.jpg%22%2C%225258.jpg%22%2C%225308.jpg%22%2C%225309.jpg%22%2C%225310.jpg%22%2C%225311.jpg%22%2C%225312.jpg%22%2C%225313.jpg%22%2C%225314.jpg%22%2C%225543.jpg%22%2C%225544.jpg%22%2C%225545.jpg%22%2C%225546.jpg%22%2C%225547.jpg%22%2C%225548.jpg%22%2C%225549.jpg%22%2C%225550.jpg%22%2C%225551.jpg%22%2C%225552.jpg%22%2C%225553.jpg%22%2C%225554.jpg%22%2C%225555.jpg%22%2C%225556.jpg%22%2C%225557.jpg%22%2C%225558.jpg%22%2C%225559.jpg%22%2C%225560.jpg%22%2C%225561.jpg%22%2C%225562.jpg%22%2C%225563.jpg%22%2C%225564.jpg%22%2C%225565.jpg%22%2C%225566.jpg%22%2C%225567.jpg%22%2C%225568.jpg%22%2C%225569.jpg%22%2C%225570.jpg%22%2C%225571.jpg%22%2C%225572.jpg%22%2C%225573.jpg%22%2C%225574.jpg%22%2C%225575.jpg%22%2C%225576.jpg%22%2C%225577.jpg%22%2C%226039.jpg%22%2C%226040.jpg%22%2C%226041.jpg%22%2C%226042.jpg%22%2C%226043.jpg%22%2C%226044.jpg%22%2C%226045.jpg%22%2C%226046.jpg%22%2C%226047.jpg%22%2C%226048.jpg%22%2C%226049.jpg%22%2C%226050.jpg%22%2C%226051.jpg%22%2C%226052.jpg%22%2C%226053.jpg%22%2C%226054.jpg%22%2C%226055.jpg%22%2C%226056.jpg%22%2C%226057.jpg%22%2C%226058.jpg%22%2C%226059.jpg%22%2C%226060.jpg%22%2C%226061.jpg%22%2C%226062.jpg%22%2C%226063.jpg%22%2C%226064.jpg%22%2C%226065.jpg%22%2C%226066.jpg%22%2C%226067.jpg%22%2C%226068.jpg%22%2C%226069.jpg%22%2C%226070.jpg%22%2C%226071.jpg%22%2C%226072.jpg%22%2C%226073.jpg%22%2C%226074.jpg%22%2C%226075.jpg%22%2C%226076.jpg%22%2C%226077.jpg%22%2C%226078.jpg%22%2C%226079.jpg%22%2C%226080.jpg%22%2C%226081.jpg%22%2C%226082.jpg%22%2C%226083.jpg%22%2C%226084.jpg%22%2C%226085.jpg%22%2C%226086.jpg%22%2C%226087.jpg%22%2C%226088.jpg%22%2C%226089.jpg%22%2C%226090.jpg%22%2C%226091.jpg%22%2C%226092.jpg%22%2C%226093.jpg%22%2C%226094.jpg%22%2C%226095.jpg%22%2C%226096.jpg%22%2C%226097.jpg%22%2C%226098.jpg%22%2C%226641.jpg%22%2C%226642.jpg%22%2C%226643.jpg%22%2C%226644.jpg%22%2C%226645.jpg%22%2C%226646.jpg%22%2C%226647.jpg%22%2C%226648.jpg%22%2C%226649.jpg%22%2C%226650.jpg%22%2C%226651.jpg%22%2C%226652.jpg%22%2C%226653.jpg%22%2C%226654.jpg%22%2C%226655.jpg%22%2C%226656.jpg%22%2C%226657.jpg%22%2C%226658.jpg%22%2C%226659.jpg%22%2C%226660.jpg%22%2C%226661.jpg%22%2C%226662.jpg%22%2C%226663.jpg%22%2C%226664.jpg%22%2C%226665.jpg%22%2C%226666.jpg%22%2C%226667.jpg%22%2C%226668.jpg%22%2C%226669.jpg%22%2C%226670.jpg%22%2C%226671.jpg%22%2C%226672.jpg%22%2C%226673.jpg%22%2C%226674.jpg%22%2C%226675.jpg%22%2C%226676.jpg%22%2C%226677.jpg%22%2C%226678.jpg%22%2C%226679.jpg%22%2C%226680.jpg%22%2C%226681.jpg%22%2C%226682.jpg%22%2C%226683.jpg%22%2C%226684.jpg%22%2C%226685.jpg%22%2C%226686.jpg%22%2C%226687.jpg%22%2C%226688.jpg%22%2C%226689.jpg%22%2C%226690.jpg%22%2C%226691.jpg%22%2C%226692.jpg%22%2C%226693.jpg%22%2C%226694.jpg%22%2C%226695.jpg%22%2C%226696.jpg%22%2C%226697.jpg%22%2C%226698.jpg%22%2C%226699.jpg%22%2C%226700.jpg%22%2C%227172.jpg%22%2C%227173.jpg%22%2C%227174.jpg%22%2C%227175.jpg%22%2C%227176.jpg%22%2C%227177.jpg%22%2C%227178.jpg%22%2C%227179.jpg%22%2C%227180.jpg%22%2C%227181.jpg%22%2C%227182.jpg%22%2C%227183.jpg%22%2C%227184.jpg%22%2C%227185.jpg%22%2C%227186.jpg%22%2C%227187.jpg%22%2C%227188.jpg%22%2C%227189.jpg%22%2C%227190.jpg%22%2C%227191.jpg%22%2C%227192.jpg%22%2C%227193.jpg%22%2C%227194.jpg%22%2C%227195.jpg%22%2C%227196.jpg%22%2C%227197.jpg%22%2C%227198.jpg%22%2C%227199.jpg%22%2C%227200.jpg%22%2C%227201.jpg%22%2C%227202.jpg%22%2C%227203.jpg%22%2C%227204.jpg%22%2C%227205.jpg%22%2C%227206.jpg%22%2C%227207.jpg%22%2C%227208.jpg%22%2C%227209.jpg%22%2C%227210.jpg%22%2C%227211.jpg%22%2C%227212.jpg%22%2C%227213.jpg%22%2C%227214.jpg%22%2C%227215.jpg%22%2C%227216.jpg%22%2C%227217.jpg%22%2C%227218.jpg%22%2C%227219.jpg%22%2C%227220.jpg%22%2C%227747.jpg%22%2C%227748.jpg%22%2C%227749.jpg%22%2C%227750.jpg%22%2C%227751.jpg%22%2C%227752.jpg%22%2C%227753.jpg%22%2C%227754.jpg%22%2C%227755.jpg%22%2C%227756.jpg%22%2C%227757.jpg%22%2C%227758.jpg%22%2C%227759.jpg%22%2C%227760.jpg%22%2C%227761.jpg%22%2C%227762.jpg%22%2C%227763.jpg%22%2C%227764.jpg%22%2C%227765.jpg%22%2C%227766.jpg%22%2C%227767.jpg%22%2C%227768.jpg%22%2C%227769.jpg%22%2C%227770.jpg%22%2C%227771.jpg%22%2C%227772.jpg%22%2C%227773.jpg%22%2C%227774.jpg%22%2C%227775.jpg%22%2C%227776.jpg%22%2C%227777.jpg%22%2C%227778.jpg%22%2C%227779.jpg%22%2C%227780.jpg%22%2C%227781.jpg%22%2C%227782.jpg%22%2C%227783.jpg%22%2C%227784.jpg%22%2C%227785.jpg%22%2C%227786.jpg%22%2C%227787.jpg%22%2C%227788.jpg%22%2C%227789.jpg%22%2C%227790.jpg%22%2C%227791.jpg%22%2C%227792.jpg%22%2C%227793.jpg%22%2C%227794.jpg%22%2C%227795.jpg%22%2C%227796.jpg%22%2C%227797.jpg%22%2C%227798.jpg%22%2C%227799.jpg%22%2C%227800.jpg%22%2C%227801.jpg%22%2C%227802.jpg%22%2C%227803.jpg%22%2C%227804.jpg%22%2C%227805.jpg%22%2C%227806.jpg%22%2C%228115.jpg%22%2C%228116.jpg%22%2C%228117.jpg%22%2C%228118.jpg%22%2C%228119.jpg%22%2C%228618.jpg%22%2C%228619.jpg%22%2C%228620.jpg%22%2C%228621.jpg%22%2C%228622.jpg%22%2C%228623.jpg%22%2C%228624.jpg%22%2C%228625.jpg%22%2C%228626.jpg%22%2C%228627.jpg%22%2C%228628.jpg%22%2C%228629.jpg%22%2C%228630.jpg%22%2C%228631.jpg%22%2C%228632.jpg%22%2C%228633.jpg%22%2C%228634.jpg%22%2C%228635.jpg%22%2C%228636.jpg%22%2C%228637.jpg%22%2C%228638.jpg%22%2C%228639.jpg%22%2C%228640.jpg%22%2C%228641.jpg%22%2C%228642.jpg%22%2C%228643.jpg%22%2C%228644.jpg%22%2C%228645.jpg%22%2C%228646.jpg%22%2C%228647.jpg%22%2C%228648.jpg%22%2C%228649.jpg%22%2C%228650.jpg%22%2C%228651.jpg%22%2C%228652.jpg%22%2C%228653.jpg%22%2C%228654.jpg%22%2C%228655.jpg%22%2C%228656.jpg%22%2C%228657.jpg%22%2C%228658.jpg%22%2C%228659.jpg%22%2C%228660.jpg%22%2C%228661.jpg%22%2C%228662.jpg%22%2C%228663.jpg%22%2C%228664.jpg%22%2C%228665.jpg%22%2C%228666.jpg%22%2C%228667.jpg%22%2C%228668.jpg%22%2C%228669.jpg%22%2C%228670.jpg%22%2C%228671.jpg%22%2C%228672.jpg%22%2C%228673.jpg%22%2C%228674.jpg%22%2C%228675.jpg%22%2C%228676.jpg%22%2C%228677.jpg%22%2C%229216.jpg%22%2C%229217.jpg%22%2C%229218.jpg%22%2C%229219.jpg%22%2C%229220.jpg%22%2C%229221.jpg%22%2C%229222.jpg%22%2C%229223.jpg%22%2C%229224.jpg%22%2C%229225.jpg%22%2C%229226.jpg%22%2C%229227.jpg%22%2C%229228.jpg%22%2C%229229.jpg%22%2C%229230.jpg%22%2C%229231.jpg%22%2C%229232.jpg%22%2C%229233.jpg%22%2C%229234.jpg%22%2C%229235.jpg%22%2C%229236.jpg%22%2C%229237.jpg%22%2C%229238.jpg%22%2C%229239.jpg%22%2C%229240.jpg%22%2C%229241.jpg%22%2C%229242.jpg%22%2C%229243.jpg%22%2C%229244.jpg%22%2C%229245.jpg%22%2C%229246.jpg%22%2C%229247.jpg%22%2C%229248.jpg%22%2C%229249.jpg%22%2C%229250.jpg%22%2C%229251.jpg%22%2C%229252.jpg%22%2C%229253.jpg%22%2C%229254.jpg%22%2C%229255.jpg%22%2C%229256.jpg%22%2C%229257.jpg%22%2C%229258.jpg%22%2C%229259.jpg%22%2C%229260.jpg%22%2C%229261.jpg%22%2C%229262.jpg%22%2C%229263.jpg%22%2C%229264.jpg%22%2C%229265.jpg%22%2C%229266.jpg%22%2C%229267.jpg%22%2C%229268.jpg%22%2C%229269.jpg%22%2C%229270.jpg%22%2C%229271.jpg%22%2C%229272.jpg%22%2C%229273.jpg%22%2C%229274.jpg%22%2C%229812.jpg%22%2C%229813.jpg%22%2C%229814.jpg%22%2C%229815.jpg%22%2C%229816.jpg%22%2C%229817.jpg%22%2C%229818.jpg%22%2C%229819.jpg%22%2C%229820.jpg%22%2C%229821.jpg%22%2C%229822.jpg%22%2C%229823.jpg%22%2C%229824.jpg%22%2C%229825.jpg%22%2C%229826.jpg%22%2C%229827.jpg%22%2C%229828.jpg%22%2C%229829.jpg%22%2C%229830.jpg%22%2C%229831.jpg%22%2C%229832.jpg%22%2C%229833.jpg%22%2C%229834.jpg%22%2C%229835.jpg%22%2C%229836.jpg%22%2C%229837.jpg%22%2C%229838.jpg%22%2C%229839.jpg%22%2C%229840.jpg%22%2C%229841.jpg%22%2C%229842.jpg%22%2C%229843.jpg%22%2C%229844.jpg%22%2C%229845.jpg%22%2C%229846.jpg%22%2C%229847.jpg%22%2C%229848.jpg%22%2C%229849.jpg%22%2C%229850.jpg%22%2C%229851.jpg%22%5D",
            filename="val2.zip",
            md5=False,
            checksum="6512acabfb1fe5e9b0514ea6b2fae7ed9171b489274870516a3a079043451ff6",
        ),
        DataLocation(
            url="https://cloud.cs.uni-tuebingen.de/public.php/dav/files/ZZxX65FGnQ8zjBP/Compressed%20Version/images/val/?accept=zip&files=%5B%229852.jpg%22%2C%229853.jpg%22%2C%229854.jpg%22%2C%229855.jpg%22%2C%229856.jpg%22%2C%229857.jpg%22%2C%229858.jpg%22%2C%229859.jpg%22%2C%229860.jpg%22%2C%229861.jpg%22%2C%229862.jpg%22%2C%2210416.jpg%22%2C%2210417.jpg%22%2C%2210418.jpg%22%2C%2210419.jpg%22%2C%2210420.jpg%22%2C%2210421.jpg%22%2C%2210422.jpg%22%2C%2210423.jpg%22%2C%2210424.jpg%22%2C%2210425.jpg%22%2C%2210426.jpg%22%2C%2210427.jpg%22%2C%2210428.jpg%22%2C%2210429.jpg%22%2C%2210430.jpg%22%2C%2210431.jpg%22%2C%2210432.jpg%22%2C%2210433.jpg%22%2C%2210434.jpg%22%2C%2210435.jpg%22%2C%2210436.jpg%22%2C%2210437.jpg%22%2C%2210438.jpg%22%2C%2210439.jpg%22%2C%2210440.jpg%22%2C%2210441.jpg%22%2C%2210442.jpg%22%2C%2210443.jpg%22%2C%2210444.jpg%22%2C%2210445.jpg%22%2C%2210446.jpg%22%2C%2210447.jpg%22%2C%2210448.jpg%22%2C%2210449.jpg%22%2C%2210450.jpg%22%2C%2210451.jpg%22%2C%2210452.jpg%22%2C%2210453.jpg%22%2C%2210454.jpg%22%2C%2210455.jpg%22%2C%2210456.jpg%22%2C%2210457.jpg%22%2C%2210458.jpg%22%2C%2210459.jpg%22%2C%2210460.jpg%22%2C%2210461.jpg%22%2C%2210462.jpg%22%2C%2210463.jpg%22%2C%2210464.jpg%22%2C%2210465.jpg%22%2C%2210740.jpg%22%2C%2210741.jpg%22%2C%2210742.jpg%22%2C%2210743.jpg%22%2C%2210744.jpg%22%2C%2210745.jpg%22%2C%2210746.jpg%22%2C%2210747.jpg%22%2C%2210748.jpg%22%2C%2210749.jpg%22%2C%2210750.jpg%22%2C%2210751.jpg%22%2C%2210752.jpg%22%2C%2210753.jpg%22%2C%2210754.jpg%22%2C%2210755.jpg%22%2C%2210756.jpg%22%2C%2210757.jpg%22%2C%2210758.jpg%22%2C%2210759.jpg%22%2C%2210760.jpg%22%2C%2210761.jpg%22%2C%2210762.jpg%22%2C%2210763.jpg%22%2C%2210764.jpg%22%2C%2210765.jpg%22%2C%2210766.jpg%22%2C%2210767.jpg%22%2C%2210768.jpg%22%2C%2210769.jpg%22%2C%2210770.jpg%22%2C%2210771.jpg%22%2C%2210772.jpg%22%2C%2210773.jpg%22%2C%2210774.jpg%22%2C%2210775.jpg%22%2C%2210776.jpg%22%2C%2210777.jpg%22%2C%2210778.jpg%22%2C%2210779.jpg%22%2C%2210780.jpg%22%2C%2210781.jpg%22%2C%2210782.jpg%22%2C%2210783.jpg%22%2C%2210784.jpg%22%2C%2210785.jpg%22%2C%2210786.jpg%22%2C%2210787.jpg%22%2C%2210788.jpg%22%2C%2210789.jpg%22%2C%2210790.jpg%22%2C%2210791.jpg%22%2C%2210792.jpg%22%2C%2210793.jpg%22%2C%2210794.jpg%22%2C%2210795.jpg%22%2C%2210796.jpg%22%2C%2210797.jpg%22%2C%2210798.jpg%22%2C%2210799.jpg%22%2C%2210800.jpg%22%2C%2210801.jpg%22%2C%2210802.jpg%22%2C%2210803.jpg%22%2C%2210804.jpg%22%2C%2210805.jpg%22%2C%2210806.jpg%22%2C%2210807.jpg%22%2C%2210808.jpg%22%2C%2210809.jpg%22%2C%2210810.jpg%22%2C%2210811.jpg%22%2C%2210812.jpg%22%2C%2210813.jpg%22%2C%2210814.jpg%22%2C%2210815.jpg%22%2C%2210816.jpg%22%2C%2210817.jpg%22%2C%2210818.jpg%22%2C%2210819.jpg%22%2C%2210820.jpg%22%2C%2210821.jpg%22%2C%2210822.jpg%22%2C%2210823.jpg%22%2C%2210824.jpg%22%2C%2210825.jpg%22%2C%2210826.jpg%22%2C%2210827.jpg%22%2C%2210828.jpg%22%2C%2210829.jpg%22%2C%2210830.jpg%22%2C%2210831.jpg%22%2C%2210832.jpg%22%2C%2210833.jpg%22%2C%2210834.jpg%22%2C%2210835.jpg%22%2C%2211233.jpg%22%2C%2211234.jpg%22%2C%2211235.jpg%22%2C%2211236.jpg%22%2C%2211237.jpg%22%2C%2211247.jpg%22%2C%2211248.jpg%22%2C%2211249.jpg%22%2C%2211250.jpg%22%2C%2211251.jpg%22%2C%2211252.jpg%22%2C%2211253.jpg%22%2C%2211580.jpg%22%2C%2211581.jpg%22%2C%2211582.jpg%22%2C%2211583.jpg%22%2C%2211584.jpg%22%2C%2211585.jpg%22%2C%2211586.jpg%22%2C%2211587.jpg%22%2C%2211588.jpg%22%2C%2211589.jpg%22%2C%2211590.jpg%22%2C%2211591.jpg%22%2C%2211592.jpg%22%2C%2211593.jpg%22%2C%2211594.jpg%22%2C%2211595.jpg%22%2C%2211596.jpg%22%2C%2211597.jpg%22%2C%2211598.jpg%22%2C%2211599.jpg%22%2C%2211600.jpg%22%2C%2211601.jpg%22%2C%2211602.jpg%22%2C%2211603.jpg%22%2C%2211604.jpg%22%2C%2211605.jpg%22%2C%2211606.jpg%22%2C%2211607.jpg%22%2C%2211608.jpg%22%2C%2211609.jpg%22%2C%2211610.jpg%22%2C%2211611.jpg%22%2C%2211612.jpg%22%2C%2211613.jpg%22%2C%2211614.jpg%22%2C%2211615.jpg%22%2C%2211616.jpg%22%2C%2211617.jpg%22%2C%2211618.jpg%22%2C%2211619.jpg%22%2C%2211620.jpg%22%2C%2211621.jpg%22%2C%2211622.jpg%22%2C%2211623.jpg%22%2C%2212006.jpg%22%2C%2212007.jpg%22%2C%2212008.jpg%22%2C%2212009.jpg%22%2C%2212010.jpg%22%2C%2212011.jpg%22%2C%2212012.jpg%22%2C%2212013.jpg%22%2C%2212014.jpg%22%2C%2212015.jpg%22%2C%2212016.jpg%22%2C%2212017.jpg%22%2C%2212018.jpg%22%2C%2212019.jpg%22%2C%2212020.jpg%22%2C%2212021.jpg%22%2C%2212022.jpg%22%2C%2212023.jpg%22%2C%2212024.jpg%22%2C%2212025.jpg%22%2C%2212026.jpg%22%2C%2212027.jpg%22%2C%2212028.jpg%22%2C%2212029.jpg%22%2C%2212030.jpg%22%2C%2212031.jpg%22%2C%2212032.jpg%22%2C%2212033.jpg%22%2C%2212034.jpg%22%2C%2212035.jpg%22%2C%2212036.jpg%22%2C%2212037.jpg%22%2C%2212038.jpg%22%2C%2212039.jpg%22%2C%2212040.jpg%22%2C%2212041.jpg%22%2C%2212042.jpg%22%2C%2212043.jpg%22%2C%2212044.jpg%22%2C%2212045.jpg%22%2C%2212046.jpg%22%2C%2212047.jpg%22%2C%2212048.jpg%22%2C%2212049.jpg%22%2C%2212050.jpg%22%2C%2212553.jpg%22%2C%2212554.jpg%22%2C%2212555.jpg%22%2C%2212556.jpg%22%2C%2212557.jpg%22%2C%2212558.jpg%22%2C%2212559.jpg%22%2C%2212560.jpg%22%2C%2212561.jpg%22%2C%2212562.jpg%22%2C%2212563.jpg%22%2C%2212564.jpg%22%2C%2212565.jpg%22%2C%2212566.jpg%22%2C%2212567.jpg%22%2C%2212568.jpg%22%2C%2212569.jpg%22%2C%2212570.jpg%22%2C%2212571.jpg%22%2C%2212572.jpg%22%2C%2212573.jpg%22%2C%2212574.jpg%22%2C%2212575.jpg%22%2C%2212576.jpg%22%2C%2212577.jpg%22%2C%2212578.jpg%22%2C%2212579.jpg%22%2C%2212580.jpg%22%2C%2212581.jpg%22%2C%2212582.jpg%22%2C%2212583.jpg%22%2C%2212584.jpg%22%2C%2212585.jpg%22%2C%2212586.jpg%22%2C%2212592.jpg%22%2C%2212593.jpg%22%2C%2212594.jpg%22%2C%2212595.jpg%22%2C%2213907.jpg%22%2C%2213908.jpg%22%2C%2213909.jpg%22%2C%2213910.jpg%22%2C%2213911.jpg%22%2C%2213912.jpg%22%2C%2213913.jpg%22%2C%2213914.jpg%22%2C%2213915.jpg%22%2C%2213916.jpg%22%2C%2213917.jpg%22%2C%2213918.jpg%22%2C%2213919.jpg%22%2C%2213920.jpg%22%2C%2213921.jpg%22%2C%2213922.jpg%22%2C%2213923.jpg%22%2C%2213924.jpg%22%2C%2213925.jpg%22%2C%2213926.jpg%22%2C%2213927.jpg%22%2C%2213928.jpg%22%2C%2213929.jpg%22%2C%2213930.jpg%22%2C%2213931.jpg%22%2C%2213932.jpg%22%2C%2213933.jpg%22%2C%2213934.jpg%22%2C%2213935.jpg%22%2C%2213936.jpg%22%2C%2213937.jpg%22%2C%2213938.jpg%22%2C%2213939.jpg%22%2C%2213940.jpg%22%2C%2213941.jpg%22%2C%2213942.jpg%22%2C%2213943.jpg%22%2C%2213944.jpg%22%2C%2213945.jpg%22%2C%2213946.jpg%22%2C%2213947.jpg%22%2C%2213948.jpg%22%2C%2213949.jpg%22%2C%2213950.jpg%22%2C%2213951.jpg%22%2C%2213952.jpg%22%2C%2213953.jpg%22%2C%2213954.jpg%22%2C%2213955.jpg%22%2C%2213956.jpg%22%2C%2213957.jpg%22%2C%2213958.jpg%22%2C%2213959.jpg%22%2C%2213960.jpg%22%2C%2213961.jpg%22%2C%2213962.jpg%22%2C%2213963.jpg%22%2C%2213964.jpg%22%2C%2213965.jpg%22%2C%2213966.jpg%22%2C%2213967.jpg%22%2C%2213968.jpg%22%2C%2213969.jpg%22%2C%2213970.jpg%22%2C%2213971.jpg%22%2C%2213972.jpg%22%2C%2213973.jpg%22%2C%2213974.jpg%22%2C%2213975.jpg%22%5D",
            filename="val3.zip",
            md5=False,
            checksum="bf46361583b6e7d4e8c7450bede239f975a5b70d1ece2b5e3ce9c4d036c252c0",
        ),
        DataLocation(
            url="https://cloud.cs.uni-tuebingen.de/public.php/dav/files/ZZxX65FGnQ8zjBP/Compressed%20Version/images/val/?accept=zip&files=%5B%2214061.jpg%22%2C%2214062.jpg%22%2C%2214063.jpg%22%2C%2214064.jpg%22%2C%2214065.jpg%22%2C%2214066.jpg%22%2C%2214067.jpg%22%2C%2214068.jpg%22%2C%2214069.jpg%22%2C%2214070.jpg%22%2C%2214071.jpg%22%2C%2214072.jpg%22%2C%2214073.jpg%22%2C%2214074.jpg%22%2C%2214075.jpg%22%2C%2214076.jpg%22%2C%2214077.jpg%22%2C%2214078.jpg%22%2C%2214079.jpg%22%2C%2214080.jpg%22%2C%2214081.jpg%22%2C%2214082.jpg%22%2C%2214083.jpg%22%2C%2214084.jpg%22%2C%2214085.jpg%22%2C%2214086.jpg%22%2C%2214087.jpg%22%2C%2215338.jpg%22%2C%2215339.jpg%22%2C%2215340.jpg%22%2C%2215341.jpg%22%2C%2215342.jpg%22%2C%2215343.jpg%22%2C%2215344.jpg%22%2C%2215345.jpg%22%2C%2215346.jpg%22%2C%2215347.jpg%22%2C%2215348.jpg%22%2C%2215349.jpg%22%2C%2215350.jpg%22%2C%2215351.jpg%22%2C%2215352.jpg%22%2C%2215353.jpg%22%2C%2215354.jpg%22%2C%2215355.jpg%22%2C%2215356.jpg%22%2C%2215357.jpg%22%2C%2215358.jpg%22%2C%2215359.jpg%22%2C%2215360.jpg%22%2C%2215361.jpg%22%2C%2215362.jpg%22%2C%2215363.jpg%22%2C%2215364.jpg%22%2C%2215365.jpg%22%2C%2215366.jpg%22%2C%2215367.jpg%22%2C%2215368.jpg%22%2C%2215369.jpg%22%2C%2215370.jpg%22%2C%2215371.jpg%22%2C%2215372.jpg%22%2C%2215373.jpg%22%2C%2215374.jpg%22%2C%2215375.jpg%22%2C%2215376.jpg%22%2C%2215377.jpg%22%2C%2215378.jpg%22%2C%2215379.jpg%22%2C%2215380.jpg%22%2C%2215381.jpg%22%2C%2215382.jpg%22%2C%2215383.jpg%22%2C%2215384.jpg%22%2C%2215385.jpg%22%2C%2215386.jpg%22%2C%2215387.jpg%22%2C%2215388.jpg%22%2C%2215389.jpg%22%2C%2215390.jpg%22%2C%2215391.jpg%22%2C%2215392.jpg%22%2C%2215393.jpg%22%2C%2215394.jpg%22%2C%2215395.jpg%22%2C%2215396.jpg%22%2C%2215397.jpg%22%2C%2215398.jpg%22%2C%2215399.jpg%22%2C%2215400.jpg%22%2C%2215401.jpg%22%2C%2215402.jpg%22%2C%2215403.jpg%22%2C%2215404.jpg%22%2C%2215405.jpg%22%2C%2215406.jpg%22%2C%2215407.jpg%22%2C%2215408.jpg%22%2C%2215409.jpg%22%2C%2215410.jpg%22%2C%2215411.jpg%22%2C%2215413.jpg%22%2C%2215414.jpg%22%2C%2215415.jpg%22%2C%2215970.jpg%22%2C%2215971.jpg%22%2C%2215972.jpg%22%2C%2215973.jpg%22%2C%2215974.jpg%22%2C%2215975.jpg%22%2C%2215976.jpg%22%2C%2215977.jpg%22%2C%2215978.jpg%22%2C%2215979.jpg%22%2C%2215980.jpg%22%2C%2215981.jpg%22%2C%2215982.jpg%22%2C%2215983.jpg%22%2C%2216075.jpg%22%2C%2216076.jpg%22%2C%2216077.jpg%22%2C%2216078.jpg%22%2C%2216079.jpg%22%2C%2216080.jpg%22%2C%2216081.jpg%22%2C%2216082.jpg%22%2C%2216083.jpg%22%2C%2216193.jpg%22%2C%2216194.jpg%22%2C%2216195.jpg%22%2C%2216196.jpg%22%2C%2216197.jpg%22%2C%2216198.jpg%22%2C%2216199.jpg%22%2C%2216200.jpg%22%2C%2216201.jpg%22%2C%2216202.jpg%22%2C%2216203.jpg%22%2C%2216204.jpg%22%2C%2216205.jpg%22%2C%2216206.jpg%22%2C%2216732.jpg%22%2C%2216733.jpg%22%2C%2216734.jpg%22%2C%2216735.jpg%22%2C%2216736.jpg%22%2C%2216737.jpg%22%2C%2216738.jpg%22%2C%2216739.jpg%22%2C%2216740.jpg%22%2C%2216741.jpg%22%2C%2216742.jpg%22%2C%2216743.jpg%22%2C%2216744.jpg%22%2C%2216745.jpg%22%2C%2216746.jpg%22%2C%2216747.jpg%22%2C%2216748.jpg%22%2C%2216749.jpg%22%2C%2216750.jpg%22%2C%2216751.jpg%22%2C%2216752.jpg%22%2C%2216753.jpg%22%2C%2216754.jpg%22%2C%2216755.jpg%22%2C%2216756.jpg%22%2C%2216757.jpg%22%2C%2216758.jpg%22%2C%2216759.jpg%22%2C%2216760.jpg%22%2C%2216761.jpg%22%2C%2216762.jpg%22%2C%2216763.jpg%22%2C%2216764.jpg%22%2C%2216765.jpg%22%2C%2216766.jpg%22%2C%2216767.jpg%22%2C%2216768.jpg%22%2C%2216769.jpg%22%2C%2216770.jpg%22%2C%2216771.jpg%22%2C%2216772.jpg%22%2C%2216773.jpg%22%2C%2216774.jpg%22%2C%2216775.jpg%22%2C%2216776.jpg%22%2C%2216777.jpg%22%2C%2216778.jpg%22%2C%2216779.jpg%22%2C%2216780.jpg%22%2C%2216781.jpg%22%2C%2216782.jpg%22%2C%2216783.jpg%22%2C%2216784.jpg%22%2C%2216785.jpg%22%2C%2216786.jpg%22%2C%2216787.jpg%22%2C%2213976.jpg%22%2C%2213977.jpg%22%2C%2213978.jpg%22%2C%2213979.jpg%22%2C%2213980.jpg%22%2C%2213981.jpg%22%2C%2213982.jpg%22%2C%2213983.jpg%22%2C%2213984.jpg%22%2C%2213985.jpg%22%2C%2213986.jpg%22%2C%2213987.jpg%22%2C%2213988.jpg%22%2C%2213989.jpg%22%2C%2213990.jpg%22%2C%2213991.jpg%22%2C%2213992.jpg%22%2C%2213993.jpg%22%2C%2213994.jpg%22%2C%2213995.jpg%22%2C%2213996.jpg%22%2C%2213997.jpg%22%2C%2213998.jpg%22%2C%2213999.jpg%22%2C%2214000.jpg%22%2C%2214001.jpg%22%2C%2214002.jpg%22%2C%2214003.jpg%22%2C%2214004.jpg%22%2C%2214005.jpg%22%2C%2214006.jpg%22%2C%2214007.jpg%22%2C%2214008.jpg%22%2C%2214009.jpg%22%2C%2214010.jpg%22%2C%2214011.jpg%22%2C%2214012.jpg%22%2C%2214013.jpg%22%2C%2214014.jpg%22%2C%2214015.jpg%22%2C%2214016.jpg%22%2C%2214017.jpg%22%2C%2214018.jpg%22%2C%2214019.jpg%22%2C%2214020.jpg%22%2C%2214021.jpg%22%2C%2214022.jpg%22%2C%2214023.jpg%22%2C%2214024.jpg%22%2C%2214025.jpg%22%2C%2214026.jpg%22%2C%2214027.jpg%22%2C%2214028.jpg%22%2C%2214029.jpg%22%2C%2214030.jpg%22%2C%2214031.jpg%22%2C%2214032.jpg%22%2C%2214033.jpg%22%2C%2214034.jpg%22%2C%2214035.jpg%22%2C%2214036.jpg%22%2C%2214037.jpg%22%2C%2214038.jpg%22%2C%2214039.jpg%22%2C%2214040.jpg%22%2C%2214041.jpg%22%2C%2214042.jpg%22%2C%2214043.jpg%22%2C%2214044.jpg%22%2C%2214045.jpg%22%2C%2214046.jpg%22%2C%2214047.jpg%22%2C%2214048.jpg%22%2C%2214049.jpg%22%2C%2214050.jpg%22%2C%2214051.jpg%22%2C%2214052.jpg%22%2C%2214053.jpg%22%2C%2214054.jpg%22%2C%2214055.jpg%22%2C%2214056.jpg%22%2C%2214057.jpg%22%2C%2214058.jpg%22%2C%2214059.jpg%22%2C%2214060.jpg%22%5D",
            filename="val4.zip",
            md5=False,
            checksum="361b48cf43ef3f16bbe65242f003b789a3e898218eac3e4ba77c0d76e58eb8a7",
        ),
        DataLocation(
            url="https://cloud.cs.uni-tuebingen.de/public.php/dav/files/ZZxX65FGnQ8zjBP/Compressed%20Version/images/test/?accept=zip&files=%5B%223.jpg%22%2C%224.jpg%22%2C%2214.jpg%22%2C%2223.jpg%22%2C%2245.jpg%22%2C%2246.jpg%22%2C%2249.jpg%22%2C%2251.jpg%22%2C%2254.jpg%22%2C%2256.jpg%22%2C%2257.jpg%22%2C%2258.jpg%22%2C%2262.jpg%22%2C%2264.jpg%22%2C%2265.jpg%22%2C%2267.jpg%22%2C%2274.jpg%22%2C%2275.jpg%22%2C%2277.jpg%22%2C%2279.jpg%22%2C%2282.jpg%22%2C%2289.jpg%22%2C%2292.jpg%22%2C%2293.jpg%22%2C%2295.jpg%22%2C%2297.jpg%22%2C%2298.jpg%22%2C%22103.jpg%22%2C%22105.jpg%22%2C%22109.jpg%22%2C%22118.jpg%22%2C%22126.jpg%22%2C%22128.jpg%22%2C%22130.jpg%22%2C%22132.jpg%22%2C%22136.jpg%22%2C%22142.jpg%22%2C%22145.jpg%22%2C%22150.jpg%22%2C%22155.jpg%22%2C%22161.jpg%22%2C%22163.jpg%22%2C%22166.jpg%22%2C%22173.jpg%22%2C%22175.jpg%22%2C%22177.jpg%22%2C%22185.jpg%22%2C%22186.jpg%22%2C%22202.jpg%22%2C%22203.jpg%22%2C%22204.jpg%22%2C%22206.jpg%22%2C%22209.jpg%22%2C%22210.jpg%22%2C%22215.jpg%22%2C%22216.jpg%22%2C%22219.jpg%22%2C%22220.jpg%22%2C%22221.jpg%22%2C%22225.jpg%22%2C%22226.jpg%22%2C%22234.jpg%22%2C%22236.jpg%22%2C%22243.jpg%22%2C%22252.jpg%22%2C%22253.jpg%22%2C%22256.jpg%22%2C%22258.jpg%22%2C%22264.jpg%22%2C%22265.jpg%22%2C%22268.jpg%22%2C%22269.jpg%22%2C%22270.jpg%22%2C%22272.jpg%22%2C%22277.jpg%22%2C%22284.jpg%22%2C%22286.jpg%22%2C%22294.jpg%22%2C%22297.jpg%22%2C%22298.jpg%22%2C%22299.jpg%22%2C%22302.jpg%22%2C%22306.jpg%22%2C%22312.jpg%22%2C%22314.jpg%22%2C%22317.jpg%22%2C%22337.jpg%22%2C%22338.jpg%22%2C%22339.jpg%22%2C%22340.jpg%22%2C%22342.jpg%22%2C%22346.jpg%22%2C%22347.jpg%22%2C%22350.jpg%22%2C%22355.jpg%22%2C%22357.jpg%22%2C%22371.jpg%22%2C%22373.jpg%22%2C%22377.jpg%22%2C%22381.jpg%22%2C%22383.jpg%22%2C%22398.jpg%22%2C%22401.jpg%22%2C%22405.jpg%22%2C%22407.jpg%22%2C%22408.jpg%22%2C%22425.jpg%22%2C%22426.jpg%22%2C%22439.jpg%22%2C%22444.jpg%22%2C%22447.jpg%22%2C%22448.jpg%22%2C%22449.jpg%22%2C%22454.jpg%22%2C%22455.jpg%22%2C%22456.jpg%22%2C%22458.jpg%22%2C%22462.jpg%22%2C%22464.jpg%22%2C%22466.jpg%22%2C%22471.jpg%22%2C%22472.jpg%22%2C%22476.jpg%22%2C%22478.jpg%22%2C%22480.jpg%22%2C%22485.jpg%22%2C%22486.jpg%22%2C%22489.jpg%22%2C%22495.jpg%22%2C%22496.jpg%22%2C%22497.jpg%22%2C%22498.jpg%22%2C%22500.jpg%22%2C%22502.jpg%22%2C%22505.jpg%22%2C%22512.jpg%22%2C%22520.jpg%22%2C%22521.jpg%22%2C%22523.jpg%22%2C%22525.jpg%22%2C%22528.jpg%22%2C%22530.jpg%22%2C%22535.jpg%22%2C%22542.jpg%22%2C%22543.jpg%22%2C%22544.jpg%22%2C%22549.jpg%22%2C%22551.jpg%22%2C%22555.jpg%22%2C%22559.jpg%22%2C%22560.jpg%22%2C%22563.jpg%22%2C%22565.jpg%22%2C%22569.jpg%22%2C%22573.jpg%22%2C%22575.jpg%22%2C%22577.jpg%22%2C%22579.jpg%22%2C%22580.jpg%22%2C%22584.jpg%22%2C%22585.jpg%22%2C%22586.jpg%22%2C%22589.jpg%22%2C%22593.jpg%22%2C%22597.jpg%22%2C%22603.jpg%22%2C%22605.jpg%22%2C%22607.jpg%22%2C%22616.jpg%22%2C%22629.jpg%22%2C%22630.jpg%22%2C%22633.jpg%22%2C%22634.jpg%22%2C%22635.jpg%22%2C%22654.jpg%22%2C%22657.jpg%22%2C%22658.jpg%22%2C%22659.jpg%22%2C%22666.jpg%22%2C%22670.jpg%22%2C%22671.jpg%22%2C%22677.jpg%22%2C%22685.jpg%22%2C%22687.jpg%22%2C%22697.jpg%22%2C%22700.jpg%22%2C%22707.jpg%22%2C%22708.jpg%22%2C%22713.jpg%22%2C%22718.jpg%22%2C%22719.jpg%22%2C%22722.jpg%22%2C%22723.jpg%22%2C%22729.jpg%22%2C%22734.jpg%22%2C%22737.jpg%22%2C%22741.jpg%22%2C%22748.jpg%22%2C%22749.jpg%22%2C%22751.jpg%22%2C%22757.jpg%22%2C%22758.jpg%22%2C%22761.jpg%22%2C%22769.jpg%22%2C%22770.jpg%22%2C%22771.jpg%22%2C%22772.jpg%22%2C%22773.jpg%22%2C%22777.jpg%22%2C%22779.jpg%22%2C%22781.jpg%22%2C%22784.jpg%22%2C%22789.jpg%22%2C%22790.jpg%22%2C%22791.jpg%22%2C%22793.jpg%22%2C%22794.jpg%22%2C%22795.jpg%22%2C%22796.jpg%22%2C%22798.jpg%22%2C%22805.jpg%22%2C%22807.jpg%22%2C%22808.jpg%22%2C%22815.jpg%22%2C%22818.jpg%22%2C%22827.jpg%22%2C%22832.jpg%22%2C%22838.jpg%22%2C%22840.jpg%22%2C%22842.jpg%22%2C%22844.jpg%22%2C%22845.jpg%22%2C%22852.jpg%22%2C%22854.jpg%22%2C%22862.jpg%22%2C%22866.jpg%22%2C%22869.jpg%22%2C%22870.jpg%22%2C%22871.jpg%22%2C%22874.jpg%22%2C%22878.jpg%22%2C%22892.jpg%22%2C%22894.jpg%22%2C%22896.jpg%22%2C%22898.jpg%22%2C%22899.jpg%22%2C%22905.jpg%22%2C%22908.jpg%22%2C%22915.jpg%22%2C%22929.jpg%22%2C%22932.jpg%22%2C%22933.jpg%22%2C%22934.jpg%22%2C%22940.jpg%22%2C%22942.jpg%22%2C%22946.jpg%22%2C%22947.jpg%22%2C%22948.jpg%22%2C%22949.jpg%22%2C%22951.jpg%22%2C%22952.jpg%22%2C%22956.jpg%22%2C%22957.jpg%22%2C%22962.jpg%22%2C%22965.jpg%22%2C%22969.jpg%22%2C%22973.jpg%22%2C%22979.jpg%22%2C%22982.jpg%22%2C%22983.jpg%22%2C%22985.jpg%22%2C%22986.jpg%22%2C%22987.jpg%22%2C%22992.jpg%22%2C%221006.jpg%22%2C%221007.jpg%22%2C%221008.jpg%22%2C%221015.jpg%22%2C%221017.jpg%22%2C%221023.jpg%22%2C%221024.jpg%22%2C%221026.jpg%22%2C%221027.jpg%22%2C%221029.jpg%22%2C%221030.jpg%22%2C%221039.jpg%22%2C%221045.jpg%22%2C%221047.jpg%22%2C%221049.jpg%22%2C%221050.jpg%22%2C%221052.jpg%22%2C%221054.jpg%22%2C%221062.jpg%22%2C%221065.jpg%22%2C%221070.jpg%22%2C%221071.jpg%22%2C%221076.jpg%22%2C%221088.jpg%22%2C%221105.jpg%22%2C%221106.jpg%22%2C%221107.jpg%22%2C%221114.jpg%22%2C%221118.jpg%22%2C%221120.jpg%22%2C%221126.jpg%22%2C%221131.jpg%22%2C%221133.jpg%22%2C%221135.jpg%22%2C%221141.jpg%22%2C%221146.jpg%22%2C%221158.jpg%22%2C%221164.jpg%22%2C%221166.jpg%22%2C%221180.jpg%22%2C%221183.jpg%22%2C%221196.jpg%22%2C%221202.jpg%22%2C%221215.jpg%22%2C%221216.jpg%22%2C%221217.jpg%22%2C%221236.jpg%22%2C%221250.jpg%22%2C%221251.jpg%22%2C%221259.jpg%22%2C%221266.jpg%22%2C%221272.jpg%22%2C%221275.jpg%22%2C%221314.jpg%22%2C%221341.jpg%22%2C%221353.jpg%22%2C%221357.jpg%22%2C%221360.jpg%22%2C%221361.jpg%22%2C%221362.jpg%22%2C%221393.jpg%22%2C%221407.jpg%22%2C%221410.jpg%22%2C%221416.jpg%22%2C%221418.jpg%22%2C%221422.jpg%22%2C%221426.jpg%22%2C%221434.jpg%22%2C%221457.jpg%22%2C%221475.jpg%22%2C%221481.jpg%22%2C%221483.jpg%22%2C%221487.jpg%22%2C%221491.jpg%22%2C%221493.jpg%22%2C%221494.jpg%22%2C%221502.jpg%22%2C%221503.jpg%22%2C%221505.jpg%22%2C%221507.jpg%22%2C%221510.jpg%22%2C%221515.jpg%22%2C%221517.jpg%22%2C%221526.jpg%22%2C%221527.jpg%22%2C%221528.jpg%22%2C%221530.jpg%22%2C%221539.jpg%22%2C%221544.jpg%22%2C%221545.jpg%22%2C%221548.jpg%22%2C%221551.jpg%22%2C%221553.jpg%22%2C%221556.jpg%22%2C%221558.jpg%22%2C%221561.jpg%22%2C%221566.jpg%22%2C%221568.jpg%22%2C%221573.jpg%22%2C%221574.jpg%22%2C%221578.jpg%22%2C%221585.jpg%22%2C%221586.jpg%22%2C%221587.jpg%22%2C%221589.jpg%22%2C%221605.jpg%22%2C%221610.jpg%22%2C%221612.jpg%22%2C%221615.jpg%22%2C%221623.jpg%22%2C%221624.jpg%22%2C%221633.jpg%22%2C%221637.jpg%22%2C%221639.jpg%22%2C%221641.jpg%22%2C%221659.jpg%22%2C%221660.jpg%22%2C%221664.jpg%22%2C%221667.jpg%22%2C%221669.jpg%22%2C%221680.jpg%22%2C%221686.jpg%22%2C%221687.jpg%22%2C%221688.jpg%22%2C%221689.jpg%22%2C%221691.jpg%22%2C%221692.jpg%22%2C%221693.jpg%22%2C%221702.jpg%22%2C%221703.jpg%22%2C%221706.jpg%22%2C%221709.jpg%22%2C%221717.jpg%22%2C%221718.jpg%22%2C%221721.jpg%22%2C%221726.jpg%22%2C%221727.jpg%22%2C%221731.jpg%22%2C%221733.jpg%22%2C%221735.jpg%22%2C%221737.jpg%22%2C%221741.jpg%22%2C%221747.jpg%22%2C%221750.jpg%22%2C%221752.jpg%22%2C%221756.jpg%22%2C%221758.jpg%22%2C%221762.jpg%22%2C%221767.jpg%22%2C%221768.jpg%22%2C%221769.jpg%22%2C%221772.jpg%22%2C%221773.jpg%22%2C%221775.jpg%22%2C%221778.jpg%22%2C%221789.jpg%22%2C%221800.jpg%22%2C%221810.jpg%22%2C%221812.jpg%22%2C%221817.jpg%22%2C%221819.jpg%22%2C%221820.jpg%22%2C%221823.jpg%22%2C%221826.jpg%22%2C%221828.jpg%22%2C%221830.jpg%22%2C%221832.jpg%22%2C%221844.jpg%22%2C%221848.jpg%22%2C%221850.jpg%22%2C%221853.jpg%22%2C%221859.jpg%22%2C%221863.jpg%22%2C%221865.jpg%22%2C%221866.jpg%22%2C%221867.jpg%22%5D",
            filename="test1.zip",
            md5=False,
            checksum="0fe288fa46e245c61f4ffd84cb4e650eb9cd12ee6641edb67b95f92e67290ca9",
        ),
        DataLocation(
            url="https://cloud.cs.uni-tuebingen.de/public.php/dav/files/ZZxX65FGnQ8zjBP/Compressed%20Version/images/test/?accept=zip&files=%5B%221870.jpg%22%2C%221872.jpg%22%2C%221875.jpg%22%2C%221876.jpg%22%2C%221879.jpg%22%2C%221884.jpg%22%2C%221893.jpg%22%2C%221894.jpg%22%2C%221913.jpg%22%2C%221921.jpg%22%2C%221924.jpg%22%2C%221925.jpg%22%2C%221940.jpg%22%2C%221943.jpg%22%2C%221945.jpg%22%2C%221948.jpg%22%2C%221965.jpg%22%2C%221966.jpg%22%2C%221968.jpg%22%2C%221969.jpg%22%2C%221974.jpg%22%2C%221976.jpg%22%2C%221980.jpg%22%2C%221985.jpg%22%2C%221986.jpg%22%2C%221987.jpg%22%2C%221988.jpg%22%2C%221999.jpg%22%2C%222001.jpg%22%2C%222002.jpg%22%2C%222003.jpg%22%2C%222008.jpg%22%2C%222010.jpg%22%2C%222026.jpg%22%2C%222027.jpg%22%2C%222030.jpg%22%2C%222033.jpg%22%2C%222036.jpg%22%2C%222051.jpg%22%2C%222052.jpg%22%2C%222068.jpg%22%2C%222071.jpg%22%2C%222073.jpg%22%2C%222076.jpg%22%2C%222082.jpg%22%2C%222088.jpg%22%2C%222097.jpg%22%2C%222100.jpg%22%2C%222102.jpg%22%2C%222103.jpg%22%2C%222106.jpg%22%2C%222107.jpg%22%2C%222118.jpg%22%2C%222119.jpg%22%2C%222121.jpg%22%2C%222133.jpg%22%2C%222134.jpg%22%2C%222136.jpg%22%2C%222139.jpg%22%2C%222140.jpg%22%2C%222150.jpg%22%2C%222158.jpg%22%2C%222171.jpg%22%2C%222175.jpg%22%2C%222186.jpg%22%2C%222187.jpg%22%2C%222188.jpg%22%2C%222192.jpg%22%2C%222193.jpg%22%2C%222201.jpg%22%2C%222206.jpg%22%2C%222208.jpg%22%2C%222209.jpg%22%2C%222227.jpg%22%2C%222238.jpg%22%2C%222239.jpg%22%2C%222242.jpg%22%2C%222244.jpg%22%2C%222248.jpg%22%2C%222249.jpg%22%2C%222250.jpg%22%2C%222254.jpg%22%2C%222265.jpg%22%2C%222275.jpg%22%2C%222278.jpg%22%2C%222284.jpg%22%2C%222285.jpg%22%2C%222288.jpg%22%2C%222291.jpg%22%2C%222292.jpg%22%2C%222296.jpg%22%2C%222297.jpg%22%2C%222303.jpg%22%2C%222314.jpg%22%2C%222318.jpg%22%2C%222325.jpg%22%2C%222328.jpg%22%2C%222333.jpg%22%2C%222336.jpg%22%2C%222337.jpg%22%2C%222340.jpg%22%2C%222341.jpg%22%2C%222345.jpg%22%2C%222349.jpg%22%2C%222353.jpg%22%2C%222354.jpg%22%2C%222359.jpg%22%2C%222365.jpg%22%2C%222370.jpg%22%2C%222373.jpg%22%2C%222375.jpg%22%2C%222377.jpg%22%2C%222379.jpg%22%2C%222381.jpg%22%2C%222387.jpg%22%2C%222389.jpg%22%2C%222395.jpg%22%2C%222415.jpg%22%2C%222417.jpg%22%2C%222422.jpg%22%2C%222433.jpg%22%2C%222436.jpg%22%2C%222443.jpg%22%2C%222447.jpg%22%2C%222452.jpg%22%2C%222456.jpg%22%2C%222457.jpg%22%2C%222469.jpg%22%2C%222472.jpg%22%2C%222491.jpg%22%2C%222492.jpg%22%2C%222498.jpg%22%2C%222502.jpg%22%2C%222504.jpg%22%2C%222515.jpg%22%2C%222517.jpg%22%2C%222519.jpg%22%2C%222520.jpg%22%2C%222527.jpg%22%2C%222528.jpg%22%2C%222531.jpg%22%2C%222537.jpg%22%2C%222541.jpg%22%2C%222542.jpg%22%2C%222544.jpg%22%2C%222547.jpg%22%2C%222548.jpg%22%2C%222552.jpg%22%2C%222566.jpg%22%2C%222574.jpg%22%2C%222575.jpg%22%2C%222579.jpg%22%2C%222585.jpg%22%2C%222588.jpg%22%2C%222589.jpg%22%2C%222590.jpg%22%2C%222592.jpg%22%2C%222594.jpg%22%2C%222596.jpg%22%2C%222597.jpg%22%2C%222602.jpg%22%2C%222604.jpg%22%2C%222607.jpg%22%2C%222609.jpg%22%2C%222612.jpg%22%2C%222619.jpg%22%2C%222620.jpg%22%2C%222625.jpg%22%2C%222631.jpg%22%2C%222632.jpg%22%2C%222635.jpg%22%2C%222640.jpg%22%2C%222641.jpg%22%2C%222644.jpg%22%2C%222646.jpg%22%2C%222653.jpg%22%2C%222659.jpg%22%2C%222660.jpg%22%2C%222662.jpg%22%2C%222663.jpg%22%2C%222665.jpg%22%2C%222668.jpg%22%2C%222674.jpg%22%2C%222676.jpg%22%2C%222680.jpg%22%2C%222686.jpg%22%2C%222688.jpg%22%2C%222693.jpg%22%2C%222696.jpg%22%2C%222701.jpg%22%2C%222702.jpg%22%2C%222705.jpg%22%2C%222707.jpg%22%2C%222708.jpg%22%2C%222710.jpg%22%2C%222712.jpg%22%2C%222714.jpg%22%2C%222717.jpg%22%2C%222721.jpg%22%2C%222722.jpg%22%2C%222723.jpg%22%2C%222727.jpg%22%2C%222729.jpg%22%2C%222730.jpg%22%2C%222735.jpg%22%2C%222737.jpg%22%2C%222738.jpg%22%2C%222740.jpg%22%2C%222743.jpg%22%2C%222751.jpg%22%2C%222753.jpg%22%2C%222760.jpg%22%2C%222763.jpg%22%2C%222764.jpg%22%2C%222767.jpg%22%2C%222768.jpg%22%2C%222770.jpg%22%2C%222772.jpg%22%2C%222775.jpg%22%2C%222783.jpg%22%2C%222784.jpg%22%2C%222790.jpg%22%2C%222799.jpg%22%2C%222800.jpg%22%2C%222803.jpg%22%2C%222805.jpg%22%2C%222806.jpg%22%2C%222808.jpg%22%2C%222812.jpg%22%2C%222817.jpg%22%2C%222818.jpg%22%2C%222819.jpg%22%2C%222820.jpg%22%2C%222821.jpg%22%2C%222823.jpg%22%2C%222824.jpg%22%2C%222825.jpg%22%2C%222832.jpg%22%2C%222833.jpg%22%2C%222837.jpg%22%2C%222839.jpg%22%2C%222843.jpg%22%2C%222848.jpg%22%2C%222850.jpg%22%2C%222852.jpg%22%2C%222856.jpg%22%2C%222861.jpg%22%2C%222863.jpg%22%2C%222867.jpg%22%2C%222868.jpg%22%2C%222873.jpg%22%2C%222877.jpg%22%2C%222878.jpg%22%2C%222881.jpg%22%2C%222882.jpg%22%2C%222886.jpg%22%2C%222888.jpg%22%2C%222891.jpg%22%2C%222896.jpg%22%2C%222897.jpg%22%2C%222902.jpg%22%2C%222904.jpg%22%2C%222912.jpg%22%2C%222914.jpg%22%2C%222917.jpg%22%2C%222918.jpg%22%2C%222919.jpg%22%2C%222920.jpg%22%2C%222921.jpg%22%2C%222925.jpg%22%2C%222926.jpg%22%2C%222932.jpg%22%2C%222936.jpg%22%2C%222937.jpg%22%2C%222938.jpg%22%2C%222939.jpg%22%2C%222943.jpg%22%2C%222946.jpg%22%2C%222947.jpg%22%2C%222949.jpg%22%2C%222951.jpg%22%2C%222954.jpg%22%2C%222955.jpg%22%2C%222956.jpg%22%2C%222959.jpg%22%2C%222960.jpg%22%2C%222964.jpg%22%2C%222966.jpg%22%2C%222968.jpg%22%2C%222973.jpg%22%2C%222974.jpg%22%2C%222979.jpg%22%2C%222982.jpg%22%2C%222984.jpg%22%2C%222988.jpg%22%2C%222992.jpg%22%2C%222993.jpg%22%2C%222994.jpg%22%2C%222995.jpg%22%2C%222996.jpg%22%2C%222999.jpg%22%2C%223005.jpg%22%2C%223014.jpg%22%2C%223018.jpg%22%2C%223019.jpg%22%2C%223021.jpg%22%2C%223031.jpg%22%2C%223034.jpg%22%2C%223036.jpg%22%2C%223037.jpg%22%2C%223045.jpg%22%2C%223048.jpg%22%2C%223054.jpg%22%2C%223055.jpg%22%2C%223057.jpg%22%2C%223061.jpg%22%2C%223064.jpg%22%2C%223077.jpg%22%2C%223079.jpg%22%2C%223083.jpg%22%2C%223088.jpg%22%2C%223089.jpg%22%2C%223090.jpg%22%2C%223093.jpg%22%2C%223097.jpg%22%2C%223099.jpg%22%2C%223106.jpg%22%2C%223115.jpg%22%2C%223118.jpg%22%2C%223119.jpg%22%2C%223123.jpg%22%2C%223124.jpg%22%2C%223131.jpg%22%2C%223132.jpg%22%2C%223135.jpg%22%2C%223136.jpg%22%2C%223137.jpg%22%2C%223141.jpg%22%2C%223145.jpg%22%2C%223150.jpg%22%2C%223153.jpg%22%2C%223154.jpg%22%2C%223155.jpg%22%2C%223158.jpg%22%2C%223159.jpg%22%2C%223161.jpg%22%2C%223163.jpg%22%2C%223168.jpg%22%2C%223169.jpg%22%2C%223172.jpg%22%2C%223174.jpg%22%2C%223180.jpg%22%2C%223195.jpg%22%2C%223198.jpg%22%2C%223201.jpg%22%2C%223207.jpg%22%2C%223209.jpg%22%2C%223210.jpg%22%2C%223215.jpg%22%2C%223216.jpg%22%2C%223222.jpg%22%2C%223228.jpg%22%2C%223232.jpg%22%2C%223237.jpg%22%2C%223238.jpg%22%2C%223243.jpg%22%2C%223253.jpg%22%2C%223254.jpg%22%2C%223259.jpg%22%2C%223266.jpg%22%2C%223269.jpg%22%2C%223271.jpg%22%2C%223273.jpg%22%2C%223277.jpg%22%2C%223279.jpg%22%2C%223284.jpg%22%2C%223293.jpg%22%2C%223299.jpg%22%2C%223300.jpg%22%2C%223305.jpg%22%2C%223306.jpg%22%2C%223307.jpg%22%2C%223310.jpg%22%2C%223316.jpg%22%2C%223322.jpg%22%2C%223337.jpg%22%2C%223348.jpg%22%2C%223350.jpg%22%2C%223351.jpg%22%2C%223355.jpg%22%2C%223359.jpg%22%2C%223360.jpg%22%2C%223365.jpg%22%2C%223369.jpg%22%2C%223373.jpg%22%2C%223375.jpg%22%2C%223377.jpg%22%2C%223384.jpg%22%2C%224055.jpg%22%2C%224056.jpg%22%2C%224057.jpg%22%2C%224058.jpg%22%2C%224059.jpg%22%2C%224060.jpg%22%2C%224061.jpg%22%2C%224062.jpg%22%2C%224063.jpg%22%2C%224064.jpg%22%2C%224065.jpg%22%2C%224066.jpg%22%2C%224067.jpg%22%2C%224068.jpg%22%2C%224069.jpg%22%2C%224070.jpg%22%2C%224071.jpg%22%2C%224072.jpg%22%2C%224073.jpg%22%2C%224074.jpg%22%2C%224075.jpg%22%2C%224076.jpg%22%2C%224077.jpg%22%2C%224078.jpg%22%2C%224079.jpg%22%2C%224080.jpg%22%2C%224081.jpg%22%2C%224082.jpg%22%2C%224083.jpg%22%2C%224084.jpg%22%2C%224085.jpg%22%2C%224086.jpg%22%2C%224087.jpg%22%2C%224088.jpg%22%2C%224089.jpg%22%2C%224090.jpg%22%2C%224091.jpg%22%2C%224092.jpg%22%2C%224093.jpg%22%2C%224094.jpg%22%2C%224095.jpg%22%2C%224096.jpg%22%2C%224097.jpg%22%2C%224098.jpg%22%2C%224099.jpg%22%2C%224100.jpg%22%2C%224101.jpg%22%2C%224102.jpg%22%2C%224103.jpg%22%2C%224104.jpg%22%2C%224105.jpg%22%2C%224106.jpg%22%5D",
            filename="test2.zip",
            md5=False,
            checksum="e2f643fc55a19ec09f83758ce8364cf61e29a6b7c2b179d05e8c1813fcc6f353",
        ),
        DataLocation(
            url="https://cloud.cs.uni-tuebingen.de/public.php/dav/files/ZZxX65FGnQ8zjBP/Compressed%20Version/images/test/?accept=zip&files=%5B%224107.jpg%22%2C%224108.jpg%22%2C%224109.jpg%22%2C%224110.jpg%22%2C%224111.jpg%22%2C%224112.jpg%22%2C%224113.jpg%22%2C%224114.jpg%22%2C%224115.jpg%22%2C%224116.jpg%22%2C%224117.jpg%22%2C%224118.jpg%22%2C%224119.jpg%22%2C%224120.jpg%22%2C%224121.jpg%22%2C%224122.jpg%22%2C%224123.jpg%22%2C%224124.jpg%22%2C%224125.jpg%22%2C%224126.jpg%22%2C%224127.jpg%22%2C%224128.jpg%22%2C%224129.jpg%22%2C%224130.jpg%22%2C%224131.jpg%22%2C%224132.jpg%22%2C%224133.jpg%22%2C%224134.jpg%22%2C%224135.jpg%22%2C%224136.jpg%22%2C%224137.jpg%22%2C%224138.jpg%22%2C%224139.jpg%22%2C%224140.jpg%22%2C%224141.jpg%22%2C%224142.jpg%22%2C%224143.jpg%22%2C%224144.jpg%22%2C%224145.jpg%22%2C%224146.jpg%22%2C%224147.jpg%22%2C%224148.jpg%22%2C%224149.jpg%22%2C%224150.jpg%22%2C%224151.jpg%22%2C%224152.jpg%22%2C%224153.jpg%22%2C%224154.jpg%22%2C%224155.jpg%22%2C%224156.jpg%22%2C%224157.jpg%22%2C%224158.jpg%22%2C%224159.jpg%22%2C%224160.jpg%22%2C%224161.jpg%22%2C%224162.jpg%22%2C%224163.jpg%22%2C%224164.jpg%22%2C%224165.jpg%22%2C%224166.jpg%22%2C%224167.jpg%22%2C%224168.jpg%22%2C%224169.jpg%22%2C%224170.jpg%22%2C%224171.jpg%22%2C%224172.jpg%22%2C%224173.jpg%22%2C%224174.jpg%22%2C%224175.jpg%22%2C%224176.jpg%22%2C%224177.jpg%22%2C%224178.jpg%22%2C%224179.jpg%22%2C%224180.jpg%22%2C%224181.jpg%22%2C%224182.jpg%22%2C%224331.jpg%22%2C%224332.jpg%22%2C%224333.jpg%22%2C%224334.jpg%22%2C%224335.jpg%22%2C%224336.jpg%22%2C%224337.jpg%22%2C%224338.jpg%22%2C%224339.jpg%22%2C%224340.jpg%22%2C%224341.jpg%22%2C%224342.jpg%22%2C%224343.jpg%22%2C%224344.jpg%22%2C%224345.jpg%22%2C%224346.jpg%22%2C%224347.jpg%22%2C%224348.jpg%22%2C%224349.jpg%22%2C%224350.jpg%22%2C%224351.jpg%22%2C%224352.jpg%22%2C%224353.jpg%22%2C%224354.jpg%22%2C%224355.jpg%22%2C%224356.jpg%22%2C%224357.jpg%22%2C%224358.jpg%22%2C%224359.jpg%22%2C%224360.jpg%22%2C%224361.jpg%22%2C%224362.jpg%22%2C%224363.jpg%22%2C%224364.jpg%22%2C%224365.jpg%22%2C%224366.jpg%22%2C%224367.jpg%22%2C%224368.jpg%22%2C%224369.jpg%22%2C%224370.jpg%22%2C%224371.jpg%22%2C%224372.jpg%22%2C%224373.jpg%22%2C%224374.jpg%22%2C%224375.jpg%22%2C%224376.jpg%22%2C%224377.jpg%22%2C%224378.jpg%22%2C%224379.jpg%22%2C%224380.jpg%22%2C%224381.jpg%22%2C%224382.jpg%22%2C%224383.jpg%22%2C%224384.jpg%22%2C%224385.jpg%22%2C%224386.jpg%22%2C%224387.jpg%22%2C%224388.jpg%22%2C%224389.jpg%22%2C%224390.jpg%22%2C%224391.jpg%22%2C%224392.jpg%22%2C%224393.jpg%22%2C%224394.jpg%22%2C%224395.jpg%22%2C%224925.jpg%22%2C%224926.jpg%22%2C%224927.jpg%22%2C%224928.jpg%22%2C%224929.jpg%22%2C%224930.jpg%22%2C%224931.jpg%22%2C%224932.jpg%22%2C%224933.jpg%22%2C%224934.jpg%22%2C%224935.jpg%22%2C%224936.jpg%22%2C%224937.jpg%22%2C%224938.jpg%22%2C%224939.jpg%22%2C%224940.jpg%22%2C%224941.jpg%22%2C%224942.jpg%22%2C%224943.jpg%22%2C%224944.jpg%22%2C%224945.jpg%22%2C%224946.jpg%22%2C%224947.jpg%22%2C%224948.jpg%22%2C%224949.jpg%22%2C%224950.jpg%22%2C%224951.jpg%22%2C%224952.jpg%22%2C%224953.jpg%22%2C%224954.jpg%22%2C%224955.jpg%22%2C%224956.jpg%22%2C%224957.jpg%22%2C%224958.jpg%22%2C%224959.jpg%22%2C%224960.jpg%22%2C%224961.jpg%22%2C%224962.jpg%22%2C%224963.jpg%22%2C%224964.jpg%22%2C%224965.jpg%22%2C%224966.jpg%22%2C%224967.jpg%22%2C%224968.jpg%22%2C%224969.jpg%22%2C%224970.jpg%22%2C%224971.jpg%22%2C%224972.jpg%22%2C%224973.jpg%22%2C%224974.jpg%22%2C%224975.jpg%22%2C%224976.jpg%22%2C%224977.jpg%22%2C%224978.jpg%22%2C%224979.jpg%22%2C%224980.jpg%22%2C%224981.jpg%22%2C%224982.jpg%22%2C%224983.jpg%22%2C%224984.jpg%22%2C%224985.jpg%22%2C%224986.jpg%22%2C%224987.jpg%22%2C%224988.jpg%22%2C%224989.jpg%22%2C%224990.jpg%22%2C%224991.jpg%22%2C%224992.jpg%22%2C%224993.jpg%22%2C%224994.jpg%22%2C%224995.jpg%22%2C%224996.jpg%22%2C%224997.jpg%22%2C%224998.jpg%22%2C%224999.jpg%22%2C%225000.jpg%22%2C%225001.jpg%22%2C%225002.jpg%22%2C%225003.jpg%22%2C%225004.jpg%22%2C%225005.jpg%22%2C%225006.jpg%22%2C%225007.jpg%22%2C%225008.jpg%22%2C%225009.jpg%22%2C%225010.jpg%22%2C%225011.jpg%22%2C%225012.jpg%22%2C%225013.jpg%22%2C%225014.jpg%22%2C%225015.jpg%22%2C%225016.jpg%22%2C%225017.jpg%22%2C%225018.jpg%22%2C%225019.jpg%22%2C%225020.jpg%22%2C%225021.jpg%22%2C%225022.jpg%22%2C%225023.jpg%22%2C%225024.jpg%22%2C%225025.jpg%22%2C%225026.jpg%22%2C%225027.jpg%22%2C%225028.jpg%22%2C%225029.jpg%22%2C%225030.jpg%22%2C%225031.jpg%22%2C%225032.jpg%22%2C%225033.jpg%22%2C%225034.jpg%22%2C%225035.jpg%22%2C%225036.jpg%22%2C%225037.jpg%22%2C%225038.jpg%22%2C%225039.jpg%22%2C%225040.jpg%22%2C%225041.jpg%22%2C%225042.jpg%22%2C%225043.jpg%22%2C%225044.jpg%22%2C%225045.jpg%22%2C%225046.jpg%22%2C%225047.jpg%22%2C%225048.jpg%22%2C%225049.jpg%22%2C%225050.jpg%22%2C%225051.jpg%22%2C%225052.jpg%22%2C%225053.jpg%22%2C%225054.jpg%22%2C%225055.jpg%22%2C%225056.jpg%22%2C%225057.jpg%22%2C%225058.jpg%22%2C%225059.jpg%22%2C%225060.jpg%22%2C%225061.jpg%22%2C%225062.jpg%22%2C%225063.jpg%22%2C%225064.jpg%22%2C%225065.jpg%22%2C%225066.jpg%22%2C%225067.jpg%22%2C%225068.jpg%22%2C%225069.jpg%22%2C%225070.jpg%22%2C%225071.jpg%22%2C%225072.jpg%22%2C%225073.jpg%22%2C%225074.jpg%22%2C%225075.jpg%22%2C%225076.jpg%22%2C%225077.jpg%22%2C%225078.jpg%22%2C%225079.jpg%22%2C%225080.jpg%22%2C%225081.jpg%22%2C%225082.jpg%22%2C%225083.jpg%22%2C%225084.jpg%22%2C%225085.jpg%22%2C%225086.jpg%22%2C%225087.jpg%22%2C%225088.jpg%22%2C%225089.jpg%22%2C%225090.jpg%22%2C%225091.jpg%22%2C%225092.jpg%22%2C%225093.jpg%22%2C%225094.jpg%22%2C%225095.jpg%22%2C%225096.jpg%22%2C%225097.jpg%22%2C%225098.jpg%22%2C%225099.jpg%22%2C%225100.jpg%22%2C%225101.jpg%22%2C%225102.jpg%22%2C%225103.jpg%22%2C%225104.jpg%22%2C%225105.jpg%22%2C%226099.jpg%22%2C%226100.jpg%22%2C%226101.jpg%22%2C%226102.jpg%22%2C%226103.jpg%22%2C%226104.jpg%22%2C%226105.jpg%22%2C%226106.jpg%22%2C%226107.jpg%22%2C%226108.jpg%22%2C%226109.jpg%22%2C%226110.jpg%22%2C%226111.jpg%22%2C%226112.jpg%22%2C%226113.jpg%22%2C%226114.jpg%22%2C%226115.jpg%22%2C%226116.jpg%22%2C%226117.jpg%22%2C%226118.jpg%22%2C%226119.jpg%22%2C%226120.jpg%22%2C%226121.jpg%22%2C%226122.jpg%22%2C%226123.jpg%22%2C%226124.jpg%22%2C%226125.jpg%22%2C%226126.jpg%22%2C%226127.jpg%22%2C%226128.jpg%22%2C%226129.jpg%22%2C%226130.jpg%22%2C%226131.jpg%22%2C%226132.jpg%22%2C%226133.jpg%22%2C%226134.jpg%22%2C%226135.jpg%22%2C%226136.jpg%22%2C%226137.jpg%22%2C%226138.jpg%22%2C%226139.jpg%22%2C%226140.jpg%22%2C%226141.jpg%22%2C%226142.jpg%22%2C%226143.jpg%22%2C%226144.jpg%22%2C%226145.jpg%22%2C%226146.jpg%22%2C%226147.jpg%22%2C%226148.jpg%22%2C%226149.jpg%22%2C%226150.jpg%22%2C%226151.jpg%22%2C%226152.jpg%22%2C%226153.jpg%22%2C%226154.jpg%22%2C%226155.jpg%22%2C%226156.jpg%22%2C%226157.jpg%22%2C%226158.jpg%22%2C%226159.jpg%22%2C%226160.jpg%22%2C%226161.jpg%22%2C%226162.jpg%22%2C%226163.jpg%22%2C%226164.jpg%22%2C%226165.jpg%22%2C%226166.jpg%22%2C%226167.jpg%22%2C%226168.jpg%22%2C%226169.jpg%22%2C%226170.jpg%22%2C%226171.jpg%22%2C%226172.jpg%22%2C%226173.jpg%22%2C%226174.jpg%22%2C%226175.jpg%22%2C%226176.jpg%22%2C%226177.jpg%22%2C%226178.jpg%22%2C%226179.jpg%22%2C%226180.jpg%22%2C%226181.jpg%22%2C%226182.jpg%22%2C%226183.jpg%22%2C%226184.jpg%22%2C%226185.jpg%22%2C%226186.jpg%22%2C%226187.jpg%22%2C%226188.jpg%22%2C%226189.jpg%22%2C%226190.jpg%22%2C%226191.jpg%22%2C%226192.jpg%22%2C%226193.jpg%22%2C%226194.jpg%22%2C%226195.jpg%22%2C%226196.jpg%22%2C%226197.jpg%22%2C%226198.jpg%22%2C%226199.jpg%22%2C%226200.jpg%22%2C%226201.jpg%22%2C%226202.jpg%22%2C%226203.jpg%22%2C%226204.jpg%22%2C%226205.jpg%22%2C%226206.jpg%22%2C%226207.jpg%22%2C%226208.jpg%22%2C%226209.jpg%22%2C%226210.jpg%22%2C%226211.jpg%22%2C%226212.jpg%22%2C%226213.jpg%22%2C%226214.jpg%22%2C%226215.jpg%22%2C%226216.jpg%22%2C%226217.jpg%22%2C%226218.jpg%22%2C%226219.jpg%22%2C%226220.jpg%22%2C%226221.jpg%22%2C%226222.jpg%22%2C%226223.jpg%22%2C%226224.jpg%22%2C%226225.jpg%22%2C%226226.jpg%22%5D",
            filename="test3.zip",
            md5=False,
            checksum="f0e601c78f2eabfabe53298489b3353f878eb00b5f30333576068f0497cc50c0",
        ),
        DataLocation(
            url="https://cloud.cs.uni-tuebingen.de/public.php/dav/files/ZZxX65FGnQ8zjBP/Compressed%20Version/images/test/?accept=zip&files=%5B%226227.jpg%22%2C%226228.jpg%22%2C%226229.jpg%22%2C%226230.jpg%22%2C%226231.jpg%22%2C%226232.jpg%22%2C%226233.jpg%22%2C%226234.jpg%22%2C%226235.jpg%22%2C%226236.jpg%22%2C%226237.jpg%22%2C%226238.jpg%22%2C%226239.jpg%22%2C%226240.jpg%22%2C%226241.jpg%22%2C%226242.jpg%22%2C%226243.jpg%22%2C%226244.jpg%22%2C%226245.jpg%22%2C%226246.jpg%22%2C%226247.jpg%22%2C%226248.jpg%22%2C%226249.jpg%22%2C%226250.jpg%22%2C%226251.jpg%22%2C%226252.jpg%22%2C%226253.jpg%22%2C%226254.jpg%22%2C%226255.jpg%22%2C%226256.jpg%22%2C%226257.jpg%22%2C%226258.jpg%22%2C%226259.jpg%22%2C%226260.jpg%22%2C%226261.jpg%22%2C%226262.jpg%22%2C%226263.jpg%22%2C%226264.jpg%22%2C%226265.jpg%22%2C%226266.jpg%22%2C%226267.jpg%22%2C%226268.jpg%22%2C%226269.jpg%22%2C%226270.jpg%22%2C%226271.jpg%22%2C%226272.jpg%22%2C%226273.jpg%22%2C%226274.jpg%22%2C%226275.jpg%22%2C%226276.jpg%22%2C%226277.jpg%22%2C%226278.jpg%22%2C%226279.jpg%22%2C%226701.jpg%22%2C%226702.jpg%22%2C%226703.jpg%22%2C%226704.jpg%22%2C%226705.jpg%22%2C%226706.jpg%22%2C%226707.jpg%22%2C%226708.jpg%22%2C%226709.jpg%22%2C%226710.jpg%22%2C%226711.jpg%22%2C%226712.jpg%22%2C%226713.jpg%22%2C%226714.jpg%22%2C%226715.jpg%22%2C%226716.jpg%22%2C%226717.jpg%22%2C%226718.jpg%22%2C%226719.jpg%22%2C%226720.jpg%22%2C%226721.jpg%22%2C%226722.jpg%22%2C%226723.jpg%22%2C%226724.jpg%22%2C%226725.jpg%22%2C%226726.jpg%22%2C%226727.jpg%22%2C%226728.jpg%22%2C%226729.jpg%22%2C%226730.jpg%22%2C%226731.jpg%22%2C%226732.jpg%22%2C%226733.jpg%22%2C%226734.jpg%22%2C%226735.jpg%22%2C%226736.jpg%22%2C%226737.jpg%22%2C%226738.jpg%22%2C%226739.jpg%22%2C%226740.jpg%22%2C%226741.jpg%22%2C%226742.jpg%22%2C%226743.jpg%22%2C%226744.jpg%22%2C%226745.jpg%22%2C%226746.jpg%22%2C%226747.jpg%22%2C%226748.jpg%22%2C%226749.jpg%22%2C%226750.jpg%22%2C%226751.jpg%22%2C%226752.jpg%22%2C%226753.jpg%22%2C%226754.jpg%22%2C%226755.jpg%22%2C%226756.jpg%22%2C%226757.jpg%22%2C%226758.jpg%22%2C%226759.jpg%22%2C%226760.jpg%22%2C%226761.jpg%22%2C%226762.jpg%22%2C%226763.jpg%22%2C%226764.jpg%22%2C%226765.jpg%22%2C%226766.jpg%22%2C%226767.jpg%22%2C%226768.jpg%22%2C%226769.jpg%22%2C%226770.jpg%22%2C%226771.jpg%22%2C%226772.jpg%22%2C%226773.jpg%22%2C%226774.jpg%22%2C%226775.jpg%22%2C%226776.jpg%22%2C%226777.jpg%22%2C%226778.jpg%22%2C%226779.jpg%22%2C%226780.jpg%22%2C%226781.jpg%22%2C%226782.jpg%22%2C%226783.jpg%22%2C%226784.jpg%22%2C%226785.jpg%22%2C%226786.jpg%22%2C%226787.jpg%22%2C%226788.jpg%22%2C%226789.jpg%22%2C%226790.jpg%22%2C%226791.jpg%22%2C%226792.jpg%22%2C%226793.jpg%22%2C%226794.jpg%22%2C%226795.jpg%22%2C%226796.jpg%22%2C%226797.jpg%22%2C%226798.jpg%22%2C%226799.jpg%22%2C%226800.jpg%22%2C%226801.jpg%22%2C%226802.jpg%22%2C%226803.jpg%22%2C%226804.jpg%22%2C%226805.jpg%22%2C%226806.jpg%22%2C%226807.jpg%22%2C%226808.jpg%22%2C%226809.jpg%22%2C%226810.jpg%22%2C%226811.jpg%22%2C%226812.jpg%22%2C%226813.jpg%22%2C%226814.jpg%22%2C%226815.jpg%22%2C%226816.jpg%22%2C%226817.jpg%22%2C%226818.jpg%22%2C%226819.jpg%22%2C%226820.jpg%22%2C%226821.jpg%22%2C%226822.jpg%22%2C%226823.jpg%22%2C%226824.jpg%22%2C%226825.jpg%22%2C%226826.jpg%22%2C%226827.jpg%22%2C%226828.jpg%22%2C%226829.jpg%22%2C%226830.jpg%22%2C%226831.jpg%22%2C%226832.jpg%22%2C%226833.jpg%22%2C%226834.jpg%22%2C%226835.jpg%22%2C%226836.jpg%22%2C%226837.jpg%22%2C%226838.jpg%22%2C%226839.jpg%22%2C%226840.jpg%22%2C%226841.jpg%22%2C%226842.jpg%22%2C%226843.jpg%22%2C%226844.jpg%22%2C%226845.jpg%22%2C%226846.jpg%22%2C%226847.jpg%22%2C%226848.jpg%22%2C%226849.jpg%22%2C%226850.jpg%22%2C%226851.jpg%22%2C%226852.jpg%22%2C%226853.jpg%22%2C%226854.jpg%22%2C%226855.jpg%22%2C%226856.jpg%22%2C%226857.jpg%22%2C%226858.jpg%22%2C%226859.jpg%22%2C%226860.jpg%22%2C%226861.jpg%22%2C%226862.jpg%22%2C%226863.jpg%22%2C%226864.jpg%22%2C%226865.jpg%22%2C%226866.jpg%22%2C%226867.jpg%22%2C%226868.jpg%22%2C%226869.jpg%22%2C%226870.jpg%22%2C%226871.jpg%22%2C%226872.jpg%22%2C%226873.jpg%22%2C%226874.jpg%22%2C%226875.jpg%22%2C%226876.jpg%22%2C%226877.jpg%22%2C%226878.jpg%22%2C%226879.jpg%22%2C%226880.jpg%22%2C%226881.jpg%22%2C%227221.jpg%22%2C%227222.jpg%22%2C%227223.jpg%22%2C%227224.jpg%22%2C%227225.jpg%22%2C%227226.jpg%22%2C%227227.jpg%22%2C%227228.jpg%22%2C%227229.jpg%22%2C%227230.jpg%22%2C%227231.jpg%22%2C%227232.jpg%22%2C%227233.jpg%22%2C%227234.jpg%22%2C%227235.jpg%22%2C%227352.jpg%22%2C%227353.jpg%22%2C%227354.jpg%22%2C%227355.jpg%22%2C%227356.jpg%22%2C%227357.jpg%22%2C%227358.jpg%22%2C%227359.jpg%22%2C%227360.jpg%22%2C%227361.jpg%22%2C%227362.jpg%22%2C%227363.jpg%22%2C%227364.jpg%22%2C%227365.jpg%22%2C%227807.jpg%22%2C%227808.jpg%22%2C%227809.jpg%22%2C%227810.jpg%22%2C%227811.jpg%22%2C%227812.jpg%22%2C%227813.jpg%22%2C%227814.jpg%22%2C%227815.jpg%22%2C%227816.jpg%22%2C%227817.jpg%22%2C%227818.jpg%22%2C%227819.jpg%22%2C%227820.jpg%22%2C%227821.jpg%22%2C%227822.jpg%22%2C%227823.jpg%22%2C%227824.jpg%22%2C%227825.jpg%22%2C%227826.jpg%22%2C%227827.jpg%22%2C%227828.jpg%22%2C%227829.jpg%22%2C%227830.jpg%22%2C%227831.jpg%22%2C%227832.jpg%22%2C%227833.jpg%22%2C%227834.jpg%22%2C%227835.jpg%22%2C%227836.jpg%22%2C%227837.jpg%22%2C%227838.jpg%22%2C%227839.jpg%22%2C%227840.jpg%22%2C%227841.jpg%22%2C%227842.jpg%22%2C%227843.jpg%22%2C%227844.jpg%22%2C%227845.jpg%22%2C%227846.jpg%22%2C%227847.jpg%22%2C%227848.jpg%22%2C%227849.jpg%22%2C%227850.jpg%22%2C%227851.jpg%22%2C%227852.jpg%22%2C%227853.jpg%22%2C%227854.jpg%22%2C%227855.jpg%22%2C%227856.jpg%22%2C%227857.jpg%22%2C%227858.jpg%22%2C%227859.jpg%22%2C%227860.jpg%22%2C%227861.jpg%22%2C%227862.jpg%22%2C%227863.jpg%22%2C%227864.jpg%22%2C%227865.jpg%22%2C%227866.jpg%22%2C%227867.jpg%22%2C%227868.jpg%22%2C%227869.jpg%22%2C%227870.jpg%22%2C%227871.jpg%22%2C%227872.jpg%22%2C%227873.jpg%22%2C%227874.jpg%22%2C%227875.jpg%22%2C%227876.jpg%22%2C%227877.jpg%22%2C%227878.jpg%22%2C%227879.jpg%22%2C%227880.jpg%22%2C%227881.jpg%22%2C%227882.jpg%22%2C%227883.jpg%22%2C%227884.jpg%22%2C%227885.jpg%22%2C%227886.jpg%22%2C%227887.jpg%22%2C%227888.jpg%22%2C%227889.jpg%22%2C%227890.jpg%22%2C%227891.jpg%22%2C%227892.jpg%22%2C%227893.jpg%22%2C%227894.jpg%22%2C%227895.jpg%22%2C%227896.jpg%22%2C%227897.jpg%22%2C%227898.jpg%22%2C%227899.jpg%22%2C%227900.jpg%22%2C%227901.jpg%22%2C%227902.jpg%22%2C%227903.jpg%22%2C%227904.jpg%22%2C%227905.jpg%22%2C%227906.jpg%22%2C%227907.jpg%22%2C%227908.jpg%22%2C%227909.jpg%22%2C%227910.jpg%22%2C%227911.jpg%22%2C%227912.jpg%22%2C%227913.jpg%22%2C%227914.jpg%22%2C%227915.jpg%22%2C%227916.jpg%22%2C%227917.jpg%22%2C%227918.jpg%22%2C%227919.jpg%22%2C%227920.jpg%22%2C%227921.jpg%22%2C%227922.jpg%22%2C%227923.jpg%22%2C%227924.jpg%22%2C%227925.jpg%22%2C%227926.jpg%22%2C%227927.jpg%22%2C%227928.jpg%22%2C%227929.jpg%22%2C%227930.jpg%22%2C%227931.jpg%22%2C%227932.jpg%22%2C%227933.jpg%22%2C%227934.jpg%22%2C%227935.jpg%22%2C%227936.jpg%22%2C%227937.jpg%22%2C%227938.jpg%22%2C%227939.jpg%22%2C%227940.jpg%22%2C%227941.jpg%22%2C%227942.jpg%22%2C%227943.jpg%22%2C%227944.jpg%22%2C%227945.jpg%22%2C%227946.jpg%22%2C%227947.jpg%22%2C%227948.jpg%22%2C%227949.jpg%22%2C%227950.jpg%22%2C%227951.jpg%22%2C%227952.jpg%22%2C%227953.jpg%22%2C%227954.jpg%22%2C%227955.jpg%22%2C%227956.jpg%22%2C%227957.jpg%22%2C%227958.jpg%22%2C%227959.jpg%22%2C%227960.jpg%22%2C%227961.jpg%22%2C%227962.jpg%22%2C%227963.jpg%22%2C%227964.jpg%22%2C%227965.jpg%22%2C%227966.jpg%22%2C%227967.jpg%22%2C%227968.jpg%22%2C%227969.jpg%22%2C%227970.jpg%22%2C%227971.jpg%22%2C%227972.jpg%22%2C%227973.jpg%22%2C%227974.jpg%22%2C%227975.jpg%22%2C%227976.jpg%22%2C%227977.jpg%22%2C%227978.jpg%22%2C%227979.jpg%22%2C%227980.jpg%22%2C%227981.jpg%22%2C%228120.jpg%22%2C%228121.jpg%22%2C%228122.jpg%22%2C%228123.jpg%22%2C%228124.jpg%22%2C%228125.jpg%22%2C%228126.jpg%22%2C%228127.jpg%22%2C%228128.jpg%22%2C%228129.jpg%22%2C%228130.jpg%22%2C%228131.jpg%22%5D",
            filename="test4.zip",
            md5=False,
            checksum="c0dcc36b156539f1f94678b94791904ab3bfe1885812a309c21287daf507e009",
        ),
        DataLocation(
            url="https://cloud.cs.uni-tuebingen.de/public.php/dav/files/ZZxX65FGnQ8zjBP/Compressed%20Version/images/test/?accept=zip&files=%5B%228132.jpg%22%2C%228133.jpg%22%2C%228134.jpg%22%2C%228135.jpg%22%2C%228136.jpg%22%2C%228137.jpg%22%2C%228138.jpg%22%2C%228139.jpg%22%2C%228140.jpg%22%2C%228141.jpg%22%2C%228142.jpg%22%2C%228143.jpg%22%2C%228678.jpg%22%2C%228679.jpg%22%2C%228680.jpg%22%2C%228681.jpg%22%2C%228682.jpg%22%2C%228683.jpg%22%2C%228684.jpg%22%2C%228685.jpg%22%2C%228686.jpg%22%2C%228687.jpg%22%2C%228688.jpg%22%2C%228689.jpg%22%2C%228690.jpg%22%2C%228691.jpg%22%2C%228692.jpg%22%2C%228693.jpg%22%2C%228694.jpg%22%2C%228695.jpg%22%2C%228696.jpg%22%2C%228697.jpg%22%2C%228698.jpg%22%2C%228699.jpg%22%2C%228700.jpg%22%2C%228701.jpg%22%2C%228702.jpg%22%2C%228703.jpg%22%2C%228704.jpg%22%2C%228705.jpg%22%2C%228706.jpg%22%2C%228707.jpg%22%2C%228708.jpg%22%2C%228709.jpg%22%2C%228710.jpg%22%2C%228711.jpg%22%2C%228712.jpg%22%2C%228713.jpg%22%2C%228714.jpg%22%2C%228715.jpg%22%2C%228716.jpg%22%2C%228717.jpg%22%2C%228718.jpg%22%2C%228719.jpg%22%2C%228720.jpg%22%2C%228721.jpg%22%2C%228722.jpg%22%2C%228723.jpg%22%2C%228724.jpg%22%2C%228725.jpg%22%2C%228726.jpg%22%2C%228727.jpg%22%2C%228728.jpg%22%2C%228729.jpg%22%2C%228730.jpg%22%2C%228731.jpg%22%2C%228732.jpg%22%2C%228733.jpg%22%2C%228734.jpg%22%2C%228735.jpg%22%2C%228736.jpg%22%2C%228737.jpg%22%2C%228738.jpg%22%2C%228739.jpg%22%2C%228740.jpg%22%2C%228741.jpg%22%2C%228742.jpg%22%2C%228743.jpg%22%2C%228744.jpg%22%2C%228745.jpg%22%2C%228746.jpg%22%2C%228747.jpg%22%2C%228748.jpg%22%2C%228749.jpg%22%2C%228750.jpg%22%2C%228751.jpg%22%2C%228752.jpg%22%2C%228753.jpg%22%2C%228754.jpg%22%2C%228755.jpg%22%2C%228756.jpg%22%2C%228757.jpg%22%2C%228758.jpg%22%2C%228759.jpg%22%2C%228760.jpg%22%2C%228761.jpg%22%2C%228762.jpg%22%2C%228763.jpg%22%2C%228764.jpg%22%2C%228765.jpg%22%2C%228766.jpg%22%2C%228767.jpg%22%2C%228768.jpg%22%2C%228769.jpg%22%2C%228770.jpg%22%2C%228771.jpg%22%2C%228772.jpg%22%2C%228773.jpg%22%2C%228774.jpg%22%2C%228775.jpg%22%2C%228776.jpg%22%2C%228777.jpg%22%2C%228778.jpg%22%2C%228779.jpg%22%2C%228780.jpg%22%2C%228781.jpg%22%2C%228782.jpg%22%2C%228783.jpg%22%2C%228784.jpg%22%2C%228785.jpg%22%2C%228786.jpg%22%2C%228787.jpg%22%2C%228788.jpg%22%2C%228789.jpg%22%2C%228790.jpg%22%2C%228791.jpg%22%2C%228792.jpg%22%2C%228793.jpg%22%2C%228794.jpg%22%2C%228795.jpg%22%2C%228796.jpg%22%2C%228797.jpg%22%2C%228798.jpg%22%2C%228799.jpg%22%2C%228800.jpg%22%2C%228801.jpg%22%2C%228802.jpg%22%2C%228803.jpg%22%2C%228804.jpg%22%2C%228805.jpg%22%2C%228806.jpg%22%2C%228807.jpg%22%2C%228808.jpg%22%2C%228809.jpg%22%2C%228810.jpg%22%2C%228811.jpg%22%2C%228812.jpg%22%2C%228813.jpg%22%2C%228814.jpg%22%2C%228815.jpg%22%2C%228816.jpg%22%2C%228817.jpg%22%2C%228818.jpg%22%2C%228819.jpg%22%2C%228820.jpg%22%2C%228821.jpg%22%2C%228822.jpg%22%2C%228823.jpg%22%2C%228824.jpg%22%2C%228825.jpg%22%2C%228826.jpg%22%2C%228827.jpg%22%2C%228828.jpg%22%2C%228829.jpg%22%2C%228830.jpg%22%2C%228831.jpg%22%2C%228832.jpg%22%2C%228833.jpg%22%2C%228834.jpg%22%2C%228835.jpg%22%2C%228836.jpg%22%2C%228837.jpg%22%2C%228838.jpg%22%2C%228839.jpg%22%2C%228840.jpg%22%2C%228841.jpg%22%2C%228842.jpg%22%2C%228843.jpg%22%2C%228844.jpg%22%2C%228845.jpg%22%2C%228846.jpg%22%2C%228847.jpg%22%2C%228848.jpg%22%2C%228849.jpg%22%2C%228850.jpg%22%2C%228851.jpg%22%2C%228852.jpg%22%2C%228853.jpg%22%2C%228854.jpg%22%2C%228855.jpg%22%2C%228856.jpg%22%2C%228857.jpg%22%2C%228858.jpg%22%2C%229275.jpg%22%2C%229276.jpg%22%2C%229277.jpg%22%2C%229278.jpg%22%2C%229279.jpg%22%2C%229280.jpg%22%2C%229281.jpg%22%2C%229282.jpg%22%2C%229283.jpg%22%2C%229284.jpg%22%2C%229285.jpg%22%2C%229286.jpg%22%2C%229287.jpg%22%2C%229288.jpg%22%2C%229289.jpg%22%2C%229290.jpg%22%2C%229291.jpg%22%2C%229292.jpg%22%2C%229293.jpg%22%2C%229294.jpg%22%2C%229295.jpg%22%2C%229296.jpg%22%2C%229297.jpg%22%2C%229298.jpg%22%2C%229299.jpg%22%2C%229300.jpg%22%2C%229301.jpg%22%2C%229302.jpg%22%2C%229303.jpg%22%2C%229304.jpg%22%2C%229305.jpg%22%2C%229306.jpg%22%2C%229307.jpg%22%2C%229308.jpg%22%2C%229309.jpg%22%2C%229310.jpg%22%2C%229311.jpg%22%2C%229312.jpg%22%2C%229313.jpg%22%2C%229314.jpg%22%2C%229315.jpg%22%2C%229316.jpg%22%2C%229317.jpg%22%2C%229318.jpg%22%2C%229319.jpg%22%2C%229320.jpg%22%2C%229321.jpg%22%2C%229322.jpg%22%2C%229323.jpg%22%2C%229324.jpg%22%2C%229325.jpg%22%2C%229326.jpg%22%2C%229327.jpg%22%2C%229328.jpg%22%2C%229329.jpg%22%2C%229330.jpg%22%2C%229331.jpg%22%2C%229332.jpg%22%2C%229333.jpg%22%2C%229334.jpg%22%2C%229335.jpg%22%2C%229336.jpg%22%2C%229337.jpg%22%2C%229338.jpg%22%2C%229339.jpg%22%2C%229340.jpg%22%2C%229341.jpg%22%2C%229342.jpg%22%2C%229343.jpg%22%2C%229344.jpg%22%2C%229345.jpg%22%2C%229346.jpg%22%2C%229347.jpg%22%2C%229348.jpg%22%2C%229349.jpg%22%2C%229350.jpg%22%2C%229351.jpg%22%2C%229352.jpg%22%2C%229353.jpg%22%2C%229354.jpg%22%2C%229355.jpg%22%2C%229356.jpg%22%2C%229357.jpg%22%2C%229358.jpg%22%2C%229359.jpg%22%2C%229360.jpg%22%2C%229361.jpg%22%2C%229362.jpg%22%2C%229363.jpg%22%2C%229364.jpg%22%2C%229365.jpg%22%2C%229366.jpg%22%2C%229367.jpg%22%2C%229368.jpg%22%2C%229369.jpg%22%2C%229370.jpg%22%2C%229371.jpg%22%2C%229372.jpg%22%2C%229373.jpg%22%2C%229374.jpg%22%2C%229375.jpg%22%2C%229376.jpg%22%2C%229377.jpg%22%2C%229378.jpg%22%2C%229379.jpg%22%2C%229380.jpg%22%2C%229381.jpg%22%2C%229382.jpg%22%2C%229383.jpg%22%2C%229384.jpg%22%2C%229385.jpg%22%2C%229386.jpg%22%2C%229387.jpg%22%2C%229388.jpg%22%2C%229389.jpg%22%2C%229390.jpg%22%2C%229391.jpg%22%2C%229392.jpg%22%2C%229393.jpg%22%2C%229394.jpg%22%2C%229395.jpg%22%2C%229396.jpg%22%2C%229397.jpg%22%2C%229398.jpg%22%2C%229399.jpg%22%2C%229400.jpg%22%2C%229401.jpg%22%2C%229402.jpg%22%2C%229403.jpg%22%2C%229404.jpg%22%2C%229405.jpg%22%2C%229406.jpg%22%2C%229407.jpg%22%2C%229408.jpg%22%2C%229409.jpg%22%2C%229410.jpg%22%2C%229411.jpg%22%2C%229412.jpg%22%2C%229413.jpg%22%2C%229414.jpg%22%2C%229415.jpg%22%2C%229416.jpg%22%2C%229417.jpg%22%2C%229418.jpg%22%2C%229419.jpg%22%2C%229420.jpg%22%2C%229421.jpg%22%2C%229422.jpg%22%2C%229423.jpg%22%2C%229424.jpg%22%2C%229425.jpg%22%2C%229426.jpg%22%2C%229427.jpg%22%2C%229428.jpg%22%2C%229429.jpg%22%2C%229430.jpg%22%2C%229431.jpg%22%2C%229432.jpg%22%2C%229433.jpg%22%2C%229434.jpg%22%2C%229435.jpg%22%2C%229436.jpg%22%2C%229437.jpg%22%2C%229438.jpg%22%2C%229439.jpg%22%2C%229440.jpg%22%2C%229441.jpg%22%2C%229442.jpg%22%2C%229443.jpg%22%2C%229444.jpg%22%2C%229445.jpg%22%2C%229446.jpg%22%2C%229447.jpg%22%2C%229448.jpg%22%2C%229449.jpg%22%2C%229863.jpg%22%2C%229864.jpg%22%2C%229865.jpg%22%2C%229866.jpg%22%2C%229867.jpg%22%2C%229868.jpg%22%2C%229869.jpg%22%2C%229870.jpg%22%2C%229871.jpg%22%2C%229872.jpg%22%2C%229873.jpg%22%2C%229874.jpg%22%2C%229875.jpg%22%2C%229876.jpg%22%2C%229877.jpg%22%2C%229878.jpg%22%2C%229879.jpg%22%2C%229880.jpg%22%2C%229881.jpg%22%2C%229882.jpg%22%2C%229883.jpg%22%2C%229884.jpg%22%2C%229885.jpg%22%2C%229886.jpg%22%2C%229887.jpg%22%2C%229888.jpg%22%2C%229889.jpg%22%2C%229890.jpg%22%2C%229891.jpg%22%2C%229892.jpg%22%2C%229893.jpg%22%2C%229894.jpg%22%2C%229895.jpg%22%2C%229896.jpg%22%2C%229897.jpg%22%2C%229898.jpg%22%2C%229899.jpg%22%2C%229900.jpg%22%2C%229901.jpg%22%2C%229902.jpg%22%2C%229903.jpg%22%2C%229904.jpg%22%2C%229905.jpg%22%2C%229906.jpg%22%2C%229907.jpg%22%2C%229908.jpg%22%2C%229909.jpg%22%2C%229910.jpg%22%2C%229911.jpg%22%2C%229912.jpg%22%2C%229913.jpg%22%2C%229914.jpg%22%2C%229915.jpg%22%2C%229916.jpg%22%2C%229917.jpg%22%2C%229918.jpg%22%2C%229919.jpg%22%2C%229920.jpg%22%2C%229921.jpg%22%2C%229922.jpg%22%2C%229923.jpg%22%2C%229924.jpg%22%2C%229925.jpg%22%2C%229926.jpg%22%2C%229927.jpg%22%2C%229928.jpg%22%2C%229929.jpg%22%2C%229930.jpg%22%2C%229931.jpg%22%2C%229932.jpg%22%2C%229933.jpg%22%2C%229934.jpg%22%2C%229935.jpg%22%2C%229936.jpg%22%2C%229937.jpg%22%2C%229938.jpg%22%2C%229939.jpg%22%2C%229940.jpg%22%2C%229941.jpg%22%2C%229942.jpg%22%2C%229943.jpg%22%2C%229944.jpg%22%5D",
            filename="test5.zip",
            md5=False,
            checksum="e90fbde16ca97a6b7e2e201e71a428a6867e9a4b2a8f5ed42a57444d397a0a1f",
        ),
        DataLocation(
            url="https://cloud.cs.uni-tuebingen.de/public.php/dav/files/ZZxX65FGnQ8zjBP/Compressed%20Version/images/test/?accept=zip&files=%5B%229945.jpg%22%2C%229946.jpg%22%2C%229947.jpg%22%2C%229948.jpg%22%2C%229949.jpg%22%2C%229950.jpg%22%2C%229951.jpg%22%2C%229952.jpg%22%2C%229953.jpg%22%2C%229954.jpg%22%2C%229955.jpg%22%2C%229956.jpg%22%2C%229957.jpg%22%2C%229958.jpg%22%2C%229959.jpg%22%2C%229960.jpg%22%2C%229961.jpg%22%2C%229962.jpg%22%2C%229963.jpg%22%2C%229964.jpg%22%2C%229965.jpg%22%2C%229966.jpg%22%2C%229967.jpg%22%2C%229968.jpg%22%2C%229969.jpg%22%2C%229970.jpg%22%2C%229971.jpg%22%2C%229972.jpg%22%2C%229973.jpg%22%2C%229974.jpg%22%2C%229975.jpg%22%2C%229976.jpg%22%2C%229977.jpg%22%2C%229978.jpg%22%2C%229979.jpg%22%2C%229980.jpg%22%2C%229981.jpg%22%2C%229982.jpg%22%2C%229983.jpg%22%2C%229984.jpg%22%2C%229985.jpg%22%2C%229986.jpg%22%2C%229987.jpg%22%2C%229988.jpg%22%2C%229989.jpg%22%2C%229990.jpg%22%2C%229991.jpg%22%2C%229992.jpg%22%2C%229993.jpg%22%2C%229994.jpg%22%2C%229995.jpg%22%2C%229996.jpg%22%2C%229997.jpg%22%2C%229998.jpg%22%2C%229999.jpg%22%2C%2210000.jpg%22%2C%2210001.jpg%22%2C%2210002.jpg%22%2C%2210003.jpg%22%2C%2210004.jpg%22%2C%2210005.jpg%22%2C%2210006.jpg%22%2C%2210007.jpg%22%2C%2210008.jpg%22%2C%2210009.jpg%22%2C%2210010.jpg%22%2C%2210011.jpg%22%2C%2210012.jpg%22%2C%2210013.jpg%22%2C%2210014.jpg%22%2C%2210015.jpg%22%2C%2210016.jpg%22%2C%2210017.jpg%22%2C%2210018.jpg%22%2C%2210466.jpg%22%2C%2210467.jpg%22%2C%2210468.jpg%22%2C%2210469.jpg%22%2C%2210470.jpg%22%2C%2210471.jpg%22%2C%2210472.jpg%22%2C%2210473.jpg%22%2C%2210474.jpg%22%2C%2210475.jpg%22%2C%2210476.jpg%22%2C%2210477.jpg%22%2C%2210478.jpg%22%2C%2210479.jpg%22%2C%2210480.jpg%22%2C%2210481.jpg%22%2C%2210482.jpg%22%2C%2210483.jpg%22%2C%2210484.jpg%22%2C%2210485.jpg%22%2C%2210486.jpg%22%2C%2210487.jpg%22%2C%2210488.jpg%22%2C%2210489.jpg%22%2C%2210490.jpg%22%2C%2210491.jpg%22%2C%2210492.jpg%22%2C%2210493.jpg%22%2C%2210494.jpg%22%2C%2210495.jpg%22%2C%2210496.jpg%22%2C%2210497.jpg%22%2C%2210498.jpg%22%2C%2210499.jpg%22%2C%2210500.jpg%22%2C%2210501.jpg%22%2C%2210502.jpg%22%2C%2210503.jpg%22%2C%2210504.jpg%22%2C%2210505.jpg%22%2C%2210506.jpg%22%2C%2210507.jpg%22%2C%2210508.jpg%22%2C%2210509.jpg%22%2C%2210510.jpg%22%2C%2210511.jpg%22%2C%2210512.jpg%22%2C%2210513.jpg%22%2C%2210514.jpg%22%2C%2210515.jpg%22%2C%2210516.jpg%22%2C%2210517.jpg%22%2C%2210518.jpg%22%2C%2210519.jpg%22%2C%2210520.jpg%22%2C%2210521.jpg%22%2C%2210522.jpg%22%2C%2210523.jpg%22%2C%2210524.jpg%22%2C%2210525.jpg%22%2C%2210526.jpg%22%2C%2210527.jpg%22%2C%2210528.jpg%22%2C%2210529.jpg%22%2C%2210530.jpg%22%2C%2210531.jpg%22%2C%2210532.jpg%22%2C%2210533.jpg%22%2C%2210534.jpg%22%2C%2210535.jpg%22%2C%2210536.jpg%22%2C%2210537.jpg%22%2C%2210538.jpg%22%2C%2210539.jpg%22%2C%2210540.jpg%22%2C%2210541.jpg%22%2C%2210542.jpg%22%2C%2210543.jpg%22%2C%2210544.jpg%22%2C%2210545.jpg%22%2C%2210546.jpg%22%2C%2210547.jpg%22%2C%2210548.jpg%22%2C%2210549.jpg%22%2C%2210550.jpg%22%2C%2210551.jpg%22%2C%2210552.jpg%22%2C%2210553.jpg%22%2C%2210554.jpg%22%2C%2210555.jpg%22%2C%2210556.jpg%22%2C%2210557.jpg%22%2C%2210558.jpg%22%2C%2210559.jpg%22%2C%2210560.jpg%22%2C%2210561.jpg%22%2C%2210562.jpg%22%2C%2210563.jpg%22%2C%2210564.jpg%22%2C%2210565.jpg%22%2C%2210566.jpg%22%2C%2210567.jpg%22%2C%2210568.jpg%22%2C%2210569.jpg%22%2C%2210570.jpg%22%2C%2210571.jpg%22%2C%2210572.jpg%22%2C%2210573.jpg%22%2C%2210574.jpg%22%2C%2210575.jpg%22%2C%2210576.jpg%22%2C%2210577.jpg%22%2C%2210578.jpg%22%2C%2210579.jpg%22%2C%2210580.jpg%22%2C%2210581.jpg%22%2C%2210582.jpg%22%2C%2210583.jpg%22%2C%2210584.jpg%22%2C%2210585.jpg%22%2C%2210586.jpg%22%2C%2210587.jpg%22%2C%2210588.jpg%22%2C%2210589.jpg%22%2C%2210590.jpg%22%2C%2210591.jpg%22%2C%2210592.jpg%22%2C%2210593.jpg%22%2C%2210594.jpg%22%2C%2210595.jpg%22%2C%2210596.jpg%22%2C%2210597.jpg%22%2C%2210598.jpg%22%2C%2210599.jpg%22%2C%2210600.jpg%22%2C%2210601.jpg%22%2C%2210602.jpg%22%2C%2210603.jpg%22%2C%2210604.jpg%22%2C%2210605.jpg%22%2C%2210606.jpg%22%2C%2210607.jpg%22%2C%2210608.jpg%22%2C%2210609.jpg%22%2C%2210610.jpg%22%2C%2210611.jpg%22%2C%2210612.jpg%22%2C%2210613.jpg%22%2C%2210614.jpg%22%2C%2210615.jpg%22%2C%2210616.jpg%22%2C%2211254.jpg%22%2C%2211255.jpg%22%2C%2211256.jpg%22%2C%2211257.jpg%22%2C%2211258.jpg%22%2C%2211259.jpg%22%2C%2211260.jpg%22%2C%2211261.jpg%22%2C%2211262.jpg%22%2C%2211263.jpg%22%2C%2211264.jpg%22%2C%2211265.jpg%22%2C%2211266.jpg%22%2C%2211267.jpg%22%2C%2211268.jpg%22%2C%2211269.jpg%22%2C%2211270.jpg%22%2C%2211271.jpg%22%2C%2211272.jpg%22%2C%2211273.jpg%22%2C%2211274.jpg%22%2C%2211288.jpg%22%2C%2211289.jpg%22%2C%2211290.jpg%22%2C%2211291.jpg%22%2C%2211292.jpg%22%2C%2211293.jpg%22%2C%2211294.jpg%22%2C%2211295.jpg%22%2C%2211296.jpg%22%2C%2211297.jpg%22%2C%2211298.jpg%22%2C%2211299.jpg%22%2C%2211300.jpg%22%2C%2211301.jpg%22%2C%2211302.jpg%22%2C%2211303.jpg%22%2C%2211304.jpg%22%2C%2211305.jpg%22%2C%2211306.jpg%22%2C%2211307.jpg%22%2C%2211308.jpg%22%2C%2211309.jpg%22%2C%2211310.jpg%22%2C%2211311.jpg%22%2C%2211312.jpg%22%2C%2211313.jpg%22%2C%2211314.jpg%22%2C%2211624.jpg%22%2C%2211625.jpg%22%2C%2211626.jpg%22%2C%2211627.jpg%22%2C%2211628.jpg%22%2C%2211629.jpg%22%2C%2211630.jpg%22%2C%2211631.jpg%22%2C%2211632.jpg%22%2C%2211633.jpg%22%2C%2211634.jpg%22%2C%2211635.jpg%22%2C%2211636.jpg%22%2C%2211637.jpg%22%2C%2211638.jpg%22%2C%2211639.jpg%22%2C%2211640.jpg%22%2C%2211641.jpg%22%2C%2211642.jpg%22%2C%2211643.jpg%22%2C%2211644.jpg%22%2C%2211645.jpg%22%2C%2211646.jpg%22%2C%2211647.jpg%22%2C%2211648.jpg%22%2C%2211649.jpg%22%2C%2211650.jpg%22%2C%2211651.jpg%22%2C%2211652.jpg%22%2C%2211653.jpg%22%2C%2211654.jpg%22%2C%2211655.jpg%22%2C%2211656.jpg%22%2C%2211657.jpg%22%2C%2211658.jpg%22%2C%2211659.jpg%22%2C%2211660.jpg%22%2C%2211661.jpg%22%2C%2211662.jpg%22%2C%2211663.jpg%22%2C%2211664.jpg%22%2C%2211665.jpg%22%2C%2211666.jpg%22%2C%2211667.jpg%22%2C%2211668.jpg%22%2C%2211669.jpg%22%2C%2211670.jpg%22%2C%2211671.jpg%22%2C%2211672.jpg%22%2C%2211673.jpg%22%2C%2211674.jpg%22%2C%2211675.jpg%22%2C%2211676.jpg%22%2C%2211677.jpg%22%2C%2211678.jpg%22%2C%2211679.jpg%22%2C%2211680.jpg%22%2C%2211681.jpg%22%2C%2211682.jpg%22%2C%2211683.jpg%22%2C%2211684.jpg%22%2C%2211685.jpg%22%2C%2211686.jpg%22%2C%2211687.jpg%22%2C%2211688.jpg%22%2C%2211689.jpg%22%2C%2211690.jpg%22%2C%2211691.jpg%22%2C%2211692.jpg%22%2C%2211693.jpg%22%2C%2211694.jpg%22%2C%2211695.jpg%22%2C%2211696.jpg%22%2C%2211697.jpg%22%2C%2211698.jpg%22%2C%2211699.jpg%22%2C%2211700.jpg%22%2C%2211701.jpg%22%2C%2211702.jpg%22%2C%2211703.jpg%22%2C%2211704.jpg%22%2C%2211705.jpg%22%2C%2211706.jpg%22%2C%2211707.jpg%22%2C%2211708.jpg%22%2C%2211709.jpg%22%2C%2211710.jpg%22%2C%2211711.jpg%22%2C%2211712.jpg%22%2C%2211713.jpg%22%2C%2211714.jpg%22%2C%2211715.jpg%22%2C%2211716.jpg%22%2C%2211717.jpg%22%2C%2211718.jpg%22%2C%2211719.jpg%22%2C%2211720.jpg%22%2C%2211721.jpg%22%2C%2211722.jpg%22%2C%2211723.jpg%22%2C%2211724.jpg%22%2C%2211725.jpg%22%2C%2211726.jpg%22%2C%2211727.jpg%22%2C%2211728.jpg%22%2C%2211729.jpg%22%2C%2211730.jpg%22%2C%2211731.jpg%22%2C%2211732.jpg%22%2C%2211733.jpg%22%2C%2211734.jpg%22%2C%2211735.jpg%22%2C%2211736.jpg%22%2C%2211737.jpg%22%2C%2211738.jpg%22%2C%2211739.jpg%22%2C%2211740.jpg%22%2C%2211741.jpg%22%2C%2211742.jpg%22%2C%2211743.jpg%22%2C%2211744.jpg%22%2C%2211745.jpg%22%2C%2211746.jpg%22%2C%2211747.jpg%22%2C%2211748.jpg%22%2C%2211749.jpg%22%2C%2211750.jpg%22%2C%2211751.jpg%22%2C%2211752.jpg%22%2C%2211753.jpg%22%2C%2211754.jpg%22%2C%2211755.jpg%22%2C%2212051.jpg%22%2C%2212052.jpg%22%2C%2212053.jpg%22%2C%2212054.jpg%22%2C%2212055.jpg%22%2C%2212056.jpg%22%2C%2212057.jpg%22%2C%2212058.jpg%22%2C%2212059.jpg%22%2C%2212060.jpg%22%2C%2212061.jpg%22%2C%2212062.jpg%22%2C%2212063.jpg%22%2C%2212064.jpg%22%2C%2212065.jpg%22%2C%2212066.jpg%22%2C%2212067.jpg%22%2C%2212068.jpg%22%2C%2212069.jpg%22%2C%2212070.jpg%22%2C%2212071.jpg%22%2C%2212072.jpg%22%2C%2212073.jpg%22%2C%2212074.jpg%22%2C%2212075.jpg%22%2C%2212076.jpg%22%2C%2212077.jpg%22%2C%2212078.jpg%22%2C%2212079.jpg%22%2C%2212080.jpg%22%2C%2212081.jpg%22%2C%2212082.jpg%22%2C%2212083.jpg%22%2C%2212084.jpg%22%2C%2212085.jpg%22%2C%2212086.jpg%22%2C%2212087.jpg%22%2C%2212088.jpg%22%2C%2212089.jpg%22%2C%2212090.jpg%22%2C%2212091.jpg%22%2C%2212092.jpg%22%2C%2212093.jpg%22%2C%2212094.jpg%22%2C%2212095.jpg%22%5D",
            filename="test6.zip",
            md5=False,
            checksum="2b418f3e0aae6252e64f9ee0a0cceb9f824107e5dea9b94fa21a646dae14545d",
        ),
        DataLocation(
            url="https://cloud.cs.uni-tuebingen.de/public.php/dav/files/ZZxX65FGnQ8zjBP/Compressed%20Version/images/test/?accept=zip&files=%5B%2212096.jpg%22%2C%2212097.jpg%22%2C%2212098.jpg%22%2C%2212099.jpg%22%2C%2212100.jpg%22%2C%2212101.jpg%22%2C%2212102.jpg%22%2C%2212103.jpg%22%2C%2212104.jpg%22%2C%2212105.jpg%22%2C%2212106.jpg%22%2C%2212107.jpg%22%2C%2212108.jpg%22%2C%2212109.jpg%22%2C%2212110.jpg%22%2C%2212111.jpg%22%2C%2212112.jpg%22%2C%2212113.jpg%22%2C%2212114.jpg%22%2C%2212115.jpg%22%2C%2212116.jpg%22%2C%2212117.jpg%22%2C%2212118.jpg%22%2C%2212119.jpg%22%2C%2212120.jpg%22%2C%2212121.jpg%22%2C%2212122.jpg%22%2C%2212123.jpg%22%2C%2212124.jpg%22%2C%2212125.jpg%22%2C%2212126.jpg%22%2C%2212127.jpg%22%2C%2212128.jpg%22%2C%2212129.jpg%22%2C%2212130.jpg%22%2C%2212131.jpg%22%2C%2212132.jpg%22%2C%2212133.jpg%22%2C%2212134.jpg%22%2C%2212135.jpg%22%2C%2212136.jpg%22%2C%2212137.jpg%22%2C%2212138.jpg%22%2C%2212139.jpg%22%2C%2212140.jpg%22%2C%2212141.jpg%22%2C%2212142.jpg%22%2C%2212143.jpg%22%2C%2212144.jpg%22%2C%2212145.jpg%22%2C%2212146.jpg%22%2C%2212147.jpg%22%2C%2212148.jpg%22%2C%2212149.jpg%22%2C%2212150.jpg%22%2C%2212151.jpg%22%2C%2212152.jpg%22%2C%2212153.jpg%22%2C%2212154.jpg%22%2C%2212155.jpg%22%2C%2212156.jpg%22%2C%2212157.jpg%22%2C%2212158.jpg%22%2C%2212159.jpg%22%2C%2212160.jpg%22%2C%2212161.jpg%22%2C%2212162.jpg%22%2C%2212163.jpg%22%2C%2212164.jpg%22%2C%2212165.jpg%22%2C%2212166.jpg%22%2C%2212167.jpg%22%2C%2212168.jpg%22%2C%2212169.jpg%22%2C%2212170.jpg%22%2C%2212171.jpg%22%2C%2212172.jpg%22%2C%2212173.jpg%22%2C%2212174.jpg%22%2C%2212175.jpg%22%2C%2212176.jpg%22%2C%2212177.jpg%22%2C%2212178.jpg%22%2C%2212179.jpg%22%2C%2212180.jpg%22%2C%2212181.jpg%22%2C%2212182.jpg%22%2C%2212183.jpg%22%2C%2212184.jpg%22%2C%2212185.jpg%22%2C%2212186.jpg%22%2C%2212596.jpg%22%2C%2212597.jpg%22%2C%2212598.jpg%22%2C%2212599.jpg%22%2C%2212600.jpg%22%2C%2212601.jpg%22%2C%2212602.jpg%22%2C%2212603.jpg%22%2C%2212604.jpg%22%2C%2212605.jpg%22%2C%2212606.jpg%22%2C%2212607.jpg%22%2C%2212608.jpg%22%2C%2212609.jpg%22%2C%2212610.jpg%22%2C%2212611.jpg%22%2C%2212612.jpg%22%2C%2212613.jpg%22%2C%2212614.jpg%22%2C%2212615.jpg%22%2C%2212616.jpg%22%2C%2212617.jpg%22%2C%2212618.jpg%22%2C%2212619.jpg%22%2C%2212620.jpg%22%2C%2212621.jpg%22%2C%2212622.jpg%22%2C%2212623.jpg%22%2C%2212624.jpg%22%2C%2212625.jpg%22%2C%2212626.jpg%22%2C%2212627.jpg%22%2C%2212628.jpg%22%2C%2212629.jpg%22%2C%2212630.jpg%22%2C%2212631.jpg%22%2C%2212632.jpg%22%2C%2212633.jpg%22%2C%2212634.jpg%22%2C%2212635.jpg%22%2C%2212636.jpg%22%2C%2212637.jpg%22%2C%2212638.jpg%22%2C%2212639.jpg%22%2C%2212640.jpg%22%2C%2212641.jpg%22%2C%2212642.jpg%22%2C%2212643.jpg%22%2C%2212644.jpg%22%2C%2212645.jpg%22%2C%2212646.jpg%22%2C%2212647.jpg%22%2C%2212648.jpg%22%2C%2212649.jpg%22%2C%2212650.jpg%22%2C%2212651.jpg%22%2C%2212652.jpg%22%2C%2212653.jpg%22%2C%2212654.jpg%22%2C%2212655.jpg%22%2C%2212656.jpg%22%2C%2212657.jpg%22%2C%2212658.jpg%22%2C%2212659.jpg%22%2C%2212660.jpg%22%2C%2212661.jpg%22%2C%2212662.jpg%22%2C%2212663.jpg%22%2C%2212664.jpg%22%2C%2212665.jpg%22%2C%2212666.jpg%22%2C%2212667.jpg%22%2C%2212668.jpg%22%2C%2212669.jpg%22%2C%2212670.jpg%22%2C%2212671.jpg%22%2C%2212672.jpg%22%2C%2212673.jpg%22%2C%2212674.jpg%22%2C%2212675.jpg%22%2C%2212676.jpg%22%2C%2212677.jpg%22%2C%2212678.jpg%22%2C%2212679.jpg%22%2C%2212680.jpg%22%2C%2212681.jpg%22%2C%2212682.jpg%22%2C%2212683.jpg%22%2C%2212684.jpg%22%2C%2212685.jpg%22%2C%2212686.jpg%22%2C%2212687.jpg%22%2C%2212688.jpg%22%2C%2212689.jpg%22%2C%2212690.jpg%22%2C%2212691.jpg%22%2C%2212692.jpg%22%2C%2212693.jpg%22%2C%2212694.jpg%22%2C%2212695.jpg%22%2C%2212696.jpg%22%2C%2212697.jpg%22%2C%2212698.jpg%22%2C%2212699.jpg%22%2C%2212700.jpg%22%2C%2212701.jpg%22%2C%2212702.jpg%22%2C%2212703.jpg%22%2C%2212704.jpg%22%2C%2212705.jpg%22%2C%2212706.jpg%22%2C%2212707.jpg%22%2C%2212708.jpg%22%2C%2212709.jpg%22%2C%2212710.jpg%22%2C%2212711.jpg%22%2C%2212712.jpg%22%2C%2212713.jpg%22%2C%2212714.jpg%22%2C%2212715.jpg%22%2C%2212716.jpg%22%2C%2212717.jpg%22%2C%2212718.jpg%22%2C%2212719.jpg%22%2C%2212720.jpg%22%2C%2212721.jpg%22%2C%2212722.jpg%22%2C%2212723.jpg%22%2C%2212724.jpg%22%2C%2212725.jpg%22%2C%2212726.jpg%22%2C%2212727.jpg%22%2C%2212728.jpg%22%2C%2212729.jpg%22%2C%2212730.jpg%22%2C%2212731.jpg%22%2C%2212732.jpg%22%2C%2212733.jpg%22%2C%2212734.jpg%22%2C%2212735.jpg%22%2C%2212736.jpg%22%2C%2212737.jpg%22%2C%2212738.jpg%22%2C%2212739.jpg%22%2C%2212740.jpg%22%2C%2212741.jpg%22%2C%2212742.jpg%22%2C%2212743.jpg%22%2C%2212744.jpg%22%2C%2212745.jpg%22%2C%2212746.jpg%22%2C%2212747.jpg%22%2C%2212748.jpg%22%2C%2212749.jpg%22%2C%2212750.jpg%22%2C%2212751.jpg%22%2C%2212752.jpg%22%2C%2212753.jpg%22%2C%2212754.jpg%22%2C%2212755.jpg%22%2C%2212756.jpg%22%2C%2212757.jpg%22%2C%2212758.jpg%22%2C%2212759.jpg%22%2C%2212760.jpg%22%2C%2212761.jpg%22%2C%2212762.jpg%22%2C%2212763.jpg%22%2C%2214088.jpg%22%2C%2214089.jpg%22%2C%2214090.jpg%22%2C%2214091.jpg%22%2C%2214092.jpg%22%2C%2214093.jpg%22%2C%2214094.jpg%22%2C%2214095.jpg%22%2C%2214096.jpg%22%2C%2214097.jpg%22%2C%2214098.jpg%22%2C%2214099.jpg%22%2C%2214100.jpg%22%2C%2214101.jpg%22%2C%2214102.jpg%22%2C%2214103.jpg%22%2C%2214104.jpg%22%2C%2214105.jpg%22%2C%2214106.jpg%22%2C%2214107.jpg%22%2C%2214108.jpg%22%2C%2214109.jpg%22%2C%2214110.jpg%22%2C%2214111.jpg%22%2C%2214112.jpg%22%2C%2214113.jpg%22%2C%2214114.jpg%22%2C%2214115.jpg%22%2C%2214116.jpg%22%2C%2214117.jpg%22%2C%2214118.jpg%22%2C%2214119.jpg%22%2C%2214120.jpg%22%2C%2214121.jpg%22%2C%2214122.jpg%22%2C%2214123.jpg%22%2C%2214124.jpg%22%2C%2214125.jpg%22%2C%2214126.jpg%22%2C%2214127.jpg%22%2C%2214128.jpg%22%2C%2214129.jpg%22%2C%2214130.jpg%22%2C%2214131.jpg%22%2C%2214132.jpg%22%2C%2214133.jpg%22%2C%2214134.jpg%22%2C%2214135.jpg%22%2C%2214136.jpg%22%2C%2214137.jpg%22%2C%2214138.jpg%22%2C%2214139.jpg%22%2C%2214140.jpg%22%2C%2214141.jpg%22%2C%2214142.jpg%22%2C%2214143.jpg%22%2C%2214144.jpg%22%2C%2214145.jpg%22%2C%2214146.jpg%22%2C%2214147.jpg%22%2C%2214148.jpg%22%2C%2214149.jpg%22%2C%2214150.jpg%22%2C%2214151.jpg%22%2C%2214152.jpg%22%2C%2214153.jpg%22%2C%2214154.jpg%22%2C%2214155.jpg%22%2C%2214156.jpg%22%2C%2214157.jpg%22%2C%2214158.jpg%22%2C%2214159.jpg%22%2C%2214160.jpg%22%2C%2214161.jpg%22%2C%2214162.jpg%22%2C%2214163.jpg%22%2C%2214164.jpg%22%2C%2214165.jpg%22%2C%2214166.jpg%22%2C%2214167.jpg%22%2C%2214168.jpg%22%2C%2214169.jpg%22%2C%2214170.jpg%22%2C%2214171.jpg%22%2C%2214172.jpg%22%2C%2214173.jpg%22%2C%2214174.jpg%22%2C%2214175.jpg%22%2C%2214176.jpg%22%2C%2214177.jpg%22%2C%2214178.jpg%22%2C%2214179.jpg%22%2C%2214180.jpg%22%2C%2214181.jpg%22%2C%2214182.jpg%22%2C%2214183.jpg%22%2C%2214184.jpg%22%2C%2214185.jpg%22%2C%2214186.jpg%22%2C%2214187.jpg%22%2C%2214188.jpg%22%2C%2214189.jpg%22%2C%2214190.jpg%22%2C%2214191.jpg%22%2C%2214192.jpg%22%2C%2214193.jpg%22%2C%2214194.jpg%22%2C%2214195.jpg%22%2C%2214196.jpg%22%2C%2214197.jpg%22%2C%2214198.jpg%22%2C%2214199.jpg%22%2C%2214200.jpg%22%2C%2214201.jpg%22%2C%2214202.jpg%22%2C%2214203.jpg%22%2C%2214204.jpg%22%2C%2214205.jpg%22%2C%2214206.jpg%22%2C%2214207.jpg%22%2C%2214208.jpg%22%2C%2214209.jpg%22%2C%2214210.jpg%22%2C%2214211.jpg%22%2C%2214212.jpg%22%2C%2214213.jpg%22%2C%2214214.jpg%22%2C%2214215.jpg%22%2C%2214216.jpg%22%2C%2214217.jpg%22%2C%2214218.jpg%22%2C%2214219.jpg%22%2C%2214220.jpg%22%5D",
            filename="test7.zip",
            md5=False,
            checksum="7770d2a9f544076f9fba70b8bb16ae515d541a6bed1a8a5a84a61c2d0b43a9c1",
        ),
        DataLocation(
            url="https://cloud.cs.uni-tuebingen.de/public.php/dav/files/ZZxX65FGnQ8zjBP/Compressed%20Version/images/test/?accept=zip&files=%5B%2214279.jpg%22%2C%2214280.jpg%22%2C%2214281.jpg%22%2C%2214282.jpg%22%2C%2214283.jpg%22%2C%2214284.jpg%22%2C%2214285.jpg%22%2C%2214286.jpg%22%2C%2214287.jpg%22%2C%2214288.jpg%22%2C%2214289.jpg%22%2C%2214290.jpg%22%2C%2214291.jpg%22%2C%2214292.jpg%22%2C%2214293.jpg%22%2C%2214294.jpg%22%2C%2214295.jpg%22%2C%2214296.jpg%22%2C%2214297.jpg%22%2C%2214298.jpg%22%2C%2214299.jpg%22%2C%2214300.jpg%22%2C%2214301.jpg%22%2C%2214302.jpg%22%2C%2214303.jpg%22%2C%2214304.jpg%22%2C%2214305.jpg%22%2C%2214306.jpg%22%2C%2214307.jpg%22%2C%2214308.jpg%22%2C%2214309.jpg%22%2C%2214310.jpg%22%2C%2214311.jpg%22%2C%2214312.jpg%22%2C%2214313.jpg%22%2C%2214314.jpg%22%2C%2214315.jpg%22%2C%2214316.jpg%22%2C%2214317.jpg%22%2C%2214318.jpg%22%2C%2214319.jpg%22%2C%2214320.jpg%22%2C%2214321.jpg%22%2C%2214322.jpg%22%2C%2214323.jpg%22%2C%2214324.jpg%22%2C%2214325.jpg%22%2C%2214326.jpg%22%2C%2214327.jpg%22%2C%2214328.jpg%22%2C%2214329.jpg%22%2C%2214330.jpg%22%2C%2214331.jpg%22%2C%2214332.jpg%22%2C%2214333.jpg%22%2C%2214334.jpg%22%2C%2214335.jpg%22%2C%2214336.jpg%22%2C%2214337.jpg%22%2C%2214338.jpg%22%2C%2214339.jpg%22%2C%2214340.jpg%22%2C%2214341.jpg%22%2C%2214342.jpg%22%2C%2214343.jpg%22%2C%2214344.jpg%22%2C%2214345.jpg%22%2C%2214346.jpg%22%2C%2214347.jpg%22%2C%2214348.jpg%22%2C%2214349.jpg%22%2C%2214350.jpg%22%2C%2214351.jpg%22%2C%2214352.jpg%22%2C%2214353.jpg%22%2C%2214354.jpg%22%2C%2214355.jpg%22%2C%2214356.jpg%22%2C%2214357.jpg%22%2C%2214358.jpg%22%2C%2214359.jpg%22%2C%2214360.jpg%22%2C%2214361.jpg%22%2C%2214362.jpg%22%2C%2214363.jpg%22%2C%2214364.jpg%22%2C%2214365.jpg%22%2C%2214366.jpg%22%2C%2214367.jpg%22%2C%2214368.jpg%22%2C%2214369.jpg%22%2C%2214370.jpg%22%2C%2214371.jpg%22%2C%2214372.jpg%22%2C%2214373.jpg%22%2C%2214374.jpg%22%2C%2214375.jpg%22%2C%2214376.jpg%22%2C%2214377.jpg%22%2C%2214378.jpg%22%2C%2214379.jpg%22%2C%2214380.jpg%22%2C%2214381.jpg%22%2C%2214382.jpg%22%2C%2214383.jpg%22%2C%2214384.jpg%22%2C%2214385.jpg%22%2C%2214386.jpg%22%2C%2214387.jpg%22%2C%2214388.jpg%22%2C%2214389.jpg%22%2C%2214390.jpg%22%2C%2214391.jpg%22%2C%2214392.jpg%22%2C%2214393.jpg%22%2C%2214394.jpg%22%2C%2214395.jpg%22%2C%2214396.jpg%22%2C%2214397.jpg%22%2C%2214398.jpg%22%2C%2214399.jpg%22%2C%2214400.jpg%22%2C%2214401.jpg%22%2C%2214402.jpg%22%2C%2214403.jpg%22%2C%2214404.jpg%22%2C%2214405.jpg%22%2C%2214406.jpg%22%2C%2214407.jpg%22%2C%2214408.jpg%22%2C%2214409.jpg%22%2C%2214410.jpg%22%2C%2214411.jpg%22%2C%2214412.jpg%22%2C%2214413.jpg%22%2C%2214414.jpg%22%2C%2214415.jpg%22%2C%2214416.jpg%22%2C%2214417.jpg%22%2C%2214418.jpg%22%2C%2214419.jpg%22%2C%2214420.jpg%22%2C%2214421.jpg%22%2C%2214422.jpg%22%2C%2214423.jpg%22%2C%2214424.jpg%22%2C%2214425.jpg%22%2C%2214426.jpg%22%2C%2214427.jpg%22%2C%2214428.jpg%22%2C%2214429.jpg%22%2C%2214430.jpg%22%2C%2214431.jpg%22%2C%2214432.jpg%22%2C%2214433.jpg%22%2C%2214434.jpg%22%2C%2214435.jpg%22%2C%2214436.jpg%22%2C%2214437.jpg%22%2C%2214438.jpg%22%2C%2214439.jpg%22%2C%2214440.jpg%22%2C%2214441.jpg%22%2C%2214442.jpg%22%2C%2214443.jpg%22%2C%2214444.jpg%22%2C%2214445.jpg%22%2C%2214446.jpg%22%2C%2214447.jpg%22%2C%2214448.jpg%22%2C%2214449.jpg%22%2C%2214450.jpg%22%2C%2214451.jpg%22%2C%2214452.jpg%22%2C%2214453.jpg%22%2C%2214454.jpg%22%2C%2214455.jpg%22%2C%2214456.jpg%22%2C%2214457.jpg%22%2C%2214458.jpg%22%2C%2214459.jpg%22%2C%2214460.jpg%22%2C%2214461.jpg%22%2C%2214462.jpg%22%2C%2214463.jpg%22%2C%2214464.jpg%22%2C%2214465.jpg%22%2C%2214466.jpg%22%2C%2214467.jpg%22%2C%2214468.jpg%22%2C%2214469.jpg%22%2C%2214470.jpg%22%2C%2214471.jpg%22%2C%2214472.jpg%22%2C%2214473.jpg%22%2C%2214474.jpg%22%2C%2214475.jpg%22%2C%2214476.jpg%22%2C%2214477.jpg%22%2C%2214478.jpg%22%2C%2214479.jpg%22%2C%2214480.jpg%22%2C%2214481.jpg%22%2C%2214482.jpg%22%2C%2214483.jpg%22%2C%2214484.jpg%22%2C%2214485.jpg%22%2C%2214486.jpg%22%2C%2214487.jpg%22%2C%2214488.jpg%22%2C%2214489.jpg%22%2C%2214490.jpg%22%2C%2214491.jpg%22%2C%2214492.jpg%22%2C%2214493.jpg%22%2C%2214494.jpg%22%2C%2214495.jpg%22%2C%2214496.jpg%22%2C%2214497.jpg%22%2C%2214498.jpg%22%2C%2214499.jpg%22%2C%2214500.jpg%22%2C%2214501.jpg%22%2C%2214502.jpg%22%2C%2214503.jpg%22%2C%2214504.jpg%22%2C%2214505.jpg%22%2C%2214506.jpg%22%2C%2214507.jpg%22%2C%2214508.jpg%22%2C%2214509.jpg%22%2C%2214510.jpg%22%2C%2214511.jpg%22%2C%2214512.jpg%22%2C%2214513.jpg%22%2C%2214514.jpg%22%2C%2214515.jpg%22%2C%2214516.jpg%22%2C%2214517.jpg%22%2C%2214518.jpg%22%2C%2214519.jpg%22%2C%2214520.jpg%22%2C%2214521.jpg%22%2C%2214522.jpg%22%2C%2214523.jpg%22%2C%2214524.jpg%22%2C%2214525.jpg%22%2C%2214526.jpg%22%2C%2214527.jpg%22%2C%2214528.jpg%22%2C%2214529.jpg%22%2C%2214530.jpg%22%2C%2214531.jpg%22%2C%2214532.jpg%22%2C%2214533.jpg%22%2C%2214534.jpg%22%2C%2214535.jpg%22%2C%2214536.jpg%22%2C%2214537.jpg%22%2C%2214538.jpg%22%2C%2214539.jpg%22%2C%2214540.jpg%22%2C%2214541.jpg%22%2C%2214542.jpg%22%2C%2214543.jpg%22%2C%2214544.jpg%22%2C%2214545.jpg%22%2C%2214546.jpg%22%2C%2214547.jpg%22%2C%2214548.jpg%22%2C%2214549.jpg%22%2C%2214550.jpg%22%2C%2214551.jpg%22%2C%2214552.jpg%22%2C%2214553.jpg%22%2C%2214554.jpg%22%2C%2214555.jpg%22%2C%2214556.jpg%22%2C%2214557.jpg%22%2C%2214558.jpg%22%2C%2214559.jpg%22%2C%2214560.jpg%22%2C%2214561.jpg%22%2C%2214562.jpg%22%2C%2214563.jpg%22%2C%2214564.jpg%22%2C%2214565.jpg%22%2C%2214566.jpg%22%2C%2214567.jpg%22%2C%2214568.jpg%22%2C%2214569.jpg%22%2C%2214570.jpg%22%2C%2214571.jpg%22%2C%2214572.jpg%22%2C%2214573.jpg%22%2C%2214574.jpg%22%2C%2214575.jpg%22%2C%2214576.jpg%22%2C%2214577.jpg%22%2C%2214578.jpg%22%2C%2214579.jpg%22%2C%2214580.jpg%22%2C%2214581.jpg%22%2C%2214582.jpg%22%2C%2214583.jpg%22%2C%2214584.jpg%22%2C%2214585.jpg%22%2C%2214586.jpg%22%2C%2214587.jpg%22%2C%2214588.jpg%22%2C%2214589.jpg%22%2C%2214590.jpg%22%2C%2214591.jpg%22%2C%2214592.jpg%22%2C%2214593.jpg%22%2C%2214594.jpg%22%2C%2214595.jpg%22%2C%2214596.jpg%22%2C%2214597.jpg%22%2C%2214598.jpg%22%2C%2214599.jpg%22%2C%2214600.jpg%22%2C%2214601.jpg%22%2C%2214602.jpg%22%2C%2214603.jpg%22%2C%2214604.jpg%22%2C%2214605.jpg%22%2C%2214606.jpg%22%2C%2214607.jpg%22%2C%2214608.jpg%22%2C%2214609.jpg%22%2C%2214610.jpg%22%2C%2214611.jpg%22%2C%2214612.jpg%22%2C%2214613.jpg%22%2C%2214614.jpg%22%2C%2214615.jpg%22%2C%2214616.jpg%22%2C%2214617.jpg%22%2C%2214618.jpg%22%2C%2214619.jpg%22%2C%2214620.jpg%22%2C%2214621.jpg%22%2C%2214622.jpg%22%2C%2214623.jpg%22%2C%2214624.jpg%22%2C%2214625.jpg%22%2C%2214626.jpg%22%2C%2214627.jpg%22%2C%2214628.jpg%22%2C%2214629.jpg%22%2C%2214630.jpg%22%2C%2214221.jpg%22%2C%2214222.jpg%22%2C%2214223.jpg%22%2C%2214224.jpg%22%2C%2214225.jpg%22%2C%2214226.jpg%22%2C%2214227.jpg%22%2C%2214228.jpg%22%2C%2214229.jpg%22%2C%2214230.jpg%22%2C%2214231.jpg%22%2C%2214232.jpg%22%2C%2214233.jpg%22%2C%2214234.jpg%22%2C%2214235.jpg%22%2C%2214236.jpg%22%2C%2214237.jpg%22%2C%2214238.jpg%22%2C%2214239.jpg%22%2C%2214240.jpg%22%2C%2214241.jpg%22%2C%2214242.jpg%22%2C%2214243.jpg%22%2C%2214244.jpg%22%2C%2214245.jpg%22%2C%2214246.jpg%22%2C%2214247.jpg%22%2C%2214248.jpg%22%2C%2214249.jpg%22%2C%2214250.jpg%22%2C%2214251.jpg%22%2C%2214252.jpg%22%2C%2214253.jpg%22%2C%2214254.jpg%22%2C%2214255.jpg%22%2C%2214256.jpg%22%2C%2214257.jpg%22%2C%2214258.jpg%22%2C%2214259.jpg%22%2C%2214260.jpg%22%2C%2214261.jpg%22%2C%2214262.jpg%22%2C%2214263.jpg%22%2C%2214264.jpg%22%2C%2214265.jpg%22%2C%2214266.jpg%22%2C%2214267.jpg%22%2C%2214268.jpg%22%2C%2214269.jpg%22%2C%2214270.jpg%22%2C%2214271.jpg%22%2C%2214272.jpg%22%2C%2214273.jpg%22%2C%2214274.jpg%22%2C%2214275.jpg%22%2C%2214276.jpg%22%2C%2214277.jpg%22%2C%2214278.jpg%22%5D",
            filename="test8.zip",
            md5=False,
            checksum="3ab2f8b509e80bcef2e7f2e767682cbca7417e74f6734e8e7b7eae62f47d461d",
        ),
        DataLocation(
            url="https://cloud.cs.uni-tuebingen.de/public.php/dav/files/ZZxX65FGnQ8zjBP/Compressed%20Version/images/test/?accept=zip&files=%5B%2214631.jpg%22%2C%2215416.jpg%22%2C%2215417.jpg%22%2C%2215418.jpg%22%2C%2215419.jpg%22%2C%2215420.jpg%22%2C%2215421.jpg%22%2C%2215422.jpg%22%2C%2215423.jpg%22%2C%2215424.jpg%22%2C%2215425.jpg%22%2C%2215426.jpg%22%2C%2215427.jpg%22%2C%2215428.jpg%22%2C%2215429.jpg%22%2C%2215430.jpg%22%2C%2215431.jpg%22%2C%2215432.jpg%22%2C%2215433.jpg%22%2C%2215434.jpg%22%2C%2215435.jpg%22%2C%2215436.jpg%22%2C%2215437.jpg%22%2C%2215438.jpg%22%2C%2215439.jpg%22%2C%2215440.jpg%22%2C%2215441.jpg%22%2C%2215442.jpg%22%2C%2215443.jpg%22%2C%2215444.jpg%22%2C%2215445.jpg%22%2C%2215446.jpg%22%2C%2215447.jpg%22%2C%2215448.jpg%22%2C%2215449.jpg%22%2C%2215450.jpg%22%2C%2215451.jpg%22%2C%2215452.jpg%22%2C%2215453.jpg%22%2C%2215454.jpg%22%2C%2215455.jpg%22%2C%2215456.jpg%22%2C%2215457.jpg%22%2C%2215458.jpg%22%2C%2215459.jpg%22%2C%2215460.jpg%22%2C%2215461.jpg%22%2C%2215462.jpg%22%2C%2215463.jpg%22%2C%2215464.jpg%22%2C%2215465.jpg%22%2C%2215466.jpg%22%2C%2215467.jpg%22%2C%2215468.jpg%22%2C%2215469.jpg%22%2C%2215470.jpg%22%2C%2215471.jpg%22%2C%2215472.jpg%22%2C%2215473.jpg%22%2C%2215474.jpg%22%2C%2215475.jpg%22%2C%2215476.jpg%22%2C%2215477.jpg%22%2C%2215478.jpg%22%2C%2215479.jpg%22%2C%2215480.jpg%22%2C%2215481.jpg%22%2C%2215482.jpg%22%2C%2215483.jpg%22%2C%2215484.jpg%22%2C%2215485.jpg%22%2C%2215486.jpg%22%2C%2215487.jpg%22%2C%2215488.jpg%22%2C%2215489.jpg%22%2C%2215490.jpg%22%2C%2215491.jpg%22%2C%2215492.jpg%22%2C%2215493.jpg%22%2C%2215494.jpg%22%2C%2215495.jpg%22%2C%2215496.jpg%22%2C%2215497.jpg%22%2C%2215498.jpg%22%2C%2215499.jpg%22%2C%2215500.jpg%22%2C%2215501.jpg%22%2C%2215502.jpg%22%2C%2215503.jpg%22%2C%2215504.jpg%22%2C%2215505.jpg%22%2C%2215506.jpg%22%2C%2215507.jpg%22%2C%2215508.jpg%22%2C%2215509.jpg%22%2C%2215510.jpg%22%2C%2215511.jpg%22%2C%2215512.jpg%22%2C%2215513.jpg%22%2C%2215514.jpg%22%2C%2215515.jpg%22%2C%2215516.jpg%22%2C%2215517.jpg%22%2C%2215518.jpg%22%2C%2215519.jpg%22%2C%2215520.jpg%22%2C%2215521.jpg%22%2C%2215522.jpg%22%2C%2215523.jpg%22%2C%2215524.jpg%22%2C%2215525.jpg%22%2C%2215526.jpg%22%2C%2215527.jpg%22%2C%2215528.jpg%22%2C%2215529.jpg%22%2C%2215530.jpg%22%2C%2215531.jpg%22%2C%2215532.jpg%22%2C%2215533.jpg%22%2C%2215534.jpg%22%2C%2215535.jpg%22%2C%2215536.jpg%22%2C%2215537.jpg%22%2C%2215538.jpg%22%2C%2215539.jpg%22%2C%2215540.jpg%22%2C%2215541.jpg%22%2C%2215542.jpg%22%2C%2215543.jpg%22%2C%2215544.jpg%22%2C%2215545.jpg%22%2C%2215546.jpg%22%2C%2215547.jpg%22%2C%2215548.jpg%22%2C%2215549.jpg%22%2C%2215550.jpg%22%2C%2215551.jpg%22%2C%2215552.jpg%22%2C%2215553.jpg%22%2C%2215554.jpg%22%2C%2215555.jpg%22%2C%2215556.jpg%22%2C%2215557.jpg%22%2C%2215558.jpg%22%2C%2215559.jpg%22%2C%2215560.jpg%22%2C%2215561.jpg%22%2C%2215562.jpg%22%2C%2215563.jpg%22%2C%2215564.jpg%22%2C%2215565.jpg%22%2C%2215566.jpg%22%2C%2215567.jpg%22%2C%2215568.jpg%22%2C%2215569.jpg%22%2C%2215570.jpg%22%2C%2215571.jpg%22%2C%2215572.jpg%22%2C%2215573.jpg%22%2C%2215574.jpg%22%2C%2215575.jpg%22%2C%2215576.jpg%22%2C%2215577.jpg%22%2C%2215578.jpg%22%2C%2215579.jpg%22%2C%2215580.jpg%22%2C%2215581.jpg%22%2C%2215582.jpg%22%2C%2215583.jpg%22%2C%2215584.jpg%22%2C%2215585.jpg%22%2C%2215586.jpg%22%2C%2215587.jpg%22%2C%2215588.jpg%22%2C%2215589.jpg%22%2C%2215590.jpg%22%2C%2215591.jpg%22%2C%2215592.jpg%22%2C%2215593.jpg%22%2C%2215594.jpg%22%2C%2215595.jpg%22%2C%2215596.jpg%22%2C%2215597.jpg%22%2C%2215598.jpg%22%2C%2215599.jpg%22%2C%2215600.jpg%22%2C%2215601.jpg%22%2C%2215602.jpg%22%2C%2215603.jpg%22%2C%2215604.jpg%22%2C%2215605.jpg%22%2C%2215606.jpg%22%2C%2215607.jpg%22%2C%2215608.jpg%22%2C%2215609.jpg%22%2C%2215610.jpg%22%2C%2215611.jpg%22%2C%2215612.jpg%22%2C%2215613.jpg%22%2C%2215614.jpg%22%2C%2215615.jpg%22%2C%2215616.jpg%22%2C%2215617.jpg%22%2C%2215618.jpg%22%2C%2215619.jpg%22%2C%2215620.jpg%22%2C%2215621.jpg%22%2C%2215622.jpg%22%2C%2215623.jpg%22%2C%2215624.jpg%22%2C%2215625.jpg%22%2C%2215626.jpg%22%2C%2215627.jpg%22%2C%2215628.jpg%22%2C%2215629.jpg%22%2C%2215630.jpg%22%2C%2215631.jpg%22%2C%2215632.jpg%22%2C%2215633.jpg%22%2C%2215634.jpg%22%2C%2215635.jpg%22%2C%2215636.jpg%22%2C%2215637.jpg%22%2C%2215638.jpg%22%2C%2215639.jpg%22%2C%2215640.jpg%22%2C%2215641.jpg%22%2C%2215642.jpg%22%2C%2215643.jpg%22%2C%2215644.jpg%22%2C%2215645.jpg%22%2C%2215646.jpg%22%2C%2215647.jpg%22%2C%2215648.jpg%22%2C%2215649.jpg%22%2C%2215650.jpg%22%2C%2215651.jpg%22%2C%2215652.jpg%22%2C%2215653.jpg%22%2C%2215654.jpg%22%2C%2215655.jpg%22%2C%2215656.jpg%22%2C%2215657.jpg%22%2C%2215658.jpg%22%2C%2215659.jpg%22%2C%2215660.jpg%22%2C%2215661.jpg%22%2C%2215662.jpg%22%5D",
            filename="test9.zip",
            md5=False,
            checksum="ff4acdf34b8bdd45ca0a7281b6eacaea46fdbec33628ddfb290b8b3483b786e7",
        ),
        DataLocation(
            url="https://cloud.cs.uni-tuebingen.de/public.php/dav/files/ZZxX65FGnQ8zjBP/Compressed%20Version/annotations/instances_val.json",
            filename="instances_val.json",
            md5=False,
            checksum="9e965c832263e6b9b9dd299637833e5b93924ecf9a1780b16fbd7c53e4a63f5e",
        ),
        DataLocation(
            url="https://cloud.cs.uni-tuebingen.de/public.php/dav/files/ZZxX65FGnQ8zjBP/Compressed%20Version/annotations/instances_train.json",
            filename="instances_train.json",
            md5=False,
            checksum="5706b042f6b6c50267ed06076b8c27594ff4da76569edbd99ed53ed8bfa1db95",
        ),
        DataLocation(
            url="https://cloud.cs.uni-tuebingen.de/public.php/dav/files/ZZxX65FGnQ8zjBP/Uncompressed%20Version/annotations/instances_test_nogt.json",
            filename="instances_test_nogt.json",
            md5=False,
            checksum="d9ee6d4c4da466ac3ab7555807ff8573cb3e7302e4fd2d218d7e5391c09514e5",
        ),
    ]

    index2label: dict[int, str] = {
        0: "not_used",
        1: "swimmer",
        2: "boat",
        3: "jetski",
        4: "life_saving_appliances",
        5: "buoy",
    }

    def __init__(
        self,
        root: str | Path,
        image_set: Literal["train", "val", "test", "base"] = "train",
        transforms: NumpyObjectDetectionTransform | Sequence[NumpyObjectDetectionTransform] | None = None,
        download: bool = False,
        verbose: bool = False,
    ) -> None:
        super().__init__(
            root,
            image_set,
            transforms,
            download,
            verbose,
        )

    def _create_structure(self) -> None:
        # Create file structure
        annotation_path = self.path / "annotations"
        annotation_path.mkdir(exist_ok=True)
        images_path = self.path / "images"
        images_path.mkdir(exist_ok=True)
        train_path = images_path / "train"
        train_path.mkdir(exist_ok=True)
        val_path = images_path / "val"
        val_path.mkdir(exist_ok=True)
        test_path = images_path / "test"
        test_path.mkdir(exist_ok=True)

    def _check_annotations(self) -> None:
        for idx in range(1, 4):
            annotation_path = self.path / "annotations"
            if not (annotation_path / self._resources[-idx].filename).exists():
                _ensure_exists(
                    *self._resources[-idx],
                    annotation_path,
                    self._root,
                    self._download,
                    self._verbose,
                )

    def _check_images(self, resource_indices: list[int], download_path: Path) -> None:
        if not list(download_path.glob("*.jpg")):
            for idx in resource_indices:
                _ensure_exists(
                    *self._resources[idx],
                    download_path,
                    self._root,
                    self._download,
                    self._verbose,
                )

    def _load_data(
        self,
    ) -> tuple[list[str], list[tuple[list[int], list[list[float]]]], dict[str, list[Any]]]:
        image_sets: dict[str, list[int]] = {
            "train": list(range(20)),
            "val": list(range(20, 24)),
            "test": list(range(24, len(self._resources) - 3)),
        }

        self._create_structure()
        self._check_annotations()

        # Load the data
        if self.image_set != "base":
            download_path = self.path / "images" / self.image_set
            self._check_images(image_sets[self.image_set], download_path)
        else:
            for folder in ["train", "val", "test"]:
                download_path = self.path / "images" / folder
                self._check_images(image_sets[folder], download_path)

        filepaths, targets, datum_metadata = super()._load_data()

        return filepaths, list(targets), datum_metadata

    def _load_images(self, data_folder: Path, file_data: dict[int, dict[str, Any]]) -> dict[int, dict[str, Any]]:
        for entry in data_folder.iterdir():
            if entry.is_file() and entry.suffix == ".jpg":
                if int(entry.stem) not in file_data:
                    file_data[int(entry.stem)] = {}
                file_data[int(entry.stem)]["data_path"] = str(entry)

        return file_data

    def _create_per_image_annotations(
        self, annotation_file: Path, file_data: dict[int, dict[str, Any]]
    ) -> dict[int, dict[str, Any]]:
        mappings = {
            "longitude": (["longitude", "gps_longitude_ref"], -1),
            "latitude": (["latitude", "gps_latitude_ref"], -1),
            "altitude": (["altitude", "height_above_takeoff(meter)"], -1),
            "gimbal_pitch": (["gimbal_pitch", "gimbal_pitch(degrees)"], -1),
            "compass_heading": (["compass_heading", "compass_heading(degrees)"], -1),
            "gimbal_heading": (["gimbal_heading", "gimbal_heading(degrees)"], -1),
            "speed": (["speed", "speed(m/s)"], -1),
            "xspeed": (["xspeed", "xspeed(m/s)"], -1),
            "yspeed": (["yspeed", "yspeed(m/s)"], -1),
            "zspeed": (["zspeed", "zspeed(m/s)"], -1),
        }

        with open(annotation_file) as af:
            result = json.load(af)

        # Retrieve the metadata
        for file_meta in result["images"]:
            current_id = file_meta["id"]

            # Skip if file_id was not in file list
            if current_id not in file_data:
                continue

            meta = file_meta["meta"] if file_meta["meta"] is not None else {}
            source = file_meta["source"]
            if not isinstance(source, dict):
                source = {"drone": source}
            current_file = file_data[current_id]

            # Basic file metadata
            current_file["height"] = file_meta["height"]
            current_file["width"] = file_meta["width"]
            current_file["drone"] = source["drone"]
            current_file["storage"] = source.get("folder_name", "")

            # Handle non-standard file metadata
            current_file["date_time"] = file_meta.get("date_time") or meta.get("date_time") or ""
            if "frame" in file_meta:
                frame = file_meta["frame"][:-4]
                current_file["frame"] = int(frame.split("_")[-1]) if "IMG_" in frame else int(frame[3:])
            elif "frame_no" in source:
                current_file["frame"] = source["frame_no"]
            else:
                current_file["frame"] = -1

            # Grab additional metadata if available
            for output_key, (possible_keys, default) in mappings.items():
                current_file[output_key] = next((meta.get(key) for key in possible_keys if key in meta), default)

        # Retrieve the label and bounding box
        for annotation in result["annotations"]:
            current_id = annotation["image_id"]

            if current_id not in file_data:
                continue

            current_file = file_data[current_id]

            if "label" not in current_file:
                current_file["label"] = []
                current_file["box"] = []
                current_file["object_id"] = []
                current_file["object_size"] = []

            current_file["label"].append(annotation["category_id"])
            current_file["box"].append([float(val) for val in annotation["bbox"]])
            current_file["object_id"].append(annotation["id"])
            current_file["object_size"].append(annotation["area"])

        return file_data

    def _restructure_file_data(self, file_data: dict[int, dict[str, Any]]) -> dict[str, list[Any]]:
        """Restructure file_data from dictionary of dictionaries to a dictionary of lists"""
        # Get the keys from the dictionary
        all_keys = set()
        for file_dict in file_data.values():
            all_keys.update(file_dict.keys())
        all_keys.discard("label")
        all_keys.discard("box")

        # Initialize the restructured dictionary
        restructured_data = {key: [] for key in all_keys}
        restructured_data["image_id"] = []
        restructured_data["label_box"] = []

        # Create the lists
        for file_id, file_dict in file_data.items():
            restructured_data["image_id"].append(file_id)
            restructured_data["label_box"].append((file_dict.get("label", []), file_dict.get("box", [])))
            for key in all_keys:
                restructured_data[key].append(file_dict.get(key, None))

        return restructured_data

    def _load_data_inner(
        self,
    ) -> tuple[list[str], list[tuple[list[int], list[list[float]]]], dict[str, Any]]:
        file_data = {}
        if self.image_set != "base":
            data_folder = self.path / "images" / self.image_set
            file_data = self._load_images(data_folder, file_data)
            json_name = self.image_set
            if json_name == "test":
                json_name += "_nogt"
            annotation_file = self.path / "annotations" / f"instances_{json_name}.json"
            file_data = self._create_per_image_annotations(annotation_file, file_data)
        else:
            for folder in ["train", "val", "test"]:
                data_folder = self.path / "images" / folder
                file_data = self._load_images(data_folder, file_data)
                json_name = folder
                if json_name == "test":
                    json_name += "_nogt"
                annotation_file = self.path / "annotations" / f"instances_{json_name}.json"
                file_data = self._create_per_image_annotations(annotation_file, file_data)

        meta_data = self._restructure_file_data(file_data)
        data = meta_data.pop("data_path")
        annotations = meta_data.pop("label_box")

        return data, annotations, meta_data

    def _read_annotations(
        self, annotation: tuple[list[int], list[list[float]]]
    ) -> tuple[list[list[float]], list[int], dict[str, Any]]:
        """Function for extracting the info for the label and boxes"""
        labels: list[int] = annotation[0]
        boxes: list[list[float]] = annotation[1]

        def xywh_to_xyxy(xywh: list[float]) -> list[float]:
            return [xywh[0], xywh[1], xywh[0] + xywh[2], xywh[1] + xywh[3]]

        xyxy = [xywh_to_xyxy(box) for box in boxes]

        return xyxy, labels, {}
