"""Utilities for vasp files"""

from typing import Optional

from pypolymlp.core.data_format import PolymlpStructure
from pypolymlp.utils.yaml_utils import save_cell


def print_poscar(st: PolymlpStructure):
    """Print structure in POSCAR format."""
    if st.comment is not None:
        print(st.comment)
    else:
        print("Generated by pypolymlp")

    print("1.0")
    for n in st.axis.T:
        print(
            "  ",
            "{0:15.15f}".format(n[0]),
            "{0:15.15f}".format(n[1]),
            "{0:15.15f}".format(n[2]),
        )
    index = 0
    print(" ", end="")
    for n in st.n_atoms:
        print(st.elements[index], " ", end="")
        index += n
    print()
    print(" ", end="")
    for n in st.n_atoms:
        print(n, " ", end="")
    print()
    print("Direct")
    for n in st.positions.T:
        print(
            "  ",
            "{0:15.15f}".format(n[0]),
            "{0:15.15f}".format(n[1]),
            "{0:15.15f}".format(n[2]),
        )


def write_poscar_file(
    st: PolymlpStructure,
    filename: str = "poscar_pypolymlp",
    header: Optional[str] = None,
):
    """Write structure in POSCAR file."""
    f = open(filename, "w")

    if header is not None:
        print(header, file=f)
    else:
        if st.comment is not None:
            print(st.comment, file=f)
        else:
            print("Generated by pypolymlp", file=f)

    print("1.0", file=f)
    for n in st.axis.T:
        print(
            "  ",
            "{0:15.15f}".format(n[0]),
            "{0:15.15f}".format(n[1]),
            "{0:15.15f}".format(n[2]),
            file=f,
        )
    index = 0
    print(" ", end="", file=f)
    for n in st.n_atoms:
        if n > 0:
            print(st.elements[index], " ", end="", file=f)
            index += n
        else:
            print("Z  ", end="", file=f)
    print("", file=f)
    print(" ", end="", file=f)
    for n in st.n_atoms:
        print(n, " ", end="", file=f)
    print("", file=f)
    print("Direct", file=f)
    for n in st.positions.T:
        print(
            "  ",
            "{0:15.15f}".format(float(n[0])),
            "{0:15.15f}".format(float(n[1])),
            "{0:15.15f}".format(float(n[2])),
            file=f,
        )

    f.close()


def load_electronic_properties_from_vasprun(
    input_filename: str = "vasprun.xml",
    output_filename: Optional[str] = None,
    temp_max: float = 1000.0,
    temp_step: float = 10.0,
):
    """Load electronic properties from vasprun.xml.

    phonopy is required.
    """
    from phonopy.interface.vasp import parse_vasprunxml

    from pypolymlp.utils.electron import ElectronProperties
    from pypolymlp.utils.phonopy_utils import phonopy_cell_to_structure

    vxml = parse_vasprunxml(input_filename)
    st = phonopy_cell_to_structure(vxml.cell)

    weights = vxml.k_weights
    eigenvalues = vxml.eigenvalues[:, :, :, 0]
    n_electrons = vxml.NELECT
    el = ElectronProperties(eigenvalues, weights, n_electrons)
    el.run(temp_max, temp_step)

    if output_filename is not None:
        with open(output_filename, mode="w") as f:
            save_cell(st, tag="structure", file=f)
            print("unit:", file=f)
            print("  free_energy   : eV/cell", file=f)
            print("  energy        : eV/cell", file=f)
            print("  entropy       : eV/K/cell", file=f)
            print("  specific_heat : eV/K/cell", file=f)
            print("", file=f)

            print("properties:", file=f)
            for t, e, s, helm, sh in zip(
                el.temperatures,
                el.energy,
                el.entropy,
                el.free_energy,
                el.cv,
            ):
                print("- temperature:  ", t, file=f)
                print("  free_energy:  ", helm, file=f)
                print("  energy:       ", e, file=f)
                print("  entropy:      ", s, file=f)
                print("  specific_heat:", sh, file=f)
                print("", file=f)

    return (st, el)
