# Structure Complète du Projet GestVenv v1.1

## 📁 Arborescence Générale

```
gestvenv/                                   # Racine du projet
├── pyproject.toml                          # Configuration moderne du projet
├── setup.py                               # Compatibilité installation
├── README.md                              # Documentation principale
├── LICENSE                                # Licence MIT
├── CHANGELOG.md                           # Historique des versions
├── .gitignore                             # Fichiers ignorés par Git
├── .pre-commit-config.yaml               # Configuration pre-commit
│
├── gestvenv/                              # Package principal
│   ├── __init__.py                       # Version et exports
│   ├── __version__.py                    # Gestion de version
│   ├── cli.py                            # Interface ligne de commande
│   │
│   ├── core/                             # Modules centraux
│   │   ├── __init__.py
│   │   ├── models.py                     # Modèles de données
│   │   ├── environment_manager.py        # Gestionnaire environnements
│   │   ├── config_manager.py             # Gestionnaire configuration
│   │   └── exceptions.py                 # Exceptions personnalisées
│   │
│   ├── services/                         # Services métier
│   │   ├── __init__.py
│   │   ├── package_service.py            # Service packages
│   │   ├── cache_service.py              # Service cache
│   │   ├── migration_service.py          # Service migration
│   │   ├── system_service.py             # Service système
│   │   ├── diagnostic_service.py         # Service diagnostic
│   │   └── template_service.py           # Service templates
│   │
│   ├── backends/                         # Backends modulaires
│   │   ├── __init__.py
│   │   ├── base.py                       # Interface abstraite
│   │   ├── pip_backend.py                # Backend pip
│   │   ├── uv_backend.py                 # Backend uv
│   │   ├── poetry_backend.py             # Backend Poetry (futur)
│   │   ├── pdm_backend.py                # Backend PDM (futur)
│   │   └── backend_manager.py            # Gestionnaire backends
│   │
│   ├── utils/                            # Utilitaires
│   │   ├── __init__.py
│   │   ├── toml_handler.py               # Gestionnaire TOML
│   │   ├── pyproject_parser.py           # Parser pyproject.toml
│   │   ├── validation.py                 # Utilitaires validation
│   │   ├── path_utils.py                 # Utilitaires chemins
│   │   ├── security.py                   # Utilitaires sécurité
│   │   └── performance.py                # Monitoring performance
│   │
│   └── templates/                        # Templates projets
│       ├── __init__.py
│       ├── base_template.py              # Template de base
│       ├── builtin/                      # Templates intégrés
│       │   ├── __init__.py
│       │   ├── basic.py                  # Template basique
│       │   ├── web.py                    # Template web
│       │   ├── data_science.py           # Template data science
│       │   ├── cli.py                    # Template CLI
│       │   ├── fastapi.py                # Template FastAPI
│       │   ├── flask.py                  # Template Flask
│       │   └── django.py                 # Template Django
│       └── user/                         # Templates utilisateur
│
├── docs/                                 # Documentation
│   ├── index.md                         # Page d'accueil docs
│   ├── installation.md                  # Guide installation
│   ├── quickstart.md                    # Démarrage rapide
│   ├── user_guide/                      # Guide utilisateur
│   │   ├── basics.md                    # Utilisation basique
│   │   ├── pyproject_support.md         # Support pyproject.toml
│   │   ├── backends.md                  # Gestion backends
│   │   ├── cache_offline.md             # Cache et mode hors ligne
│   │   ├── templates.md                 # Templates projets
│   │   ├── migration.md                 # Guide migration
│   │   └── troubleshooting.md           # Résolution problèmes
│   ├── api/                             # Documentation API
│   │   ├── core.md                      # API core
│   │   ├── services.md                  # API services
│   │   └── backends.md                  # API backends
│   ├── development/                     # Guide développeur
│   │   ├── contributing.md              # Guide contribution
│   │   ├── architecture.md              # Architecture
│   │   ├── testing.md                   # Guide tests
│   │   └── release.md                   # Processus release
│   └── examples/                        # Exemples d'utilisation
│
├── scripts/                             # Scripts utilitaires
│   ├── install_dev.py                   # Installation développement
│   ├── run_tests.py                     # Lancement tests
│   ├── build_release.py                 # Construction release
│   ├── generate_docs.py                 # Génération documentation
│   └── performance_benchmark.py         # Benchmarks performance
│
├── benchmarks/                          # Tests de performance
│   ├── __init__.py
│   ├── benchmark_backends.py            # Benchmark backends
│   ├── benchmark_cache.py               # Benchmark cache
│   ├── benchmark_parsing.py             # Benchmark parsing
│   └── results/                         # Résultats benchmarks
│
├── examples/                            # Exemples d'utilisation
│   ├── basic_usage/                     # Utilisation basique
│   │   ├── simple_project/
│   │   └── requirements_migration/
│   ├── pyproject_examples/              # Exemples pyproject.toml
│   │   ├── web_app/
│   │   ├── data_science/
│   │   └── cli_tool/
│   ├── advanced_workflows/              # Workflows avancés
│   │   ├── multi_environment/
│   │   ├── ci_cd_integration/
│   │   └── team_collaboration/
│   └── migration_scenarios/             # Scénarios migration
│       ├── v1_to_v1.1/
│       ├── requirements_to_pyproject/
│       └── poetry_migration/
│
└── .github/                             # Configuration GitHub
    ├── workflows/                       # GitHub Actions
    │   ├── ci.yml                      # Pipeline CI/CD
    │   ├── release.yml                 # Release automatique
    │   └── security.yml                # Audit sécurité
    ├── ISSUE_TEMPLATE/                  # Templates d'issues
    │   ├── bug_report.md
    │   ├── feature_request.md
    │   └── performance_issue.md
    └── pull_request_template.md         # Template PR
```

## 📦 Configuration du Projet

### pyproject.toml

```toml
[project]
name = "gestvenv"
version = "1.1.0"
description = "Gestionnaire d'environnements virtuels Python moderne"
authors = [
    {name = "GestVenv Team", email = "contact@gestvenv.dev"}
]
readme = "README.md"
license = {text = "MIT"}
requires-python = ">=3.9"
keywords = ["python", "virtualenv", "package-manager", "environment", "pyproject"]
classifiers = [
    "Development Status :: 5 - Production/Stable",
    "Intended Audience :: Developers",
    "License :: OSI Approved :: MIT License",
    "Operating System :: OS Independent",
    "Programming Language :: Python :: 3",
    "Programming Language :: Python :: 3.9",
    "Programming Language :: Python :: 3.10", 
    "Programming Language :: Python :: 3.11",
    "Programming Language :: Python :: 3.12",
    "Programming Language :: Python :: 3.13",
    "Topic :: Software Development :: Libraries :: Python Modules",
    "Topic :: System :: Software Distribution",
    "Topic :: Software Development :: Build Tools",
]

dependencies = [
    "packaging>=21.0",
    "tomli>=2.0.0; python_version<'3.11'",
    "tomlkit>=0.11.0",
    "click>=8.0.0",
    "rich>=13.0.0",
]

[project.optional-dependencies]
dev = [
    "pytest>=7.0",
    "pytest-cov>=4.0",
    "pytest-mock>=3.6",
    "pytest-asyncio>=0.21",
    "black>=23.0",
    "isort>=5.10",
    "mypy>=1.0",
    "flake8>=6.0",
    "pre-commit>=3.0",
]
docs = [
    "sphinx>=6.0",
    "sphinx-rtd-theme>=1.0",
    "myst-parser>=1.0",
    "sphinx-click>=4.0",
]
performance = [
    "uv>=0.1.0",
]
full = [
    "uv>=0.1.0",
    "tomli-w>=1.0.0",
    "psutil>=5.9.0",
]

[project.scripts]
gestvenv = "gestvenv.cli:main"

[project.urls]
Homepage = "https://github.com/gestvenv/gestvenv"
Documentation = "https://gestvenv.readthedocs.io"
Repository = "https://github.com/gestvenv/gestvenv.git"
Changelog = "https://github.com/gestvenv/gestvenv/blob/main/CHANGELOG.md"
Issues = "https://github.com/gestvenv/gestvenv/issues"

[build-system]
requires = ["setuptools>=68.0", "wheel", "setuptools_scm[toml]>=6.2"]
build-backend = "setuptools.build_meta"

[tool.setuptools_scm]
write_to = "gestvenv/__version__.py"

[tool.black]
line-length = 88
target-version = ['py39']
include = '\.pyi?$'
extend-exclude = '''
/(
  \.eggs
  | \.git
  | \.mypy_cache
  | \.tox
  | \.venv
  | build
  | dist
)/
'''

[tool.isort]
profile = "black"
multi_line_output = 3
line_length = 88
known_first_party = ["gestvenv"]

[tool.mypy]
python_version = "3.9"
warn_return_any = true
warn_unused_configs = true
disallow_untyped_defs = true
disallow_incomplete_defs = true
check_untyped_defs = true
disallow_untyped_decorators = true
no_implicit_optional = true
warn_redundant_casts = true
warn_unused_ignores = true
warn_no_return = true
warn_unreachable = true

[tool.coverage.run]
source = ["gestvenv"]
omit = [
    "gestvenv/__version__.py",
    "*/tests/*",
    "*/test_*.py",
]

[tool.coverage.report]
exclude_lines = [
    "pragma: no cover",
    "def __repr__",
    "raise AssertionError",
    "raise NotImplementedError",
    "if __name__ == .__main__.:",
    "if TYPE_CHECKING:",
]

[tool.flake8]
max-line-length = 88
extend-ignore = ["E203", "W503"]
exclude = [
    ".git",
    "__pycache__",
    ".mypy_cache",
    ".pytest_cache",
    ".venv",
    "build",
    "dist",
]
```

## 🔧 Fichiers de Configuration

### .gitignore

```gitignore
# Byte-compiled / optimized / DLL files
__pycache__/
*.py[cod]
*$py.class

# C extensions
*.so

# Distribution / packaging
.Python
build/
develop-eggs/
dist/
downloads/
eggs/
.eggs/
lib/
lib64/
parts/
sdist/
var/
wheels/
share/python-wheels/
*.egg-info/
.installed.cfg
*.egg
MANIFEST

# PyInstaller
*.manifest
*.spec

# Installer logs
pip-log.txt
pip-delete-this-directory.txt

# Unit test / coverage reports
htmlcov/
.tox/
.nox/
.coverage
.coverage.*
.cache
nosetests.xml
coverage.xml
*.cover
*.py,cover
.hypothesis/
.pytest_cache/
cover/

# Translations
*.mo
*.pot

# Django stuff:
*.log
local_settings.py
db.sqlite3
db.sqlite3-journal

# Flask stuff:
instance/
.webassets-cache

# Scrapy stuff:
.scrapy

# Sphinx documentation
docs/_build/

# PyBuilder
.pybuilder/
target/

# Jupyter Notebook
.ipynb_checkpoints

# IPython
profile_default/
ipython_config.py

# pyenv
.python-version

# pipenv
Pipfile.lock

# poetry
poetry.lock

# pdm
.pdm.toml

# PEP 582
__pypackages__/

# Celery stuff
celerybeat-schedule
celerybeat.pid

# SageMath parsed files
*.sage.py

# Environments
.env
.venv
env/
venv/
ENV/
env.bak/
venv.bak/

# Spyder project settings
.spyderproject
.spyproject

# Rope project settings
.ropeproject

# mkdocs documentation
/site

# mypy
.mypy_cache/
.dmypy.json
dmypy.json

# Pyre type checker
.pyre/

# pytype static type analyzer
.pytype/

# Cython debug symbols
cython_debug/

# IDE
.vscode/
.idea/
*.swp
*.swo

# GestVenv specific
.gestvenv/
.gestvenv_v1/
*.gestvenv-metadata.json
gestvenv-cache/
performance_results/
benchmark_results/

# OS
.DS_Store
Thumbs.db
```

### .pre-commit-config.yaml

```yaml
repos:
  - repo: https://github.com/pre-commit/pre-commit-hooks
    rev: v4.4.0
    hooks:
      - id: trailing-whitespace
      - id: end-of-file-fixer
      - id: check-yaml
      - id: check-added-large-files
      - id: check-merge-conflict
      - id: debug-statements

  - repo: https://github.com/psf/black
    rev: 23.3.0
    hooks:
      - id: black
        language_version: python3.11

  - repo: https://github.com/pycqa/isort
    rev: 5.12.0
    hooks:
      - id: isort
        args: ["--profile", "black"]

  - repo: https://github.com/pycqa/flake8
    rev: 6.0.0
    hooks:
      - id: flake8

  - repo: https://github.com/pre-commit/mirrors-mypy
    rev: v1.3.0
    hooks:
      - id: mypy
        additional_dependencies: [types-all]
        exclude: ^(docs|examples)/

  - repo: https://github.com/pycqa/bandit
    rev: 1.7.5
    hooks:
      - id: bandit
        args: ["-c", "pyproject.toml"]
        additional_dependencies: ["bandit[toml]"]
```

## 📋 Documentation

### README.md

```markdown
# GestVenv v1.1 🐍

**Gestionnaire d'environnements virtuels Python moderne et performant**

[![PyPI version](https://badge.fury.io/py/gestvenv.svg)](https://badge.fury.io/py/gestvenv)
[![Python Support](https://img.shields.io/pypi/pyversions/gestvenv.svg)](https://pypi.org/project/gestvenv/)
[![Tests](https://github.com/gestvenv/gestvenv/workflows/Tests/badge.svg)](https://github.com/gestvenv/gestvenv/actions)
[![Coverage](https://codecov.io/gh/gestvenv/gestvenv/branch/main/graph/badge.svg)](https://codecov.io/gh/gestvenv/gestvenv)
[![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg)](https://opensource.org/licenses/MIT)

## 🚀 Fonctionnalités

- **🔥 Performance** : 10x plus rapide avec le backend uv
- **📦 Support pyproject.toml** : Conformité complète PEP 621
- **🎯 Backends modulaires** : pip, uv, poetry, pdm
- **💾 Cache intelligent** : Mode hors ligne efficace
- **🔧 Diagnostic automatique** : Détection et réparation des problèmes
- **📋 Templates intégrés** : Démarrage rapide pour tous projets
- **🔄 Migration transparente** : Compatible 100% avec v1.0

## ⚡ Installation Rapide

```bash
# Installation standard
pip install gestvenv

# Installation avec performances optimisées
pip install gestvenv[performance]

# Installation complète
pip install gestvenv[full]
```

## 🎯 Utilisation

### Création d'environnements

```bash
# Environnement basique
gestvenv create monapp

# Depuis pyproject.toml
gestvenv create-from-pyproject ./pyproject.toml monapp

# Avec template
gestvenv create-from-template web monwebapp
```

### Gestion des packages

```bash
# Installation
gestvenv install requests flask --env monapp

# Installation avec groupes
gestvenv install --group dev --env monapp

# Synchronisation
gestvenv sync monapp
```

### Cache et mode hors ligne

```bash
# Pré-téléchargement
gestvenv cache add numpy pandas matplotlib

# Installation hors ligne
gestvenv --offline install requests
```

## 📚 Documentation

- [Guide d'installation](docs/installation.md)
- [Démarrage rapide](docs/quickstart.md) 
- [Guide utilisateur](docs/user_guide/)
- [Migration v1.0 → v1.1](docs/user_guide/migration.md)
- [Documentation API](docs/api/)

## 🤝 Contribution

Les contributions sont les bienvenues ! Consultez le [guide de contribution](docs/development/contributing.md).

## 📄 Licence

Ce projet est sous licence MIT. Voir [LICENSE](LICENSE) pour plus de détails.

## 🙏 Remerciements

- Communauté Python pour les standards PEP
- Équipes uv, poetry, PDM pour l'inspiration
- Tous les contributeurs et utilisateurs
```

### CHANGELOG.md

```markdown
# Changelog

Toutes les modifications notables de ce projet seront documentées dans ce fichier.

Le format est basé sur [Keep a Changelog](https://keepachangelog.com/en/1.0.0/),
et ce projet adhère au [Semantic Versioning](https://semver.org/spec/v2.0.0.html).

## [1.1.0] - 2025-01-27

### ✨ Nouvelles fonctionnalités

- **Support pyproject.toml natif** : Conformité complète PEP 621
- **Backend uv** : Performance 10x supérieure avec fallback pip automatique
- **Architecture multi-backend** : Support extensible pip/uv/poetry/pdm
- **Cache intelligent** : Mode hors ligne avec compression et LRU
- **Service de diagnostic** : Détection automatique et réparation des problèmes
- **Templates de projets** : Templates intégrés (web, data science, CLI)
- **Migration automatique** : Transition transparente v1.0 → v1.1

### 🔧 Améliorations

- Interface CLI moderne avec sous-commandes intuitives
- Validation de sécurité renforcée
- Monitoring de performance intégré
- Messages d'erreur plus informatifs
- Support émojis dans l'interface

### 🐛 Corrections

- Gestion robuste des erreurs de réseau
- Correction des permissions sur Windows
- Amélioration de la détection des versions Python
- Fix de la gestion des dépendances circulaires

### 🔄 Migration

- Migration automatique des environnements v1.0
- Conversion assistée requirements.txt → pyproject.toml
- Préservation totale de la compatibilité ascendante

### ⚠️ Changements

- **AUCUN BREAKING CHANGE** : Compatibilité 100% avec v1.0
- Nouveau répertoire de configuration : `~/.gestvenv/` (migration auto)
- Cache reorganisé par backend pour optimisation

## [1.0.0] - 2024-12-15

### ✨ Version initiale

- Gestion d'environnements virtuels basique
- Support requirements.txt
- Backend pip uniquement
- Interface CLI simple
- Export/import JSON

---

**Note** : Les versions pre-1.0 sont considérées comme expérimentales et ne sont pas documentées ici.
```

## 🚀 Scripts d'Automatisation

### scripts/install_dev.py

```python
#!/usr/bin/env python3
"""Script d'installation pour développement"""

import subprocess
import sys
from pathlib import Path

def install_dev():
    """Installation complète environnement de développement"""
    
    print("🔧 Installation GestVenv en mode développement...")
    
    # Installation package en mode éditable
    subprocess.run([sys.executable, "-m", "pip", "install", "-e", ".[dev,docs,full]"])
    
    # Installation pre-commit hooks
    print("🪝 Installation pre-commit hooks...")
    subprocess.run(["pre-commit", "install"])
    
    # Installation uv si pas présent
    try:
        subprocess.run(["uv", "--version"], check=True, capture_output=True)
        print("✅ uv déjà installé")
    except (subprocess.CalledProcessError, FileNotFoundError):
        print("📦 Installation uv...")
        subprocess.run([sys.executable, "-m", "pip", "install", "uv"])
    
    print("🎉 Installation développement terminée!")
    print("\n📋 Commandes utiles:")
    print("  pytest                    # Exécuter les tests")
    print("  black .                   # Formatter le code")
    print("  mypy gestvenv/            # Vérifier les types")
    print("  pre-commit run --all-files # Vérifier tous les hooks")

if __name__ == "__main__":
    install_dev()
```

### scripts/build_release.py

```python
#!/usr/bin/env python3
"""Script de build pour release"""

import subprocess
import sys
import shutil
from pathlib import Path

def build_release():
    """Build complet pour release"""
    
    print("🏗️  Build release GestVenv...")
    
    # Nettoyage
    print("🧹 Nettoyage...")
    for path in ["build", "dist", "*.egg-info"]:
        if Path(path).exists():
            shutil.rmtree(path, ignore_errors=True)
    
    # Tests
    print("🧪 Exécution tests...")
    result = subprocess.run(["python", "-m", "pytest", "--cov=gestvenv", "--cov-fail-under=85"])
    if result.returncode != 0:
        print("❌ Tests échoués")
        return False
    
    # Linting
    print("🔍 Vérifications qualité...")
    checks = [
        ["black", "--check", "."],
        ["isort", "--check-only", "."],
        ["mypy", "gestvenv/"],
        ["flake8", "gestvenv/"]
    ]
    
    for check in checks:
        result = subprocess.run(check)
        if result.returncode != 0:
            print(f"❌ Échec: {' '.join(check)}")
            return False
    
    # Build
    print("📦 Construction package...")
    subprocess.run([sys.executable, "-m", "build"])
    
    # Vérification
    print("✅ Vérification package...")
    subprocess.run(["twine", "check", "dist/*"])
    
    print("🎉 Release prête dans dist/")
    return True

if __name__ == "__main__":
    success = build_release()
    sys.exit(0 if success else 1)
```
