"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NIST80053R5Checks = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const aws_cdk_lib_1 = require("aws-cdk-lib");
const nag_pack_1 = require("../nag-pack");
const nag_rules_1 = require("../nag-rules");
const apigw_1 = require("../rules/apigw");
const autoscaling_1 = require("../rules/autoscaling");
const cloudtrail_1 = require("../rules/cloudtrail");
const cloudwatch_1 = require("../rules/cloudwatch");
const dms_1 = require("../rules/dms");
const dynamodb_1 = require("../rules/dynamodb");
const ec2_1 = require("../rules/ec2");
const ecs_1 = require("../rules/ecs");
const efs_1 = require("../rules/efs");
const elasticache_1 = require("../rules/elasticache");
const elasticbeanstalk_1 = require("../rules/elasticbeanstalk");
const elb_1 = require("../rules/elb");
const iam_1 = require("../rules/iam");
const kms_1 = require("../rules/kms");
const lambda_1 = require("../rules/lambda");
const opensearch_1 = require("../rules/opensearch");
const rds_1 = require("../rules/rds");
const redshift_1 = require("../rules/redshift");
const s3_1 = require("../rules/s3");
const sagemaker_1 = require("../rules/sagemaker");
const secretsmanager_1 = require("../rules/secretsmanager");
const sns_1 = require("../rules/sns");
const vpc_1 = require("../rules/vpc");
const waf_1 = require("../rules/waf");
/**
 * Check for NIST 800-53 rev 5 compliance.
 * Based on the NIST 800-53 rev 5 AWS operational best practices: https://docs.aws.amazon.com/config/latest/developerguide/operational-best-practices-for-nist-800-53_rev_5.html
 */
class NIST80053R5Checks extends nag_pack_1.NagPack {
    constructor(props) {
        super(props);
        this.packName = 'NIST.800.53.R5';
    }
    visit(node) {
        if (node instanceof aws_cdk_lib_1.CfnResource) {
            this.checkAPIGW(node);
            this.checkAutoScaling(node);
            this.checkCloudTrail(node);
            this.checkCloudWatch(node);
            this.checkDMS(node);
            this.checkDynamoDB(node);
            this.checkEC2(node);
            this.checkECS(node);
            this.checkEFS(node);
            this.checkElastiCache(node);
            this.checkElasticBeanstalk(node);
            this.checkELB(node);
            this.checkIAM(node);
            this.checkKMS(node);
            this.checkLambda(node);
            this.checkOpenSearch(node);
            this.checkRDS(node);
            this.checkRedshift(node);
            this.checkS3(node);
            this.checkSageMaker(node);
            this.checkSecretsManager(node);
            this.checkSNS(node);
            this.checkVPC(node);
            this.checkWAF(node);
        }
    }
    /**
     * Check API Gateway Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkAPIGW(node) {
        this.applyRule({
            info: 'The REST API stage is not associated with AWS WAFv2 web ACL - (Control ID: AC-4(21)).',
            explanation: 'AWS WAF enables you to configure a set of rules (called a web access control list (web ACL)) that allow, block, or count web requests based on customizable web security rules and conditions that you define. Ensure your Amazon API Gateway stage is associated with a WAF Web ACL to protect it from malicious attacks.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: apigw_1.APIGWAssociatedWithWAF,
            node: node,
        });
        this.applyRule({
            info: 'The API Gateway stage does not have caching enabled and encrypted for all methods - (Control IDs: AU-9(3), CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)).',
            explanation: "To help protect data at rest, ensure encryption is enabled for your API Gateway stage's cache. Because sensitive data can be captured for the API method, enable encryption at rest to help protect that data.",
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: apigw_1.APIGWCacheEnabledAndEncrypted,
            node: node,
        });
        this.applyRule({
            info: 'The API Gateway stage does not have execution logging enabled for all methods - (Control IDs: AC-4(26), AU-2b, AU-3a, AU-3b, AU-3c, AU-3d, AU-3e, AU-3f, AU-6(3), AU-6(4), AU-6(6), AU-6(9), AU-8b, AU-10, AU-12a, AU-12c, AU-12(1), AU-12(2), AU-12(3), AU-12(4), AU-14a, AU-14b, AU-14b, AU-14(3), CA-7b, CM-5(1)(b), IA-3(3)(b), MA-4(1)(a), PM-14a.1, PM-14b, PM-31, SC-7(9)(b), SI-4(17), SI-7(8)).',
            explanation: 'API Gateway logging displays detailed views of users who accessed the API and the way they accessed the API. This insight enables visibility of user activities.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: apigw_1.APIGWExecutionLoggingEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The API Gateway REST API stage is not configured with SSL certificates - (Control IDs: AC-4, AC-4(22), AC-17(2), AC-24(1), AU-9(3), CA-9b, IA-5(1)(c), PM-17b, SC-7(4)(b), SC-7(4)(g), SC-8, SC-8(1), SC-8(2), SC-8(3), SC-8(4), SC-8(5), SC-13a, SC-23, SI-1a.2, SI-1a.2, SI-1c.2).',
            explanation: 'Ensure Amazon API Gateway REST API stages are configured with SSL certificates to allow backend systems to authenticate that requests originate from API Gateway.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: apigw_1.APIGWSSLEnabled,
            node: node,
        });
    }
    /**
     * Check Auto Scaling Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkAutoScaling(node) {
        this.applyRule({
            info: 'The Auto Scaling group (which is associated with a load balancer) does not utilize ELB health checks - (Control IDs: AU-12(3), AU-14a, AU-14b, CA-2(2), CA-7, CA-7b, CM-6a, CM-9b, PM-14a.1, PM-14b, PM-31, SC-6, SC-36(1)(a), SI-2a).',
            explanation: 'The Elastic Load Balancer (ELB) health checks for Amazon Elastic Compute Cloud (Amazon EC2) Auto Scaling groups support maintenance of adequate capacity and availability. The load balancer periodically sends pings, attempts connections, or sends requests to test Amazon EC2 instances health in an auto-scaling group. If an instance is not reporting back, traffic is sent to a new Amazon EC2 instance.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: autoscaling_1.AutoScalingGroupELBHealthCheckRequired,
            node: node,
        });
        this.applyRule({
            info: 'The Auto Scaling launch configuration does not have public IP addresses disabled - (Control IDs: AC-3, AC-4(21), CM-6a, SC-7(3)).',
            explanation: 'If you configure your Network Interfaces with a public IP address, then the associated resources to those Network Interfaces are reachable from the internet. EC2 resources should not be publicly accessible, as this may allow unintended access to your applications or servers.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: autoscaling_1.AutoScalingLaunchConfigPublicIpDisabled,
            node: node,
        });
    }
    /**
     * Check CloudTrail Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkCloudTrail(node) {
        this.applyRule({
            info: 'The trail does not have CloudWatch logs enabled - (Control IDs: AC-2(4), AC-3(1), AC-3(10), AC-4(26), AC-6(9), AU-2b, AU-3a, AU-3b, AU-3c, AU-3d, AU-3e, AU-3f, AU-4(1), AU-6(1), AU-6(3), AU-6(4), AU-6(5), AU-6(6), AU-6(9), AU-7(1), AU-8b, AU-9(7), AU-10, AU-12a, AU-12c, AU-12(1), AU-12(2), AU-12(3), AU-12(4), AU-14a, AU-14b, AU-14b, AU-14(3), AU-16, CA-7b, CM-5(1)(b), CM-6a, CM-9b, IA-3(3)(b), MA-4(1)(a), PM-14a.1, PM-14b, PM-31, SC-7(9)(b), SI-1(1)(c), SI-3(8)(b), SI-4(2), SI-4(17), SI-4(20), SI-7(8), SI-10(1)(c)).',
            explanation: 'Use Amazon CloudWatch to centrally collect and manage log event activity. Inclusion of AWS CloudTrail data provides details of API call activity within your AWS account.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: cloudtrail_1.CloudTrailCloudWatchLogsEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The trail does not have encryption enabled - (Control IDs: AU-9(3), CM-6a, CM-9b, CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)).',
            explanation: 'Because sensitive data may exist and to help protect data at rest, ensure encryption is enabled for your AWS CloudTrail trails.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: cloudtrail_1.CloudTrailEncryptionEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The trail does not have log file validation enabled - (Control IDs: AU-9a, CM-6a, CM-9b, PM-11b, PM-17b, SA-1(1), SA-10(1), SC-16(1), SI-1a.2, SI-1a.2, SI-1c.2, SI-4d, SI-7a, SI-7(1), SI-7(3), SI-7(7)).',
            explanation: 'Utilize AWS CloudTrail log file validation to check the integrity of CloudTrail logs. Log file validation helps determine if a log file was modified or deleted or unchanged after CloudTrail delivered it. This feature is built using industry standard algorithms: SHA-256 for hashing and SHA-256 with RSA for digital signing. This makes it computationally infeasible to modify, delete or forge CloudTrail log files without detection.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: cloudtrail_1.CloudTrailLogFileValidationEnabled,
            node: node,
        });
    }
    /**
     * Check CloudWatch Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkCloudWatch(node) {
        this.applyRule({
            info: 'The CloudWatch alarm does not have at least one alarm action, one INSUFFICIENT_DATA action, or one OK action enabled - (Control IDs: AU-6(1), AU-6(5), AU-12(3), AU-14a, AU-14b, CA-2(2), CA-7, CA-7b, PM-14a.1, PM-14b, PM-31, SC-36(1)(a), SI-2a, SI-4(12), SI-5b, SI-5(1)).',
            explanation: 'Amazon CloudWatch alarms alert when a metric breaches the threshold for a specified number of evaluation periods. The alarm performs one or more actions based on the value of the metric or expression relative to a threshold over a number of time periods.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: cloudwatch_1.CloudWatchAlarmAction,
            node: node,
        });
        this.applyRule({
            info: 'The CloudWatch Log Group is not encrypted with an AWS KMS key - (Control IDs: AU-9(3), CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)).',
            explanation: 'To help protect sensitive data at rest, ensure encryption is enabled for your Amazon CloudWatch Log Groups.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: cloudwatch_1.CloudWatchLogGroupEncrypted,
            node: node,
        });
        this.applyRule({
            info: 'The CloudWatch Log Group does not have an explicit retention period configured - (Control IDs: AC-16b, AT-4b, AU-6(3), AU-6(4), AU-6(6), AU-6(9), AU-10, AU-11(1), AU-11, AU-12(1), AU-12(2), AU-12(3), AU-14a, AU-14b, CA-7b, PM-14a.1, PM-14b, PM-21b, PM-31, SC-28(2), SI-4(17), SI-12).',
            explanation: 'Ensure a minimum duration of event log data is retained for your log groups to help with troubleshooting and forensics investigations. The lack of available past event log data makes it difficult to reconstruct and identify potentially malicious events.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: cloudwatch_1.CloudWatchLogGroupRetentionPeriod,
            node: node,
        });
    }
    /**
     * Check DMS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkDMS(node) {
        this.applyRule({
            info: 'The DMS replication instance is public - (Control IDs: AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25).',
            explanation: 'DMS replication instances can contain sensitive information and access control is required for such accounts.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: dms_1.DMSReplicationNotPublic,
            node: node,
        });
    }
    /**
     * Check DynamoDB Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkDynamoDB(node) {
        this.applyRule({
            info: "The provisioned capacity DynamoDB table does not have Auto Scaling enabled on it's indexes - (Control IDs: CP-1a.1(b), CP-1a.2, CP-2a, CP-2a.6, CP-2a.7, CP-2d, CP-2e, CP-2(5), CP-2(6), CP-6(2), CP-10, SC-5(2), SC-6, SC-22, SC-36, SI-13(5)).",
            explanation: 'Amazon DynamoDB auto scaling uses the AWS Application Auto Scaling service to adjust provisioned throughput capacity that automatically responds to actual traffic patterns. This enables a table or a global secondary index to increase its provisioned read/write capacity to handle sudden increases in traffic, without throttling.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: dynamodb_1.DynamoDBAutoScalingEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The DynamoDB table is not in an AWS Backup plan - (Control IDs: CP-1(2), CP-2(5), CP-6a, CP-6(1), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), SC-5(2), SI-13(5)).',
            explanation: 'To help with data back-up processes, ensure your Amazon DynamoDB tables are a part of an AWS Backup plan. AWS Backup is a fully managed backup service with a policy-based backup solution. This solution simplifies your backup management and enables you to meet your business and regulatory backup compliance requirements.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: dynamodb_1.DynamoDBInBackupPlan,
            node: node,
        });
        this.applyRule({
            info: 'The DynamoDB table does not have Point-in-time Recovery enabled - (Control IDs: CP-1(2), CP-2(5), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), SC-5(2), SI-13(5)).',
            explanation: 'The recovery maintains continuous backups of your table for the last 35 days.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: dynamodb_1.DynamoDBPITREnabled,
            node: node,
        });
    }
    /**
     * Check EC2 Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkEC2(node) {
        this.applyRule({
            info: 'The EBS volume is not in an AWS Backup plan - (Control IDs: CP-1(2), CP-2(5), CP-6a, CP-6(1), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), SC-5(2), SI-13(5)).',
            explanation: 'To help with data back-up processes, ensure your Amazon Elastic Block Store (Amazon EBS) volumes are a part of an AWS Backup plan. AWS Backup is a fully managed backup service with a policy-based backup solution. This solution simplifies your backup management and enables you to meet your business and regulatory backup compliance requirements.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2EBSInBackupPlan,
            node: node,
        });
        this.applyRule({
            info: "The EC2 instance type 'supports' EBS optimization and does not have EBS optimization enabled - (Control IDs: CP-2(5), CP-9a, CP-9b, CP-9c, CP-10, SC-5(2)).",
            explanation: 'An optimized instance in Amazon Elastic Block Store (Amazon EBS) provides additional, dedicated capacity for Amazon EBS I/O operations. This optimization provides the most efficient performance for your EBS volumes by minimizing contention between Amazon EBS I/O operations and other traffic from your instance.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2EBSOptimizedInstance,
            node: node,
        });
        this.applyRule({
            info: 'The EC2 instance does not have IMDSV2 (Instance Metadata Service Version 2) enabled - (Control ID: AC-2(6), AC-3, AC-3(3)(a), AC-3(3)(b)(1), AC-3(3)(b)(2), AC-3(3)(b)(3), AC-3(3)(b)(4), AC-3(3)(b)(5), AC-3(3)(c), AC-3(3), AC-3(4)(a), AC-3(4)(b), AC-3(4)(c), AC-3(4)(d), AC-3(4)(e), AC-3(4), AC-3(7), AC-3(8), AC-3(12)(a), AC-3(13), AC-3(15)(a), AC-3(15)(b), AC-4(28), AC-6, AC-24, CM-5(1)(a), MP-2, SC-23(3)).',
            explanation: 'Instance Metadata Service Version 2 (IMDSv2) helps protect access and control of Amazon Elastic Compute Cloud (Amazon EC2) instance metadata. The IMDSv2 method uses session-based controls. With IMDSv2, controls can be implemented to restrict changes to instance metadata.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2IMDSv2Enabled,
            node: node,
        });
        this.applyRule({
            info: 'The EC2 instance is associated with a public IP address - (Control IDs: AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25).',
            explanation: 'Manage access to the AWS Cloud by ensuring Amazon Elastic Compute Cloud (Amazon EC2) instances cannot be publicly accessed. Amazon EC2 instances can contain sensitive information and access control is required for such accounts.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2InstanceNoPublicIp,
            node: node,
        });
        this.applyRule({
            info: 'The EC2 instance does not have an instance profile attached - (Control IDs: AC-3, CM-5(1)(a), CM-6a).',
            explanation: 'EC2 instance profiles pass an IAM role to an EC2 instance. Attaching an instance profile to your instances can assist with least privilege and permissions management.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2InstanceProfileAttached,
            node: node,
        });
        this.applyRule({
            info: 'The EC2 instance is not within a VPC - (Control IDs: AC-2(6), AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-25).',
            explanation: 'Deploy Amazon Elastic Compute Cloud (Amazon EC2) instances within an Amazon Virtual Private Cloud (Amazon VPC) to enable secure communication between an instance and other services within the amazon VPC, without requiring an internet gateway, NAT device, or VPN connection. All traffic remains securely within the AWS Cloud. Because of their logical isolation, domains that reside within anAmazon VPC have an extra layer of security when compared to domains that use public endpoints. Assign Amazon EC2 instances to an Amazon VPC to properly manage access.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2InstancesInVPC,
            node: node,
        });
        this.applyRule({
            info: 'The EC2 instance allows unrestricted inbound IPv4 TCP traffic on one or more common ports (by default these ports include 20, 21, 3389, 3309, 3306, 4333) - (Control IDs: AC-4(21), AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), CM-2a, CM-2(2), CM-6a, CM-7b, CM-8(6), CM-9b, SC-7a, SC-7c, SC-7(5), SC-7(7), SC-7(11), SC-7(12), SC-7(16), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28)).',
            explanation: 'Not restricting access to ports to trusted sources can lead to attacks against the availability, integrity and confidentiality of systems. By default, common ports which should be restricted include port numbers 20, 21, 3389, 3306, and 4333.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2RestrictedCommonPorts,
            node: node,
        });
        this.applyRule({
            info: 'The Security Group allows unrestricted SSH access - (Control IDs: AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), CM-9b, SC-7a, SC-7c, SC-7(7), SC-7(11), SC-7(12), SC-7(16), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28)).',
            explanation: 'Not allowing ingress (or remote) traffic from 0.0.0.0/0 or ::/0 to port 22 on your resources helps to restrict remote access.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2RestrictedSSH,
            node: node,
        });
    }
    /**
     * Check ECS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkECS(node) {
        this.applyRule({
            info: "The ECS task definition is configured for host networking and has at least one container with definitions with 'privileged' set to false or empty or 'user' set to root or empty - (Control IDs: AC-3, AC-5b, CM-5(1)(a)).",
            explanation: 'If a task definition has elevated privileges it is because you have specifically opted-in to those configurations. This rule checks for unexpected privilege escalation when a task definition has host networking enabled but the customer has not opted-in to elevated privileges.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: ecs_1.ECSTaskDefinitionUserForHostMode,
            node: node,
        });
    }
    /**
     * Check EFS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkEFS(node) {
        this.applyRule({
            info: 'The EFS is not in an AWS Backup plan - (Control IDs: CP-1(2), CP-2(5), CP-6a, CP-6(1), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), SC-5(2), SI-13(5)).',
            explanation: 'To help with data back-up processes, ensure your Amazon Elastic File System (Amazon EFS) file systems are a part of an AWS Backup plan. AWS Backup is a fully managed backup service with a policy-based backup solution. This solution simplifies your backup management and enables you to meet your business and regulatory backup compliance requirements.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: efs_1.EFSInBackupPlan,
            node: node,
        });
        this.applyRule({
            info: 'The EFS does not have encryption at rest enabled - (Control IDs: AU-9(3), CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)).',
            explanation: 'Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon Elastic File System (EFS).',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: efs_1.EFSEncrypted,
            node: node,
        });
    }
    /**
     * Check ElastiCache Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkElastiCache(node) {
        this.applyRule({
            info: 'The ElastiCache Redis cluster does not retain automatic backups for at least 15 days - (Control IDs: CP-1(2), CP-2(5), CP-6a, CP-6(1), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), SC-5(2), SI-13(5)).',
            explanation: 'Automatic backups can help guard against data loss. If a failure occurs, you can create a new cluster, which restores your data from the most recent backup.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elasticache_1.ElastiCacheRedisClusterAutomaticBackup,
            node: node,
        });
    }
    /**
     * Check Elastic Beanstalk Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkElasticBeanstalk(node) {
        this.applyRule({
            info: 'The Elastic Beanstalk environment does not have enhanced health reporting enabled - (Control IDs: AU-12(3), AU-14a, AU-14b, CA-2(2), CA-7, CA-7b, PM-14a.1, PM-14b, PM-31, SC-6, SC-36(1)(a), SI-2a).',
            explanation: 'AWS Elastic Beanstalk enhanced health reporting enables a more rapid response to changes in the health of the underlying infrastructure. These changes could result in a lack of availability of the application. Elastic Beanstalk enhanced health reporting provides a status descriptor to gauge the severity of the identified issues and identify possible causes to investigate.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elasticbeanstalk_1.ElasticBeanstalkEnhancedHealthReportingEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The Elastic Beanstalk environment does not have managed updates enabled - (Control IDs: SI-2c, SI-2d, SI-2(2), SI-2(5)).',
            explanation: 'Enabling managed platform updates for an Amazon Elastic Beanstalk environment ensures that the latest available platform fixes, updates, and features for the environment are installed. Keeping up to date with patch installation is a best practice in securing systems.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elasticbeanstalk_1.ElasticBeanstalkManagedUpdatesEnabled,
            node: node,
        });
    }
    /**
     * Check Elastic Load Balancer Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkELB(node) {
        this.applyRule({
            info: "The ALB's HTTP listeners are not configured to redirect to HTTPS - (Control IDs: AC-4, AC-4(22), AC-17(2), AC-24(1), AU-9(3), CA-9b, IA-5(1)(c), PM-17b, SC-7(4)(b), SC-7(4)(g), SC-8, SC-8(1), SC-8(2), SC-8(3), SC-8(4), SC-8(5), SC-13a, SC-23, SI-1a.2, SI-1a.2, SI-1c.2).",
            explanation: 'To help protect data in transit, ensure that your Application Load Balancer automatically redirects unencrypted HTTP requests to HTTPS. Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elb_1.ALBHttpToHttpsRedirection,
            node: node,
        });
        this.applyRule({
            info: 'The ALB is not associated with AWS WAFv2 web ACL - (Control ID: AC-4(21)).',
            explanation: 'A WAF helps to protect your web applications or APIs against common web exploits. These web exploits may affect availability, compromise security, or consume excessive resources within your environment.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elb_1.ALBWAFEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The CLB does not utilize an SSL certificate provided by ACM (Amazon Certificate Manager) - (Control IDs: AC-4, AC-4(22), AC-17(2), AC-24(1), AU-9(3), CA-9b, IA-5(1)(c), PM-17b, SC-7(4)(b), SC-7(4)(g), SC-8, SC-8(1), SC-8(2), SC-8(3), SC-8(4), SC-8(5), SC-13a, SC-23, SC-23(5), SI-1a.2, SI-1a.2, SI-1c.2).',
            explanation: 'Because sensitive data can exist and to help protect data at transit, ensure encryption is enabled for your Elastic Load Balancing. Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBACMCertificateRequired,
            node: node,
        });
        this.applyRule({
            info: 'The CLB does not balance traffic between at least 2 Availability Zones - (Control IDs: CP-1a.1(b), CP-1a.2, CP-2a, CP-2a.6, CP-2a.7, CP-2d, CP-2e, CP-2(5), CP-2(6), CP-6(2), CP-10, SC-5(2), SC-6, SC-22, SC-36, SI-13(5)).',
            explanation: "Enable cross-zone load balancing for your Classic Load Balancers (CLBs) to help maintain adequate capacity and availability. The cross-zone load balancing reduces the need to maintain equivalent numbers of instances in each enabled availability zone. It also improves your application's ability to handle the loss of one or more instances.",
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBCrossZoneLoadBalancingEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The ALB, NLB, or GLB does not have deletion protection enabled - (Control IDs: CA-7(4)(c), CM-2a, CM-2(2), CM-3a, CM-8(6), CP-1a.1(b), CP-1a.2, CP-2a, CP-2a.6, CP-2a.7, CP-2d, CP-2e, CP-2(5), SA-15a.4, SC-5(2), SC-22).',
            explanation: 'This rule ensures that Elastic Load Balancing has deletion protection enabled. Use this feature to prevent your load balancer from being accidentally or maliciously deleted, which can lead to loss of availability for your applications.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBDeletionProtectionEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The ELB does not have logging enabled - (Control IDs: AC-4(26), AU-2b, AU-3a, AU-3b, AU-3c, AU-3d, AU-3e, AU-3f, AU-6(3), AU-6(4), AU-6(6), AU-6(9), AU-8b, AU-10, AU-12a, AU-12c, AU-12(1), AU-12(2), AU-12(3), AU-12(4), AU-14a, AU-14b, AU-14b, AU-14(3), CA-7b, CM-5(1)(b), IA-3(3)(b), MA-4(1)(a), PM-14a.1, PM-14b, PM-31, SC-7(9)(b), SI-4(17), SI-7(8)).',
            explanation: "Elastic Load Balancing activity is a central point of communication within an environment. Ensure ELB logging is enabled. The collected data provides detailed information about requests sent to The ELB. Each log contains information such as the time the request was received, the client's IP address, latencies, request paths, and server responses.",
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBLoggingEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The CLB does not restrict its listeners to only the SSL and HTTPS protocols - (Control IDs: AC-4, AC-4(22), AC-17(2), AC-24(1), AU-9(3), CA-9b, IA-5(1)(c), PM-17b, PM-17b, SC-7(4)(b), SC-7(4)(g), SC-8, SC-8(1), SC-8(2), SC-8(2), SC-8(3), SC-8(4), SC-8(5), SC-13a, SC-23, SI-1a.2, SI-1a.2, SI-1a.2, SI-1a.2, SI-1c.2, SI-1c.2).',
            explanation: 'Ensure that your Classic Load Balancers (CLBs) are configured with SSL or HTTPS listeners. Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBTlsHttpsListenersOnly,
            node: node,
        });
        this.applyRule({
            info: 'The ALB, NLB, or GLB listener does not utilize an SSL certificate provided by ACM (Amazon Certificate Manager) - (Control IDs: SC-8(1), SC-23(5)).',
            explanation: 'Because sensitive data can exist and to help protect data at transit, ensure encryption is enabled for your Elastic Load Balancing. Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBv2ACMCertificateRequired,
            node: node,
        });
    }
    /**
     * Check IAM Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkIAM(node) {
        this.applyRule({
            info: 'The IAM Group, User, or Role contains an inline policy - (Control IDs: AC-2i.2, AC-2(1), AC-2(6), AC-3, AC-3(3)(a), AC-3(3)(b)(1), AC-3(3)(b)(2), AC-3(3)(b)(3), AC-3(3)(b)(4), AC-3(3)(b)(5), AC-3(3)(c), AC-3(3), AC-3(4)(a), AC-3(4)(b), AC-3(4)(c), AC-3(4)(d), AC-3(4)(e), AC-3(4), AC-3(7), AC-3(8), AC-3(12)(a), AC-3(13), AC-3(15)(a), AC-3(15)(b), AC-4(28), AC-6, AC-6(3), AC-24, CM-5(1)(a), CM-6a, CM-9b, MP-2, SC-23(3)).',
            explanation: 'AWS recommends to use managed policies instead of inline policies. The managed policies allow reusability, versioning and rolling back, and delegating permissions management.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMNoInlinePolicy,
            node: node,
        });
        this.applyRule({
            info: 'The IAM policy grants admin access, meaning the policy allows a principal to perform all actions on all resources - (Control IDs: AC-2i.2, AC-2(1), AC-2(6), AC-3, AC-3(3)(a), AC-3(3)(b)(1), AC-3(3)(b)(2), AC-3(3)(b)(3), AC-3(3)(b)(4), AC-3(3)(b)(5), AC-3(3)(c), AC-3(3), AC-3(4)(a), AC-3(4)(b), AC-3(4)(c), AC-3(4)(d), AC-3(4)(e), AC-3(4), AC-3(7), AC-3(8), AC-3(12)(a), AC-3(13), AC-3(15)(a), AC-3(15)(b), AC-4(28), AC-5b, AC-6, AC-6(2), AC-6(3), AC-6(10), AC-24, CM-5(1)(a), CM-6a, CM-9b, MP-2, SC-23(3), SC-25).',
            explanation: 'AWS Identity and Access Management (IAM) can help you incorporate the principles of least privilege and separation of duties with access permissions and authorizations, by ensuring that IAM groups have at least one IAM user. Placing IAM users in groups based on their associated permissions or job function is one way to incorporate least privilege.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMPolicyNoStatementsWithAdminAccess,
            node: node,
        });
        this.applyRule({
            info: 'The IAM policy grants full access, meaning the policy allows a principal to perform all actions on individual resources - (Control IDs: AC-3, AC-5b, AC-6(2), AC-6(10), CM-5(1)(a)).',
            explanation: 'Ensure IAM Actions are restricted to only those actions that are needed. Allowing users to have more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMPolicyNoStatementsWithFullAccess,
            node: node,
        });
        this.applyRule({
            info: 'The IAM user does not belong to any group(s) - (Control IDs: AC-2i.2, AC-2(1), AC-2(6), AC-3, AC-3(3)(a), AC-3(3)(b)(1), AC-3(3)(b)(2), AC-3(3)(b)(3), AC-3(3)(b)(4), AC-3(3)(b)(5), AC-3(3)(c), AC-3(3), AC-3(4)(a), AC-3(4)(b), AC-3(4)(c), AC-3(4)(d), AC-3(4)(e), AC-3(4), AC-3(7), AC-3(8), AC-3(12)(a), AC-3(13), AC-3(15)(a), AC-3(15)(b), AC-4(28), AC-6, AC-6(3), AC-24, CM-5(1)(a), CM-6a, CM-9b, MP-2, SC-23(3)).',
            explanation: 'AWS Identity and Access Management (IAM) can help you restrict access permissions and authorizations by ensuring IAM users are members of at least one group. Allowing users more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMUserGroupMembership,
            node: node,
        });
        this.applyRule({
            info: 'The IAM policy is attached at the user level - (Control IDs: AC-2i.2, AC-2(1), AC-2(6), AC-3, AC-3(3)(a), AC-3(3)(b)(1), AC-3(3)(b)(2), AC-3(3)(b)(3), AC-3(3)(b)(4), AC-3(3)(b)(5), AC-3(3)(c), AC-3(3), AC-3(4)(a), AC-3(4)(b), AC-3(4)(c), AC-3(4)(d), AC-3(4)(e), AC-3(4), AC-3(7), AC-3(8), AC-3(12)(a), AC-3(13), AC-3(15)(a), AC-3(15)(b), AC-4(28), AC-6, AC-6(3), AC-24, CM-5(1)(a), CM-6a, CM-9b, MP-2, SC-23(3), SC-25).',
            explanation: 'Assigning privileges at the group or the role level helps to reduce opportunity for an identity to receive or retain excessive privileges.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMUserNoPolicies,
            node: node,
        });
    }
    /**
     * Check KMS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkKMS(node) {
        this.applyRule({
            info: 'The KMS Symmetric key does not have automatic key rotation enabled - (Control IDs: CM-6a, CM-9b, SA-9(6), SC-12, SC-12(2), SC-12(6)).',
            explanation: 'Enable key rotation to ensure that keys are rotated once they have reached the end of their crypto period.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: kms_1.KMSBackingKeyRotationEnabled,
            node: node,
        });
    }
    /**
     * Check Lambda Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkLambda(node) {
        this.applyRule({
            info: 'The Lambda function is not configured with function-level concurrent execution limits - (Control IDs: AU-12(3), AU-14a, AU-14b, CA-7, CA-7b, PM-14a.1, PM-14b, PM-31, SC-6).',
            explanation: "Ensure that a Lambda function's concurrency high and low limits are established. This can assist in baselining the number of requests that your function is serving at any given time.",
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: lambda_1.LambdaConcurrency,
            node: node,
        });
        this.applyRule({
            info: 'The Lambda function is not configured with a dead-letter configuration - (Control IDs: AU-12(3), AU-14a, AU-14b, CA-2(2), CA-7, CA-7b, PM-14a.1, PM-14b, PM-31, SC-36(1)(a), SI-2a).',
            explanation: 'Notify the appropriate personnel through Amazon Simple Queue Service (Amazon SQS) or Amazon Simple Notification Service (Amazon SNS) when a function has failed.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: lambda_1.LambdaDLQ,
            node: node,
        });
        this.applyRule({
            info: 'The Lambda function permission grants public access - (Control IDs: AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25).',
            explanation: 'Public access allows anyone on the internet to perform unauthenticated actions on your function and can potentially lead to degraded availability.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: lambda_1.LambdaFunctionPublicAccessProhibited,
            node: node,
        });
        this.applyRule({
            info: 'The Lambda function is not VPC enabled - (Control IDs: AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-25).',
            explanation: 'Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: lambda_1.LambdaInsideVPC,
            node: node,
        });
    }
    /**
     * Check OpenSearch Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkOpenSearch(node) {
        this.applyRule({
            info: 'The OpenSearch Service domain does not have encryption at rest enabled - (Control IDs: AU-9(3), CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)).',
            explanation: 'Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon OpenSearch Service (OpenSearch Service) domains.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchEncryptedAtRest,
            node: node,
        });
        this.applyRule({
            info: 'The OpenSearch Service domain is not running within a VPC - (Control IDs: AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-25).',
            explanation: 'VPCs help secure your AWS resources and provide an extra layer of protection.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchInVPCOnly,
            node: node,
        });
        this.applyRule({
            info: 'The OpenSearch Service domain does not stream error logs (ES_APPLICATION_LOGS) to CloudWatch Logs - (Control ID: AU-10).',
            explanation: 'Ensure Amazon OpenSearch Service domains have error logs enabled and streamed to Amazon CloudWatch Logs for retention and response. Domain error logs can assist with security and access audits, and can help to diagnose availability issues.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchErrorLogsToCloudWatch,
            node: node,
        });
        this.applyRule({
            info: 'The OpenSearch Service domain does not have node-to-node encryption enabled - (Control IDs: AC-4, AC-4(22), AC-24(1), AU-9(3), CA-9b, PM-17b, SC-7(4)(b), SC-7(4)(g), SC-8, SC-8(1), SC-8(2), SC-8(3), SC-8(4), SC-8(5), SC-13a, SC-23, SI-1a.2, SI-1a.2, SI-1c.2).',
            explanation: 'Because sensitive data can exist, enable encryption in transit to help protect that data within your Amazon OpenSearch Service (OpenSearch Service) domains.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchNodeToNodeEncryption,
            node: node,
        });
    }
    /**
     * Check RDS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkRDS(node) {
        this.applyRule({
            info: 'The RDS DB Instance does not have enhanced monitoring enabled - (Control IDs: AU-12(3), AU-14a, AU-14b, CA-2(2), CA-7, CA-7b, PM-14a.1, PM-14b, PM-31, SC-36(1)(a), SI-2a).',
            explanation: 'Enable enhanced monitoring to help monitor Amazon RDS availability. This provides detailed visibility into the health of your Amazon RDS database instances.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSEnhancedMonitoringEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The RDS DB instance is not in an AWS Backup plan - (Control IDs: CP-1(2), CP-2(5), CP-6a, CP-6(1), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), SC-5(2), SI-13(5)).',
            explanation: 'To help with data back-up processes, ensure your Amazon Relational Database Service (Amazon RDS) instances are a part of an AWS Backup plan. AWS Backup is a fully managed backup service with a policy-based backup solution. This solution simplifies your backup management and enables you to meet your business and regulatory backup compliance requirements.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSInBackupPlan,
            node: node,
        });
        this.applyRule({
            info: 'The RDS DB Instance does not have backup enabled - (Control IDs: CP-1(2), CP-2(5), CP-6a, CP-6(1), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), SC-5(2), SI-13(5)).',
            explanation: 'The backup feature of Amazon RDS creates backups of your databases and transaction logs.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSInstanceBackupEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The RDS DB Instance or Aurora Cluster does not have deletion protection enabled - (Control IDs: CA-7(4)(c), CM-3a, CP-1a.1(b), CP-1a.2, CP-2a, CP-2a.6, CP-2a.7, CP-2d, CP-2e, CP-2(5), SA-15a.4, SC-5(2), SC-22, SI-13(5)).',
            explanation: 'Ensure Amazon Relational Database Service (Amazon RDS) instances and clusters have deletion protection enabled. Use deletion protection to prevent your Amazon RDS DB instances and clusters from being accidentally or maliciously deleted, which can lead to loss of availability for your applications.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSInstanceDeletionProtectionEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The RDS DB Instance allows public access - (Control IDs: AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25).',
            explanation: 'Amazon RDS database instances can contain sensitive information, and principles and access control is required for such accounts.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSInstancePublicAccess,
            node: node,
        });
        this.applyRule({
            info: 'The non-Aurora RDS DB instance or Aurora cluster does not have all CloudWatch log types exported - (Control IDs: AC-2(4), AC-3(1), AC-3(10), AC-4(26), AC-6(9), AU-2b, AU-3a, AU-3b, AU-3c, AU-3d, AU-3e, AU-3f, AU-6(3), AU-6(4), AU-6(6), AU-6(9), AU-8b, AU-10, AU-12a, AU-12c, AU-12(1), AU-12(2), AU-12(3), AU-12(4), AU-14a, AU-14b, AU-14b, AU-14(3), CA-7b, CM-5(1)(b), IA-3(3)(b), MA-4(1)(a), PM-14a.1, PM-14b, PM-31, SC-7(9)(b), SI-1(1)(c), SI-3(8)(b), SI-4(2), SI-4(17), SI-4(20), SI-7(8), SI-10(1)(c)).',
            explanation: 'To help with logging and monitoring within your environment, ensure Amazon Relational Database Service (Amazon RDS) logging is enabled. With Amazon RDS logging, you can capture events such as connections, disconnections, queries, or tables queried.' +
                "This is a granular rule that returns individual findings that can be suppressed with 'appliesTo'. The findings are in the format 'LogExport::<log>' for exported logs. Example: appliesTo: ['LogExport::audit'].",
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSLoggingEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The RDS DB Instance does not have multi-AZ support - (Control IDs: CP-1a.1(b), CP-1a.2, CP-2a, CP-2a.6, CP-2a.7, CP-2d, CP-2e, CP-2(5), CP-2(6), CP-6(2), CP-10, SC-5(2), SC-6, SC-22, SC-36, SI-13(5)).',
            explanation: 'Multi-AZ support in Amazon Relational Database Service (Amazon RDS) provides enhanced availability and durability for database instances. When you provision a Multi-AZ database instance, Amazon RDS automatically creates a primary database instance, and synchronously replicates the data to a standby instance in a different Availability Zone. In case of an infrastructure failure, Amazon RDS performs an automatic failover to the standby so that you can resume database operations as soon as the failover is complete.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSMultiAZSupport,
            node: node,
        });
        this.applyRule({
            info: 'The RDS DB Instance or Aurora Cluster does not have storage encrypted - (Control IDs: AU-9(3), CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)).',
            explanation: 'Because sensitive data can exist at rest in Amazon RDS instances, enable encryption at rest to help protect that data.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSStorageEncrypted,
            node: node,
        });
    }
    /**
     * Check Redshift Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkRedshift(node) {
        this.applyRule({
            info: 'The Redshift cluster does not have automated snapshots enabled or the retention period is not between 1 and 35 days - (Control IDs: CP-1(2), CP-2(5), CP-6a, CP-6(1), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), SC-5(2), SI-13(5)).',
            explanation: 'To help with data back-up processes, ensure your Amazon Redshift clusters have automated snapshots. When automated snapshots are enabled for a cluster, Redshift periodically takes snapshots of that cluster. By default, Redshift takes a snapshot every eight hours or every 5 GB per node of data changes, or whichever comes first.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftBackupEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The Redshift cluster does not have encryption or audit logging enabled - (Control IDs: AC-2(4), AC-3(1), AC-3(10), AC-4(26), AC-6(9), AU-2b, AU-3a, AU-3b, AU-3c, AU-3d, AU-3e, AU-3f, AU-6(3), AU-6(4), AU-6(6), AU-6(9), AU-8b, AU-9(3), AU-10, AU-12a, AU-12c, AU-12(1), AU-12(2), AU-12(3), AU-12(4), AU-14a, AU-14b, AU-14b, AU-14(3), CA-7b, CM-5(1)(b), CP-9d, IA-3(3)(b), MA-4(1)(a), PM-14a.1, PM-14b, PM-31, SC-7(9)(b), SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-1(1)(c), SI-3(8)(b), SI-4(2), SI-4(17), SI-4(20), SI-7(8), SI-10(1)(c), SI-19(4)).',
            explanation: 'To protect data at rest, ensure that encryption is enabled for your Amazon Redshift clusters. You must also ensure that required configurations are deployed on Amazon Redshift clusters. The audit logging should be enabled to provide information about connections and user activities in the database.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterConfiguration,
            node: node,
        });
        this.applyRule({
            info: 'The Redshift cluster does not have version upgrades enabled, automated snapshot retention periods enabled, and an explicit maintenance window configured - (Control IDs: CM-2b, CM-2b.1, CM-2b.2, CM-2b.3, CM-3(3), CP-9a, CP-9b, CP-9c, SC-5(2), SI-2c, SI-2d, SI-2(2), SI-2(5)).',
            explanation: 'Ensure that Amazon Redshift clusters have the preferred settings for your organization. Specifically, that they have preferred maintenance windows and automated snapshot retention periods for the database.                                                                            ',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterMaintenanceSettings,
            node: node,
        });
        this.applyRule({
            info: 'The Redshift cluster allows public access - (Control IDs: AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25).',
            explanation: 'Amazon Redshift clusters can contain sensitive information and principles and access control is required for such accounts.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterPublicAccess,
            node: node,
        });
        this.applyRule({
            info: 'The Redshift cluster does not have enhanced VPC routing enabled - (Control IDs: AC-4(21), SC-7b).',
            explanation: 'Enhanced VPC routing forces all COPY and UNLOAD traffic between the cluster and data repositories to go through your Amazon VPC. You can then use VPC features such as security groups and network access control lists to secure network traffic. You can also use VPC flow logs to monitor network traffic.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftEnhancedVPCRoutingEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The Redshift cluster does not require TLS/SSL encryption - (Control IDs: AC-4, AC-4(22), AC-24(1), AU-9(3), CA-9b, PM-17b, SC-7(4)(b), SC-7(4)(g), SC-8, SC-8(1), SC-8(2), SC-8(3), SC-8(4), SC-8(5), SC-13a, SC-23, SI-1a.2, SI-1a.2, SI-1c.2).',
            explanation: 'Ensure that your Amazon Redshift clusters require TLS/SSL encryption to connect to SQL clients. Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftRequireTlsSSL,
            node: node,
        });
    }
    /**
     * Check Amazon S3 Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkS3(node) {
        this.applyRule({
            info: 'The S3 bucket does not prohibit public access through bucket level settings - (Control IDs: AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25).',
            explanation: 'Keep sensitive data safe from unauthorized remote users by preventing public access at the bucket level.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketLevelPublicAccessProhibited,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Buckets does not have server access logs enabled - (Control IDs: AC-2(4), AC-3(1), AC-3(10), AC-4(26), AC-6(9), AU-2b, AU-3a, AU-3b, AU-3c, AU-3d, AU-3e, AU-3f, AU-6(3), AU-6(4), AU-6(6), AU-6(9), AU-8b, AU-10, AU-12a, AU-12c, AU-12(1), AU-12(2), AU-12(3), AU-12(4), AU-14a, AU-14b, AU-14b, AU-14(3), CA-7b, CM-5(1)(b), CM-6a, CM-9b, IA-3(3)(b), MA-4(1)(a), PM-14a.1, PM-14b, PM-31, SC-7(9)(b), SI-1(1)(c), SI-3(8)(b), SI-4(2), SI-4(17), SI-4(20), SI-7(8), SI-10(1)(c)).',
            explanation: 'Amazon Simple Storage Service (Amazon S3) server access logging provides a method to monitor the network for potential cybersecurity events. The events are monitored by capturing detailed records for the requests that are made to an Amazon S3 bucket. Each access log record provides details about a single access request. The details include the requester, bucket name, request time, request action, response status, and an error code, if relevant.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketLoggingEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Bucket does not prohibit public read access through its Block Public Access configurations and bucket ACLs - (Control IDs: AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), CM-6a, CM-9b, MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25).',
            explanation: 'The management of access should be consistent with the classification of the data.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketPublicReadProhibited,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Bucket does not prohibit public write access through its Block Public Access configurations and bucket ACLs - (Control IDs: AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), CM-6a, CM-9b, MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25).',
            explanation: 'The management of access should be consistent with the classification of the data.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketPublicWriteProhibited,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Bucket does not have replication enabled - (Control IDs: AU-9(2), CM-6a, CM-9b, CP-1(2), CP-2(5), CP-6a, CP-6(1), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), SC-5(2), SI-13(5)).',
            explanation: 'Amazon Simple Storage Service (Amazon S3) Cross-Region Replication (CRR) supports maintaining adequate capacity and availability. CRR enables automatic, asynchronous copying of objects across Amazon S3 buckets to help ensure that data availability is maintained.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketReplicationEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Bucket or bucket policy does not require requests to use SSL - (Control IDs: AC-4, AC-4(22), AC-17(2), AC-24(1), AU-9(3), CA-9b, CM-6a, CM-9b, IA-5(1)(c), PM-11b, PM-17b, SC-7(4)(b), SC-7(4)(g), SC-8, SC-8(1), SC-8(2), SC-8(3), SC-8(4), SC-8(5), SC-13a, SC-16(1), SC-23, SI-1a.2, SI-1a.2, SI-1c.2).',
            explanation: 'To help protect data in transit, ensure that your Amazon Simple Storage Service (Amazon S3) buckets require requests to use Secure Socket Layer (SSL). Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketSSLRequestsOnly,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Bucket does not have versioning enabled - (Control IDs: AU-9(2), CP-1(2), CP-2(5), CP-6a, CP-6(1), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), PM-11b, PM-17b, SC-5(2), SC-16(1), SI-1a.2, SI-1a.2, SI-1c.2, SI-13(5)).',
            explanation: 'Use versioning to preserve, retrieve, and restore every version of every object stored in your Amazon S3 bucket. Versioning helps you to easily recover from unintended user actions and application failures.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketVersioningEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Bucket is not encrypted with a KMS Key by default - (Control IDs: AU-9(3), CP-9d, CP-9(8), SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)).',
            explanation: 'Ensure that encryption is enabled for your Amazon Simple Storage Service (Amazon S3) buckets. Because sensitive data can exist at rest in an Amazon S3 bucket, enable encryption at rest to help protect that data.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: s3_1.S3DefaultEncryptionKMS,
            node: node,
        });
    }
    /**
     * Check SageMaker Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkSageMaker(node) {
        this.applyRule({
            info: 'The SageMaker resource endpoint is not encrypted with a KMS key - (Control IDs: AU-9(3), CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)).',
            explanation: 'Because sensitive data can exist at rest in SageMaker endpoint, enable encryption at rest to help protect that data.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.SageMakerEndpointConfigurationKMSKeyConfigured,
            node: node,
        });
        this.applyRule({
            info: 'The SageMaker notebook is not encrypted with a KMS key - (Control IDs: AU-9(3), CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)).',
            explanation: 'Because sensitive data can exist at rest in SageMaker notebook, enable encryption at rest to help protect that data.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.SageMakerNotebookInstanceKMSKeyConfigured,
            node: node,
        });
        this.applyRule({
            info: 'The SageMaker notebook does not disable direct internet access - (Control IDs: AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25).',
            explanation: 'By preventing direct internet access, you can keep sensitive data from being accessed by unauthorized users.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.SageMakerNotebookNoDirectInternetAccess,
            node: node,
        });
    }
    /**
     * Check Secrets Manager Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkSecretsManager(node) {
        this.applyRule({
            info: 'The secret does not have automatic rotation scheduled - (Control IDs: AC-4, AC-4(22), AC-24(1), AU-9(3), CA-9b, PM-17b, SC-7(4)(b), SC-7(4)(g), SC-8, SC-8(1), SC-8(2), SC-8(3), SC-8(4), SC-8(5), SC-13a, SC-23, SI-1a.2, SI-1a.2, SI-1c.2).',
            explanation: 'Rotating secrets on a regular schedule can shorten the period a secret is active, and potentially reduce the business impact if the secret is compromised.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: secretsmanager_1.SecretsManagerRotationEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The secret is not encrypted with a KMS Customer managed key - (Control IDs: AU-9(3), CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)).',
            explanation: 'To help protect data at rest, ensure encryption with AWS Key Management Service (AWS KMS) is enabled for AWS Secrets Manager secrets. Because sensitive data can exist at rest in Secrets Manager secrets, enable encryption at rest to help protect that data.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: secretsmanager_1.SecretsManagerUsingKMSKey,
            node: node,
        });
    }
    /**
     * Check Amazon SNS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkSNS(node) {
        this.applyRule({
            info: 'The SNS topic does not have KMS encryption enabled - (Control IDs: AU-9(3), CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1)).',
            explanation: 'To help protect data at rest, ensure that your Amazon Simple Notification Service (Amazon SNS) topics require encryption using AWS Key Management Service (AWS KMS) Because sensitive data can exist at rest in published messages, enable encryption at rest to help protect that data.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: sns_1.SNSEncryptedKMS,
            node: node,
        });
    }
    /**
     * Check VPC Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkVPC(node) {
        this.applyRule({
            info: "The VPC's default security group allows inbound or outbound traffic - (Control IDs: AC-4(21), AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), CM-6a, CM-9b, SC-7a, SC-7c, SC-7(5), SC-7(7), SC-7(11), SC-7(12), SC-7(16), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28)).",
            explanation: 'Amazon Elastic Compute Cloud (Amazon EC2) security groups can help in the management of network access by providing stateful filtering of ingress and egress network traffic to AWS resources. Restricting all the traffic on the default security group helps in restricting remote access to your AWS resources.',
            level: nag_rules_1.NagMessageLevel.WARN,
            rule: vpc_1.VPCDefaultSecurityGroupClosed,
            node: node,
        });
        this.applyRule({
            info: 'The VPC does not have an associated Flow Log - (Control IDs: AC-4(26), AU-2b, AU-3a, AU-3b, AU-3c, AU-3d, AU-3e, AU-6(3), AU-6(4), AU-6(6), AU-6(9), AU-8b, AU-12a, AU-12c, AU-12(1), AU-12(2), AU-12(3), AU-12(4), AU-14a, AU-14b, AU-14b, AU-14(3), CA-7b, CM-5(1)(b), CM-6a, CM-9b, IA-3(3)(b), MA-4(1)(a), PM-14a.1, PM-14b, PM-31, SI-4(17), SI-7(8)).',
            explanation: 'The VPC flow logs provide detailed records for information about the IP traffic going to and from network interfaces in your Amazon Virtual Private Cloud (Amazon VPC). By default, the flow log record includes values for the different components of the IP flow, including the source, destination, and protocol.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: vpc_1.VPCFlowLogsEnabled,
            node: node,
        });
        this.applyRule({
            info: "The route table may contain one or more unrestricted route(s) to an IGW ('0.0.0.0/0' or '::/0') - (Control IDs: AC-4(21), CM-7b).",
            explanation: 'Ensure Amazon EC2 route tables do not have unrestricted routes to an internet gateway. Removing or limiting the access to the internet for workloads within Amazon VPCs can reduce unintended access within your environment.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: vpc_1.VPCNoUnrestrictedRouteToIGW,
            node: node,
        });
        this.applyRule({
            info: 'The subnet auto-assigns public IP addresses - (Control IDs: AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25).',
            explanation: 'Manage access to the AWS Cloud by ensuring Amazon Virtual Private Cloud (VPC) subnets are not automatically assigned a public IP address. Amazon Elastic Compute Cloud (EC2) instances that are launched into subnets that have this attribute enabled have a public IP address assigned to their primary network interface.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: vpc_1.VPCSubnetAutoAssignPublicIpDisabled,
            node: node,
        });
    }
    /**
     * Check WAF Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkWAF(node) {
        this.applyRule({
            info: 'The WAFv2 web ACL does not have logging enabled - (Control IDs: AC-4(26), AU-2b, AU-3a, AU-3b, AU-3c, AU-3d, AU-3e, AU-3f, AU-6(3), AU-6(4), AU-6(6), AU-6(9), AU-8b, AU-10, AU-12a, AU-12c, AU-12(1), AU-12(2), AU-12(3), AU-12(4), AU-14a, AU-14b, AU-14b, AU-14(3), CA-7b, CM-5(1)(b), IA-3(3)(b), MA-4(1)(a), PM-14a.1, PM-14b, PM-31, SC-7(9)(b), SI-4(17), SI-7(8)).',
            explanation: 'AWS WAF logging provides detailed information about the traffic that is analyzed by your web ACL. The logs record the time that AWS WAF received the request from your AWS resource, information about the request, and an action for the rule that each request matched.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: waf_1.WAFv2LoggingEnabled,
            node: node,
        });
    }
}
exports.NIST80053R5Checks = NIST80053R5Checks;
_a = JSII_RTTI_SYMBOL_1;
NIST80053R5Checks[_a] = { fqn: "cdk-nag.NIST80053R5Checks", version: "2.28.126" };
//# sourceMappingURL=data:application/json;base64,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