"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const nag_rules_1 = require("../../nag-rules");
const flatten_cfn_reference_1 = require("../../utils/flatten-cfn-reference");
/**
 * S3 Buckets have server access logs enabled
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_s3_1.CfnBucket) {
        const logging = aws_cdk_lib_1.Stack.of(node).resolve(node.loggingConfiguration);
        if (logging == undefined ||
            (logging.destinationBucketName == undefined &&
                logging.logFilePrefix == undefined)) {
            let found = false;
            const bucketLogicalId = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.ref);
            const bucketName = aws_cdk_lib_1.Stack.of(node).resolve(node.bucketName);
            for (const child of aws_cdk_lib_1.Stack.of(node).node.findAll()) {
                if (child instanceof aws_s3_1.CfnBucket) {
                    if (isMatchingBucket(child, bucketLogicalId, bucketName)) {
                        found = true;
                        break;
                    }
                }
            }
            if (!found) {
                return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function to check whether a given S3 Bucket uses the target S3 bucket for access logs
 * @param node the CfnBucket to check
 * @param bucketLogicalId the Cfn Logical ID of the target bucket
 * @param bucketName the name of the target bucket
 * returns whether the CfnBucket uses the target S3 bucket for access logs
 */
function isMatchingBucket(node, bucketLogicalId, bucketName) {
    const destinationBucketName = flatten_cfn_reference_1.flattenCfnReference(aws_cdk_lib_1.Stack.of(node).resolve(aws_cdk_lib_1.Stack.of(node).resolve(node.loggingConfiguration)?.destinationBucketName));
    if (new RegExp(`${bucketLogicalId}(?![\\w])`).test(destinationBucketName) ||
        bucketName === destinationBucketName) {
        return true;
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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