"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_secretsmanager_1 = require("aws-cdk-lib/aws-secretsmanager");
const nag_rules_1 = require("../../nag-rules");
/**
 * Secrets have automatic rotation scheduled
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_secretsmanager_1.CfnSecret) {
        const secretLogicalId = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.ref);
        const secretTargetAttachmentLogicalIds = Array();
        const cfnSecretTargetAttachments = Array();
        const cfnRotationSchedules = Array();
        for (const child of aws_cdk_lib_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_secretsmanager_1.CfnSecretTargetAttachment) {
                cfnSecretTargetAttachments.push(child);
            }
            else if (child instanceof aws_secretsmanager_1.CfnRotationSchedule) {
                cfnRotationSchedules.push(child);
            }
        }
        if (cfnRotationSchedules.length === 0) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        let found = false;
        for (const child of cfnSecretTargetAttachments) {
            const attachmentLogicalId = getMatchingSecretTargetAttachment(child, secretLogicalId);
            if (attachmentLogicalId) {
                secretTargetAttachmentLogicalIds.push(attachmentLogicalId);
            }
        }
        for (const child of cfnRotationSchedules) {
            if (isMatchingRotationSchedule(child, secretLogicalId, secretTargetAttachmentLogicalIds)) {
                found = true;
                break;
            }
        }
        if (!found) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function to check whether a given Secret Target Attachment is associated with the given secret.
 * @param node The CfnTargetAttachment to check.
 * @param secretLogicalId The Cfn Logical ID of the secret.
 * Returns the Logical ID if the attachment if is associated with the secret, otherwise and empty string.
 */
function getMatchingSecretTargetAttachment(node, secretLogicalId) {
    const resourceSecretId = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.secretId);
    if (secretLogicalId === resourceSecretId) {
        return nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.ref);
    }
    return '';
}
/**
 * Helper function to check whether a given Rotation Schedule is associated with the given secret.
 * @param node The CfnRotationSchedule to check.
 * @param secretLogicalId The Cfn Logical ID of the secret.
 * @param secretTargetAttachmentLogicalIds The Cfn Logical IDs of any Secret Target Attachments associated with the given secret.
 * Returns whether the CfnRotationSchedule is associated with the given secret.
 */
function isMatchingRotationSchedule(node, secretLogicalId, secretTargetAttachmentLogicalIds) {
    const resourceSecretId = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.secretId);
    if (secretLogicalId === resourceSecretId ||
        secretTargetAttachmentLogicalIds.includes(resourceSecretId)) {
        if (aws_cdk_lib_1.Stack.of(node).resolve(node.hostedRotationLambda) === undefined &&
            aws_cdk_lib_1.Stack.of(node).resolve(node.rotationLambdaArn) === undefined) {
            return false;
        }
        const rotationRules = aws_cdk_lib_1.Stack.of(node).resolve(node.rotationRules);
        if (rotationRules !== undefined) {
            const automaticallyAfterDays = aws_cdk_lib_1.Stack.of(node).resolve(rotationRules.automaticallyAfterDays);
            const scheduleExpression = aws_cdk_lib_1.Stack.of(node).resolve(rotationRules.scheduleExpression);
            if (automaticallyAfterDays !== undefined ||
                scheduleExpression !== undefined) {
                return true;
            }
        }
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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