"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const nag_rules_1 = require("../../nag-rules");
/**
 * VPCs have their default security group closed
 * VPCs created via CloudFormation will not have their default security group closed.
 * The L2 VPC Construct provides a way to remmediate this via a custom resource.
 * @see https://docs.aws.amazon.com/vpc/latest/userguide/VPC_SecurityGroups.html#DefaultSecurityGroup
 * @see https://github.com/aws/aws-cdk/pull/25297
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_ec2_1.CfnVPC) {
        const parent = node.node.scope;
        if (parent) {
            const restrictSgCR = parent.node.tryFindChild('RestrictDefaultSecurityGroupCustomResource');
            if (restrictSgCR &&
                restrictSgCR.node.defaultChild.cfnResourceType ==
                    'Custom::VpcRestrictDefaultSG') {
                return nag_rules_1.NagRuleCompliance.COMPLIANT;
            }
            else {
                return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
            }
        }
        else {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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