"""module to automatically wrap methods in autogenerated low-level code and re-expose them as high-level functions"""

import inspect
from typing import Optional
import warnings

from beartype import beartype
from box import Box

from deeporigin.auth import get_tokens
from deeporigin.config import get_value
from deeporigin.utils.core import _get_method

warnings.filterwarnings(
    "ignore",
    module="pydantic._internal._fields",
)

# we're importing this here after the warnings are supressed
# because importing this raises UserWarnings we don't want to show
import do_sdk_platform  # noqa: E402


class PlatformClients:
    """
    A container for all DeepOrigin platform API clients, instantiated with a token, base_url, and org_key.

    Each API client is available as an attribute of this class, e.g., self.FilesApi, self.ToolsApi, etc.

    Args:
        token (str): The authentication token to use for all API clients.
        base_url (str): The base URL of the DeepOrigin platform (e.g., 'https://os.deeporigin.io').
        org_key (str): The organization-friendly ID to use for API requests.
    """

    def __init__(
        self,
        *,
        token: str,
        base_url: str,
        org_key: str,
    ) -> None:
        self.org_key = org_key

        api_endpoint = base_url + "/api/"
        apis = [attr for attr in do_sdk_platform.__dir__() if attr.endswith("Api")]

        for api in apis:
            setattr(
                self,
                api,
                _get_api_client(
                    api_name=api,
                    token=token,
                    api_endpoint=api_endpoint,
                ),
            )

    def list_clients(self) -> list:
        """Return a list of all API client attribute names available on this instance."""
        return [attr for attr in self.__dict__.keys() if attr.endswith("Api")]


@beartype
def _add_functions_to_module(
    module,
    api_name: str,
) -> list:
    """utility function to dynamically add functions to a module

    This function works by calling setattr on the module.

    Args:
        module (str): name of the module
        api_name (str): name of the API

    Returns:
        set of methods that were added
    """
    methods = _get_client_methods(
        _get_api_client(
            api_name=api_name,
            configure=False,
        )
    )

    sanitized_methods = []

    for method in methods:
        # clean up the name so that it's more readable
        # here we're removing the first word, which is the same as the api name
        sanitized_method_name = "_".join(method.split("_")[1:])

        sanitized_method_name = sanitized_method_name.replace(
            "_without_preload_content", ""
        ).lstrip("_")

        sanitized_methods.append(sanitized_method_name)

        # add this function as an attribute to this module
        # so that we can call it
        setattr(
            module,
            sanitized_method_name,
            _create_function(
                method_path=method,
                api_name=api_name,
            ),
        )

    return sanitized_methods


@beartype
def _get_api_client(
    *,
    api_name: str,
    configure: bool = True,
    token: Optional[str] = None,
    api_endpoint: Optional[str] = None,
):
    """return a configured client for the API we want to access

    Args:
        api_name (str): name of the API
        configure (bool): whether to configure the client with authentication

    Returns:
        configured client

    """

    if configure:
        if api_endpoint is None:
            api_endpoint = get_value()["api_endpoint"]

        if token is None:
            token = get_tokens()["access"]

        configuration = do_sdk_platform.Configuration(
            host=api_endpoint,
            access_token=token,
        )

        client = do_sdk_platform.ApiClient(configuration=configuration)
    else:
        client = do_sdk_platform.ApiClient()

    api_class = getattr(do_sdk_platform, api_name)
    client = api_class(api_client=client)
    return client


@beartype
def _get_client_methods(client) -> set:
    """utility function to get methods from the client that return raw responses from the server"""
    methods = set(
        [
            attr
            for attr in dir(client)
            if callable(getattr(client, attr))
            and not attr.startswith("_")
            and "without_preload_content" in attr
        ]
    )

    return methods


def _create_function(
    *,
    method_path: str,
    api_name: str,
):
    """utility function the dynamically creates functions
    that wrap low-level functions in the DeepOrigin data API"""

    # we're constructing a client solely for the purposes
    # of inspecting its methods and extracting
    # function signatures. So we don't need any
    # authentication
    client = _get_api_client(
        configure=False,
        api_name=api_name,
    )

    method = _get_method(client, method_path)

    signature = inspect.signature(method)

    def dynamic_function(
        *,
        client=None,
        **kwargs,
    ):
        """dynamic function that wraps low-level functions in the DeepOrigin platform API"""

        if client is None:
            client = _get_api_client(
                api_name=api_name,
            )
        method = _get_method(client, method_path)

        # Insert org_key if not present in kwargs, and
        # if it's required by the method
        method_sig = inspect.signature(method)
        if "org_key" in method_sig.parameters and (kwargs.get("org_key") is None):
            kwargs["org_key"] = get_value()["organization_id"]

        # call the low level API method
        response = method(**kwargs)

        if 400 <= response.status < 600:
            content = response.read().decode("utf-8", errors="replace")

            raise ValueError(
                f"HTTP request failed with status: {response.status} - {response.reason} - {content}"
            )

        if not isinstance(response, dict):
            response = response.json()

        if isinstance(response, bool):
            return response

        elif isinstance(response, dict):
            if "data" in response.keys():
                response = response["data"]
                if isinstance(response, list):
                    response = [Box(item) for item in response]
                else:
                    response = Box(response)
            else:
                response = Box(response)

        elif isinstance(response, list):
            response = [Box(item) for item in response]
        else:
            raise NotImplementedError(f"Unexpected response type: {type(response)}")

        return response

    # attach the signature of the underlying method to the
    # function so that IDEs can display it properly
    dynamic_function.__signature__ = signature

    return dynamic_function
