import click
from rich.console import Console
from rich.prompt import Prompt, Confirm
from rich.panel import Panel
from rich.progress import Progress, SpinnerColumn, TextColumn
from pathlib import Path
import sys

from .generator import ProjectGenerator
from .config import AVAILABLE_TEMPLATES
from .commands.add import add as add_group
from .commands.validate import register_commands
from .commands.env import env as env_group
from .commands.deps import deps as deps_group
from .utils.license_utils import generate_license
from .utils.gitignore_utils import generate_gitignore

console = Console()


@click.group()
@click.version_option(version="4.0.0", prog_name="Projex")
def main():
    """
    🔨 Projex - Instantly generate Python project boilerplates!
    """
    pass


@main.command()
@click.argument("project_name", required=False)
@click.option(
    "--template",
    "-t",
    type=click.Choice(["fastapi", "django", "flask", "bottle", "pyramid", "tornado", "sanic", "cherrypy"]),
    help="Choose the project template type",
)
@click.option("--path", "-p", default=".", help="Directory path to create the project")
@click.option("--author", "-a", help="Author name")
@click.option("--description", "-d", help="Project description")
@click.option(
    "--db",
    type=click.Choice(["postgresql", "mysql", "mongodb", "sqlite", "redis"]),
    default="postgresql",
    help="Database type (default: postgresql)",
)
@click.option(
    "--style",
    type=click.Choice(["minimal", "standard", "full"]),
    default="standard",
    help="Template style: minimal (bare minimum), standard (default), full (everything)",
)
@click.option(
    "--auth",
    type=click.Choice(["jwt", "oauth2", "apikey", "basic"]),
    help="Authentication method (jwt, oauth2, apikey, basic)",
)
@click.option("--no-git", is_flag=True, help="Skip git initialization")
@click.option("--no-venv", is_flag=True, help="Skip virtual environment creation")
@click.option(
    "--license",
    type=click.Choice(["mit", "apache", "gpl", "bsd", "unlicense"]),
    help="License type (mit, apache, gpl, bsd, unlicense)",
)
@click.option(
    "--gitignore",
    help="Comma-separated gitignore templates (e.g., python,venv,pycharm,vscode)",
)
def create(project_name, template, path, author, description, db, style, auth, no_git, no_venv, license, gitignore):
    """Create a new Python project using Projex templates."""

    console.print(
        Panel.fit(
            "[bold cyan]🔨 Projex[/bold cyan]\n"
            "[dim]Generate production-ready Python projects in seconds[/dim]",
            border_style="cyan",
        )
    )

    # Interactive mode if not provided via CLI
    if not project_name:
        project_name = Prompt.ask("[cyan]Project name[/cyan]")

    if not template:
        console.print("\n[yellow]Available templates:[/yellow]")
        for i, (key, value) in enumerate(AVAILABLE_TEMPLATES.items(), 1):
            console.print(f"  {i}. [cyan]{key}[/cyan] - {value['description']}")

        template_choice = Prompt.ask(
            "\n[cyan]Choose template[/cyan]",
            choices=["1", "2", "3", "4", "5", "6", "7", "8", "fastapi", "django", "flask", "bottle", "pyramid", "tornado", "sanic", "cherrypy"],
            default="1",
        )

        template_map = {
            "1": "fastapi", 
            "2": "django", 
            "3": "flask",
            "4": "bottle",
            "5": "pyramid",
            "6": "tornado",
            "7": "sanic",
            "8": "cherrypy"
        }
        template = template_map.get(template_choice, template_choice)

    if not author:
        author = Prompt.ask("[cyan]Author name[/cyan]", default="Developer")

    if not description:
        description = Prompt.ask(
            "[cyan]Project description[/cyan]",
            default=f"A {template} project generated by Projex",
        )

    # Database selection (interactive if not provided via CLI)
    # Note: db has a default value, so we check if it's explicitly None
    # Since click provides default, we'll make it optional in interactive mode
    # For now, we'll prompt if in interactive mode (when template is also prompted)
    if not template:  # Only prompt for DB if we're in full interactive mode
        console.print("\n[yellow]Available databases:[/yellow]")
        databases = ["postgresql", "mysql", "mongodb", "sqlite", "redis"]
        for i, db_name in enumerate(databases, 1):
            console.print(f"  {i}. [cyan]{db_name}[/cyan]")
        
        db_choice = Prompt.ask(
            "\n[cyan]Choose database[/cyan]",
            choices=["1", "2", "3", "4", "5", "postgresql", "mysql", "mongodb", "sqlite", "redis"],
            default="1",
        )
        
        db_map = {
            "1": "postgresql",
            "2": "mysql",
            "3": "mongodb",
            "4": "sqlite",
            "5": "redis"
        }
        db = db_map.get(db_choice, db_choice)

    # Show summary
    console.print("\n[bold yellow]Project Configuration:[/bold yellow]")
    console.print(f"  Name: [cyan]{project_name}[/cyan]")
    console.print(f"  Template: [cyan]{template}[/cyan]")
    console.print(f"  Database: [cyan]{db}[/cyan]")
    console.print(f"  Style: [cyan]{style}[/cyan]")
    if auth:
        console.print(f"  Authentication: [cyan]{auth}[/cyan]")
    console.print(f"  Path: [cyan]{Path(path).absolute() / project_name}[/cyan]")
    console.print(f"  Author: [cyan]{author}[/cyan]")
    console.print(f"  Git Init: [cyan]{'No' if no_git else 'Yes'}[/cyan]")
    console.print(f"  Virtual Env: [cyan]{'No' if no_venv else 'Yes'}[/cyan]")

    if not Confirm.ask("\n[cyan]Proceed with project creation?[/cyan]", default=True):
        console.print("[yellow]Cancelled by user.[/yellow]")
        return

    # Project generation logic
    try:
        generator = ProjectGenerator(
            project_name=project_name,
            template_type=template,
            base_path=path,
            author=author,
            description=description,
            database=db,
            style=style,
            auth=auth,
            init_git=not no_git,
            create_venv=not no_venv,
        )

        with Progress(
            SpinnerColumn(),
            TextColumn("[progress.description]{task.description}"),
            console=console,
        ) as progress:
            task = progress.add_task("[cyan]Creating project...", total=None)
            project_path = generator.generate()
            progress.update(task, completed=True)

        console.print(
            f"\n[bold green]✓[/bold green] Project created successfully at: [cyan]{project_path}[/cyan]"
        )
        
        # Post-generation: Add license if specified
        if license:
            try:
                license_file = generate_license(project_path, license, author)
                console.print(f"[green]✓[/green] License file created: {license_file.name}")
            except Exception as e:
                console.print(f"[yellow]Warning:[/yellow] Could not create license: {str(e)}")
        
        # Post-generation: Add custom gitignore if specified
        if gitignore:
            try:
                templates = [t.strip() for t in gitignore.split(',')]
                gitignore_file = generate_gitignore(project_path, templates)
                console.print(f"[green]✓[/green] Custom .gitignore created")
            except Exception as e:
                console.print(f"[yellow]Warning:[/yellow] Could not create custom gitignore: {str(e)}")

        # Post-creation guidance
        console.print("\n[yellow]Next steps:[/yellow]")
        console.print(f"  cd {project_name}")
        if not no_venv:
            console.print("  source venv/bin/activate  # (Windows: venv\\Scripts\\activate)")
        console.print("  pip install -r requirements.txt")

        if template == "fastapi":
            console.print("  uvicorn app.main:app --reload")
        elif template == "django":
            console.print("  python manage.py migrate")
            console.print("  python manage.py runserver")
        elif template == "flask":
            console.print("  python run.py")
        elif template == "bottle":
            console.print("  python app/main.py")
        elif template == "pyramid":
            console.print("  python run.py")
        elif template == "tornado":
            console.print("  python app/main.py")
        elif template == "sanic":
            console.print("  python app/main.py")
        elif template == "cherrypy":
            console.print("  python app/main.py")

    except Exception as e:
        console.print(f"[bold red]✗ Error:[/bold red] {str(e)}")
        sys.exit(1)


@main.command()
def list():
    """List all available Projex templates."""
    console.print("\n[bold cyan]Available Templates:[/bold cyan]\n")

    for name, details in AVAILABLE_TEMPLATES.items():
        console.print(
            Panel(
                f"[bold]{details['description']}[/bold]\n"
                f"[dim]Features: {', '.join(details['features'])}[/dim]",
                title=f"[cyan]{name}[/cyan]",
                border_style="cyan",
            )
        )


# Register add command group
main.add_command(add_group)

# Register env command group
main.add_command(env_group)

# Register deps command group
main.add_command(deps_group)

# Register validation and info commands
register_commands(main)


if __name__ == "__main__":
    main()
