"""Mwr module, containing the :class:`Mwr` class."""

from os import PathLike

import numpy.ma as ma
import numpy.typing as npt

from cloudnetpy import utils
from cloudnetpy.constants import G_TO_KG
from cloudnetpy.datasource import DataSource
from cloudnetpy.utils import interpolate_1d


class Mwr(DataSource):
    """Microwave radiometer class, child of DataSource.

    Args:
         full_path: Cloudnet Level 1b mwr file.

    """

    def __init__(self, full_path: str | PathLike) -> None:
        super().__init__(full_path)
        self._init_lwp_data()
        self._init_lwp_error()

    def interpolate_to_grid(self, time_grid: npt.NDArray, max_time: float = 1) -> None:
        for key, array in self.data.items():
            self.data[key].data = interpolate_1d(
                self.time, array.data, time_grid, max_time=max_time
            )

    def _init_lwp_data(self) -> None:
        lwp = self.dataset.variables["lwp"][:]
        if "lwp_quality_flag" in self.dataset.variables:
            quality_flag = self.dataset.variables["lwp_quality_flag"][:]
            lwp[quality_flag != 0] = ma.masked
        self.append_data(lwp, "lwp")

    def _init_lwp_error(self) -> None:
        random_error, bias = 0.25, 20
        lwp_error = utils.l2norm(self.data["lwp"][:] * random_error, bias * G_TO_KG)
        self.append_data(lwp_error, "lwp_error", units="kg m-2")
        self.data["lwp_error"].comment = (
            "This variable is a rough estimate of the one-standard-deviation\n"
            f"error in liquid water path, calculated as a combination of\n"
            f"a {bias} g m-2 linear error and a {round(random_error * 100)} %\n"
            "fractional error."
        )
