# v0.11.6 Migration Guide: Nested Object Filtering

**Release Date**: [TBD]
**Breaking Changes**: None
**New Features**: Nested object filtering for hybrid tables

## Overview

FraiseQL v0.11.6 introduces nested object filtering, allowing you to filter on nested JSONB objects in hybrid tables. This enables complex queries on nested data structures without requiring SQL JOINs.

## New Features

### Nested Object Filtering

You can now filter on nested objects stored in JSONB columns:

```python
# Before v0.11.6: Could only filter on top-level fields
where = {"status": {"eq": "active"}}

# v0.11.6+: Can filter on nested objects
where = {
    "machine": {
        "name": {"eq": "Server-01"}
    }
}
results = await repo.find("allocations", where=where)
```

This generates SQL: `WHERE data->'machine'->>'name' = 'Server-01'`

### Multiple Nesting Levels

Filter on deeply nested objects:

```python
where = {
    "location": {
        "address": {
            "city": {"eq": "Seattle"}
        }
    }
}
# SQL: WHERE data->'location'->'address'->>'city' = 'Seattle'
```

### Combined Filters

Mix flat and nested filters:

```python
where = {
    "status": {"eq": "active"},
    "machine": {
        "type": {"eq": "Server"},
        "power": {"gte": 100}
    }
}
```

### GraphQL WhereInput Support

Type-safe filtering with generated WhereInput classes:

```python
MachineWhereInput = create_graphql_where_input(Machine)
AllocationWhereInput = create_graphql_where_input(Allocation)

where = AllocationWhereInput(
    machine=MachineWhereInput(
        name=StringFilter(eq="Server-01")
    )
)
```

## Migration Steps

### 1. Update Dependencies

```bash
pip install fraiseql>=0.11.6
```

### 2. Test Existing Queries

All existing WHERE clause functionality continues to work unchanged. No breaking changes.

### 3. Add Nested Filtering (Optional)

Update your queries to use nested object filtering where beneficial:

```python
# Old approach: Required custom SQL or multiple queries
# New approach: Simple nested filter
where = {"machine": {"id": {"eq": machine_id}}}
```

### 4. Update GraphQL Resolvers

If using GraphQL WhereInput objects, they now support nested filtering automatically.

## Performance Considerations

- Nested filtering uses JSONB path operators (`->`, `->>`)
- Consider adding GIN indexes on JSONB columns for better performance
- For frequently queried nested paths, consider computed columns

## Examples

See `examples/nested_filtering_example.py` for complete working examples.

## Backward Compatibility

- ✅ All existing WHERE clause syntax continues to work
- ✅ No breaking changes to existing APIs
- ✅ Existing tests pass without modification
- ✅ Performance characteristics unchanged for non-nested queries

## Troubleshooting

### Common Issues

**Issue**: "Unsupported operator" errors
**Solution**: Ensure you're using valid operators (eq, neq, gt, gte, lt, lte, in, notin, contains, startswith, endswith, is_null)

**Issue**: Queries return no results
**Solution**: Verify JSONB structure matches your filter paths. Use database queries to inspect actual JSONB data.

**Issue**: Performance degradation
**Solution**: Add GIN indexes on JSONB columns: `CREATE INDEX idx_data ON table USING GIN (data)`

## Support

For questions about nested object filtering:
- Check the documentation: `docs/core/database-api.md`
- See examples: `examples/nested_filtering_example.py`
- File issues: https://github.com/fraiseql/fraiseql/issues
