"""Analysis of image series (base class)"""

# Standard library imports
from abc import abstractmethod
from pathlib import Path

# Nonstandard
from filo import FormattedAnalysisBase

# Local imports
from .results import ResultsBase


class Analysis(FormattedAnalysisBase):
    """Base class for analysis subclasses (GreyLevel, ContourTracking, etc.).

    Class attributes (to redefine in subclasses)
    --------------------------------------------

    Viewer : class
        (subclass of AnalysisViewer)
        Viewer class/subclasses that is used to display and inspect analysis data

    Formatter: class
        (subclass of Formatter)
        class used to format results spit out by the raw analysis into
        something storable/saveable by the Results class.

    Results : class
        (subclass of ResultsBase)
        Results class/subclasses that is used to store, save and load
        analysis data and metadata.
    """
    Viewer = None
    Formatter = None
    Results = ResultsBase

    def __init__(
        self,
        img_series,
        savepath=None,
    ):
        """Initialize Analysis object

        Parameters
        ----------
        img_series : ImgSeries or ImgStack object
            image series on which the analysis will be run

        savepath : str or Path object
            folder in which to save analysis data & metadata
                    (if not specified, the img_series savepath is used)
        """
        self.img_series = img_series
        savepath = Path(savepath) if savepath else img_series.savepath

        results = self.Results(savepath=savepath)
        formatter = self.Formatter(analysis=self) if self.Formatter else None
        viewer = self.Viewer(analysis=self) if self.Viewer else None

        super().__init__(
            data_series=img_series,
            results=results,
            formatter=formatter,
            viewer=viewer,
        )

    # ================= Subclassed Methods of AnalysisBase ===================

    def analyze(self, num, details=False):
        """Same as _analyze, but with num as input instead of img.

        Can be subclassed if necessary.

        Parameters
        ----------
        num : int
            file number identifier across the image file series

        details : bool
            whether to include more details (e.g. for debugging or live view)

        Returns
        -------
        dict
            data that can be used by formatter._store_data(), with at least
            the key 'num' indicating the data series identifier
        """
        img = self.img_series.read(num=num)
        data = self._analyze(img=img)
        data['num'] = num
        if details:
            data['image'] = img
        return data

    # ================== AnalysisBase methods to subclass ====================

    def _init_analysis(self):
        """Any necessary initialization outside of data storage preparation

        [OPTIONAL]
        """
        pass

    def _end_analysis(self):
        """Any necessary initialization outside of data storage preparation

        [OPTIONAL]
        """
        pass

    # ======================= New methods to subclass ========================

    @abstractmethod
    def _analyze(self, img):
        """Analysis process on single image. Must return a dict.

        Parameters
        ----------
        img : array-like
            image array to be analyzed (e.g. numpy array).

        details : bool
            whether to include more details (e.g. for debugging or live view)

        Returns
        -------
        dict
            dict of data, handled by formatter._store_data()

        Define in subclasses."""
        pass

    # ========================== New public methods ==========================

    def regenerate(self, filename=None):
        """Load saved data, metadata and regenerate objects from them.

        Is used to reset the system in a state similar to the end of the
        analysis that was made before saving the results.

        Parameters
        ----------
        filename : str
            name of the analysis results file (if None, use default)

        Notes
        -----
            More or less equivalent to:
            >>> analysis.results.load(filename=filename)
            >>> image_series.load_transforms()
            (except that transforms are loaded from the metadata file of the
            analysis, not from a file generated by
            image_series.save_transforms())
        """
        # load data from files
        self.results.load(filename=filename)

        # re-apply transforms (rotation, crop etc.)
        # Note: if there are corrections, they need to be loaded manually
        # because typically all corrections parameters and data are
        # not stored in metadata.
        for name, transform in self.img_series.transforms.items():
            transform.reset()
            transform_data = self.results.metadata.get('transforms', {})
            transform.data = transform_data.get(name, {})
