import asyncio
import contextvars
import functools
import importlib.util
import logging
import os
import sys
from pathlib import Path
from typing import TYPE_CHECKING, Protocol, Union, runtime_checkable

import click
import matplotlib.pyplot as plt
import yaml
from autogen_agentchat.agents import AssistantAgent, UserProxyAgent
from autogen_agentchat.base import TaskResult
from autogen_agentchat.base._termination import OrTerminationCondition
from autogen_agentchat.conditions import TextMentionTermination
from autogen_core.models import RequestUsage
from autogen_ext.cache_store.diskcache import DiskCacheStore
from autogen_ext.models.cache import CHAT_CACHE_VALUE_TYPE, ChatCompletionCache
from autogen_ext.models.openai import OpenAIChatCompletionClient
from diskcache import Cache

from .logger import DVLogger, log_to_console, log_to_json_file, log_to_markdown_file
from .patched import (
    AssistantAgentWithStructuredToolCalling,
    MagenticOneOrchestratorPatch,
    PlotPatcher,
)
from .utils import DataVoyagerShell
from .models import ToolCall

if TYPE_CHECKING:
    from autogen_agentchat.teams import MagenticOneGroupChat, SelectorGroupChat


Orchestrator = Union["SelectorGroupChat", "MagenticOneGroupChat"]
dv_shell = DataVoyagerShell()

logger = logging.getLogger(__name__)


@runtime_checkable
class UsageTrackingClientProtocol(Protocol):
    def total_usage(self) -> RequestUsage:
        """
        Return the total token usage of the completion client
        """
        ...


class Config:
    """
    Parse the config file and return
    the config as a dictionary

    Parameters
    ----------
    config_file (str): path to the config file

    Returns
    -------
    config (dict): config as a dictionary
    """

    def __init__(self, config_file: str) -> None:
        self._config_file: str = config_file
        self._config: dict = self._parse_config()

    def _parse_config(self) -> dict:
        with open(self._config_file, "r") as stream:
            try:
                return yaml.safe_load(stream)
            except yaml.YAMLError as exc:
                print(exc)

    def get_config(self) -> dict:
        return self._config


class DataVoyager:
    """
    DataVoyager Agent

    Parameters
    ----------
    config_file (str): path to the config file
    log_file (str): path to the log file
    enable_markdown_logging (bool): enable markdown logging

    Returns
    -------
    None
    """

    def __init__(
        self,
        config_file: str,
        log_file: str,
        enable_markdown_logging: bool = False,
        enable_console_logging: bool = False,
    ) -> None:
        """
        Initialize the DataVoyager Agent

        Parameters
        ----------
        config_file (str): path to the config file
        log_file (str): path to the log file
        enable_markdown_logging (bool): enable markdown logging

        Returns
        -------
        None
        """
        self._config = Config(config_file).get_config()
        self.log_file = log_file
        self.enable_markdown_logging = enable_markdown_logging
        self.enable_console_logging = enable_console_logging

        # Keep track of all completion clients so we can get the usage later
        self._completion_clients: list[UsageTrackingClientProtocol] = []

        # TODO: use UUID4 or some other unique identifier
        if self.enable_markdown_logging:
            self.md_logger = DVLogger(
                logger_name=f"dv_{log_file}_md",
                log_filename=log_file.split(".")[0] + ".md",
                use_json_formatter=False,
            )
        if self.enable_console_logging:
            self.console_logger = DVLogger(
                logger_name=f"dv_{log_file}_console",
                log_filename=None,
                use_json_formatter=False,
            )
        self.json_logger = DVLogger(
            logger_name=f"dv_{log_file}_json", log_filename=log_file
        )

        self._tool_config: dict = self._config["tools"]
        self._agent_config: dict = self._config["agents"]
        self._orchestrator_config: dict = self._config["orchestrator"]
        self._error_handling: dict = self._config["error_handling"]

        self.cache_dir = (
            self._config["cache_dir"] if "cache_dir" in self._config else ".cache"
        )
        os.makedirs(self.cache_dir, exist_ok=True)
        self.cache_store = DiskCacheStore[CHAT_CACHE_VALUE_TYPE](Cache(self.cache_dir))

        self.tools: dict = self._get_tools()
        self.agents: dict = self._create_agents()
        self.orchestrator: Orchestrator = self._create_orchestrator()

        # NOTE: If we create generalized tool for
        # patching then make a separate function
        # Patch the plot function
        plot_patcher_client = self._make_openai_completion_client(
            model="gpt-4o-2024-05-13",
            seed=42,
            api_key=os.getenv("OPENAI_API_KEY"),
        )
        plot_patcher = PlotPatcher(
            model_client=plot_patcher_client,
            console_logger=self.json_logger,
        )
        plt.show = plot_patcher.get_patched_plt_show()

        # inject tools in the environment
        self._inject_tools()

    def get_tools(self) -> dict:
        return self.tools

    def get_agents(self) -> dict:
        return self.agents

    def get_orchestrator(self) -> Orchestrator:
        return self.orchestrator

    def get_config(self) -> dict:
        return self._config

    def _register_completion_client(self, client: UsageTrackingClientProtocol):
        """
        Register a completion client to track usage

        Parameters
        ----------
        client (UsageTrackingClientProtocol): completion client to register

        Returns
        -------
        None
        """
        self._completion_clients.append(client)

    def _make_openai_completion_client(
        self, *args, cache: bool = False, **kwargs
    ) -> OpenAIChatCompletionClient | ChatCompletionCache:
        """
        Create an OpenAI completion client, optionally with cache, and register it for usage tracking

        Parameters
        ----------
        *args: positional arguments for the client
        cache (bool): whether to use cache
        **kwargs: keyword arguments for the client

        Returns
        -------
        UsageTrackingClientProtocol: OpenAI completion client
        """
        model_client = OpenAIChatCompletionClient(*args, **kwargs)
        if cache:
            model_client = ChatCompletionCache(model_client, self.cache_store)
        self._register_completion_client(model_client)
        return model_client

    def _create_agents(self) -> dict:
        """
        Create agents from the agent config

        Parameters
        ----------
        None

        Returns
        -------
        agents (dict): dictionary of agents
        """
        agents = {}
        config = self._agent_config
        for agent_name, agent_config in config.items():
            if agent_name == "user":
                agent_config["input_func"] = (
                    self.tools[agent_name]
                    if agent_name in self.tools
                    else lambda val: input(val)
                )
                agents[agent_name] = UserProxyAgent(**agent_config)
            elif (
                agent_name == "programmer"
                and isinstance(agent_config["model_client"]["model"], str)
                and agent_config["model_client"]["model"].startswith("o")
            ):
                logger.warning(
                    "OpenAI's reasoning model has known tool calling issues. "
                    "Forcing structured output via patch to ensure reliable execution."
                )

                model_client = self._make_openai_completion_client(
                    **agent_config["model_client"],
                    cache=agent_config.pop("cache", False),
                )
                agent_config["model_client"] = model_client

                agent_config["output_content_type"] = ToolCall
                agent_config["tools"] = (
                    self.tools[agent_name] if agent_name in self.tools else None
                )
                agents[agent_name] = AssistantAgentWithStructuredToolCalling(
                    **agent_config
                )
            else:
                model_client = self._make_openai_completion_client(
                    **agent_config["model_client"],
                    cache=agent_config.pop("cache", False),
                )

                agent_config["model_client"] = model_client
                agent_config["tools"] = (
                    self.tools[agent_name] if agent_name in self.tools else None
                )
                agents[agent_name] = AssistantAgent(**agent_config)
        return agents

    def _create_orchestrator(self) -> Orchestrator:
        """
        Create orchestrator from the config

        Parameters
        ----------
        None

        Returns
        -------
        orchestrator (Orchestrator): orchestrator instance
        """
        config = self._orchestrator_config
        model_client = self._make_openai_completion_client(
            **config["model_client"],
            cache=config.pop("cache", False),
        )
        config["model_client"] = model_client

        termination: OrTerminationCondition = None
        for condition in config["termination_condition"]:
            if termination is None:
                termination = TextMentionTermination(condition)
            termination = termination | TextMentionTermination(condition)
        config["termination_condition"] = termination
        config["participants"] = [self.agents[name] for name in self.agents]

        orchestrator_type = config.pop("type")
        if orchestrator_type == "default":
            from autogen_agentchat.teams import SelectorGroupChat

            if hasattr(config, "prompts"):
                config.pop("prompts")  # No use of prompts for default

            return SelectorGroupChat(**config)

        elif orchestrator_type == "magentic-one":
            prompts = config.pop("prompts")

            from autogen_agentchat.teams._group_chat._magentic_one._magentic_one_orchestrator import (
                MagenticOneOrchestrator,
            )

            from .patched import MagenticOnePromptPatcher

            # patch the prompts
            patcher = MagenticOnePromptPatcher(prompts)

            # patch the orchestrator method
            MagenticOneOrchestrator._thread_to_context = (
                MagenticOneOrchestratorPatch._thread_to_context
            )

            for key in prompts:
                setattr(
                    MagenticOneOrchestrator,
                    patcher.patched_methods[key].__name__,
                    patcher.get_patched_method(key),
                )

            from autogen_agentchat.teams import MagenticOneGroupChat

            return MagenticOneGroupChat(**config)

        else:
            raise NotImplementedError

    def _get_tools(self) -> dict:
        """
        Load tools from the config

        Parameters
        ----------
        None

        Returns
        -------
        tools (dict): dictionary of agent names and their tools
        """
        tools = {}
        config = self._tool_config
        suppress_errors = self._error_handling["suppress_missing_tool_errors"]
        for agent_name, tool_config in config.items():
            if tool_config is None:
                continue
            agent_tools = []
            for tool in tool_config:
                script_path = tool.split("::")[0]
                function_name = tool.split("::")[1]
                tool = self._load_tool(
                    script_path=script_path,
                    function_name=function_name,
                    suppress_errors=suppress_errors,
                )
                if not tool:
                    continue
                agent_tools.append(tool)
            tools[agent_name] = agent_tools
        return tools

    def _inject_tools(
        self,
    ) -> None:
        """
        Inject tools into the python environment

        Parameters
        ----------
        None

        Returns
        -------
        None
        """
        if "injected_functions" not in self._config:
            return
        injected_functions = self._config["injected_functions"]
        suppress_errors = self._error_handling["suppress_missing_tool_errors"]

        for function in injected_functions:
            script_path = function.split("::")[0]
            function_name = function.split("::")[1]
            tool = self._load_tool(
                script_path=script_path,
                function_name=function_name,
                suppress_errors=suppress_errors,
            )
            if not tool:
                continue
            dv_shell.inject_tool(tool_name=function_name, tool=tool)

    def _load_tool(
        self, script_path: str, function_name: str, suppress_errors: bool = False
    ) -> callable:
        """
        Load the tool from the script path

        Parameters
        ----------
        script_path (str): path to the script
        function_name (str): name of the function to load
        suppress_errors (bool): suppress errors

        Returns
        -------
        tool (callable): tool function
        """
        try:
            module_name = script_path.split("/")[-1].split(".")[0]
            spec = importlib.util.spec_from_file_location(
                name=module_name, location=str(Path(__file__).parent / script_path)
            )
            module = importlib.util.module_from_spec(spec)
            sys.modules[module_name] = module
            spec.loader.exec_module(module)

            # Need to preserve original context in case libs like inspect_ai used ContextVars
            ctx = contextvars.copy_context()

            @functools.wraps(getattr(module, function_name))
            def fn_in_context(*args, **kwargs):
                try:
                    return ctx.copy().run(
                        getattr(module, function_name), *args, **kwargs
                    )
                except Exception as e:
                    logger.error(f"Error in tool {function_name}: {str(e)}")
                    if suppress_errors:
                        return {"internal_tool_error": str(e)}
                    raise e
                return {
                    "internal_tool_error": "Reached unreachable code branch.  This should never happen."
                }

            return fn_in_context
        except Exception as e:
            if suppress_errors:
                return None
            raise e

    async def run(self, query: str) -> str:
        """
        Run the agent with the given query

        Parameters
        ----------
        query (str): query to run

        Returns
        -------
        str: The final response from the agent
        """
        if "domain_knowledge" in self._config:
            domain_knowledge = self._config["domain_knowledge"]
            query = (
                "# Query:\n"
                + query
                + "\n\n"
                + "# Domain Knowledge:\n"
                + domain_knowledge
            )

        final_response = ""

        stream = self.orchestrator.run_stream(task=query)
        async for message in stream:
            if self.enable_markdown_logging:
                log_to_markdown_file(self.md_logger, message)
            if self.enable_console_logging:
                log_to_console(self.console_logger, message)
            log_to_json_file(self.json_logger, message)

            # Could be a risk that an agent gives the final resp and
            # a different agent then says 'TERMINATE'; should consider a more
            # robust solution
            if isinstance(message, TaskResult):
                final_response = message.messages[-1].content

        return final_response

    def total_usage(self) -> dict[str, RequestUsage]:
        """
        Get the total usage of all completion clients.

        Returns: a dictionary mapping model name to total RequestUsage
        """
        total_usages = dict()

        for client in self._completion_clients:
            if not isinstance(client, UsageTrackingClientProtocol):
                logger.warning(
                    f"Client {client} does not implement total_usage method, skipping."
                )
                continue

            model_name: str = "unknown"
            if isinstance(client, OpenAIChatCompletionClient):
                model_name = client.model_info.get("family", model_name)
            elif isinstance(client, ChatCompletionCache):
                model_name = client.model_info.get("family", model_name)

                logger.warning(
                    "Trying to get usage from a cached completion client; this reports 0 token usage when requests are found in the cache."
                )
            elif hasattr(client, "model_info"):
                logger.warning(
                    f"Client {client} is unknown but has 'model_info' attribute; using it for model name."
                )
                model_name = client.model_info.get("family", model_name)
            elif hasattr(client, "model"):
                logger.warning(
                    f"Client {client} is unknown but has 'model' attribute; using it for model name."
                )
                model_name = client.model

            if model_name == "unknown" or not model_name:
                logger.warning(f"Could not identify model name from client; skipping.")
                continue

            total_usages.setdefault(
                model_name, RequestUsage(prompt_tokens=0, completion_tokens=0)
            )

            client_usage = client.total_usage()
            total_usages[model_name].prompt_tokens += client_usage.prompt_tokens
            total_usages[model_name].completion_tokens += client_usage.completion_tokens
        return total_usages


@click.command()
@click.option(
    "--config_file",
    default="./config/default_config.yaml",
    help="Path to the config file",
)
@click.option("--log_file", default="./tmp.json", help="Path to the log file")
@click.option(
    "--enable_markdown_logging",
    is_flag=True,
    default=False,
    help="Enable markdown logging",
)
@click.option(
    "--enable_console_logging",
    is_flag=True,
    default=False,
    help="Enable console logging [Works for Jupiter notebook]",
)
@click.argument("query")
def main(
    config_file: str,
    log_file: str,
    enable_markdown_logging: bool,
    enable_console_logging: bool,
    query: str,
):
    agent = DataVoyager(
        config_file=config_file,
        log_file=log_file,
        enable_markdown_logging=enable_markdown_logging,
        enable_console_logging=enable_console_logging,
    )
    asyncio.run(agent.run(query=query))


if __name__ == "__main__":
    main()
