"""Factory to create native tools for model providers"""

import logging

from inspect_ai._util.registry import registry_info
from inspect_ai.model import get_model
from inspect_ai.model._providers._openai_web_search import maybe_web_search_tool
from inspect_ai.model._providers.anthropic import AnthropicAPI
from inspect_ai.model._providers.google import GoogleGenAIAPI
from inspect_ai.model._providers.openai import OpenAIAPI
from inspect_ai.model._providers.perplexity import PerplexityAPI
from inspect_ai.tool import Tool, web_search
from inspect_ai.tool._tool import TOOL_OPTIONS
from inspect_ai.tool._tool_info import ToolInfo, parse_tool_info

logger = logging.getLogger(__name__)


def parse_tool_info_with_options(tool: Tool) -> ToolInfo:
    """Parse tool info including options from registry metadata.

    This is a workaround for the bug in inspect_ai's parse_tool_info function
    which doesn't extract options from the registry metadata.
    """
    # Get basic tool info
    tool_info = parse_tool_info(tool)

    # Get options from registry metadata
    try:
        info = registry_info(tool)
        options = info.metadata.get(TOOL_OPTIONS, None)
        tool_info.options = options
    except Exception:
        # If we can't get registry info, leave options as None
        pass

    return tool_info


def make_native_search_tools(inserted_before: str | None = None) -> list[Tool]:
    """Create native tools for the current model provider.

    Args:
        inserted_before (str): A string indicating the cutoff date for the search tools.
        Only Perplexity native search tools will support this parameter.

    Returns a list of search tools supported by the current model provider:
    - For OpenAI: web search with high context size for compatible models
    - For Anthropic: web search for compatible models

    Raises RuntimeError if the current model doesn't support native search tools.
    """
    model = get_model()
    if isinstance(model.api, OpenAIAPI):
        tool = web_search({"openai": {"search_context_size": "high"}})
        # This function checks for the subset of supported models
        tool_info = parse_tool_info_with_options(tool)
        if maybe_web_search_tool(model.name, tool_info):
            return [tool]
    elif isinstance(model.api, AnthropicAPI):
        tool = web_search({"anthropic": True})
        tool_info = parse_tool_info_with_options(tool)
        if model.api.web_search_tool_param(tool_info) is not None:
            return [tool]
    elif isinstance(model.api, GoogleGenAIAPI):
        tool = web_search({"gemini": True})
        tool_info = parse_tool_info_with_options(tool)
        if model.api.chat_tools([tool_info]) is not None:
            return [tool]
    elif isinstance(model.api, PerplexityAPI):
        perplexity_options = {
            "search_mode": "academic",
            "web_search_options": {"search_context_size": "high"},
        }
        if inserted_before:
            perplexity_options["search_before_date_filter"] = inserted_before
        tool = web_search({"perplexity": perplexity_options})
        return [tool]

    raise RuntimeError(
        f"No native search tools defined for {model.api.__class__.__name__} ({model.name})"
    )


__all__ = ["make_native_search_tools"]
