# Copyright Kevin Deldycke <kevin@deldycke.com> and contributors.
#
# This program is Free Software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

from __future__ import annotations

from textwrap import dedent

from gha_utils.cli import remove_header
from gha_utils.mailmap import Mailmap, Record


def test_remove_header():
    content = dedent("""\
        # Generated by gha-utils mailmap-sync v4.4.3 - https://github.com/kdeldycke/workflows
        # Timestamp: 2024-08-12T08:15:41.083405
        # Format is:
        #   Preferred Name <preferred e-mail> Other Name <other e-mail>
        #
        # Reference: https://git-scm.com/docs/git-blame#_mapping_authors

        GitHub <noreply@github.com>
        """)

    assert remove_header(content) == dedent("""\
        # Format is:
        #   Preferred Name <preferred e-mail> Other Name <other e-mail>
        #
        # Reference: https://git-scm.com/docs/git-blame#_mapping_authors

        GitHub <noreply@github.com>""")


def test_mailmap():
    mailmap = Mailmap()

    mailmap.parse(
        dedent(
            """

            github-actions[bot] <41898282+github-actions[bot]@users.noreply.github.com> GitHub Actions <actions@github.com>
            GitHub <noreply@github.com>

            GitHub <noreply@github.com>

            GitHub <noreply@github.com>                   dependabot[bot] <49699333+dependabot[bot]@users.noreply.github.com>
            GitHub <noreply@github.com>         GitHub <noreply@github.com>
            """
        )
    )

    assert mailmap.render() == dedent(
        """\
        GitHub <noreply@github.com>
        GitHub <noreply@github.com>
        GitHub <noreply@github.com> dependabot[bot] <49699333+dependabot[bot]@users.noreply.github.com>
        GitHub <noreply@github.com>
        github-actions[bot] <41898282+github-actions[bot]@users.noreply.github.com> GitHub Actions <actions@github.com>"""
    )


def test_new_entry():
    mailmap = Mailmap()

    header = dedent(
        """\
        # Generated by gha-utils mailmap-sync v4.16.2 - https://github.com/kdeldycke/workflows
        # Timestamp: 2025-05-13T13:14:51.128939
        # Format is:
        #   Preferred Name <preferred e-mail> Other Name <other e-mail>
        #
        # Reference: https://git-scm.com/docs/git-blame#_mapping_authors"""
    )

    mailmap.parse(
        dedent(
            f"""\
            {header}
            dependabot[bot] <49699333+dependabot[bot]@users.noreply.github.com>


            """
        )
    )

    assert len(mailmap.records) == 1
    assert (
        mailmap.records[0].canonical
        == "dependabot[bot] <49699333+dependabot[bot]@users.noreply.github.com>"
    )
    assert mailmap.records[0].aliases == set()
    assert mailmap.records[0].pre_comment == header

    # Add new record manually
    new_contributor = "AA BB <aa@bb.example.org>"
    mailmap.records.append(Record(canonical=new_contributor))

    assert mailmap.render() == (
        header
        + "\n"
        + new_contributor
        + "\n"
        + "dependabot[bot] <49699333+dependabot[bot]@users.noreply.github.com>"
    )
