import { AccessLevelList } from '../shared/access-level';
import { PolicyStatement, Operator } from '../shared';
import { aws_iam as iam } from "aws-cdk-lib";
/**
 * Statement provider for service [eks](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonelastickubernetesservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class Eks extends PolicyStatement {
    servicePrefix: string;
    /**
     * Grants permission to view Kubernetes objects via AWS EKS console
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/view-workloads.html
     */
    toAccessKubernetesApi(): this;
    /**
     * Grants permission to associate encryption configuration to a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_AssociateEncryptionConfig.html
     */
    toAssociateEncryptionConfig(): this;
    /**
     * Grants permission to associate an identity provider configuration to a cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifClientId()
     * - .ifIssuerUrl()
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_AssociateIdentityProviderConfig.html
     */
    toAssociateIdentityProviderConfig(): this;
    /**
     * Grants permission to create an Amazon EKS add-on
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_CreateAddon.html
     */
    toCreateAddon(): this;
    /**
     * Grants permission to create an Amazon EKS cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_CreateCluster.html
     */
    toCreateCluster(): this;
    /**
     * Grants permission to create an AWS Fargate profile
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_CreateFargateProfile.html
     */
    toCreateFargateProfile(): this;
    /**
     * Grants permission to create an Amazon EKS Nodegroup
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_CreateNodegroup.html
     */
    toCreateNodegroup(): this;
    /**
     * Grants permission to delete an Amazon EKS add-on
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DeleteAddon.html
     */
    toDeleteAddon(): this;
    /**
     * Grants permission to delete an Amazon EKS cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DeleteCluster.html
     */
    toDeleteCluster(): this;
    /**
     * Grants permission to delete an AWS Fargate profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DeleteFargateProfile.html
     */
    toDeleteFargateProfile(): this;
    /**
     * Grants permission to delete an Amazon EKS Nodegroup
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DeleteNodegroup.html
     */
    toDeleteNodegroup(): this;
    /**
     * Grants permission to deregister an External cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DeregisterCluster.html
     */
    toDeregisterCluster(): this;
    /**
     * Grants permission to retrieve descriptive information about an Amazon EKS add-on
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DescribeAddon.html
     */
    toDescribeAddon(): this;
    /**
     * Grants permission to retrieve descriptive version information about the add-ons that Amazon EKS Add-ons supports
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DescribeAddonVersions.html
     */
    toDescribeAddonVersions(): this;
    /**
     * Grants permission to retrieve descriptive information about an Amazon EKS cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DescribeCluster.html
     */
    toDescribeCluster(): this;
    /**
     * Grants permission to retrieve descriptive information about an AWS Fargate profile associated with a cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DescribeFargateProfile.html
     */
    toDescribeFargateProfile(): this;
    /**
     * Grants permission to retrieve descriptive information about an Idp config associated with a cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DescribeIdentityProviderConfig.html
     */
    toDescribeIdentityProviderConfig(): this;
    /**
     * Grants permission to retrieve descriptive information about an Amazon EKS nodegroup
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DescribeNodegroup.html
     */
    toDescribeNodegroup(): this;
    /**
     * Grants permission to retrieve a given update for a given Amazon EKS cluster/nodegroup/add-on (in the specified or default region)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DescribeUpdate.html
     */
    toDescribeUpdate(): this;
    /**
     * Grants permission to delete an asssociated Idp config
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DisassociateIdentityProviderConfig.html
     */
    toDisassociateIdentityProviderConfig(): this;
    /**
     * Grants permission to list the Amazon EKS add-ons in your AWS account (in the specified or default region) for a given cluster
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_ListAddons.html
     */
    toListAddons(): this;
    /**
     * Grants permission to list the Amazon EKS clusters in your AWS account (in the specified or default region)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_ListClusters.html
     */
    toListClusters(): this;
    /**
     * Grants permission to list the AWS Fargate profiles in your AWS account (in the specified or default region) associated with a given cluster
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_ListFargateProfiles.html
     */
    toListFargateProfiles(): this;
    /**
     * Grants permission to list the Idp configs in your AWS account (in the specified or default region) associated with a given cluster
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_ListIdentityProviderConfigs.html
     */
    toListIdentityProviderConfigs(): this;
    /**
     * Grants permission to list the Amazon EKS nodegroups in your AWS account (in the specified or default region) attached to given cluster
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_ListNodegroups.html
     */
    toListNodegroups(): this;
    /**
     * Grants permission to list tags for the specified resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource(): this;
    /**
     * Grants permission to list the updates for a given Amazon EKS cluster/nodegroup/add-on (in the specified or default region)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_ListUpdates.html
     */
    toListUpdates(): this;
    /**
     * Grants permission to register an External cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_RegisterCluster.html
     */
    toRegisterCluster(): this;
    /**
     * Grants permission to tag the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_TagResource.html
     */
    toTagResource(): this;
    /**
     * Grants permission to untag the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_UntagResource.html
     */
    toUntagResource(): this;
    /**
     * Grants permission to update Amazon EKS add-on configurations, such as the VPC-CNI version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_UpdateAddon.html
     */
    toUpdateAddon(): this;
    /**
     * Grants permission to update Amazon EKS cluster configurations (eg: API server endpoint access)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_UpdateClusterConfig.html
     */
    toUpdateClusterConfig(): this;
    /**
     * Grants permission to update the Kubernetes version of an Amazon EKS cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_UpdateClusterVersion.html
     */
    toUpdateClusterVersion(): this;
    /**
     * Grants permission to update Amazon EKS nodegroup configurations (eg: min/max/desired capacity or labels)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_UpdateNodegroupConfig.html
     */
    toUpdateNodegroupConfig(): this;
    /**
     * Grants permission to update the Kubernetes version of an Amazon EKS nodegroup
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_UpdateNodegroupVersion.html
     */
    toUpdateNodegroupVersion(): this;
    protected accessLevelList: AccessLevelList;
    /**
     * Adds a resource of type cluster to the statement
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/clusters.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCluster(clusterName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type nodegroup to the statement
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/managed-node-groups.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param nodegroupName - Identifier for the nodegroupName.
     * @param uUID - Identifier for the uUID.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onNodegroup(clusterName: string, nodegroupName: string, uUID: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type addon to the statement
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/managed-addons.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param addonName - Identifier for the addonName.
     * @param uUID - Identifier for the uUID.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAddon(clusterName: string, addonName: string, uUID: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type fargateprofile to the statement
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/fargate-profile.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param fargateProfileName - Identifier for the fargateProfileName.
     * @param uUID - Identifier for the uUID.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFargateprofile(clusterName: string, fargateProfileName: string, uUID: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type identityproviderconfig to the statement
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/authenticate-oidc-identity-provider.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param identityProviderType - Identifier for the identityProviderType.
     * @param identityProviderConfigName - Identifier for the identityProviderConfigName.
     * @param uUID - Identifier for the uUID.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIdentityproviderconfig(clusterName: string, identityProviderType: string, identityProviderConfigName: string, uUID: string, account?: string, region?: string, partition?: string): this;
    /**
     * Filters access by the clientId present in the associateIdentityProviderConfig request the user makes to the EKS service
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies
     *
     * Applies to actions:
     * - .toAssociateIdentityProviderConfig()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifClientId(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the issuerUrl present in the associateIdentityProviderConfig request the user makes to the EKS service
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies
     *
     * Applies to actions:
     * - .toAssociateIdentityProviderConfig()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifIssuerUrl(value: string | string[], operator?: Operator | string): this;
    /**
     * Statement provider for service [eks](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonelastickubernetesservice.html).
     *
     */
    constructor(props?: iam.PolicyStatementProps);
}
//# sourceMappingURL=elastickubernetesservice.d.ts.map