import { AccessLevelList } from '../shared/access-level';
import { PolicyStatement, Operator } from '../shared';
import { aws_iam as iam } from "aws-cdk-lib";
/**
 * Statement provider for service [refactor-spaces](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsmigrationhubrefactorspaces.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class RefactorSpaces extends PolicyStatement {
    servicePrefix: string;
    /**
     * Grants permission to create an application within an environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_CreateApplication.html
     */
    toCreateApplication(): this;
    /**
     * Grants permission to create an environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_CreateEnvironment.html
     */
    toCreateEnvironment(): this;
    /**
     * Grants permission to create a route within an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifServiceCreatedByAccount()
     * - .ifRouteCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifSourcePath()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_CreateRoute.html
     */
    toCreateRoute(): this;
    /**
     * Grants permission to create a service within an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifServiceCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_CreateService.html
     */
    toCreateService(): this;
    /**
     * Grants permission to delete an application from an environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_DeleteApplication.html
     */
    toDeleteApplication(): this;
    /**
     * Grants permission to delete an environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_DeleteEnvironment.html
     */
    toDeleteEnvironment(): this;
    /**
     * Grants permission to delete a resource policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_DeleteResourcePolicy.html
     */
    toDeleteResourcePolicy(): this;
    /**
     * Grants permission to delete a route from an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifServiceCreatedByAccount()
     * - .ifRouteCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifSourcePath()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_DeleteRoute.html
     */
    toDeleteRoute(): this;
    /**
     * Grants permission to delete a service from an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifServiceCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_DeleteService.html
     */
    toDeleteService(): this;
    /**
     * Grants permission to get more information about an application
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_GetApplication.html
     */
    toGetApplication(): this;
    /**
     * Grants permission to get more information for an environment
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_GetEnvironment.html
     */
    toGetEnvironment(): this;
    /**
     * Grants permission to get the details about a resource policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_GetResourcePolicy.html
     */
    toGetResourcePolicy(): this;
    /**
     * Grants permission to get more information about a route
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifServiceCreatedByAccount()
     * - .ifRouteCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifSourcePath()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_GetRoute.html
     */
    toGetRoute(): this;
    /**
     * Grants permission to get more information about a service
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifServiceCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_GetService.html
     */
    toGetService(): this;
    /**
     * Grants permission to list all the applications in an environment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_ListApplications.html
     */
    toListApplications(): this;
    /**
     * Grants permission to list all the VPCs for the environment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_ListEnvironmentVpcs.html
     */
    toListEnvironmentVpcs(): this;
    /**
     * Grants permission to list all environments
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_ListEnvironment.html
     */
    toListEnvironments(): this;
    /**
     * Grants permission to list all the routes in an application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_ListRoutes.html
     */
    toListRoutes(): this;
    /**
     * Grants permission to list all the services in an environment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_ListServices.html
     */
    toListServices(): this;
    /**
     * Grants permission to list all the tags for a given resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource(): this;
    /**
     * Grants permission to add a resource policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_PutResourcePolicy.html
     */
    toPutResourcePolicy(): this;
    /**
     * Grants permission to tag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifServiceCreatedByAccount()
     * - .ifRouteCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifSourcePath()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_TagResource.html
     */
    toTagResource(): this;
    /**
     * Grants permission to remove a tag from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifServiceCreatedByAccount()
     * - .ifRouteCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifSourcePath()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_UntagResource.html
     */
    toUntagResource(): this;
    protected accessLevelList: AccessLevelList;
    /**
     * Adds a resource of type environment to the statement
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-resources
     *
     * @param environmentId - Identifier for the environmentId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEnvironment(environmentId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type application to the statement
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-resources
     *
     * @param environmentId - Identifier for the environmentId.
     * @param applicationId - Identifier for the applicationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifApplicationCreatedByAccount()
     * - .ifCreatedByAccountIds()
     */
    onApplication(environmentId: string, applicationId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type service to the statement
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-resources
     *
     * @param environmentId - Identifier for the environmentId.
     * @param applicationId - Identifier for the applicationId.
     * @param serviceId - Identifier for the serviceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifApplicationCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifServiceCreatedByAccount()
     */
    onService(environmentId: string, applicationId: string, serviceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type route to the statement
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-resources
     *
     * @param environmentId - Identifier for the environmentId.
     * @param applicationId - Identifier for the applicationId.
     * @param routeId - Identifier for the routeId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifApplicationCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifRouteCreatedByAccount()
     * - .ifServiceCreatedByAccount()
     * - .ifSourcePath()
     */
    onRoute(environmentId: string, applicationId: string, routeId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Filters access by restricting the action to only those accounts that created the application within an environment
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * Applies to actions:
     * - .toCreateApplication()
     * - .toCreateRoute()
     * - .toCreateService()
     * - .toDeleteApplication()
     * - .toDeleteRoute()
     * - .toDeleteService()
     * - .toGetApplication()
     * - .toGetRoute()
     * - .toGetService()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * Applies to resource types:
     * - application
     * - service
     * - route
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifApplicationCreatedByAccount(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the accounts that created the resource
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * Applies to actions:
     * - .toCreateApplication()
     * - .toCreateRoute()
     * - .toCreateService()
     * - .toDeleteApplication()
     * - .toDeleteRoute()
     * - .toDeleteService()
     * - .toGetApplication()
     * - .toGetRoute()
     * - .toGetService()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * Applies to resource types:
     * - application
     * - service
     * - route
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCreatedByAccountIds(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by restricting the action to only those accounts that created the route within an application
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * Applies to actions:
     * - .toCreateRoute()
     * - .toDeleteRoute()
     * - .toGetRoute()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * Applies to resource types:
     * - route
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRouteCreatedByAccount(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by restricting the action to only those accounts that created the service within an application
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * Applies to actions:
     * - .toCreateRoute()
     * - .toCreateService()
     * - .toDeleteRoute()
     * - .toDeleteService()
     * - .toGetRoute()
     * - .toGetService()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * Applies to resource types:
     * - service
     * - route
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifServiceCreatedByAccount(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the path of the route
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * Applies to actions:
     * - .toCreateRoute()
     * - .toDeleteRoute()
     * - .toGetRoute()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * Applies to resource types:
     * - route
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSourcePath(value: string | string[], operator?: Operator | string): this;
    /**
     * Statement provider for service [refactor-spaces](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsmigrationhubrefactorspaces.html).
     *
     */
    constructor(props?: iam.PolicyStatementProps);
}
//# sourceMappingURL=migrationhubrefactorspaces.d.ts.map