import { AccessLevelList } from '../shared/access-level';
import { PolicyStatement, Operator } from '../shared';
import { aws_iam as iam } from "aws-cdk-lib";
/**
 * Statement provider for service [quicksight](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonquicksight.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class Quicksight extends PolicyStatement {
    servicePrefix: string;
    /**
     * Grants permission to enable setting default access to AWS resources
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/accessing-data-sources.html
     */
    toAccountConfigurations(): this;
    /**
     * Grants permission to cancel a SPICE ingestions on a dataset
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CancelIngestion.html
     */
    toCancelIngestion(): this;
    /**
     * Grants permission to create an account customization for QuickSight account or namespace
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateAccountCustomization.html
     */
    toCreateAccountCustomization(): this;
    /**
     * Grants permission to provision Amazon QuickSight administrators, authors, and readers
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    toCreateAdmin(): this;
    /**
     * Grants permission to create an analysis from a template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateAnalysis.html
     */
    toCreateAnalysis(): this;
    /**
     * Grants permission to create a custom permissions resource for restricting user access
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    toCreateCustomPermissions(): this;
    /**
     * Grants permission to create a QuickSight Dashboard
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateDashboard.html
     */
    toCreateDashboard(): this;
    /**
     * Grants permission to create a dataset
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - quicksight:PassDataSource
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateDataSet.html
     */
    toCreateDataSet(): this;
    /**
     * Grants permission to create a data source
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateDataSource.html
     */
    toCreateDataSource(): this;
    /**
     * Grants permission to create a QuickSight email customization template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/customizing-quicksight-email-templates.html
     */
    toCreateEmailCustomizationTemplate(): this;
    /**
     * Grants permission to create a QuickSight folder
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateFolder.html
     */
    toCreateFolder(): this;
    /**
     * Grants permission to add a QuickSight Dashboard, Analysis or Dataset to a QuickSight Folder
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateFolderMembership.html
     */
    toCreateFolderMembership(): this;
    /**
     * Grants permission to create a QuickSight group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateGroup.html
     */
    toCreateGroup(): this;
    /**
     * Grants permission to add a QuickSight user to a QuickSight group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateGroupMembership.html
     */
    toCreateGroupMembership(): this;
    /**
     * Grants permission to create an assignment with one specified IAM Policy ARN that will be assigned to specified groups or users of QuickSight
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateIAMPolicyAssignment.html
     */
    toCreateIAMPolicyAssignment(): this;
    /**
     * Grants permission to start a SPICE ingestion on a dataset
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateIngestion.html
     */
    toCreateIngestion(): this;
    /**
     * Grants permission to create an QuickSight namespace
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ds:CreateIdentityPoolDirectory
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateNamespace.html
     */
    toCreateNamespace(): this;
    /**
     * Grants permission to provision Amazon QuickSight readers
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    toCreateReader(): this;
    /**
     * Grants permission to create a template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateTemplate.html
     */
    toCreateTemplate(): this;
    /**
     * Grants permission to create a template alias
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateTemplateAlias.html
     */
    toCreateTemplateAlias(): this;
    /**
     * Grants permission to create a theme
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateTheme.html
     */
    toCreateTheme(): this;
    /**
     * Grants permission to create an alias for a theme version
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateThemeAlias.html
     */
    toCreateThemeAlias(): this;
    /**
     * Grants permission to provision Amazon QuickSight authors and readers
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    toCreateUser(): this;
    /**
     * Grants permission to create a VPC connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/vpc-creating-a-connection-in-quicksight.html
     */
    toCreateVPCConnection(): this;
    /**
     * Grants permission to delete an account customization for QuickSight account or namespace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteAccountCustomization.html
     */
    toDeleteAccountCustomization(): this;
    /**
     * Grants permission to delete an analysis
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteAnalysis.html
     */
    toDeleteAnalysis(): this;
    /**
     * Grants permission to delete a custom permissions resource
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    toDeleteCustomPermissions(): this;
    /**
     * Grants permission to delete a QuickSight Dashboard
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteDashboard.html
     */
    toDeleteDashboard(): this;
    /**
     * Grants permission to delete a dataset
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteDataSet.html
     */
    toDeleteDataSet(): this;
    /**
     * Grants permission to delete a data source
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteDataSource.html
     */
    toDeleteDataSource(): this;
    /**
     * Grants permission to delete a QuickSight email customization template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/customizing-quicksight-email-templates.html
     */
    toDeleteEmailCustomizationTemplate(): this;
    /**
     * Grants permission to delete a QuickSight Folder
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteFolder.html
     */
    toDeleteFolder(): this;
    /**
     * Grants permission to remove a QuickSight Dashboard, Analysis or Dataset from a QuickSight Folder
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteFolderMembership.html
     */
    toDeleteFolderMembership(): this;
    /**
     * Grants permission to remove a user group from QuickSight
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteGroup.html
     */
    toDeleteGroup(): this;
    /**
     * Grants permission to remove a user from a group so that he/she is no longer a member of the group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteGroupMembership.html
     */
    toDeleteGroupMembership(): this;
    /**
     * Grants permission to update an existing assignment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteIAMPolicyAssignment.html
     */
    toDeleteIAMPolicyAssignment(): this;
    /**
     * Grants permission to delete a QuickSight namespace
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ds:DeleteDirectory
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteNamespace.html
     */
    toDeleteNamespace(): this;
    /**
     * Grants permission to delete a template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteTemplate.html
     */
    toDeleteTemplate(): this;
    /**
     * Grants permission to delete a template alias
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteTemplateAlias.html
     */
    toDeleteTemplateAlias(): this;
    /**
     * Grants permission to delete a theme
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteTheme.html
     */
    toDeleteTheme(): this;
    /**
     * Grants permission to delete the alias of a theme
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteThemeAlias.html
     */
    toDeleteThemeAlias(): this;
    /**
     * Grants permission to delete a QuickSight user, given the user name
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteUser.html
     */
    toDeleteUser(): this;
    /**
     * Grants permission to deletes a user identified by its principal ID
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteUserByPrincipalId.html
     */
    toDeleteUserByPrincipalId(): this;
    /**
     * Grants permission to delete a VPC connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/vpc-creating-a-connection-in-quicksight.html
     */
    toDeleteVPCConnection(): this;
    /**
     * Grants permission to describe an account customization for QuickSight account or namespace
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeAccountCustomization.html
     */
    toDescribeAccountCustomization(): this;
    /**
     * Grants permission to describe the administrative account settings for QuickSight account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeAccountSettings.html
     */
    toDescribeAccountSettings(): this;
    /**
     * Grants permission to describe an analysis
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeAnalysis.html
     */
    toDescribeAnalysis(): this;
    /**
     * Grants permission to describe permissions for an analysis
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeAnalysisPermissions.html
     */
    toDescribeAnalysisPermissions(): this;
    /**
     * Grants permission to describe a custom permissions resource in a QuickSight account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    toDescribeCustomPermissions(): this;
    /**
     * Grants permission to describe a QuickSight Dashboard
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeDashboard.html
     */
    toDescribeDashboard(): this;
    /**
     * Grants permission to describe permissions for a QuickSight Dashboard
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeDashboardPermissions.html
     */
    toDescribeDashboardPermissions(): this;
    /**
     * Grants permission to describe a dataset
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeDataSet.html
     */
    toDescribeDataSet(): this;
    /**
     * Grants permission to describe the resource policy of a dataset
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeDataSetPermissions.html
     */
    toDescribeDataSetPermissions(): this;
    /**
     * Grants permission to describe a data source
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeDataSource.html
     */
    toDescribeDataSource(): this;
    /**
     * Grants permission to describe the resource policy of a data source
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeDataSourcePermissions.html
     */
    toDescribeDataSourcePermissions(): this;
    /**
     * Grants permission to describe a QuickSight email customization template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/customizing-quicksight-email-templates.html
     */
    toDescribeEmailCustomizationTemplate(): this;
    /**
     * Grants permission to describe a QuickSight Folder
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeFolder.html
     */
    toDescribeFolder(): this;
    /**
     * Grants permission to describe permissions for a QuickSight Folder
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeFolderPermissions.html
     */
    toDescribeFolderPermissions(): this;
    /**
     * Grants permission to describe resolved permissions for a QuickSight Folder
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeFolderResolvedPermissions.html
     */
    toDescribeFolderResolvedPermissions(): this;
    /**
     * Grants permission to describe a QuickSight group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeGroup.html
     */
    toDescribeGroup(): this;
    /**
     * Grants permission to describe a QuickSight group member
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeGroupMembership.html
     */
    toDescribeGroupMembership(): this;
    /**
     * Grants permission to describe an existing assignment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeIAMPolicyAssignment.html
     */
    toDescribeIAMPolicyAssignment(): this;
    /**
     * Grants permission to describe a SPICE ingestion on a dataset
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeIngestion.html
     */
    toDescribeIngestion(): this;
    /**
     * Grants permission to describe the IP restrictions for QuickSight account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeIpRestriction.html
     */
    toDescribeIpRestriction(): this;
    /**
     * Grants permission to describe a QuickSight namespace
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeNamespace.html
     */
    toDescribeNamespace(): this;
    /**
     * Grants permission to describe a template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeTemplate.html
     */
    toDescribeTemplate(): this;
    /**
     * Grants permission to describe a template alias
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeTemplateAlias.html
     */
    toDescribeTemplateAlias(): this;
    /**
     * Grants permission to describe permissions for a template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeTemplatePermissions.html
     */
    toDescribeTemplatePermissions(): this;
    /**
     * Grants permission to describe a theme
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeTheme.html
     */
    toDescribeTheme(): this;
    /**
     * Grants permission to describe a theme alias
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeThemeAlias.html
     */
    toDescribeThemeAlias(): this;
    /**
     * Grants permission to describe permissions for a theme
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeThemePermissions.html
     */
    toDescribeThemePermissions(): this;
    /**
     * Grants permission to describe a QuickSight user given the user name
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeUser.html
     */
    toDescribeUser(): this;
    /**
     * Grants permission to generate a URL used to embed a QuickSight Dashboard for a user not registered with QuickSight
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_GenerateEmbedUrlForAnonymousUser.html
     */
    toGenerateEmbedUrlForAnonymousUser(): this;
    /**
     * Grants permission to generate a URL used to embed a QuickSight Dashboard for a user registered with QuickSight
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_GenerateEmbedUrlForRegisteredUser.html
     */
    toGenerateEmbedUrlForRegisteredUser(): this;
    /**
     * Grants permission to get a URL used to embed a QuickSight Dashboard for a user not registered with QuickSight
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_GetDashboardEmbedUrl.html
     */
    toGetAnonymousUserEmbedUrl(): this;
    /**
     * Grants permission to get an auth code representing a QuickSight user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    toGetAuthCode(): this;
    /**
     * Grants permission to get a URL used to embed a QuickSight Dashboard
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_GetDashboardEmbedUrl.html
     */
    toGetDashboardEmbedUrl(): this;
    /**
     * Grants permission to use Amazon QuickSight, in Enterprise edition, to identify and display the Microsoft Active Directory (Microsoft Active Directory) directory groups that are mapped to roles in Amazon QuickSight
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    toGetGroupMapping(): this;
    /**
     * Grants permission to get a URL to embed QuickSight console experience
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_GetSessionEmbedUrl.html
     */
    toGetSessionEmbedUrl(): this;
    /**
     * Grants permission to list all analyses in an account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListAnalyses.html
     */
    toListAnalyses(): this;
    /**
     * Grants permission to list custom permissions resources in QuickSight account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    toListCustomPermissions(): this;
    /**
     * Grants permission to list all versions of a QuickSight Dashboard
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListDashboardVersions.html
     */
    toListDashboardVersions(): this;
    /**
     * Grants permission to list all Dashboards in a QuickSight Account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListDashboards.html
     */
    toListDashboards(): this;
    /**
     * Grants permission to list all datasets
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListDataSets.html
     */
    toListDataSets(): this;
    /**
     * Grants permission to list all data sources
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListDataSources.html
     */
    toListDataSources(): this;
    /**
     * Grants permission to list all members in a folder
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListFolderMembers.html
     */
    toListFolderMembers(): this;
    /**
     * Grants permission to list all Folders in a QuickSight Account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListFolders.html
     */
    toListFolders(): this;
    /**
     * Grants permission to list member users in a group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListGroupMemberships.html
     */
    toListGroupMemberships(): this;
    /**
     * Grants permission to list all user groups in QuickSight
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListGroups.html
     */
    toListGroups(): this;
    /**
     * Grants permission to list all assignments in the current Amazon QuickSight account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListIAMPolicyAssignments.html
     */
    toListIAMPolicyAssignments(): this;
    /**
     * Grants permission to list all assignments assigned to a user and the groups it belongs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListIAMPolicyAssignmentsForUser.html
     */
    toListIAMPolicyAssignmentsForUser(): this;
    /**
     * Grants permission to list all SPICE ingestions on a dataset
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListIngestions.html
     */
    toListIngestions(): this;
    /**
     * Grants permission to lists all namespaces in a QuickSight account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListNamespaces.html
     */
    toListNamespaces(): this;
    /**
     * Grants permission to list tags of a QuickSight resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource(): this;
    /**
     * Grants permission to list all aliases for a template
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListTemplateAliases.html
     */
    toListTemplateAliases(): this;
    /**
     * Grants permission to list all versions of a template
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListTemplateVersions.html
     */
    toListTemplateVersions(): this;
    /**
     * Grants permission to list all templates in a QuickSight account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListTemplates.html
     */
    toListTemplates(): this;
    /**
     * Grants permission to list all aliases of a theme
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListThemeAliases.html
     */
    toListThemeAliases(): this;
    /**
     * Grants permission to list all versions of a theme
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListThemeVersions.html
     */
    toListThemeVersions(): this;
    /**
     * Grants permission to list all themes in an account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListThemes.html
     */
    toListThemes(): this;
    /**
     * Grants permission to list groups that a given user is a member of
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListUserGroups.html
     */
    toListUserGroups(): this;
    /**
     * Grants permission to list all of the QuickSight users belonging to this account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListUsers.html
     */
    toListUsers(): this;
    /**
     * Grants permission to use a dataset for a template
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/qs-api-overview.html
     */
    toPassDataSet(): this;
    /**
     * Grants permission to use a data source for a data set
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/qs-api-overview.html
     */
    toPassDataSource(): this;
    /**
     * Grants permission to create a QuickSight user, whose identity is associated with the IAM identity/role specified in the request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_RegisterUser.html
     */
    toRegisterUser(): this;
    /**
     * Grants permission to restore a deleted analysis
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_RestoreAnalysis.html
     */
    toRestoreAnalysis(): this;
    /**
     * Grants permission to manage scoping policies for permissions to AWS resources
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/accessing-data-sources.html
     */
    toScopeDownPolicy(): this;
    /**
     * Grants permission to search for a sub-set of analyses
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_SearchAnalyses.html
     */
    toSearchAnalyses(): this;
    /**
     * Grants permission to search for a sub-set of QuickSight Dashboards
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_SearchDashboards.html
     */
    toSearchDashboards(): this;
    /**
     * Grants permission to use Amazon QuickSight, in Enterprise edition, to display your Microsoft Active Directory directory groups so that you can choose which ones to map to roles in Amazon QuickSight
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    toSearchDirectoryGroups(): this;
    /**
     * Grants permission to search for a sub-set of QuickSight Folders
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_SearchFolders.html
     */
    toSearchFolders(): this;
    /**
     * Grants permission to search for a sub-set of QuickSight groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_SearchGroups.html
     */
    toSearchGroups(): this;
    /**
     * Grants permission to use Amazon QuickSight, in Enterprise edition, to display your Microsoft Active Directory directory groups so that you can choose which ones to map to roles in Amazon QuickSight
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    toSetGroupMapping(): this;
    /**
     * Grants permission to subscribe to Amazon QuickSight, and also to allow the user to upgrade the subscription to Enterprise edition
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifEdition()
     * - .ifDirectoryType()
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    toSubscribe(): this;
    /**
     * Grants permission to add tags to a QuickSight resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_TagResource.html
     */
    toTagResource(): this;
    /**
     * Grants permission to unsubscribe from Amazon QuickSight, which permanently deletes all users and their resources from Amazon QuickSight
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    toUnsubscribe(): this;
    /**
     * Grants permission to remove tags from a QuickSight resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UntagResource.html
     */
    toUntagResource(): this;
    /**
     * Grants permission to update an account customization for QuickSight account or namespace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateAccountCustomization.html
     */
    toUpdateAccountCustomization(): this;
    /**
     * Grants permission to update the administrative account settings for QuickSight account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateAccountSettings.html
     */
    toUpdateAccountSettings(): this;
    /**
     * Grants permission to update an analysis
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateAnalysis.html
     */
    toUpdateAnalysis(): this;
    /**
     * Grants permission to update permissions for an analysis
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateAnalysisPermissions.html
     */
    toUpdateAnalysisPermissions(): this;
    /**
     * Grants permission to update a custom permissions resource
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    toUpdateCustomPermissions(): this;
    /**
     * Grants permission to update a QuickSight Dashboard
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateDashboard.html
     */
    toUpdateDashboard(): this;
    /**
     * Grants permission to update permissions for a QuickSight Dashboard
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateDashboardPermissions.html
     */
    toUpdateDashboardPermissions(): this;
    /**
     * Grants permission to update a QuickSight Dashboard’s Published Version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateDashboardPublishedVersion.html
     */
    toUpdateDashboardPublishedVersion(): this;
    /**
     * Grants permission to update a dataset
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - quicksight:PassDataSource
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateDataSet.html
     */
    toUpdateDataSet(): this;
    /**
     * Grants permission to update the resource policy of a dataset
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateDataSetPermissions.html
     */
    toUpdateDataSetPermissions(): this;
    /**
     * Grants permission to update a data source
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateDataSource.html
     */
    toUpdateDataSource(): this;
    /**
     * Grants permission to update the resource policy of a data source
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateDataSourcePermissions.html
     */
    toUpdateDataSourcePermissions(): this;
    /**
     * Grants permission to update a QuickSight email customization template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/customizing-quicksight-email-templates.html
     */
    toUpdateEmailCustomizationTemplate(): this;
    /**
     * Grants permission to update a QuickSight Folder
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateFolder.html
     */
    toUpdateFolder(): this;
    /**
     * Grants permission to update permissions for a QuickSight Folder
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateFolderPermissions.html
     */
    toUpdateFolderPermissions(): this;
    /**
     * Grants permission to change group description
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateGroup.html
     */
    toUpdateGroup(): this;
    /**
     * Grants permission to update an existing assignment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateIAMPolicyAssignment.html
     */
    toUpdateIAMPolicyAssignment(): this;
    /**
     * Grants permission to update the IP restrictions for QuickSight account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateIpRestriction.html
     */
    toUpdateIpRestriction(): this;
    /**
     * Grants permission to enable or disable public sharing on an account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdatePublicSharingSettings.html
     */
    toUpdatePublicSharingSettings(): this;
    /**
     * Grants permission to update a template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateTemplate.html
     */
    toUpdateTemplate(): this;
    /**
     * Grants permission to update a template alias
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateTemplateAlias.html
     */
    toUpdateTemplateAlias(): this;
    /**
     * Grants permission to update permissions for a template
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateTemplatePermissions.html
     */
    toUpdateTemplatePermissions(): this;
    /**
     * Grants permission to update a theme
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateTheme.html
     */
    toUpdateTheme(): this;
    /**
     * Grants permission to update the alias of a theme
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateThemeAlias.html
     */
    toUpdateThemeAlias(): this;
    /**
     * Grants permission to update permissions for a theme
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateThemePermissions.html
     */
    toUpdateThemePermissions(): this;
    /**
     * Grants permission to update an Amazon QuickSight user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateUser.html
     */
    toUpdateUser(): this;
    protected accessLevelList: AccessLevelList;
    /**
     * Adds a resource of type user to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_User.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onUser(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type group to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_Group.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onGroup(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type analysis to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_Analysis.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAnalysis(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type dashboard to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_Dashboard.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDashboard(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type template to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_Template.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTemplate(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type datasource to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DataSource.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDatasource(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type dataset to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DataSet.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDataset(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type ingestion to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_Ingestion.html
     *
     * @param datasetId - Identifier for the datasetId.
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIngestion(datasetId: string, resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type theme to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_Theme.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTheme(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type assignment to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_IAMPolicyAssignment.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAssignment(resourceId: string, account?: string, partition?: string): this;
    /**
     * Adds a resource of type customization to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_AccountCustomization.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCustomization(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type namespace to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_Namespace.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onNamespace(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type folder to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_Folder.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFolder(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type emailCustomizationTemplate to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/customizing-quicksight-email-templates.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onEmailCustomizationTemplate(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Filters access by the user management options
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/security-scp.html
     *
     * Applies to actions:
     * - .toSubscribe()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDirectoryType(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the edition of QuickSight
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/security-scp.html
     *
     * Applies to actions:
     * - .toSubscribe()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEdition(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by IAM user or role ARN
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     *
     * Applies to actions:
     * - .toRegisterUser()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifIamArn(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by session name
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     *
     * Applies to actions:
     * - .toRegisterUser()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSessionName(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by user name
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     *
     * Applies to actions:
     * - .toCreateGroupMembership()
     * - .toDeleteGroupMembership()
     * - .toDescribeGroupMembership()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifUserName(value: string | string[], operator?: Operator | string): this;
    /**
     * Statement provider for service [quicksight](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonquicksight.html).
     *
     */
    constructor(props?: iam.PolicyStatementProps);
}
//# sourceMappingURL=quicksight.d.ts.map