import { AccessLevelList } from '../shared/access-level';
import { PolicyStatement, Operator } from '../shared';
import { aws_iam as iam } from "aws-cdk-lib";
/**
 * Statement provider for service [ssm](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssystemsmanager.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class Ssm extends PolicyStatement {
    servicePrefix: string;
    /**
     * Grants permission to add or overwrite one or more tags for a specified AWS resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_AddTagsToResource.html
     */
    toAddTagsToResource(): this;
    /**
     * Grants permission to associate RelatedItem to an OpsItem
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_AssociateOpsItemRelatedItem.html
     */
    toAssociateOpsItemRelatedItem(): this;
    /**
     * Grants permission to cancel a specified Run Command command
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CancelCommand.html
     */
    toCancelCommand(): this;
    /**
     * Grants permission to cancel an in-progress maintenance window execution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CancelMaintenanceWindowExecution.html
     */
    toCancelMaintenanceWindowExecution(): this;
    /**
     * Grants permission to create an activation that is used to register on-premises servers and virtual machines (VMs) with Systems Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CreateActivation.html
     */
    toCreateActivation(): this;
    /**
     * Grants permission to associate a specified Systems Manager document with specified instances or other targets
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CreateAssociation.html
     */
    toCreateAssociation(): this;
    /**
     * Grants permission to combine entries for multiple CreateAssociation operations in a single command
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CreateAssociationBatch.html
     */
    toCreateAssociationBatch(): this;
    /**
     * Grants permission to create a Systems Manager SSM document
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CreateDocument.html
     */
    toCreateDocument(): this;
    /**
     * Grants permission to create a maintenance window
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CreateMaintenanceWindow.html
     */
    toCreateMaintenanceWindow(): this;
    /**
     * Grants permission to create an OpsItem in OpsCenter
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CreateOpsItem.html
     */
    toCreateOpsItem(): this;
    /**
     * Grants permission to create an OpsMetadata object for an AWS resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CreateOpsMetadata.html
     */
    toCreateOpsMetadata(): this;
    /**
     * Grants permission to create a patch baseline
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CreatePatchBaseline.html
     */
    toCreatePatchBaseline(): this;
    /**
     * Grants permission to create a resource data sync configuration, which regularly collects inventory data from managed instances and updates the data in an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSyncType()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_CreateResourceDataSync.html
     */
    toCreateResourceDataSync(): this;
    /**
     * Grants permission to delete a specified activation for managed instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeleteActivation.html
     */
    toDeleteActivation(): this;
    /**
     * Grants permission to disassociate a specified SSM document from a specified instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeleteAssociation.html
     */
    toDeleteAssociation(): this;
    /**
     * Grants permission to delete a specified SSM document and its instance associations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeleteDocument.html
     */
    toDeleteDocument(): this;
    /**
     * Grants permission to delete a specified custom inventory type, or the data associated with a custom inventory type
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeleteInventory.html
     */
    toDeleteInventory(): this;
    /**
     * Grants permission to delete a specified maintenance window
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeleteMaintenanceWindow.html
     */
    toDeleteMaintenanceWindow(): this;
    /**
     * Grants permission to delete an OpsMetadata object
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeleteOpsMetadata.html
     */
    toDeleteOpsMetadata(): this;
    /**
     * Grants permission to delete a specified SSM parameter
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeleteParameter.html
     */
    toDeleteParameter(): this;
    /**
     * Grants permission to delete multiple specified SSM parameters
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeleteParameters.html
     */
    toDeleteParameters(): this;
    /**
     * Grants permission to delete a specified patch baseline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeletePatchBaseline.html
     */
    toDeletePatchBaseline(): this;
    /**
     * Grants permission to delete a specified resource data sync
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSyncType()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeleteResourceDataSync.html
     */
    toDeleteResourceDataSync(): this;
    /**
     * Grants permission to deregister a specified on-premises server or virtual machine (VM) from Systems Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeregisterManagedInstance.html
     */
    toDeregisterManagedInstance(): this;
    /**
     * Grants permission to deregister a specified patch baseline from being the default patch baseline for a specified patch group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeregisterPatchBaselineForPatchGroup.html
     */
    toDeregisterPatchBaselineForPatchGroup(): this;
    /**
     * Grants permission to deregister a specified target from a maintenance window
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeregisterTargetFromMaintenanceWindow.html
     */
    toDeregisterTargetFromMaintenanceWindow(): this;
    /**
     * Grants permission to deregister a specified task from a maintenance window
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DeregisterTaskFromMaintenanceWindow.html
     */
    toDeregisterTaskFromMaintenanceWindow(): this;
    /**
     * Grants permission to view details about a specified managed instance activation, such as when it was created and the number of instances registered using the activation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeActivations.html
     */
    toDescribeActivations(): this;
    /**
     * Grants permission to view details about the specified association for a specified instance or target
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeAssociation.html
     */
    toDescribeAssociation(): this;
    /**
     * Grants permission to view information about a specified association execution
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeAssociationExecutionTargets.html
     */
    toDescribeAssociationExecutionTargets(): this;
    /**
     * Grants permission to view all executions for a specified association
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeAssociationExecutions.html
     */
    toDescribeAssociationExecutions(): this;
    /**
     * Grants permission to view details about all active and terminated Automation executions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeAutomationExecutions.html
     */
    toDescribeAutomationExecutions(): this;
    /**
     * Grants permission to view information about all active and terminated step executions in an Automation workflow
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeAutomationStepExecutions.html
     */
    toDescribeAutomationStepExecutions(): this;
    /**
     * Grants permission to view all patches eligible to include in a patch baseline
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeAvailablePatches.html
     */
    toDescribeAvailablePatches(): this;
    /**
     * Grants permission to view details about a specified SSM document
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeDocument.html
     */
    toDescribeDocument(): this;
    /**
     * Grants permission to display information about SSM document parameters in the Systems Manager console (internal Systems Manager action)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-setting-up-messageAPIs.html
     */
    toDescribeDocumentParameters(): this;
    /**
     * Grants permission to view the permissions for a specified SSM document
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeDocumentPermission.html
     */
    toDescribeDocumentPermission(): this;
    /**
     * Grants permission to view all current associations for a specified instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeEffectiveInstanceAssociations.html
     */
    toDescribeEffectiveInstanceAssociations(): this;
    /**
     * Grants permission to view details about the patches currently associated with the specified patch baseline (Windows only)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeEffectivePatchesForPatchBaseline.html
     */
    toDescribeEffectivePatchesForPatchBaseline(): this;
    /**
     * Grants permission to view the status of the associations for a specified instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeInstanceAssociationsStatus.html
     */
    toDescribeInstanceAssociationsStatus(): this;
    /**
     * Grants permission to view details about a specified instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeInstanceInformation.html
     */
    toDescribeInstanceInformation(): this;
    /**
     * Grants permission to view status details about patches on a specified instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeInstancePatchStates.html
     */
    toDescribeInstancePatchStates(): this;
    /**
     * Grants permission to describe the high-level patch state for the instances in the specified patch group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeInstancePatchStatesForPatchGroup.html
     */
    toDescribeInstancePatchStatesForPatchGroup(): this;
    /**
     * Grants permission to view general details about the patches on a specified instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeInstancePatches.html
     */
    toDescribeInstancePatches(): this;
    /**
     * Grants permission to user's Amazon EC2 console to render managed instances' nodes
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-setting-up-messageAPIs.html
     */
    toDescribeInstanceProperties(): this;
    /**
     * Grants permission to view details about a specified inventory deletion
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeInventoryDeletions.html
     */
    toDescribeInventoryDeletions(): this;
    /**
     * Grants permission to view details of a specified task execution for a maintenance window
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeMaintenanceWindowExecutionTaskInvocations.html
     */
    toDescribeMaintenanceWindowExecutionTaskInvocations(): this;
    /**
     * Grants permission to view details about the tasks that ran during a specified maintenance window execution
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeMaintenanceWindowExecutionTasks.html
     */
    toDescribeMaintenanceWindowExecutionTasks(): this;
    /**
     * Grants permission to view the executions of a specified maintenance window
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeMaintenanceWindowExecutions.html
     */
    toDescribeMaintenanceWindowExecutions(): this;
    /**
     * Grants permission to view details about upcoming executions of a specified maintenance window
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeMaintenanceWindowSchedule.html
     */
    toDescribeMaintenanceWindowSchedule(): this;
    /**
     * Grants permission to view a list of the targets associated with a specified maintenance window
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeMaintenanceWindowTargets.html
     */
    toDescribeMaintenanceWindowTargets(): this;
    /**
     * Grants permission to view a list of the tasks associated with a specified maintenance window
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeMaintenanceWindowTasks.html
     */
    toDescribeMaintenanceWindowTasks(): this;
    /**
     * Grants permission to view information about all or specified maintenance windows
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeMaintenanceWindows.html
     */
    toDescribeMaintenanceWindows(): this;
    /**
     * Grants permission to view information about the maintenance window targets and tasks associated with a specified instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeMaintenanceWindowsForTarget.html
     */
    toDescribeMaintenanceWindowsForTarget(): this;
    /**
     * Grants permission to view details about specified OpsItems
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeOpsItems.html
     */
    toDescribeOpsItems(): this;
    /**
     * Grants permission to view details about a specified SSM parameter
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeParameters.html
     */
    toDescribeParameters(): this;
    /**
     * Grants permission to view information about patch baselines that meet the specified criteria
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribePatchBaselines.html
     */
    toDescribePatchBaselines(): this;
    /**
     * Grants permission to view aggregated status details for patches for a specified patch group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribePatchGroupState.html
     */
    toDescribePatchGroupState(): this;
    /**
     * Grants permission to view information about the patch baseline for a specified patch group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribePatchGroups.html
     */
    toDescribePatchGroups(): this;
    /**
     * Grants permission to view details of available patches for a specified operating system and patch property
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribePatchProperties.html
     */
    toDescribePatchProperties(): this;
    /**
     * Grants permission to view a list of recent Session Manager sessions that meet the specified search criteria
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DescribeSessions.html
     */
    toDescribeSessions(): this;
    /**
     * Grants permission to disassociate RelatedItem from an OpsItem
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_DisassociateOpsItemRelatedItem.html
     */
    toDisassociateOpsItemRelatedItem(): this;
    /**
     * Grants permission to view details of a specified Automation execution
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_AutomationExecution.html
     */
    toGetAutomationExecution(): this;
    /**
     * Grants permission to view the calendar state for a change calendar or a list of change calendars
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetCalendarState.html
     */
    toGetCalendarState(): this;
    /**
     * Grants permission to view details about the command execution of a specified invocation or plugin
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetCommandInvocation.html
     */
    toGetCommandInvocation(): this;
    /**
     * Grants permission to view the Session Manager connection status for a specified managed instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetConnectionStatus.html
     */
    toGetConnectionStatus(): this;
    /**
     * Grants permission to view the current default patch baseline for a specified operating system type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetDefaultPatchBaseline.html
     */
    toGetDefaultPatchBaseline(): this;
    /**
     * Grants permission to retrieve the current patch baseline snapshot for a specified instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetDeployablePatchSnapshotForInstance.html
     */
    toGetDeployablePatchSnapshotForInstance(): this;
    /**
     * Grants permission to view the contents of a specified SSM document
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDocumentCategories()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetDocument.html
     */
    toGetDocument(): this;
    /**
     * Grants permission to view instance inventory details per the specified criteria
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetInventory.html
     */
    toGetInventory(): this;
    /**
     * Grants permission to view a list of inventory types or attribute names for a specified inventory item type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetInventorySchema.html
     */
    toGetInventorySchema(): this;
    /**
     * Grants permission to view details about a specified maintenance window
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetMaintenanceWindow.html
     */
    toGetMaintenanceWindow(): this;
    /**
     * Grants permission to view details about a specified maintenance window execution
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetMaintenanceWindowExecution.html
     */
    toGetMaintenanceWindowExecution(): this;
    /**
     * Grants permission to view details about a specified maintenance window execution task
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetMaintenanceWindowExecutionTask.html
     */
    toGetMaintenanceWindowExecutionTask(): this;
    /**
     * Grants permission to view details about a specific maintenance window task running on a specific target
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetMaintenanceWindowExecutionTaskInvocation.html
     */
    toGetMaintenanceWindowExecutionTaskInvocation(): this;
    /**
     * Grants permission to view details about tasks registered with a specified maintenance window
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetMaintenanceWindowTask.html
     */
    toGetMaintenanceWindowTask(): this;
    /**
     * Grants permission to Systems Manager and SSM Agent to determine package installation requirements for an instance (internal Systems Manager call)
     *
     * Access Level: Read
     */
    toGetManifest(): this;
    /**
     * Grants permission to view information about a specified OpsItem
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetOpsItem.html
     */
    toGetOpsItem(): this;
    /**
     * Grants permission to retrieve an OpsMetadata object
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetOpsMetadata.html
     */
    toGetOpsMetadata(): this;
    /**
     * Grants permission to view summary information about OpsItems based on specified filters and aggregators
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetOpsSummary.html
     */
    toGetOpsSummary(): this;
    /**
     * Grants permission to view information about a specified parameter
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetParameter.html
     */
    toGetParameter(): this;
    /**
     * Grants permission to view details and changes for a specified parameter
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetParameterHistory.html
     */
    toGetParameterHistory(): this;
    /**
     * Grants permission to view information about multiple specified parameters
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetParameters.html
     */
    toGetParameters(): this;
    /**
     * Grants permission to view information about parameters in a specified hierarchy
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifRecursive()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetParametersByPath.html
     */
    toGetParametersByPath(): this;
    /**
     * Grants permission to view information about a specified patch baseline
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetPatchBaseline.html
     */
    toGetPatchBaseline(): this;
    /**
     * Grants permission to view the ID of the current patch baseline for a specified patch group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetPatchBaselineForPatchGroup.html
     */
    toGetPatchBaselineForPatchGroup(): this;
    /**
     * Grants permission to view the account-level setting for an AWS service
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_GetServiceSetting.html
     */
    toGetServiceSetting(): this;
    /**
     * Grants permission to apply an identifying label to a specified version of a parameter
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_LabelParameterVersion.html
     */
    toLabelParameterVersion(): this;
    /**
     * Grants permission to list versions of the specified association
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListAssociationVersions.html
     */
    toListAssociationVersions(): this;
    /**
     * Grants permission to list the associations for a specified SSM document or managed instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListAssociations.html
     */
    toListAssociations(): this;
    /**
     * Grants permission to list information about command invocations sent to a specified instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListCommandInvocations.html
     */
    toListCommandInvocations(): this;
    /**
     * Grants permission to list the commands sent to a specified instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListCommands.html
     */
    toListCommands(): this;
    /**
     * Grants permission to list compliance status for specified resource types on a specified resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListComplianceItems.html
     */
    toListComplianceItems(): this;
    /**
     * Grants permission to list a summary count of compliant and noncompliant resources for a specified compliance type
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListComplianceSummaries.html
     */
    toListComplianceSummaries(): this;
    /**
     * Grants permission to view metadata history about a specified SSM document
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListDocumentMetadataHistory.html
     */
    toListDocumentMetadataHistory(): this;
    /**
     * Grants permission to list all versions of a specified document
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListDocumentVersions.html
     */
    toListDocumentVersions(): this;
    /**
     * Grants permission to view information about a specified SSM document
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListDocuments.html
     */
    toListDocuments(): this;
    /**
     * Grants permission to SSM Agent to check for new State Manager associations (internal Systems Manager call)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-setting-up-messageAPIs.html
     */
    toListInstanceAssociations(): this;
    /**
     * Grants permission to view a list of specified inventory types for a specified instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListInventoryEntries.html
     */
    toListInventoryEntries(): this;
    /**
     * Grants permission to view details about OpsItemEvents
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListOpsItemEvents.html
     */
    toListOpsItemEvents(): this;
    /**
     * Grants permission to view details about OpsItem RelatedItems
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListOpsItemRelatedItems.html
     */
    toListOpsItemRelatedItems(): this;
    /**
     * Grants permission to view a list of OpsMetadata objects
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListOpsMetadata.html
     */
    toListOpsMetadata(): this;
    /**
     * Grants permission to list resource-level summary count
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListResourceComplianceSummaries.html
     */
    toListResourceComplianceSummaries(): this;
    /**
     * Grants permission to list information about resource data sync configurations in an account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifSyncType()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListResourceDataSync.html
     */
    toListResourceDataSync(): this;
    /**
     * Grants permission to view a list of resource tags for a specified resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource(): this;
    /**
     * Grants permission to share a custom SSM document publicly or privately with specified AWS accounts
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ModifyDocumentPermission.html
     */
    toModifyDocumentPermission(): this;
    /**
     * Grants permission to register a compliance type and other compliance details on a specified resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_PutComplianceItems.html
     */
    toPutComplianceItems(): this;
    /**
     * Grants permission to SSM Agent to generate a report of the results of specific agent requests (internal Systems Manager call)
     *
     * Access Level: Read
     */
    toPutConfigurePackageResult(): this;
    /**
     * Grants permission to add or update inventory items on multiple specified managed instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_PutInventory.html
     */
    toPutInventory(): this;
    /**
     * Grants permission to create an SSM parameter
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifOverwrite()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_PutParameter.html
     */
    toPutParameter(): this;
    /**
     * Grants permission to specify the default patch baseline for an operating system type
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_RegisterDefaultPatchBaseline.html
     */
    toRegisterDefaultPatchBaseline(): this;
    /**
     * Grants permission to register a Systems Manager Agent
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-setting-up-messageAPIs.html
     */
    toRegisterManagedInstance(): this;
    /**
     * Grants permission to specify the default patch baseline for a specified patch group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_RegisterPatchBaselineForPatchGroup.html
     */
    toRegisterPatchBaselineForPatchGroup(): this;
    /**
     * Grants permission to register a target with a specified maintenance window
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_RegisterTargetWithMaintenanceWindow.html
     */
    toRegisterTargetWithMaintenanceWindow(): this;
    /**
     * Grants permission to register a task with a specified maintenance window
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_RegisterTaskWithMaintenanceWindow.html
     */
    toRegisterTaskWithMaintenanceWindow(): this;
    /**
     * Grants permission to remove a specified tag key from a specified resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_RemoveTagsFromResource.html
     */
    toRemoveTagsFromResource(): this;
    /**
     * Grants permission to reset the service setting for an AWS account to the default value
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ResetServiceSetting.html
     */
    toResetServiceSetting(): this;
    /**
     * Grants permission to reconnect a Session Manager session to a managed instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_ResumeSession.html
     */
    toResumeSession(): this;
    /**
     * Grants permission to send a signal to change the current behavior or status of a specified Automation execution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_SendAutomationSignal.html
     */
    toSendAutomationSignal(): this;
    /**
     * Grants permission to run commands on one or more specified managed instances
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_SendCommand.html
     */
    toSendCommand(): this;
    /**
     * Grants permission to run a specified association manually
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_StartAssociationsOnce.html
     */
    toStartAssociationsOnce(): this;
    /**
     * Grants permission to initiate the execution of an Automation document
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_StartAutomationExecution.html
     */
    toStartAutomationExecution(): this;
    /**
     * Grants permission to initiate the execution of an Automation Change Template document
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_StartChangeRequestExecution.html
     */
    toStartChangeRequestExecution(): this;
    /**
     * Grants permission to initiate a connection to a specified target for a Session Manager session
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSessionDocumentAccessCheck()
     * - .ifResourceTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_StartSession.html
     */
    toStartSession(): this;
    /**
     * Grants permission to stop a specified Automation execution that is already in progress
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_StopAutomationExecution.html
     */
    toStopAutomationExecution(): this;
    /**
     * Grants permission to permanently end a Session Manager connection to an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_TerminateSession.html
     */
    toTerminateSession(): this;
    /**
     * Grants permission to remove an identifying label from a specified version of a parameter
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UnlabelParameterVersion.html
     */
    toUnlabelParameterVersion(): this;
    /**
     * Grants permission to update an association and immediately run the association on the specified targets
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateAssociation.html
     */
    toUpdateAssociation(): this;
    /**
     * Grants permission to update the status of the SSM document associated with a specified instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateAssociationStatus.html
     */
    toUpdateAssociationStatus(): this;
    /**
     * Grants permission to update one or more values for an SSM document
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateDocument.html
     */
    toUpdateDocument(): this;
    /**
     * Grants permission to change the default version of an SSM document
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateDocumentDefaultVersion.html
     */
    toUpdateDocumentDefaultVersion(): this;
    /**
     * Grants permission to update the metadata of an SSM document
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateDocumentMetadata.html
     */
    toUpdateDocumentMetadata(): this;
    /**
     * Grants permission to SSM Agent to update the status of the association that it is currently running (internal Systems Manager call)
     *
     * Access Level: Write
     */
    toUpdateInstanceAssociationStatus(): this;
    /**
     * Grants permission to SSM Agent to send a heartbeat signal to the Systems Manager service in the cloud
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-setting-up-messageAPIs.html
     */
    toUpdateInstanceInformation(): this;
    /**
     * Grants permission to update a specified maintenance window
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateMaintenanceWindow.html
     */
    toUpdateMaintenanceWindow(): this;
    /**
     * Grants permission to update a specified maintenance window target
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateMaintenanceWindowTarget.html
     */
    toUpdateMaintenanceWindowTarget(): this;
    /**
     * Grants permission to update a specified maintenance window task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateMaintenanceWindowTask.html
     */
    toUpdateMaintenanceWindowTask(): this;
    /**
     * Grants permission to assign or change the IAM role assigned to a specified managed instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateManagedInstanceRole.html
     */
    toUpdateManagedInstanceRole(): this;
    /**
     * Grants permission to edit or change an OpsItem
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateOpsItem.html
     */
    toUpdateOpsItem(): this;
    /**
     * Grants permission to update an OpsMetadata object
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateOpsMetadata.html
     */
    toUpdateOpsMetadata(): this;
    /**
     * Grants permission to update a specified patch baseline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdatePatchBaseline.html
     */
    toUpdatePatchBaseline(): this;
    /**
     * Grants permission to update a resource data sync
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSyncType()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateResourceDataSync.html
     */
    toUpdateResourceDataSync(): this;
    /**
     * Grants permission to update the service setting for an AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/APIReference/API_UpdateServiceSetting.html
     */
    toUpdateServiceSetting(): this;
    protected accessLevelList: AccessLevelList;
    /**
     * Adds a resource of type association to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-associations.html
     *
     * @param associationId - Identifier for the associationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAssociation(associationId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type automation-execution to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-working.html
     *
     * @param automationExecutionId - Identifier for the automationExecutionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onAutomationExecution(automationExecutionId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type automation-definition to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-documents.html
     *
     * @param automationDefinitionName - Identifier for the automationDefinitionName.
     * @param versionId - Identifier for the versionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAutomationDefinition(automationDefinitionName: string, versionId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type bucket to the statement
     *
     * @param bucketName - Identifier for the bucketName.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onBucket(bucketName: string, partition?: string): this;
    /**
     * Adds a resource of type document to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-ssm-docs.html
     *
     * @param documentName - Identifier for the documentName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onDocument(documentName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type instance to the statement
     *
     * @param instanceId - Identifier for the instanceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onInstance(instanceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type maintenancewindow to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-maintenance.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onMaintenancewindow(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type managed-instance to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/what-is-systems-manager.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onManagedInstance(instanceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type managed-instance-inventory to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-inventory-configuring.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onManagedInstanceInventory(instanceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type opsitem to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-working-with-OpsItems.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onOpsitem(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type opsmetadata to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/API_OpsMetadata.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onOpsmetadata(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type parameter to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-parameter-store.html
     *
     * @param parameterNameWithoutLeadingSlash - Identifier for the parameterNameWithoutLeadingSlash.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onParameter(parameterNameWithoutLeadingSlash: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type patchbaseline to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-patch.html
     *
     * @param patchBaselineIdResourceId - Identifier for the patchBaselineIdResourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onPatchbaseline(patchBaselineIdResourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type session to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/session-manager.html
     *
     * @param sessionId - Identifier for the sessionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onSession(sessionId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type resourcedatasync to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-inventory-datasync.html
     *
     * @param syncName - Identifier for the syncName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onResourcedatasync(syncName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type servicesetting to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/API_ServiceSetting.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onServicesetting(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type windowtarget to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-maintenance-assign-targets.html
     *
     * @param windowTargetId - Identifier for the windowTargetId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onWindowtarget(windowTargetId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type windowtask to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-maintenance-assign-tasks.html
     *
     * @param windowTaskId - Identifier for the windowTaskId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onWindowtask(windowTaskId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type task to the statement
     *
     * https://docs.aws.amazon.com/AmazonECS/latest/developerguide/scheduling_tasks.html
     *
     * @param taskId - Identifier for the taskId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTask(taskId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Filters access by verifying that a user has permission to access a document belonging to a specific category enum
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#policy-conditions
     *
     * Applies to actions:
     * - .toGetDocument()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDocumentCategories(value: string | string[], operator?: Operator | string): this;
    /**
     * Controls whether Systems Manager parameters can be overwritten
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#policy-conditions
     *
     * Applies to actions:
     * - .toPutParameter()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifOverwrite(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access to Systems Manager parameters created in a hierarchical structure
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#policy-conditions
     *
     * Applies to actions:
     * - .toGetParametersByPath()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRecursive(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by verifying that a user has permission to access either the default Session Manager configuration document or the custom configuration document specified in a request
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/getting-started-sessiondocumentaccesscheck.html
     *
     * Applies to actions:
     * - .toStartSession()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifSessionDocumentAccessCheck(value?: boolean): this;
    /**
     * Filters access by verifying that a user also has access to the ResourceDataSync SyncType specified in the request
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#policy-conditions
     *
     * Applies to actions:
     * - .toCreateResourceDataSync()
     * - .toDeleteResourceDataSync()
     * - .toListResourceDataSync()
     * - .toUpdateResourceDataSync()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSyncType(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by based on a tag key-value pair assigned to the Systems Manager resource
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#policy-conditions
     *
     * Applies to actions:
     * - .toSendCommand()
     * - .toStartSession()
     *
     * Applies to resource types:
     * - automation-execution
     * - document
     * - instance
     * - maintenancewindow
     * - managed-instance
     * - opsmetadata
     * - parameter
     * - patchbaseline
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Statement provider for service [ssm](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssystemsmanager.html).
     *
     */
    constructor(props?: iam.PolicyStatementProps);
}
//# sourceMappingURL=systemsmanager.d.ts.map