use super::{
    super::{
        Blocks, Connectivity, Coordinates, FiniteElementMethods, FiniteElementSpecifics, Nodes,
        Smoothing, VecConnectivity,
    },
    HEX, HexahedralFiniteElements,
};
use conspire::math::{Tensor, TensorVec};

const EPSILON: f64 = 1.0e-14;
const SMOOTHING_SCALE: f64 = 0.3;

const ZERO: f64 = 0.0;
const ONE_FIFTH: f64 = 1.0 / 5.0;
const ONE_FOURTH: f64 = 1.0 / 4.0;
const ONE_THIRD: f64 = 1.0 / 3.0;

trait FooClone {
    fn clone_foo(&self) -> Self;
}

impl FooClone for Coordinates {
    fn clone_foo(&self) -> Self {
        self.iter().cloned().collect()
    }
}

#[allow(clippy::too_many_arguments)]
fn test_finite_elements(
    element_blocks: Blocks,
    element_node_connectivity: Connectivity<HEX>,
    nodal_coordinates: Coordinates,
    node_element_connectivity_gold: VecConnectivity,
    node_node_connectivity_gold: VecConnectivity,
    exterior_nodes_gold: Nodes,
    interface_nodes_gold: Nodes,
    interior_nodes_gold: Nodes,
    laplacian_gold: Option<Coordinates>,
    smoothed_coordinates_gold: Option<Vec<Coordinates>>,
    nodal_influencers_gold: VecConnectivity,
) {
    let mut finite_elements = HexahedralFiniteElements::from((
        element_blocks.clone(),
        element_node_connectivity.clone(),
        nodal_coordinates.clone_foo(),
    ));
    assert_eq!(
        finite_elements.node_node_connectivity(),
        Err("Need to calculate the node-to-element connectivity first")
    );
    assert_eq!(
        finite_elements.nodal_hierarchy(),
        Err("Need to calculate the node-to-element connectivity first")
    );
    finite_elements.node_element_connectivity().unwrap();
    finite_elements.node_node_connectivity().unwrap();
    finite_elements.nodal_hierarchy().unwrap();
    finite_elements.nodal_influencers();
    assert_eq!(
        finite_elements.get_nodal_influencers(),
        &nodal_influencers_gold
    );
    assert_eq!(
        finite_elements.get_node_element_connectivity(),
        &node_element_connectivity_gold
    );
    assert_eq!(
        finite_elements.get_node_node_connectivity(),
        &node_node_connectivity_gold
    );
    assert_eq!(finite_elements.get_exterior_nodes(), &exterior_nodes_gold);
    assert_eq!(finite_elements.get_interface_nodes(), &interface_nodes_gold);
    assert_eq!(finite_elements.get_interior_nodes(), &interior_nodes_gold);
    if let Some(gold) = laplacian_gold {
        let laplacian = finite_elements.laplacian(finite_elements.get_node_node_connectivity());
        assert!(laplacian.len() == gold.len());
        laplacian
            .iter()
            .zip(gold.iter())
            .for_each(|(coordinates, gold_coordinates)| {
                coordinates.iter().zip(gold_coordinates.iter()).for_each(
                    |(coordinate, gold_coordinate)| {
                        assert!((coordinate - gold_coordinate).abs() < EPSILON)
                    },
                )
            });
    }
    if let Some(gold_set) = smoothed_coordinates_gold {
        gold_set.iter().enumerate().for_each(|(index, gold)| {
            let iterations = index + 1;
            let mut finite_elements = HexahedralFiniteElements::from((
                element_blocks.clone(),
                element_node_connectivity.clone(),
                nodal_coordinates.clone_foo(),
            ));
            finite_elements.node_element_connectivity().unwrap();
            finite_elements.node_node_connectivity().unwrap();
            finite_elements.nodal_hierarchy().unwrap();
            finite_elements.nodal_influencers();
            finite_elements
                .smooth(&Smoothing::Laplacian(iterations, SMOOTHING_SCALE))
                .unwrap();
            let smoothed_nodal_coordinates = finite_elements.get_nodal_coordinates();
            assert!(smoothed_nodal_coordinates.len() == gold.len());
            smoothed_nodal_coordinates.iter().zip(gold.iter()).for_each(
                |(coordinates, gold_coordinates)| {
                    coordinates.iter().zip(gold_coordinates.iter()).for_each(
                        |(coordinate, gold_coordinate)| {
                            assert!((coordinate - gold_coordinate).abs() < EPSILON)
                        },
                    )
                },
            );
        });
        let mut finite_elements = HexahedralFiniteElements::from((
            element_blocks.clone(),
            element_node_connectivity.clone(),
            nodal_coordinates.clone_foo(),
        ));
        finite_elements.node_element_connectivity().unwrap();
        finite_elements.node_node_connectivity().unwrap();
        finite_elements.nodal_hierarchy().unwrap();
        let prescribed_nodes = finite_elements.get_boundary_nodes().clone();
        finite_elements
            .set_prescribed_nodes(Some(prescribed_nodes), None)
            .unwrap();
        finite_elements
            .smooth(&Smoothing::Laplacian(1, SMOOTHING_SCALE))
            .unwrap();
        finite_elements
            .get_nodal_coordinates()
            .iter()
            .zip(nodal_coordinates.iter())
            .for_each(|(a, b)| assert_eq!(a, b));
    }
}

#[test]
fn single() {
    let element_blocks = vec![11];
    let element_node_connectivity = vec![[0, 1, 3, 2, 4, 5, 7, 6]];
    let nodal_coordinates = Coordinates::new(&[
        [0.0, 0.0, 0.0],
        [1.0, 0.0, 0.0],
        [0.0, 1.0, 0.0],
        [1.0, 1.0, 0.0],
        [0.0, 0.0, 1.0],
        [1.0, 0.0, 1.0],
        [0.0, 1.0, 1.0],
        [1.0, 1.0, 1.0],
    ]);
    let node_element_connectivity_gold = vec![vec![0]; 8];
    let node_node_connectivity_gold = vec![
        vec![1, 2, 4],
        vec![0, 3, 5],
        vec![0, 3, 6],
        vec![1, 2, 7],
        vec![0, 5, 6],
        vec![1, 4, 7],
        vec![2, 4, 7],
        vec![3, 5, 6],
    ];
    let exterior_nodes_gold = (0..8).collect();
    let interface_nodes_gold = vec![];
    let interior_nodes_gold = vec![];
    let laplacian_gold = Coordinates::new(&[
        [ONE_THIRD, ONE_THIRD, ONE_THIRD],
        [-ONE_THIRD, ONE_THIRD, ONE_THIRD],
        [ONE_THIRD, -ONE_THIRD, ONE_THIRD],
        [-ONE_THIRD, -ONE_THIRD, ONE_THIRD],
        [ONE_THIRD, ONE_THIRD, -ONE_THIRD],
        [-ONE_THIRD, ONE_THIRD, -ONE_THIRD],
        [ONE_THIRD, -ONE_THIRD, -ONE_THIRD],
        [-ONE_THIRD, -ONE_THIRD, -ONE_THIRD],
    ]);
    let nodal_influencers_gold = node_node_connectivity_gold.clone();
    test_finite_elements(
        element_blocks,
        element_node_connectivity,
        nodal_coordinates,
        node_element_connectivity_gold,
        node_node_connectivity_gold,
        exterior_nodes_gold,
        interface_nodes_gold,
        interior_nodes_gold,
        Some(laplacian_gold),
        None,
        nodal_influencers_gold,
    );
}

#[test]
fn double_x() {
    let element_blocks = vec![11; 2];
    let element_node_connectivity = vec![[0, 1, 4, 3, 6, 7, 10, 9], [1, 2, 5, 4, 7, 8, 11, 10]];
    let nodal_coordinates = Coordinates::new(&[
        [0.0, 0.0, 0.0],
        [1.0, 0.0, 0.0],
        [2.0, 0.0, 0.0],
        [0.0, 1.0, 0.0],
        [1.0, 1.0, 0.0],
        [2.0, 1.0, 0.0],
        [0.0, 0.0, 1.0],
        [1.0, 0.0, 1.0],
        [2.0, 0.0, 1.0],
        [0.0, 1.0, 1.0],
        [1.0, 1.0, 1.0],
        [2.0, 1.0, 1.0],
    ]);
    let node_element_connectivity_gold = vec![
        vec![0],
        vec![0, 1],
        vec![1],
        vec![0],
        vec![0, 1],
        vec![1],
        vec![0],
        vec![0, 1],
        vec![1],
        vec![0],
        vec![0, 1],
        vec![1],
    ];
    let node_node_connectivity_gold = vec![
        vec![1, 3, 6],
        vec![0, 2, 4, 7],
        vec![1, 5, 8],
        vec![0, 4, 9],
        vec![1, 3, 5, 10],
        vec![2, 4, 11],
        vec![0, 7, 9],
        vec![1, 6, 8, 10],
        vec![2, 7, 11],
        vec![3, 6, 10],
        vec![4, 7, 9, 11],
        vec![5, 8, 10],
    ];
    let exterior_nodes_gold = (0..12).collect();
    let interface_nodes_gold = vec![];
    let interior_nodes_gold = vec![];
    let laplacian_gold = Coordinates::new(&[
        [ONE_THIRD, ONE_THIRD, ONE_THIRD],
        [ZERO, ONE_FOURTH, ONE_FOURTH],
        [-ONE_THIRD, ONE_THIRD, ONE_THIRD],
        [ONE_THIRD, -ONE_THIRD, ONE_THIRD],
        [ZERO, -ONE_FOURTH, ONE_FOURTH],
        [-ONE_THIRD, -ONE_THIRD, ONE_THIRD],
        [ONE_THIRD, ONE_THIRD, -ONE_THIRD],
        [ZERO, ONE_FOURTH, -ONE_FOURTH],
        [-ONE_THIRD, ONE_THIRD, -ONE_THIRD],
        [ONE_THIRD, -ONE_THIRD, -ONE_THIRD],
        [ZERO, -ONE_FOURTH, -ONE_FOURTH],
        [-ONE_THIRD, -ONE_THIRD, -ONE_THIRD],
    ]);
    let smoothed_coordinates_gold = vec![
        Coordinates::new(&[
            [0.1, 0.100, 0.100],
            [1.0, 0.075, 0.075],
            [1.9, 0.100, 0.100],
            [0.1, 0.900, 0.100],
            [1.0, 0.925, 0.075],
            [1.9, 0.900, 0.100],
            [0.1, 0.100, 0.900],
            [1.0, 0.075, 0.925],
            [1.9, 0.100, 0.900],
            [0.1, 0.900, 0.900],
            [1.0, 0.925, 0.925],
            [1.9, 0.900, 0.900],
        ]),
        Coordinates::new(&[
            [0.19, 0.1775, 0.1775],
            [1.00, 0.1425, 0.1425],
            [1.81, 0.1775, 0.1775],
            [0.19, 0.8225, 0.1775],
            [1.00, 0.8575, 0.1425],
            [1.81, 0.8225, 0.1775],
            [0.19, 0.1775, 0.8225],
            [1.00, 0.1425, 0.8575],
            [1.81, 0.1775, 0.8225],
            [0.19, 0.8225, 0.8225],
            [1.00, 0.8575, 0.8575],
            [1.81, 0.8225, 0.8225],
        ]),
    ];
    let nodal_influencers_gold = node_node_connectivity_gold.clone();
    test_finite_elements(
        element_blocks,
        element_node_connectivity,
        nodal_coordinates,
        node_element_connectivity_gold,
        node_node_connectivity_gold,
        exterior_nodes_gold,
        interface_nodes_gold,
        interior_nodes_gold,
        Some(laplacian_gold),
        Some(smoothed_coordinates_gold),
        nodal_influencers_gold,
    );
}

#[test]
fn double_y() {
    let element_blocks = vec![11; 2];
    let element_node_connectivity = vec![[0, 1, 3, 2, 6, 7, 9, 8], [2, 3, 5, 4, 8, 9, 11, 10]];
    let nodal_coordinates = Coordinates::new(&[
        [0.0, 0.0, 0.0],
        [1.0, 0.0, 0.0],
        [0.0, 1.0, 0.0],
        [1.0, 1.0, 0.0],
        [0.0, 2.0, 0.0],
        [1.0, 2.0, 0.0],
        [0.0, 0.0, 1.0],
        [1.0, 0.0, 1.0],
        [0.0, 1.0, 1.0],
        [1.0, 1.0, 1.0],
        [0.0, 2.0, 1.0],
        [1.0, 2.0, 1.0],
    ]);
    let node_element_connectivity_gold = vec![
        vec![0],
        vec![0],
        vec![0, 1],
        vec![0, 1],
        vec![1],
        vec![1],
        vec![0],
        vec![0],
        vec![0, 1],
        vec![0, 1],
        vec![1],
        vec![1],
    ];
    let node_node_connectivity_gold = vec![
        vec![1, 2, 6],
        vec![0, 3, 7],
        vec![0, 3, 4, 8],
        vec![1, 2, 5, 9],
        vec![2, 5, 10],
        vec![3, 4, 11],
        vec![0, 7, 8],
        vec![1, 6, 9],
        vec![2, 6, 9, 10],
        vec![3, 7, 8, 11],
        vec![4, 8, 11],
        vec![5, 9, 10],
    ];
    let exterior_nodes_gold = (0..12).collect();
    let interface_nodes_gold = vec![];
    let interior_nodes_gold = vec![];
    let nodal_influencers_gold = node_node_connectivity_gold.clone();
    test_finite_elements(
        element_blocks,
        element_node_connectivity,
        nodal_coordinates,
        node_element_connectivity_gold,
        node_node_connectivity_gold,
        exterior_nodes_gold,
        interface_nodes_gold,
        interior_nodes_gold,
        None,
        None,
        nodal_influencers_gold,
    );
}

#[test]
fn triple() {
    let element_blocks = vec![11; 3];
    let element_node_connectivity = vec![
        [0, 1, 5, 4, 8, 9, 13, 12],
        [1, 2, 6, 5, 9, 10, 14, 13],
        [2, 3, 7, 6, 10, 11, 15, 14],
    ];
    let nodal_coordinates = Coordinates::new(&[
        [0.0, 0.0, 0.0],
        [1.0, 0.0, 0.0],
        [2.0, 0.0, 0.0],
        [3.0, 0.0, 0.0],
        [0.0, 1.0, 0.0],
        [1.0, 1.0, 0.0],
        [2.0, 1.0, 0.0],
        [3.0, 1.0, 0.0],
        [0.0, 0.0, 1.0],
        [1.0, 0.0, 1.0],
        [2.0, 0.0, 1.0],
        [3.0, 0.0, 1.0],
        [0.0, 1.0, 1.0],
        [1.0, 1.0, 1.0],
        [2.0, 1.0, 1.0],
        [3.0, 1.0, 1.0],
    ]);
    let node_element_connectivity_gold = vec![
        vec![0],
        vec![0, 1],
        vec![1, 2],
        vec![2],
        vec![0],
        vec![0, 1],
        vec![1, 2],
        vec![2],
        vec![0],
        vec![0, 1],
        vec![1, 2],
        vec![2],
        vec![0],
        vec![0, 1],
        vec![1, 2],
        vec![2],
    ];
    let node_node_connectivity_gold = vec![
        vec![1, 4, 8],
        vec![0, 2, 5, 9],
        vec![1, 3, 6, 10],
        vec![2, 7, 11],
        vec![0, 5, 12],
        vec![1, 4, 6, 13],
        vec![2, 5, 7, 14],
        vec![3, 6, 15],
        vec![0, 9, 12],
        vec![1, 8, 10, 13],
        vec![2, 9, 11, 14],
        vec![3, 10, 15],
        vec![4, 8, 13],
        vec![5, 9, 12, 14],
        vec![6, 10, 13, 15],
        vec![7, 11, 14],
    ];
    let exterior_nodes_gold = (0..16).collect();
    let interface_nodes_gold = vec![];
    let interior_nodes_gold = vec![];
    let nodal_influencers_gold = node_node_connectivity_gold.clone();
    test_finite_elements(
        element_blocks,
        element_node_connectivity,
        nodal_coordinates,
        node_element_connectivity_gold,
        node_node_connectivity_gold,
        exterior_nodes_gold,
        interface_nodes_gold,
        interior_nodes_gold,
        None,
        None,
        nodal_influencers_gold,
    );
}

#[test]
fn quadruple() {
    let element_blocks = vec![11; 4];
    let element_node_connectivity = vec![
        [0, 1, 6, 5, 10, 11, 16, 15],
        [1, 2, 7, 6, 11, 12, 17, 16],
        [2, 3, 8, 7, 12, 13, 18, 17],
        [3, 4, 9, 8, 13, 14, 19, 18],
    ];
    let nodal_coordinates = Coordinates::new(&[
        [0.0, 0.0, 0.0],
        [1.0, 0.0, 0.0],
        [2.0, 0.0, 0.0],
        [3.0, 0.0, 0.0],
        [4.0, 0.0, 0.0],
        [0.0, 1.0, 0.0],
        [1.0, 1.0, 0.0],
        [2.0, 1.0, 0.0],
        [3.0, 1.0, 0.0],
        [4.0, 1.0, 0.0],
        [0.0, 0.0, 1.0],
        [1.0, 0.0, 1.0],
        [2.0, 0.0, 1.0],
        [3.0, 0.0, 1.0],
        [4.0, 0.0, 1.0],
        [0.0, 1.0, 1.0],
        [1.0, 1.0, 1.0],
        [2.0, 1.0, 1.0],
        [3.0, 1.0, 1.0],
        [4.0, 1.0, 1.0],
    ]);
    let node_element_connectivity_gold = vec![
        vec![0],
        vec![0, 1],
        vec![1, 2],
        vec![2, 3],
        vec![3],
        vec![0],
        vec![0, 1],
        vec![1, 2],
        vec![2, 3],
        vec![3],
        vec![0],
        vec![0, 1],
        vec![1, 2],
        vec![2, 3],
        vec![3],
        vec![0],
        vec![0, 1],
        vec![1, 2],
        vec![2, 3],
        vec![3],
    ];
    let node_node_connectivity_gold = vec![
        vec![1, 5, 10],
        vec![0, 2, 6, 11],
        vec![1, 3, 7, 12],
        vec![2, 4, 8, 13],
        vec![3, 9, 14],
        vec![0, 6, 15],
        vec![1, 5, 7, 16],
        vec![2, 6, 8, 17],
        vec![3, 7, 9, 18],
        vec![4, 8, 19],
        vec![0, 11, 15],
        vec![1, 10, 12, 16],
        vec![2, 11, 13, 17],
        vec![3, 12, 14, 18],
        vec![4, 13, 19],
        vec![5, 10, 16],
        vec![6, 11, 15, 17],
        vec![7, 12, 16, 18],
        vec![8, 13, 17, 19],
        vec![9, 14, 18],
    ];
    let exterior_nodes_gold = (0..20).collect();
    let interface_nodes_gold = vec![];
    let interior_nodes_gold = vec![];
    let nodal_influencers_gold = node_node_connectivity_gold.clone();
    test_finite_elements(
        element_blocks,
        element_node_connectivity,
        nodal_coordinates,
        node_element_connectivity_gold,
        node_node_connectivity_gold,
        exterior_nodes_gold,
        interface_nodes_gold,
        interior_nodes_gold,
        None,
        None,
        nodal_influencers_gold,
    );
}

#[test]
fn quadruple_2_voids() {
    let element_blocks = vec![99; 2];
    let element_node_connectivity = vec![[0, 1, 5, 4, 8, 9, 13, 12], [2, 3, 7, 6, 10, 11, 15, 14]];
    let nodal_coordinates = Coordinates::new(&[
        [0.0, 0.0, 0.0],
        [1.0, 0.0, 0.0],
        [3.0, 0.0, 0.0],
        [4.0, 0.0, 0.0],
        [0.0, 1.0, 0.0],
        [1.0, 1.0, 0.0],
        [3.0, 1.0, 0.0],
        [4.0, 1.0, 0.0],
        [0.0, 0.0, 1.0],
        [1.0, 0.0, 1.0],
        [3.0, 0.0, 1.0],
        [4.0, 0.0, 1.0],
        [0.0, 1.0, 1.0],
        [1.0, 1.0, 1.0],
        [3.0, 1.0, 1.0],
        [4.0, 1.0, 1.0],
    ]);
    let node_element_connectivity_gold = vec![
        vec![0],
        vec![0],
        vec![1],
        vec![1],
        vec![0],
        vec![0],
        vec![1],
        vec![1],
        vec![0],
        vec![0],
        vec![1],
        vec![1],
        vec![0],
        vec![0],
        vec![1],
        vec![1],
    ];
    let node_node_connectivity_gold = vec![
        vec![1, 4, 8],
        vec![0, 5, 9],
        vec![3, 6, 10],
        vec![2, 7, 11],
        vec![0, 5, 12],
        vec![1, 4, 13],
        vec![2, 7, 14],
        vec![3, 6, 15],
        vec![0, 9, 12],
        vec![1, 8, 13],
        vec![2, 11, 14],
        vec![3, 10, 15],
        vec![4, 8, 13],
        vec![5, 9, 12],
        vec![6, 10, 15],
        vec![7, 11, 14],
    ];
    let exterior_nodes_gold = (0..16).collect();
    let interface_nodes_gold = vec![];
    let interior_nodes_gold = vec![];
    let nodal_influencers_gold = node_node_connectivity_gold.clone();
    test_finite_elements(
        element_blocks,
        element_node_connectivity,
        nodal_coordinates,
        node_element_connectivity_gold,
        node_node_connectivity_gold,
        exterior_nodes_gold,
        interface_nodes_gold,
        interior_nodes_gold,
        None,
        None,
        nodal_influencers_gold,
    );
}

#[test]
fn quadruple_2_blocks() {
    let element_blocks = vec![11, 21, 21, 11];
    let element_node_connectivity = vec![
        [0, 1, 6, 5, 10, 11, 16, 15],
        [1, 2, 7, 6, 11, 12, 17, 16],
        [2, 3, 8, 7, 12, 13, 18, 17],
        [3, 4, 9, 8, 13, 14, 19, 18],
    ];
    let nodal_coordinates = Coordinates::new(&[
        [0.0, 0.0, 0.0],
        [1.0, 0.0, 0.0],
        [2.0, 0.0, 0.0],
        [3.0, 0.0, 0.0],
        [4.0, 0.0, 0.0],
        [0.0, 1.0, 0.0],
        [1.0, 1.0, 0.0],
        [2.0, 1.0, 0.0],
        [3.0, 1.0, 0.0],
        [4.0, 1.0, 0.0],
        [0.0, 0.0, 1.0],
        [1.0, 0.0, 1.0],
        [2.0, 0.0, 1.0],
        [3.0, 0.0, 1.0],
        [4.0, 0.0, 1.0],
        [0.0, 1.0, 1.0],
        [1.0, 1.0, 1.0],
        [2.0, 1.0, 1.0],
        [3.0, 1.0, 1.0],
        [4.0, 1.0, 1.0],
    ]);
    let node_element_connectivity_gold = vec![
        vec![0],
        vec![0, 1],
        vec![1, 2],
        vec![2, 3],
        vec![3],
        vec![0],
        vec![0, 1],
        vec![1, 2],
        vec![2, 3],
        vec![3],
        vec![0],
        vec![0, 1],
        vec![1, 2],
        vec![2, 3],
        vec![3],
        vec![0],
        vec![0, 1],
        vec![1, 2],
        vec![2, 3],
        vec![3],
    ];
    let node_node_connectivity_gold = vec![
        vec![1, 5, 10],
        vec![0, 2, 6, 11],
        vec![1, 3, 7, 12],
        vec![2, 4, 8, 13],
        vec![3, 9, 14],
        vec![0, 6, 15],
        vec![1, 5, 7, 16],
        vec![2, 6, 8, 17],
        vec![3, 7, 9, 18],
        vec![4, 8, 19],
        vec![0, 11, 15],
        vec![1, 10, 12, 16],
        vec![2, 11, 13, 17],
        vec![3, 12, 14, 18],
        vec![4, 13, 19],
        vec![5, 10, 16],
        vec![6, 11, 15, 17],
        vec![7, 12, 16, 18],
        vec![8, 13, 17, 19],
        vec![9, 14, 18],
    ];
    let exterior_nodes_gold = (0..20).collect();
    let interface_nodes_gold = vec![1, 3, 6, 8, 11, 13, 16, 18];
    let interior_nodes_gold = vec![];
    let nodal_influencers_gold = node_node_connectivity_gold.clone();
    test_finite_elements(
        element_blocks,
        element_node_connectivity,
        nodal_coordinates,
        node_element_connectivity_gold,
        node_node_connectivity_gold,
        exterior_nodes_gold,
        interface_nodes_gold,
        interior_nodes_gold,
        None,
        None,
        nodal_influencers_gold,
    );
}

#[test]
fn quadruple_2_blocks_void() {
    let element_blocks = vec![11, 21, 11];
    let element_node_connectivity = vec![
        [0, 1, 6, 5, 10, 11, 16, 15],
        [1, 2, 7, 6, 11, 12, 17, 16],
        [3, 4, 9, 8, 13, 14, 19, 18],
    ];
    let nodal_coordinates = Coordinates::new(&[
        [0.0, 0.0, 0.0],
        [1.0, 0.0, 0.0],
        [2.0, 0.0, 0.0],
        [3.0, 0.0, 0.0],
        [4.0, 0.0, 0.0],
        [0.0, 1.0, 0.0],
        [1.0, 1.0, 0.0],
        [2.0, 1.0, 0.0],
        [3.0, 1.0, 0.0],
        [4.0, 1.0, 0.0],
        [0.0, 0.0, 1.0],
        [1.0, 0.0, 1.0],
        [2.0, 0.0, 1.0],
        [3.0, 0.0, 1.0],
        [4.0, 0.0, 1.0],
        [0.0, 1.0, 1.0],
        [1.0, 1.0, 1.0],
        [2.0, 1.0, 1.0],
        [3.0, 1.0, 1.0],
        [4.0, 1.0, 1.0],
    ]);
    let node_element_connectivity_gold = vec![
        vec![0],
        vec![0, 1],
        vec![1],
        vec![2],
        vec![2],
        vec![0],
        vec![0, 1],
        vec![1],
        vec![2],
        vec![2],
        vec![0],
        vec![0, 1],
        vec![1],
        vec![2],
        vec![2],
        vec![0],
        vec![0, 1],
        vec![1],
        vec![2],
        vec![2],
    ];
    let node_node_connectivity_gold = vec![
        vec![1, 5, 10],
        vec![0, 2, 6, 11],
        vec![1, 7, 12],
        vec![4, 8, 13],
        vec![3, 9, 14],
        vec![0, 6, 15],
        vec![1, 5, 7, 16],
        vec![2, 6, 17],
        vec![3, 9, 18],
        vec![4, 8, 19],
        vec![0, 11, 15],
        vec![1, 10, 12, 16],
        vec![2, 11, 17],
        vec![3, 14, 18],
        vec![4, 13, 19],
        vec![5, 10, 16],
        vec![6, 11, 15, 17],
        vec![7, 12, 16],
        vec![8, 13, 19],
        vec![9, 14, 18],
    ];
    let exterior_nodes_gold = (0..20).collect();
    let interface_nodes_gold = vec![1, 6, 11, 16];
    let interior_nodes_gold = vec![];
    let nodal_influencers_gold = node_node_connectivity_gold.clone();
    test_finite_elements(
        element_blocks,
        element_node_connectivity,
        nodal_coordinates,
        node_element_connectivity_gold,
        node_node_connectivity_gold,
        exterior_nodes_gold,
        interface_nodes_gold,
        interior_nodes_gold,
        None,
        None,
        nodal_influencers_gold,
    );
}

#[test]
fn cube() {
    let element_blocks = vec![11; 8];
    let element_node_connectivity = vec![
        [0, 1, 4, 3, 9, 10, 13, 12],
        [1, 2, 5, 4, 10, 11, 14, 13],
        [3, 4, 7, 6, 12, 13, 16, 15],
        [4, 5, 8, 7, 13, 14, 17, 16],
        [9, 10, 13, 12, 18, 19, 22, 21],
        [10, 11, 14, 13, 19, 20, 23, 22],
        [12, 13, 16, 15, 21, 22, 25, 24],
        [13, 14, 17, 16, 22, 23, 26, 25],
    ];
    let nodal_coordinates = Coordinates::new(&[
        [0.0, 0.0, 0.0],
        [1.0, 0.0, 0.0],
        [2.0, 0.0, 0.0],
        [0.0, 1.0, 0.0],
        [1.0, 1.0, 0.0],
        [2.0, 1.0, 0.0],
        [0.0, 2.0, 0.0],
        [1.0, 2.0, 0.0],
        [2.0, 2.0, 0.0],
        [0.0, 0.0, 1.0],
        [1.0, 0.0, 1.0],
        [2.0, 0.0, 1.0],
        [0.0, 1.0, 1.0],
        [1.0, 1.0, 1.0],
        [2.0, 1.0, 1.0],
        [0.0, 2.0, 1.0],
        [1.0, 2.0, 1.0],
        [2.0, 2.0, 1.0],
        [0.0, 0.0, 2.0],
        [1.0, 0.0, 2.0],
        [2.0, 0.0, 2.0],
        [0.0, 1.0, 2.0],
        [1.0, 1.0, 2.0],
        [2.0, 1.0, 2.0],
        [0.0, 2.0, 2.0],
        [1.0, 2.0, 2.0],
        [2.0, 2.0, 2.0],
    ]);
    let node_element_connectivity_gold = vec![
        vec![0],
        vec![0, 1],
        vec![1],
        vec![0, 2],
        vec![0, 1, 2, 3],
        vec![1, 3],
        vec![2],
        vec![2, 3],
        vec![3],
        vec![0, 4],
        vec![0, 1, 4, 5],
        vec![1, 5],
        vec![0, 2, 4, 6],
        vec![0, 1, 2, 3, 4, 5, 6, 7],
        vec![1, 3, 5, 7],
        vec![2, 6],
        vec![2, 3, 6, 7],
        vec![3, 7],
        vec![4],
        vec![4, 5],
        vec![5],
        vec![4, 6],
        vec![4, 5, 6, 7],
        vec![5, 7],
        vec![6],
        vec![6, 7],
        vec![7],
    ];
    let node_node_connectivity_gold = vec![
        vec![1, 3, 9],
        vec![0, 2, 4, 10],
        vec![1, 5, 11],
        vec![0, 4, 6, 12],
        vec![1, 3, 5, 7, 13],
        vec![2, 4, 8, 14],
        vec![3, 7, 15],
        vec![4, 6, 8, 16],
        vec![5, 7, 17],
        vec![0, 10, 12, 18],
        vec![1, 9, 11, 13, 19],
        vec![2, 10, 14, 20],
        vec![3, 9, 13, 15, 21],
        vec![4, 10, 12, 14, 16, 22],
        vec![5, 11, 13, 17, 23],
        vec![6, 12, 16, 24],
        vec![7, 13, 15, 17, 25],
        vec![8, 14, 16, 26],
        vec![9, 19, 21],
        vec![10, 18, 20, 22],
        vec![11, 19, 23],
        vec![12, 18, 22, 24],
        vec![13, 19, 21, 23, 25],
        vec![14, 20, 22, 26],
        vec![15, 21, 25],
        vec![16, 22, 24, 26],
        vec![17, 23, 25],
    ];
    let mut exterior_nodes_gold = (0..27).collect::<Nodes>();
    exterior_nodes_gold.remove(13);
    let interface_nodes_gold = vec![];
    let interior_nodes_gold = vec![13];
    let nodal_influencers_gold = vec![
        vec![1, 3, 9],
        vec![0, 2, 4, 10],
        vec![1, 5, 11],
        vec![0, 4, 6, 12],
        vec![1, 3, 5, 7],
        vec![2, 4, 8, 14],
        vec![3, 7, 15],
        vec![4, 6, 8, 16],
        vec![5, 7, 17],
        vec![0, 10, 12, 18],
        vec![1, 9, 11, 19],
        vec![2, 10, 14, 20],
        vec![3, 9, 15, 21],
        vec![4, 10, 12, 14, 16, 22],
        vec![5, 11, 17, 23],
        vec![6, 12, 16, 24],
        vec![7, 15, 17, 25],
        vec![8, 14, 16, 26],
        vec![9, 19, 21],
        vec![10, 18, 20, 22],
        vec![11, 19, 23],
        vec![12, 18, 22, 24],
        vec![19, 21, 23, 25],
        vec![14, 20, 22, 26],
        vec![15, 21, 25],
        vec![16, 22, 24, 26],
        vec![17, 23, 25],
    ];
    test_finite_elements(
        element_blocks,
        element_node_connectivity,
        nodal_coordinates,
        node_element_connectivity_gold,
        node_node_connectivity_gold,
        exterior_nodes_gold,
        interface_nodes_gold,
        interior_nodes_gold,
        None,
        None,
        nodal_influencers_gold,
    );
}

#[test]
fn cube_multi() {
    let element_blocks = vec![82, 2, 2, 2, 31, 44];
    let element_node_connectivity = vec![
        [0, 1, 4, 3, 9, 10, 13, 12],
        [1, 2, 5, 4, 10, 11, 14, 13],
        [3, 4, 7, 6, 12, 13, 16, 15],
        [4, 5, 8, 7, 13, 14, 17, 16],
        [10, 11, 14, 13, 18, 19, 21, 20],
        [13, 14, 17, 16, 20, 21, 23, 22],
    ];
    let nodal_coordinates = Coordinates::new(&[
        [0.0, 0.0, 0.0],
        [1.0, 0.0, 0.0],
        [2.0, 0.0, 0.0],
        [0.0, 1.0, 0.0],
        [1.0, 1.0, 0.0],
        [2.0, 1.0, 0.0],
        [0.0, 2.0, 0.0],
        [1.0, 2.0, 0.0],
        [2.0, 2.0, 0.0],
        [0.0, 0.0, 1.0],
        [1.0, 0.0, 1.0],
        [2.0, 0.0, 1.0],
        [0.0, 1.0, 1.0],
        [1.0, 1.0, 1.0],
        [2.0, 1.0, 1.0],
        [0.0, 2.0, 1.0],
        [1.0, 2.0, 1.0],
        [2.0, 2.0, 1.0],
        [1.0, 0.0, 2.0],
        [2.0, 0.0, 2.0],
        [1.0, 1.0, 2.0],
        [2.0, 1.0, 2.0],
        [1.0, 2.0, 2.0],
        [2.0, 2.0, 2.0],
    ]);
    let node_element_connectivity_gold = vec![
        vec![0],
        vec![0, 1],
        vec![1],
        vec![0, 2],
        vec![0, 1, 2, 3],
        vec![1, 3],
        vec![2],
        vec![2, 3],
        vec![3],
        vec![0],
        vec![0, 1, 4],
        vec![1, 4],
        vec![0, 2],
        vec![0, 1, 2, 3, 4, 5],
        vec![1, 3, 4, 5],
        vec![2],
        vec![2, 3, 5],
        vec![3, 5],
        vec![4],
        vec![4],
        vec![4, 5],
        vec![4, 5],
        vec![5],
        vec![5],
    ];
    let node_node_connectivity_gold = vec![
        vec![1, 3, 9],
        vec![0, 2, 4, 10],
        vec![1, 5, 11],
        vec![0, 4, 6, 12],
        vec![1, 3, 5, 7, 13],
        vec![2, 4, 8, 14],
        vec![3, 7, 15],
        vec![4, 6, 8, 16],
        vec![5, 7, 17],
        vec![0, 10, 12],
        vec![1, 9, 11, 13, 18],
        vec![2, 10, 14, 19],
        vec![3, 9, 13, 15],
        vec![4, 10, 12, 14, 16, 20],
        vec![5, 11, 13, 17, 21],
        vec![6, 12, 16],
        vec![7, 13, 15, 17, 22],
        vec![8, 14, 16, 23],
        vec![10, 19, 20],
        vec![11, 18, 21],
        vec![13, 18, 21, 22],
        vec![14, 19, 20, 23],
        vec![16, 20, 23],
        vec![17, 21, 22],
    ];
    let exterior_nodes_gold = (0..24).collect();
    let interface_nodes_gold = vec![1, 3, 4, 10, 11, 12, 13, 14, 16, 17, 20, 21];
    let interior_nodes_gold = vec![];
    let nodal_influencers_gold = node_node_connectivity_gold.clone();
    test_finite_elements(
        element_blocks,
        element_node_connectivity,
        nodal_coordinates,
        node_element_connectivity_gold,
        node_node_connectivity_gold,
        exterior_nodes_gold,
        interface_nodes_gold,
        interior_nodes_gold,
        None,
        None,
        nodal_influencers_gold,
    );
}

#[test]
fn cube_with_inclusion() {
    let mut element_blocks = vec![11; 27];
    element_blocks[13] = 88;
    let element_node_connectivity = vec![
        [0, 1, 5, 4, 16, 17, 21, 20],
        [1, 2, 6, 5, 17, 18, 22, 21],
        [2, 3, 7, 6, 18, 19, 23, 22],
        [4, 5, 9, 8, 20, 21, 25, 24],
        [5, 6, 10, 9, 21, 22, 26, 25],
        [6, 7, 11, 10, 22, 23, 27, 26],
        [8, 9, 13, 12, 24, 25, 29, 28],
        [9, 10, 14, 13, 25, 26, 30, 29],
        [10, 11, 15, 14, 26, 27, 31, 30],
        [16, 17, 21, 20, 32, 33, 37, 36],
        [17, 18, 22, 21, 33, 34, 38, 37],
        [18, 19, 23, 22, 34, 35, 39, 38],
        [20, 21, 25, 24, 36, 37, 41, 40],
        [21, 22, 26, 25, 37, 38, 42, 41],
        [22, 23, 27, 26, 38, 39, 43, 42],
        [24, 25, 29, 28, 40, 41, 45, 44],
        [25, 26, 30, 29, 41, 42, 46, 45],
        [26, 27, 31, 30, 42, 43, 47, 46],
        [32, 33, 37, 36, 48, 49, 53, 52],
        [33, 34, 38, 37, 49, 50, 54, 53],
        [34, 35, 39, 38, 50, 51, 55, 54],
        [36, 37, 41, 40, 52, 53, 57, 56],
        [37, 38, 42, 41, 53, 54, 58, 57],
        [38, 39, 43, 42, 54, 55, 59, 58],
        [40, 41, 45, 44, 56, 57, 61, 60],
        [41, 42, 46, 45, 57, 58, 62, 61],
        [42, 43, 47, 46, 58, 59, 63, 62],
    ];
    let nodal_coordinates = Coordinates::new(&[
        [0.0, 0.0, 0.0],
        [1.0, 0.0, 0.0],
        [2.0, 0.0, 0.0],
        [3.0, 0.0, 0.0],
        [0.0, 1.0, 0.0],
        [1.0, 1.0, 0.0],
        [2.0, 1.0, 0.0],
        [3.0, 1.0, 0.0],
        [0.0, 2.0, 0.0],
        [1.0, 2.0, 0.0],
        [2.0, 2.0, 0.0],
        [3.0, 2.0, 0.0],
        [0.0, 3.0, 0.0],
        [1.0, 3.0, 0.0],
        [2.0, 3.0, 0.0],
        [3.0, 3.0, 0.0],
        [0.0, 0.0, 1.0],
        [1.0, 0.0, 1.0],
        [2.0, 0.0, 1.0],
        [3.0, 0.0, 1.0],
        [0.0, 1.0, 1.0],
        [1.0, 1.0, 1.0],
        [2.0, 1.0, 1.0],
        [3.0, 1.0, 1.0],
        [0.0, 2.0, 1.0],
        [1.0, 2.0, 1.0],
        [2.0, 2.0, 1.0],
        [3.0, 2.0, 1.0],
        [0.0, 3.0, 1.0],
        [1.0, 3.0, 1.0],
        [2.0, 3.0, 1.0],
        [3.0, 3.0, 1.0],
        [0.0, 0.0, 2.0],
        [1.0, 0.0, 2.0],
        [2.0, 0.0, 2.0],
        [3.0, 0.0, 2.0],
        [0.0, 1.0, 2.0],
        [1.0, 1.0, 2.0],
        [2.0, 1.0, 2.0],
        [3.0, 1.0, 2.0],
        [0.0, 2.0, 2.0],
        [1.0, 2.0, 2.0],
        [2.0, 2.0, 2.0],
        [3.0, 2.0, 2.0],
        [0.0, 3.0, 2.0],
        [1.0, 3.0, 2.0],
        [2.0, 3.0, 2.0],
        [3.0, 3.0, 2.0],
        [0.0, 0.0, 3.0],
        [1.0, 0.0, 3.0],
        [2.0, 0.0, 3.0],
        [3.0, 0.0, 3.0],
        [0.0, 1.0, 3.0],
        [1.0, 1.0, 3.0],
        [2.0, 1.0, 3.0],
        [3.0, 1.0, 3.0],
        [0.0, 2.0, 3.0],
        [1.0, 2.0, 3.0],
        [2.0, 2.0, 3.0],
        [3.0, 2.0, 3.0],
        [0.0, 3.0, 3.0],
        [1.0, 3.0, 3.0],
        [2.0, 3.0, 3.0],
        [3.0, 3.0, 3.0],
    ]);
    let node_element_connectivity_gold = vec![
        vec![0],
        vec![0, 1],
        vec![1, 2],
        vec![2],
        vec![0, 3],
        vec![0, 1, 3, 4],
        vec![1, 2, 4, 5],
        vec![2, 5],
        vec![3, 6],
        vec![3, 4, 6, 7],
        vec![4, 5, 7, 8],
        vec![5, 8],
        vec![6],
        vec![6, 7],
        vec![7, 8],
        vec![8],
        vec![0, 9],
        vec![0, 1, 9, 10],
        vec![1, 2, 10, 11],
        vec![2, 11],
        vec![0, 3, 9, 12],
        vec![0, 1, 3, 4, 9, 10, 12, 13],
        vec![1, 2, 4, 5, 10, 11, 13, 14],
        vec![2, 5, 11, 14],
        vec![3, 6, 12, 15],
        vec![3, 4, 6, 7, 12, 13, 15, 16],
        vec![4, 5, 7, 8, 13, 14, 16, 17],
        vec![5, 8, 14, 17],
        vec![6, 15],
        vec![6, 7, 15, 16],
        vec![7, 8, 16, 17],
        vec![8, 17],
        vec![9, 18],
        vec![9, 10, 18, 19],
        vec![10, 11, 19, 20],
        vec![11, 20],
        vec![9, 12, 18, 21],
        vec![9, 10, 12, 13, 18, 19, 21, 22],
        vec![10, 11, 13, 14, 19, 20, 22, 23],
        vec![11, 14, 20, 23],
        vec![12, 15, 21, 24],
        vec![12, 13, 15, 16, 21, 22, 24, 25],
        vec![13, 14, 16, 17, 22, 23, 25, 26],
        vec![14, 17, 23, 26],
        vec![15, 24],
        vec![15, 16, 24, 25],
        vec![16, 17, 25, 26],
        vec![17, 26],
        vec![18],
        vec![18, 19],
        vec![19, 20],
        vec![20],
        vec![18, 21],
        vec![18, 19, 21, 22],
        vec![19, 20, 22, 23],
        vec![20, 23],
        vec![21, 24],
        vec![21, 22, 24, 25],
        vec![22, 23, 25, 26],
        vec![23, 26],
        vec![24],
        vec![24, 25],
        vec![25, 26],
        vec![26],
    ];
    let node_node_connectivity_gold = vec![
        vec![1, 4, 16],
        vec![0, 2, 5, 17],
        vec![1, 3, 6, 18],
        vec![2, 7, 19],
        vec![0, 5, 8, 20],
        vec![1, 4, 6, 9, 21],
        vec![2, 5, 7, 10, 22],
        vec![3, 6, 11, 23],
        vec![4, 9, 12, 24],
        vec![5, 8, 10, 13, 25],
        vec![6, 9, 11, 14, 26],
        vec![7, 10, 15, 27],
        vec![8, 13, 28],
        vec![9, 12, 14, 29],
        vec![10, 13, 15, 30],
        vec![11, 14, 31],
        vec![0, 17, 20, 32],
        vec![1, 16, 18, 21, 33],
        vec![2, 17, 19, 22, 34],
        vec![3, 18, 23, 35],
        vec![4, 16, 21, 24, 36],
        vec![5, 17, 20, 22, 25, 37],
        vec![6, 18, 21, 23, 26, 38],
        vec![7, 19, 22, 27, 39],
        vec![8, 20, 25, 28, 40],
        vec![9, 21, 24, 26, 29, 41],
        vec![10, 22, 25, 27, 30, 42],
        vec![11, 23, 26, 31, 43],
        vec![12, 24, 29, 44],
        vec![13, 25, 28, 30, 45],
        vec![14, 26, 29, 31, 46],
        vec![15, 27, 30, 47],
        vec![16, 33, 36, 48],
        vec![17, 32, 34, 37, 49],
        vec![18, 33, 35, 38, 50],
        vec![19, 34, 39, 51],
        vec![20, 32, 37, 40, 52],
        vec![21, 33, 36, 38, 41, 53],
        vec![22, 34, 37, 39, 42, 54],
        vec![23, 35, 38, 43, 55],
        vec![24, 36, 41, 44, 56],
        vec![25, 37, 40, 42, 45, 57],
        vec![26, 38, 41, 43, 46, 58],
        vec![27, 39, 42, 47, 59],
        vec![28, 40, 45, 60],
        vec![29, 41, 44, 46, 61],
        vec![30, 42, 45, 47, 62],
        vec![31, 43, 46, 63],
        vec![32, 49, 52],
        vec![33, 48, 50, 53],
        vec![34, 49, 51, 54],
        vec![35, 50, 55],
        vec![36, 48, 53, 56],
        vec![37, 49, 52, 54, 57],
        vec![38, 50, 53, 55, 58],
        vec![39, 51, 54, 59],
        vec![40, 52, 57, 60],
        vec![41, 53, 56, 58, 61],
        vec![42, 54, 57, 59, 62],
        vec![43, 55, 58, 63],
        vec![44, 56, 61],
        vec![45, 57, 60, 62],
        vec![46, 58, 61, 63],
        vec![47, 59, 62],
    ];
    let interface_nodes_gold = vec![21, 22, 25, 26, 37, 38, 41, 42];
    let exterior_nodes_gold = (0..64)
        .filter(|node| !interface_nodes_gold.contains(node))
        .collect();
    let interior_nodes_gold = vec![];
    let nodal_influencers_gold = node_node_connectivity_gold.clone();
    test_finite_elements(
        element_blocks,
        element_node_connectivity,
        nodal_coordinates,
        node_element_connectivity_gold,
        node_node_connectivity_gold,
        exterior_nodes_gold,
        interface_nodes_gold,
        interior_nodes_gold,
        None,
        None,
        nodal_influencers_gold,
    );
}

#[test]
fn bracket() {
    let element_blocks = vec![1; 12];
    let element_node_connectivity = vec![
        [0, 1, 6, 5, 21, 22, 27, 26],
        [1, 2, 7, 6, 22, 23, 28, 27],
        [2, 3, 8, 7, 23, 24, 29, 28],
        [3, 4, 9, 8, 24, 25, 30, 29],
        [5, 6, 11, 10, 26, 27, 32, 31],
        [6, 7, 12, 11, 27, 28, 33, 32],
        [7, 8, 13, 12, 28, 29, 34, 33],
        [8, 9, 14, 13, 29, 30, 35, 34],
        [10, 11, 16, 15, 31, 32, 37, 36],
        [11, 12, 17, 16, 32, 33, 38, 37],
        [15, 16, 19, 18, 36, 37, 40, 39],
        [16, 17, 20, 19, 37, 38, 41, 40],
    ];
    let nodal_coordinates = Coordinates::new(&[
        [0.0, 0.0, 0.0],
        [1.0, 0.0, 0.0],
        [2.0, 0.0, 0.0],
        [3.0, 0.0, 0.0],
        [4.0, 0.0, 0.0],
        [0.0, 1.0, 0.0],
        [1.0, 1.0, 0.0],
        [2.0, 1.0, 0.0],
        [3.0, 1.0, 0.0],
        [4.0, 1.0, 0.0],
        [0.0, 2.0, 0.0],
        [1.0, 2.0, 0.0],
        [2.0, 2.0, 0.0],
        [3.0, 2.0, 0.0],
        [4.0, 2.0, 0.0],
        [0.0, 3.0, 0.0],
        [1.0, 3.0, 0.0],
        [2.0, 3.0, 0.0],
        [0.0, 4.0, 0.0],
        [1.0, 4.0, 0.0],
        [2.0, 4.0, 0.0],
        [0.0, 0.0, 1.0],
        [1.0, 0.0, 1.0],
        [2.0, 0.0, 1.0],
        [3.0, 0.0, 1.0],
        [4.0, 0.0, 1.0],
        [0.0, 1.0, 1.0],
        [1.0, 1.0, 1.0],
        [2.0, 1.0, 1.0],
        [3.0, 1.0, 1.0],
        [4.0, 1.0, 1.0],
        [0.0, 2.0, 1.0],
        [1.0, 2.0, 1.0],
        [2.0, 2.0, 1.0],
        [3.0, 2.0, 1.0],
        [4.0, 2.0, 1.0],
        [0.0, 3.0, 1.0],
        [1.0, 3.0, 1.0],
        [2.0, 3.0, 1.0],
        [0.0, 4.0, 1.0],
        [1.0, 4.0, 1.0],
        [2.0, 4.0, 1.0],
    ]);
    let node_element_connectivity_gold = vec![
        vec![0],
        vec![0, 1],
        vec![1, 2],
        vec![2, 3],
        vec![3],
        vec![0, 4],
        vec![0, 1, 4, 5],
        vec![1, 2, 5, 6],
        vec![2, 3, 6, 7],
        vec![3, 7],
        vec![4, 8],
        vec![4, 5, 8, 9],
        vec![5, 6, 9],
        vec![6, 7],
        vec![7],
        vec![8, 10],
        vec![8, 9, 10, 11],
        vec![9, 11],
        vec![10],
        vec![10, 11],
        vec![11],
        vec![0],
        vec![0, 1],
        vec![1, 2],
        vec![2, 3],
        vec![3],
        vec![0, 4],
        vec![0, 1, 4, 5],
        vec![1, 2, 5, 6],
        vec![2, 3, 6, 7],
        vec![3, 7],
        vec![4, 8],
        vec![4, 5, 8, 9],
        vec![5, 6, 9],
        vec![6, 7],
        vec![7],
        vec![8, 10],
        vec![8, 9, 10, 11],
        vec![9, 11],
        vec![10],
        vec![10, 11],
        vec![11],
    ];
    let node_node_connectivity_gold = vec![
        vec![1, 5, 21],
        vec![0, 2, 6, 22],
        vec![1, 3, 7, 23],
        vec![2, 4, 8, 24],
        vec![3, 9, 25],
        vec![0, 6, 10, 26],
        vec![1, 5, 7, 11, 27],
        vec![2, 6, 8, 12, 28],
        vec![3, 7, 9, 13, 29],
        vec![4, 8, 14, 30],
        vec![5, 11, 15, 31],
        vec![6, 10, 12, 16, 32],
        vec![7, 11, 13, 17, 33],
        vec![8, 12, 14, 34],
        vec![9, 13, 35],
        vec![10, 16, 18, 36],
        vec![11, 15, 17, 19, 37],
        vec![12, 16, 20, 38],
        vec![15, 19, 39],
        vec![16, 18, 20, 40],
        vec![17, 19, 41],
        vec![0, 22, 26],
        vec![1, 21, 23, 27],
        vec![2, 22, 24, 28],
        vec![3, 23, 25, 29],
        vec![4, 24, 30],
        vec![5, 21, 27, 31],
        vec![6, 22, 26, 28, 32],
        vec![7, 23, 27, 29, 33],
        vec![8, 24, 28, 30, 34],
        vec![9, 25, 29, 35],
        vec![10, 26, 32, 36],
        vec![11, 27, 31, 33, 37],
        vec![12, 28, 32, 34, 38],
        vec![13, 29, 33, 35],
        vec![14, 30, 34],
        vec![15, 31, 37, 39],
        vec![16, 32, 36, 38, 40],
        vec![17, 33, 37, 41],
        vec![18, 36, 40],
        vec![19, 37, 39, 41],
        vec![20, 38, 40],
    ];
    let exterior_nodes_gold = (0..42).collect();
    let interface_nodes_gold = vec![];
    let interior_nodes_gold = vec![];
    let nodal_influencers_gold = node_node_connectivity_gold.clone();
    let laplacian_gold = Coordinates::new(&[
        [ONE_THIRD, ONE_THIRD, ONE_THIRD],
        [ZERO, ONE_FOURTH, ONE_FOURTH],
        [ZERO, ONE_FOURTH, ONE_FOURTH],
        [ZERO, ONE_FOURTH, ONE_FOURTH],
        [-ONE_THIRD, ONE_THIRD, ONE_THIRD],
        [ONE_FOURTH, ZERO, ONE_FOURTH],
        [ZERO, ZERO, ONE_FIFTH],
        [ZERO, ZERO, ONE_FIFTH],
        [ZERO, ZERO, ONE_FIFTH],
        [-ONE_FOURTH, ZERO, ONE_FOURTH],
        [ONE_FOURTH, ZERO, ONE_FOURTH],
        [ZERO, ZERO, ONE_FIFTH],
        [ZERO, ZERO, ONE_FIFTH],
        [ZERO, -ONE_FOURTH, ONE_FOURTH],
        [-ONE_THIRD, -ONE_THIRD, ONE_THIRD],
        [ONE_FOURTH, ZERO, ONE_FOURTH],
        [ZERO, ZERO, ONE_FIFTH],
        [-ONE_FOURTH, ZERO, ONE_FOURTH],
        [ONE_THIRD, -ONE_THIRD, ONE_THIRD],
        [ZERO, -ONE_FOURTH, ONE_FOURTH],
        [-ONE_THIRD, -ONE_THIRD, ONE_THIRD],
        [ONE_THIRD, ONE_THIRD, -ONE_THIRD],
        [ZERO, ONE_FOURTH, -ONE_FOURTH],
        [ZERO, ONE_FOURTH, -ONE_FOURTH],
        [ZERO, ONE_FOURTH, -ONE_FOURTH],
        [-ONE_THIRD, ONE_THIRD, -ONE_THIRD],
        [ONE_FOURTH, ZERO, -ONE_FOURTH],
        [ZERO, ZERO, -ONE_FIFTH],
        [ZERO, ZERO, -ONE_FIFTH],
        [ZERO, ZERO, -ONE_FIFTH],
        [-ONE_FOURTH, ZERO, -ONE_FOURTH],
        [ONE_FOURTH, ZERO, -ONE_FOURTH],
        [ZERO, ZERO, -ONE_FIFTH],
        [ZERO, ZERO, -ONE_FIFTH],
        [ZERO, -ONE_FOURTH, -ONE_FOURTH],
        [-ONE_THIRD, -ONE_THIRD, -ONE_THIRD],
        [ONE_FOURTH, ZERO, -ONE_FOURTH],
        [ZERO, ZERO, -ONE_FIFTH],
        [-ONE_FOURTH, ZERO, -ONE_FOURTH],
        [ONE_THIRD, -ONE_THIRD, -ONE_THIRD],
        [ZERO, -ONE_FOURTH, -ONE_FOURTH],
        [-ONE_THIRD, -ONE_THIRD, -ONE_THIRD],
    ]);
    let smoothed_coordinates_gold = vec![
        Coordinates::new(&[
            [0.1, 0.1, 0.1],
            [1.0, 0.075, 0.075],
            [2.0, 0.075, 0.075],
            [3.0, 0.075, 0.075],
            [3.9, 0.1, 0.1],
            [0.075, 1.0, 0.075],
            [1.0, 1.0, 0.06],
            [2.0, 1.0, 0.06],
            [3.0, 1.0, 0.06],
            [3.925, 1.0, 0.075],
            [0.075, 2.0, 0.075],
            [1.0, 2.0, 0.06],
            [2.0, 2.0, 0.06],
            [3.0, 1.925, 0.075],
            [3.9, 1.9, 0.1],
            [0.075, 3.0, 0.075],
            [1.0, 3.0, 0.06],
            [1.925, 3.0, 0.075],
            [0.1, 3.9, 0.1],
            [1.0, 3.925, 0.075],
            [1.9, 3.9, 0.1],
            [0.1, 0.1, 0.9],
            [1.0, 0.075, 0.925],
            [2.0, 0.075, 0.925],
            [3.0, 0.075, 0.925],
            [3.9, 0.1, 0.9],
            [0.075, 1.0, 0.925],
            [1.0, 1.0, 0.94],
            [2.0, 1.0, 0.94],
            [3.0, 1.0, 0.94],
            [3.925, 1.0, 0.925],
            [0.075, 2.0, 0.925],
            [1.0, 2.0, 0.94],
            [2.0, 2.0, 0.94],
            [3.0, 1.925, 0.925],
            [3.9, 1.9, 0.9],
            [0.075, 3.0, 0.925],
            [1.0, 3.0, 0.94],
            [1.925, 3.0, 0.925],
            [0.1, 3.9, 0.9],
            [1.0, 3.925, 0.925],
            [1.9, 3.9, 0.9],
        ]),
        Coordinates::new(&[
            [0.1875, 0.1875, 0.175],
            [1.0075, 0.14625, 0.1395],
            [2.0, 0.144375, 0.137625],
            [2.9925, 0.14625, 0.1395],
            [3.8125, 0.1875, 0.175],
            [0.14625, 1.0075, 0.1395],
            [1.0045, 1.0045, 0.1146],
            [2.0, 1.0045, 0.1137],
            [2.9955, 1.0, 0.1155],
            [3.851875, 1.0, 0.141375],
            [0.144375, 2.0, 0.137625],
            [1.0045, 2.0, 0.1137],
            [1.9955, 1.9955, 0.1146],
            [2.9925, 1.859375, 0.138375],
            [3.8125, 1.8125, 0.175],
            [0.14625, 2.9925, 0.1395],
            [1.0, 2.9955, 0.1155],
            [1.859375, 2.9925, 0.138375],
            [0.1875, 3.8125, 0.175],
            [1.0, 3.851875, 0.141375],
            [1.8125, 3.8125, 0.175],
            [0.1875, 0.1875, 0.8250],
            [1.0075, 0.14625, 0.8605],
            [2.0, 0.144375, 0.862375],
            [2.9925, 0.14625, 0.8605],
            [3.8125, 0.1875, 0.8250],
            [0.14625, 1.0075, 0.8605],
            [1.0045, 1.0045, 0.8854],
            [2.0, 1.0045, 0.8863],
            [2.9955, 1.0, 0.8845],
            [3.851875, 1.0, 0.858625],
            [0.144375, 2.0, 0.862375],
            [1.0045, 2.0, 0.8863],
            [1.9955, 1.9955, 0.8854],
            [2.9925, 1.859375, 0.861625],
            [3.8125, 1.8125, 0.8250],
            [0.14625, 2.9925, 0.8605],
            [1.0, 2.9955, 0.8845],
            [1.859375, 2.9925, 0.861625],
            [0.1875, 3.8125, 0.8250],
            [1.0, 3.851875, 0.858625],
            [1.8125, 3.8125, 0.8250],
        ]),
    ];
    test_finite_elements(
        element_blocks.clone(),
        element_node_connectivity.clone(),
        nodal_coordinates.clone_foo(),
        node_element_connectivity_gold,
        node_node_connectivity_gold,
        exterior_nodes_gold,
        interface_nodes_gold,
        interior_nodes_gold,
        Some(laplacian_gold),
        Some(smoothed_coordinates_gold),
        nodal_influencers_gold,
    );
    let cos_15 = 15.0_f64.to_radians().cos();
    let cos_30 = 30.0_f64.to_radians().cos();
    let sin_15 = 15.0_f64.to_radians().sin();
    let sin_30 = 30.0_f64.to_radians().sin();
    let prescribed_nodes_homogeneous = vec![
        0, 1, 2, 3, 4, 5, 10, 15, 18, 21, 22, 23, 24, 25, 26, 31, 36, 39,
    ];
    let prescribed_nodes_inhomogeneous = vec![9, 14, 19, 20, 30, 35, 40, 41];
    let prescribed_nodes_inhomogeneous_coordinates = Coordinates::new(&[
        [4.5 * cos_15, 4.5 * sin_15, 0.0],
        [4.5 * cos_30, 4.5 * sin_30, 0.0],
        [1.5, 4.0, 0.0],
        [3.5, 4.0, 0.0],
        [4.5 * cos_15, 4.5 * sin_15, 1.0],
        [4.5 * cos_30, 4.5 * sin_30, 1.0],
        [1.5, 4.0, 1.0],
        [3.5, 4.0, 1.0],
    ]);
    let mut finite_elements = HexahedralFiniteElements::from((
        element_blocks,
        element_node_connectivity,
        nodal_coordinates,
    ));
    finite_elements.node_element_connectivity().unwrap();
    finite_elements.node_node_connectivity().unwrap();
    finite_elements.nodal_hierarchy().unwrap();
    finite_elements
        .set_prescribed_nodes(
            Some(prescribed_nodes_homogeneous),
            Some((
                prescribed_nodes_inhomogeneous_coordinates,
                prescribed_nodes_inhomogeneous,
            )),
        )
        .unwrap();
    finite_elements.nodal_influencers();
    finite_elements
        .smooth(&Smoothing::Laplacian(10, SMOOTHING_SCALE))
        .unwrap();
    finite_elements
        .get_nodal_coordinates()
        .iter()
        .zip(
            vec![
                vec![0.0, 0.0, 0.0],
                vec![1.0, 0.0, 0.0],
                vec![2.0, 0.0, 0.0],
                vec![3.0, 0.0, 0.0],
                vec![4.0, 0.0, 0.0],
                vec![0.0, 1.0, 0.0],
                vec![1.0076218690550747, 0.9988829259123082, 0.24593434133370803],
                vec![2.0218051968023, 0.993985105791881, 0.2837944855813176],
                vec![3.0816593568068398, 0.9931227966186256, 0.24898414051620496],
                vec![4.346666218300808, 1.1646857029613433, 0.0],
                vec![0.0, 2.0, 0.0],
                vec![1.0346002406957664, 1.992982526945126, 0.2837944855813176],
                vec![2.0408618916639916, 1.9528647520642073, 0.3332231502067546],
                vec![2.9955771790244468, 1.7619821132207711, 0.29909606343914835],
                vec![3.897114317029974, 2.2499999999999996, 0.0],
                vec![0.0, 3.0, 0.0],
                vec![1.157261281731803, 2.9982665159532105, 0.24898414051620493],
                vec![2.1973691292662734, 2.991054895165017, 0.29909606343914835],
                vec![0.0, 4.0, 0.0],
                vec![1.5, 4.0, 0.0],
                vec![3.5, 4.0, 0.0],
                vec![0.0, 0.0, 1.0],
                vec![1.0, 0.0, 1.0],
                vec![2.0, 0.0, 1.0],
                vec![3.0, 0.0, 1.0],
                vec![4.0, 0.0, 1.0],
                vec![0.0, 1.0, 1.0],
                vec![1.0076218690550747, 0.9988829259123082, 0.7540656586662919],
                vec![2.0218051968023, 0.993985105791881, 0.7162055144186824],
                vec![3.0816593568068398, 0.9931227966186257, 0.7510158594837951],
                vec![4.346666218300808, 1.1646857029613433, 1.0],
                vec![0.0, 2.0, 1.0],
                vec![1.0346002406957664, 1.9929825269451262, 0.7162055144186824],
                vec![2.0408618916639916, 1.9528647520642073, 0.6667768497932453],
                vec![2.9955771790244468, 1.7619821132207711, 0.7009039365608517],
                vec![3.897114317029974, 2.2499999999999996, 1.0],
                vec![0.0, 3.0, 1.0],
                vec![1.157261281731803, 2.9982665159532105, 0.751015859483795],
                vec![2.1973691292662734, 2.991054895165017, 0.7009039365608516],
                vec![0.0, 4.0, 1.0],
                vec![1.5, 4.0, 1.0],
                vec![3.5, 4.0, 1.0],
            ]
            .iter(),
        )
        .for_each(|(data, gold)| {
            data.iter()
                .zip(gold.iter())
                .for_each(|(data_entry, gold_entry)| assert_eq!(data_entry, gold_entry))
        });
}

#[test]
fn letter_f() {
    let element_blocks = vec![11; 8];
    let element_node_connectivity = vec![
        [0, 1, 3, 2, 18, 19, 21, 20],
        [2, 3, 5, 4, 20, 21, 23, 22],
        [4, 5, 8, 7, 22, 23, 26, 25],
        [5, 6, 9, 8, 23, 24, 27, 26],
        [7, 8, 11, 10, 25, 26, 29, 28],
        [10, 11, 15, 14, 28, 29, 33, 32],
        [11, 12, 16, 15, 29, 30, 34, 33],
        [12, 13, 17, 16, 30, 31, 35, 34],
    ];
    let nodal_coordinates = Coordinates::new(&[
        [0.0, 0.0, 0.0],
        [1.0, 0.0, 0.0],
        [0.0, 1.0, 0.0],
        [1.0, 1.0, 0.0],
        [0.0, 2.0, 0.0],
        [1.0, 2.0, 0.0],
        [2.0, 2.0, 0.0],
        [0.0, 3.0, 0.0],
        [1.0, 3.0, 0.0],
        [2.0, 3.0, 0.0],
        [0.0, 4.0, 0.0],
        [1.0, 4.0, 0.0],
        [2.0, 4.0, 0.0],
        [3.0, 4.0, 0.0],
        [0.0, 5.0, 0.0],
        [1.0, 5.0, 0.0],
        [2.0, 5.0, 0.0],
        [3.0, 5.0, 0.0],
        [0.0, 0.0, 1.0],
        [1.0, 0.0, 1.0],
        [0.0, 1.0, 1.0],
        [1.0, 1.0, 1.0],
        [0.0, 2.0, 1.0],
        [1.0, 2.0, 1.0],
        [2.0, 2.0, 1.0],
        [0.0, 3.0, 1.0],
        [1.0, 3.0, 1.0],
        [2.0, 3.0, 1.0],
        [0.0, 4.0, 1.0],
        [1.0, 4.0, 1.0],
        [2.0, 4.0, 1.0],
        [3.0, 4.0, 1.0],
        [0.0, 5.0, 1.0],
        [1.0, 5.0, 1.0],
        [2.0, 5.0, 1.0],
        [3.0, 5.0, 1.0],
    ]);
    let node_element_connectivity_gold = vec![
        vec![0],
        vec![0],
        vec![0, 1],
        vec![0, 1],
        vec![1, 2],
        vec![1, 2, 3],
        vec![3],
        vec![2, 4],
        vec![2, 3, 4],
        vec![3],
        vec![4, 5],
        vec![4, 5, 6],
        vec![6, 7],
        vec![7],
        vec![5],
        vec![5, 6],
        vec![6, 7],
        vec![7],
        vec![0],
        vec![0],
        vec![0, 1],
        vec![0, 1],
        vec![1, 2],
        vec![1, 2, 3],
        vec![3],
        vec![2, 4],
        vec![2, 3, 4],
        vec![3],
        vec![4, 5],
        vec![4, 5, 6],
        vec![6, 7],
        vec![7],
        vec![5],
        vec![5, 6],
        vec![6, 7],
        vec![7],
    ];
    let node_node_connectivity_gold = vec![
        vec![1, 2, 18],
        vec![0, 3, 19],
        vec![0, 3, 4, 20],
        vec![1, 2, 5, 21],
        vec![2, 5, 7, 22],
        vec![3, 4, 6, 8, 23],
        vec![5, 9, 24],
        vec![4, 8, 10, 25],
        vec![5, 7, 9, 11, 26],
        vec![6, 8, 27],
        vec![7, 11, 14, 28],
        vec![8, 10, 12, 15, 29],
        vec![11, 13, 16, 30],
        vec![12, 17, 31],
        vec![10, 15, 32],
        vec![11, 14, 16, 33],
        vec![12, 15, 17, 34],
        vec![13, 16, 35],
        vec![0, 19, 20],
        vec![1, 18, 21],
        vec![2, 18, 21, 22],
        vec![3, 19, 20, 23],
        vec![4, 20, 23, 25],
        vec![5, 21, 22, 24, 26],
        vec![6, 23, 27],
        vec![7, 22, 26, 28],
        vec![8, 23, 25, 27, 29],
        vec![9, 24, 26],
        vec![10, 25, 29, 32],
        vec![11, 26, 28, 30, 33],
        vec![12, 29, 31, 34],
        vec![13, 30, 35],
        vec![14, 28, 33],
        vec![15, 29, 32, 34],
        vec![16, 30, 33, 35],
        vec![17, 31, 34],
    ];
    let exterior_nodes_gold = (0..36).collect();
    let interface_nodes_gold = vec![];
    let interior_nodes_gold = vec![];
    let nodal_influencers_gold = node_node_connectivity_gold.clone();
    test_finite_elements(
        element_blocks,
        element_node_connectivity,
        nodal_coordinates,
        node_element_connectivity_gold,
        node_node_connectivity_gold,
        exterior_nodes_gold,
        interface_nodes_gold,
        interior_nodes_gold,
        None,
        None,
        nodal_influencers_gold,
    );
}

#[test]
fn letter_f_3d() {
    let element_blocks = vec![11; 39];
    let element_node_connectivity = vec![
        [0, 1, 6, 5, 30, 31, 36, 35],
        [1, 2, 7, 6, 31, 32, 37, 36],
        [2, 3, 8, 7, 32, 33, 38, 37],
        [3, 4, 9, 8, 33, 34, 39, 38],
        [5, 6, 11, 10, 35, 36, 41, 40],
        [6, 7, 12, 11, 36, 37, 42, 41],
        [7, 8, 13, 12, 37, 38, 43, 42],
        [8, 9, 14, 13, 38, 39, 44, 43],
        [10, 11, 16, 15, 40, 41, 46, 45],
        [11, 12, 17, 16, 41, 42, 47, 46],
        [12, 13, 18, 17, 42, 43, 48, 47],
        [13, 14, 19, 18, 43, 44, 49, 48],
        [15, 16, 21, 20, 45, 46, 51, 50],
        [16, 17, 22, 21, 46, 47, 52, 51],
        [17, 18, 23, 22, 47, 48, 53, 52],
        [18, 19, 24, 23, 48, 49, 54, 53],
        [20, 21, 26, 25, 50, 51, 56, 55],
        [21, 22, 27, 26, 51, 52, 57, 56],
        [22, 23, 28, 27, 52, 53, 58, 57],
        [23, 24, 29, 28, 53, 54, 59, 58],
        [30, 31, 36, 35, 60, 61, 63, 62],
        [35, 36, 41, 40, 62, 63, 65, 64],
        [40, 41, 46, 45, 64, 65, 70, 69],
        [41, 42, 47, 46, 65, 66, 71, 70],
        [42, 43, 48, 47, 66, 67, 72, 71],
        [43, 44, 49, 48, 67, 68, 73, 72],
        [45, 46, 51, 50, 69, 70, 75, 74],
        [50, 51, 56, 55, 74, 75, 80, 79],
        [51, 52, 57, 56, 75, 76, 81, 80],
        [52, 53, 58, 57, 76, 77, 82, 81],
        [53, 54, 59, 58, 77, 78, 83, 82],
        [60, 61, 63, 62, 84, 85, 87, 86],
        [62, 63, 65, 64, 86, 87, 89, 88],
        [64, 65, 70, 69, 88, 89, 91, 90],
        [69, 70, 75, 74, 90, 91, 93, 92],
        [74, 75, 80, 79, 92, 93, 98, 97],
        [75, 76, 81, 80, 93, 94, 99, 98],
        [76, 77, 82, 81, 94, 95, 100, 99],
        [77, 78, 83, 82, 95, 96, 101, 100],
    ];
    let nodal_coordinates = Coordinates::new(&[
        [0.0, 0.0, 0.0],
        [1.0, 0.0, 0.0],
        [2.0, 0.0, 0.0],
        [3.0, 0.0, 0.0],
        [4.0, 0.0, 0.0],
        [0.0, 1.0, 0.0],
        [1.0, 1.0, 0.0],
        [2.0, 1.0, 0.0],
        [3.0, 1.0, 0.0],
        [4.0, 1.0, 0.0],
        [0.0, 2.0, 0.0],
        [1.0, 2.0, 0.0],
        [2.0, 2.0, 0.0],
        [3.0, 2.0, 0.0],
        [4.0, 2.0, 0.0],
        [0.0, 3.0, 0.0],
        [1.0, 3.0, 0.0],
        [2.0, 3.0, 0.0],
        [3.0, 3.0, 0.0],
        [4.0, 3.0, 0.0],
        [0.0, 4.0, 0.0],
        [1.0, 4.0, 0.0],
        [2.0, 4.0, 0.0],
        [3.0, 4.0, 0.0],
        [4.0, 4.0, 0.0],
        [0.0, 5.0, 0.0],
        [1.0, 5.0, 0.0],
        [2.0, 5.0, 0.0],
        [3.0, 5.0, 0.0],
        [4.0, 5.0, 0.0],
        [0.0, 0.0, 1.0],
        [1.0, 0.0, 1.0],
        [2.0, 0.0, 1.0],
        [3.0, 0.0, 1.0],
        [4.0, 0.0, 1.0],
        [0.0, 1.0, 1.0],
        [1.0, 1.0, 1.0],
        [2.0, 1.0, 1.0],
        [3.0, 1.0, 1.0],
        [4.0, 1.0, 1.0],
        [0.0, 2.0, 1.0],
        [1.0, 2.0, 1.0],
        [2.0, 2.0, 1.0],
        [3.0, 2.0, 1.0],
        [4.0, 2.0, 1.0],
        [0.0, 3.0, 1.0],
        [1.0, 3.0, 1.0],
        [2.0, 3.0, 1.0],
        [3.0, 3.0, 1.0],
        [4.0, 3.0, 1.0],
        [0.0, 4.0, 1.0],
        [1.0, 4.0, 1.0],
        [2.0, 4.0, 1.0],
        [3.0, 4.0, 1.0],
        [4.0, 4.0, 1.0],
        [0.0, 5.0, 1.0],
        [1.0, 5.0, 1.0],
        [2.0, 5.0, 1.0],
        [3.0, 5.0, 1.0],
        [4.0, 5.0, 1.0],
        [0.0, 0.0, 2.0],
        [1.0, 0.0, 2.0],
        [0.0, 1.0, 2.0],
        [1.0, 1.0, 2.0],
        [0.0, 2.0, 2.0],
        [1.0, 2.0, 2.0],
        [2.0, 2.0, 2.0],
        [3.0, 2.0, 2.0],
        [4.0, 2.0, 2.0],
        [0.0, 3.0, 2.0],
        [1.0, 3.0, 2.0],
        [2.0, 3.0, 2.0],
        [3.0, 3.0, 2.0],
        [4.0, 3.0, 2.0],
        [0.0, 4.0, 2.0],
        [1.0, 4.0, 2.0],
        [2.0, 4.0, 2.0],
        [3.0, 4.0, 2.0],
        [4.0, 4.0, 2.0],
        [0.0, 5.0, 2.0],
        [1.0, 5.0, 2.0],
        [2.0, 5.0, 2.0],
        [3.0, 5.0, 2.0],
        [4.0, 5.0, 2.0],
        [0.0, 0.0, 3.0],
        [1.0, 0.0, 3.0],
        [0.0, 1.0, 3.0],
        [1.0, 1.0, 3.0],
        [0.0, 2.0, 3.0],
        [1.0, 2.0, 3.0],
        [0.0, 3.0, 3.0],
        [1.0, 3.0, 3.0],
        [0.0, 4.0, 3.0],
        [1.0, 4.0, 3.0],
        [2.0, 4.0, 3.0],
        [3.0, 4.0, 3.0],
        [4.0, 4.0, 3.0],
        [0.0, 5.0, 3.0],
        [1.0, 5.0, 3.0],
        [2.0, 5.0, 3.0],
        [3.0, 5.0, 3.0],
        [4.0, 5.0, 3.0],
    ]);
    let node_element_connectivity_gold = vec![
        vec![0],
        vec![0, 1],
        vec![1, 2],
        vec![2, 3],
        vec![3],
        vec![0, 4],
        vec![0, 1, 4, 5],
        vec![1, 2, 5, 6],
        vec![2, 3, 6, 7],
        vec![3, 7],
        vec![4, 8],
        vec![4, 5, 8, 9],
        vec![5, 6, 9, 10],
        vec![6, 7, 10, 11],
        vec![7, 11],
        vec![8, 12],
        vec![8, 9, 12, 13],
        vec![9, 10, 13, 14],
        vec![10, 11, 14, 15],
        vec![11, 15],
        vec![12, 16],
        vec![12, 13, 16, 17],
        vec![13, 14, 17, 18],
        vec![14, 15, 18, 19],
        vec![15, 19],
        vec![16],
        vec![16, 17],
        vec![17, 18],
        vec![18, 19],
        vec![19],
        vec![0, 20],
        vec![0, 1, 20],
        vec![1, 2],
        vec![2, 3],
        vec![3],
        vec![0, 4, 20, 21],
        vec![0, 1, 4, 5, 20, 21],
        vec![1, 2, 5, 6],
        vec![2, 3, 6, 7],
        vec![3, 7],
        vec![4, 8, 21, 22],
        vec![4, 5, 8, 9, 21, 22, 23],
        vec![5, 6, 9, 10, 23, 24],
        vec![6, 7, 10, 11, 24, 25],
        vec![7, 11, 25],
        vec![8, 12, 22, 26],
        vec![8, 9, 12, 13, 22, 23, 26],
        vec![9, 10, 13, 14, 23, 24],
        vec![10, 11, 14, 15, 24, 25],
        vec![11, 15, 25],
        vec![12, 16, 26, 27],
        vec![12, 13, 16, 17, 26, 27, 28],
        vec![13, 14, 17, 18, 28, 29],
        vec![14, 15, 18, 19, 29, 30],
        vec![15, 19, 30],
        vec![16, 27],
        vec![16, 17, 27, 28],
        vec![17, 18, 28, 29],
        vec![18, 19, 29, 30],
        vec![19, 30],
        vec![20, 31],
        vec![20, 31],
        vec![20, 21, 31, 32],
        vec![20, 21, 31, 32],
        vec![21, 22, 32, 33],
        vec![21, 22, 23, 32, 33],
        vec![23, 24],
        vec![24, 25],
        vec![25],
        vec![22, 26, 33, 34],
        vec![22, 23, 26, 33, 34],
        vec![23, 24],
        vec![24, 25],
        vec![25],
        vec![26, 27, 34, 35],
        vec![26, 27, 28, 34, 35, 36],
        vec![28, 29, 36, 37],
        vec![29, 30, 37, 38],
        vec![30, 38],
        vec![27, 35],
        vec![27, 28, 35, 36],
        vec![28, 29, 36, 37],
        vec![29, 30, 37, 38],
        vec![30, 38],
        vec![31],
        vec![31],
        vec![31, 32],
        vec![31, 32],
        vec![32, 33],
        vec![32, 33],
        vec![33, 34],
        vec![33, 34],
        vec![34, 35],
        vec![34, 35, 36],
        vec![36, 37],
        vec![37, 38],
        vec![38],
        vec![35],
        vec![35, 36],
        vec![36, 37],
        vec![37, 38],
        vec![38],
    ];
    let node_node_connectivity_gold = vec![
        vec![1, 5, 30],
        vec![0, 2, 6, 31],
        vec![1, 3, 7, 32],
        vec![2, 4, 8, 33],
        vec![3, 9, 34],
        vec![0, 6, 10, 35],
        vec![1, 5, 7, 11, 36],
        vec![2, 6, 8, 12, 37],
        vec![3, 7, 9, 13, 38],
        vec![4, 8, 14, 39],
        vec![5, 11, 15, 40],
        vec![6, 10, 12, 16, 41],
        vec![7, 11, 13, 17, 42],
        vec![8, 12, 14, 18, 43],
        vec![9, 13, 19, 44],
        vec![10, 16, 20, 45],
        vec![11, 15, 17, 21, 46],
        vec![12, 16, 18, 22, 47],
        vec![13, 17, 19, 23, 48],
        vec![14, 18, 24, 49],
        vec![15, 21, 25, 50],
        vec![16, 20, 22, 26, 51],
        vec![17, 21, 23, 27, 52],
        vec![18, 22, 24, 28, 53],
        vec![19, 23, 29, 54],
        vec![20, 26, 55],
        vec![21, 25, 27, 56],
        vec![22, 26, 28, 57],
        vec![23, 27, 29, 58],
        vec![24, 28, 59],
        vec![0, 31, 35, 60],
        vec![1, 30, 32, 36, 61],
        vec![2, 31, 33, 37],
        vec![3, 32, 34, 38],
        vec![4, 33, 39],
        vec![5, 30, 36, 40, 62],
        vec![6, 31, 35, 37, 41, 63],
        vec![7, 32, 36, 38, 42],
        vec![8, 33, 37, 39, 43],
        vec![9, 34, 38, 44],
        vec![10, 35, 41, 45, 64],
        vec![11, 36, 40, 42, 46, 65],
        vec![12, 37, 41, 43, 47, 66],
        vec![13, 38, 42, 44, 48, 67],
        vec![14, 39, 43, 49, 68],
        vec![15, 40, 46, 50, 69],
        vec![16, 41, 45, 47, 51, 70],
        vec![17, 42, 46, 48, 52, 71],
        vec![18, 43, 47, 49, 53, 72],
        vec![19, 44, 48, 54, 73],
        vec![20, 45, 51, 55, 74],
        vec![21, 46, 50, 52, 56, 75],
        vec![22, 47, 51, 53, 57, 76],
        vec![23, 48, 52, 54, 58, 77],
        vec![24, 49, 53, 59, 78],
        vec![25, 50, 56, 79],
        vec![26, 51, 55, 57, 80],
        vec![27, 52, 56, 58, 81],
        vec![28, 53, 57, 59, 82],
        vec![29, 54, 58, 83],
        vec![30, 61, 62, 84],
        vec![31, 60, 63, 85],
        vec![35, 60, 63, 64, 86],
        vec![36, 61, 62, 65, 87],
        vec![40, 62, 65, 69, 88],
        vec![41, 63, 64, 66, 70, 89],
        vec![42, 65, 67, 71],
        vec![43, 66, 68, 72],
        vec![44, 67, 73],
        vec![45, 64, 70, 74, 90],
        vec![46, 65, 69, 71, 75, 91],
        vec![47, 66, 70, 72],
        vec![48, 67, 71, 73],
        vec![49, 68, 72],
        vec![50, 69, 75, 79, 92],
        vec![51, 70, 74, 76, 80, 93],
        vec![52, 75, 77, 81, 94],
        vec![53, 76, 78, 82, 95],
        vec![54, 77, 83, 96],
        vec![55, 74, 80, 97],
        vec![56, 75, 79, 81, 98],
        vec![57, 76, 80, 82, 99],
        vec![58, 77, 81, 83, 100],
        vec![59, 78, 82, 101],
        vec![60, 85, 86],
        vec![61, 84, 87],
        vec![62, 84, 87, 88],
        vec![63, 85, 86, 89],
        vec![64, 86, 89, 90],
        vec![65, 87, 88, 91],
        vec![69, 88, 91, 92],
        vec![70, 89, 90, 93],
        vec![74, 90, 93, 97],
        vec![75, 91, 92, 94, 98],
        vec![76, 93, 95, 99],
        vec![77, 94, 96, 100],
        vec![78, 95, 101],
        vec![79, 92, 98],
        vec![80, 93, 97, 99],
        vec![81, 94, 98, 100],
        vec![82, 95, 99, 101],
        vec![83, 96, 100],
    ];
    let exterior_nodes_gold = (0..102).collect();
    let interface_nodes_gold = vec![];
    let interior_nodes_gold = vec![];
    let nodal_influencers_gold = node_node_connectivity_gold.clone();
    test_finite_elements(
        element_blocks,
        element_node_connectivity,
        nodal_coordinates,
        node_element_connectivity_gold,
        node_node_connectivity_gold,
        exterior_nodes_gold,
        interface_nodes_gold,
        interior_nodes_gold,
        None,
        None,
        nodal_influencers_gold,
    );
}

#[test]
fn sparse() {
    let element_blocks = vec![
        2, 1, 2, 1, 1, 2, 1, 1, 2, 1, 1, 2, 1, 1, 1, 1, 2, 2, 1, 1, 1, 1, 2, 1, 2, 2, 2, 2, 1, 1,
        2, 1, 1, 1, 2, 2, 1, 2, 2, 2, 2, 1, 1, 2, 2, 2, 2, 2, 2, 2, 1, 2, 2, 1, 1, 1, 2, 1,
    ];
    let element_node_connectivity = vec![
        [0, 1, 3, 2, 28, 29, 35, 34],
        [2, 3, 9, 8, 34, 35, 41, 40],
        [4, 5, 11, 10, 36, 37, 43, 42],
        [5, 6, 12, 11, 37, 38, 44, 43],
        [7, 8, 14, 13, 39, 40, 46, 45],
        [8, 9, 15, 14, 40, 41, 47, 46],
        [10, 11, 17, 16, 42, 43, 49, 48],
        [14, 15, 21, 20, 46, 47, 53, 52],
        [16, 17, 23, 22, 48, 49, 55, 54],
        [17, 18, 24, 23, 49, 50, 56, 55],
        [19, 20, 26, 25, 51, 52, 58, 57],
        [20, 21, 27, 26, 52, 53, 59, 58],
        [30, 31, 37, 36, 63, 64, 70, 69],
        [31, 32, 38, 37, 64, 65, 71, 70],
        [33, 34, 40, 39, 66, 67, 73, 72],
        [34, 35, 41, 40, 67, 68, 74, 73],
        [39, 40, 46, 45, 72, 73, 79, 78],
        [42, 43, 49, 48, 75, 76, 82, 81],
        [43, 44, 50, 49, 76, 77, 83, 82],
        [45, 46, 52, 51, 78, 79, 85, 84],
        [48, 49, 55, 54, 81, 82, 88, 87],
        [53, 54, 60, 59, 86, 87, 92, 91],
        [61, 62, 68, 67, 95, 96, 101, 100],
        [62, 63, 69, 68, 96, 97, 102, 101],
        [63, 64, 70, 69, 97, 98, 103, 102],
        [69, 70, 76, 75, 102, 103, 109, 108],
        [74, 75, 81, 80, 107, 108, 113, 112],
        [75, 76, 82, 81, 108, 109, 114, 113],
        [80, 81, 87, 86, 112, 113, 118, 117],
        [81, 82, 88, 87, 113, 114, 119, 118],
        [85, 86, 91, 90, 116, 117, 122, 121],
        [87, 88, 93, 92, 118, 119, 124, 123],
        [88, 89, 94, 93, 119, 120, 125, 124],
        [97, 98, 103, 102, 129, 130, 136, 135],
        [98, 99, 104, 103, 130, 131, 137, 136],
        [100, 101, 107, 106, 133, 134, 140, 139],
        [101, 102, 108, 107, 134, 135, 141, 140],
        [105, 106, 111, 110, 138, 139, 145, 144],
        [107, 108, 113, 112, 140, 141, 147, 146],
        [110, 111, 116, 115, 144, 145, 150, 149],
        [111, 112, 117, 116, 145, 146, 151, 150],
        [113, 114, 119, 118, 147, 148, 153, 152],
        [117, 118, 123, 122, 151, 152, 158, 157],
        [119, 120, 125, 124, 153, 154, 160, 159],
        [126, 127, 133, 132, 161, 162, 167, 166],
        [128, 129, 135, 134, 163, 164, 169, 168],
        [129, 130, 136, 135, 164, 165, 170, 169],
        [132, 133, 139, 138, 166, 167, 173, 172],
        [133, 134, 140, 139, 167, 168, 174, 173],
        [134, 135, 141, 140, 168, 169, 175, 174],
        [135, 136, 142, 141, 169, 170, 176, 175],
        [136, 137, 143, 142, 170, 171, 177, 176],
        [140, 141, 147, 146, 174, 175, 179, 178],
        [146, 147, 152, 151, 178, 179, 184, 183],
        [147, 148, 153, 152, 179, 180, 185, 184],
        [149, 150, 156, 155, 181, 182, 188, 187],
        [150, 151, 157, 156, 182, 183, 189, 188],
        [153, 154, 160, 159, 185, 186, 191, 190],
    ];
    let nodal_coordinates = Coordinates::new(&[
        [1.0, 0.0, 0.0],
        [2.0, 0.0, 0.0],
        [1.0, 1.0, 0.0],
        [2.0, 1.0, 0.0],
        [3.0, 1.0, 0.0],
        [4.0, 1.0, 0.0],
        [5.0, 1.0, 0.0],
        [0.0, 2.0, 0.0],
        [1.0, 2.0, 0.0],
        [2.0, 2.0, 0.0],
        [3.0, 2.0, 0.0],
        [4.0, 2.0, 0.0],
        [5.0, 2.0, 0.0],
        [0.0, 3.0, 0.0],
        [1.0, 3.0, 0.0],
        [2.0, 3.0, 0.0],
        [3.0, 3.0, 0.0],
        [4.0, 3.0, 0.0],
        [5.0, 3.0, 0.0],
        [0.0, 4.0, 0.0],
        [1.0, 4.0, 0.0],
        [2.0, 4.0, 0.0],
        [3.0, 4.0, 0.0],
        [4.0, 4.0, 0.0],
        [5.0, 4.0, 0.0],
        [0.0, 5.0, 0.0],
        [1.0, 5.0, 0.0],
        [2.0, 5.0, 0.0],
        [1.0, 0.0, 1.0],
        [2.0, 0.0, 1.0],
        [3.0, 0.0, 1.0],
        [4.0, 0.0, 1.0],
        [5.0, 0.0, 1.0],
        [0.0, 1.0, 1.0],
        [1.0, 1.0, 1.0],
        [2.0, 1.0, 1.0],
        [3.0, 1.0, 1.0],
        [4.0, 1.0, 1.0],
        [5.0, 1.0, 1.0],
        [0.0, 2.0, 1.0],
        [1.0, 2.0, 1.0],
        [2.0, 2.0, 1.0],
        [3.0, 2.0, 1.0],
        [4.0, 2.0, 1.0],
        [5.0, 2.0, 1.0],
        [0.0, 3.0, 1.0],
        [1.0, 3.0, 1.0],
        [2.0, 3.0, 1.0],
        [3.0, 3.0, 1.0],
        [4.0, 3.0, 1.0],
        [5.0, 3.0, 1.0],
        [0.0, 4.0, 1.0],
        [1.0, 4.0, 1.0],
        [2.0, 4.0, 1.0],
        [3.0, 4.0, 1.0],
        [4.0, 4.0, 1.0],
        [5.0, 4.0, 1.0],
        [0.0, 5.0, 1.0],
        [1.0, 5.0, 1.0],
        [2.0, 5.0, 1.0],
        [3.0, 5.0, 1.0],
        [1.0, 0.0, 2.0],
        [2.0, 0.0, 2.0],
        [3.0, 0.0, 2.0],
        [4.0, 0.0, 2.0],
        [5.0, 0.0, 2.0],
        [0.0, 1.0, 2.0],
        [1.0, 1.0, 2.0],
        [2.0, 1.0, 2.0],
        [3.0, 1.0, 2.0],
        [4.0, 1.0, 2.0],
        [5.0, 1.0, 2.0],
        [0.0, 2.0, 2.0],
        [1.0, 2.0, 2.0],
        [2.0, 2.0, 2.0],
        [3.0, 2.0, 2.0],
        [4.0, 2.0, 2.0],
        [5.0, 2.0, 2.0],
        [0.0, 3.0, 2.0],
        [1.0, 3.0, 2.0],
        [2.0, 3.0, 2.0],
        [3.0, 3.0, 2.0],
        [4.0, 3.0, 2.0],
        [5.0, 3.0, 2.0],
        [0.0, 4.0, 2.0],
        [1.0, 4.0, 2.0],
        [2.0, 4.0, 2.0],
        [3.0, 4.0, 2.0],
        [4.0, 4.0, 2.0],
        [5.0, 4.0, 2.0],
        [1.0, 5.0, 2.0],
        [2.0, 5.0, 2.0],
        [3.0, 5.0, 2.0],
        [4.0, 5.0, 2.0],
        [5.0, 5.0, 2.0],
        [1.0, 0.0, 3.0],
        [2.0, 0.0, 3.0],
        [3.0, 0.0, 3.0],
        [4.0, 0.0, 3.0],
        [5.0, 0.0, 3.0],
        [1.0, 1.0, 3.0],
        [2.0, 1.0, 3.0],
        [3.0, 1.0, 3.0],
        [4.0, 1.0, 3.0],
        [5.0, 1.0, 3.0],
        [0.0, 2.0, 3.0],
        [1.0, 2.0, 3.0],
        [2.0, 2.0, 3.0],
        [3.0, 2.0, 3.0],
        [4.0, 2.0, 3.0],
        [0.0, 3.0, 3.0],
        [1.0, 3.0, 3.0],
        [2.0, 3.0, 3.0],
        [3.0, 3.0, 3.0],
        [4.0, 3.0, 3.0],
        [0.0, 4.0, 3.0],
        [1.0, 4.0, 3.0],
        [2.0, 4.0, 3.0],
        [3.0, 4.0, 3.0],
        [4.0, 4.0, 3.0],
        [5.0, 4.0, 3.0],
        [1.0, 5.0, 3.0],
        [2.0, 5.0, 3.0],
        [3.0, 5.0, 3.0],
        [4.0, 5.0, 3.0],
        [5.0, 5.0, 3.0],
        [0.0, 0.0, 4.0],
        [1.0, 0.0, 4.0],
        [2.0, 0.0, 4.0],
        [3.0, 0.0, 4.0],
        [4.0, 0.0, 4.0],
        [5.0, 0.0, 4.0],
        [0.0, 1.0, 4.0],
        [1.0, 1.0, 4.0],
        [2.0, 1.0, 4.0],
        [3.0, 1.0, 4.0],
        [4.0, 1.0, 4.0],
        [5.0, 1.0, 4.0],
        [0.0, 2.0, 4.0],
        [1.0, 2.0, 4.0],
        [2.0, 2.0, 4.0],
        [3.0, 2.0, 4.0],
        [4.0, 2.0, 4.0],
        [5.0, 2.0, 4.0],
        [0.0, 3.0, 4.0],
        [1.0, 3.0, 4.0],
        [2.0, 3.0, 4.0],
        [3.0, 3.0, 4.0],
        [4.0, 3.0, 4.0],
        [0.0, 4.0, 4.0],
        [1.0, 4.0, 4.0],
        [2.0, 4.0, 4.0],
        [3.0, 4.0, 4.0],
        [4.0, 4.0, 4.0],
        [5.0, 4.0, 4.0],
        [0.0, 5.0, 4.0],
        [1.0, 5.0, 4.0],
        [2.0, 5.0, 4.0],
        [3.0, 5.0, 4.0],
        [4.0, 5.0, 4.0],
        [5.0, 5.0, 4.0],
        [0.0, 0.0, 5.0],
        [1.0, 0.0, 5.0],
        [2.0, 0.0, 5.0],
        [3.0, 0.0, 5.0],
        [4.0, 0.0, 5.0],
        [0.0, 1.0, 5.0],
        [1.0, 1.0, 5.0],
        [2.0, 1.0, 5.0],
        [3.0, 1.0, 5.0],
        [4.0, 1.0, 5.0],
        [5.0, 1.0, 5.0],
        [0.0, 2.0, 5.0],
        [1.0, 2.0, 5.0],
        [2.0, 2.0, 5.0],
        [3.0, 2.0, 5.0],
        [4.0, 2.0, 5.0],
        [5.0, 2.0, 5.0],
        [2.0, 3.0, 5.0],
        [3.0, 3.0, 5.0],
        [4.0, 3.0, 5.0],
        [0.0, 4.0, 5.0],
        [1.0, 4.0, 5.0],
        [2.0, 4.0, 5.0],
        [3.0, 4.0, 5.0],
        [4.0, 4.0, 5.0],
        [5.0, 4.0, 5.0],
        [0.0, 5.0, 5.0],
        [1.0, 5.0, 5.0],
        [2.0, 5.0, 5.0],
        [4.0, 5.0, 5.0],
        [5.0, 5.0, 5.0],
    ]);
    let node_element_connectivity_gold = vec![
        vec![0],
        vec![0],
        vec![0, 1],
        vec![0, 1],
        vec![2],
        vec![2, 3],
        vec![3],
        vec![4],
        vec![1, 4, 5],
        vec![1, 5],
        vec![2, 6],
        vec![2, 3, 6],
        vec![3],
        vec![4],
        vec![4, 5, 7],
        vec![5, 7],
        vec![6, 8],
        vec![6, 8, 9],
        vec![9],
        vec![10],
        vec![7, 10, 11],
        vec![7, 11],
        vec![8],
        vec![8, 9],
        vec![9],
        vec![10],
        vec![10, 11],
        vec![11],
        vec![0],
        vec![0],
        vec![12],
        vec![12, 13],
        vec![13],
        vec![14],
        vec![0, 1, 14, 15],
        vec![0, 1, 15],
        vec![2, 12],
        vec![2, 3, 12, 13],
        vec![3, 13],
        vec![4, 14, 16],
        vec![1, 4, 5, 14, 15, 16],
        vec![1, 5, 15],
        vec![2, 6, 17],
        vec![2, 3, 6, 17, 18],
        vec![3, 18],
        vec![4, 16, 19],
        vec![4, 5, 7, 16, 19],
        vec![5, 7],
        vec![6, 8, 17, 20],
        vec![6, 8, 9, 17, 18, 20],
        vec![9, 18],
        vec![10, 19],
        vec![7, 10, 11, 19],
        vec![7, 11, 21],
        vec![8, 20, 21],
        vec![8, 9, 20],
        vec![9],
        vec![10],
        vec![10, 11],
        vec![11, 21],
        vec![21],
        vec![22],
        vec![22, 23],
        vec![12, 23, 24],
        vec![12, 13, 24],
        vec![13],
        vec![14],
        vec![14, 15, 22],
        vec![15, 22, 23],
        vec![12, 23, 24, 25],
        vec![12, 13, 24, 25],
        vec![13],
        vec![14, 16],
        vec![14, 15, 16],
        vec![15, 26],
        vec![17, 25, 26, 27],
        vec![17, 18, 25, 27],
        vec![18],
        vec![16, 19],
        vec![16, 19],
        vec![26, 28],
        vec![17, 20, 26, 27, 28, 29],
        vec![17, 18, 20, 27, 29],
        vec![18],
        vec![19],
        vec![19, 30],
        vec![21, 28, 30],
        vec![20, 21, 28, 29, 31],
        vec![20, 29, 31, 32],
        vec![32],
        vec![30],
        vec![21, 30],
        vec![21, 31],
        vec![31, 32],
        vec![32],
        vec![22],
        vec![22, 23],
        vec![23, 24, 33],
        vec![24, 33, 34],
        vec![34],
        vec![22, 35],
        vec![22, 23, 35, 36],
        vec![23, 24, 25, 33, 36],
        vec![24, 25, 33, 34],
        vec![34],
        vec![37],
        vec![35, 37],
        vec![26, 35, 36, 38],
        vec![25, 26, 27, 36, 38],
        vec![25, 27],
        vec![37, 39],
        vec![37, 39, 40],
        vec![26, 28, 38, 40],
        vec![26, 27, 28, 29, 38, 41],
        vec![27, 29, 41],
        vec![39],
        vec![30, 39, 40],
        vec![28, 30, 40, 42],
        vec![28, 29, 31, 41, 42],
        vec![29, 31, 32, 41, 43],
        vec![32, 43],
        vec![30],
        vec![30, 42],
        vec![31, 42],
        vec![31, 32, 43],
        vec![32, 43],
        vec![44],
        vec![44],
        vec![45],
        vec![33, 45, 46],
        vec![33, 34, 46],
        vec![34],
        vec![44, 47],
        vec![35, 44, 47, 48],
        vec![35, 36, 45, 48, 49],
        vec![33, 36, 45, 46, 49, 50],
        vec![33, 34, 46, 50, 51],
        vec![34, 51],
        vec![37, 47],
        vec![35, 37, 47, 48],
        vec![35, 36, 38, 48, 49, 52],
        vec![36, 38, 49, 50, 52],
        vec![50, 51],
        vec![51],
        vec![37, 39],
        vec![37, 39, 40],
        vec![38, 40, 52, 53],
        vec![38, 41, 52, 53, 54],
        vec![41, 54],
        vec![39, 55],
        vec![39, 40, 55, 56],
        vec![40, 42, 53, 56],
        vec![41, 42, 53, 54],
        vec![41, 43, 54, 57],
        vec![43, 57],
        vec![55],
        vec![55, 56],
        vec![42, 56],
        vec![42],
        vec![43, 57],
        vec![43, 57],
        vec![44],
        vec![44],
        vec![45],
        vec![45, 46],
        vec![46],
        vec![44, 47],
        vec![44, 47, 48],
        vec![45, 48, 49],
        vec![45, 46, 49, 50],
        vec![46, 50, 51],
        vec![51],
        vec![47],
        vec![47, 48],
        vec![48, 49, 52],
        vec![49, 50, 52],
        vec![50, 51],
        vec![51],
        vec![52, 53],
        vec![52, 53, 54],
        vec![54],
        vec![55],
        vec![55, 56],
        vec![53, 56],
        vec![53, 54],
        vec![54, 57],
        vec![57],
        vec![55],
        vec![55, 56],
        vec![56],
        vec![57],
        vec![57],
    ];
    let node_node_connectivity_gold = vec![
        vec![1, 2, 28],
        vec![0, 3, 29],
        vec![0, 3, 8, 34],
        vec![1, 2, 9, 35],
        vec![5, 10, 36],
        vec![4, 6, 11, 37],
        vec![5, 12, 38],
        vec![8, 13, 39],
        vec![2, 7, 9, 14, 40],
        vec![3, 8, 15, 41],
        vec![4, 11, 16, 42],
        vec![5, 10, 12, 17, 43],
        vec![6, 11, 44],
        vec![7, 14, 45],
        vec![8, 13, 15, 20, 46],
        vec![9, 14, 21, 47],
        vec![10, 17, 22, 48],
        vec![11, 16, 18, 23, 49],
        vec![17, 24, 50],
        vec![20, 25, 51],
        vec![14, 19, 21, 26, 52],
        vec![15, 20, 27, 53],
        vec![16, 23, 54],
        vec![17, 22, 24, 55],
        vec![18, 23, 56],
        vec![19, 26, 57],
        vec![20, 25, 27, 58],
        vec![21, 26, 59],
        vec![0, 29, 34],
        vec![1, 28, 35],
        vec![31, 36, 63],
        vec![30, 32, 37, 64],
        vec![31, 38, 65],
        vec![34, 39, 66],
        vec![2, 28, 33, 35, 40, 67],
        vec![3, 29, 34, 41, 68],
        vec![4, 30, 37, 42, 69],
        vec![5, 31, 36, 38, 43, 70],
        vec![6, 32, 37, 44, 71],
        vec![7, 33, 40, 45, 72],
        vec![8, 34, 39, 41, 46, 73],
        vec![9, 35, 40, 47, 74],
        vec![10, 36, 43, 48, 75],
        vec![11, 37, 42, 44, 49, 76],
        vec![12, 38, 43, 50, 77],
        vec![13, 39, 46, 51, 78],
        vec![14, 40, 45, 47, 52, 79],
        vec![15, 41, 46, 53],
        vec![16, 42, 49, 54, 81],
        vec![17, 43, 48, 50, 55, 82],
        vec![18, 44, 49, 56, 83],
        vec![19, 45, 52, 57, 84],
        vec![20, 46, 51, 53, 58, 85],
        vec![21, 47, 52, 54, 59, 86],
        vec![22, 48, 53, 55, 60, 87],
        vec![23, 49, 54, 56, 88],
        vec![24, 50, 55],
        vec![25, 51, 58],
        vec![26, 52, 57, 59],
        vec![27, 53, 58, 60, 91],
        vec![54, 59, 92],
        vec![62, 67, 95],
        vec![61, 63, 68, 96],
        vec![30, 62, 64, 69, 97],
        vec![31, 63, 65, 70, 98],
        vec![32, 64, 71],
        vec![33, 67, 72],
        vec![34, 61, 66, 68, 73, 100],
        vec![35, 62, 67, 69, 74, 101],
        vec![36, 63, 68, 70, 75, 102],
        vec![37, 64, 69, 71, 76, 103],
        vec![38, 65, 70],
        vec![39, 66, 73, 78],
        vec![40, 67, 72, 74, 79],
        vec![41, 68, 73, 75, 80, 107],
        vec![42, 69, 74, 76, 81, 108],
        vec![43, 70, 75, 77, 82, 109],
        vec![44, 76, 83],
        vec![45, 72, 79, 84],
        vec![46, 73, 78, 85],
        vec![74, 81, 86, 112],
        vec![48, 75, 80, 82, 87, 113],
        vec![49, 76, 81, 83, 88, 114],
        vec![50, 77, 82],
        vec![51, 78, 85],
        vec![52, 79, 84, 86, 90, 116],
        vec![53, 80, 85, 87, 91, 117],
        vec![54, 81, 86, 88, 92, 118],
        vec![55, 82, 87, 89, 93, 119],
        vec![88, 94, 120],
        vec![85, 91, 121],
        vec![59, 86, 90, 92, 122],
        vec![60, 87, 91, 93, 123],
        vec![88, 92, 94, 124],
        vec![89, 93, 125],
        vec![61, 96, 100],
        vec![62, 95, 97, 101],
        vec![63, 96, 98, 102, 129],
        vec![64, 97, 99, 103, 130],
        vec![98, 104, 131],
        vec![67, 95, 101, 106, 133],
        vec![68, 96, 100, 102, 107, 134],
        vec![69, 97, 101, 103, 108, 135],
        vec![70, 98, 102, 104, 109, 136],
        vec![99, 103, 137],
        vec![106, 110, 138],
        vec![100, 105, 107, 111, 139],
        vec![74, 101, 106, 108, 112, 140],
        vec![75, 102, 107, 109, 113, 141],
        vec![76, 103, 108, 114],
        vec![105, 111, 115, 144],
        vec![106, 110, 112, 116, 145],
        vec![80, 107, 111, 113, 117, 146],
        vec![81, 108, 112, 114, 118, 147],
        vec![82, 109, 113, 119, 148],
        vec![110, 116, 149],
        vec![85, 111, 115, 117, 121, 150],
        vec![86, 112, 116, 118, 122, 151],
        vec![87, 113, 117, 119, 123, 152],
        vec![88, 114, 118, 120, 124, 153],
        vec![89, 119, 125, 154],
        vec![90, 116, 122],
        vec![91, 117, 121, 123, 157],
        vec![92, 118, 122, 124, 158],
        vec![93, 119, 123, 125, 159],
        vec![94, 120, 124, 160],
        vec![127, 132, 161],
        vec![126, 133, 162],
        vec![129, 134, 163],
        vec![97, 128, 130, 135, 164],
        vec![98, 129, 131, 136, 165],
        vec![99, 130, 137],
        vec![126, 133, 138, 166],
        vec![100, 127, 132, 134, 139, 167],
        vec![101, 128, 133, 135, 140, 168],
        vec![102, 129, 134, 136, 141, 169],
        vec![103, 130, 135, 137, 142, 170],
        vec![104, 131, 136, 143, 171],
        vec![105, 132, 139, 144, 172],
        vec![106, 133, 138, 140, 145, 173],
        vec![107, 134, 139, 141, 146, 174],
        vec![108, 135, 140, 142, 147, 175],
        vec![136, 141, 143, 176],
        vec![137, 142, 177],
        vec![110, 138, 145, 149],
        vec![111, 139, 144, 146, 150],
        vec![112, 140, 145, 147, 151, 178],
        vec![113, 141, 146, 148, 152, 179],
        vec![114, 147, 153, 180],
        vec![115, 144, 150, 155, 181],
        vec![116, 145, 149, 151, 156, 182],
        vec![117, 146, 150, 152, 157, 183],
        vec![118, 147, 151, 153, 158, 184],
        vec![119, 148, 152, 154, 159, 185],
        vec![120, 153, 160, 186],
        vec![149, 156, 187],
        vec![150, 155, 157, 188],
        vec![122, 151, 156, 158, 189],
        vec![123, 152, 157],
        vec![124, 153, 160, 190],
        vec![125, 154, 159, 191],
        vec![126, 162, 166],
        vec![127, 161, 167],
        vec![128, 164, 168],
        vec![129, 163, 165, 169],
        vec![130, 164, 170],
        vec![132, 161, 167, 172],
        vec![133, 162, 166, 168, 173],
        vec![134, 163, 167, 169, 174],
        vec![135, 164, 168, 170, 175],
        vec![136, 165, 169, 171, 176],
        vec![137, 170, 177],
        vec![138, 166, 173],
        vec![139, 167, 172, 174],
        vec![140, 168, 173, 175, 178],
        vec![141, 169, 174, 176, 179],
        vec![142, 170, 175, 177],
        vec![143, 171, 176],
        vec![146, 174, 179, 183],
        vec![147, 175, 178, 180, 184],
        vec![148, 179, 185],
        vec![149, 182, 187],
        vec![150, 181, 183, 188],
        vec![151, 178, 182, 184, 189],
        vec![152, 179, 183, 185],
        vec![153, 180, 184, 186, 190],
        vec![154, 185, 191],
        vec![155, 181, 188],
        vec![156, 182, 187, 189],
        vec![157, 183, 188],
        vec![159, 185, 191],
        vec![160, 186, 190],
    ];
    let exterior_nodes_gold = (0..192).collect();
    let interface_nodes_gold = vec![
        2, 3, 5, 8, 9, 10, 11, 14, 15, 16, 17, 20, 21, 23, 26, 34, 35, 36, 37, 39, 40, 41, 42, 43,
        45, 46, 47, 48, 49, 52, 53, 54, 55, 58, 59, 62, 63, 64, 67, 68, 69, 70, 72, 73, 74, 76, 78,
        79, 80, 81, 82, 85, 86, 91, 96, 97, 98, 101, 102, 103, 107, 108, 112, 113, 114, 117, 119,
        120, 122, 124, 125, 129, 130, 134, 135, 136, 140, 141, 142, 146, 147, 149, 150, 151, 153,
        154, 156, 157, 159, 160, 169, 170, 175, 176, 178, 179, 182, 183, 188,
    ];
    let interior_nodes_gold = vec![];
    let nodal_influencers_gold = node_node_connectivity_gold.clone();
    test_finite_elements(
        element_blocks,
        element_node_connectivity,
        nodal_coordinates,
        node_element_connectivity_gold,
        node_node_connectivity_gold,
        exterior_nodes_gold,
        interface_nodes_gold,
        interior_nodes_gold,
        None,
        None,
        nodal_influencers_gold,
    );
}

#[test]
fn valence_3_and_4_noised() {
    // Reference: https://autotwin.github.io/automesh/cli/metrics.html#hexahedral-unit-tests
    // We test both of the noised elements, valence_03' (noised)
    // valence_04' (noised)

    let maximum_edge_ratios_gold = [1.2922598186116965, 1.167883631481492];
    let mininum_scaled_jacobians_gold = [0.19173666980464177, 0.3743932367172326];
    let maximum_skews_gold = [0.6797482929789989, 0.4864935739781938];
    let element_volumes_gold = [1.24779970625, 0.9844007500000004];

    let element_node_connectivity = vec![[0, 1, 3, 2, 4, 5, 7, 6]];

    let nodal_coordinates_set = [
        Coordinates::new(&[
            [0.110000e0, 0.120000e0, -0.130000e0],
            [1.200000e0, -0.200000e0, 0.000000e0],
            [-0.500000e0, 1.866025e0, -0.200000e0],
            [0.500000e0, 0.866025e0, -0.400000e0],
            [0.000000e0, 0.000000e0, 1.000000e0],
            [1.000000e0, 0.000000e0, 1.000000e0],
            [-0.500000e0, 0.600000e0, 1.400000e0],
            [0.500000e0, 0.866025e0, 1.200000e0],
        ]),
        Coordinates::new(&[
            [0.100000e0, 0.200000e0, 0.300000e0],
            [1.200000e0, 0.300000e0, 0.400000e0],
            [-0.200000e0, 1.200000e0, -0.100000e0],
            [1.030000e0, 1.102000e0, -0.250000e0],
            [-0.001000e0, -0.021000e0, 1.002000e0],
            [1.200000e0, -0.100000e0, 1.100000e0],
            [0.000000e0, 1.000000e0, 1.000000e0],
            [1.010000e0, 1.020000e0, 1.030000e0],
        ]),
    ];

    let blocks: Vec<u8> = element_node_connectivity.iter().map(|_| 1).collect();

    nodal_coordinates_set
        .into_iter()
        .zip(
            maximum_edge_ratios_gold.into_iter().zip(
                maximum_skews_gold.into_iter().zip(
                    mininum_scaled_jacobians_gold
                        .into_iter()
                        .zip(element_volumes_gold),
                ),
            ),
        )
        .for_each(
            |(
                nodal_coordinates,
                (
                    maximum_edge_ratio_gold,
                    (maximum_skew_gold, (mininum_scaled_jacobian_gold, element_volume_gold)),
                ),
            )| {
                let block = HexahedralFiniteElements::from((
                    blocks.clone(),
                    element_node_connectivity.clone(),
                    nodal_coordinates,
                ));
                assert!((block.maximum_edge_ratios()[0] - maximum_edge_ratio_gold).abs() < EPSILON);
                assert!((block.maximum_skews()[0] - maximum_skew_gold).abs() < EPSILON);
                assert!(
                    (block.minimum_scaled_jacobians()[0] - mininum_scaled_jacobian_gold).abs()
                        < EPSILON
                );
                assert!((block.volumes()[0] - element_volume_gold).abs() < EPSILON);
            },
        );
}
