import { Connections, IConnectable, IVpc, SecurityGroup } from "aws-cdk-lib/aws-ec2";
import type { IAccessPoint, IFileSystem } from "aws-cdk-lib/aws-efs";
import { CfnCluster } from "aws-cdk-lib/aws-emr";
import { IGrantable, IPrincipal, InstanceProfile, Role } from "aws-cdk-lib/aws-iam";
import { Duration, RemovalPolicy, Resource } from "aws-cdk-lib/core";
import { Construct } from "constructs";
import { Home } from "../workspace/home";
import type { MountFileSystemOptions, Workspace } from "../workspace/workspace";
import { BootstrapAction } from "./bootstrap-action";
import { ICatalog } from "./catalog";
import { Configuration } from "./configuration";
import { InstanceFleet } from "./instance-fleet";
import type { InstanceGroup, PrimaryInstanceGroup } from "./instance-group";
import { Jdbc, JdbcProps } from "./jdbc";
import { ManagedScalingPolicy, ScaleDownBehavior } from "./managed-scaling";
import { ReleaseLabel } from "./release-label";
import { Step } from "./step";
export interface BaseClusterProps {
    /**
     * Name of the EMR Cluster.
     */
    readonly clusterName: string;
    /**
     * The VPC to deploy the EMR cluster into.
     */
    readonly vpc: IVpc;
    /**
     * @default None
     */
    readonly idleTimeout?: Duration;
    /**
     * @default - {@link ReleaseLabel.LATEST}
     */
    readonly releaseLabel?: ReleaseLabel;
    /**
     * The catalogs to use for the EMR cluster.
     */
    readonly catalogs: Record<string, ICatalog>;
    /**
     * @default - {@link ScaleDownBehavior.TERMINATE_AT_TASK_COMPLETION}
     */
    readonly scaleDownBehavior?: ScaleDownBehavior;
    /**
     * @default - No managed scaling policy
     */
    readonly managedScalingPolicy?: ManagedScalingPolicy;
    /**
     * Override EMR Configurations.
     *
     * @default - the {@link catalog}'s configurations + .venv for the user code.
     */
    readonly configurations?: Configuration[];
    /**
     * @default {@link RemovalPolicy.DESTROY}
     */
    readonly removalPolicy?: RemovalPolicy;
    /**
     * @default - No bootstrap actions
     */
    readonly bootstrapActions?: BootstrapAction[];
    /**
     * The EMR Steps to submit to the cluster.
     *
     * @see https://docs.aws.amazon.com/emr/latest/ReleaseGuide/emr-spark-submit-step.html
     */
    readonly steps?: Step[];
    /**
     * The concurrency level of the cluster.
     *
     * @default 1
     */
    readonly stepConcurrencyLevel?: number;
    /**
     * Extra java options to include in the Spark context by default.
     */
    readonly extraJavaOptions?: Record<string, string>;
    /**
     * Installs and configures the SSM agent to run on all Primary, Core and Task nodes.
     *
     * @default - `true` if {@link enableSSMTunnelOverSSH} is also `true`, otherwise `false`
     */
    readonly enableSSMAgent?: boolean;
    /**
     * Install the GitHub CLI on the EMR cluster.
     *
     * @default false
     */
    readonly installGitHubCLI?: boolean;
    /**
     * Mount a shared filesystem to the EMR cluster
     */
    readonly home?: Workspace;
    /**
     * Enable the Spark Rapids plugin.
     *
     * @default false
     * @see https://docs.aws.amazon.com/emr/latest/ReleaseGuide/emr-spark-rapids.html
     */
    readonly enableSparkRapids?: boolean;
    /**
     * Enable the XGBoost spark library.
     *
     * @default false
     * @see https://docs.nvidia.com/spark-rapids/user-guide/latest/getting-started/aws-emr.html
     */
    readonly enableXGBoost?: boolean;
    /**
     * Enable Docker support on the cluster.
     *
     * @default true
     */
    readonly enableDocker?: boolean;
    /**
     * Additional registries to trust for Docker containers.
     *
     * @default - trust the `local` registry and all container registries in the account/region pair
     */
    readonly additionalTrustedRegistries?: string[];
    /**
     * Additional registries to allow privileged containers from.
     *
     * @default - trust the `local` registry and all container registries in the account/region pair
     */
    readonly additionalPrivilegedRegistries?: string[];
}
export interface ClusterProps extends BaseClusterProps {
    /**
     * Describes the EC2 instances and instance configurations for the master
     * {@link InstanceGroup} when using {@link UniformCluster}s.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-emr-cluster-jobflowinstancesconfig.html#cfn-emr-cluster-jobflowinstancesconfig-masterinstancegroup
     */
    readonly primaryInstanceGroup?: PrimaryInstanceGroup;
    /**
     * Describes the EC2 instances and instance configurations for the master
     * {@link InstanceFleet} when using {@link FleetCluster}s.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-emr-cluster-jobflowinstancesconfig.html#cfn-emr-cluster-jobflowinstancesconfig-masterinstancefleet
     */
    readonly primaryInstanceFleet?: InstanceFleet;
    /**
     * Describes the EC2 instances and instance configurations for core
     * {@link InstanceGroup}s when using {@link UniformCluster}s.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-emr-cluster-jobflowinstancesconfig.html#cfn-emr-cluster-jobflowinstancesconfig-coreinstancegroup
     */
    readonly coreInstanceGroup?: InstanceGroup;
    /**
     * Describes the EC2 instances and instance configurations for the core {@link InstanceFleet} when
     * using {@link FleetCluster}s.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-emr-cluster-jobflowinstancesconfig.html#cfn-emr-cluster-jobflowinstancesconfig-coreinstancefleet
     */
    readonly coreInstanceFleet?: InstanceFleet;
    /**
     * Describes the EC2 instances and instance configurations for task {@link InstanceGroup}s
     * when using {@link UniformCluster}s.
     *
     * These task {@link InstanceGroup}s are added to the cluster as part of the cluster launch.
     * Each task {@link InstanceGroup} must have a unique name specified so that CloudFormation
     * can differentiate between the task {@link InstanceGroup}s.
     *
     * > After creating the cluster, you can only modify the mutable properties of `InstanceGroupConfig` , which are `AutoScalingPolicy` and `InstanceCount` . Modifying any other property results in cluster replacement.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-emr-cluster-jobflowinstancesconfig.html#cfn-emr-cluster-jobflowinstancesconfig-taskinstancegroups
     */
    readonly taskInstanceGroups?: InstanceGroup[];
    /**
     * Describes the EC2 instances and instance configurations for the task {@link InstanceFleet}s
     * when using {@link FleetCluster}s.
     *
     * These task {@link InstanceFleet}s are added to the cluster as part of the cluster launch.
     * Each task {@link InstanceFleet} must have a unique name specified so that CloudFormation
     * can differentiate between the task {@link InstanceFleet}s.
     *
     * > You can currently specify only one task instance fleet for a cluster. After creating the cluster, you can only modify the mutable properties of `InstanceFleetConfig` , which are `TargetOnDemandCapacity` and `TargetSpotCapacity` . Modifying any other property results in cluster replacement. > To allow a maximum of 30 Amazon EC2 instance types per fleet, include `TaskInstanceFleets` when you create your cluster. If you create your cluster without `TaskInstanceFleets` , Amazon EMR uses its default allocation strategy, which allows for a maximum of five Amazon EC2 instance types.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-emr-cluster-jobflowinstancesconfig.html#cfn-emr-cluster-jobflowinstancesconfig-taskinstancefleets
     */
    readonly taskInstanceFleets?: InstanceFleet[];
}
/**
 * An EMR Cluster.
 */
export declare class Cluster extends Resource implements IGrantable, IConnectable {
    readonly release: ReleaseLabel;
    readonly primarySg: SecurityGroup;
    readonly coreSg: SecurityGroup;
    readonly serviceAccessSg: SecurityGroup;
    readonly connections: Connections;
    readonly grantPrincipal: IPrincipal;
    readonly extraJavaOptions: Readonly<Record<string, string>>;
    readonly jobFlowRole: Role;
    readonly instanceProfile: InstanceProfile;
    readonly serviceRole: Role;
    private readonly steps;
    private readonly bootstrapActions;
    private readonly configurations;
    private readonly clusterID;
    private readonly enableSparkRapids;
    private readonly enableXGBoost;
    private readonly enableDocker;
    protected readonly taskInstanceGroups: InstanceGroup[];
    protected readonly taskInstanceFleets: InstanceFleet[];
    protected readonly resource: CfnCluster;
    constructor(scope: Construct, id: string, props: ClusterProps);
    /**
     * Configure the EMR cluster start the Thrift Server and serve JDBC requests on the specified port.
     *
     * @param options to set when running the JDBC server
     * @returns a reference to the JDBC server
     * @example
     * const sparkSQL = cluster.jdbc({
     *  port: 10000,
     * });
     * sparkSQL.allowFrom(sageMakerDomain);
     */
    jdbc(options: JdbcProps): Jdbc;
    /**
     * Add an EMR Step to the cluster.
     *
     * This step will run when the cluster is started.
     *
     * @param step the step to add
     */
    addStep(step: Step): void;
    /**
     * Add EMR Configurations to the cluster.
     *
     * E.g. spark or hive configurations.
     *
     * @param configurations additional configurations to add
     */
    addConfig(...configurations: Configuration[]): void;
    /**
     * Add a Bootstrap Action to the cluster.
     *
     * Bootstrap actions are scripts that run on the cluster before Hadoop starts.
     *
     * @param action the bootstrap action to add
     * @see https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-plan-bootstrap.html
     */
    addBootstrapAction(action: BootstrapAction): void;
    private getInstanceGroupConfig;
    private getInstanceFleetConfig;
    private getInstanceConfigurations;
    private getEbsConfigurations;
    /**
     * Installs the SSM Agent on Primary, Core, and Task nodes.
     *
     * Authorizes the EC2 instances to communicate with the SSM service.
     *
     * @see https://aws.amazon.com/blogs/big-data/securing-access-to-emr-clusters-using-aws-systems-manager/
     */
    enableSSMAgent(): void;
    private isYarnCGroupsMounted;
    /**
     * Mounts YARN cgroups if not already mounted.
     */
    mountYarnCGroups(): void;
    /**
     * Install the NVidia drivers on the EMR cluster.
     */
    installNvidiaDrivers(): void;
    /**
     * Install the NVidia drivers on the EMR cluster.
     */
    installNvidiaContainerToolkit(): void;
    /**
     * Setup Hadoop Users on the EMR cluster.
     */
    setupHadoopUsers(): void;
    private isGitHubCLIInstalled;
    /**
     * Install the GitHub CLI on the EMR cluster.
     */
    installGitHubCLI(): void;
    /**
     * Mount a {@link Home} directory onto the File System.
     *
     * @param home the home directory to mount
     */
    mount(home: Home): void;
    /**
     * Mount an EFS Access Point on the EMR cluster.
     *
     * @param accessPoint the EFS Access Point to mount
     * @param options the options to use when mounting the Access Point
     */
    mountAccessPoint(accessPoint: IAccessPoint, options: MountFileSystemOptions): void;
    /**
     * Mount an EFS File System on the EMR cluster.
     *
     * @param fileSystem the EFS File System to mount
     * @param options the options to use when mounting the File System
     */
    mountFileSystem(fileSystem: IFileSystem, options: MountFileSystemOptions): void;
    private grantMountPermissions;
    private addMountBootstrapAction;
    private getScript;
    /**
     * Grant an permission to start an SSM Session on the EMR cluster.
     *
     * @param grantee the principal to grant the permission to
     *
     * // TODO: figure out how to use SSM Session Documents to:
     * //       1. customize where state is store and encrypt it
     * //       2. customize other session properties
     * //       3. constrain access with IAM Condition: ssm:SessionDocumentAccessCheck
     * @see https://docs.aws.amazon.com/systems-manager/latest/userguide/getting-started-specify-session-document.html
     */
    grantStartSSMSession(grantee: IGrantable): void;
    /**
     * Allows connections to the Livy server on port 8998 from the specified {@link other} security group.
     */
    allowLivyFrom(other: IConnectable): void;
    /**
     * Configure the rules for the Primary, Core, and Service Access security groups.
     */
    private configureSecurityGroups;
    /**
     * Configure security group for Primary instance (master)
     *
     * All traffic to/from the Primary and Core/Task security groups.
     * All outbound traffic to any IPv4 address.
     *
     * @see https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-man-sec-groups.html#emr-sg-elasticmapreduce-master-private
     */
    private configureMasterSecurityGroup;
    /**
     * Configure security group for Core & Task nodes
     *
     * All traffic to/from the Primary and Core/Task security groups.
     * All outbound traffic to any IPv4 address.
     *
     * @see https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-man-sec-groups.html#emr-sg-elasticmapreduce-slave-private
     */
    private configureCoreSecurityGroup;
    /**
     * Configure security group for Service Access.
     *
     * It allows inbound traffic on 8443 from the primary security group.
     * It allows outbound traffic on 8443 to the primary and core security groups.
     *
     * @see https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-man-sec-groups.html#emr-sg-elasticmapreduce-sa-private
     */
    private configureServiceAccessSecurityGroup;
}
