"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cluster = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_emr_1 = require("aws-cdk-lib/aws-emr");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
const core_1 = require("aws-cdk-lib/core");
const path = require("path");
const application_1 = require("./application");
const configuration_1 = require("./configuration");
const instance_fleet_1 = require("./instance-fleet");
const instance_market_1 = require("./instance-market");
const jdbc_1 = require("./jdbc");
const managed_scaling_1 = require("./managed-scaling");
const release_label_1 = require("./release-label");
const spark_config_1 = require("./spark-config");
/**
 * An EMR Cluster.
 */
class Cluster extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.extraJavaOptions = { ...(props.extraJavaOptions ?? {}) };
        this.steps = [...(props.steps ?? [])];
        this.configurations = [...(props.configurations ?? [])];
        this.bootstrapActions = [...(props.bootstrapActions ?? [])];
        this.release = props.releaseLabel ?? release_label_1.ReleaseLabel.EMR_7_0_0;
        // const taskInstanceType = props.taskInstanceGroup?.instanceType ?? m5xlarge;
        this.taskInstanceGroups = [...(props.taskInstanceGroups ?? [])];
        this.taskInstanceFleets = [...(props.taskInstanceFleets ?? [])];
        // for least privileges, see:
        // https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-iam-role-for-ec2.html#emr-ec2-role-least-privilege
        this.jobFlowRole = new aws_iam_1.Role(this, "JobFlowRole", {
            assumedBy: new aws_iam_1.ServicePrincipal("ec2.amazonaws.com"),
        });
        // see: https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-iam-role-for-ec2.html
        this.instanceProfile = new aws_iam_1.InstanceProfile(this, "InstanceProfile", {
            role: this.jobFlowRole,
        });
        this.grantPrincipal = this.jobFlowRole;
        this.serviceRole = new aws_iam_1.Role(this, "ServiceRole", {
            assumedBy: new aws_iam_1.ServicePrincipal("elasticmapreduce.amazonaws.com"),
            managedPolicies: [
                // TODO: fine-grained policies
                // see: https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-iam-policies.html
                // see: https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-iam-role.html
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName("service-role/AmazonEMRServicePolicy_v2"),
                // TODO: remove
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName("AdministratorAccess"),
            ],
        });
        this.serviceRole.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            actions: ["ec2:CreateSecurityGroup"],
            effect: aws_iam_1.Effect.ALLOW,
            resources: [
                props.vpc.vpcArn,
                core_1.Arn.format({
                    service: "ec2",
                    resource: "security-group",
                    resourceName: "*",
                }, core_1.Stack.of(this)),
            ],
        }));
        this.primarySg = new aws_ec2_1.SecurityGroup(this, "PrimarySG", {
            vpc: props.vpc,
            description: "The security group for the primary instance (private subnets). See: https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-man-sec-groups.html#emr-sg-elasticmapreduce-master-private",
            allowAllOutbound: true,
        });
        this.connections = this.primarySg.connections;
        this.coreSg = new aws_ec2_1.SecurityGroup(this, "CoreSG", {
            vpc: props.vpc,
            description: "Security group for core and task instances (private subnets). See: https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-man-sec-groups.html#emr-sg-elasticmapreduce-slave-private",
            allowAllOutbound: true,
        });
        this.serviceAccessSg = new aws_ec2_1.SecurityGroup(this, "ServiceAccessSG", {
            vpc: props.vpc,
            allowAllOutbound: false,
        });
        this.configureSecurityGroups();
        const logsBucket = new aws_s3_1.Bucket(this, "ClusterLogs", {
            removalPolicy: props.removalPolicy ?? core_1.RemovalPolicy.DESTROY,
        });
        const awsAccount = core_1.Stack.of(this).account;
        const awsRegion = core_1.Stack.of(this).region;
        this.enableDocker = props.enableDocker ?? true;
        const enableGpuAcceleration = [
            ...(props.primaryInstanceGroup
                ? [props.primaryInstanceGroup.instanceType]
                : []),
            ...(props.coreInstanceGroup
                ? [props.coreInstanceGroup.instanceType]
                : []),
            ...(props.taskInstanceGroups?.flatMap((group) => [group.instanceType]) ??
                []),
            ...(props.primaryInstanceFleet?.instanceTypes.map((type) => type.instanceType) ?? []),
            ...(props.coreInstanceFleet?.instanceTypes.map((type) => type.instanceType) ?? []),
            ...(props.taskInstanceFleets?.flatMap((fleet) => fleet.instanceTypes.map((type) => type.instanceType)) ?? []),
        ].some((instanceType) => isGPUInstance(instanceType));
        const isUniformCluster = !!props.primaryInstanceGroup ||
            !!props.coreInstanceGroup ||
            !!props.taskInstanceGroups;
        const isFleetCluster = props.primaryInstanceFleet ||
            props.coreInstanceFleet ||
            props.taskInstanceFleets;
        if (isUniformCluster && isFleetCluster) {
            throw new Error("Cannot specify both Instance Groups and Instance Fleets for Primary, Core and Task nodes. You must use either a UniformCluster or a FleetCluster.");
        }
        if (enableGpuAcceleration) {
            if (this.release.majorVersion < 6) {
                throw new Error(
                // TODO: confirm that 5 is not supported
                `GPUs are not supported in EMR ${this.release.label}. You must >= 6.x`);
            }
        }
        if (props.enableSparkRapids) {
            if (this.release.majorVersion < 6) {
                throw new Error(`The Spark Rapids plugin is not supported in EMR ${this.release.label}. You must >= 6.x`);
            }
            if (!enableGpuAcceleration) {
                throw new Error("You cannot enable the Spark Rapids plugin because none of the Instance Types are GPU instances.");
            }
        }
        if (props.enableXGBoost && !props.enableSparkRapids) {
            throw new Error("You cannot enable the XGBoost plugin without also enabling the Spark Rapids plugin.");
        }
        this.enableSparkRapids = props.enableSparkRapids;
        this.enableXGBoost = props.enableXGBoost;
        this.mountYarnCGroups();
        if (this.enableDocker && enableGpuAcceleration) {
            this.installNvidiaContainerToolkit();
        }
        // this constructs a globally unique identifier for the cluster for use in ResourceTag IAM policies
        // should work when clusters are deployed via CDK or Service Catalog
        this.clusterID = `${core_1.Aws.ACCOUNT_ID}/${core_1.Aws.REGION}/${core_1.Aws.STACK_NAME}/${props.clusterName}`;
        const cluster = new aws_emr_1.CfnCluster(this, "Resource", {
            name: props.clusterName,
            // see: https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-policy-fullaccess-v2.html
            tags: [
                {
                    key: "for-use-with-amazon-emr-managed-policies",
                    value: "true",
                },
                {
                    key: "ClusterID",
                    value: this.clusterID,
                },
            ],
            logUri: `s3://${logsBucket.bucketName}/`,
            jobFlowRole: this.instanceProfile.instanceProfileArn,
            serviceRole: this.serviceRole.roleArn,
            releaseLabel: props.releaseLabel?.label ?? release_label_1.ReleaseLabel.LATEST.label,
            applications: [
                this.release.majorVersion >= 7
                    ? // Only available on EMR 7: https://docs.aws.amazon.com/emr/latest/ReleaseGuide/emr-AmazonCloudWatchAgent.html
                        { name: application_1.Application.AMAZON_CLOUDWATCH_AGENT }
                    : undefined,
                { name: application_1.Application.LIVY },
                { name: application_1.Application.SPARK },
            ].filter((app) => app !== undefined),
            steps: core_1.Lazy.any({
                produce: () => this.steps,
            }),
            stepConcurrencyLevel: props.stepConcurrencyLevel,
            bootstrapActions: core_1.Lazy.any({
                produce: () => this.bootstrapActions.map((action) => ({
                    name: action.name,
                    scriptBootstrapAction: {
                        path: action.script.s3ObjectUrl,
                        args: action.args,
                    },
                })),
            }),
            instances: {
                ec2SubnetId: isUniformCluster
                    ? props.vpc.privateSubnets[0].subnetId
                    : undefined,
                ec2SubnetIds: isFleetCluster
                    ? props.vpc.privateSubnets.map((s) => s.subnetId)
                    : undefined,
                emrManagedMasterSecurityGroup: this.primarySg.securityGroupId,
                emrManagedSlaveSecurityGroup: this.coreSg.securityGroupId,
                serviceAccessSecurityGroup: this.serviceAccessSg.securityGroupId,
                // Instance Scaling:
                masterInstanceGroup: props.primaryInstanceGroup
                    ? this.getInstanceGroupConfig({
                        ...props.primaryInstanceGroup,
                        instanceCount: props.primaryInstanceGroup.instanceCount ?? 1,
                        market: props.primaryInstanceGroup?.market ?? instance_market_1.InstanceMarket.ON_DEMAND,
                    })
                    : undefined,
                masterInstanceFleet: props.primaryInstanceFleet
                    ? this.getInstanceFleetConfig(props.primaryInstanceFleet)
                    : undefined,
                coreInstanceGroup: props.coreInstanceGroup
                    ? this.getInstanceGroupConfig({
                        ...props.coreInstanceGroup,
                        instanceCount: props.coreInstanceGroup.instanceCount ?? 1,
                        market: props.coreInstanceGroup?.market ?? instance_market_1.InstanceMarket.ON_DEMAND,
                    })
                    : undefined,
                coreInstanceFleet: props.coreInstanceFleet
                    ? this.getInstanceFleetConfig(props.coreInstanceFleet)
                    : undefined,
                taskInstanceGroups: core_1.Lazy.any({
                    produce: () => this.taskInstanceGroups.length > 0
                        ? this.taskInstanceGroups.map((group) => this.getInstanceGroupConfig(group))
                        : undefined,
                }),
                taskInstanceFleets: core_1.Lazy.any({
                    produce: () => this.taskInstanceFleets.length > 0
                        ? this.taskInstanceFleets?.map((fleet) => fleet ? this.getInstanceFleetConfig(fleet) : undefined)
                        : undefined,
                }),
            },
            autoTerminationPolicy: props.idleTimeout
                ? {
                    idleTimeout: props.idleTimeout.toSeconds(),
                }
                : undefined,
            configurations: core_1.Lazy.any({
                produce: () => (0, configuration_1.combineConfigurations)({
                    classification: "spark-defaults",
                    configurationProperties: {
                        // configure spark to use the virtual environment
                        "spark.driver.extraJavaOptions": (0, spark_config_1.toCLIArgs)(this.extraJavaOptions),
                    },
                }, 
                // this is in the docs for gpus, but seems like a best practice default config
                // -> use c-groups to isolate containers from each other
                {
                    classification: "yarn-site",
                    configurationProperties: {
                        "yarn.node-labels.enabled": "true",
                        "yarn.nodemanager.linux-container-executor.cgroups.mount": "true",
                        "yarn.nodemanager.linux-container-executor.cgroups.mount-path": this.release.majorVersion === 7
                            ? // EMR 7
                                "/yarn-cgroup"
                            : // EMR 6
                                "/sys/fs/cgroup",
                        "yarn.nodemanager.linux-container-executor.cgroups.hierarchy": "yarn",
                        "yarn.nodemanager.container-executor.class": "org.apache.hadoop.yarn.server.nodemanager.LinuxContainerExecutor",
                        // I think this is causing non-GPU Core instances to fail, moving to the per instances group config
                        // ...(enableGpuAcceleration
                        //   ? {
                        //       "yarn.nodemanager.resource-plugins": "yarn.io/gpu",
                        //     }
                        //   : {}),
                    },
                }, {
                    classification: "container-executor",
                    configurationProperties: {},
                    configurations: (0, configuration_1.combineConfigurations)([
                        // required for device isolation of non-docker yarn containers
                        {
                            classification: "cgroups",
                            configurationProperties: {
                                root: this.release.majorVersion === 7
                                    ? "/yarn-cgroup"
                                    : // EMR 6
                                        "/sys/fs/cgroup",
                                "yarn-hierarchy": "yarn",
                            },
                        },
                        this.enableDocker
                            ? {
                                // TODO: support more configuration
                                // see: https://hadoop.apache.org/docs/r3.1.1/hadoop-yarn/hadoop-yarn-site/DockerContainers.html
                                classification: "docker",
                                configurationProperties: {
                                    // by default, trust all container registries in the account/region pair
                                    "docker.trusted.registries": [
                                        "local",
                                        `${awsAccount}.dkr.ecr.${awsRegion}.amazonaws.com`,
                                        ...(props.additionalTrustedRegistries ?? []),
                                    ].join(","),
                                    "docker.privileged-containers.registries": [
                                        "local",
                                        `${awsAccount}.dkr.ecr.${awsRegion}.amazonaws.com`,
                                        ...(props.additionalPrivilegedRegistries ?? []),
                                    ].join(","),
                                },
                            }
                            : undefined,
                    ]),
                }, enableGpuAcceleration
                    ? [
                        {
                            classification: "yarn-site",
                            configurationProperties: {
                                "yarn.resource-types": "yarn.io/gpu",
                            },
                        },
                        {
                            classification: "capacity-scheduler",
                            configurationProperties: {
                                "yarn.scheduler.capacity.resource-calculator": "org.apache.hadoop.yarn.util.resource.DominantResourceCalculator",
                            },
                        },
                        {
                            classification: "container-executor",
                            configurationProperties: {},
                            configurations: [
                                {
                                    classification: "gpu",
                                    configurationProperties: {
                                        "module.enabled": "true",
                                    },
                                },
                            ],
                        },
                    ]
                    : undefined, 
                // @see https://docs.aws.amazon.com/emr/latest/ReleaseGuide/emr-spark-rapids.html
                this.enableSparkRapids
                    ? [
                        {
                            classification: "spark",
                            configurationProperties: {
                                enableSparkRapids: "true",
                            },
                        },
                        {
                            classification: "spark-defaults",
                            configurationProperties: {
                                "spark.plugins": "com.nvidia.spark.SQLPlugin",
                                // TODO: do these need to be on the executor nodes that have GPUs only or can it be global?
                                "spark.executor.resource.gpu.discoveryScript": "/usr/lib/spark/scripts/gpu/getGpusResources.sh",
                                "spark.executor.extraLibraryPath": "/usr/local/cuda/targets/x86_64-linux/lib:/usr/local/cuda/extras/CUPTI/lib64:/usr/local/cuda/compat/lib:/usr/local/cuda/lib:/usr/local/cuda/lib64:/usr/lib/hadoop/lib/native:/usr/lib/hadoop-lzo/lib/native:/docker/usr/lib/hadoop/lib/native:/docker/usr/lib/hadoop-lzo/lib/native",
                            },
                        },
                    ]
                    : undefined, 
                // @see https://docs.nvidia.com/spark-rapids/user-guide/latest/getting-started/aws-emr.html#submit-spark-jobs-to-an-emr-cluster-accelerated-by-gpus
                this.enableXGBoost
                    ? {
                        classification: "spark-defaults",
                        configurationProperties: {
                            "spark.submit.pyFiles": 
                            // note: spark_3.0 is used for all spark versions on EMR right now, this could be confusing if you look at it - it does not need to match the spark version, e.g. spark_3.5
                            "/usr/lib/spark/jars/xgboost4j-spark_3.0-1.4.2-0.3.0.jar",
                        },
                    }
                    : undefined, ...this.configurations),
            }),
            scaleDownBehavior: props.scaleDownBehavior ??
                managed_scaling_1.ScaleDownBehavior.TERMINATE_AT_TASK_COMPLETION,
            managedScalingPolicy: props.managedScalingPolicy,
            // TODO: configure specific Role
            // autoScalingRole: "EMR_AutoScaling_DefaultRole",
        });
        logsBucket.grantReadWrite(this.jobFlowRole);
        for (const [catalogName, catalog] of Object.entries(props.catalogs)) {
            catalog.bind(this, catalogName);
        }
        this.resource = cluster;
        cluster.applyRemovalPolicy(props.removalPolicy ?? core_1.RemovalPolicy.DESTROY);
        if (props.enableSSMAgent) {
            this.enableSSMAgent();
        }
        if (props.installGitHubCLI) {
            this.installGitHubCLI();
        }
    }
    /**
     * Configure the EMR cluster start the Thrift Server and serve JDBC requests on the specified port.
     *
     * @param options to set when running the JDBC server
     * @returns a reference to the JDBC server
     * @example
     * const sparkSQL = cluster.jdbc({
     *  port: 10000,
     * });
     * sparkSQL.allowFrom(sageMakerDomain);
     */
    jdbc(options) {
        return new jdbc_1.Jdbc(this, options);
    }
    /**
     * Add an EMR Step to the cluster.
     *
     * This step will run when the cluster is started.
     *
     * @param step the step to add
     */
    addStep(step) {
        this.steps.push(step);
    }
    /**
     * Add EMR Configurations to the cluster.
     *
     * E.g. spark or hive configurations.
     *
     * @param configurations additional configurations to add
     */
    addConfig(...configurations) {
        this.configurations.push(...configurations);
    }
    /**
     * Add a Bootstrap Action to the cluster.
     *
     * Bootstrap actions are scripts that run on the cluster before Hadoop starts.
     *
     * @param action the bootstrap action to add
     * @see https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-plan-bootstrap.html
     */
    addBootstrapAction(action) {
        action.script.grantRead(this.jobFlowRole);
        this.bootstrapActions.push(action);
    }
    getInstanceGroupConfig(instanceGroup) {
        return {
            name: instanceGroup.name,
            instanceCount: instanceGroup.instanceCount,
            instanceType: instanceGroup.instanceType.toString(),
            customAmiId: instanceGroup.customAmi?.getImage(this).imageId,
            market: instanceGroup.market,
            configurations: (0, configuration_1.combineConfigurations)(instanceGroup.configurations, this.getInstanceConfigurations(instanceGroup.instanceType)),
            ebsConfiguration: this.getEbsConfigurations(instanceGroup),
            bidPrice: instanceGroup.bidPrice,
            autoScalingPolicy: instanceGroup.autoScalingPolicy,
        };
    }
    getInstanceFleetConfig(instanceFleet) {
        const timeoutDuration = instanceFleet.timeoutDuration?.toMinutes() ?? 60;
        if (timeoutDuration < 5 || timeoutDuration > 1440) {
            throw new Error("timeoutDuration must be between 5 and 1440 minutes");
        }
        // check timeoutDuration is a whole minute
        if (timeoutDuration % 1 !== 0) {
            throw new Error("timeoutDuration must be a whole number of minutes");
        }
        const targetOnDemandCapacity = instanceFleet.targetOnDemandCapacity ?? 0;
        const targetSpotCapacity = instanceFleet.targetSpotCapacity ?? 0;
        if (targetOnDemandCapacity + targetSpotCapacity === 0) {
            throw new Error("targetOnDemandCapacity and targetSpotCapacity cannot both be 0");
        }
        return {
            name: instanceFleet.name,
            targetOnDemandCapacity: instanceFleet?.targetOnDemandCapacity,
            targetSpotCapacity: instanceFleet?.targetSpotCapacity,
            launchSpecifications: {
                onDemandSpecification: {
                    allocationStrategy: instance_fleet_1.AllocationStrategy.LOWEST_PRICE,
                },
                spotSpecification: {
                    // deprecated by AWS
                    // blockDurationMinutes: ,
                    timeoutAction: instanceFleet?.timeoutAction ?? instance_fleet_1.TimeoutAction.SWITCH_TO_ON_DEMAND,
                    timeoutDurationMinutes: timeoutDuration,
                    allocationStrategy: instanceFleet?.allocationStrategy ??
                        instance_fleet_1.AllocationStrategy.LOWEST_PRICE,
                },
            },
            instanceTypeConfigs: instanceFleet.instanceTypes.map((instance) => ({
                customAmiId: instance.customAmi?.getImage(this).imageId,
                instanceType: instance.instanceType.toString(),
                weightedCapacity: instance.weightedCapacity,
                bidPriceAsPercentageOfOnDemandPrice: instance.bidPriceAsPercentageOfOnDemandPrice,
                bidPrice: instance.bidPrice,
                ebsConfiguration: this.getEbsConfigurations(instance),
                configurations: (0, configuration_1.combineConfigurations)(...(instance.configurations ?? []), ...(this.getInstanceConfigurations(instance.instanceType) ?? [])),
            })),
        };
    }
    getInstanceConfigurations(instance) {
        if (isGPUInstance(instance)) {
            const nvidiaK80 = {
                "p2.xlarge": 1,
                "p2.8xlarge": 8,
                "p2.16xlarge": 16,
            };
            const nvidiaTeslaV100 = {
                "p3.2xlarge": 1,
                "p3.8xlarge": 4,
                "p3.16xlarge": 8,
                "p3dn.24xlarge": 8,
            };
            const nvidiaA100 = {
                "p4d.24xlarge	": 8,
                "p4de.24xlarge": 8,
            };
            const nvidiaH100 = {
                "p5.48xlarge": 8,
            };
            const nvidiaTeslaM60 = {
                "g3s.xlarge": 1,
                "g3.4xlarge": 1,
                "g3.8xlarge": 2,
                "g3.16xlarge": 4,
            };
            // const amdRadeonProV520 = {
            //   "g4ad.xlarge": 1,
            //   "g4ad.2xlarge": 1,
            //   "g4ad.4xlarge": 1,
            //   "g4ad.8xlarge": 2,
            //   "g4ad.16xlarge": 4,
            // };
            const nvidiaT4 = {
                "g4dn.xlarge": 1,
                "g4dn.2xlarge": 1,
                "g4dn.4xlarge": 1,
                "g4dn.8xlarge": 1,
                "g4dn.16xlarge": 1,
                "g4dn.12xlarge": 4,
                "g4dn.metal": 8,
            };
            const nvidiaA10G = {
                "g5.xlarge": 1,
                "g5.2xlarge": 1,
                "g5.4xlarge": 1,
                "g5.8xlarge": 1,
                "g5.16xlarge": 1,
                "g5.12xlarge": 4,
                "g5.24xlarge": 4,
                "g5.48xlarge": 8,
            };
            const nvidiaT4G = {
                "g5g.xlarge": 1,
                "g5g.2xlarge": 1,
                "g5g.4xlarge": 1,
                "g5g.8xlarge": 1,
                "g5g.16xlarge": 2,
                "g5g.metal": 2,
            };
            // const trainium = {
            //   "trn1.2xlarge": 1,
            //   "trn1.32xlarge": 16,
            //   "trn1n.32xlarge": 16,
            // };
            // TODO: support configuring non-NVIDA chips
            // const inferentia2 = {
            //   "inf2.xlarge": 1,
            //   "inf2.8xlarge": 1,
            //   "inf2.24xlarge": 6,
            //   "inf2.48xlarge": 12,
            // };
            // const inferentia1 = {
            //   "inf1.xlarge": 1,
            //   "inf1.2xlarge": 1,
            //   "inf1.6xlarge": 4,
            //   "inf1.24xlarge": 16,
            // };
            // const gaudi = {
            //   "dl1.24xlarge": 8,
            // };
            // const qualcommAI100 = {
            //   "dl2q.24xlarge": 8,
            // };
            const gpuMappings = {
                // ...amdRadeonProV520,
                ...nvidiaK80,
                ...nvidiaTeslaV100,
                ...nvidiaA100,
                ...nvidiaH100,
                ...nvidiaTeslaM60,
                ...nvidiaT4,
                ...nvidiaA10G,
                ...nvidiaT4G,
                // ...trainium,
                // ...inferentia2,
                // ...inferentia1,
                // ...gaudi,
                // ...qualcommAI100,
            };
            const it = instance.toString();
            if (!(it in gpuMappings)) {
                throw new Error(`Instance type ${it} does not support GPU`);
            }
            const numDevices = gpuMappings[it];
            const gpuDevicesList = Array.from({ length: numDevices }, (_, index) => `/dev/nvidia${index}`).join(",");
            return [
                {
                    classification: "yarn-site",
                    configurationProperties: {
                        "yarn.nodemanager.resource-plugins": "yarn.io/gpu",
                        "yarn.nodemanager.resource-plugins.gpu.allowed-gpu-devices": "auto",
                        "yarn.nodemanager.resource-plugins.gpu.path-to-discovery-executables": "/usr/bin",
                        ...(this.enableDocker
                            ? {
                                "yarn.nodemanager.resource-plugins.gpu.docker-plugin": "nvidia-docker-v2",
                            }
                            : {}),
                    },
                },
                {
                    classification: "container-executor",
                    configurationProperties: {},
                    configurations: [
                        {
                            classification: "docker",
                            configurationProperties: {
                                "docker.allowed.runtimes": "nvidia",
                                "docker.allowed.devices": `/dev/nvidiactl,/dev/nvidia-uvm,/dev/nvidia-uvm-tools,${gpuDevicesList}`,
                            },
                        },
                    ],
                },
            ];
            // TODO:
        }
        return undefined;
    }
    getEbsConfigurations(instance) {
        return {
            // TODO: is there a good reason not to use this?
            ebsOptimized: instance.ebsOptimized ?? true,
            ebsBlockDeviceConfigs: instance.ebsBlockDevices
                ? instance.ebsBlockDevices.map((device) => ({
                    volumeSpecification: {
                        sizeInGb: device.sizeInGb,
                        volumeType: device.volumeType,
                        iops: device.iops,
                        throughput: device.throughput,
                    },
                    volumesPerInstance: device.volumesPerInstance ?? 1,
                }))
                : undefined,
        };
    }
    /**
     * Installs the SSM Agent on Primary, Core, and Task nodes.
     *
     * Authorizes the EC2 instances to communicate with the SSM service.
     *
     * @see https://aws.amazon.com/blogs/big-data/securing-access-to-emr-clusters-using-aws-systems-manager/
     */
    enableSSMAgent() {
        // this allows the SSM agent to communicate with the SSM service
        this.jobFlowRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName("AmazonSSMManagedInstanceCore"));
    }
    /**
     * Mounts YARN cgroups if not already mounted.
     */
    mountYarnCGroups() {
        if (this.isYarnCGroupsMounted) {
            return;
        }
        this.isYarnCGroupsMounted = true;
        this.addBootstrapAction({
            name: "Mount Yarn cgroups",
            script: this.getScript("mount-yarn-cgroups.sh"),
            args: ["--emr-version", this.release.majorVersion.toString()],
        });
    }
    /**
     * Install the NVidia drivers on the EMR cluster.
     */
    installNvidiaDrivers() {
        this.addBootstrapAction({
            name: "Install NVIDIA Drivers",
            script: this.getScript("install-nvidia-drivers.sh"),
        });
    }
    /**
     * Install the NVidia drivers on the EMR cluster.
     */
    installNvidiaContainerToolkit() {
        this.addBootstrapAction({
            name: "Install NVIDIA Container Toolkit",
            script: this.getScript("install-nvidia-container-toolkit.sh"),
        });
    }
    /**
     * Setup Hadoop Users on the EMR cluster.
     */
    setupHadoopUsers() {
        this.addBootstrapAction({
            name: "Setup Hadoop Users",
            script: this.getScript("setup-hadoop-users.sh"),
        });
    }
    /**
     * Install the GitHub CLI on the EMR cluster.
     */
    installGitHubCLI() {
        if (this.isGitHubCLIInstalled) {
            return;
        }
        this.isGitHubCLIInstalled = false;
        this.addBootstrapAction({
            name: "Install GitHub CLI",
            script: this.getScript("install-github-cli.sh"),
        });
    }
    /**
     * Mount a {@link Home} directory onto the File System.
     *
     * @param home the home directory to mount
     */
    mount(home) {
        this.resource.node.addDependency(home.accessPoint.fileSystem.mountTargetsAvailable);
        home.grantReadWrite(this.jobFlowRole);
        home.allowFrom(this.primarySg);
        home.allowFrom(this.coreSg);
        this.addMountBootstrapAction({
            target: home.accessPoint,
            mountPoint: home.path,
            username: home.username,
            uid: home.uid,
            gid: home.gid,
        });
    }
    /**
     * Mount an EFS Access Point on the EMR cluster.
     *
     * @param accessPoint the EFS Access Point to mount
     * @param options the options to use when mounting the Access Point
     */
    mountAccessPoint(accessPoint, options) {
        this.grantMountPermissions(accessPoint.fileSystem, accessPoint);
        this.addMountBootstrapAction({
            target: accessPoint,
            mountPoint: options.mountPoint,
            username: options.username,
            uid: `${options.uid}`,
            gid: `${options.gid}`,
        });
    }
    /**
     * Mount an EFS File System on the EMR cluster.
     *
     * @param fileSystem the EFS File System to mount
     * @param options the options to use when mounting the File System
     */
    mountFileSystem(fileSystem, options) {
        this.grantMountPermissions(fileSystem);
        this.addMountBootstrapAction({
            target: fileSystem,
            mountPoint: options.mountPoint,
            username: options.username,
            uid: `${options.uid}`,
            gid: `${options.gid}`,
        });
    }
    grantMountPermissions(fileSystem, accessPoint) {
        this.resource.node.addDependency(fileSystem.mountTargetsAvailable);
        this.jobFlowRole.addToPolicy(new aws_iam_1.PolicyStatement({
            actions: ["elasticfilesystem:DescribeMountTargets"],
            resources: [
                accessPoint?.fileSystem?.fileSystemArn,
                accessPoint?.accessPointArn,
            ].filter((s) => !!s),
        }));
        this.jobFlowRole.addToPolicy(new aws_iam_1.PolicyStatement({
            actions: [
                "elasticfilesystem:ClientMount",
                "elasticfilesystem:ClientWrite",
            ],
            resources: [fileSystem.fileSystemArn],
            conditions: {
                ...(accessPoint
                    ? {
                        StringEquals: {
                            "elasticfilesystem:AccessPointArn": accessPoint.accessPointArn,
                        },
                    }
                    : {}),
                Bool: {
                    "elasticfilesystem:AccessedViaMountTarget": true,
                },
            },
        }));
    }
    addMountBootstrapAction({ target, mountPoint, username, uid, gid, }) {
        const [fileSystemId, accessPointId] = "accessPointId" in target
            ? [target.fileSystem.fileSystemId, target.accessPointId]
            : [target.fileSystemId, undefined];
        this.addBootstrapAction({
            name: `Mount ${mountPoint}`,
            script: this.getScript("mount-efs.sh"),
            args: [
                "--file-system-id",
                fileSystemId,
                "--mount-point",
                mountPoint,
                ...(accessPointId ? ["--access-point-id", accessPointId] : []),
                "--user",
                username,
                "--uid",
                `${uid}`,
                "--gid",
                `${gid}`,
            ],
        });
    }
    getScript(name) {
        const singletonId = `packyak::emr::${name}`;
        const stack = core_1.Stack.of(this);
        const bootstrapScript = stack.node.tryFindChild(singletonId) ??
            new aws_s3_assets_1.Asset(stack, singletonId, {
                path: path.join(__dirname, "..", "..", "scripts", name),
            });
        return bootstrapScript;
    }
    /**
     * Grant an permission to start an SSM Session on the EMR cluster.
     *
     * @param grantee the principal to grant the permission to
     *
     * // TODO: figure out how to use SSM Session Documents to:
     * //       1. customize where state is store and encrypt it
     * //       2. customize other session properties
     * //       3. constrain access with IAM Condition: ssm:SessionDocumentAccessCheck
     * @see https://docs.aws.amazon.com/systems-manager/latest/userguide/getting-started-specify-session-document.html
     */
    grantStartSSMSession(grantee) {
        grantee.grantPrincipal.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            actions: [
                "ssm:DescribeInstanceProperties",
                "ssm:DescribeSessions",
                "ec2:describeInstances",
                "ssm:GetConnectionStatus",
            ],
            // TODO: not sure if this can be constrained
            resources: ["*"],
        }));
        grantee.grantPrincipal.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            actions: ["ssm:StartSession"],
            resources: [
                core_1.Arn.format({
                    service: "ec2",
                    resource: "instance",
                    resourceName: "*",
                }, core_1.Stack.of(this)),
            ],
            conditions: {
                StringEquals: {
                    // restrict access to only this cluster, as identified by AccountID, Region, StackName and ClusterName
                    "ssm:resourceTag/ClusterID": this.clusterID,
                },
            },
        }));
    }
    /**
     * Allows connections to the Livy server on port 8998 from the specified {@link other} security group.
     */
    allowLivyFrom(other) {
        this.connections.allowFrom(other, aws_ec2_1.Port.tcp(8998));
    }
    /**
     * Configure the rules for the Primary, Core, and Service Access security groups.
     */
    configureSecurityGroups() {
        this.configureMasterSecurityGroup();
        this.configureCoreSecurityGroup();
        this.configureServiceAccessSecurityGroup();
    }
    /**
     * Configure security group for Primary instance (master)
     *
     * All traffic to/from the Primary and Core/Task security groups.
     * All outbound traffic to any IPv4 address.
     *
     * @see https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-man-sec-groups.html#emr-sg-elasticmapreduce-master-private
     */
    configureMasterSecurityGroup() {
        this.primarySg.connections.allowFrom(this.primarySg, aws_ec2_1.Port.allTraffic(), "Allows the primary (aka. master) node(s) to communicate with each other over ICMP or any TCP or UDP port.");
        this.primarySg.connections.allowFrom(this.coreSg, aws_ec2_1.Port.allTraffic(), "Allows the primary (aka. master) node(s) to communicate with the core and task nodes over ICMP or any TCP or UDP port.");
        this.primarySg.connections.allowFrom(this.serviceAccessSg, aws_ec2_1.Port.tcp(8443), "This rule allows the cluster manager to communicate with the primary node.");
    }
    /**
     * Configure security group for Core & Task nodes
     *
     * All traffic to/from the Primary and Core/Task security groups.
     * All outbound traffic to any IPv4 address.
     *
     * @see https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-man-sec-groups.html#emr-sg-elasticmapreduce-slave-private
     */
    configureCoreSecurityGroup() {
        this.coreSg.connections.allowFrom(this.primarySg, aws_ec2_1.Port.allTraffic(), "Allows the primary (aka. master) node(s) to communicate with the core and task nodes over ICMP or any TCP or UDP port.");
        this.coreSg.connections.allowFrom(this.coreSg, aws_ec2_1.Port.allTraffic(), "Allows core and task node(s) to communicate with each other over ICMP or any TCP or UDP port.");
        this.coreSg.connections.allowFrom(this.serviceAccessSg, aws_ec2_1.Port.tcp(8443), "This rule allows the cluster manager to communicate with core and task nodes.");
    }
    /**
     * Configure security group for Service Access.
     *
     * It allows inbound traffic on 8443 from the primary security group.
     * It allows outbound traffic on 8443 to the primary and core security groups.
     *
     * @see https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-man-sec-groups.html#emr-sg-elasticmapreduce-sa-private
     */
    configureServiceAccessSecurityGroup() {
        this.serviceAccessSg.connections.allowFrom(this.primarySg, aws_ec2_1.Port.tcp(9443));
        this.serviceAccessSg.connections.allowTo(this.primarySg, aws_ec2_1.Port.tcp(8443));
        this.serviceAccessSg.connections.allowTo(this.coreSg, aws_ec2_1.Port.tcp(8443));
    }
}
exports.Cluster = Cluster;
_a = JSII_RTTI_SYMBOL_1;
Cluster[_a] = { fqn: "@packyak/aws-cdk.Cluster", version: "0.4.20" };
function isGPUInstance(instanceType) {
    const instanceTypeString = instanceType.toString().toLowerCase();
    return (instanceTypeString.startsWith("p") || instanceTypeString.startsWith("g"));
}
//# sourceMappingURL=data:application/json;base64,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