"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const client_efs_1 = require("@aws-sdk/client-efs");
const client_sagemaker_1 = require("@aws-sdk/client-sagemaker");
const efs = new client_efs_1.EFSClient();
const sagemaker = new client_sagemaker_1.SageMakerClient();
async function handler(event) {
    console.log(JSON.stringify(event, null, 2));
    const fileSystemId = event.ResourceProperties.FileSystemId;
    const domainId = event.ResourceProperties.DomainId;
    const removalPolicy = event.ResourceProperties.RemovalPolicy;
    if (typeof domainId !== "string" && !domainId.trim()) {
        return {
            ...event,
            Status: "FAILED",
            Reason: "DomainId is required",
            PhysicalResourceId: "none",
        };
    }
    if (event.RequestType === "Delete" &&
        removalPolicy.toLowerCase() === "destroy") {
        await stopAndDeleteSpaces();
        await deleteFileSystemAndMountTargets();
    }
    return {
        ...event,
        Status: "SUCCESS",
        PhysicalResourceId: domainId,
    };
    async function sleep(ms) {
        return new Promise((resolve) => setTimeout(resolve, ms));
    }
    async function deleteFileSystemAndMountTargets() {
        let nextToken = undefined;
        const mountTargets = [];
        while (true) {
            console.log(`Listing Mount Targets in FileSystem: ${fileSystemId}`, nextToken);
            // find all the mount targets we need to delete
            const mountTargetsResponse = await retry(() => efs.send(new client_efs_1.DescribeMountTargetsCommand({
                FileSystemId: fileSystemId,
                Marker: nextToken,
            })), {
                okErrorCodes: [
                    "MountTargetNotFound",
                    "FileSystemNotFound",
                    "AccessPointNotFound",
                ],
            });
            if (!mountTargetsResponse?.MountTargets?.length) {
                console.log("No Mount Targets found, breaking.");
                break;
            }
            console.log(`Found ${mountTargetsResponse.MountTargets.length} Mount Targets`, mountTargetsResponse);
            mountTargets.push(...mountTargetsResponse.MountTargets);
            nextToken = mountTargetsResponse.NextMarker;
            if (nextToken === undefined) {
                console.log("No more Mount Targets, breaking.");
                break;
            }
        }
        // call delete o all those mount targets
        // this will return early (before the resource is deleted)
        console.log(`Deleting ${mountTargets.length} mount targets`, mountTargets);
        await Promise.all(mountTargets.map(async (mountTarget) => {
            if (mountTarget.MountTargetId) {
                await retry(() => efs.send(new client_efs_1.DeleteMountTargetCommand({
                    MountTargetId: mountTarget.MountTargetId,
                })), {
                    okErrorCodes: ["MountTargetNotFound"],
                    retryableErrorCodes: ["DependencyTimeout"],
                });
            }
        }));
        // now, wait until all the mount targets are gone
        while (true) {
            console.log(`Waiting for all mount targets to be deleted in FileSystem: ${fileSystemId}`);
            const response = await retry(() => efs.send(new client_efs_1.DescribeMountTargetsCommand({ FileSystemId: fileSystemId })));
            if (!response?.MountTargets?.length) {
                break;
            }
            else {
                console.log(`Found ${response.MountTargets.length} mount targets`, response.MountTargets);
                await sleep(2000);
            }
        }
        // now that all mount targets have been removed, we can finally delete the file system
        console.log(`Deleting file system ${fileSystemId}`);
        await retry(() => efs.send(new client_efs_1.DeleteFileSystemCommand({ FileSystemId: fileSystemId })), {
            okErrorCodes: ["FileSystemNotFound"],
        });
    }
    async function stopAndDeleteSpaces() {
        const spaces = [];
        let nextToken = undefined;
        while (true) {
            console.log(`Listing Spaces in Domain ${domainId}`);
            const response = await retry(() => sagemaker.send(new client_sagemaker_1.ListSpacesCommand({
                DomainIdEquals: domainId,
                NextToken: nextToken,
            })));
            nextToken = response?.NextToken;
            if (!response?.Spaces?.length) {
                console.log("No Spaces found, breaking.");
                break;
            }
            console.log(`Found ${response.Spaces.length} Spaces`, response.Spaces);
            spaces.push(...response.Spaces);
            if (nextToken === undefined) {
                console.log("No more Spaces, breaking.");
                break;
            }
        }
        console.log(`Deleting ${spaces.length} Spaces`, spaces);
        await Promise.all(spaces.map(async (space) => {
            if (space.SpaceName) {
                await stopAndDeleteSpace(domainId, space.SpaceName);
            }
        }));
    }
    /**
     * List all Apps in the Space and delete them
     */
    async function stopAndDeleteSpace(domainId, spaceName) {
        await deleteApps();
        await deleteSpace();
        async function deleteApps() {
            let nextToken = undefined;
            const apps = [];
            while (true) {
                console.log(`Listing Apps in Space ${spaceName}`);
                const response = await retry(() => sagemaker.send(new client_sagemaker_1.ListAppsCommand({
                    DomainIdEquals: domainId,
                    SpaceNameEquals: spaceName,
                    NextToken: nextToken,
                })), {
                    okErrorCodes: ["ResourceNotFound"],
                });
                nextToken = response?.NextToken;
                if (!response?.Apps?.length) {
                    console.log("No Apps found, breaking.");
                    break;
                }
                apps.push(...response.Apps);
                if (nextToken === undefined) {
                    console.log("No more Apps, breaking.");
                    break;
                }
            }
            console.log(`Deleting ${apps.length} Apps`, apps);
            await Promise.all(apps.map(async (app) => {
                if (app.Status !== "Deleted") {
                    console.log(`Deleting App ${app.AppName} in Space ${app.SpaceName}`);
                    const response = await retry(() => sagemaker.send(new client_sagemaker_1.DeleteAppCommand({
                        DomainId: domainId,
                        AppName: app.AppName,
                        AppType: app.AppType,
                        SpaceName: app.SpaceName,
                    })), {
                        okErrorCodes: ["ResourceNotFound"],
                    });
                    if (response === undefined) {
                        return;
                    }
                    while (true) {
                        console.log(`Waiting for App ${app.AppName} in Space ${app.SpaceName} to be deleted`);
                        const status = await retry(() => sagemaker.send(new client_sagemaker_1.DescribeAppCommand({
                            DomainId: domainId,
                            AppName: app.AppName,
                            AppType: app.AppType,
                            SpaceName: app.SpaceName,
                        })), {
                            okErrorCodes: ["ResourceNotFound"],
                        });
                        if (status === undefined ||
                            status.Status === "Deleted" ||
                            status.Status === "Failed") {
                            return;
                        }
                        else {
                            console.log(`App ${app.AppName} in Space ${app.SpaceName} is ${status.Status}`);
                            await sleep(2000);
                        }
                    }
                }
            }));
        }
        async function deleteSpace() {
            console.log(`Deleting Space ${spaceName}`);
            const space = await retry(() => sagemaker.send(new client_sagemaker_1.DescribeSpaceCommand({
                DomainId: domainId,
                SpaceName: spaceName,
            })), {
                okErrorCodes: ["ResourceNotFound"],
            });
            console.log(`Space ${spaceName} Status: ${space?.Status}`);
            if (space?.Status !== client_sagemaker_1.SpaceStatus.Deleting) {
                console.log(`Deleting Space ${spaceName}`);
                await retry(() => sagemaker.send(new client_sagemaker_1.DeleteSpaceCommand({
                    DomainId: domainId,
                    SpaceName: spaceName,
                })), {
                    okErrorCodes: ["ResourceNotFound"],
                });
            }
            while (true) {
                console.log(`Waiting for Space ${spaceName} to be deleted`);
                const status = await retry(() => sagemaker.send(new client_sagemaker_1.DescribeSpaceCommand({
                    DomainId: domainId,
                    SpaceName: spaceName,
                })), {
                    retriesLeft: 5,
                    interval: 1000,
                    okErrorCodes: ["ResourceNotFound"],
                });
                if (status === undefined) {
                    return;
                }
                else if (status.Status === client_sagemaker_1.SpaceStatus.Delete_Failed) {
                    throw new Error(`Failed to delete Space ${spaceName}`);
                }
                else {
                    console.log(`Space ${spaceName} is ${status.Status}. Waiting for 2s and then checking agian.`);
                    await sleep(2000);
                }
            }
        }
    }
    async function retry(fn, { retriesLeft = 5, interval = 1000, retryableErrorCodes = [
        "InternalServerError",
        "InternalFailure",
        "ServiceUnavailable",
        "ThrottlingException",
    ], okErrorCodes, } = {}) {
        try {
            return await fn();
        }
        catch (err) {
            if (retriesLeft === 0) {
                throw err;
            }
            if (okErrorCodes?.includes(err.name)) {
                return undefined;
            }
            if (retryableErrorCodes?.includes(err.name)) {
                await new Promise((resolve) => setTimeout(resolve, interval));
                return retry(fn, {
                    retriesLeft: retriesLeft - 1,
                    interval,
                    retryableErrorCodes,
                });
            }
            throw err;
        }
    }
}
exports.handler = handler;
//# sourceMappingURL=data:application/json;base64,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