"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Domain = exports.AppNetworkAccessType = exports.AuthMode = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_sagemaker_1 = require("aws-cdk-lib/aws-sagemaker");
const core_1 = require("aws-cdk-lib/core");
const constructs_1 = require("constructs");
const path = require("path");
const packyak_resource_1 = require("../packyak-resource");
const sage_maker_image_1 = require("./sage-maker-image");
const user_profile_1 = require("./user-profile");
var AuthMode;
(function (AuthMode) {
    AuthMode["SSO"] = "SSO";
    AuthMode["IAM"] = "IAM";
})(AuthMode || (exports.AuthMode = AuthMode = {}));
// export enum DomainEncryption {
//   KMS = "KMS",
// }
var AppNetworkAccessType;
(function (AppNetworkAccessType) {
    AppNetworkAccessType["VPC_ONLY"] = "VpcOnly";
    AppNetworkAccessType["PUBLIC_INTERNET_ONLY"] = "PublicInternetOnly";
})(AppNetworkAccessType || (exports.AppNetworkAccessType = AppNetworkAccessType = {}));
class Domain extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        const removalPolicy = props.removalPolicy ?? core_1.RemovalPolicy.RETAIN;
        this.users = new constructs_1.Construct(this, "Users");
        const defaultImage = props.defaultImage ?? sage_maker_image_1.SageMakerImage.CPU_V1;
        const domainExecutionRole = new aws_iam_1.Role(this, "ExecutionRole", {
            assumedBy: new aws_iam_1.CompositePrincipal(new aws_iam_1.ServicePrincipal("sagemaker.amazonaws.com"), new aws_iam_1.ServicePrincipal("glue.amazonaws.com")),
        });
        this.grantPrincipal = domainExecutionRole;
        domainExecutionRole.applyRemovalPolicy(removalPolicy);
        // sagemaker needs permission to call GetRole and PassRole on the Role it assumed
        // e.g. arn:aws:iam::123456789012:role/role-name/SageMaker will call GetRole on arn:aws:iam::123456789012:role/role-name
        // When you run `spark` in a Jupyter notebook, it will:
        // 1. GetRole
        // 2. CreateSession
        // 3. PassRole to the Session
        domainExecutionRole.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            actions: ["iam:GetRole", "iam:PassRole"],
            resources: [domainExecutionRole.roleArn],
        }));
        // If EMR, the driver node must be in the same private subnet as the SageMaker notebook
        //
        // Principal Tags (for SSO)
        // AWS Global Config file on the local instance of the notebook
        // Lifecycle script
        // -> Git
        // note: image/lifecycle script should include ssh
        // error: cannot run ssh: No such file or directory
        // or: GitHub CLI ...
        // %glue_version 4.0
        // Athena -> External Catalog (or sync to Glue Iceberg)
        //  https://docs.aws.amazon.com/athena/latest/ug/connect-to-data-source-hive.html
        // https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonelasticmapreduce.html
        const sageMakerSg = props.sageMakerSg ??
            new aws_ec2_1.SecurityGroup(this, "SageMakerSecurityGroup", {
                vpc: props.vpc,
            });
        this.sageMakerSg = sageMakerSg;
        this.connections = sageMakerSg.connections;
        this.resource = new aws_sagemaker_1.CfnDomain(this, "Resource", {
            authMode: props.authMode ?? AuthMode.SSO,
            domainName: props.domainName,
            vpcId: props.vpc.vpcId,
            subnetIds: (props.subnetSelection
                ? props.vpc.selectSubnets(props.subnetSelection).subnets
                : props.vpc.privateSubnets).map((subnet) => subnet.subnetId),
            defaultUserSettings: {
                executionRole: domainExecutionRole.roleArn,
                studioWebPortal: props.defaultUserSettings?.studioWebPortal ?? true
                    ? "ENABLED"
                    : "DISABLED",
                securityGroups: [sageMakerSg.securityGroupId],
            },
            appNetworkAccessType: props.appNetworkAccessType ?? AppNetworkAccessType.VPC_ONLY,
            defaultSpaceSettings: {
                executionRole: domainExecutionRole.roleArn,
                kernelGatewayAppSettings: {
                    defaultResourceSpec: {
                        instanceType: "system",
                        sageMakerImageArn: defaultImage.getArnForStack(core_1.Stack.of(this)),
                        // TODO:
                        // lifecycleConfigArn: ??
                    },
                },
                // jupyterServerAppSettings: {
                //   defaultResourceSpec: {
                //     what is the image
                //   },
                // },
            },
        });
        this.resource.applyRemovalPolicy(removalPolicy);
        this.domainId = this.resource.ref;
        this.domainArn = this.resource.attrDomainArn;
        this.domainUrl = this.resource.attrUrl;
        this.homeEfsFileSystemId = this.resource.attrHomeEfsFileSystemId;
        this.homeEfsFileSystemArn = core_1.Arn.format({
            service: "elasticfilesystem",
            resource: "file-system",
            resourceName: this.homeEfsFileSystemId,
        }, core_1.Stack.of(this));
        this.singleSignOnManagedApplicationInstanceId =
            this.resource.attrSingleSignOnManagedApplicationInstanceId;
        this.singleSignOnApplicationArn =
            this.resource.attrSingleSignOnApplicationArn;
        this.enableCleanup(removalPolicy);
        // TODO: CustomResource to spin down Spaces when destroyed
        // TODO: should this be configurable?
        this.grantStudioAccess(domainExecutionRole);
        // TODO: should this be configurable?
        this.grantGlueInteractiveSession(domainExecutionRole);
    }
    grantGlueInteractiveSession(grantee) {
        grantee.grantPrincipal.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            actions: [
                "glue:CreateSession",
                "glue:DeleteSession",
                "glue:GetSession",
                "glue:StopSession",
                "glue:CancelStatement",
                "glue:GetStatement",
                "glue:ListStatements",
                "glue:RunStatement",
            ],
            resources: [
                core_1.Arn.format({
                    service: "glue",
                    resource: "session",
                    resourceName: "*",
                }, core_1.Stack.of(this)),
            ],
        }));
    }
    /**
     * Creates a CustomResource that will clean up the domain prior to it being destroyed:
     * 1. Delete any running Apps (i.e. instances of a Space)
     * 2. Delete the Domain's spaces.
     * 2. Delete the Domain's EFS file system (first, by deleting any mounted access points, then the FS).
     */
    enableCleanup(removalPolicy) {
        if (this.cleanup) {
            return;
        }
        const cleanup = new packyak_resource_1.PackYakResource(this, "CleanupDomain", {
            resourceType: "Custom::PackYakCleanupDomain",
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, "cleanup")),
            properties: {
                FileSystemId: this.homeEfsFileSystemId,
                DomainId: this.domainId,
                RemovalPolicy: removalPolicy,
            },
        });
        cleanup.grantPrincipal.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            actions: [
                "elasticfilesystem:DeleteFileSystem",
                "elasticfilesystem:DescribeMountTargets",
            ],
            resources: [this.homeEfsFileSystemArn],
        }));
        cleanup.grantPrincipal.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            actions: ["elasticfilesystem:DeleteMountTarget"],
            resources: [
                // See: https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonelasticfilesystem.html#amazonelasticfilesystem-actions-as-permissions
                core_1.Arn.format({
                    service: "elasticfilesystem",
                    resource: "file-system",
                    // TODO: can we constrain this more?
                    resourceName: this.homeEfsFileSystemId,
                }, core_1.Stack.of(this)),
            ],
        }));
        this.grantDeleteApp(cleanup);
        this.grantDeleteSpace(cleanup);
        this.grantDescribeApp(cleanup);
        this.grantDescribeSpace(cleanup);
        this.grantListApps(cleanup);
        this.grantListSpaces(cleanup);
    }
    grantStudioAccess(grantee) {
        this.grantCreateSpace(grantee);
        this.grantCreateApp(grantee);
        this.grantDeleteSpace(grantee);
        this.grantDeleteApp(grantee);
        this.grantUpdateSpace(grantee);
        this.grantCreatePresignedDomainUrl(grantee);
        this.grantDescribeApp(grantee);
        this.grantDescribeDomain(grantee);
        this.grantDescribeSpace(grantee);
        this.grantDescribeUserProfile(grantee);
        this.grantListApps(grantee);
        this.grantListSessions(grantee);
        this.grantListTags(grantee);
        this.grantListSpaces(grantee);
        this.grantEMRClusterAccess(grantee);
    }
    /**
     * Grants access to list and describe clusters in the JupyterNotebook.
     */
    grantEMRClusterAccess(grantee) {
        grantee.grantPrincipal.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            actions: [
                "elasticmapreduce:ListClusters",
                "elasticmapreduce:ListInstances",
                "elasticmapreduce:ListInstanceFleets",
                "elasticmapreduce:ListInstanceGroups",
                "elasticmapreduce:DescribeCluster",
                // TODO: this should be cluster specific
                "elasticmapreduce:GetOnClusterAppUIPresignedURL",
            ],
            resources: ["*"],
        }));
    }
    // sagemaker:Search
    grantSageMakerSearch(grantee) {
        this.grant(grantee, {
            actions: ["sagemaker:Search"],
            resource: "user-profile",
        });
    }
    grantListApps(grantee) {
        this.grant(grantee, {
            actions: ["sagemaker:ListApps"],
            resource: "app",
        });
    }
    grantListSessions(grantee) {
        grantee.grantPrincipal.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            actions: ["glue:ListSessions"],
            // TODO: tag-based auth
            resources: ["*"],
            effect: aws_iam_1.Effect.ALLOW,
        }));
    }
    grantListTags(grantee) {
        this.grant(grantee, {
            actions: ["sagemaker:ListTags"],
            resource: "user-profile",
        });
        // grantee.grantPrincipal.addToPrincipalPolicy(
        //   new PolicyStatement({
        //     actions: ["sagemaker:ListTags"],
        //     resources: [this.domainArn],
        //     effect: Effect.ALLOW,
        //   }),
        // );
    }
    grantSearchServiceCatalogProducts(grantee) {
        grantee.grantPrincipal.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            actions: ["servicecatalog:SearchProducts"],
            // sagemaker scans the whole account
            resources: ["*"],
            effect: aws_iam_1.Effect.ALLOW,
        }));
    }
    grantListSpaces(grantee) {
        this.grant(grantee, {
            actions: ["sagemaker:ListSpaces"],
            resource: "space",
        });
    }
    grantCreateApp(grantee) {
        this.grant(grantee, {
            actions: ["sagemaker:CreateApp"],
            resource: "app",
        });
    }
    grantCreatePresignedDomainUrl(grantee) {
        this.grant(grantee, {
            actions: ["sagemaker:CreatePresignedDomainUrl"],
            resource: "user-profile",
        });
    }
    grantCreateSpace(grantee) {
        this.grant(grantee, {
            actions: ["sagemaker:CreateSpace"],
            resource: "space",
        });
    }
    grantDeleteApp(grantee) {
        this.grant(grantee, {
            actions: ["sagemaker:DeleteApp"],
            resource: "app",
        });
    }
    grantDeleteSpace(grantee) {
        this.grant(grantee, {
            actions: ["sagemaker:DeleteSpace"],
            resource: "space",
        });
    }
    grantDescribeApp(grantee) {
        this.grant(grantee, {
            actions: ["sagemaker:DescribeApp"],
            resource: "app",
        });
    }
    grantDescribeDomain(grantee) {
        grantee.grantPrincipal.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            actions: ["sagemaker:DescribeDomain"],
            resources: [this.domainArn],
            effect: aws_iam_1.Effect.ALLOW,
        }));
    }
    grantDescribeSpace(grantee) {
        this.grant(grantee, {
            actions: ["sagemaker:DescribeSpace"],
            resource: "space",
        });
    }
    grantDescribeUserProfile(grantee) {
        this.grant(grantee, {
            actions: ["sagemaker:DescribeUserProfile"],
            resource: "user-profile",
        });
    }
    grantUpdateSpace(grantee) {
        this.grant(grantee, {
            actions: ["sagemaker:UpdateSpace"],
            resource: "space",
        });
    }
    grant(grantee, props) {
        grantee.grantPrincipal.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            actions: props.actions,
            resources: [
                core_1.Arn.format({
                    service: "sagemaker",
                    resource: props.resource,
                    resourceName: `${this.domainId}/*`,
                }, core_1.Stack.of(this)),
            ],
            effect: aws_iam_1.Effect.ALLOW,
        }));
    }
    addUserProfile(username, props) {
        return new user_profile_1.UserProfile(this.users, username, {
            domain: this,
            userProfileName: username,
            executionRole: props?.executionRole,
        });
    }
}
exports.Domain = Domain;
_a = JSII_RTTI_SYMBOL_1;
Domain[_a] = { fqn: "@packyak/aws-cdk.Domain", version: "0.4.20" };
//# sourceMappingURL=data:application/json;base64,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