"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Workspace = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_efs_1 = require("aws-cdk-lib/aws-efs");
const core_1 = require("aws-cdk-lib/core");
const constructs_1 = require("constructs");
const home_1 = require("./home");
/**
 * A Workspace is a shared environment for a team of developers to work on a project together.
 *
 * A Workspace contains a shared EFS {@link FileSystem} with {@link AccessPoint}s
 * for each {@link User} granted access to the system.
 *
 * A Workspace can be mounted to EC2 machines, SageMaker Domains and AWS EMR Clusters.
 */
class Workspace extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // isolated scoping for Home directories
        this.homes = new constructs_1.Construct(this, "Homes");
        this.fileSystem = new aws_efs_1.FileSystem(this, "FileSystem", {
            ...props,
            vpc: props.vpc,
            // switch default to Elastic as it seems more hands off
            throughputMode: props.throughputMode ?? aws_efs_1.ThroughputMode.ELASTIC,
            // switch the default to encrypted, this is designed to store sensitive user data in a home directory
            // e.g. ssh keys, .env files, API keys, credentials, proprietary code
            encrypted: props.encrypted ?? true,
            removalPolicy: props.removalPolicy ?? core_1.RemovalPolicy.RETAIN,
        });
        this.connections = this.fileSystem.connections;
        // TODO: disable root permissions from ssm-user
        // https://docs.aws.amazon.com/systems-manager/latest/userguide/session-manager-getting-started-ssm-user-permissions.html
        this.ssm = this.addHome({
            username: "ssm-user",
            // TODO: what's the default UID and GID for the ssm-user when created by AWS?
            uid: "2000",
            gid: "2000",
        });
    }
    /**
     * Allow access to the EFS file system from a connectable, e.g. SecurityGroup.
     *
     * @param connectable the connectable to allow access to the shared EFS file system
     */
    allowFrom(connectable) {
        this.fileSystem.connections.allowDefaultPortFrom(connectable);
    }
    /**
     * Add a home directory to the workspace
     */
    addHome(props) {
        return new home_1.Home(this.homes, props.username, {
            fileSystem: this.fileSystem,
            username: props.username,
            uid: props.uid,
            gid: props.gid,
        });
    }
}
exports.Workspace = Workspace;
_a = JSII_RTTI_SYMBOL_1;
Workspace[_a] = { fqn: "@packyak/aws-cdk.Workspace", version: "0.4.20" };
//# sourceMappingURL=data:application/json;base64,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