import requests
from pydantic import BaseModel
from typing import List, Dict, Any, Optional, Union
from urllib.parse import urljoin
from uuid import UUID
from . import models


class DatasinkAPIClient:
    def __init__(self, base_url, credentials, timeout=15):
        self.base_url = base_url
        self.credentials = credentials
        self.session = requests.Session()
        self.session.timeout = timeout
        self.session.headers.update({
            'Authorization': f'Basic {self.credentials}',
            'Content-Type': 'application/json'
        })

    def _request(self, method, endpoint, object_ctors, json_data=None, **kwargs):
        url = urljoin(self.base_url, endpoint)
        if json_data is not None:
            is_valid_list_payload = isinstance(json_data, list) and json_data
            if type(json_data.__class__) == type(BaseModel):
                json_data = json_data.model_dump()
            elif is_valid_list_payload and type(json_data[0].__class__) == type(BaseModel):
                json_data = [item.model_dump() for item in json_data]
            kwargs['json'] = json_data
        response = self.session.request(method, url, **kwargs)
        object_ctor = object_ctors.get(response.status_code, dict)
        response.raise_for_status()
        if response.status_code == 204:
            return None
        data = response.json()
        if isinstance(data, list):
            return [object_ctor(**item) for item in data]
        return object_ctor(**data)

    def read_root(self) -> models.ReadRootResponse:
        """Read Root"""
        return self._request(
            'GET',
            '/',
            object_ctors={
                200: models.ReadRootResponse
            }
        )

    def health_check(self) -> models.HealthCheckResponse:
        """Health Check"""
        return self._request(
            'GET',
            '/health',
            object_ctors={
                200: models.HealthCheckResponse
            }
        )

    def get_models(self) -> List[models.EmbeddingModel]:
        """Get Models"""
        return self._request(
            'GET',
            '/models',
            object_ctors={
                200: models.EmbeddingModel
            }
        )

    def create_model(self, embedding_model: models.EmbeddingModel) -> Union[dict, models.UnauthorizedResponse, models.EmbeddingModel, models.HTTPValidationError]:
        """Create Model"""
        return self._request(
            'POST',
            '/models',
            json_data=embedding_model,
            object_ctors={
                200: dict,
                201: models.EmbeddingModel,
                401: models.UnauthorizedResponse,
                422: models.HTTPValidationError
            }
        )

    def create_embedding(self, embedding_request: models.EmbeddingRequest) -> Union[models.UnauthorizedResponse, models.ModelNotFoundResponse, dict, models.HTTPValidationError, models.EmbeddingResponse]:
        """Create Embedding"""
        return self._request(
            'POST',
            '/embed',
            json_data=embedding_request,
            object_ctors={
                200: dict,
                201: models.EmbeddingResponse,
                401: models.UnauthorizedResponse,
                404: models.ModelNotFoundResponse,
                422: models.HTTPValidationError
            }
        )

    def get_collections(self) -> Union[models.UnauthorizedResponse, List[models.CollectionInfo]]:
        """Get Collections"""
        return self._request(
            'GET',
            '/collections',
            object_ctors={
                200: models.CollectionInfo,
                401: models.UnauthorizedResponse
            }
        )

    def create_collection(self, collection_info: models.CollectionInfo) -> Union[models.ErrorResponse, models.UnauthorizedResponse, models.QdrantCollectionInfo, dict, models.HTTPValidationError]:
        """Create Collection"""
        return self._request(
            'POST',
            '/collections',
            json_data=collection_info,
            object_ctors={
                200: dict,
                201: models.QdrantCollectionInfo,
                400: models.ErrorResponse,
                401: models.UnauthorizedResponse,
                422: models.HTTPValidationError
            }
        )

    def update_collection(self, collection_id, partial_collection_info: models.PartialCollectionInfo) -> Union[models.ErrorResponse, models.UnauthorizedResponse, models.HTTPValidationError, models.QdrantCollectionInfo]:
        """Update Collection"""
        return self._request(
            'PATCH',
            f'/collections/{collection_id}',
            json_data=partial_collection_info,
            object_ctors={
                200: models.QdrantCollectionInfo,
                400: models.ErrorResponse,
                401: models.UnauthorizedResponse,
                404: models.ErrorResponse,
                422: models.HTTPValidationError
            }
        )

    def delete_collection(self, collection_id) -> Union[dict, models.ErrorResponse, models.UnauthorizedResponse, models.HTTPValidationError]:
        """Delete Collection"""
        return self._request(
            'DELETE',
            f'/collections/{collection_id}',
            object_ctors={
                200: dict,
                204: dict,
                401: models.UnauthorizedResponse,
                404: models.ErrorResponse,
                422: models.HTTPValidationError
            }
        )

    def get_collection(self, collection_id) -> Union[models.UnauthorizedResponse, models.HTTPValidationError, models.CollectionNotFoundResponse, models.CollectionInfo]:
        """Get Collection"""
        return self._request(
            'GET',
            f'/collections/{collection_id}',
            object_ctors={
                200: models.CollectionInfo,
                401: models.UnauthorizedResponse,
                404: models.CollectionNotFoundResponse,
                422: models.HTTPValidationError
            }
        )

    def query(self, collection_id, query_request: models.QueryRequest) -> Union[models.UnauthorizedResponse, dict, models.HTTPValidationError, models.UnsupportedCollectionTypeResponse, models.CollectionNotFoundResponse]:
        """Query"""
        return self._request(
            'POST',
            f'/collections/{collection_id}/query',
            json_data=query_request,
            object_ctors={
                200: dict,
                400: models.UnsupportedCollectionTypeResponse,
                401: models.UnauthorizedResponse,
                404: models.CollectionNotFoundResponse,
                422: models.HTTPValidationError
            }
        )

    def get_collection_entities_list(self, collection_id, limit=100, offset=0) -> Union[models.UnauthorizedResponse, models.CollectionEntityListResponse, models.HTTPValidationError, models.CollectionNotFoundResponse]:
        """Get Collection Entities List"""
        return self._request(
            'GET',
            f'/collections/{collection_id}/entities',
            params=dict(limit=limit, offset=offset),
            object_ctors={
                200: models.CollectionEntityListResponse,
                401: models.UnauthorizedResponse,
                404: models.CollectionNotFoundResponse,
                422: models.HTTPValidationError
            }
        )

    def create_collection_entity(self, collection_id, collection_entity: models.CollectionEntity) -> Union[models.UnauthorizedResponse, dict, models.CollectionEntityAlreadyExistsResponse, models.HTTPValidationError, models.CollectionEntityResponse, models.CollectionNotFoundResponse]:
        """Create Collection Entity"""
        return self._request(
            'POST',
            f'/collections/{collection_id}/entities',
            json_data=collection_entity,
            object_ctors={
                200: dict,
                201: models.CollectionEntityResponse,
                401: models.UnauthorizedResponse,
                404: models.CollectionNotFoundResponse,
                409: models.CollectionEntityAlreadyExistsResponse,
                422: models.HTTPValidationError
            }
        )

    def create_collection_entities(self, collection_id, collection_entity: list[models.CollectionEntity]) -> Union[models.UnauthorizedResponse, dict, models.HTTPValidationError, List[models.CollectionEntityResponse], models.CollectionNotFoundResponse]:
        """Create Collection Entities"""
        return self._request(
            'POST',
            f'/collections/{collection_id}/entities/batch',
            json_data=collection_entity,
            object_ctors={
                200: dict,
                201: models.CollectionEntityResponse,
                401: models.UnauthorizedResponse,
                404: models.CollectionNotFoundResponse,
                422: models.HTTPValidationError
            }
        )

    def get_collection_entity(self, collection_id, entity_id) -> Union[models.UnauthorizedResponse, models.ErrorResponse, models.HTTPValidationError, models.CollectionEntityResponse]:
        """Get Collection Entity"""
        return self._request(
            'GET',
            f'/collections/{collection_id}/entities/{entity_id}',
            object_ctors={
                200: models.CollectionEntityResponse,
                401: models.UnauthorizedResponse,
                404: models.ErrorResponse,
                422: models.HTTPValidationError
            }
        )

    def update_collection_entity(self, collection_id, entity_id, partial_collection_entity: models.PartialCollectionEntity) -> Union[models.UnauthorizedResponse, models.ErrorResponse, models.HTTPValidationError, models.CollectionEntityResponse]:
        """Update Collection Entity"""
        return self._request(
            'PUT',
            f'/collections/{collection_id}/entities/{entity_id}',
            json_data=partial_collection_entity,
            object_ctors={
                200: models.CollectionEntityResponse,
                401: models.UnauthorizedResponse,
                404: models.ErrorResponse,
                422: models.HTTPValidationError
            }
        )

    def delete_collection_entity(self, collection_id, entity_id) -> Union[dict, models.HTTPValidationError, models.CollectionNotFoundResponse]:
        """Delete Collection Entity"""
        return self._request(
            'DELETE',
            f'/collections/{collection_id}/entities/{entity_id}',
            object_ctors={
                200: dict,
                204: dict,
                404: models.CollectionNotFoundResponse,
                422: models.HTTPValidationError
            }
        )

    def delete_collection_entities(self, collection_id, u_u_i_d: list[Union[UUID, int, str]]) -> Union[dict, models.UnauthorizedResponse, models.HTTPValidationError, models.CollectionNotFoundResponse]:
        """Delete Collection Entities"""
        return self._request(
            'POST',
            f'/collections/{collection_id}/entities/delete',
            json_data=u_u_i_d,
            object_ctors={
                200: dict,
                204: dict,
                401: models.UnauthorizedResponse,
                404: models.CollectionNotFoundResponse,
                422: models.HTTPValidationError
            }
        )

    def add_data(self, collection_id, data_point: Union['DataPoint', List['DataPoint']]) -> Union[dict, models.UnsupportedCollectionTypeResponse, models.CollectionNotFoundResponse, models.HTTPValidationError]:
        """Add Data"""
        return self._request(
            'POST',
            f'/collections/{collection_id}/data',
            json_data=data_point,
            object_ctors={
                200: dict,
                201: dict,
                404: models.CollectionNotFoundResponse,
                400: models.UnsupportedCollectionTypeResponse,
                422: models.HTTPValidationError
            }
        )
