"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const iam = require("aws-cdk-lib/aws-iam");
const sns = require("aws-cdk-lib/aws-sns");
const sqs = require("aws-cdk-lib/aws-sqs");
const kms = require("aws-cdk-lib/aws-kms");
const core_1 = require("@aws-solutions-constructs/core");
// Setup
const app = new aws_cdk_lib_1.App();
const stack = new aws_cdk_lib_1.Stack(app, core_1.generateIntegStackName(__filename));
stack.templateOptions.description = 'Integration Test for aws-sns-sqs';
const sqsEncryptionKey = new kms.Key(stack, 'ImportedSQSEncryptionKey', {
    enableKeyRotation: true
});
const dlq = new sqs.Queue(stack, 'existing-dlq-obj', {
    encryptionMasterKey: sqsEncryptionKey
});
// Definitions
const props = {
    sqsSubscriptionProps: {
        rawMessageDelivery: true,
        filterPolicy: {
            color: sns.SubscriptionFilter.stringFilter({
                allowlist: ['red', 'orange'],
                matchPrefixes: ['bl'],
            }),
            size: sns.SubscriptionFilter.stringFilter({
                denylist: ['small', 'medium'],
            }),
            price: sns.SubscriptionFilter.numericFilter({
                between: { start: 100, stop: 200 },
                greaterThan: 300,
            }),
            store: sns.SubscriptionFilter.existsFilter(),
        },
        deadLetterQueue: dlq
    }
};
const snsToSqsStack = new lib_1.SnsToSqs(stack, 'test-sns-sqs-stack', props);
// Grant yourself permissions to use the Customer Managed KMS Key
const policyStatement = new iam.PolicyStatement({
    actions: ["kms:Encrypt", "kms:Decrypt"],
    effect: iam.Effect.ALLOW,
    principals: [new iam.AccountRootPrincipal()],
    resources: ["*"]
});
snsToSqsStack.encryptionKey?.addToResourcePolicy(policyStatement);
// Synth
app.synth();
//# sourceMappingURL=data:application/json;base64,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