#!/usr/bin/env python3
"""
build_heuristic_from_tsv.py — **v10**
====================================
Simple heuristic that:
1. **Keeps every sequence**, including SBRef.
2. **Uses the raw SeriesDescription** (cleaned) as the filename stem – no
   added `rep-*`, task, or echo logic.
3. Skips only modalities listed in `SKIP_BY_DEFAULT` (`report`,
   `physio`, `refscan`).
"""

from __future__ import annotations
from pathlib import Path
from textwrap import dedent
import pandas as pd
import re

# -----------------------------------------------------------------------------
# Configuration
# -----------------------------------------------------------------------------
SKIP_BY_DEFAULT = {"report", "physio", "refscan"}

# -----------------------------------------------------------------------------
# Helper functions
# -----------------------------------------------------------------------------

def clean(text: str) -> str:
    """Return alphanumerics only (for variable names)."""
    return re.sub(r"[^0-9A-Za-z]+", "", str(text))


def safe_stem(seq: str) -> str:
    """Clean SeriesDescription for use in a filename."""
    return re.sub(r"[^0-9A-Za-z_-]+", "_", seq.strip()).strip("_")


def dedup_parts(*parts: str) -> str:
    """Return underscore-joined *parts* with consecutive repeats removed."""
    tokens: list[str] = []
    for part in parts:
        for t in str(part).split("_"):
            if t and (not tokens or t != tokens[-1]):
                tokens.append(t)
    return "_".join(tokens)


# -----------------------------------------------------------------------------
# Core writer
# -----------------------------------------------------------------------------

def write_heuristic(df: pd.DataFrame, dst: Path) -> None:
    """Write a HeuDiConv heuristic from ``df`` to ``dst``.

    Parameters
    ----------
    df : pandas.DataFrame
        Table generated by :mod:`dicom_inventory` describing the DICOM series.
    dst : Path
        Destination ``heuristic_<name>.py`` file.
    """

    print("Building heuristic (v10)…")
    buf: list[str] = []

    # 1 ─ header -----------------------------------------------------------
    buf.append(
        dedent(
            '''\
            """AUTO-GENERATED HeuDiConv heuristic (v10)."""
            from typing import Tuple

            def create_key(template: str,
                           outtype: Tuple[str, ...] = ("nii.gz",),
                           annotation_classes=None):
                if not template:
                    raise ValueError("Template must be non-empty")
                return template, outtype, annotation_classes
            '''
        )
    )

    # 2 ─ SID_MAP ----------------------------------------------------------
    sid_pairs = {(clean(str(r.source_folder)) or clean(Path(r.source_folder or '.').name), r.BIDS_name) for r in df.itertuples()}
    buf.append("\nSID_MAP = {\n")
    for folder, bids in sorted(sid_pairs):
        buf.append(f"    '{folder}': '{bids}',\n")
    buf.append("}\n\n")

    # 3 ─ template keys ----------------------------------------------------
    # Include series UID (or rep) in the key to handle repeated sequences
    seq2key: dict[tuple[str, str, str, str, str], str] = {}
    key_defs: list[tuple[str, str]] = []

    rep_counts = (
        df.groupby(["BIDS_name", "session", "sequence"], dropna=False)["sequence"].transform("count")
    )
    rep_index = (
        df.groupby(["BIDS_name", "session", "sequence"], dropna=False).cumcount() + 1
    )

    key_def_set = set()
    for idx, row in df.iterrows():
        ses_raw = row.get("session", "")
        ses = "" if pd.isna(ses_raw) else str(ses_raw).strip()
        folder = Path(str(row.get("source_folder", "."))).name
        rep_num = rep_index.loc[idx]
        uid_field = str(row.get("series_uid", ""))
        bids = row["BIDS_name"]
        container = row.get("modality_bids", "misc") or "misc"
        stem = safe_stem(row["sequence"])

        base_parts = [bids, ses, stem]
        if rep_counts.loc[idx] > 1:
            base_parts.append(f"rep-{rep_num}")
        base = dedup_parts(*base_parts)
        path = "/".join(p for p in [bids, ses, container] if p)
        template = f"{path}/{base}"

        key_parts = [bids, ses, stem]
        if rep_counts.loc[idx] > 1:
            key_parts.append(f"rep-{rep_num}")
        key_var = "key_" + clean("_".join(p for p in key_parts if p))
        if key_var not in key_def_set:
            key_defs.append((key_var, template))
            key_def_set.add(key_var)

        uid_list = [u for u in uid_field.split("|") if u] or [""]
        for uid in uid_list:
            key_id = (row["sequence"], row["BIDS_name"], ses, folder, uid)
            if key_id in seq2key:
                continue
            seq2key[key_id] = key_var

    for var, tpl in key_defs:
        buf.append(f"{var} = create_key('{tpl}')\n")
    buf.append("\n")

    # 4 ─ infotodict() ----------------------------------------------------
    buf.append("def infotodict(seqinfo):\n    \"\"\"Return mapping SeriesDescription → key list.\"\"\"\n")
    for var in seq2key.values():
        buf.append(f"    {var}_list = []\n")
    buf.append("    info = {\n")
    for var in seq2key.values():
        buf.append(f"        {var}: {var}_list,\n")
    buf.append("    }\n\n")

    buf.append("    for s in seqinfo:\n")
    for (seq, _b, _s, folder, uid), var in seq2key.items():
        seq_esc = seq.replace("'", "\\'")
        fol_esc = folder.replace("'", "\\'")
        uid_esc = str(uid).replace("'", "\\'")
        buf.append(
            f"        if s.series_description == '{seq_esc}' and s.dcm_dir_name == '{fol_esc}' and getattr(s, 'series_uid', '') == '{uid_esc}':\n"
        )
        buf.append(f"            {var}_list.append(s.series_id)\n")
    buf.append("    return info\n")

    dst.write_text("".join(buf), encoding="utf-8")
    print("Heuristic written →", dst.resolve())


# -----------------------------------------------------------------------------
# Driver
# -----------------------------------------------------------------------------

def generate(tsv: Path, out_dir: Path) -> None:
    """Generate heuristic files for each study described in ``tsv``.

    Parameters
    ----------
    tsv : Path
        Path to ``subject_summary.tsv`` produced by :mod:`dicom_inventory`.
    out_dir : Path
        Directory where the heuristic files will be written.
    """

    df = pd.read_csv(tsv, sep="\t", keep_default_na=False)

    # Drop rows with unwanted modalities
    mask = df.modality.isin(SKIP_BY_DEFAULT)
    if mask.any():
        df.loc[mask, "include"] = 0
        print(f"Auto‑skipped {mask.sum()} rows ({', '.join(SKIP_BY_DEFAULT)})")

    df = df[df.include == 1]

    out_dir.mkdir(parents=True, exist_ok=True)

    for study, sub_df in df.groupby("StudyDescription"):
        fname = safe_stem(study or "unknown")
        heur = out_dir / f"heuristic_{fname}.py"
        write_heuristic(sub_df, heur)
        folders = " ".join(sorted({clean(f) or clean(Path(f or '.').name) for f in sub_df.source_folder.unique()}))
        print(dedent(f"""
        heudiconv -d "<RAW_ROOT>/{{subject}}/**/*.*" -s {folders} -f {heur.name} -c dcm2niix -o <BIDS_OUT>/{fname} -b --minmeta --overwrite"""))


def main() -> None:
    """Entry point for the ``build-heuristic`` command line utility."""

    import argparse

    parser = argparse.ArgumentParser(description="Generate HeuDiConv heuristic(s) from TSV")
    parser.add_argument("tsv", help="Path to subject_summary.tsv file")
    parser.add_argument("out_dir", help="Directory to write heuristic files")
    args = parser.parse_args()

    generate(Path(args.tsv), Path(args.out_dir))


if __name__ == "__main__":
    main()

