"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AmplifyAssetDeploymentHandler = void 0;
const common_1 = require("./common");
class AmplifyAssetDeploymentHandler extends common_1.ResourceHandler {
    constructor(amplify, s3, event) {
        super(event);
        this.props = parseProps(this.event.ResourceProperties);
        this.amplify = amplify;
        this.s3 = s3;
    }
    // ------
    // CREATE
    // ------
    async onCreate() {
        // eslint-disable-next-line no-console
        console.log('deploying to Amplify with options:', JSON.stringify(this.props, undefined, 2));
        // Verify no jobs are currently running.
        const jobs = await this.amplify
            .listJobs({
            appId: this.props.AppId,
            branchName: this.props.BranchName,
            maxResults: 1,
        })
            .promise();
        if (jobs.jobSummaries &&
            jobs.jobSummaries.find(summary => summary.status === 'PENDING')) {
            return Promise.reject('Amplify job already running. Aborting deployment.');
        }
        // Create a pre-signed get URL of the asset so Amplify can retrieve it.
        const assetUrl = this.s3.getSignedUrl('getObject', {
            Bucket: this.props.S3BucketName,
            Key: this.props.S3ObjectKey,
        });
        // Deploy the asset to Amplify.
        const deployment = await this.amplify
            .startDeployment({
            appId: this.props.AppId,
            branchName: this.props.BranchName,
            sourceUrl: assetUrl,
        })
            .promise();
        return {
            AmplifyJobId: deployment.jobSummary.jobId,
        };
    }
    async isCreateComplete() {
        return this.isActive(this.event.AmplifyJobId);
    }
    // ------
    // DELETE
    // ------
    async onDelete() {
        // We can't delete this resource as it's a deployment.
        return;
    }
    async isDeleteComplete() {
        // We can't delete this resource as it's a deployment.
        return {
            IsComplete: true,
        };
    }
    // ------
    // UPDATE
    // ------
    async onUpdate() {
        return this.onCreate();
    }
    async isUpdateComplete() {
        return this.isActive(this.event.AmplifyJobId);
    }
    async isActive(jobId) {
        if (!jobId) {
            throw new Error('Unable to determine Amplify job status without job id');
        }
        const job = await this.amplify
            .getJob({
            appId: this.props.AppId,
            branchName: this.props.BranchName,
            jobId: jobId,
        })
            .promise();
        if (job.job.summary.status === 'SUCCEED') {
            return {
                IsComplete: true,
                Data: {
                    JobId: jobId,
                    Status: job.job.summary.status,
                },
            };
        }
        if (job.job.summary.status === 'FAILED' || job.job.summary.status === 'CANCELLED') {
            throw new Error(`Amplify job failed with status: ${job.job.summary.status}`);
        }
        else {
            return {
                IsComplete: false,
            };
        }
    }
}
exports.AmplifyAssetDeploymentHandler = AmplifyAssetDeploymentHandler;
function parseProps(props) {
    return props;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaGFuZGxlci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImhhbmRsZXIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBR0EscUNBQTRGO0FBVTVGLE1BQWEsNkJBQThCLFNBQVEsd0JBQWU7SUFLaEUsWUFBWSxPQUFnQixFQUFFLEVBQU0sRUFBRSxLQUFvQjtRQUN4RCxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUM7UUFFYixJQUFJLENBQUMsS0FBSyxHQUFHLFVBQVUsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLGtCQUFrQixDQUFDLENBQUM7UUFDdkQsSUFBSSxDQUFDLE9BQU8sR0FBRyxPQUFPLENBQUM7UUFDdkIsSUFBSSxDQUFDLEVBQUUsR0FBRyxFQUFFLENBQUM7S0FDZDtJQUVELFNBQVM7SUFDVCxTQUFTO0lBQ1QsU0FBUztJQUVDLEtBQUssQ0FBQyxRQUFRO1FBQ3RCLHNDQUFzQztRQUN0QyxPQUFPLENBQUMsR0FBRyxDQUFDLG9DQUFvQyxFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxTQUFTLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUU1Rix3Q0FBd0M7UUFDeEMsTUFBTSxJQUFJLEdBQUcsTUFBTSxJQUFJLENBQUMsT0FBTzthQUM1QixRQUFRLENBQUM7WUFDUixLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxLQUFLO1lBQ3ZCLFVBQVUsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFVBQVU7WUFDakMsVUFBVSxFQUFFLENBQUM7U0FDZCxDQUFDO2FBQ0QsT0FBTyxFQUFFLENBQUM7UUFFYixJQUNFLElBQUksQ0FBQyxZQUFZO1lBQ2pCLElBQUksQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUMsT0FBTyxDQUFDLE1BQU0sS0FBSyxTQUFTLENBQUMsRUFDL0Q7WUFDQSxPQUFPLE9BQU8sQ0FBQyxNQUFNLENBQUMsbURBQW1ELENBQUMsQ0FBQztTQUM1RTtRQUVELHVFQUF1RTtRQUN2RSxNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsRUFBRSxDQUFDLFlBQVksQ0FBQyxXQUFXLEVBQUU7WUFDakQsTUFBTSxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsWUFBWTtZQUMvQixHQUFHLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxXQUFXO1NBQzVCLENBQUMsQ0FBQztRQUVILCtCQUErQjtRQUMvQixNQUFNLFVBQVUsR0FBRyxNQUFNLElBQUksQ0FBQyxPQUFPO2FBQ2xDLGVBQWUsQ0FBQztZQUNmLEtBQUssRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEtBQUs7WUFDdkIsVUFBVSxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsVUFBVTtZQUNqQyxTQUFTLEVBQUUsUUFBUTtTQUNwQixDQUFDO2FBQ0QsT0FBTyxFQUFFLENBQUM7UUFFYixPQUFPO1lBQ0wsWUFBWSxFQUFFLFVBQVUsQ0FBQyxVQUFVLENBQUMsS0FBSztTQUMxQyxDQUFDO0tBQ0g7SUFFUyxLQUFLLENBQUMsZ0JBQWdCO1FBQzlCLE9BQU8sSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLFlBQVksQ0FBQyxDQUFDO0tBQy9DO0lBRUQsU0FBUztJQUNULFNBQVM7SUFDVCxTQUFTO0lBRUMsS0FBSyxDQUFDLFFBQVE7UUFDdEIsc0RBQXNEO1FBQ3RELE9BQU87S0FDUjtJQUVTLEtBQUssQ0FBQyxnQkFBZ0I7UUFDOUIsc0RBQXNEO1FBQ3RELE9BQU87WUFDTCxVQUFVLEVBQUUsSUFBSTtTQUNqQixDQUFDO0tBQ0g7SUFFRCxTQUFTO0lBQ1QsU0FBUztJQUNULFNBQVM7SUFFQyxLQUFLLENBQUMsUUFBUTtRQUN0QixPQUFPLElBQUksQ0FBQyxRQUFRLEVBQUUsQ0FBQztLQUN4QjtJQUVTLEtBQUssQ0FBQyxnQkFBZ0I7UUFDOUIsT0FBTyxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDLENBQUM7S0FDL0M7SUFFTyxLQUFLLENBQUMsUUFBUSxDQUFDLEtBQWM7UUFDbkMsSUFBSSxDQUFDLEtBQUssRUFBRTtZQUNWLE1BQU0sSUFBSSxLQUFLLENBQUMsdURBQXVELENBQUMsQ0FBQztTQUMxRTtRQUVELE1BQU0sR0FBRyxHQUFHLE1BQU0sSUFBSSxDQUFDLE9BQU87YUFDM0IsTUFBTSxDQUFDO1lBQ04sS0FBSyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsS0FBSztZQUN2QixVQUFVLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxVQUFVO1lBQ2pDLEtBQUssRUFBRSxLQUFLO1NBQ2IsQ0FBQzthQUNELE9BQU8sRUFBRSxDQUFDO1FBRWIsSUFBSSxHQUFHLENBQUMsR0FBRyxDQUFDLE9BQU8sQ0FBQyxNQUFNLEtBQUssU0FBUyxFQUFFO1lBQ3hDLE9BQU87Z0JBQ0wsVUFBVSxFQUFFLElBQUk7Z0JBQ2hCLElBQUksRUFBRTtvQkFDSixLQUFLLEVBQUUsS0FBSztvQkFDWixNQUFNLEVBQUUsR0FBRyxDQUFDLEdBQUcsQ0FBQyxPQUFPLENBQUMsTUFBTTtpQkFDL0I7YUFDRixDQUFDO1NBQ0g7UUFBQyxJQUFJLEdBQUcsQ0FBQyxHQUFHLENBQUMsT0FBTyxDQUFDLE1BQU0sS0FBSyxRQUFRLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxPQUFPLENBQUMsTUFBTSxLQUFLLFdBQVcsRUFBRTtZQUNuRixNQUFNLElBQUksS0FBSyxDQUFDLG1DQUFtQyxHQUFHLENBQUMsR0FBRyxDQUFDLE9BQU8sQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDO1NBQzlFO2FBQU07WUFDTCxPQUFPO2dCQUNMLFVBQVUsRUFBRSxLQUFLO2FBQ2xCLENBQUM7U0FDSDtLQUNGO0NBQ0Y7QUF0SEQsc0VBc0hDO0FBRUQsU0FBUyxVQUFVLENBQUMsS0FBVTtJQUM1QixPQUFPLEtBQUssQ0FBQztBQUNmLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvLyBhd3Mtc2RrIGF2YWlsYWJsZSBhdCBydW50aW1lIGZvciBsYW1iZGFzXG4vLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUgaW1wb3J0L25vLWV4dHJhbmVvdXMtZGVwZW5kZW5jaWVzXG5pbXBvcnQgeyBBbXBsaWZ5LCBTMyB9IGZyb20gJ2F3cy1zZGsnO1xuaW1wb3J0IHsgQW1wbGlmeUpvYklkLCBJc0NvbXBsZXRlUmVzcG9uc2UsIFJlc291cmNlRXZlbnQsIFJlc291cmNlSGFuZGxlciB9IGZyb20gJy4vY29tbW9uJztcblxuZXhwb3J0IGludGVyZmFjZSBBbXBsaWZ5QXNzZXREZXBsb3ltZW50UHJvcHMge1xuICBBcHBJZDogc3RyaW5nO1xuICBCcmFuY2hOYW1lOiBzdHJpbmc7XG4gIFMzQnVja2V0TmFtZTogc3RyaW5nO1xuICBTM09iamVjdEtleTogc3RyaW5nO1xuICBUaW1lb3V0U2Vjb25kczogbnVtYmVyO1xufVxuXG5leHBvcnQgY2xhc3MgQW1wbGlmeUFzc2V0RGVwbG95bWVudEhhbmRsZXIgZXh0ZW5kcyBSZXNvdXJjZUhhbmRsZXIge1xuICBwcml2YXRlIHJlYWRvbmx5IHByb3BzOiBBbXBsaWZ5QXNzZXREZXBsb3ltZW50UHJvcHM7XG4gIHByb3RlY3RlZCByZWFkb25seSBhbXBsaWZ5OiBBbXBsaWZ5O1xuICBwcm90ZWN0ZWQgcmVhZG9ubHkgczM6IFMzO1xuXG4gIGNvbnN0cnVjdG9yKGFtcGxpZnk6IEFtcGxpZnksIHMzOiBTMywgZXZlbnQ6IFJlc291cmNlRXZlbnQpIHtcbiAgICBzdXBlcihldmVudCk7XG5cbiAgICB0aGlzLnByb3BzID0gcGFyc2VQcm9wcyh0aGlzLmV2ZW50LlJlc291cmNlUHJvcGVydGllcyk7XG4gICAgdGhpcy5hbXBsaWZ5ID0gYW1wbGlmeTtcbiAgICB0aGlzLnMzID0gczM7XG4gIH1cblxuICAvLyAtLS0tLS1cbiAgLy8gQ1JFQVRFXG4gIC8vIC0tLS0tLVxuXG4gIHByb3RlY3RlZCBhc3luYyBvbkNyZWF0ZSgpOiBQcm9taXNlPEFtcGxpZnlKb2JJZD4ge1xuICAgIC8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBuby1jb25zb2xlXG4gICAgY29uc29sZS5sb2coJ2RlcGxveWluZyB0byBBbXBsaWZ5IHdpdGggb3B0aW9uczonLCBKU09OLnN0cmluZ2lmeSh0aGlzLnByb3BzLCB1bmRlZmluZWQsIDIpKTtcblxuICAgIC8vIFZlcmlmeSBubyBqb2JzIGFyZSBjdXJyZW50bHkgcnVubmluZy5cbiAgICBjb25zdCBqb2JzID0gYXdhaXQgdGhpcy5hbXBsaWZ5XG4gICAgICAubGlzdEpvYnMoe1xuICAgICAgICBhcHBJZDogdGhpcy5wcm9wcy5BcHBJZCxcbiAgICAgICAgYnJhbmNoTmFtZTogdGhpcy5wcm9wcy5CcmFuY2hOYW1lLFxuICAgICAgICBtYXhSZXN1bHRzOiAxLFxuICAgICAgfSlcbiAgICAgIC5wcm9taXNlKCk7XG5cbiAgICBpZiAoXG4gICAgICBqb2JzLmpvYlN1bW1hcmllcyAmJlxuICAgICAgam9icy5qb2JTdW1tYXJpZXMuZmluZChzdW1tYXJ5ID0+IHN1bW1hcnkuc3RhdHVzID09PSAnUEVORElORycpXG4gICAgKSB7XG4gICAgICByZXR1cm4gUHJvbWlzZS5yZWplY3QoJ0FtcGxpZnkgam9iIGFscmVhZHkgcnVubmluZy4gQWJvcnRpbmcgZGVwbG95bWVudC4nKTtcbiAgICB9XG5cbiAgICAvLyBDcmVhdGUgYSBwcmUtc2lnbmVkIGdldCBVUkwgb2YgdGhlIGFzc2V0IHNvIEFtcGxpZnkgY2FuIHJldHJpZXZlIGl0LlxuICAgIGNvbnN0IGFzc2V0VXJsID0gdGhpcy5zMy5nZXRTaWduZWRVcmwoJ2dldE9iamVjdCcsIHtcbiAgICAgIEJ1Y2tldDogdGhpcy5wcm9wcy5TM0J1Y2tldE5hbWUsXG4gICAgICBLZXk6IHRoaXMucHJvcHMuUzNPYmplY3RLZXksXG4gICAgfSk7XG5cbiAgICAvLyBEZXBsb3kgdGhlIGFzc2V0IHRvIEFtcGxpZnkuXG4gICAgY29uc3QgZGVwbG95bWVudCA9IGF3YWl0IHRoaXMuYW1wbGlmeVxuICAgICAgLnN0YXJ0RGVwbG95bWVudCh7XG4gICAgICAgIGFwcElkOiB0aGlzLnByb3BzLkFwcElkLFxuICAgICAgICBicmFuY2hOYW1lOiB0aGlzLnByb3BzLkJyYW5jaE5hbWUsXG4gICAgICAgIHNvdXJjZVVybDogYXNzZXRVcmwsXG4gICAgICB9KVxuICAgICAgLnByb21pc2UoKTtcblxuICAgIHJldHVybiB7XG4gICAgICBBbXBsaWZ5Sm9iSWQ6IGRlcGxveW1lbnQuam9iU3VtbWFyeS5qb2JJZCxcbiAgICB9O1xuICB9XG5cbiAgcHJvdGVjdGVkIGFzeW5jIGlzQ3JlYXRlQ29tcGxldGUoKSB7XG4gICAgcmV0dXJuIHRoaXMuaXNBY3RpdmUodGhpcy5ldmVudC5BbXBsaWZ5Sm9iSWQpO1xuICB9XG5cbiAgLy8gLS0tLS0tXG4gIC8vIERFTEVURVxuICAvLyAtLS0tLS1cblxuICBwcm90ZWN0ZWQgYXN5bmMgb25EZWxldGUoKTogUHJvbWlzZTx2b2lkPiB7XG4gICAgLy8gV2UgY2FuJ3QgZGVsZXRlIHRoaXMgcmVzb3VyY2UgYXMgaXQncyBhIGRlcGxveW1lbnQuXG4gICAgcmV0dXJuO1xuICB9XG5cbiAgcHJvdGVjdGVkIGFzeW5jIGlzRGVsZXRlQ29tcGxldGUoKTogUHJvbWlzZTxJc0NvbXBsZXRlUmVzcG9uc2U+IHtcbiAgICAvLyBXZSBjYW4ndCBkZWxldGUgdGhpcyByZXNvdXJjZSBhcyBpdCdzIGEgZGVwbG95bWVudC5cbiAgICByZXR1cm4ge1xuICAgICAgSXNDb21wbGV0ZTogdHJ1ZSxcbiAgICB9O1xuICB9XG5cbiAgLy8gLS0tLS0tXG4gIC8vIFVQREFURVxuICAvLyAtLS0tLS1cblxuICBwcm90ZWN0ZWQgYXN5bmMgb25VcGRhdGUoKSB7XG4gICAgcmV0dXJuIHRoaXMub25DcmVhdGUoKTtcbiAgfVxuXG4gIHByb3RlY3RlZCBhc3luYyBpc1VwZGF0ZUNvbXBsZXRlKCkge1xuICAgIHJldHVybiB0aGlzLmlzQWN0aXZlKHRoaXMuZXZlbnQuQW1wbGlmeUpvYklkKTtcbiAgfVxuXG4gIHByaXZhdGUgYXN5bmMgaXNBY3RpdmUoam9iSWQ/OiBzdHJpbmcpOiBQcm9taXNlPElzQ29tcGxldGVSZXNwb25zZT4ge1xuICAgIGlmICgham9iSWQpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignVW5hYmxlIHRvIGRldGVybWluZSBBbXBsaWZ5IGpvYiBzdGF0dXMgd2l0aG91dCBqb2IgaWQnKTtcbiAgICB9XG5cbiAgICBjb25zdCBqb2IgPSBhd2FpdCB0aGlzLmFtcGxpZnlcbiAgICAgIC5nZXRKb2Ioe1xuICAgICAgICBhcHBJZDogdGhpcy5wcm9wcy5BcHBJZCxcbiAgICAgICAgYnJhbmNoTmFtZTogdGhpcy5wcm9wcy5CcmFuY2hOYW1lLFxuICAgICAgICBqb2JJZDogam9iSWQsXG4gICAgICB9KVxuICAgICAgLnByb21pc2UoKTtcblxuICAgIGlmIChqb2Iuam9iLnN1bW1hcnkuc3RhdHVzID09PSAnU1VDQ0VFRCcpIHtcbiAgICAgIHJldHVybiB7XG4gICAgICAgIElzQ29tcGxldGU6IHRydWUsXG4gICAgICAgIERhdGE6IHtcbiAgICAgICAgICBKb2JJZDogam9iSWQsXG4gICAgICAgICAgU3RhdHVzOiBqb2Iuam9iLnN1bW1hcnkuc3RhdHVzLFxuICAgICAgICB9LFxuICAgICAgfTtcbiAgICB9IGlmIChqb2Iuam9iLnN1bW1hcnkuc3RhdHVzID09PSAnRkFJTEVEJyB8fCBqb2Iuam9iLnN1bW1hcnkuc3RhdHVzID09PSAnQ0FOQ0VMTEVEJykge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBBbXBsaWZ5IGpvYiBmYWlsZWQgd2l0aCBzdGF0dXM6ICR7am9iLmpvYi5zdW1tYXJ5LnN0YXR1c31gKTtcbiAgICB9IGVsc2Uge1xuICAgICAgcmV0dXJuIHtcbiAgICAgICAgSXNDb21wbGV0ZTogZmFsc2UsXG4gICAgICB9O1xuICAgIH1cbiAgfVxufVxuXG5mdW5jdGlvbiBwYXJzZVByb3BzKHByb3BzOiBhbnkpOiBBbXBsaWZ5QXNzZXREZXBsb3ltZW50UHJvcHMge1xuICByZXR1cm4gcHJvcHM7XG59XG4iXX0=