"""
Simplified Redis Protocol Implementation for AetherMagic
Uses only Redis Pub/Sub but with separate connections for publishing and subscribing
"""

import asyncio
import json
from typing import Optional, Callable, Dict, Any, List
import redis.asyncio as redis
from redis.asyncio.client import PubSub

from . import ProtocolInterface, ConnectionConfig, AetherMessage, ProtocolType


class RedisProtocol(ProtocolInterface):
    """Simple Redis PubSub with separate publish/subscribe connections"""
    
    def __init__(self, config: ConnectionConfig):
        super().__init__(config)
        self.publish_client: Optional[redis.Redis] = None
        self.subscribe_client: Optional[redis.Redis] = None
        self.pubsub: Optional[PubSub] = None
        self.subscribed_channels: set = set()
        
    async def connect(self) -> bool:
        """Connect to Redis server"""
        try:
            # Create Redis connection parameters
            connection_params = {
                'host': self.config.host,
                'port': self.config.port,
                'decode_responses': True,
                'socket_timeout': self.config.timeout,
                'socket_keepalive': True,
                'socket_keepalive_options': {}
            }
            
            # Add SSL if needed
            if self.config.ssl:
                connection_params['ssl'] = True
                connection_params['ssl_cert_reqs'] = None
            
            # Add auth if provided
            if self.config.username and self.config.password:
                connection_params['username'] = self.config.username
                connection_params['password'] = self.config.password
            elif self.config.password:
                connection_params['password'] = self.config.password
                
            # Add extra parameters
            connection_params.update(self.config.extra_params)
            
            # Create single client for both publishing and subscribing
            self.client = redis.Redis(**connection_params)
            self.publish_client = self.client  # Alias for backward compatibility
            self.subscribe_client = self.client  # Alias for backward compatibility
            
            # Test connection - ping is sync in redis.asyncio
            self.client.ping()
            
            # Create pub/sub connection
            self.pubsub = self.client.pubsub()
            
            self.connected = True
            print(f"Redis: Connected (channel: {self.config.channel})")
            return True
            
        except Exception as e:
            print(f"Redis: Connection failed - {e}")
            self.connected = False
            return False
    
    async def disconnect(self) -> bool:
        """Disconnect from Redis"""
        try:
            if self.pubsub:
                await self.pubsub.close()
            if self.client:
                await self.client.close()
            self.connected = False
            return True
        except Exception as e:
            print(f"Redis: Disconnect failed - {e}")
            return False
    
    async def publish(self, topic: str, message: AetherMessage, retain: bool = False) -> bool:
        """Publish message to Redis channel"""
        try:
            if not self.publish_client or not self.connected:
                return False
                
            # Use async publish
            await self.client.publish(topic, message.to_json())
            print(f"Redis: Sending {topic} (channel: {self.config.channel})")
            
            return True
            
        except Exception as e:
            print(f"Redis: Publish failed - {e}")
            return False
    
    async def subscribe(self, topic: str, callback: Optional[Callable] = None) -> bool:
        """Subscribe to Redis channel"""
        try:
            if not self.pubsub or not self.connected:
                return False
            
            # Convert MQTT wildcards to Redis wildcards
            redis_topic = topic.replace('+', '*')  # MQTT single-level -> Redis wildcard
            redis_topic = redis_topic.replace('#', '*')  # MQTT multi-level -> Redis wildcard
                
            # Handle wildcard patterns (Redis uses * for wildcards)
            if '*' in redis_topic or '?' in redis_topic or '[' in redis_topic:
                await self.pubsub.psubscribe(redis_topic)
                print(f"Redis: Pattern subscribed to {redis_topic} (original: {topic}) (channel: {self.config.channel})")
            else:
                await self.pubsub.subscribe(redis_topic)
                print(f"Redis: Subscribed to {redis_topic} (channel: {self.config.channel})")
                
            self.subscribed_channels.add(redis_topic)
            return True
            
        except Exception as e:
            print(f"Redis: Subscribe failed - {e}")
            return False
    
    async def unsubscribe(self, topic: str) -> bool:
        """Unsubscribe from Redis channel"""
        try:
            if not self.pubsub or not self.connected:
                return False
            
            # Convert MQTT wildcards to Redis wildcards
            redis_topic = topic.replace('+', '*')  # MQTT single-level -> Redis wildcard
            redis_topic = redis_topic.replace('#', '*')  # MQTT multi-level -> Redis wildcard
                
            if '*' in redis_topic or '?' in redis_topic or '[' in redis_topic:
                await self.pubsub.punsubscribe(redis_topic)
            else:
                await self.pubsub.unsubscribe(redis_topic)
                
            self.subscribed_channels.discard(redis_topic)
            return True
            
        except Exception as e:
            print(f"Redis: Unsubscribe failed - {e}")
            return False
    
    async def receive_messages(self) -> List[Dict[str, Any]]:
        """Receive messages from subscribed channels"""
        messages = []
        
        if not self.pubsub or not self.connected:
            return messages
            
        # Don't try to read if no subscriptions
        if not self.subscribed_channels:
            return messages
            
        try:
            # Get message with timeout to avoid blocking
            message = await asyncio.wait_for(
                self.pubsub.get_message(ignore_subscribe_messages=True),
                timeout=0.1
            )
            
            if message and message['type'] in ['message', 'pmessage']:
                channel = message['channel']
                data = message['data']
                
                if data:
                    print(f"Redis: Received {channel} (channel: {self.config.channel})")
                    messages.append({
                        'topic': channel,
                        'payload': data
                    })
                    
        except asyncio.TimeoutError:
            pass  # No messages available
        except Exception as e:
            print(f"Redis: Receive failed - {e}")
            
        return messages
    
    def generate_topic(self, job: str, task: str, context: str, tid: str, action: str, shared: bool = False, workgroup: str = "") -> str:
        """Generate Redis channel name - use colon format for Redis"""
        return f"{self.config.union}:{job}:{task}:{context}:{tid}:{action}"
    
    def parse_topic(self, topic: str) -> Dict[str, str]:
        """Parse Redis channel name with colon separators"""
        parts = topic.split(':')
        
        if len(parts) >= 6:
            return {
                'union': parts[0],
                'job': parts[1], 
                'task': parts[2],
                'context': parts[3],
                'tid': parts[4],
                'action': parts[5]
            }
        return {}