# greenlang/agents/energy_balance_agent.py

try:
    import pandas as pd
except ImportError:
    raise ImportError(
        "pandas is required for the EnergyBalanceAgent. "
        "Install it with: pip install greenlang[analytics]"
    )

from typing import Dict, Any
from greenlang.agents.base import BaseAgent, AgentResult

# Performance constants for a typical parabolic trough collector
OPTICAL_EFFICIENCY = 0.75  # Includes reflectivity, intercept factor, etc.
THERMAL_LOSS_COEFF_U1 = 0.2  # W/m^2-K
THERMAL_LOSS_COEFF_U2 = 0.003  # W/m^2-K^2


class EnergyBalanceAgent(BaseAgent):
    """
    Performs an hourly energy balance simulation for the solar field.
    """

    def execute(self, input_data: Dict[str, Any]) -> AgentResult:
        """
        Args:
            input_data: Dictionary containing solar and load data.

        Returns:
            AgentResult with final solar fraction and performance data.
        """
        try:
            solar_resource_df_json = input_data.get("solar_resource_df_json")
            load_profile_df_json = input_data.get("load_profile_df_json")
            required_aperture_area_m2 = input_data.get("required_aperture_area_m2")

            if not all(
                [
                    solar_resource_df_json,
                    load_profile_df_json,
                    required_aperture_area_m2 is not None,
                ]
            ):
                return AgentResult(success=False, error="Missing required inputs")

            solar_df = pd.read_json(solar_resource_df_json, orient="split")
            load_df = pd.read_json(load_profile_df_json, orient="split")

            # Merge the two dataframes on their hourly index
            df = solar_df.join(load_df, how="inner")

            # Assume average fluid temperature for thermal loss calculation
            avg_fluid_temp_C = 100  # A reasonable average for this process

            # Calculate hourly performance
            def calculate_hourly_yield(row):
                dni = row["dni_w_per_m2"]
                ambient_temp = row["temp_c"]

                # 1. Calculate incident energy
                incident_energy_w = dni * required_aperture_area_m2

                # 2. Calculate absorbed energy
                absorbed_energy_w = incident_energy_w * OPTICAL_EFFICIENCY

                # 3. Calculate thermal losses
                delta_t = avg_fluid_temp_C - ambient_temp
                if delta_t < 0:
                    delta_t = 0

                thermal_loss_w = (
                    THERMAL_LOSS_COEFF_U1 * delta_t + THERMAL_LOSS_COEFF_U2 * delta_t**2
                ) * required_aperture_area_m2

                # 4. Calculate useful energy generated by the field
                useful_energy_w = absorbed_energy_w - thermal_loss_w
                if useful_energy_w < 0:
                    useful_energy_w = 0

                # Convert from W to kWh
                return useful_energy_w / 1000

            df["solar_yield_kWh"] = df.apply(calculate_hourly_yield, axis=1)

            # The actual heat delivered is the minimum of what's generated and what's demanded
            df["heat_delivered_kWh"] = df[["solar_yield_kWh", "demand_kWh"]].min(axis=1)

            # Calculate the final solar fraction
            total_heat_delivered = df["heat_delivered_kWh"].sum()
            total_demand = df["demand_kWh"].sum()

            solar_fraction = (
                total_heat_delivered / total_demand if total_demand > 0 else 0
            )

            return AgentResult(
                success=True,
                data={
                    "solar_fraction": solar_fraction,
                    "performance_df_json": df.to_json(orient="split"),
                    "total_solar_yield_gwh": df["solar_yield_kWh"].sum() / 1e6,
                },
            )
        except Exception as e:
            return AgentResult(success=False, error=str(e))
