
# the file specification, defined as the earlieast version to handle
# current file standard. (revision number). specification versions are not
# continuous. historically taken [1, 39]

SPECIFICATION = 39

import importlib.metadata
ver_string = importlib.metadata.version("exprmat")
MAJOR, MINOR, REVISION = [int(x) for x in ver_string.split('.')]

import os
os.environ['OPENBLAS_NUM_THREADS'] = '1'
os.environ['GOTO_NUM_THREADS'] = '1'
os.environ['OMP_NUM_THREADS'] = '1'

import mudata as mu
import anndata as ad
import scanpy as sc
import pandas as pd
import pathlib
from PIL import Image

# build cython package

try: import exprmat.peaks.idr.invcdf
except ImportError:
    
    from exprmat.ansi import info
    info('seems to be your first use of exprmat. the C modules are not compiled yet.')
    info('compiling cython extension modules ...')

    from distutils.core import setup
    from distutils.extension import Extension
    from Cython.Distutils import build_ext
    from Cython.Build import cythonize
    import numpy
    import sys

    # change to the project source directory
    # thus we can build the extension in the source directory.
    sys.argv = ['', 'build_ext', '--inplace']
    cwd = os.getcwd()
    os.chdir(os.path.dirname(__file__))
    info(f'entering working directory: {os.getcwd()}')

    extensions = [
        Extension(
            'peaks.idr.invcdf',
            [__file__.replace('__init__.py', os.path.join('peaks', 'idr', 'invcdf.pyx'))],
            include_dirs = [numpy.get_include()],
            define_macros = [('NPY_NO_DEPRECATED_API', 'NPY_1_7_API_VERSION')]
        )
    ]
    
    setup(
        name = 'peaks.idr.invcdf',
        cmdclass = { 'build_ext': build_ext },
        ext_modules = cythonize(extensions, language_level = 3, quiet = True)
    )

    os.chdir(cwd)
    # remove the autogenerated build directory. since we compile inplace.
    os.system(f'rm -r {os.path.join(os.path.dirname(__file__), "build")}')
    info('compilation finished.')


from exprmat.configuration import Configuration as configuration
from exprmat.ansi import warning, info

config = configuration()

mu.set_options(pull_on_update = False)
Image.MAX_IMAGE_PIXELS = config['max.image']

DATABASE_SETUP_MESSAGE = """
The exprmat package do not come with an included database after version 0.1.25, 
due to the extensive size of the reference data. You need to setup the database
and configure it properly with configuration file `.exprmatrc` under your
user root or current working directory

For more guide on how to setup the database folder, and more information about
what the database includes, see <https://github.com/yang-z-03/exprmat>
"""

# load configuration
PERFORM_DATABASE_CHECK = True
default_finders = [
    'exprmat.config',
    '.exprmat.config',
    '.exprmatrc',
    os.path.join(str(pathlib.Path.home()), 'exprmat.config'),
    os.path.join(str(pathlib.Path.home()), '.exprmat.config'),
    os.path.join(str(pathlib.Path.home()), '.exprmatrc')
]

DEFAULT_CONFIG = '[internals]'
for finder in default_finders:
    if os.path.exists(finder):
        DEFAULT_CONFIG = finder
        if False: info(f'load configuration from {finder}')
        with open(finder, 'r') as f:
            import json
            workspace_config = json.load(f)
            config.update(workspace_config)
            break

basepath = config['data']

# core method exports
from exprmat.utils import setup_styles
from exprmat.ansi import error, warning, info, pprog, pproga
from exprmat.reader.experiment import experiment, load_experiment
from exprmat.reader.metadata import metadata, load_metadata


setup_styles()


def version_db():
    dbpath = config['data']
    version_file = os.path.join(dbpath, '.bsync', 'current')

    if not os.path.exists(version_file): 
        db_ver = None
    else:
        with open(version_file, 'r') as vf:
            db_ver = vf.read().strip()
    
    return db_ver


# perform database integrity check.
if PERFORM_DATABASE_CHECK:
    db_ver = version_db()
    if db_ver is None: 
        warning('the database is not installed.')
        print(DATABASE_SETUP_MESSAGE)
    elif db_ver != importlib.metadata.version("exprmat"):
        warning('the database version do not match the package version')
        warning(f'db version: {db_ver}  package version: {importlib.metadata.version("exprmat")}')


def cuda():
    import torch
    from exprmat.utils import supports_tensorcore
    info(f'PyTorch version: {torch.__version__}')
    info(f'GPU acceleration availability: {"YES" if torch.cuda.is_available() else "NO"}')
    if torch.cuda.is_available():
        info(f'CUDA version: {torch.version.cuda}')
        info(f'Number of installed GPUs: {torch.cuda.device_count()}')
        info(f'Supporting BF16 format: {"YES" if torch.cuda.is_bf16_supported() else "NO"}')
        info(f'Devices:')
        for i in range(torch.cuda.device_count()):
            total_mem = torch.cuda.get_device_properties(i).total_memory / 1024 / 1024 / 1024
            dedicated_mem = torch.cuda.memory_allocated(i) / 1024 / 1024 / 1024
            info(f'[{i}] {torch.cuda.get_device_name(i)} {"*" if torch.cuda.current_device() == i else ""}')
            info(f'    CUDA capability: {torch.cuda.get_device_capability(i)}')
            info(f'    Installed VRAM (GiB): {total_mem:.2f} GiB')
            info(f'    Supporting TensorCore: {"YES" if supports_tensorcore() else "NO"}')
            info(f'    Current dedicated memory: {dedicated_mem:.2f} / {total_mem:.2f} GiB ({100 * dedicated_mem / total_mem:.1f}%)')


def version(): 
    import os
    import sys
    import platform
    from exprmat.ansi import info, error, format_file_size

    db_ver = version_db()
    info(f'exprmat {MAJOR}.{MINOR}.{REVISION} / exprmat-db {db_ver if db_ver is not None else "(Not installed)"}')
    info(f'os: {os.name} ({sys.platform})  platform version: {platform.release()}')
    info(f'loaded configuration from {finder}')
    info(f'current working directory: {os.getcwd()}')
    info(f'current database directory: {config["data"]} ({db_ver if db_ver is not None else "(Not found)"})')
    memory()
    return (MAJOR, MINOR, REVISION)


def memory():
    import psutil
    from exprmat.ansi import info, error, format_file_size
    meminfo = psutil.Process().memory_info()
    resident = meminfo.rss
    virtual = meminfo.vms
    info(f'resident memory: {format_file_size(resident)}')
    info(f'virtual memory: {format_file_size(virtual)}')


setwd = os.chdir
getwd = os.getcwd


__all__ = [
    'setup_styles',
    'experiment',
    'metadata',
    'load_experiment',
    'load_metadata',
    'version',
    'memory',
    'cuda',
    'setwd',
    'getwd',
    'error',
    'warning',
    'info',
    'pprog',
    'pproga'
]