
from __future__ import annotations

from typing import Literal
from pathlib import Path
import numpy as np
from anndata import AnnData
import warnings

import exprmat.snapatac as internal
from exprmat.ansi import error, info, warning
from exprmat.utils import get_file_format, anndata_rs_ipar, anndata_rs_par
from exprmat.data.finders import get_genome_size
from exprmat.preprocessing.emptydrops import filter_cellular_barcodes_ordmag


def make_fragment_file(
    bam_file: Path,
    output_file: Path,
    is_paired: bool = True,
    barcode_tag: str | None = None,
    barcode_regex: str | None = None,
    umi_tag: str | None = None,
    umi_regex: str | None = None,
    shift_left: int = 4,
    shift_right: int = -5,
    min_mapq: int | None = 30,
    chunk_size: int = 50000000,
    mitochondria: list[str] | None = ["chrM", "MT", "M"],
    source: Literal["10x"] | None = None,
    compression: Literal["gzip", "zstandard"] | None = None,
    compression_level: int | None = None,
    tempdir: Path | None = None,
) -> dict[str, float]:
    """
    Convert a BAM file to a fragment file by performing the following steps:

    1. Filtering: remove reads that are unmapped, not primary alignment, mapq < 30,
       fails platform/vendor quality checks, or optical duplicate.
       For paired-end sequencing, it also removes reads that are not properly aligned.
    2. Deduplicate: Sort the reads by cell barcodes and remove duplicated reads
       for each unique cell barcode.
    3. Output: Convert BAM records to fragments (if paired-end) or single-end reads.

    The bam file need not to be sorted or filtered.

    Note
    ----
    - When using `barcode_regex` or `umi_regex`, the regex must contain exactly one 
      capturing group (Parentheses group the regex between them) that matches the barcodes or UMIs.
      BAM files produced by the 10X Genomics Cell Ranger pipeline are not supported,
      as they contain invalid BAM headers. Specifically, Cell Ranger ATAC <= 2.0 produces BAM
      files with no @VN tag in the header, and Cell Ranger ATAC >= 2.1 produces BAM files
      with invalid @VN tag in the header. You should either manually fix the BAM header using
      tools like samtools, or it is recommended to use the fragment files produced by 10x
      pipeline instead.

    - This function generates large temporary files in `tempdir` during sorting.
      For large files, it is recommended to set `tempdir` to a location with
      sufficient space in order to avoid running out of disk space.

    - Internally, this is done by a call to MACS3 executable.

    Parameters
    ----------
    bam_file
        File name of the BAM file.

    output_file
        File name of the output fragment file.

    is_paired
        Indicate whether the BAM file contain paired-end reads

    barcode_tag
        Extract barcodes from TAG fields of BAM records, e.g., `"CB"`.

    barcode_regex
        Extract barcodes from read names of BAM records using regular expressions.
        Reguler expressions should contain exactly one capturing group (Parentheses group and 
        regex between them) that matches the barcodes. For example, `barcode_regex = "(..:..:..:..):\\\\w+$"`
        extracts `bd:69:Y6:10` from `A01535:24:HW2MMDSX2:2:1359:8513:3458:bd:69:Y6:10:TGATAGGTTG`.

    umi_tag
        Extract UMI from TAG fields of BAM records.

    umi_regex
        Extract UMI from read names of BAM records using regular expressions.
        See `barcode_regex` for more details.

    shift_left
        Insertion site correction for the left end. Note this has no effect on single-end reads.

    shift_right
        Insertion site correction for the right end. Note this has no effect on single-end reads.

    min_mapq
        Filter the reads based on MAPQ.

    chunk_size
        The size of data retained in memory when performing sorting. Larger chunk sizes
        result in faster sorting and greater memory usage.

    source
        The source of the BAM file. This is used for vendor-specific processing.
        For example, the BAM files generated by 10X Genomics needs special processing
        to fix the malformed BAM headers. Currently the only supported source is "10x".

    chrM
        A list of mitochondrial chromosome names, used to calculate QC metrics.

    compression
        Compression type. If `None`, it is inferred from the suffix.

    compression_level
        Compression level. 1-9 for gzip, 1-22 for zstandard.
        If `None`, it is set to 6 for gzip and 3 for zstandard.

    tempdir
        Location to store temporary files. If `None`, system temporary directory will be used.

    Returns
    -------
    dict[str, float]
        A dictionary containing the following metrics:

    - "sequenced_reads": Number of reads in the input BAM file.
    - "sequenced_read_pairs": Number of read pairs in the input BAM file.
    - "frac_duplicates": Fraction of high-quality read pairs that are deemed to be PCR duplicates. 
      This metric is a measure of sequencing saturation and is a function of library complexity 
      and sequencing depth. More specifically, this is the fraction of high-quality fragments with 
      a valid barcode that align to the same genomic position as another read pair in the library.
    - "frac_confidently_mapped": Fraction of sequenced reads or read pairs with mapping quality > 30.
    - "frac_unmapped": Fraction of sequenced reads or read pairs that have a valid barcode but 
      could not be mapped to the genome.
    - "frac_valid_barcode": Fraction of reads or read pairs with barcodes that match the whitelist 
      after error correction.
    - "frac_nonnuclear": Fraction of sequenced read pairs that have a valid barcode and map to 
      non-nuclear genome contigs, including mitochondria, with mapping quality > 30.
    - "frac_fragment_in_nucleosome_free_region": Fraction of high-quality fragments smaller than 147 basepairs.
    - "frac_fragment_flanking_single_nucleosome": Fraction of high-quality fragments between 147 and 294 basepairs.

    See Also
    --------
    import_fragments
    """

    if barcode_tag is None and barcode_regex is None:
        error("either barcode_tag or barcode_regex must be set.")
    if barcode_tag is not None and barcode_regex is not None:
        error("only one of barcode_tag or barcode_regex can be set.")

    if compression is None:
        _, compression = get_file_format(output_file)

    return internal.make_fragment_file(
        bam_file, output_file, is_paired, shift_left, shift_right, chunk_size,
        barcode_tag, barcode_regex, umi_tag, umi_regex, min_mapq, mitochondria, source,
        compression, compression_level, tempdir,
    )



def import_fragments(
    fragment_file: Path | list[Path],
    assembly: str,
    *,
    file: Path | list[Path] | None = None,
    min_num_fragments: int = 200,
    sorted_by_barcode: bool = True,
    whitelist: Path | list[str] | None = None,
    mitochondria: list[str] = ["chrM", "M", "MT"],
    shift_left: int = 0,
    shift_right: int = 0,
    chunk_size: int = 2000,
    tempdir: Path | None = None,
    backend: Literal['hdf5'] = 'hdf5',
    n_jobs: int = 8,
) -> internal.AnnData:
    """
    Import data fragment files and compute basic QC metrics.

    A fragment refers to the sequence data originating from a distinct location
    in the genome. In single-ended sequencing, one read equates to a fragment.
    However, in paired-ended sequencing, a fragment is defined by a pair of reads.
    This function is designed to handle, store, and process input files with
    fragment data, further yielding a range of basic Quality Control (QC) metrics.
    These metrics include the total number of unique fragments, duplication rates,
    and the percentage of mitochondrial DNA detected.

    How fragments are stored is dependent on the sequencing approach utilized.
    For single-ended sequencing, fragments are found in `.obsm['fragment_single']`.
    In contrast, for paired-ended sequencing, they are located in
    `.obsm['fragment_paired']`.

    Diving deeper technically, the fragments are internally structured within a
    Compressed Sparse Row (CSR) matrix. In this configuration, each row signifies
    a specific cell, while each column represents a unique genomic position.
    Fragment starting positions dictate the column indices. Matrix values
    capture the lengths of the fragments for paired-end reads or the lengths of
    the reads for single-ended scenarios. It's important to note that for
    single-ended reads, the values are signed, with the sign providing information
    on the fragment's strand orientation. Additionally, it is worth noting that
    cells may harbor duplicate fragments, leading to the presence of duplicate
    column indices within the matrix. As a result, the matrix deviates from
    the standard CSR format, and it is not advisable to use the matrix for linear
    algebra operations.
    
    Note
    ----
    - This function accepts both single-end and paired-end reads. 
      If the records in the fragment file contain 6 columns with the last column
      representing the strand of the fragment, the fragments are considered single-ended.
      Otherwise, the fragments are considered paired-ended.
    - When `file` is not `None`, this function uses constant memory regardless of
      the size of the input file.
    - When `sorted_by_barcode` is `False`, this function will sort the fragment file
      first, during which temporary files will be created in `tempdir`. The size of
      temporary files is proportional to the number of records in the fragment file.
      For large fragment files, it is recommended to set `tempdir` to a location with
      sufficient space in order to avoid running out of disk space.
    - The QC metrics are computed only for reads that are included by the `whitelist`
      or `chrom_sizes`.

    Warning
    -------
    When the input to the function is a list of files, it employs multiprocessing
    to process these files concurrently. In this case, however, it is crucial to
    safeguard the entry point of the program by encapsulating the function call
    within `if __name__ == '__main__':`. This condition ensures that the module
    is being run as the main program and not being loaded as a module from
    another script. Without this protection, each subprocess might attempt to
    spawn its own subprocesses, leading to a cascade of process spawns -- a situation
    that can cause the program to hang or crash due to infinite recursion.
    You don't need to do this in Jupyter notebook as it automatically does that.

    Parameters
    ----------
    fragment_file
        File name of the fragment file, optionally compressed with gzip or zstd.
        This can be a single file or a list of files.
        If it is a list of files, a separate AnnData object will be created for each file.
        A fragment file must contain at least 5 columns:
        [chromosome, start, end, barcode, count].
        Optionally it can contain one more column indicating the strand of the fragment.
        When strand is provided, the fragments are considered single-ended.

    assembly
        A genome assembly name. This implys the organism already. (e.g. 'grcm38', 'mm10')
    
    file
        File name of the output h5ad file used to store the result. If provided,
        result will be saved to a backed AnnData, otherwise an in-memory AnnData
        is used. If `fragment_file` is a list of files, `file` must also be a list of files if provided.
    
    min_num_fragments
        Number of unique fragments threshold used to filter cells
    
    sorted_by_barcode
        Whether the fragment file has been sorted by cell barcodes. This function will be faster 
        if `sorted_by_barcode == True`. Note the ``make_fragment_file()`` will always sort the
        fragment file by barcode for you.

    whitelist
        File name or a list of barcodes. If it is a file name, each line must contain a valid 
        barcode. When provided, only barcodes in the whitelist will be retained.

    shift_left
        Insertion site correction for the left end. This is set to 0 by default,
        as shift correction is usually done in the fragment file generation step.

    mitochondria
        A list of chromosome names that are considered mitochondrial DNA. This is
        used to compute the fraction of mitochondrial DNA.
    
    shift_right
        Insertion site correction for the right end. Note this has no effect on single-end reads.
        For single-end reads, `shift_right` will be set using the value of `shift_left`.
        This is set to 0 by default, as shift correction is usually done in the fragment
        file generation step.
    
    chunk_size
        Increasing the chunk_size may speed up I/O but will use more memory.
        The speed gain is usually not significant.
    
    tempdir
        Location to store temporary files. If `None`, system temporary directory
        will be used.
    
    backend
        The backend.
    
    n_jobs
        Number of jobs to run in parallel when `fragment_file` is a list.
        If `-1`, all CPUs will be used.

    Returns
    -------
    AnnData | ad.AnnData
        An annotated data matrix of shape `n_obs` * `n_vars`. Rows correspond to
        cells and columns to regions. If `file = None`, an in-memory AnnData will be
        returned, otherwise a backed AnnData is returned.
    
    It should be noted that after this step, the matrix inside anndata is empty, and
    the dataset seems to contain 0 columns. This is normal, because columns are
    appended after you call ``add_tile_matrix()``.
    """

    warnings.filterwarnings('ignore')
    chrom_sizes = get_genome_size(assembly)
    if len(chrom_sizes) == 0: error("chrom_size cannot be empty")

    if whitelist is not None:
        if isinstance(whitelist, str) or isinstance(whitelist, Path):
            with open(whitelist, "r") as fl:
                whitelist = set([line.strip() for line in fl])
        else: whitelist = set(whitelist)

    if isinstance(fragment_file, list):
        n = len(fragment_file)
        if file is None:
            adatas = [AnnData() for _ in range(n)]
        else:
            if len(file) != n:
                error("the length of 'file' must be the same as the length of 'fragment_file'")
            # initialize backed anndata files
            adatas = [internal.AnnData(filename = f, backend = backend) for f in file]

        anndata_rs_ipar(
            list(enumerate(adatas)),
            lambda x: internal.import_fragments(
                x[1], fragment_file[x[0]], chrom_sizes, mitochondria, min_num_fragments,
                sorted_by_barcode, shift_left, shift_right, chunk_size, whitelist, tempdir,
            ), n_jobs = n_jobs,
        )

        warnings.filterwarnings('default')
        return adatas
    
    else:

        adata = AnnData() if file is None else internal.AnnData(filename = file, backend = backend)
        internal.import_fragments(
            adata, fragment_file, chrom_sizes, mitochondria, min_num_fragments,
            sorted_by_barcode, shift_left, shift_right, chunk_size, whitelist, tempdir,
        )

        warnings.filterwarnings('default')
        return adata


def import_contacts(
    contact_file: Path,
    assembly,
    *,
    file: Path | None = None,
    sorted_by_barcode: bool = True,
    bin_size: int = 500000,
    chunk_size: int = 200,
    tempdir: Path | None = None,
    backend: Literal['hdf5'] = 'hdf5',
) -> internal.AnnData:
    """
    Import chromatin contacts. (from HiC-seq typically)

    Parameters
    ----------
    contact_file
        File name of the fragment file.
        
    sorted_by_barcode
        Whether the fragment file has been sorted by cell barcodes. If 
        `sorted_by_barcode == True`, this function makes use of small fixed amout of memory.
    
    bin_size
        The size of consecutive genomic regions used to record the counts.

    chunk_size
        Increasing the chunk_size speeds up I/O but uses more memory.

    Returns
    -------
    AnnData | ad.AnnData
        An annotated data matrix of shape `n_obs` x `n_vars`. Rows correspond to
        cells and columns to regions.
    """

    chrom_sizes = get_genome_size(assembly)
    
    adata = AnnData() if file is None else internal.AnnData(filename = file, backend = backend)
    internal.import_contacts(
        adata, contact_file, chrom_sizes, sorted_by_barcode, bin_size, chunk_size, tempdir)
    return adata


def import_values(
    input_dir: Path,
    assembly,
    *,
    file: Path | None = None,
    whitelist: Path | list[str] | None = None,
    chunk_size: int = 200,
    backend: Literal['hdf5'] = 'hdf5',
) -> internal.AnnData:
    """
    Import values associated with base pairs, typically from experiments like
    whole-genome bisulfite sequencing (WGBS).

    Parameters
    ----------
    input_dir
        Directory containing the input files. Each file corresponds to a single cell.

    whitelist
        File name or a list of barcodes. If it is a file name, each line
        must contain a valid barcode. When provided, only barcodes in the whitelist
        will be retained.
    """

    chrom_sizes = get_genome_size(assembly)

    if whitelist is not None:
        if isinstance(whitelist, str) or isinstance(whitelist, Path):
            with open(whitelist, "r") as fl:
                whitelist = set([line.strip() for line in fl])
        else: whitelist = set(whitelist)

    adata = AnnData() if file is None else internal.AnnData(filename = file, backend = backend)
    internal.import_values(adata, input_dir, chrom_sizes, chunk_size, whitelist)
    return adata


def add_tile_matrix(
    adata: internal.AnnData | list[internal.AnnData],
    *,
    bin_size: int = 500,
    inplace: bool = True,
    chunk_size: int = 500,
    exclude_chroms: list[str] | str | None = ["chrM", "chrY", "MT", "M", "Y"],
    min_frag_size: int | None = None,
    max_frag_size: int | None = None,
    counting_strategy: Literal['fragment', 'insertion', 'paired-insertion'] = 'paired-insertion',
    value_type: Literal['target', 'total', 'fraction'] = 'target',
    summary_type: Literal['sum', 'mean'] = 'sum',
    file: Path | None = None,
    backend: Literal['hdf5'] = 'hdf5',
    n_jobs: int = 8,
) -> internal.AnnData | None:
    """
    Generate cell by bin count matrix. The matrix will typically fill X and populate
    the columns of the annotated data. Note that typically, the anndata object have
    not a single column before this method is run.

    Parameters
    ----------
    bin_size
        The size of consecutive genomic regions used to record the counts. Typically,
        for a bin size of 500bp, a human genome is splitted into 6 million units.

    inplace
        Whether to add the tile matrix to the AnnData object or return a new AnnData object.

    chunk_size
        Increasing the chunk_size speeds up I/O but uses more memory.

    exclude_chroms
        A list of chromosomes to exclude. By default, all regions on the mitochondria
        (non-nuclear genome) and the chromosome Y will be excluded, since most of the
        regions on Y is not accessible, and do not yield meaningful results.

    min_frag_size
        Minimum fragment size to include.

    max_frag_size
        Maximum fragment size to include.

    counting_strategy
        The strategy to compute feature counts. It must be one of the following:
        "fragment", "insertion", or "paired-insertion". "fragment" means the
        feature counts are assigned based on the number of fragments that overlap
        with a region of interest. "insertion" means the feature counts are assigned
        based on the number of insertions that overlap with a region of interest.
        "paired-insertion" is similar to "insertion", but it only counts the insertions
        once if the pair of insertions of a fragment are both within the same region
        of interest. Note that this parameter has no effect if input are single-end reads.
    
    value_type
        The type of value to use from ``.obsm['_values']``, only available when 
        data is imported using ``import_values()``. It must be one of the following:
        "target", "total", or "fraction". "target" means the value is the number
        of recrods that are with postive measurements, e.g., number of methylated bases.
        "total" means the value is the total number of measurements, e.g., methylated bases plus
        unmethylated bases. "fraction" means the value is the fraction of the
        records that are positive, e.g., the fraction of methylated bases.

    summary_type
        The type of summary to use when multiple values are found in a bin. This parameter
        is only used when `.obsm['_values']` exists, It must be one of the following: 
        "sum" or "mean". This is only meaningful when the adata is generated by ``import_values()``.
    """
    
    def fun(data, out):
        internal.mk_tile_matrix(
            data, bin_size, chunk_size, counting_strategy, value_type, 
            summary_type, exclude_chroms, min_frag_size, max_frag_size, out
        )

    if isinstance(exclude_chroms, str):
        exclude_chroms = [exclude_chroms]

    if inplace:
        if isinstance(adata, list):
            anndata_rs_par(
                adata,
                lambda x: fun(x, None),
                n_jobs = n_jobs,
            )
        
        else: fun(adata, None)
    
    else:
        if file is None:
            if adata.isbacked: out = AnnData(obs = adata.obs[:].to_pandas())
            else: out = AnnData(obs = adata.obs[:])
        else: out = internal.AnnData(filename = file, backend = backend, obs = adata.obs[:])
        fun(adata, out)
        return out


def make_peak_matrix(
    adata: internal.AnnData | internal.AnnDataSet,
    *,
    use_rep: str | list[str] | None = None,
    inplace: bool = False,
    file: Path | None = None,
    backend: Literal['hdf5'] = 'hdf5',
    peak_file: Path | None = None,
    chunk_size: int = 500,
    use_x: bool = False,
    min_frag_size: int | None = None,
    max_frag_size: int | None = None,
    counting_strategy: Literal['fragment', 'insertion', 'paired-insertion'] = 'paired-insertion',
    value_type: Literal['target', 'total', 'fraction'] = 'target',
    summary_type: Literal['sum', 'mean'] = 'sum',
) -> internal.AnnData:
    """
    Generate cell by peak count matrix. Peaks are encoded by regions in bed files.
    Alternatively, you can use peak definitions stored in the uns slot ``use_rep``.

    Parameters
    ----------
    use_rep
        This is used to read peak information from `.uns[use_rep]`.
        The peaks can also be provided by a list of strings: ["chr1:1-100", "chr2:2-200"].
        
    peak_file
        Bed file containing the peaks. If provided, peak information will be read
        from this file.
        
    use_x
        If True, use the matrix stored in `.X` as raw counts.
        Otherwise the `.obsm['insertion']` is used.

    min_frag_size
        Minimum fragment size to include.

    max_frag_size
        Maximum fragment size to include.

    counting_strategy
        The strategy to compute feature counts. It must be one of the following:
        "fragment", "insertion", or "paired-insertion". "fragment" means the feature counts are 
        assigned based on the number of fragments that overlap with a region of interest. 
        "insertion" means the feature counts are assigned based on the number of insertions that 
        overlap with a region of interest. "paired-insertion" is similar to "insertion", but it 
        only counts the insertions once if the pair of insertions of a fragment are both within 
        the same region of interest. Note that this parameter has no effect if input are single-end reads.
    """

    import gzip

    if peak_file is not None and use_rep is not None:
        raise RuntimeError("'peak_file' and 'use_rep' cannot be both set") 

    if use_rep is None and peak_file is None:
        use_rep = "peaks"

    if isinstance(use_rep, str):
        df = adata.uns[use_rep]
        peaks = df[df.columns[0]]
    else: peaks = use_rep

    if peak_file is not None:
        if Path(peak_file).suffix == ".gz":
            with gzip.open(peak_file, 'rt') as f:
                peaks = [line.strip() for line in f]
        else:
            with open(peak_file, 'r') as f:
                peaks = [line.strip() for line in f]

    if inplace: out = None
    elif file is None:
        if adata.isbacked: out = AnnData(obs = adata.obs[:].to_pandas())
        else: out = AnnData(obs = adata.obs[:])
    else: out = internal.AnnData(filename = file, backend = backend, obs = adata.obs[:])
    
    internal.mk_peak_matrix(
        adata, peaks, chunk_size, use_x, counting_strategy, value_type, summary_type,
        min_frag_size, max_frag_size, out
    )

    return out


def make_gene_matrix(
    adata: internal.AnnData | internal.AnnDataSet,
    assembly,
    *,
    inplace: bool = False,
    file: Path | None = None,
    backend: Literal['hdf5'] | None = 'hdf5',
    chunk_size: int = 500,
    use_x: bool = False,
    id_type: Literal['gene', 'transcript'] = "gene",
    upstream: int = 2000,
    downstream: int = 0,
    include_gene_body: bool = True,
    transcript_name_key: str = "transcript_name",
    transcript_id_key: str = "transcript_id",
    gene_name_key: str = "gene_name",
    gene_id_key: str = "gene_id",
    min_frag_size: int | None = None,
    max_frag_size: int | None = None,
    counting_strategy: Literal['fragment', 'insertion', 'paired-insertion'] = 'paired-insertion',
) -> internal.AnnData:
    """
    Infer cell by gene activity matrix.

    Generate cell by gene activity matrix by counting the TN5 insertions in each gene's
    regulatory domain. The regulatory domain is initially defined as the TSS or the
    whole gene body (if `include_gene_body = True`). We then extends this domain
    by `upstream` and `downstream` base pairs on both sides.

    Parameters
    ----------
    use_x
        If True, use the matrix stored in `.X` to compute the gene activity.
        Otherwise the `.obsm['insertion']` is used.

    id_type
        "gene" or "transcript". The unit of expression to use when inferring activities.
        By default, we infer the activity for genes.

    upstream
        The number of base pairs upstream of the regulatory domain.

    downstream
        The number of base pairs downstream of the regulatory domain.

    include_gene_body
        Whether to include the gene body in the regulatory domain. If False, the
        TSS is used as the regulatory domain.

    transcript_name_key
        The key of the transcript name in the gene annotation file.

    transcript_id_key
        The key of the transcript id in the gene annotation file.

    gene_name_key
        The key of the gene name in the gene annotation file.

    gene_id_key
        The key of the gene id in the gene annotation file.

    min_frag_size
        Minimum fragment size to include.

    max_frag_size
        Maximum fragment size to include.

    counting_strategy
        The strategy to compute feature counts. It must be one of the following:
        "fragment", "insertion", or "paired-insertion". "fragment" means the
        feature counts are assigned based on the number of fragments that overlap
        with a region of interest. "insertion" means the feature counts are assigned
        based on the number of insertions that overlap with a region of interest.
        "paired-insertion" is similar to "insertion", but it only counts the insertions
        once if the pair of insertions of a fragment are both within the same region
        of interest [Miao24]_.
        Note that this parameter has no effect if input are single-end reads.
    """

    from exprmat.data.finders import get_genome_gff_fname
    gene_anno = get_genome_gff_fname(assembly)

    if inplace: out = None
    elif file is None:
        if adata.isbacked: out = AnnData(obs=adata.obs[:].to_pandas())
        else: out = AnnData(obs=adata.obs[:])
    else:
        out = internal.AnnData(filename = file, backend = backend, obs = adata.obs[:])
    internal.mk_gene_matrix(adata, gene_anno, chunk_size, use_x, id_type,
        upstream, downstream, include_gene_body,
        transcript_name_key, transcript_id_key, gene_name_key, gene_id_key,
        counting_strategy, min_frag_size, max_frag_size, out)
    return out


def call_cells(
    data: internal.AnnData | list[internal.AnnData],
    use_rep: str | np.ndarray[float],
    inplace: bool = True,
    n_jobs: int = 4,
) -> np.ndarray | None:
    """
    Calling cells based on the number of feature counts.

    Parameters
    ----------
    use_rep
        The representation to use for filtering. This can be a string or a numpy array.

    inplace
        Perform computation inplace or return result.
    
    Returns
    -------
    np.ndarray | None:
        If `inplace = True`, directly subsets the data matrix. Otherwise return 
        indices of cells that pass the filtering.
    """

    if isinstance(data, list):
        result = anndata_rs_par(
            data, lambda x: call_cells(x, inplace = inplace),
            n_jobs = n_jobs,
        )
        if inplace: return None
        else: return result

    counts = data.obs[use_rep].to_numpy() if isinstance(use_rep, str) else use_rep
    selected_cells = filter_cellular_barcodes_ordmag(counts, None)[0]
    if inplace:
        if data.isbacked: data.subset(selected_cells)
        else: data._inplace_subset_obs(selected_cells)
    else: return selected_cells


def filter_cells(
    data: internal.AnnData | list[internal.AnnData],
    min_counts: int | None = 1000,
    min_tsse: float | None = 5.0,
    max_counts: int | None = None,
    max_tsse: float | None = None,
    inplace: bool = True,
    n_jobs: int = 8,
) -> np.ndarray | None:
    """
    Filter cell outliers based on counts and numbers of genes expressed.
    For instance, only keep cells with at least `min_counts` counts or
    `min_tsse` TSS enrichment scores. This is to filter measurement outliers,
    i.e. "unreliable" observations.

    Parameters
    ----------
    min_counts
        Minimum number of counts required for a cell to pass filtering.

    min_tsse
        Minimum TSS enrichemnt score required for a cell to pass filtering.

    max_counts
        Maximum number of counts required for a cell to pass filtering.

    max_tsse
        Maximum TSS enrichment score expressed required for a cell to pass filtering.

    Returns
    -------
    np.ndarray | None:
        If `inplace = True`, directly subsets the data matrix. Otherwise return 
        indices of cells that pass the filtering.
    """

    if isinstance(data, list):
        result = anndata_rs_par(
            data, lambda x: filter_cells(x, min_counts, min_tsse, max_counts, max_tsse, inplace = inplace),
            n_jobs = n_jobs,
        )

        if inplace: return None
        else: return result

    selected_cells = True
    if min_counts: selected_cells &= data.obs["n.fragments"] >= min_counts
    if max_counts: selected_cells &= data.obs["n.fragments"] <= max_counts
    if min_tsse: selected_cells &= data.obs["tsse"] >= min_tsse
    if max_tsse: selected_cells &= data.obs["tsse"] <= max_tsse

    selected_cells = np.flatnonzero(selected_cells)
    if inplace:
        if data.isbacked: data.subset(selected_cells)
        else: data._inplace_subset_obs(selected_cells)
    else: return selected_cells
