import math
import time
from typing import Any, List, Optional

from ara_api._core.services.nav.behavior.commands._subutils import (
    CommandResult,
    NavigationCommand,
)
from ara_api._core.services.nav.checker.simple_goal_checker import (
    SimpleGoalChecker,
)
from ara_api._core.services.nav.controller.mppi.controller import (
    MPPIController,
)
from ara_api._core.services.nav.planner.global_planner import (
    GlobalNavigationPlanner,
)
from ara_api._utils import (
    Logger,
    MPPIControl,
    MPPIState,
    ObstacleBox,
    Path,
    Point3D,
    Vector3,
)
from ara_api._utils.communication.grpc_sync import gRPCSync
from ara_api._utils.config import MOVE_COMMAND, MPPI
from ara_api._utils.data.msp import RC


class MoveCommand(NavigationCommand):
    def __init__(
        self,
        target_position: Vector3,
        obstacles: Optional[List[ObstacleBox]] = None,
    ):
        self._logger = Logger(
            log_level=MOVE_COMMAND.LOG_LEVEL,
            log_to_file=MOVE_COMMAND.LOG_TO_FILE,
            log_to_terminal=MOVE_COMMAND.LOG_TO_TERMINAL,
        )
        self._grpc_sync = gRPCSync.get_instance()
        self._planner = GlobalNavigationPlanner()
        self._controller = MPPIController(obstacles)
        self._goal_checker = SimpleGoalChecker()
        self._obstacles = obstacles or []
        self._current_path: Optional[Path] = None
        self._goal_position: Vector3 = target_position
        self._is_executing: bool = False
        self._prev_yaw = 0.0
        self._logger.info(f"MoveCommand initialized to {self._goal_position}")

    def can_execute(self, request: Any, context: Any) -> bool:
        try:
            if not hasattr(request, "point") or request.point is None:
                self._logger.error("Целевая точка не указана в запросе")
                return False

            target_position = Vector3(
                x=request.point.x, y=request.point.y, z=request.point.z
            )

            if not self._is_valid_position(target_position):
                self._logger.error(
                    f"Некорректная целевая позиция: {target_position}"
                )
                return False

            # Получаем текущее состояние дрона
            current_state = self._get_current_drone_state()
            if current_state is None:
                self._logger.error(
                    "Не удалось получить текущее состояние дрона"
                )
                return False

            # Проверяем безопасность полета
            if not self._is_safe_to_fly(current_state):
                self._logger.error("Условия полета небезопасны")
                return False

            # Проверяем достижимость цели
            if not self._is_goal_reachable(
                current_state.position, target_position
            ):
                self._logger.error("Целевая точка недостижима")
                return False

            self._logger.info(
                f"Команда Move может быть выполнена к позиции {target_position}"
            )
            return True

        except Exception as e:
            self._logger.error(
                f"Ошибка при проверке возможности выполнения Move: {e}"
            )
            return False

    def execute(self, *args, **kwargs) -> CommandResult:
        try:
            self._is_executing = True
            self._logger.info(f"Moving to {self._goal_position}")

            current_state = self._get_current_drone_state()
            if current_state is None:
                return CommandResult(
                    success=False,
                    message="Failed to get drone state",
                )

            try:
                self._current_path = self._planner.plan_path(
                    start=current_state.position,
                    goal=self._goal_position,
                    obstacles=self._obstacles,
                )
                self._logger.info(
                    f"Path planned: "
                    f"{len(self._current_path.segments)} segments"
                )
            except Exception as e:
                return CommandResult(
                    success=False,
                    message=f"Path planning failed: {e}",
                )

            control_result = self._execute_navigation_loop(current_state)
            self._is_executing = False
            return control_result
        except Exception as e:
            self._is_executing = False
            self._logger.error(f"Execute error: {e}")
            return CommandResult(success=False, message=f"Move failed: {e}")

    def _execute_navigation_loop(
        self, initial_state: MPPIState
    ) -> CommandResult:
        start_time = time.time()
        max_time = MOVE_COMMAND.MAX_EXECUTION_TIME

        while self._is_executing and time.time() - start_time < max_time:
            try:
                current_state = self._get_current_drone_state()
                if current_state is None:
                    self._logger.warning("No state, retrying")
                    time.sleep(0.1)
                    continue

                if self._goal_checker.is_goal_reached(
                    current_pose=Point3D(
                        position=current_state.position,
                        yaw=current_state.yaw,
                    ),
                    goal_pose=Point3D(
                        position=self._goal_position,
                        yaw=current_state.yaw,
                    ),
                ):
                    self._logger.info("Goal reached")
                    self._stop_drone()
                    return CommandResult(success=True, message="Goal reached")

                goal_state = MPPIState(
                    position=self._goal_position,
                    velocity=Vector3(0, 0, 0),
                    yaw=current_state.yaw,
                    yaw_rate=0.0,
                    timestamp=time.time(),
                )

                control = self._controller.compute_control(
                    current_state=current_state,
                    reference_path=self._current_path,
                    goal_state=goal_state,
                )

                self._send_control_command(control, current_state)
                time.sleep(1.0 / MPPI.CONTROL_FREQUENCY)
            except Exception as e:
                self._logger.error(f"Nav loop error: {e}")
                time.sleep(0.1)

        self._stop_drone()
        if time.time() - start_time >= max_time:
            return CommandResult(success=False, message="Timeout")
        else:
            return CommandResult(success=False, message="Interrupted")

    def _send_control_command(
        self, control: MPPIControl, state: MPPIState
    ) -> None:
        try:
            rc = RC()
            rc.transform_from_vel(
                x=control.linear_velocity.x,
                y=control.linear_velocity.y,
                z=control.linear_velocity.z,
                w=control.angular_velocity,
                state=state,
                dt=1.0 / MPPI.CONTROL_FREQUENCY,
            )

            rc.grpc.ail = int(max(1000, min(2000, rc.grpc.ail)))
            rc.grpc.ele = int(max(1000, min(2000, rc.grpc.ele)))
            rc.grpc.thr = int(max(1000, min(2000, rc.grpc.thr)))
            rc.grpc.rud = int(max(1000, min(2000, rc.grpc.rud)))

            self._grpc_sync.msp_cmd_send_rc(rc.grpc, None)

            self._logger.debug(
                f"RC: ail={rc.grpc.ail}, ele={rc.grpc.ele}, "
                f"thr={rc.grpc.thr}, rud={rc.grpc.rud}"
            )
        except Exception as e:
            self._logger.error(f"Control send error: {e}")

    def _stop_drone(self) -> None:
        try:
            neutral = RC()
            neutral.grpc.ail = 1500
            neutral.grpc.ele = 1500
            neutral.grpc.thr = 1500
            neutral.grpc.rud = 1500
            neutral.grpc.aux1 = 1000
            neutral.grpc.aux2 = 1000
            neutral.grpc.aux3 = 1000
            neutral.grpc.aux4 = 1000

            for _ in range(3):
                self._grpc_sync.msp_cmd_send_rc(neutral.grpc, None)
                time.sleep(0.05)

            self._logger.info("Drone stopped")
        except Exception as e:
            self._logger.error(f"Stop error: {e}")

    def _get_current_drone_state(self) -> Optional[MPPIState]:
        try:
            position = self._grpc_sync.msp_get_position()
            if position is None:
                return None

            velocity = self._grpc_sync.msp_get_velocity()
            if velocity is None:
                return None

            attitude = self._grpc_sync.msp_get_attitude()
            if attitude is None:
                return None

            yaw_rad = math.radians(attitude.grpc.data.z)
            yaw_rate = (yaw_rad - self._prev_yaw) / 0.1
            self._prev_yaw = yaw_rad

            return MPPIState(
                position=Vector3(
                    x=position.grpc.data.x,
                    y=position.grpc.data.y,
                    z=position.grpc.data.z,
                ),
                velocity=Vector3(
                    x=velocity.grpc.data.x,
                    y=velocity.grpc.data.y,
                    z=velocity.grpc.data.z,
                ),
                yaw=yaw_rad,
                yaw_rate=yaw_rate,
                timestamp=time.time(),
            )
        except Exception as e:
            self._logger.error(f"Get state error: {e}")
            return None

    def _is_valid_position(self, position: Vector3) -> bool:
        if not all(
            isinstance(c, (int, float))
            for c in [position.x, position.y, position.z]
        ):
            return False

        max_coord = MOVE_COMMAND.MAX_COORDINATE_VALUE
        if abs(position.x) > max_coord or abs(position.y) > max_coord:
            return False

        if (
            position.z < MOVE_COMMAND.MIN_ALTITUDE
            or position.z > MOVE_COMMAND.MAX_ALTITUDE
        ):
            return False

        return True

    def _is_safe_to_fly(self, current_state: MPPIState) -> bool:
        try:
            if current_state.position.z < MOVE_COMMAND.MIN_SAFE_ALTITUDE:
                return False

            attitude = self._grpc_sync.msp_get_attitude()
            if attitude is None:
                return False

            max_tilt = MOVE_COMMAND.MAX_SAFE_TILT_DEG
            roll_deg = abs(attitude.grpc.data.x)
            pitch_deg = abs(attitude.grpc.data.y)

            if roll_deg > max_tilt or pitch_deg > max_tilt:
                return False

            return True
        except Exception as e:
            self._logger.error(f"Safety check error: {e}")
            return False

    def _is_goal_reachable(self, start: Vector3, goal: Vector3) -> bool:
        try:
            distance = (
                (goal.x - start.x) ** 2
                + (goal.y - start.y) ** 2
                + (goal.z - start.z) ** 2
            ) ** 0.5

            if distance > MOVE_COMMAND.MAX_DISTANCE:
                self._logger.error(f"Too far: {distance:.2f}m")
                return False

            if distance < MOVE_COMMAND.MIN_DISTANCE:
                self._logger.warning(f"Too close: {distance:.2f}m")
                return False

            for obstacle in self._obstacles:
                if self._line_intersects_obstacle(start, goal, obstacle):
                    self._logger.warning("Obstacle on path")
                    break

            return True
        except Exception as e:
            self._logger.error(f"Reachability error: {e}")
            return False

    def _line_intersects_obstacle(
        self, start: Vector3, end: Vector3, obstacle: ObstacleBox
    ) -> bool:
        min_x = min(start.x, end.x)
        max_x = max(start.x, end.x)
        min_y = min(start.y, end.y)
        max_y = max(start.y, end.y)
        min_z = min(start.z, end.z)
        max_z = max(start.z, end.z)

        return not (
            max_x < obstacle.min_point.x
            or min_x > obstacle.max_point.x
            or max_y < obstacle.min_point.y
            or min_y > obstacle.max_point.y
            or max_z < obstacle.min_point.z
            or min_z > obstacle.max_point.z
        )
