#!/usr/bin/env python3
"""
Тестовый скрипт для A*-планировщика с визуализацией путей.

Данный скрипт тестирует алгоритм A* в различных сценариях:
- Прямой путь без препятствий
- Обход одиночного препятствия
- Навигация через множественные препятствия
- Узкие проходы
- Оптимальные пути в сетке
- Сложные лабиринты

Примечание: A* алгоритм пока не реализован и будет возвращать NotImplementedError.
После реализации алгоритма этот скрипт готов к полноценному использованию.
"""

import os
import sys
import time
from typing import Any, Dict, List, Tuple

import numpy as np

from test.subutils.visual_path_planning import VisualPathPlanning

# Добавляем корневую директорию проекта в путь
sys.path.insert(0, os.path.join(os.path.dirname(__file__), "../../"))

from ara_api._core.services.nav.planner.global_planner import (
    GlobalNavigationPlanner,
)
from ara_api._utils import (
    ObstacleBox,
    PlanningAlgorithm,
    Vector3,
)


class AStarPlannerTestSuite:
    """Набор тестов для A*-планировщика с визуализацией."""

    def __init__(self):
        """Инициализация тестового набора."""
        self.planner = GlobalNavigationPlanner()
        self.test_results: List[Dict[str, Any]] = []

    def create_test_scenario(
        self,
        name: str,
        start: Vector3,
        goal: Vector3,
        obstacles: List[ObstacleBox],
        description: str = "",
        workspace_bounds: Tuple[
            Tuple[float, float], Tuple[float, float], Tuple[float, float]
        ] = ((-20, 20), (-20, 20), (0, 25)),
    ) -> Dict[str, Any]:
        """
        Создает тестовый сценарий для A*.

        Args:
            name: Название теста
            start: Начальная точка
            goal: Целевая точка
            obstacles: Список препятствий
            description: Описание теста
            workspace_bounds: Границы рабочего пространства

        Returns:
            Словарь с данными тестового сценария
        """
        return {
            "name": name,
            "start": start,
            "goal": goal,
            "obstacles": obstacles,
            "description": description,
            "workspace_bounds": workspace_bounds,
            "result": None,
            "path": None,
            "error": None,
            "metrics": None,
            "execution_time": 0.0,
            "implemented": True,
        }

    def run_test_scenario(self, scenario: Dict[str, Any]) -> Dict[str, Any]:
        """
        Выполняет один тестовый сценарий для A*.

        Args:
            scenario: Тестовый сценарий

        Returns:
            Обновленный сценарий с результатами
        """
        print(f"\n🧪 Тест A*: {scenario['name']}")
        print(f"📝 Описание: {scenario['description']}")
        print(
            f"🎯 Старт: ({scenario['start'].x:.1f}, {scenario['start'].y:.1f}, {scenario['start'].z:.1f})"
        )
        print(
            f"🏁 Цель: ({scenario['goal'].x:.1f}, {scenario['goal'].y:.1f}, {scenario['goal'].z:.1f})"
        )
        print(f"🚧 Препятствий: {len(scenario['obstacles'])}")

        try:
            start_time = time.time()

            # Планирование пути с использованием A*
            path = self.planner.plan_path(
                start=scenario["start"],
                goal=scenario["goal"],
                obstacles=scenario["obstacles"],
                algorithm=PlanningAlgorithm.A_STAR,
            )

            execution_time = time.time() - start_time

            scenario["result"] = "SUCCESS"
            scenario["path"] = path
            scenario["error"] = None
            scenario["execution_time"] = execution_time

            # Анализ качества пути
            visualizer = VisualPathPlanning(
                title=f"A* Тест: {scenario['name']}",
                workspace_bounds=scenario["workspace_bounds"],
            )
            scenario["metrics"] = visualizer.analyze_path_quality(path)

            print(f"✅ Успех! Найден путь из {len(path.segments)} сегментов")
            print(
                f"📊 Общая длина: {scenario['metrics']['total_distance']:.2f}м"
            )
            print(f"📊 Плавность: {scenario['metrics']['smoothness']:.3f} рад")
            print(
                f"📊 Изменение высоты: {scenario['metrics']['elevation_change']:.2f}м"
            )
            print(f"⏱️  Время выполнения: {execution_time:.3f}с")

        except NotImplementedError as e:
            scenario["result"] = "NOT_IMPLEMENTED"
            scenario["path"] = None
            scenario["error"] = str(e)
            scenario["metrics"] = None
            scenario["implemented"] = False

            print("⚠️  A* алгоритм пока не реализован")
            print(f"📋 Детали: {e}")

        except Exception as e:
            execution_time = time.time() - start_time
            scenario["result"] = "FAILED"
            scenario["path"] = None
            scenario["error"] = str(e)
            scenario["metrics"] = None
            scenario["execution_time"] = execution_time

            print(f"❌ Ошибка: {e}")

        return scenario

    def visualize_scenario(
        self, scenario: Dict[str, Any], show: bool = True
    ) -> VisualPathPlanning:
        """
        Визуализирует результаты тестового сценария A*.

        Args:
            scenario: Тестовый сценарий с результатами
            show: Показывать ли график сразу

        Returns:
            Объект визуализатора
        """
        visualizer = VisualPathPlanning(
            title=f"A* Планировщик: {scenario['name']}",
            workspace_bounds=scenario["workspace_bounds"],
        )

        # Добавляем стартовую и целевую точки
        visualizer.add_waypoint(
            scenario["start"],
            name="Старт",
            color="green",
            size=12,
            symbol="diamond",
        )
        visualizer.add_waypoint(
            scenario["goal"], name="Цель", color="red", size=12, symbol="cross"
        )

        # Добавляем препятствия
        for i, obstacle in enumerate(scenario["obstacles"]):
            visualizer.add_obstacle_box(
                min_point=obstacle.min_point,
                max_point=obstacle.max_point,
                name=f"Препятствие {i + 1}",
                color="red",
                opacity=0.4,
            )

        # Добавляем путь, если он найден
        if scenario["path"] is not None:
            visualizer.add_path(
                scenario["path"], name="A*-путь", color="blue", line_width=3
            )

        # Добавляем информацию о статусе
        if scenario["result"] == "NOT_IMPLEMENTED":
            # Можно добавить текстовую аннотацию о том, что алгоритм не реализован
            pass

        if show:
            visualizer.show()

        return visualizer

    def generate_test_scenarios(self) -> List[Dict[str, Any]]:
        """
        Генерирует набор тестовых сценариев для A*.

        A* особенно эффективен для:
        - Поиска оптимальных путей
        - Сценариев с четкой сеткой
        - Быстрого планирования в структурированных средах
        - Минимизации стоимости пути

        Returns:
            Список тестовых сценариев
        """
        scenarios = []

        # Тест 1: Базовый тест - прямой путь
        scenarios.append(
            self.create_test_scenario(
                name="Прямой путь",
                start=Vector3(0, 0, 5),
                goal=Vector3(12, 12, 5),
                obstacles=[],
                description="Базовый тест A* без препятствий - проверка оптимальности",
            )
        )

        # Тест 2: Простой обход
        scenarios.append(
            self.create_test_scenario(
                name="Простой обход",
                start=Vector3(0, 0, 5),
                goal=Vector3(10, 0, 5),
                obstacles=[
                    ObstacleBox(
                        min_point=Vector3(4, -2, 0),
                        max_point=Vector3(6, 2, 10),
                    )
                ],
                description="A* обход одиночного препятствия - тест оптимальности пути",
            )
        )

        # Тест 3: Лабиринт с четкой структурой (сильная сторона A*)
        scenarios.append(
            self.create_test_scenario(
                name="Структурированный лабиринт",
                start=Vector3(-8, -8, 4),
                goal=Vector3(8, 8, 4),
                obstacles=[
                    # Создаем структурированный лабиринт - идеален для A*
                    ObstacleBox(
                        min_point=Vector3(-10, -10, 0),
                        max_point=Vector3(10, -9, 8),
                    ),
                    ObstacleBox(
                        min_point=Vector3(-10, 9, 0),
                        max_point=Vector3(10, 10, 8),
                    ),
                    ObstacleBox(
                        min_point=Vector3(-10, -10, 0),
                        max_point=Vector3(-9, 10, 8),
                    ),
                    ObstacleBox(
                        min_point=Vector3(9, -10, 0),
                        max_point=Vector3(10, 10, 8),
                    ),
                    # Внутренние стены создающие коридоры
                    ObstacleBox(
                        min_point=Vector3(-6, -6, 0),
                        max_point=Vector3(-4, 6, 8),
                    ),
                    ObstacleBox(
                        min_point=Vector3(-2, -2, 0),
                        max_point=Vector3(2, 0, 8),
                    ),
                    ObstacleBox(
                        min_point=Vector3(4, -6, 0), max_point=Vector3(6, 6, 8)
                    ),
                ],
                description="Структурированный лабиринт - оптимальная задача для A*",
            )
        )

        # Тест 4: Узкие проходы (тест точности)
        scenarios.append(
            self.create_test_scenario(
                name="Узкие проходы",
                start=Vector3(-6, 0, 5),
                goal=Vector3(16, 0, 5),
                obstacles=[
                    ObstacleBox(
                        min_point=Vector3(1, -6, 0),
                        max_point=Vector3(3, -1, 12),
                    ),
                    ObstacleBox(
                        min_point=Vector3(1, 1, 0), max_point=Vector3(3, 6, 12)
                    ),
                    ObstacleBox(
                        min_point=Vector3(7, -6, 0),
                        max_point=Vector3(9, -1, 12),
                    ),
                    ObstacleBox(
                        min_point=Vector3(7, 1, 0), max_point=Vector3(9, 6, 12)
                    ),
                ],
                description="A* навигация через последовательность узких проходов",
            )
        )

        # Тест 5: Оптимизация пути в сложной среде
        scenarios.append(
            self.create_test_scenario(
                name="Оптимизация пути",
                start=Vector3(-7, -7, 3),
                goal=Vector3(15, 10, 3),
                obstacles=[
                    ObstacleBox(
                        min_point=Vector3(-2, -4, 0),
                        max_point=Vector3(1, -1, 8),
                    ),
                    ObstacleBox(
                        min_point=Vector3(3, 2, 0), max_point=Vector3(6, 5, 10)
                    ),
                    ObstacleBox(
                        min_point=Vector3(8, -2, 0),
                        max_point=Vector3(11, 1, 7),
                    ),
                    ObstacleBox(
                        min_point=Vector3(5, 7, 0), max_point=Vector3(8, 10, 9)
                    ),
                    ObstacleBox(
                        min_point=Vector3(12, 4, 0),
                        max_point=Vector3(14, 8, 6),
                    ),
                ],
                description="A* поиск оптимального пути через сложное поле препятствий",
            )
        )

        # Тест 6: Многоуровневая навигация
        scenarios.append(
            self.create_test_scenario(
                name="Многоуровневая навигация",
                start=Vector3(0, 0, 2),
                goal=Vector3(15, 0, 15),
                obstacles=[
                    ObstacleBox(
                        min_point=Vector3(2, -2, 0), max_point=Vector3(4, 2, 6)
                    ),
                    ObstacleBox(
                        min_point=Vector3(6, -2, 0),
                        max_point=Vector3(8, 2, 10),
                    ),
                    ObstacleBox(
                        min_point=Vector3(10, -2, 0),
                        max_point=Vector3(12, 2, 14),
                    ),
                    ObstacleBox(
                        min_point=Vector3(13, -2, 0),
                        max_point=Vector3(15, 2, 18),
                    ),
                ],
                description="A* 3D планирование с препятствиями различной высоты",
            )
        )

        # Тест 7: Стресс-тест производительности
        scenarios.append(
            self.create_test_scenario(
                name="Производительность A*",
                start=Vector3(-10, -10, 5),
                goal=Vector3(10, 10, 5),
                obstacles=[
                    # Много мелких препятствий для тестирования скорости A*
                    ObstacleBox(
                        min_point=Vector3(-8, -2, 0),
                        max_point=Vector3(-7, -1, 6),
                    ),
                    ObstacleBox(
                        min_point=Vector3(-6, 1, 0),
                        max_point=Vector3(-5, 2, 6),
                    ),
                    ObstacleBox(
                        min_point=Vector3(-4, -3, 0),
                        max_point=Vector3(-3, -2, 6),
                    ),
                    ObstacleBox(
                        min_point=Vector3(-2, 0, 0),
                        max_point=Vector3(-1, 1, 6),
                    ),
                    ObstacleBox(
                        min_point=Vector3(0, -2, 0),
                        max_point=Vector3(1, -1, 6),
                    ),
                    ObstacleBox(
                        min_point=Vector3(2, 1, 0), max_point=Vector3(3, 2, 6)
                    ),
                    ObstacleBox(
                        min_point=Vector3(4, -1, 0), max_point=Vector3(5, 0, 6)
                    ),
                    ObstacleBox(
                        min_point=Vector3(6, 2, 0), max_point=Vector3(7, 3, 6)
                    ),
                    ObstacleBox(
                        min_point=Vector3(8, -3, 0),
                        max_point=Vector3(9, -2, 6),
                    ),
                ],
                description="Стресс-тест производительности A* с множественными препятствиями",
            )
        )

        return scenarios

    def run_all_tests(
        self, visualize: bool = True, interactive: bool = True
    ) -> None:
        """
        Запускает все тесты A* и выводит результаты.

        Args:
            visualize: Показывать ли визуализацию для каждого теста
            interactive: Ждать ли подтверждения пользователя между тестами
        """
        print("🚀 Запуск тестового набора A*-планировщика")
        print("⭐ A* (A-Star) - алгоритм оптимального поиска пути")
        print("=" * 65)

        scenarios = self.generate_test_scenarios()
        success_count = 0
        not_implemented_count = 0

        for i, scenario in enumerate(scenarios, 1):
            print(f"\n[{i}/{len(scenarios)}]", end=" ")

            # Выполняем тест
            result_scenario = self.run_test_scenario(scenario)
            self.test_results.append(result_scenario)

            if result_scenario["result"] == "SUCCESS":
                success_count += 1
            elif result_scenario["result"] == "NOT_IMPLEMENTED":
                not_implemented_count += 1

            # Визуализация
            if visualize:
                try:
                    self.visualize_scenario(result_scenario, show=True)
                except Exception as e:
                    print(f"⚠️  Ошибка визуализации: {e}")

            # Интерактивная пауза
            if interactive and i < len(scenarios):
                input("\n📱 Нажмите Enter для продолжения...")

        # Итоговая статистика
        self.print_summary(
            success_count, not_implemented_count, len(scenarios)
        )

    def print_summary(
        self, success_count: int, not_implemented_count: int, total_count: int
    ) -> None:
        """
        Выводит итоговую статистику тестов A*.

        Args:
            success_count: Количество успешных тестов
            not_implemented_count: Количество тестов с NOT_IMPLEMENTED
            total_count: Общее количество тестов
        """
        failed_count = total_count - success_count - not_implemented_count

        print("\n" + "=" * 65)
        print("📊 ИТОГОВАЯ СТАТИСТИКА A*-ПЛАНИРОВЩИКА")
        print("=" * 65)
        print(f"✅ Успешных тестов: {success_count}/{total_count}")
        print(f"⚠️  Не реализовано: {not_implemented_count}/{total_count}")
        print(f"❌ Неудачных тестов: {failed_count}/{total_count}")

        if success_count > 0:
            print(
                f"📈 Процент успеха: {(success_count / total_count) * 100:.1f}%"
            )

            successful_results = [
                r for r in self.test_results if r["result"] == "SUCCESS"
            ]
            avg_distance = np.mean(
                [r["metrics"]["total_distance"] for r in successful_results]
            )
            avg_smoothness = np.mean(
                [r["metrics"]["smoothness"] for r in successful_results]
            )
            avg_segments = np.mean(
                [r["metrics"]["num_waypoints"] for r in successful_results]
            )
            avg_time = np.mean(
                [r["execution_time"] for r in successful_results]
            )

            print(f"\n📏 Средняя длина пути: {avg_distance:.2f}м")
            print(f"🌊 Средняя плавность: {avg_smoothness:.3f} рад")
            print(f"🔗 Среднее количество сегментов: {avg_segments:.1f}")
            print(f"⏱️  Среднее время выполнения: {avg_time:.3f}с")

            print("\n⭐ ПРЕИМУЩЕСТВА A*:")
            print("• Гарантированно находит оптимальный путь")
            print("• Высокая производительность на структурированных сетках")
            print("• Эффективная эвристика сокращает поиск")
            print("• Детерминированный результат")

        elif not_implemented_count > 0:
            print("\n🔨 A* алгоритм готов к тестированию после реализации!")
            print(
                "📋 Все тестовые сценарии подготовлены и ждут реализации алгоритма."
            )

        print("\n📋 ДЕТАЛИ ТЕСТОВ:")
        for result in self.test_results:
            if result["result"] == "SUCCESS":
                status_icon = "✅"
            elif result["result"] == "NOT_IMPLEMENTED":
                status_icon = "⚠️ "
            else:
                status_icon = "❌"

            print(f"{status_icon} {result['name']}: {result['result']}")
            if result["error"] and result["result"] != "NOT_IMPLEMENTED":
                print(f"   └─ Ошибка: {result['error']}")

        print("\n🎯 СЛЕДУЮЩИЕ ШАГИ:")
        if not_implemented_count > 0:
            print("1. Реализовать A* алгоритм в GlobalNavigationPlanner")
            print("2. Настроить сетку дискретизации пространства")
            print(
                "3. Реализовать эвристическую функцию (например, Евклидово расстояние)"
            )
            print("4. Запустить тесты повторно для проверки оптимальности")
            print("5. Сравнить результаты с Carrot и RRT* планировщиками")


def main():
    """Главная функция для запуска тестов A*."""
    print("🔧 Инициализация тестового набора A*-планировщика...")

    try:
        # Создаем тестовый набор
        test_suite = AStarPlannerTestSuite()

        # Запускаем все тесты
        test_suite.run_all_tests(
            visualize=True,  # Показывать визуализацию
            interactive=True,  # Интерактивный режим
        )

    except KeyboardInterrupt:
        print("\n\n⏹️  Тестирование A* прервано пользователем")
    except Exception as e:
        print(f"\n\n💥 Критическая ошибка: {e}")
        import traceback

        traceback.print_exc()


if __name__ == "__main__":
    main()
