#!/usr/bin/env python3
"""
Анализатор отчетов тестирования ARA API.

Этот скрипт предоставляет быстрый анализ созданных отчетов тестирования,
включая статистику, тренды и рекомендации по улучшению качества кода.

Использование:
    python analyze_reports.py                    # Анализ всех отчетов
    python analyze_reports.py --summary         # Только сводка
    python analyze_reports.py --coverage        # Только анализ покрытия
    python analyze_reports.py --trends          # Анализ трендов (если есть история)
"""

import argparse
import re
import sys
from pathlib import Path
from typing import Any, Dict, Optional


class ReportsAnalyzer:
    """Анализатор отчетов тестирования."""

    def __init__(self, reports_dir: str) -> None:
        """
        Инициализация анализатора.

        Args:
            reports_dir: Путь к папке с отчетами
        """
        self.reports_dir = Path(reports_dir)
        self.reports: Dict[str, Any] = {}

    def load_reports(self) -> None:
        """Загрузка всех доступных отчетов."""
        if not self.reports_dir.exists():
            print(f"❌ Папка отчетов не найдена: {self.reports_dir}")
            return

        # Загружаем различные типы отчетов
        self._load_summary_report()
        self._load_imports_report()
        self._load_unit_tests_report()
        self._load_coverage_report()
        self._load_linting_report()
        self._load_format_report()
        self._load_type_check_report()

    def _load_summary_report(self) -> None:
        """Загрузка сводного отчета."""
        summary_file = self.reports_dir / "summary_report.txt"
        if summary_file.exists():
            content = summary_file.read_text(encoding="utf-8")
            self.reports["summary"] = self._parse_summary(content)

    def _load_imports_report(self) -> None:
        """Загрузка отчета импортов."""
        imports_file = self.reports_dir / "imports_report.txt"
        if imports_file.exists():
            content = imports_file.read_text(encoding="utf-8")
            self.reports["imports"] = self._parse_imports(content)

    def _load_unit_tests_report(self) -> None:
        """Загрузка отчета unit тестов."""
        unit_file = self.reports_dir / "unit_tests_report.txt"
        if unit_file.exists():
            content = unit_file.read_text(encoding="utf-8")
            self.reports["unit_tests"] = self._parse_unit_tests(content)

    def _load_coverage_report(self) -> None:
        """Загрузка отчета покрытия."""
        coverage_file = self.reports_dir / "coverage_report.txt"
        if coverage_file.exists():
            content = coverage_file.read_text(encoding="utf-8")
            self.reports["coverage"] = self._parse_coverage(content)

    def _load_linting_report(self) -> None:
        """Загрузка отчета линтинга."""
        linting_file = self.reports_dir / "linting_report.txt"
        if linting_file.exists():
            content = linting_file.read_text(encoding="utf-8")
            self.reports["linting"] = self._parse_linting(content)

    def _load_format_report(self) -> None:
        """Загрузка отчета форматирования."""
        format_file = self.reports_dir / "format_check_report.txt"
        if format_file.exists():
            content = format_file.read_text(encoding="utf-8")
            self.reports["formatting"] = self._parse_formatting(content)

    def _load_type_check_report(self) -> None:
        """Загрузка отчета проверки типов."""
        type_file = self.reports_dir / "type_check_report.txt"
        if type_file.exists():
            content = type_file.read_text(encoding="utf-8")
            self.reports["type_check"] = self._parse_type_check(content)

    def _parse_summary(self, content: str) -> Dict[str, Any]:
        """Парсинг сводного отчета."""
        data = {"date": None, "tests_status": {}, "overall_status": None}

        # Извлекаем дату
        date_match = re.search(r"Дата: (.+)", content)
        if date_match:
            data["date"] = date_match.group(1)

        # Извлекаем статусы тестов
        status_section = re.search(
            r"СТАТУС ТЕСТИРОВАНИЯ:(.*?)СОЗДАННЫЕ ОТЧЕТЫ:", content, re.DOTALL
        )
        if status_section:
            status_text = status_section.group(1)

            # Парсим каждый тип теста
            for line in status_text.split("\n"):
                if "📋 Тесты импортов:" in line:
                    data["tests_status"]["imports"] = "ВЫПОЛНЕНЫ" in line
                elif "🧪 Unit тесты:" in line:
                    data["tests_status"]["unit_tests"] = "ВЫПОЛНЕНЫ" in line
                elif "🔍 Проверка типов:" in line:
                    data["tests_status"]["type_check"] = "ВЫПОЛНЕНА" in line
                elif "🎨 Проверка форматирования:" in line:
                    data["tests_status"]["formatting"] = "ВЫПОЛНЕНА" in line
                elif "🔧 Линтинг:" in line:
                    data["tests_status"]["linting"] = "ВЫПОЛНЕН" in line
                elif "📊 Анализ покрытия:" in line:
                    data["tests_status"]["coverage"] = "ВЫПОЛНЕН" in line

        # Общий статус
        if "🎉 ТЕСТИРОВАНИЕ ЗАВЕРШЕНО УСПЕШНО" in content:
            data["overall_status"] = "success"
        elif "ПРОБЛЕМАМИ" in content:
            data["overall_status"] = "warning"
        else:
            data["overall_status"] = "unknown"

        return data

    def _parse_imports(self, content: str) -> Dict[str, Any]:
        """Парсинг отчета импортов."""
        data = {
            "total_modules": 0,
            "successful": 0,
            "failed": 0,
            "success_rate": 0.0,
        }

        # Извлекаем статистику
        stats_match = re.search(r"Всего модулей: (\d+)", content)
        if stats_match:
            data["total_modules"] = int(stats_match.group(1))

        success_match = re.search(r"Успешных импортов: (\d+)", content)
        if success_match:
            data["successful"] = int(success_match.group(1))

        failed_match = re.search(r"Неудачных импортов: (\d+)", content)
        if failed_match:
            data["failed"] = int(failed_match.group(1))

        if data["total_modules"] > 0:
            data["success_rate"] = (
                data["successful"] / data["total_modules"]
            ) * 100

        return data

    def _parse_unit_tests(self, content: str) -> Dict[str, Any]:
        """Парсинг отчета unit тестов."""
        data = {"total_files": 0, "successful": 0, "warnings": 0}

        # Подсчет тестовых файлов
        files_match = re.search(r"Найдено тестовых файлов: (\d+)", content)
        if files_match:
            data["total_files"] = int(files_match.group(1))

        # Подсчет успешных и с предупреждениями
        successful_count = len(re.findall(r"✅.*УСПЕШНО", content))
        warnings_count = len(re.findall(r"⚠️.*ПРЕДУПРЕЖДЕНИЯ", content))

        data["successful"] = successful_count
        data["warnings"] = warnings_count

        return data

    def _parse_coverage(self, content: str) -> Dict[str, Any]:
        """Парсинг отчета покрытия."""
        data = {"total_coverage": 0.0, "files_analyzed": 0}

        # Попытка извлечь общее покрытие
        coverage_match = re.search(r"TOTAL.*?(\d+)%", content)
        if coverage_match:
            data["total_coverage"] = float(coverage_match.group(1))

        # Подсчет проанализированных файлов
        files_count = len(re.findall(r"\.py.*?\d+%", content))
        data["files_analyzed"] = files_count

        return data

    def _parse_linting(self, content: str) -> Dict[str, Any]:
        """Парсинг отчета линтинга."""
        data = {
            "ruff_issues": 0,
            "flake8_issues": 0,
            "overall_status": "unknown",
        }

        # Статус линтинга
        if "✅.*УСПЕШНО" in content:
            data["overall_status"] = "success"
        elif "⚠️.*ПРОБЛЕМЫ" in content:
            data["overall_status"] = "issues"

        return data

    def _parse_formatting(self, content: str) -> Dict[str, Any]:
        """Парсинг отчета форматирования."""
        data = {"needs_formatting": False, "status": "unknown"}

        if "✅.*КОРРЕКТНО" in content:
            data["status"] = "correct"
            data["needs_formatting"] = False
        elif "⚠️.*ПЕРЕФОРМАТИРОВАНИЯ" in content:
            data["status"] = "needs_fix"
            data["needs_formatting"] = True

        return data

    def _parse_type_check(self, content: str) -> Dict[str, Any]:
        """Парсинг отчета проверки типов."""
        data = {"status": "unknown", "issues_count": 0}

        if "✅.*УСПЕШНО" in content:
            data["status"] = "success"
        elif "⚠️.*ПРОБЛЕМЫ" in content:
            data["status"] = "issues"

        return data

    def print_summary(self) -> None:
        """Вывод краткой сводки."""
        print("🔍 АНАЛИЗ ОТЧЕТОВ ТЕСТИРОВАНИЯ ARA API")
        print("=" * 50)

        if not self.reports:
            print("❌ Отчеты не найдены")
            return

        # Общая информация
        if "summary" in self.reports:
            summary = self.reports["summary"]
            print(
                f"📅 Дата последнего тестирования: {summary.get('date', 'Неизвестно')}"
            )
            print(
                f"🎯 Общий статус: {self._status_emoji(summary.get('overall_status'))}"
            )
            print()

        # Статистика по типам тестов
        print("📊 СТАТИСТИКА ПО ТЕСТАМ:")
        print("-" * 30)

        executed_tests = 0
        successful_tests = 0

        for test_type, data in self.reports.items():
            if test_type == "summary":
                continue

            status = self._get_test_status(test_type, data)
            print(
                f"{self._test_emoji(test_type)} {test_type.replace('_', ' ').title()}: {status}"
            )

            if status != "НЕ ВЫПОЛНЯЛСЯ":
                executed_tests += 1
                if "✅" in status:
                    successful_tests += 1

        print()
        print(f"📈 Выполнено тестов: {executed_tests}")
        print(f"✅ Успешных: {successful_tests}")
        print(f"⚠️ С проблемами: {executed_tests - successful_tests}")

        if executed_tests > 0:
            success_rate = (successful_tests / executed_tests) * 100
            print(f"📊 Успешность: {success_rate:.1f}%")

    def print_detailed_analysis(self) -> None:
        """Подробный анализ всех отчетов."""
        self.print_summary()
        print("\n" + "=" * 50)
        print("📋 ПОДРОБНЫЙ АНАЛИЗ")
        print("=" * 50)

        # Анализ импортов
        if "imports" in self.reports:
            imports = self.reports["imports"]
            print("\n🔗 ИМПОРТЫ МОДУЛЕЙ:")
            print(f"   Всего модулей: {imports.get('total_modules', 0)}")
            print(f"   Успешных: {imports.get('successful', 0)}")
            print(f"   Неудачных: {imports.get('failed', 0)}")
            print(f"   Успешность: {imports.get('success_rate', 0):.1f}%")

        # Анализ unit тестов
        if "unit_tests" in self.reports:
            unit = self.reports["unit_tests"]
            print("\n🧪 UNIT ТЕСТЫ:")
            print(f"   Файлов тестов: {unit.get('total_files', 0)}")
            print(f"   Успешных: {unit.get('successful', 0)}")
            print(f"   С предупреждениями: {unit.get('warnings', 0)}")

        # Анализ покрытия
        if "coverage" in self.reports:
            coverage = self.reports["coverage"]
            print("\n📊 ПОКРЫТИЕ КОДА:")
            print(
                f"   Общее покрытие: {coverage.get('total_coverage', 0):.1f}%"
            )
            print(
                f"   Файлов проанализировано: {coverage.get('files_analyzed', 0)}"
            )

        # Рекомендации
        self._print_recommendations()

    def _print_recommendations(self) -> None:
        """Вывод рекомендаций по улучшению."""
        print("\n💡 РЕКОМЕНДАЦИИ:")
        print("-" * 20)

        recommendations = []

        # Проверка импортов
        if "imports" in self.reports:
            imports = self.reports["imports"]
            if imports.get("failed", 0) > 0:
                recommendations.append(
                    "🔧 Исправить проблемы с импортами модулей"
                )

        # Проверка форматирования
        if "formatting" in self.reports:
            formatting = self.reports["formatting"]
            if formatting.get("needs_formatting", False):
                recommendations.append(
                    "🎨 Запустить переформатирование кода: poetry run black ara_api tests"
                )

        # Проверка линтинга
        if "linting" in self.reports:
            linting = self.reports["linting"]
            if linting.get("overall_status") == "issues":
                recommendations.append(
                    "🔍 Исправить проблемы, найденные линтером"
                )

        # Проверка типизации
        if "type_check" in self.reports:
            type_check = self.reports["type_check"]
            if type_check.get("status") == "issues":
                recommendations.append("📝 Улучшить типизацию кода")

        # Проверка покрытия
        if "coverage" in self.reports:
            coverage = self.reports["coverage"]
            if coverage.get("total_coverage", 0) < 70:
                recommendations.append(
                    "📈 Увеличить покрытие кода тестами (текущее < 70%)"
                )

        if recommendations:
            for i, rec in enumerate(recommendations, 1):
                print(f"   {i}. {rec}")
        else:
            print("   ✅ Все проверки прошли успешно! Рекомендаций нет.")

    def _status_emoji(self, status: Optional[str]) -> str:
        """Эмодзи для статуса."""
        emoji_map = {
            "success": "✅ УСПЕШНО",
            "warning": "⚠️ ЕСТЬ ПРОБЛЕМЫ",
            "unknown": "❓ НЕИЗВЕСТНО",
        }
        return emoji_map.get(status, "❓ НЕИЗВЕСТНО")

    def _test_emoji(self, test_type: str) -> str:
        """Эмодзи для типа теста."""
        emoji_map = {
            "imports": "🔗",
            "unit_tests": "🧪",
            "coverage": "📊",
            "linting": "🔧",
            "formatting": "🎨",
            "type_check": "🔍",
        }
        return emoji_map.get(test_type, "📋")

    def _get_test_status(self, test_type: str, data: Dict[str, Any]) -> str:
        """Получение статуса теста."""
        if test_type == "imports":
            if data.get("failed", 0) == 0:
                return "✅ УСПЕШНО"
            else:
                return "⚠️ ЕСТЬ ПРОБЛЕМЫ"
        elif test_type == "unit_tests":
            if data.get("warnings", 0) == 0:
                return "✅ УСПЕШНО"
            else:
                return "⚠️ ЕСТЬ ПРЕДУПРЕЖДЕНИЯ"
        elif test_type == "coverage":
            coverage = data.get("total_coverage", 0)
            if coverage >= 80:
                return "✅ ОТЛИЧНО"
            elif coverage >= 60:
                return "⚠️ УДОВЛЕТВОРИТЕЛЬНО"
            else:
                return "❌ НИЗКОЕ"
        elif test_type == "formatting":
            if data.get("status") == "correct":
                return "✅ КОРРЕКТНО"
            else:
                return "⚠️ ТРЕБУЕТ ИСПРАВЛЕНИЯ"
        elif test_type == "linting":
            if data.get("overall_status") == "success":
                return "✅ УСПЕШНО"
            else:
                return "⚠️ ЕСТЬ ПРОБЛЕМЫ"
        elif test_type == "type_check":
            if data.get("status") == "success":
                return "✅ УСПЕШНО"
            else:
                return "⚠️ ЕСТЬ ПРОБЛЕМЫ"

        return "НЕ ВЫПОЛНЯЛСЯ"


def main() -> None:
    """Главная функция."""
    parser = argparse.ArgumentParser(
        description="Анализатор отчетов тестирования ARA API",
        formatter_class=argparse.RawDescriptionHelpFormatter,
    )

    parser.add_argument(
        "--reports-dir",
        default="tests/reports",
        help="Путь к папке с отчетами (по умолчанию: tests/reports)",
    )

    parser.add_argument(
        "--summary", action="store_true", help="Показать только краткую сводку"
    )

    parser.add_argument(
        "--coverage",
        action="store_true",
        help="Показать только анализ покрытия",
    )

    args = parser.parse_args()

    # Создаем анализатор
    analyzer = ReportsAnalyzer(args.reports_dir)
    analyzer.load_reports()

    if not analyzer.reports:
        print("❌ Отчеты не найдены. Запустите тестирование: ./test.sh")
        sys.exit(1)

    # Выводим результаты
    if args.summary:
        analyzer.print_summary()
    elif args.coverage and "coverage" in analyzer.reports:
        print("📊 АНАЛИЗ ПОКРЫТИЯ КОДА")
        print("=" * 30)
        coverage = analyzer.reports["coverage"]
        print(f"Общее покрытие: {coverage.get('total_coverage', 0):.1f}%")
        print(f"Файлов проанализировано: {coverage.get('files_analyzed', 0)}")

        # Открыть HTML отчет если доступен
        html_report = Path(args.reports_dir) / "coverage_html" / "index.html"
        if html_report.exists():
            print(f"HTML отчет: {html_report}")
            print("Для просмотра: open tests/reports/coverage_html/index.html")
    else:
        analyzer.print_detailed_analysis()


if __name__ == "__main__":
    main()
